"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack
     */
    static resourceServer(server, scope) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cognito_IUserPoolResourceServer(server);
            jsiiDeprecationWarnings.monocdk_aws_cognito_ResourceServerScope(scope);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resourceServer);
            }
            throw error;
        }
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
}
exports.OAuthScope = OAuthScope;
_a = JSII_RTTI_SYMBOL_1;
OAuthScope[_a] = { fqn: "monocdk.aws_cognito.OAuthScope", version: "1.185.0" };
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
_b = JSII_RTTI_SYMBOL_1;
UserPoolClientIdentityProvider[_b] = { fqn: "monocdk.aws_cognito.UserPoolClientIdentityProvider", version: "1.185.0" };
/**
 * Allow users to sign in using 'Sign In With Apple'.
 * A `UserPoolIdentityProviderApple` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.APPLE = new UserPoolClientIdentityProvider('SignInWithApple');
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_cognito_UserPoolClientProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, UserPoolClient);
            }
            throw error;
        }
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = props.oAuth?.flows ?? {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = props.oAuth?.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: props.oAuth?.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
            readAttributes: props.readAttributes?.attributes(),
            writeAttributes: props.writeAttributes?.attributes(),
            enableTokenRevocation: props.enableTokenRevocation,
        });
        this.configureTokenValidity(resource, props);
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        const scopes = oAuth?.scopes ?? [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
    configureTokenValidity(resource, props) {
        this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.idTokenValidity);
        this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.accessTokenValidity);
        this.validateDuration('refreshTokenValidity', core_1.Duration.minutes(60), core_1.Duration.days(10 * 365), props.refreshTokenValidity);
        if (props.refreshTokenValidity) {
            this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.idTokenValidity);
            this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.accessTokenValidity);
        }
        if (props.accessTokenValidity || props.idTokenValidity || props.refreshTokenValidity) {
            resource.tokenValidityUnits = {
                idToken: props.idTokenValidity ? 'minutes' : undefined,
                accessToken: props.accessTokenValidity ? 'minutes' : undefined,
                refreshToken: props.refreshTokenValidity ? 'minutes' : undefined,
            };
        }
        ;
        resource.idTokenValidity = props.idTokenValidity ? props.idTokenValidity.toMinutes() : undefined;
        resource.refreshTokenValidity = props.refreshTokenValidity ? props.refreshTokenValidity.toMinutes() : undefined;
        resource.accessTokenValidity = props.accessTokenValidity ? props.accessTokenValidity.toMinutes() : undefined;
    }
    validateDuration(name, min, max, value) {
        if (value === undefined) {
            return;
        }
        if (value.toMilliseconds() < min.toMilliseconds() || value.toMilliseconds() > max.toMilliseconds()) {
            throw new Error(`${name}: Must be a duration between ${min.toHumanString()} and ${max.toHumanString()} (inclusive); received ${value.toHumanString()}.`);
        }
    }
}
exports.UserPoolClient = UserPoolClient;
_c = JSII_RTTI_SYMBOL_1;
UserPoolClient[_c] = { fqn: "monocdk.aws_cognito.UserPoolClient", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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