"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("../../../aws-kms");
const s3 = require("../../../aws-s3");
const cdk = require("../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../../core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends core_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        const createKmsKey = props.createKmsKey ?? true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                enableKeyRotation: props.enableKeyRotation,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
            enforceSSL: true,
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
            enableKeyRotation: props.enableKeyRotation,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3Jvc3MtcmVnaW9uLXN1cHBvcnQtc3RhY2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjcm9zcy1yZWdpb24tc3VwcG9ydC1zdGFjay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBd0M7QUFDeEMsc0NBQXNDO0FBQ3RDLHFDQUFxQztBQUVyQyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELHdDQUEwQztBQUUxQyxNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQztBQUV2Qzs7OztHQUlHO0FBQ0gsTUFBTSw2QkFBOEIsU0FBUSxHQUFHLENBQUMsS0FBSztJQUN6QyxvQkFBb0I7UUFDNUIsSUFBSSxRQUFRLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixFQUFFLENBQUM7UUFDNUMsSUFBSSxRQUFRLENBQUMsVUFBVSxDQUFDLHFCQUFxQixDQUFDLEVBQUU7WUFDOUMsZ0ZBQWdGO1lBQ2hGLFFBQVEsR0FBRyxRQUFRLENBQUMsU0FBUyxDQUFDLHFCQUFxQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzdEO1FBQ0QsTUFBTSxTQUFTLEdBQUcsRUFBRSxHQUFHLHFCQUFxQixDQUFDLE1BQU0sQ0FBQztRQUNwRCxzREFBc0Q7UUFDdEQsK0NBQStDO1FBQy9DLE9BQU8scUJBQXFCLEdBQUcsZUFBZSxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztLQUNyRTtDQUNGO0FBRUQsU0FBUyxlQUFlLENBQUMsR0FBVyxFQUFFLENBQVM7SUFDN0MsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMvQyxPQUFPLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLENBQUM7QUFDbkMsQ0FBQztBQXVCRCxNQUFhLDJCQUE0QixTQUFRLGdCQUFTO0lBR3hELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBMEMsRUFBRTtRQUNwRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDO1FBRWhELElBQUksZUFBZSxDQUFDO1FBQ3BCLElBQUksWUFBWSxFQUFFO1lBQ2hCLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsdURBQXVELEVBQUU7Z0JBQy9GLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87Z0JBQ3hDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7YUFDM0MsQ0FBQyxDQUFDO1lBQ0gsZUFBZSxHQUFHLElBQUksNkJBQTZCLENBQUMsSUFBSSxFQUFFLHlEQUF5RCxFQUFFO2dCQUNuSCxTQUFTLEVBQUUsYUFBYTtnQkFDeEIsU0FBUyxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsa0JBQWtCO2dCQUM5QyxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPO2FBQ3pDLENBQUMsQ0FBQztTQUNKO1FBQ0QsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsMENBQTBDLEVBQUU7WUFDdkYsVUFBVSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsa0JBQWtCO1lBQy9DLFVBQVUsRUFBRSxlQUFlLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXO1lBQ3ZGLGFBQWEsRUFBRSxlQUFlO1lBQzlCLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO1NBQ2xELENBQUMsQ0FBQztLQUNKO0NBQ0Y7QUE1QkQsa0VBNEJDO0FBNENEOzs7R0FHRztBQUNILE1BQWEsdUJBQXdCLFNBQVEsR0FBRyxDQUFDLEtBQUs7SUFNcEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFtQztRQUMzRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxLQUFLLENBQUM7WUFDbkMsR0FBRyxFQUFFO2dCQUNILE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtnQkFDcEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO2FBQ3ZCO1lBQ0QsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQy9CLENBQUMsQ0FBQztRQUVILE1BQU0sMkJBQTJCLEdBQUcsSUFBSSwyQkFBMkIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ25GLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1NBQzNDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxpQkFBaUIsR0FBRywyQkFBMkIsQ0FBQyxpQkFBaUIsQ0FBQztLQUN4RTtDQUNGO0FBdEJELDBEQXNCQztBQUVELFNBQVMsaUJBQWlCLENBQUMsS0FBbUM7SUFDNUQsT0FBTyxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsWUFBWSxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7QUFDOUQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGttcyBmcm9tICcuLi8uLi8uLi9hd3Mta21zJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJy4uLy4uLy4uL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnLi4vLi4vLi4vY29yZSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuLi8uLi8uLi9jb3JlJztcblxuY29uc3QgUkVRVUlSRURfQUxJQVNfUFJFRklYID0gJ2FsaWFzLyc7XG5cbi8qKlxuICogQSBjbGFzcyBuZWVkZWQgdG8gd29yayBhcm91bmQgQ29kZVBpcGVsaW5lJ3MgZXh0cmVtZWx5IHNtYWxsICgxMDAgY2hhcmFjdGVycylcbiAqIGxpbWl0IGZvciB0aGUgbmFtZS9BUk4gb2YgdGhlIGtleSBpbiB0aGUgQXJ0aWZhY3RTdG9yZS5cbiAqIExpbWl0cyB0aGUgbGVuZ3RoIG9mIHRoZSBhbGlhcycgYXV0by1nZW5lcmF0ZWQgbmFtZSB0byA1MCBjaGFyYWN0ZXJzLlxuICovXG5jbGFzcyBBbGlhc1dpdGhTaG9ydGVyR2VuZXJhdGVkTmFtZSBleHRlbmRzIGttcy5BbGlhcyB7XG4gIHByb3RlY3RlZCBnZW5lcmF0ZVBoeXNpY2FsTmFtZSgpOiBzdHJpbmcge1xuICAgIGxldCBiYXNlTmFtZSA9IHN1cGVyLmdlbmVyYXRlUGh5c2ljYWxOYW1lKCk7XG4gICAgaWYgKGJhc2VOYW1lLnN0YXJ0c1dpdGgoUkVRVUlSRURfQUxJQVNfUFJFRklYKSkge1xuICAgICAgLy8gcmVtb3ZlIHRoZSBwcmVmaXgsIGJlY2F1c2Ugd2UncmUgdGFraW5nIHRoZSBsYXN0IGNoYXJhY3RlcnMgb2YgdGhlIG5hbWUgYmVsb3dcbiAgICAgIGJhc2VOYW1lID0gYmFzZU5hbWUuc3Vic3RyaW5nKFJFUVVJUkVEX0FMSUFTX1BSRUZJWC5sZW5ndGgpO1xuICAgIH1cbiAgICBjb25zdCBtYXhMZW5ndGggPSA1MCAtIFJFUVVJUkVEX0FMSUFTX1BSRUZJWC5sZW5ndGg7XG4gICAgLy8gdGFrZSB0aGUgbGFzdCBjaGFyYWN0ZXJzLCBhcyB0aGV5IGluY2x1ZGUgdGhlIGhhc2gsXG4gICAgLy8gYW5kIHNvIGhhdmUgYSBoaWdoZXIgY2hhbmNlIG9mIG5vdCBjb2xsaWRpbmdcbiAgICByZXR1cm4gUkVRVUlSRURfQUxJQVNfUFJFRklYICsgbGFzdE5DaGFyYWN0ZXJzKGJhc2VOYW1lLCBtYXhMZW5ndGgpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGxhc3ROQ2hhcmFjdGVycyhzdHI6IHN0cmluZywgbjogbnVtYmVyKSB7XG4gIGNvbnN0IHN0YXJ0SW5kZXggPSBNYXRoLm1heChzdHIubGVuZ3RoIC0gbiwgMCk7XG4gIHJldHVybiBzdHIuc3Vic3RyaW5nKHN0YXJ0SW5kZXgpO1xufVxuXG4vKipcbiAqIFByb3BzIGZvciB0aGUgc3VwcG9ydCBzdGFja1xuICovXG5leHBvcnQgaW50ZXJmYWNlIENyb3NzUmVnaW9uU3VwcG9ydENvbnN0cnVjdFByb3BzIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gY3JlYXRlIHRoZSBLTVMgQ01LXG4gICAqXG4gICAqIChSZXF1aXJlZCBmb3IgY3Jvc3MtYWNjb3VudCBkZXBsb3ltZW50cylcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY3JlYXRlS21zS2V5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRW5hYmxlcyBLTVMga2V5IHJvdGF0aW9uIGZvciBjcm9zcy1hY2NvdW50IGtleXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2UgKGtleSByb3RhdGlvbiBpcyBkaXNhYmxlZClcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZUtleVJvdGF0aW9uPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNsYXNzIENyb3NzUmVnaW9uU3VwcG9ydENvbnN0cnVjdCBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSByZXBsaWNhdGlvbkJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ3Jvc3NSZWdpb25TdXBwb3J0Q29uc3RydWN0UHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBjcmVhdGVLbXNLZXkgPSBwcm9wcy5jcmVhdGVLbXNLZXkgPz8gdHJ1ZTtcblxuICAgIGxldCBlbmNyeXB0aW9uQWxpYXM7XG4gICAgaWYgKGNyZWF0ZUttc0tleSkge1xuICAgICAgY29uc3QgZW5jcnlwdGlvbktleSA9IG5ldyBrbXMuS2V5KHRoaXMsICdDcm9zc1JlZ2lvbkNvZGVQaXBlbGluZVJlcGxpY2F0aW9uQnVja2V0RW5jcnlwdGlvbktleScsIHtcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgICAgZW5hYmxlS2V5Um90YXRpb246IHByb3BzLmVuYWJsZUtleVJvdGF0aW9uLFxuICAgICAgfSk7XG4gICAgICBlbmNyeXB0aW9uQWxpYXMgPSBuZXcgQWxpYXNXaXRoU2hvcnRlckdlbmVyYXRlZE5hbWUodGhpcywgJ0Nyb3NzUmVnaW9uQ29kZVBpcGVsaW5lUmVwbGljYXRpb25CdWNrZXRFbmNyeXB0aW9uQWxpYXMnLCB7XG4gICAgICAgIHRhcmdldEtleTogZW5jcnlwdGlvbktleSxcbiAgICAgICAgYWxpYXNOYW1lOiBjZGsuUGh5c2ljYWxOYW1lLkdFTkVSQVRFX0lGX05FRURFRCxcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0pO1xuICAgIH1cbiAgICB0aGlzLnJlcGxpY2F0aW9uQnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnQ3Jvc3NSZWdpb25Db2RlUGlwZWxpbmVSZXBsaWNhdGlvbkJ1Y2tldCcsIHtcbiAgICAgIGJ1Y2tldE5hbWU6IGNkay5QaHlzaWNhbE5hbWUuR0VORVJBVEVfSUZfTkVFREVELFxuICAgICAgZW5jcnlwdGlvbjogZW5jcnlwdGlvbkFsaWFzID8gczMuQnVja2V0RW5jcnlwdGlvbi5LTVMgOiBzMy5CdWNrZXRFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICAgICAgZW5jcnlwdGlvbktleTogZW5jcnlwdGlvbkFsaWFzLFxuICAgICAgZW5mb3JjZVNTTDogdHJ1ZSxcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBzMy5CbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3Ige0BsaW5rIENyb3NzUmVnaW9uU3VwcG9ydFN0YWNrfS5cbiAqIFRoaXMgaW50ZXJmYWNlIGlzIHByaXZhdGUgdG8gdGhlIGF3cy1jb2RlcGlwZWxpbmUgcGFja2FnZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDcm9zc1JlZ2lvblN1cHBvcnRTdGFja1Byb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBTdGFjayB0aGUgUGlwZWxpbmUgaXRzZWxmIGJlbG9uZ3MgdG8uXG4gICAqIFVzZWQgdG8gZ2VuZXJhdGUgYSBtb3JlIGZyaWVuZGx5IG5hbWUgZm9yIHRoZSBzdXBwb3J0IFN0YWNrLlxuICAgKi9cbiAgcmVhZG9ubHkgcGlwZWxpbmVTdGFja05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFXUyByZWdpb24gdGhpcyBTdGFjayByZXNpZGVzIGluLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVnaW9uOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBV1MgYWNjb3VudCBJRCB0aGlzIFN0YWNrIGJlbG9uZ3MgdG8uXG4gICAqXG4gICAqIEBleGFtcGxlICcwMTIzNDU2Nzg5MDEnXG4gICAqL1xuICByZWFkb25seSBhY2NvdW50OiBzdHJpbmc7XG5cbiAgcmVhZG9ubHkgc3ludGhlc2l6ZXI6IGNkay5JU3RhY2tTeW50aGVzaXplciB8IHVuZGVmaW5lZDtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gY3JlYXRlIGEgS01TIGtleSBpbiB0aGUgc3VwcG9ydCBzdGFja1xuICAgKlxuICAgKiAoUmVxdWlyZWQgZm9yIGNyb3NzLWFjY291bnQgZGVwbG95bWVudHMpXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGNyZWF0ZUttc0tleT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgS01TIGtleSByb3RhdGlvbiBmb3IgY3Jvc3MtYWNjb3VudCBrZXlzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlIChrZXkgcm90YXRpb24gaXMgZGlzYWJsZWQpXG4gICAqL1xuICByZWFkb25seSBlbmFibGVLZXlSb3RhdGlvbj86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQSBTdGFjayBjb250YWluaW5nIHJlc291cmNlcyByZXF1aXJlZCBmb3IgdGhlIGNyb3NzLXJlZ2lvbiBDb2RlUGlwZWxpbmUgZnVuY3Rpb25hbGl0eSB0byB3b3JrLlxuICogVGhpcyBjbGFzcyBpcyBwcml2YXRlIHRvIHRoZSBhd3MtY29kZXBpcGVsaW5lIHBhY2thZ2UuXG4gKi9cbmV4cG9ydCBjbGFzcyBDcm9zc1JlZ2lvblN1cHBvcnRTdGFjayBleHRlbmRzIGNkay5TdGFjayB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgUzMgQnVja2V0IHVzZWQgZm9yIHJlcGxpY2F0aW5nIHRoZSBQaXBlbGluZSdzIGFydGlmYWN0cyBpbnRvIHRoZSByZWdpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVwbGljYXRpb25CdWNrZXQ6IHMzLklCdWNrZXQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENyb3NzUmVnaW9uU3VwcG9ydFN0YWNrUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHN0YWNrTmFtZTogZ2VuZXJhdGVTdGFja05hbWUocHJvcHMpLFxuICAgICAgZW52OiB7XG4gICAgICAgIHJlZ2lvbjogcHJvcHMucmVnaW9uLFxuICAgICAgICBhY2NvdW50OiBwcm9wcy5hY2NvdW50LFxuICAgICAgfSxcbiAgICAgIHN5bnRoZXNpemVyOiBwcm9wcy5zeW50aGVzaXplcixcbiAgICB9KTtcblxuICAgIGNvbnN0IGNyb3NzUmVnaW9uU3VwcG9ydENvbnN0cnVjdCA9IG5ldyBDcm9zc1JlZ2lvblN1cHBvcnRDb25zdHJ1Y3QodGhpcywgJ0RlZmF1bHQnLCB7XG4gICAgICBjcmVhdGVLbXNLZXk6IHByb3BzLmNyZWF0ZUttc0tleSxcbiAgICAgIGVuYWJsZUtleVJvdGF0aW9uOiBwcm9wcy5lbmFibGVLZXlSb3RhdGlvbixcbiAgICB9KTtcbiAgICB0aGlzLnJlcGxpY2F0aW9uQnVja2V0ID0gY3Jvc3NSZWdpb25TdXBwb3J0Q29uc3RydWN0LnJlcGxpY2F0aW9uQnVja2V0O1xuICB9XG59XG5cbmZ1bmN0aW9uIGdlbmVyYXRlU3RhY2tOYW1lKHByb3BzOiBDcm9zc1JlZ2lvblN1cHBvcnRTdGFja1Byb3BzKTogc3RyaW5nIHtcbiAgcmV0dXJuIGAke3Byb3BzLnBpcGVsaW5lU3RhY2tOYW1lfS1zdXBwb3J0LSR7cHJvcHMucmVnaW9ufWA7XG59XG4iXX0=