"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("../../../aws-codepipeline");
const targets = require("../../../aws-events-targets");
const core_1 = require("../../../core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_S3SourceActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, S3SourceAction);
            }
            throw error;
        }
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(_scope);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role, this.props.bucketKey);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node) + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
_a = JSII_RTTI_SYMBOL_1;
S3SourceAction[_a] = { fqn: "monocdk.aws_codepipeline_actions.S3SourceAction", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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