"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildAction = exports.CodeBuildActionType = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("../../../aws-codebuild");
const codepipeline = require("../../../aws-codepipeline");
const iam = require("../../../aws-iam");
const cdk = require("../../../core");
const __1 = require("..");
const action_1 = require("../action");
const source_action_1 = require("../codecommit/source-action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category -
 * Build, or Test.
 * The default is Build.
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    /**
     * The action will have the Build Category.
     * This is the default.
     */
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    /**
     * The action will have the Test Category.
     */
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 */
class CodeBuildAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_CodeBuildActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CodeBuildAction);
            }
            throw error;
        }
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'exported-variables' section of the buildspec
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(scope);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(_stage);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                `codebuild:${this.props.executeBatchBuild ? 'BatchGetBuildBatches' : 'BatchGetBuilds'}`,
                `codebuild:${this.props.executeBatchBuild ? 'StartBuildBatch' : 'StartBuild'}`,
                `codebuild:${this.props.executeBatchBuild ? 'StopBuildBatch' : 'StopBuild'}`,
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        for (const inputArtifact of this.actionProperties.inputs || []) {
            // if any of the inputs come from the CodeStarConnectionsSourceAction
            // with codeBuildCloneOutput=true,
            // grant the Project's Role to use the connection
            const connectionArn = inputArtifact.getMetadata(__1.CodeStarConnectionsSourceAction._CONNECTION_ARN_PROPERTY);
            if (connectionArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codestar-connections:UseConnection'],
                    resources: [connectionArn],
                }));
            }
            // if any of the inputs come from the CodeCommitSourceAction
            // with codeBuildCloneOutput=true,
            // grant the Project's Role git pull access to the repository
            const codecommitRepositoryArn = inputArtifact.getMetadata(source_action_1.CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY);
            if (codecommitRepositoryArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codecommit:GitPull'],
                    resources: [codecommitRepositoryArn],
                }));
            }
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables, this.props.checkSecretsInPlainTextEnvVariables ?? true, this.props.project)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.string({ produce: () => this.props.input.artifactName });
        }
        if (this.props.executeBatchBuild) {
            configuration.BatchEnabled = 'true';
            this.props.project.enableBatchBuilds();
            if (this.props.combineBatchBuildArtifacts) {
                configuration.CombineArtifacts = 'true';
            }
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildAction[_a] = { fqn: "monocdk.aws_codepipeline_actions.CodeBuildAction", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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