"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseCapabilities = exports.SingletonPolicy = void 0;
const iam = require("../../../../aws-iam");
const cdk = require("../../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../../../core");
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends core_1.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStackSet(props) {
        const actions = [
            'cloudformation:CreateStackSet',
            'cloudformation:UpdateStackSet',
            'cloudformation:DescribeStackSet',
            'cloudformation:DescribeStackSetOperation',
            'cloudformation:ListStackInstances',
            'cloudformation:CreateStackInstances',
        ];
        this.statementFor({ actions }).addResources(this.stackSetArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(typeof role === 'string' ? role : role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
    stackSetArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stackset',
            resourceName: `${props.stackSetName}:*`,
        });
    }
}
exports.SingletonPolicy = SingletonPolicy;
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
exports.parseCapabilities = parseCapabilities;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2luZ2xldG9uLXBvbGljeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNpbmdsZXRvbi1wb2xpY3kudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMkNBQTJDO0FBQzNDLHdDQUF3QztBQUV4QyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELDJDQUE2QztBQUU3Qzs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxnQkFBUztJQWlCNUMsWUFBcUMsSUFBZTtRQUNsRCxLQUFLLENBQUMsSUFBNEIsRUFBRSxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7UUFEdkIsU0FBSSxHQUFKLElBQUksQ0FBVztRQUY1QyxlQUFVLEdBQTJDLEVBQUUsQ0FBQztRQUk5RCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQztLQUM1QjtJQW5CRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFlO1FBQ25DLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzRCxPQUFRLEtBQXlCLElBQUksSUFBSSxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDaEU7SUFhTSxxQkFBcUIsQ0FBQyxLQUFvRTtRQUMvRixJQUFJLENBQUMsWUFBWSxDQUFDO1lBQ2hCLE9BQU8sRUFBRTtnQkFDUCwrQkFBK0I7Z0JBQy9CLGtDQUFrQztnQkFDbEMsaUNBQWlDO2FBQ2xDO1lBQ0QsVUFBVSxFQUFFLEVBQUUsb0JBQW9CLEVBQUUsRUFBRSw4QkFBOEIsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLEVBQUU7U0FDOUYsQ0FBQyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztLQUNoRDtJQUVNLDJCQUEyQixDQUFDLEtBQW9FO1FBQ3JHLElBQUksQ0FBQyxZQUFZLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNQLGdDQUFnQztnQkFDaEMsZ0NBQWdDO2dCQUNoQyxrQ0FBa0M7Z0JBQ2xDLCtCQUErQjthQUNoQztZQUNELFVBQVUsRUFBRSxFQUFFLG9CQUFvQixFQUFFLEVBQUUsOEJBQThCLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSxFQUFFO1NBQzlGLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7S0FDaEQ7SUFFTSxzQkFBc0IsQ0FBQyxLQUF5RTtRQUNyRyxNQUFNLE9BQU8sR0FBRztZQUNkLCtCQUErQjtZQUMvQiw0QkFBNEI7WUFDNUIsNEJBQTRCO1lBQzVCLDZCQUE2QjtZQUM3QixpQ0FBaUM7WUFDakMsK0JBQStCO1lBQy9CLCtCQUErQjtTQUNoQyxDQUFDO1FBQ0YsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsT0FBTyxDQUFDLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1NBQzVDO1FBQ0QsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0tBQzVFO0lBRU0seUJBQXlCLENBQUMsS0FBZ0Q7UUFDL0UsTUFBTSxPQUFPLEdBQUc7WUFDZCwrQkFBK0I7WUFDL0IsK0JBQStCO1lBQy9CLGlDQUFpQztZQUNqQywwQ0FBMEM7WUFDMUMsbUNBQW1DO1lBQ25DLHFDQUFxQztTQUN0QyxDQUFDO1FBQ0YsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0tBQy9FO0lBRU0sZ0JBQWdCLENBQUMsS0FBNkM7UUFDbkUsSUFBSSxDQUFDLFlBQVksQ0FBQztZQUNoQixPQUFPLEVBQUU7Z0JBQ1AsK0JBQStCO2dCQUMvQiw0QkFBNEI7YUFDN0I7U0FDRixDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0tBQ2hEO0lBRU0sYUFBYSxDQUFDLElBQXdCO1FBQzNDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLENBQUMsWUFBWSxDQUFDLE9BQU8sSUFBSSxLQUFLLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7S0FDL0c7SUFFTyxZQUFZLENBQUMsUUFBMkI7UUFDOUMsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzdCLElBQUksQ0FBQyxDQUFDLEdBQUcsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDN0IsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsRUFBRSxPQUFPLEVBQUUsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDOUUsSUFBSSxRQUFRLENBQUMsVUFBVSxFQUFFO2dCQUN2QixJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7YUFDekQ7WUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDN0M7UUFDRCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFNUIsU0FBUyxNQUFNLENBQUMsS0FBd0I7WUFDdEMsTUFBTSxPQUFPLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO1lBQ3ZELE1BQU0sVUFBVSxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUN0RCxPQUFPLEdBQUcsT0FBTyxPQUFPLFVBQVUsRUFBRSxDQUFDO1lBRXJDLFNBQVMsZ0JBQWdCLENBQUMsSUFBeUI7Z0JBQ2pELElBQUksSUFBSSxJQUFJLElBQUksRUFBRTtvQkFBRSxPQUFPLEVBQUUsQ0FBQztpQkFBRTtnQkFDaEMsSUFBSSxNQUFNLEdBQUcsRUFBRSxDQUFDO2dCQUNoQixLQUFLLE1BQU0sRUFBRSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7b0JBQ3pDLE1BQU0sSUFBSSxHQUFHLEVBQUUsTUFBTSxDQUFDO29CQUN0QixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7b0JBQzNCLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTt3QkFDckQsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDO3dCQUNuQyxNQUFNLElBQUksR0FBRyxLQUFLLE1BQU0sQ0FBQztxQkFDMUI7aUJBQ0Y7Z0JBQ0QsT0FBTyxNQUFNLENBQUM7WUFDaEIsQ0FBQztRQUNILENBQUM7S0FDRjtJQUVPLGlCQUFpQixDQUFDLEtBQTZDO1FBQ3JFLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ2xDLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtZQUNwQixPQUFPLEVBQUUsZ0JBQWdCO1lBQ3pCLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFlBQVksRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUk7U0FDckMsQ0FBQyxDQUFDO0tBQ0o7SUFFTyxvQkFBb0IsQ0FBQyxLQUFnRDtRQUMzRSxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNsQyxNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDcEIsT0FBTyxFQUFFLGdCQUFnQjtZQUN6QixRQUFRLEVBQUUsVUFBVTtZQUNwQixZQUFZLEVBQUUsR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJO1NBQ3hDLENBQUMsQ0FBQztLQUNKOztBQXRJSCwwQ0F1SUM7QUE1SHlCLG9CQUFJLEdBQUcsc0NBQXNDLENBQUM7QUFxSXhFLFNBQWdCLGlCQUFpQixDQUFDLFlBQStDO0lBQy9FLElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtRQUM5QixPQUFPLFNBQVMsQ0FBQztLQUNsQjtTQUFNLElBQUksWUFBWSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDcEMsTUFBTSxVQUFVLEdBQUcsWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzNDLE9BQU8sQ0FBQyxVQUFVLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDO0tBQ3JEO1NBQU0sSUFBSSxZQUFZLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtRQUNsQyxPQUFPLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7S0FDL0I7SUFFRCxPQUFPLFNBQVMsQ0FBQztBQUNuQixDQUFDO0FBWEQsOENBV0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnLi4vLi4vLi4vLi4vYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnLi4vLi4vLi4vLi4vY29yZSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuLi8uLi8uLi8uLi9jb3JlJztcblxuLyoqXG4gKiBNYW5hZ2VzIGEgYnVuY2ggb2Ygc2luZ2xldG9uLXkgc3RhdGVtZW50cyBvbiB0aGUgcG9saWN5IG9mIGFuIElBTSBSb2xlLlxuICogRGVkaWNhdGVkIG1ldGhvZHMgY2FuIGJlIHVzZWQgdG8gYWRkIHNwZWNpZmljIHBlcm1pc3Npb25zIHRvIHRoZSByb2xlIHBvbGljeVxuICogdXNpbmcgYXMgZmV3IHN0YXRlbWVudHMgYXMgcG9zc2libGUgKGFkZGluZyByZXNvdXJjZXMgdG8gZXhpc3RpbmcgY29tcGF0aWJsZVxuICogc3RhdGVtZW50cyBpbnN0ZWFkIG9mIGFkZGluZyBuZXcgc3RhdGVtZW50cyB3aGVuZXZlciBwb3NzaWJsZSkuXG4gKlxuICogU3RhdGVtZW50cyBjcmVhdGVkIG91dHNpZGUgb2YgdGhpcyBjbGFzcyBhcmUgbm90IGNvbnNpZGVyZWQgd2hlbiBhZGRpbmcgbmV3XG4gKiBwZXJtaXNzaW9ucy5cbiAqL1xuZXhwb3J0IGNsYXNzIFNpbmdsZXRvblBvbGljeSBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIGlhbS5JR3JhbnRhYmxlIHtcbiAgLyoqXG4gICAqIE9idGFpbiBhIFNpbmdsZXRvblBvbGljeSBmb3IgYSBnaXZlbiByb2xlLlxuICAgKiBAcGFyYW0gcm9sZSB0aGUgUm9sZSB0aGlzIHBvbGljeSBpcyBib3VuZCB0by5cbiAgICogQHJldHVybnMgdGhlIFNpbmdsZXRvblBvbGljeSBmb3IgdGhpcyByb2xlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmb3JSb2xlKHJvbGU6IGlhbS5JUm9sZSk6IFNpbmdsZXRvblBvbGljeSB7XG4gICAgY29uc3QgZm91bmQgPSByb2xlLm5vZGUudHJ5RmluZENoaWxkKFNpbmdsZXRvblBvbGljeS5VVUlEKTtcbiAgICByZXR1cm4gKGZvdW5kIGFzIFNpbmdsZXRvblBvbGljeSkgfHwgbmV3IFNpbmdsZXRvblBvbGljeShyb2xlKTtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFVVSUQgPSAnODM4OWU3NWYtMDgxMC00ODM4LWJmNjQtZDZmODVhOTVjZjgzJztcblxuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuXG4gIHByaXZhdGUgc3RhdGVtZW50czogeyBba2V5OiBzdHJpbmddOiBpYW0uUG9saWN5U3RhdGVtZW50IH0gPSB7fTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcm9sZTogaWFtLklSb2xlKSB7XG4gICAgc3VwZXIocm9sZSBhcyB1bmtub3duIGFzIENvbnN0cnVjdCwgU2luZ2xldG9uUG9saWN5LlVVSUQpO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSByb2xlO1xuICB9XG5cbiAgcHVibGljIGdyYW50RXhlY3V0ZUNoYW5nZVNldChwcm9wczogeyBzdGFja05hbWU6IHN0cmluZywgY2hhbmdlU2V0TmFtZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcgfSk6IHZvaWQge1xuICAgIHRoaXMuc3RhdGVtZW50Rm9yKHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tzJyxcbiAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlQ2hhbmdlU2V0JyxcbiAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkV4ZWN1dGVDaGFuZ2VTZXQnLFxuICAgICAgXSxcbiAgICAgIGNvbmRpdGlvbnM6IHvCoFN0cmluZ0VxdWFsc0lmRXhpc3RzOiB7ICdjbG91ZGZvcm1hdGlvbjpDaGFuZ2VTZXROYW1lJzogcHJvcHMuY2hhbmdlU2V0TmFtZSB9IH0sXG4gICAgfSkuYWRkUmVzb3VyY2VzKHRoaXMuc3RhY2tBcm5Gcm9tUHJvcHMocHJvcHMpKTtcbiAgfVxuXG4gIHB1YmxpYyBncmFudENyZWF0ZVJlcGxhY2VDaGFuZ2VTZXQocHJvcHM6IHsgc3RhY2tOYW1lOiBzdHJpbmcsIGNoYW5nZVNldE5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nIH0pOiB2b2lkIHtcbiAgICB0aGlzLnN0YXRlbWVudEZvcih7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdjbG91ZGZvcm1hdGlvbjpDcmVhdGVDaGFuZ2VTZXQnLFxuICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVsZXRlQ2hhbmdlU2V0JyxcbiAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlQ2hhbmdlU2V0JyxcbiAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tzJyxcbiAgICAgIF0sXG4gICAgICBjb25kaXRpb25zOiB7IFN0cmluZ0VxdWFsc0lmRXhpc3RzOiB7ICdjbG91ZGZvcm1hdGlvbjpDaGFuZ2VTZXROYW1lJzogcHJvcHMuY2hhbmdlU2V0TmFtZSB9IH0sXG4gICAgfSkuYWRkUmVzb3VyY2VzKHRoaXMuc3RhY2tBcm5Gcm9tUHJvcHMocHJvcHMpKTtcbiAgfVxuXG4gIHB1YmxpYyBncmFudENyZWF0ZVVwZGF0ZVN0YWNrKHByb3BzOiB7IHN0YWNrTmFtZTogc3RyaW5nLCByZXBsYWNlT25GYWlsdXJlPzogYm9vbGVhbiwgcmVnaW9uPzogc3RyaW5nIH0pOiB2b2lkIHtcbiAgICBjb25zdCBhY3Rpb25zID0gW1xuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2sqJyxcbiAgICAgICdjbG91ZGZvcm1hdGlvbjpDcmVhdGVTdGFjaycsXG4gICAgICAnY2xvdWRmb3JtYXRpb246VXBkYXRlU3RhY2snLFxuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkdldFRlbXBsYXRlKicsXG4gICAgICAnY2xvdWRmb3JtYXRpb246VmFsaWRhdGVUZW1wbGF0ZScsXG4gICAgICAnY2xvdWRmb3JtYXRpb246R2V0U3RhY2tQb2xpY3knLFxuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOlNldFN0YWNrUG9saWN5JyxcbiAgICBdO1xuICAgIGlmIChwcm9wcy5yZXBsYWNlT25GYWlsdXJlKSB7XG4gICAgICBhY3Rpb25zLnB1c2goJ2Nsb3VkZm9ybWF0aW9uOkRlbGV0ZVN0YWNrJyk7XG4gICAgfVxuICAgIHRoaXMuc3RhdGVtZW50Rm9yKHsgYWN0aW9ucyB9KS5hZGRSZXNvdXJjZXModGhpcy5zdGFja0FybkZyb21Qcm9wcyhwcm9wcykpO1xuICB9XG5cbiAgcHVibGljIGdyYW50Q3JlYXRlVXBkYXRlU3RhY2tTZXQocHJvcHM6IHsgc3RhY2tTZXROYW1lOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZyB9KTogdm9pZCB7XG4gICAgY29uc3QgYWN0aW9ucyA9IFtcbiAgICAgICdjbG91ZGZvcm1hdGlvbjpDcmVhdGVTdGFja1NldCcsXG4gICAgICAnY2xvdWRmb3JtYXRpb246VXBkYXRlU3RhY2tTZXQnLFxuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tTZXQnLFxuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tTZXRPcGVyYXRpb24nLFxuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkxpc3RTdGFja0luc3RhbmNlcycsXG4gICAgICAnY2xvdWRmb3JtYXRpb246Q3JlYXRlU3RhY2tJbnN0YW5jZXMnLFxuICAgIF07XG4gICAgdGhpcy5zdGF0ZW1lbnRGb3IoeyBhY3Rpb25zIH0pLmFkZFJlc291cmNlcyh0aGlzLnN0YWNrU2V0QXJuRnJvbVByb3BzKHByb3BzKSk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnREZWxldGVTdGFjayhwcm9wczogeyBzdGFja05hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nIH0pOiB2b2lkIHtcbiAgICB0aGlzLnN0YXRlbWVudEZvcih7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrKicsXG4gICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZWxldGVTdGFjaycsXG4gICAgICBdLFxuICAgIH0pLmFkZFJlc291cmNlcyh0aGlzLnN0YWNrQXJuRnJvbVByb3BzKHByb3BzKSk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRQYXNzUm9sZShyb2xlOiBpYW0uSVJvbGUgfCBzdHJpbmcpOiB2b2lkIHtcbiAgICB0aGlzLnN0YXRlbWVudEZvcih7IGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10gfSkuYWRkUmVzb3VyY2VzKHR5cGVvZiByb2xlID09PSAnc3RyaW5nJyA/IHJvbGUgOiByb2xlLnJvbGVBcm4pO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0ZW1lbnRGb3IodGVtcGxhdGU6IFN0YXRlbWVudFRlbXBsYXRlKTogaWFtLlBvbGljeVN0YXRlbWVudCB7XG4gICAgY29uc3Qga2V5ID0ga2V5Rm9yKHRlbXBsYXRlKTtcbiAgICBpZiAoIShrZXkgaW4gdGhpcy5zdGF0ZW1lbnRzKSkge1xuICAgICAgdGhpcy5zdGF0ZW1lbnRzW2tleV0gPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7IGFjdGlvbnM6IHRlbXBsYXRlLmFjdGlvbnMgfSk7XG4gICAgICBpZiAodGVtcGxhdGUuY29uZGl0aW9ucykge1xuICAgICAgICB0aGlzLnN0YXRlbWVudHNba2V5XS5hZGRDb25kaXRpb25zKHRlbXBsYXRlLmNvbmRpdGlvbnMpO1xuICAgICAgfVxuICAgICAgdGhpcy5yb2xlLmFkZFRvUG9saWN5KHRoaXMuc3RhdGVtZW50c1trZXldKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuc3RhdGVtZW50c1trZXldO1xuXG4gICAgZnVuY3Rpb24ga2V5Rm9yKHByb3BzOiBTdGF0ZW1lbnRUZW1wbGF0ZSk6IHN0cmluZyB7XG4gICAgICBjb25zdCBhY3Rpb25zID0gYCR7cHJvcHMuYWN0aW9ucy5zb3J0KCkuam9pbignXFx4MUYnKX1gO1xuICAgICAgY29uc3QgY29uZGl0aW9ucyA9IGZvcm1hdENvbmRpdGlvbnMocHJvcHMuY29uZGl0aW9ucyk7XG4gICAgICByZXR1cm4gYCR7YWN0aW9uc31cXHgxRCR7Y29uZGl0aW9uc31gO1xuXG4gICAgICBmdW5jdGlvbiBmb3JtYXRDb25kaXRpb25zKGNvbmQ/OiBTdGF0ZW1lbnRDb25kaXRpb24pOiBzdHJpbmcge1xuICAgICAgICBpZiAoY29uZCA9PSBudWxsKSB7IHJldHVybiAnJzsgfVxuICAgICAgICBsZXQgcmVzdWx0ID0gJyc7XG4gICAgICAgIGZvciAoY29uc3Qgb3Agb2YgT2JqZWN0LmtleXMoY29uZCkuc29ydCgpKSB7XG4gICAgICAgICAgcmVzdWx0ICs9IGAke29wfVxceDFFYDtcbiAgICAgICAgICBjb25zdCBjb25kaXRpb24gPSBjb25kW29wXTtcbiAgICAgICAgICBmb3IgKGNvbnN0IGF0dHJpYnV0ZSBvZiBPYmplY3Qua2V5cyhjb25kaXRpb24pLnNvcnQoKSkge1xuICAgICAgICAgICAgY29uc3QgdmFsdWUgPSBjb25kaXRpb25bYXR0cmlidXRlXTtcbiAgICAgICAgICAgIHJlc3VsdCArPSBgJHt2YWx1ZX1cXHgxRmA7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXN1bHQ7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBzdGFja0FybkZyb21Qcm9wcyhwcm9wczogeyBzdGFja05hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nIH0pOiBzdHJpbmcge1xuICAgIHJldHVybiBjZGsuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgIHJlZ2lvbjogcHJvcHMucmVnaW9uLFxuICAgICAgc2VydmljZTogJ2Nsb3VkZm9ybWF0aW9uJyxcbiAgICAgIHJlc291cmNlOiAnc3RhY2snLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHtwcm9wcy5zdGFja05hbWV9LypgLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGFja1NldEFybkZyb21Qcm9wcyhwcm9wczogeyBzdGFja1NldE5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nIH0pOiBzdHJpbmcge1xuICAgIHJldHVybiBjZGsuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgIHJlZ2lvbjogcHJvcHMucmVnaW9uLFxuICAgICAgc2VydmljZTogJ2Nsb3VkZm9ybWF0aW9uJyxcbiAgICAgIHJlc291cmNlOiAnc3RhY2tzZXQnLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHtwcm9wcy5zdGFja1NldE5hbWV9OipgLFxuICAgIH0pO1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgU3RhdGVtZW50VGVtcGxhdGUge1xuICBhY3Rpb25zOiBzdHJpbmdbXTtcbiAgY29uZGl0aW9ucz86IFN0YXRlbWVudENvbmRpdGlvbjtcbn1cblxuZXhwb3J0IHR5cGUgU3RhdGVtZW50Q29uZGl0aW9uID0geyBbb3A6IHN0cmluZ106IHsgW2F0dHJpYnV0ZTogc3RyaW5nXTogc3RyaW5nIH0gfTtcblxuZXhwb3J0IGZ1bmN0aW9uIHBhcnNlQ2FwYWJpbGl0aWVzKGNhcGFiaWxpdGllczogY2RrLkNmbkNhcGFiaWxpdGllc1tdIHwgdW5kZWZpbmVkKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgaWYgKGNhcGFiaWxpdGllcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfSBlbHNlIGlmIChjYXBhYmlsaXRpZXMubGVuZ3RoID09PSAxKSB7XG4gICAgY29uc3QgY2FwYWJpbGl0eSA9IGNhcGFiaWxpdGllcy50b1N0cmluZygpO1xuICAgIHJldHVybiAoY2FwYWJpbGl0eSA9PT0gJycpID8gdW5kZWZpbmVkIDogY2FwYWJpbGl0eTtcbiAgfSBlbHNlIGlmIChjYXBhYmlsaXRpZXMubGVuZ3RoID4gMSkge1xuICAgIHJldHVybiBjYXBhYmlsaXRpZXMuam9pbignLCcpO1xuICB9XG5cbiAgcmV0dXJuIHVuZGVmaW5lZDtcbn0iXX0=