"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 */
class Source {
    constructor(props) {
        this.badgeSupported = false;
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_SourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Source);
            }
            throw error;
        }
        this.identifier = props.identifier;
    }
    static s3(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_S3SourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.s3);
            }
            throw error;
        }
        return new S3Source(props);
    }
    static codeCommit(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_CodeCommitSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.codeCommit);
            }
            throw error;
        }
        return new CodeCommitSource(props);
    }
    static gitHub(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_GitHubSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.gitHub);
            }
            throw error;
        }
        return new GitHubSource(props);
    }
    static gitHubEnterprise(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_GitHubEnterpriseSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.gitHubEnterprise);
            }
            throw error;
        }
        return new GitHubEnterpriseSource(props);
    }
    static bitBucket(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_BitBucketSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bitBucket);
            }
            throw error;
        }
        return new BitBucketSource(props);
    }
    /**
     * Called by the project when the source is added so that the source can perform
     * binding operations on the source. For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     */
    bind(_scope, _project) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(_scope);
            jsiiDeprecationWarnings.monocdk_aws_codebuild_IProject(_project);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "monocdk.aws_codebuild.Source", version: "1.185.0" };
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
        this.fetchSubmodules = props.fetchSubmodules;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
                gitSubmodulesConfig: this.fetchSubmodules ? {
                    fetchSubmodules: this.fetchSubmodules,
                } : undefined,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 */
var EventAction;
(function (EventAction) {
    /**
     * A push (of a branch, or a tag) to the repository.
     */
    EventAction["PUSH"] = "PUSH";
    /**
     * Creating a Pull Request.
     */
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    /**
     * Updating a Pull Request.
     */
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    /**
     * Merging a Pull Request.
     */
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    /**
     * Re-opening a previously closed Pull Request.
     * Note that this event is only supported for GitHub and GitHubEnterprise sources.
     */
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
var WebhookFilterTypes;
(function (WebhookFilterTypes) {
    WebhookFilterTypes["FILE_PATH"] = "FILE_PATH";
    WebhookFilterTypes["COMMIT_MESSAGE"] = "COMMIT_MESSAGE";
    WebhookFilterTypes["HEAD_REF"] = "HEAD_REF";
    WebhookFilterTypes["ACTOR_ACCOUNT_ID"] = "ACTOR_ACCOUNT_ID";
    WebhookFilterTypes["BASE_REF"] = "BASE_REF";
})(WebhookFilterTypes || (WebhookFilterTypes = {}));
/**
 * An object that represents a group of filter conditions for a webhook.
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on
     */
    static inEventOf(...actions) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codebuild_EventAction(actions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.inEventOf);
            }
            throw error;
        }
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression)
     */
    andCommitMessageIs(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression)
     */
    andCommitMessageIsNot(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub, Bitbucket and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must not affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub, Bitbucket and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addCommitMessageFilter(commitMessage, include) {
        return this.addFilter(WebhookFilterTypes.COMMIT_MESSAGE, commitMessage, include);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter(WebhookFilterTypes.HEAD_REF, refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter(WebhookFilterTypes.ACTOR_ACCOUNT_ID, accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter(WebhookFilterTypes.BASE_REF, refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(WebhookFilterTypes.FILE_PATH, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
_b = JSII_RTTI_SYMBOL_1;
FilterGroup[_b] = { fqn: "monocdk.aws_codebuild.FilterGroup", version: "1.185.0" };
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus ?? true;
        this.webhookFilters = props.webhookFilters || [];
        this.webhookTriggersBatchBuild = props.webhookTriggersBatchBuild;
        this.buildStatusUrl = props.buildStatusUrl;
    }
    bind(_scope, project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook ?? (anyFilterGroupsProvided ? true : undefined);
        if (!webhook && anyFilterGroupsProvided) {
            throw new Error('`webhookFilters` cannot be used when `webhook` is `false`');
        }
        if (!webhook && this.webhookTriggersBatchBuild) {
            throw new Error('`webhookTriggersBatchBuild` cannot be used when `webhook` is `false`');
        }
        const superConfig = super.bind(_scope, project);
        if (this.webhookTriggersBatchBuild) {
            project.enableBatchBuilds();
        }
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                buildType: this.webhookTriggersBatchBuild ? 'BUILD_BATCH' : undefined,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project, this.path);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
class CommonGithubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.buildStatusContext = props.buildStatusContext;
    }
    bind(scope, project) {
        const superConfig = super.bind(scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                buildStatusConfig: this.buildStatusContext !== undefined || this.buildStatusUrl !== undefined
                    ? {
                        context: this.buildStatusContext,
                        targetUrl: this.buildStatusUrl,
                    }
                    : undefined,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends CommonGithubSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends CommonGithubSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        if (this.hasCommitMessageFilterAndPrEvent()) {
            throw new Error('COMMIT_MESSAGE filters cannot be used with GitHub Enterprise Server pull request events');
        }
        if (this.hasFilePathFilterAndPrEvent()) {
            throw new Error('FILE_PATH filters cannot be used with GitHub Enterprise Server pull request events');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    hasCommitMessageFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.COMMIT_MESSAGE) &&
            this.hasPrEvent(fg._actions)));
    }
    hasFilePathFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.FILE_PATH) &&
            this.hasPrEvent(fg._actions)));
    }
    hasPrEvent(actions) {
        return actions.includes(EventAction.PULL_REQUEST_CREATED ||
            EventAction.PULL_REQUEST_MERGED ||
            EventAction.PULL_REQUEST_REOPENED ||
            EventAction.PULL_REQUEST_UPDATED);
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
        this.buildStatusName = props.buildStatusName;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                buildStatusConfig: this.buildStatusName !== undefined || this.buildStatusUrl !== undefined
                    ? {
                        context: this.buildStatusName,
                        targetUrl: this.buildStatusUrl,
                    }
                    : undefined,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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