"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataResourceType = exports.ManagementEventSources = exports.Trail = exports.ReadWriteType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const s3 = require("../../aws-s3");
const core_1 = require("../../core");
const cloudtrail_generated_1 = require("./cloudtrail.generated");
/**
 * Types of events that CloudTrail can log
 */
var ReadWriteType;
(function (ReadWriteType) {
    /**
     * Read-only events include API operations that read your resources,
     * but don't make changes.
     * For example, read-only events include the Amazon EC2 DescribeSecurityGroups
     * and DescribeSubnets API operations.
     */
    ReadWriteType["READ_ONLY"] = "ReadOnly";
    /**
     * Write-only events include API operations that modify (or might modify)
     * your resources.
     * For example, the Amazon EC2 RunInstances and TerminateInstances API
     * operations modify your instances.
     */
    ReadWriteType["WRITE_ONLY"] = "WriteOnly";
    /**
     * All events
     */
    ReadWriteType["ALL"] = "All";
    /**
     * No events
     */
    ReadWriteType["NONE"] = "None";
})(ReadWriteType = exports.ReadWriteType || (exports.ReadWriteType = {}));
/**
 * Cloud trail allows you to log events that happen in your AWS account
 * For example:
 *
 * import { CloudTrail } from '@aws-cdk/aws-cloudtrail'
 *
 * const cloudTrail = new CloudTrail(this, 'MyTrail');
 *
 * NOTE the above example creates an UNENCRYPTED bucket by default,
 * If you are required to use an Encrypted bucket you can supply a preconfigured bucket
 * via TrailProps
 *
 */
class Trail extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.trailName,
        });
        this.eventSelectors = [];
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_TrailProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Trail);
            }
            throw error;
        }
        const cloudTrailPrincipal = new iam.ServicePrincipal('cloudtrail.amazonaws.com');
        this.s3bucket = props.bucket || new s3.Bucket(this, 'S3', { encryption: s3.BucketEncryption.UNENCRYPTED, enforceSSL: true });
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.bucketArn],
            actions: ['s3:GetBucketAcl'],
            principals: [cloudTrailPrincipal],
        }));
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.arnForObjects(`${props.s3KeyPrefix ? `${props.s3KeyPrefix}/` : ''}AWSLogs/${core_1.Stack.of(this).account}/*`)],
            actions: ['s3:PutObject'],
            principals: [cloudTrailPrincipal],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        this.topic = props.snsTopic;
        if (this.topic) {
            this.topic.grantPublish(cloudTrailPrincipal);
        }
        let logsRole;
        if (props.sendToCloudWatchLogs) {
            if (props.cloudWatchLogGroup) {
                this.logGroup = props.cloudWatchLogGroup;
            }
            else {
                this.logGroup = new logs.LogGroup(this, 'LogGroup', {
                    retention: props.cloudWatchLogsRetention ?? logs.RetentionDays.ONE_YEAR,
                });
            }
            logsRole = new iam.Role(this, 'LogsRole', { assumedBy: cloudTrailPrincipal });
            logsRole.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: [this.logGroup.logGroupArn],
            }));
        }
        if (props.managementEvents) {
            let managementEvent;
            if (props.managementEvents === ReadWriteType.NONE) {
                managementEvent = {
                    includeManagementEvents: false,
                };
            }
            else {
                managementEvent = {
                    includeManagementEvents: true,
                    readWriteType: props.managementEvents,
                };
            }
            this.eventSelectors.push(managementEvent);
        }
        if (props.kmsKey && props.encryptionKey) {
            throw new Error('Both kmsKey and encryptionKey must not be specified. Use only encryptionKey');
        }
        // TODO: not all regions support validation. Use service configuration data to fail gracefully
        const trail = new cloudtrail_generated_1.CfnTrail(this, 'Resource', {
            isLogging: true,
            enableLogFileValidation: props.enableFileValidation == null ? true : props.enableFileValidation,
            isMultiRegionTrail: props.isMultiRegionTrail == null ? true : props.isMultiRegionTrail,
            includeGlobalServiceEvents: props.includeGlobalServiceEvents == null ? true : props.includeGlobalServiceEvents,
            trailName: this.physicalName,
            kmsKeyId: props.encryptionKey?.keyArn ?? props.kmsKey?.keyArn,
            s3BucketName: this.s3bucket.bucketName,
            s3KeyPrefix: props.s3KeyPrefix,
            cloudWatchLogsLogGroupArn: this.logGroup?.logGroupArn,
            cloudWatchLogsRoleArn: logsRole?.roleArn,
            snsTopicName: this.topic?.topicName,
            eventSelectors: this.eventSelectors,
        });
        this.trailArn = this.getResourceArnAttribute(trail.attrArn, {
            service: 'cloudtrail',
            resource: 'trail',
            resourceName: this.physicalName,
        });
        this.trailSnsTopicArn = trail.attrSnsTopicArn;
        // Add a dependency on the bucket policy being updated, CloudTrail will test this upon creation.
        if (this.s3bucket.policy) {
            trail.node.addDependency(this.s3bucket.policy);
        }
        // If props.sendToCloudWatchLogs is set to true then the trail needs to depend on the created logsRole
        // so that it can create the log stream for the log group. This ensures the logsRole is created and propagated
        // before the trail tries to create the log stream.
        if (logsRole !== undefined) {
            trail.node.addDependency(logsRole);
        }
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     */
    static onEvent(scope, id, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_events_OnEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onEvent);
            }
            throw error;
        }
        const rule = new events.Rule(scope, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            detailType: ['AWS API Call via CloudTrail'],
        });
        return rule;
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an Event Selector for filtering events that match either S3 or Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param dataResourceValues the list of data resource ARNs to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addEventSelector(dataResourceType, dataResourceValues, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_DataResourceType(dataResourceType);
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventSelector);
            }
            throw error;
        }
        if (dataResourceValues.length > 250) {
            throw new Error('A maximum of 250 data elements can be in one event selector');
        }
        if (this.eventSelectors.length > 5) {
            throw new Error('A maximum of 5 event selectors are supported per trail.');
        }
        this.eventSelectors.push({
            dataResources: [{
                    type: dataResourceType,
                    values: dataResourceValues,
                }],
            includeManagementEvents: options.includeManagementEvents,
            excludeManagementEventSources: options.excludeManagementEventSources,
            readWriteType: options.readWriteType,
        });
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds a Lambda Data Event Selector for filtering events that match Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param handlers the list of lambda function handlers whose data events should be logged (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addLambdaEventSelector(handlers, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLambdaEventSelector);
            }
            throw error;
        }
        if (handlers.length === 0) {
            return;
        }
        const dataResourceValues = handlers.map((h) => h.functionArn);
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, dataResourceValues, options);
    }
    /**
     * Log all Lamda data events for all lambda functions the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllLambdaDataEvents(options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.logAllLambdaDataEvents);
            }
            throw error;
        }
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, [`arn:${this.stack.partition}:lambda`], options);
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an S3 Data Event Selector for filtering events that match S3 operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param s3Selector the list of S3 bucket with optional prefix to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addS3EventSelector(s3Selector, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addS3EventSelector);
            }
            throw error;
        }
        if (s3Selector.length === 0) {
            return;
        }
        const dataResourceValues = s3Selector.map((sel) => `${sel.bucket.bucketArn}/${sel.objectPrefix ?? ''}`);
        return this.addEventSelector(DataResourceType.S3_OBJECT, dataResourceValues, options);
    }
    /**
     * Log all S3 data events for all objects for all buckets in the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllS3DataEvents(options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.logAllS3DataEvents);
            }
            throw error;
        }
        return this.addEventSelector(DataResourceType.S3_OBJECT, [`arn:${this.stack.partition}:s3:::`], options);
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @deprecated - use Trail.onEvent()
     */
    onCloudTrailEvent(id, options = {}) {
        try {
            jsiiDeprecationWarnings.print("monocdk.aws_cloudtrail.Trail#onCloudTrailEvent", "- use Trail.onEvent()");
            jsiiDeprecationWarnings.monocdk_aws_events_OnEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onCloudTrailEvent);
            }
            throw error;
        }
        return Trail.onEvent(this, id, options);
    }
}
exports.Trail = Trail;
_a = JSII_RTTI_SYMBOL_1;
Trail[_a] = { fqn: "monocdk.aws_cloudtrail.Trail", version: "1.185.0" };
/**
 * Types of management event sources that can be excluded
 */
var ManagementEventSources;
(function (ManagementEventSources) {
    /**
     * AWS Key Management Service (AWS KMS) events
     */
    ManagementEventSources["KMS"] = "kms.amazonaws.com";
    /**
     * Data API events
     */
    ManagementEventSources["RDS_DATA_API"] = "rdsdata.amazonaws.com";
})(ManagementEventSources = exports.ManagementEventSources || (exports.ManagementEventSources = {}));
/**
 * Resource type for a data event
 */
var DataResourceType;
(function (DataResourceType) {
    /**
     * Data resource type for Lambda function
     */
    DataResourceType["LAMBDA_FUNCTION"] = "AWS::Lambda::Function";
    /**
     * Data resource type for S3 objects
     */
    DataResourceType["S3_OBJECT"] = "AWS::S3::Object";
})(DataResourceType = exports.DataResourceType || (exports.DataResourceType = {}));
//# sourceMappingURL=data:application/json;base64,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