"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlan = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const backup_generated_1 = require("./backup.generated");
const rule_1 = require("./rule");
const selection_1 = require("./selection");
const vault_1 = require("./vault");
/**
 * A backup plan
 */
class BackupPlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.rules = [];
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_BackupPlanProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BackupPlan);
            }
            throw error;
        }
        const plan = new backup_generated_1.CfnBackupPlan(this, 'Resource', {
            backupPlan: {
                advancedBackupSettings: this.advancedBackupSettings(props),
                backupPlanName: props.backupPlanName || id,
                backupPlanRule: core_1.Lazy.any({ produce: () => this.rules }, { omitEmptyArray: true }),
            },
        });
        this.backupPlanId = plan.attrBackupPlanId;
        this.backupPlanArn = plan.attrBackupPlanArn;
        this.versionId = plan.attrVersionId;
        this._backupVault = props.backupVault;
        for (const rule of props.backupPlanRules || []) {
            this.addRule(rule);
        }
    }
    /**
     * Import an existing backup plan
     */
    static fromBackupPlanId(scope, id, backupPlanId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupPlanId = backupPlanId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Daily with 35 day retention
     */
    static daily35DayRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.daily35DayRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        return plan;
    }
    /**
     * Daily and monthly with 1 year retention
     */
    static dailyMonthly1YearRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dailyMonthly1YearRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.monthly1Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 5 year retention
     */
    static dailyWeeklyMonthly5YearRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dailyWeeklyMonthly5YearRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly5Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 7 year retention
     */
    static dailyWeeklyMonthly7YearRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dailyWeeklyMonthly7YearRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly7Year());
        return plan;
    }
    advancedBackupSettings(props) {
        if (!props.windowsVss) {
            return undefined;
        }
        return [{
                backupOptions: {
                    WindowsVSS: 'enabled',
                },
                resourceType: 'EC2',
            }];
    }
    /**
     * Adds a rule to a plan
     *
     * @param rule the rule to add
     */
    addRule(rule) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_BackupPlanRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRule);
            }
            throw error;
        }
        let vault;
        if (rule.props.backupVault) {
            vault = rule.props.backupVault;
        }
        else if (this._backupVault) {
            vault = this._backupVault;
        }
        else {
            this._backupVault = new vault_1.BackupVault(this, 'Vault');
            vault = this._backupVault;
        }
        this.rules.push({
            completionWindowMinutes: rule.props.completionWindow?.toMinutes(),
            lifecycle: (rule.props.deleteAfter || rule.props.moveToColdStorageAfter) && {
                deleteAfterDays: rule.props.deleteAfter?.toDays(),
                moveToColdStorageAfterDays: rule.props.moveToColdStorageAfter?.toDays(),
            },
            ruleName: rule.props.ruleName ?? `${this.node.id}Rule${this.rules.length}`,
            scheduleExpression: rule.props.scheduleExpression?.expressionString,
            startWindowMinutes: rule.props.startWindow?.toMinutes(),
            enableContinuousBackup: rule.props.enableContinuousBackup,
            targetBackupVault: vault.backupVaultName,
        });
    }
    /**
     * The backup vault where backups are stored if not defined at
     * the rule level
     */
    get backupVault() {
        if (!this._backupVault) {
            // This cannot happen but is here to make TypeScript happy
            throw new Error('No backup vault!');
        }
        return this._backupVault;
    }
    /**
     * Adds a selection to this plan
     */
    addSelection(id, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_BackupSelectionOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSelection);
            }
            throw error;
        }
        return new selection_1.BackupSelection(this, id, {
            backupPlan: this,
            ...options,
        });
    }
    validate() {
        if (this.rules.length === 0) {
            return ['A backup plan must have at least 1 rule.'];
        }
        return [];
    }
}
exports.BackupPlan = BackupPlan;
_a = JSII_RTTI_SYMBOL_1;
BackupPlan[_a] = { fqn: "monocdk.aws_backup.BackupPlan", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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