"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_common_1 = require("../../aws-autoscaling-common");
const cloudwatch = require("../../aws-cloudwatch");
const step_scaling_action_1 = require("./step-scaling-action");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../core");
/**
 * Define a acaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_StepScalingPolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, StepScalingPolicy);
            }
            throw error;
        }
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: props.metricAggregationType ?? aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                autoScalingGroup: props.autoScalingGroup,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: props.evaluationPeriods ?? 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: props.metricAggregationType ?? aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                autoScalingGroup: props.autoScalingGroup,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: props.evaluationPeriods ?? 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
_a = JSII_RTTI_SYMBOL_1;
StepScalingPolicy[_a] = { fqn: "monocdk.aws_autoscaling.StepScalingPolicy", version: "1.185.0" };
function aggregationTypeFromMetric(metric) {
    const statistic = metric.toMetricConfig().metricStat?.statistic;
    if (statistic === undefined) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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