"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionsIntegration = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const core_1 = require("../../../core");
const integration_1 = require("../integration");
const model_1 = require("../model");
const aws_1 = require("./aws");
/**
 * Options to integrate with various StepFunction API
 */
class StepFunctionsIntegration {
    /**
     * Integrates a Synchronous Express State Machine from AWS Step Functions to an API Gateway method.
     *
     * @example
     *
     *    const stateMachine = new stepfunctions.StateMachine(this, 'MyStateMachine', {
     *       stateMachineType: stepfunctions.StateMachineType.EXPRESS,
     *       definition: stepfunctions.Chain.start(new stepfunctions.Pass(this, 'Pass')),
     *    });
     *
     *    const api = new apigateway.RestApi(this, 'Api', {
     *       restApiName: 'MyApi',
     *    });
     *    api.root.addMethod('GET', apigateway.StepFunctionsIntegration.startExecution(stateMachine));
     */
    static startExecution(stateMachine, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_stepfunctions_IStateMachine(stateMachine);
            jsiiDeprecationWarnings.monocdk_aws_apigateway_StepFunctionsExecutionIntegrationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.startExecution);
            }
            throw error;
        }
        return new StepFunctionsExecutionIntegration(stateMachine, options);
    }
}
exports.StepFunctionsIntegration = StepFunctionsIntegration;
_a = JSII_RTTI_SYMBOL_1;
StepFunctionsIntegration[_a] = { fqn: "monocdk.aws_apigateway.StepFunctionsIntegration", version: "1.185.0" };
class StepFunctionsExecutionIntegration extends aws_1.AwsIntegration {
    constructor(stateMachine, options = {}) {
        super({
            service: 'states',
            action: 'StartSyncExecution',
            options: {
                credentialsRole: options.credentialsRole,
                integrationResponses: integrationResponse(),
                passthroughBehavior: integration_1.PassthroughBehavior.NEVER,
                requestTemplates: requestTemplates(stateMachine, options),
                ...options,
            },
        });
        this.stateMachine = stateMachine;
    }
    bind(method) {
        const bindResult = super.bind(method);
        const credentialsRole = bindResult.options?.credentialsRole ?? new iam.Role(method, 'StartSyncExecutionRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        });
        this.stateMachine.grantStartSyncExecution(credentialsRole);
        let stateMachineName;
        if (this.stateMachine instanceof sfn.StateMachine) {
            const stateMachineType = this.stateMachine.stateMachineType;
            if (stateMachineType !== sfn.StateMachineType.EXPRESS) {
                throw new Error('State Machine must be of type "EXPRESS". Please use StateMachineType.EXPRESS as the stateMachineType');
            }
            //if not imported, extract the name from the CFN layer to reach the
            //literal value if it is given (rather than a token)
            stateMachineName = this.stateMachine.node.defaultChild.stateMachineName;
        }
        else {
            //imported state machine
            stateMachineName = `StateMachine-${this.stateMachine.stack.node.addr}`;
        }
        let deploymentToken;
        if (stateMachineName !== undefined && !core_1.Token.isUnresolved(stateMachineName)) {
            deploymentToken = JSON.stringify({ stateMachineName });
        }
        for (const methodResponse of METHOD_RESPONSES) {
            method.addMethodResponse(methodResponse);
        }
        return {
            ...bindResult,
            options: {
                ...bindResult.options,
                credentialsRole,
            },
            deploymentToken,
        };
    }
}
/**
 * Defines the integration response that passes the result on success,
 * or the error on failure, from the synchronous execution to the caller.
 *
 * @returns integrationResponse mapping
 */
function integrationResponse() {
    const errorResponse = [
        {
            /**
             * Specifies the regular expression (regex) pattern used to choose
             * an integration response based on the response from the back end.
             * In this case it will match all '4XX' HTTP Errors
             */
            selectionPattern: '4\\d{2}',
            statusCode: '400',
            responseTemplates: {
                'application/json': `{
            "error": "Bad request!"
          }`,
            },
        },
        {
            /**
             * Match all '5XX' HTTP Errors
             */
            selectionPattern: '5\\d{2}',
            statusCode: '500',
            responseTemplates: {
                'application/json': '"error": $input.path(\'$.error\')',
            },
        },
    ];
    const integResponse = [
        {
            statusCode: '200',
            responseTemplates: {
                /* eslint-disable */
                'application/json': [
                    '#set($inputRoot = $input.path(\'$\'))',
                    '#if($input.path(\'$.status\').toString().equals("FAILED"))',
                    '#set($context.responseOverride.status = 500)',
                    '{',
                    '"error": "$input.path(\'$.error\')",',
                    '"cause": "$input.path(\'$.cause\')"',
                    '}',
                    '#else',
                    '$input.path(\'$.output\')',
                    '#end',
                ].join('\n'),
            },
        },
        ...errorResponse,
    ];
    return integResponse;
}
/**
 * Defines the request template that will be used for the integration
 * @param stateMachine
 * @param options
 * @returns requestTemplate
 */
function requestTemplates(stateMachine, options) {
    const templateStr = templateString(stateMachine, options);
    const requestTemplate = {
        'application/json': templateStr,
    };
    return requestTemplate;
}
/**
 * Reads the VTL template and returns the template string to be used
 * for the request template.
 *
 * @param stateMachine
 * @param includeRequestContext
 * @param options
 * @reutrns templateString
 */
function templateString(stateMachine, options) {
    let templateStr;
    let requestContextStr = '';
    const includeHeader = options.headers ?? false;
    const includeQueryString = options.querystring ?? true;
    const includePath = options.path ?? true;
    const includeAuthorizer = options.authorizer ?? false;
    if (options.requestContext && Object.keys(options.requestContext).length > 0) {
        requestContextStr = requestContext(options.requestContext);
    }
    templateStr = fs.readFileSync(path.join(__dirname, 'stepfunctions.vtl'), { encoding: 'utf-8' });
    templateStr = templateStr.replace('%STATEMACHINE%', stateMachine.stateMachineArn);
    templateStr = templateStr.replace('%INCLUDE_HEADERS%', String(includeHeader));
    templateStr = templateStr.replace('%INCLUDE_QUERYSTRING%', String(includeQueryString));
    templateStr = templateStr.replace('%INCLUDE_PATH%', String(includePath));
    templateStr = templateStr.replace('%INCLUDE_AUTHORIZER%', String(includeAuthorizer));
    templateStr = templateStr.replace('%REQUESTCONTEXT%', requestContextStr);
    return templateStr;
}
function requestContext(requestContextObj) {
    const context = {
        accountId: requestContextObj?.accountId ? '$context.identity.accountId' : undefined,
        apiId: requestContextObj?.apiId ? '$context.apiId' : undefined,
        apiKey: requestContextObj?.apiKey ? '$context.identity.apiKey' : undefined,
        authorizerPrincipalId: requestContextObj?.authorizerPrincipalId ? '$context.authorizer.principalId' : undefined,
        caller: requestContextObj?.caller ? '$context.identity.caller' : undefined,
        cognitoAuthenticationProvider: requestContextObj?.cognitoAuthenticationProvider ? '$context.identity.cognitoAuthenticationProvider' : undefined,
        cognitoAuthenticationType: requestContextObj?.cognitoAuthenticationType ? '$context.identity.cognitoAuthenticationType' : undefined,
        cognitoIdentityId: requestContextObj?.cognitoIdentityId ? '$context.identity.cognitoIdentityId' : undefined,
        cognitoIdentityPoolId: requestContextObj?.cognitoIdentityPoolId ? '$context.identity.cognitoIdentityPoolId' : undefined,
        httpMethod: requestContextObj?.httpMethod ? '$context.httpMethod' : undefined,
        stage: requestContextObj?.stage ? '$context.stage' : undefined,
        sourceIp: requestContextObj?.sourceIp ? '$context.identity.sourceIp' : undefined,
        user: requestContextObj?.user ? '$context.identity.user' : undefined,
        userAgent: requestContextObj?.userAgent ? '$context.identity.userAgent' : undefined,
        userArn: requestContextObj?.userArn ? '$context.identity.userArn' : undefined,
        requestId: requestContextObj?.requestId ? '$context.requestId' : undefined,
        resourceId: requestContextObj?.resourceId ? '$context.resourceId' : undefined,
        resourcePath: requestContextObj?.resourcePath ? '$context.resourcePath' : undefined,
    };
    const contextAsString = JSON.stringify(context);
    // The VTL Template conflicts with double-quotes (") for strings.
    // Before sending to the template, we replace double-quotes (") with @@ and replace it back inside the .vtl file
    const doublequotes = '"';
    const replaceWith = '@@';
    return contextAsString.split(doublequotes).join(replaceWith);
}
/**
 * Method response model for each HTTP code response
 */
const METHOD_RESPONSES = [
    {
        statusCode: '200',
        responseModels: {
            'application/json': model_1.Model.EMPTY_MODEL,
        },
    },
    {
        statusCode: '400',
        responseModels: {
            'application/json': model_1.Model.ERROR_MODEL,
        },
    },
    {
        statusCode: '500',
        responseModels: {
            'application/json': model_1.Model.ERROR_MODEL,
        },
    },
];
//# sourceMappingURL=data:application/json;base64,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