"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Template = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("../../core");
const fs = require("fs-extra");
const match_1 = require("./match");
const matcher_1 = require("./matcher");
const conditions_1 = require("./private/conditions");
const cyclic_1 = require("./private/cyclic");
const mappings_1 = require("./private/mappings");
const outputs_1 = require("./private/outputs");
const parameters_1 = require("./private/parameters");
const resources_1 = require("./private/resources");
/**
 * Suite of assertions that can be run on a CDK stack.
 * Typically used, as part of unit tests, to validate that the rendered
 * CloudFormation template has expected resources and properties.
 */
class Template {
    constructor(template) {
        this.template = template;
        cyclic_1.checkTemplateForCyclicDependencies(this.template);
    }
    /**
     * Base your assertions on the CloudFormation template synthesized by a CDK `Stack`.
     * @param stack the CDK Stack to run assertions on
     */
    static fromStack(stack) {
        try {
            jsiiDeprecationWarnings.monocdk_Stack(stack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromStack);
            }
            throw error;
        }
        return new Template(toTemplate(stack));
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as an in-memory
     * JSON object.
     * @param template the CloudFormation template formatted as a nested set of records
     */
    static fromJSON(template) {
        return new Template(template);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as a
     * JSON string.
     * @param template the CloudFormation template in
     */
    static fromString(template) {
        return new Template(JSON.parse(template));
    }
    /**
     * The CloudFormation template deserialized into an object.
     */
    toJSON() {
        return this.template;
    }
    /**
     * Assert that the given number of resources of the given type exist in the
     * template.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param count number of expected instances
     */
    resourceCountIs(type, count) {
        const counted = resources_1.countResources(this.template, type);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that a resource of the given type and properties exists in the
     * CloudFormation template.
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavour, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the 'Properties' section of the resource as should be expected in the template.
     */
    hasResourceProperties(type, props) {
        const matchError = resources_1.hasResourceProperties(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a resource of the given type and given definition exists in the
     * CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the entire defintion of the resource as should be expected in the template.
     */
    hasResource(type, props) {
        const matchError = resources_1.hasResource(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching resources of a given type and properties in the CloudFormation template.
     * @param type the type to match in the CloudFormation template
     * @param props by default, matches all resources with the given type.
     * When a literal is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findResources(type, props = {}) {
        return resources_1.findResources(this.template, type, props);
    }
    /**
     * Assert that a Parameter with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the parameter, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props the parameter as should be expected in the template.
     */
    hasParameter(logicalId, props) {
        const matchError = parameters_1.hasParameter(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Parameters that match the given properties in the CloudFormation template.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props by default, matches all Parameters in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findParameters(logicalId, props = {}) {
        return parameters_1.findParameters(this.template, logicalId, props);
    }
    /**
     * Assert that an Output with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props the output as should be expected in the template.
     */
    hasOutput(logicalId, props) {
        const matchError = outputs_1.hasOutput(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Outputs that match the given properties in the CloudFormation template.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props by default, matches all Outputs in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findOutputs(logicalId, props = {}) {
        return outputs_1.findOutputs(this.template, logicalId, props);
    }
    /**
     * Assert that a Mapping with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props the output as should be expected in the template.
     */
    hasMapping(logicalId, props) {
        const matchError = mappings_1.hasMapping(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Mappings that match the given properties in the CloudFormation template.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props by default, matches all Mappings in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findMappings(logicalId, props = {}) {
        return mappings_1.findMappings(this.template, logicalId, props);
    }
    /**
     * Assert that a Condition with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all conditions in the template.
     * @param props the output as should be expected in the template.
     */
    hasCondition(logicalId, props) {
        const matchError = conditions_1.hasCondition(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Conditions that match the given properties in the CloudFormation template.
     * @param logicalId the name of the condition. Provide `'*'` to match all conditions in the template.
     * @param props by default, matches all Conditions in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findConditions(logicalId, props = {}) {
        return conditions_1.findConditions(this.template, logicalId, props);
    }
    /**
     * Assert that the CloudFormation template matches the given value
     * @param expected the expected CloudFormation template as key-value pairs.
     */
    templateMatches(expected) {
        const matcher = matcher_1.Matcher.isMatcher(expected) ? expected : match_1.Match.objectLike(expected);
        const result = matcher.test(this.template);
        if (result.hasFailed()) {
            throw new Error([
                'Template did not match as expected. The following mismatches were found:',
                ...result.toHumanStrings().map(s => `\t${s}`),
            ].join('\n'));
        }
    }
}
exports.Template = Template;
_a = JSII_RTTI_SYMBOL_1;
Template[_a] = { fqn: "monocdk.assertions.Template", version: "1.185.0" };
function toTemplate(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    const assembly = root.synth();
    if (stack.nestedStackParent) {
        // if this is a nested stack (it has a parent), then just read the template as a string
        return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
    }
    return assembly.getStackArtifact(stack.artifactId).template;
}
//# sourceMappingURL=data:application/json;base64,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