'''
# AWS App Mesh Construct Library

AWS App Mesh is a service mesh based on the [Envoy](https://www.envoyproxy.io/) proxy that makes it easy to monitor and control microservices. App Mesh standardizes how your microservices communicate, giving you end-to-end visibility and helping to ensure high-availability for your applications.

App Mesh gives you consistent visibility and network traffic controls for every microservice in an application.

App Mesh supports microservice applications that use service discovery naming for their components. To use App Mesh, you must have an existing application running on AWS Fargate, Amazon ECS, Amazon EKS, Kubernetes on AWS, or Amazon EC2.

For further information on **AWS App Mesh**, visit the [AWS App Mesh Documentation](https://docs.aws.amazon.com/app-mesh/index.html).

## Create the App and Stack

```python
app = cdk.App()
stack = cdk.Stack(app, "stack")
```

## Creating the Mesh

A service mesh is a logical boundary for network traffic between the services that reside within it.

After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.

The following example creates the `AppMesh` service mesh with the default egress filter of `DROP_ALL`. See [the AWS CloudFormation `EgressFilter` resource](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html) for more info on egress filters.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh"
)
```

The mesh can instead be created with the `ALLOW_ALL` egress filter by providing the `egressFilter` property.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh",
    egress_filter=appmesh.MeshFilterType.ALLOW_ALL
)
```

## Adding VirtualRouters

A *mesh* uses  *virtual routers* as logical units to route requests to *virtual nodes*.

Virtual routers handle traffic for one or more virtual services within your mesh.
After you create a virtual router, you can create and associate routes to your virtual router that direct incoming requests to different virtual nodes.

```python
# mesh: appmesh.Mesh

router = mesh.add_virtual_router("router",
    listeners=[appmesh.VirtualRouterListener.http(8080)]
)
```

Note that creating the router using the `addVirtualRouter()` method places it in the same stack as the mesh
(which might be different from the current stack).
The router can also be created using the `VirtualRouter` constructor (passing in the mesh) instead of calling the `addVirtualRouter()` method.
This is particularly useful when splitting your resources between many stacks: for example, defining the mesh itself as part of an infrastructure stack, but defining the other resources, such as routers, in the application stack:

```python
# infra_stack: cdk.Stack
# app_stack: cdk.Stack


mesh = appmesh.Mesh(infra_stack, "AppMesh",
    mesh_name="myAwsMesh",
    egress_filter=appmesh.MeshFilterType.ALLOW_ALL
)

# the VirtualRouter will belong to 'appStack',
# even though the Mesh belongs to 'infraStack'
router = appmesh.VirtualRouter(app_stack, "router",
    mesh=mesh,  # notice that mesh is a required property when creating a router with the 'new' statement
    listeners=[appmesh.VirtualRouterListener.http(8081)]
)
```

The same is true for other `add*()` methods in the App Mesh construct library.

The `VirtualRouterListener` class lets you define protocol-specific listeners.
The `http()`, `http2()`, `grpc()` and `tcp()` methods create listeners for the named protocols.
They accept a single parameter that defines the port to on which requests will be matched.
The port parameter defaults to 8080 if omitted.

## Adding a VirtualService

A *virtual service* is an abstraction of a real service that is provided by a virtual node directly, or indirectly by means of a virtual router. Dependent services call your virtual service by its `virtualServiceName`, and those requests are routed to the virtual node or virtual router specified as the provider for the virtual service.

We recommend that you use the service discovery name of the real service that you're targeting (such as `my-service.default.svc.cluster.local`).

When creating a virtual service:

* If you want the virtual service to spread traffic across multiple virtual nodes, specify a virtual router.
* If you want the virtual service to reach a virtual node directly, without a virtual router, specify a virtual node.

Adding a virtual router as the provider:

```python
# router: appmesh.VirtualRouter


appmesh.VirtualService(self, "virtual-service",
    virtual_service_name="my-service.default.svc.cluster.local",  # optional
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router)
)
```

Adding a virtual node as the provider:

```python
# node: appmesh.VirtualNode


appmesh.VirtualService(self, "virtual-service",
    virtual_service_name="my-service.default.svc.cluster.local",  # optional
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_node(node)
)
```

## Adding a VirtualNode

A *virtual node* acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment.

When you create a virtual node, accept inbound traffic by specifying a *listener*. Outbound traffic that your virtual node expects to send should be specified as a *back end*.

The response metadata for your new virtual node contains the Amazon Resource Name (ARN) that is associated with the virtual node. Set this value (either the full ARN or the truncated resource name) as the `APPMESH_VIRTUAL_NODE_NAME` environment variable for your task group's Envoy proxy container in your task definition or pod spec. For example, the value could be `mesh/default/virtualNode/simpleapp`. This is then mapped to the `node.id` and `node.cluster` Envoy parameters.

> **Note**
> If you require your Envoy stats or tracing to use a different name, you can override the `node.cluster` value that is set by `APPMESH_VIRTUAL_NODE_NAME` with the `APPMESH_VIRTUAL_NODE_CLUSTER` environment variable.

```python
# mesh: appmesh.Mesh
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")
node = mesh.add_virtual_node("virtual-node",
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8081,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),  # minimum
            path="/health-check-path",
            timeout=cdk.Duration.seconds(2),  # minimum
            unhealthy_threshold=2
        )
    )],
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)
```

Create a `VirtualNode` with the constructor and add tags.

```python
# mesh: appmesh.Mesh
# service: cloudmap.Service


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),
            path="/ping",
            timeout=cdk.Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=cdk.aws_appmesh.HttpTimeout(
            idle=cdk.Duration.seconds(5)
        )
    )],
    backend_defaults=cdk.aws_appmesh.BackendDefaults(
        tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
            validation=cdk.aws_appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)

cdk.Tags.of(node).add("Environment", "Dev")
```

Create a `VirtualNode` with the constructor and add backend virtual service.

```python
# mesh: appmesh.Mesh
# router: appmesh.VirtualRouter
# service: cloudmap.Service


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),
            path="/ping",
            timeout=cdk.Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=cdk.aws_appmesh.HttpTimeout(
            idle=cdk.Duration.seconds(5)
        )
    )],
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)

virtual_service = appmesh.VirtualService(self, "service-1",
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router),
    virtual_service_name="service1.domain.local"
)

node.add_backend(appmesh.Backend.virtual_service(virtual_service))
```

The `listeners` property can be left blank and added later with the `node.addListener()` method. The `serviceDiscovery` property must be specified when specifying a listener.

The `backends` property can be added with `node.addBackend()`. In the example, we define a virtual service and add it to the virtual node to allow egress traffic to other nodes.

The `backendDefaults` property is added to the node while creating the virtual node. These are the virtual node's default settings for all backends.

The `VirtualNode.addBackend()` method is especially useful if you want to create a circular traffic flow by having a Virtual Service as a backend whose provider is that same Virtual Node:

```python
# mesh: appmesh.Mesh


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node")
)

virtual_service = appmesh.VirtualService(self, "service-1",
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_node(node),
    virtual_service_name="service1.domain.local"
)

node.add_backend(appmesh.Backend.virtual_service(virtual_service))
```

### Adding TLS to a listener

The `tls` property specifies TLS configuration when creating a listener for a virtual node or a virtual gateway.
Provide the TLS certificate to the proxy in one of the following ways:

* A certificate from AWS Certificate Manager (ACM).
* A customer-provided certificate (specify a `certificateChain` path file and a `privateKey` file path).
* A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its `secretName`).

```python
# A Virtual Node with listener TLS from an ACM provided certificate
# cert: certificatemanager.Certificate
# mesh: appmesh.Mesh


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node"),
    listeners=[appmesh.VirtualNodeListener.grpc(
        port=80,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.acm(cert)
        )
    )]
)

# A Virtual Gateway with listener TLS from a customer provided file certificate
gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.grpc(
        port=8080,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
        )
    )],
    virtual_gateway_name="gateway"
)

# A Virtual Gateway with listener TLS from a SDS provided certificate
gateway2 = appmesh.VirtualGateway(self, "gateway2",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.http2(
        port=8080,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.sds("secrete_certificate")
        )
    )],
    virtual_gateway_name="gateway2"
)
```

### Adding mutual TLS authentication

Mutual TLS authentication is an optional component of TLS that offers two-way peer authentication.
To enable mutual TLS authentication, add the `mutualTlsCertificate` property to TLS client policy and/or the `mutualTlsValidation` property to your TLS listener.

`tls.mutualTlsValidation` and `tlsClientPolicy.mutualTlsCertificate` can be sourced from either:

* A customer-provided certificate (specify a `certificateChain` path file and a `privateKey` file path).
* A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its `secretName`).

> **Note**
> Currently, a certificate from AWS Certificate Manager (ACM) cannot be used for mutual TLS authentication.

```python
# mesh: appmesh.Mesh


node1 = appmesh.VirtualNode(self, "node1",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node"),
    listeners=[appmesh.VirtualNodeListener.grpc(
        port=80,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey"),
            # Validate a file client certificates to enable mutual TLS authentication when a client provides a certificate.
            mutual_tls_validation=cdk.aws_appmesh.MutualTlsValidation(
                trust=appmesh.TlsValidationTrust.file("path-to-certificate")
            )
        )
    )]
)

certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"
node2 = appmesh.VirtualNode(self, "node2",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node2"),
    backend_defaults=cdk.aws_appmesh.BackendDefaults(
        tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
            ports=[8080, 8081],
            validation=cdk.aws_appmesh.TlsValidation(
                subject_alternative_names=appmesh.SubjectAlternativeNames.matching_exactly("mesh-endpoint.apps.local"),
                trust=appmesh.TlsValidationTrust.acm([
                    acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                ])
            ),
            # Provide a SDS client certificate when a server requests it and enable mutual TLS authentication.
            mutual_tls_certificate=appmesh.TlsCertificate.sds("secret_certificate")
        )
    )
)
```

### Adding outlier detection to a Virtual Node listener

The `outlierDetection` property adds outlier detection to a Virtual Node listener. The properties
`baseEjectionDuration`, `interval`, `maxEjectionPercent`, and `maxServerErrors` are required.

```python
# mesh: appmesh.Mesh
# Cloud Map service discovery is currently required for host ejection by outlier detection
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")
node = mesh.add_virtual_node("virtual-node",
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        outlier_detection=cdk.aws_appmesh.OutlierDetection(
            base_ejection_duration=cdk.Duration.seconds(10),
            interval=cdk.Duration.seconds(30),
            max_ejection_percent=50,
            max_server_errors=5
        )
    )]
)
```

### Adding a connection pool to a listener

The `connectionPool` property can be added to a Virtual Node listener or Virtual Gateway listener to add a request connection pool. Each listener protocol type has its own connection pool properties.

```python
# A Virtual Node with a gRPC listener with a connection pool set
# mesh: appmesh.Mesh

node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
    # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
    # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
    # By default, the response type is assumed to be LOAD_BALANCER
    service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
    listeners=[appmesh.VirtualNodeListener.http(
        port=80,
        connection_pool=cdk.aws_appmesh.HttpConnectionPool(
            max_connections=100,
            max_pending_requests=10
        )
    )]
)

# A Virtual Gateway with a gRPC listener with a connection pool set
gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.grpc(
        port=8080,
        connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
            max_requests=10
        )
    )],
    virtual_gateway_name="gateway"
)
```

## Adding a Route

A *route* matches requests with an associated virtual router and distributes traffic to its associated virtual nodes.
The route distributes matching requests to one or more target virtual nodes with relative weighting.

The `RouteSpec` class lets you define protocol-specific route specifications.
The `tcp()`, `http()`, `http2()`, and `grpc()` methods create a specification for the named protocols.

For HTTP-based routes, the match field can match on path (prefix, exact, or regex), HTTP method, scheme,
HTTP headers, and query parameters. By default, HTTP-based routes match all requests.

For gRPC-based routes, the match field can  match on service name, method name, and metadata.
When specifying the method name, the service name must also be specified.

For example, here's how to add an HTTP route that matches based on a prefix of the URL path:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.http(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=cdk.aws_appmesh.HttpRouteMatch(
            # Path that is passed to this method must start with '/'.
            path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
        )
    )
)
```

Add an HTTP2 route that matches based on exact path, method, scheme, headers, and query parameters:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http2",
    route_spec=appmesh.RouteSpec.http2(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=cdk.aws_appmesh.HttpRouteMatch(
            path=appmesh.HttpRoutePathMatch.exactly("/exact"),
            method=appmesh.HttpRouteMethod.POST,
            protocol=appmesh.HttpRouteProtocol.HTTPS,
            headers=[
                # All specified headers must match for the route to match.
                appmesh.HeaderMatch.value_is("Content-Type", "application/json"),
                appmesh.HeaderMatch.value_is_not("Content-Type", "application/json")
            ],
            query_parameters=[
                # All specified query parameters must match for the route to match.
                appmesh.QueryParameterMatch.value_is("query-field", "value")
            ]
        )
    )
)
```

Add a single route with two targets and split traffic 50/50:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.http(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node,
            weight=50
        ), cdk.aws_appmesh.WeightedTarget(
            virtual_node=node,
            weight=50
        )
        ],
        match=cdk.aws_appmesh.HttpRouteMatch(
            path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
        )
    )
)
```

Add an http2 route with retries:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http2-retry",
    route_spec=appmesh.RouteSpec.http2(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
        retry_policy=cdk.aws_appmesh.HttpRetryPolicy(
            # Retry if the connection failed
            tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
            # Retry if HTTP responds with a gateway error (502, 503, 504)
            http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
            # Retry five times
            retry_attempts=5,
            # Use a 1 second timeout per retry
            retry_timeout=cdk.Duration.seconds(1)
        )
    )
)
```

Add a gRPC route with retries:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-retry",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
        match=cdk.aws_appmesh.GrpcRouteMatch(service_name="servicename"),
        retry_policy=cdk.aws_appmesh.GrpcRetryPolicy(
            tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
            http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
            # Retry if gRPC responds that the request was cancelled, a resource
            # was exhausted, or if the service is unavailable
            grpc_retry_events=[appmesh.GrpcRetryEvent.CANCELLED, appmesh.GrpcRetryEvent.RESOURCE_EXHAUSTED, appmesh.GrpcRetryEvent.UNAVAILABLE
            ],
            retry_attempts=5,
            retry_timeout=cdk.Duration.seconds(1)
        )
    )
)
```

Add an gRPC route that matches based on method name and metadata:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-retry",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
        match=cdk.aws_appmesh.GrpcRouteMatch(
            # When method name is specified, service name must be also specified.
            method_name="methodname",
            service_name="servicename",
            metadata=[
                # All specified metadata must match for the route to match.
                appmesh.HeaderMatch.value_starts_with("Content-Type", "application/"),
                appmesh.HeaderMatch.value_does_not_start_with("Content-Type", "text/")
            ]
        )
    )
)
```

Add a gRPC route with timeout:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=cdk.aws_appmesh.GrpcRouteMatch(
            service_name="my-service.default.svc.cluster.local"
        ),
        timeout=cdk.aws_appmesh.GrpcTimeout(
            idle=cdk.Duration.seconds(2),
            per_request=cdk.Duration.seconds(1)
        )
    )
)
```

## Adding a Virtual Gateway

A *virtual gateway* allows resources outside your mesh to communicate with resources inside your mesh.
The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance.
Unlike a virtual node, which represents Envoy running with an application, a virtual gateway represents Envoy deployed by itself.

A virtual gateway is similar to a virtual node in that it has a listener that accepts traffic for a particular port and protocol (HTTP, HTTP2, gRPC).
Traffic received by the virtual gateway is directed to other services in your mesh
using rules defined in gateway routes which can be added to your virtual gateway.

Create a virtual gateway with the constructor:

```python
# mesh: appmesh.Mesh

certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"

gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.http(
        port=443,
        health_check=appmesh.HealthCheck.http(
            interval=cdk.Duration.seconds(10)
        )
    )],
    backend_defaults=cdk.aws_appmesh.BackendDefaults(
        tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
            ports=[8080, 8081],
            validation=cdk.aws_appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.acm([
                    acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                ])
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
    virtual_gateway_name="virtualGateway"
)
```

Add a virtual gateway directly to the mesh:

```python
# mesh: appmesh.Mesh


gateway = mesh.add_virtual_gateway("gateway",
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
    virtual_gateway_name="virtualGateway",
    listeners=[appmesh.VirtualGatewayListener.http(
        port=443,
        health_check=appmesh.HealthCheck.http(
            interval=cdk.Duration.seconds(10)
        )
    )]
)
```

The `listeners` field defaults to an HTTP Listener on port 8080 if omitted.
A gateway route can be added using the `gateway.addGatewayRoute()` method.

The `backendDefaults` property, provided when creating the virtual gateway, specifies the virtual gateway's default settings for all backends.

## Adding a Gateway Route

A *gateway route* is attached to a virtual gateway and routes matching traffic to an existing virtual service.

For HTTP-based gateway routes, the `match` field can be used to match on
path (prefix, exact, or regex), HTTP method, host name, HTTP headers, and query parameters.
By default, HTTP-based gateway routes match all requests.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            path=appmesh.HttpGatewayRoutePathMatch.regex("regex")
        )
    )
)
```

For gRPC-based gateway routes, the `match` field can be used to match on service name, host name, and metadata.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-grpc",
    route_spec=appmesh.GatewayRouteSpec.grpc(
        route_target=virtual_service,
        match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
            hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
        )
    )
)
```

For HTTP based gateway routes, App Mesh automatically rewrites the matched prefix path in Gateway Route to “/”.
This automatic rewrite configuration can be overwritten in following ways:

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            # This disables the default rewrite to '/', and retains original path.
            path=appmesh.HttpGatewayRoutePathMatch.starts_with("/path-to-app/", "")
        )
    )
)

gateway.add_gateway_route("gateway-route-http-1",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            # If the request full path is '/path-to-app/xxxxx', this rewrites the path to '/rewrittenUri/xxxxx'.
            # Please note both `prefixPathMatch` and `rewriteTo` must start and end with the `/` character.
            path=appmesh.HttpGatewayRoutePathMatch.starts_with("/path-to-app/", "/rewrittenUri/")
        )
    )
)
```

If matching other path (exact or regex), only specific rewrite path can be specified.
Unlike `startsWith()` method above, no default rewrite is performed.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http-2",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            # This rewrites the path from '/test' to '/rewrittenPath'.
            path=appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath")
        )
    )
)
```

For HTTP/gRPC based routes, App Mesh automatically rewrites
the original request received at the Virtual Gateway to the destination Virtual Service name.
This default host name rewrite can be configured by specifying the rewrite rule as one of the `match` property:

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-grpc",
    route_spec=appmesh.GatewayRouteSpec.grpc(
        route_target=virtual_service,
        match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
            hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
            # This disables the default rewrite to virtual service name and retain original request.
            rewrite_request_hostname=False
        )
    )
)
```

## Importing Resources

Each App Mesh resource class comes with two static methods, `from<Resource>Arn` and `from<Resource>Attributes` (where `<Resource>` is replaced with the resource name, such as `VirtualNode`) for importing a reference to an existing App Mesh resource.
These imported resources can be used with other resources in your mesh as if they were defined directly in your CDK application.

```python
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh/virtualNode/testNode"
appmesh.VirtualNode.from_virtual_node_arn(self, "importedVirtualNode", arn)
```

```python
virtual_node_name = "my-virtual-node"
appmesh.VirtualNode.from_virtual_node_attributes(self, "imported-virtual-node",
    mesh=appmesh.Mesh.from_mesh_name(self, "Mesh", "testMesh"),
    virtual_node_name=virtual_node_name
)
```

To import a mesh, again there are two static methods, `fromMeshArn` and `fromMeshName`.

```python
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh"
appmesh.Mesh.from_mesh_arn(self, "imported-mesh", arn)
```

```python
appmesh.Mesh.from_mesh_name(self, "imported-mesh", "abc")
```

## IAM Grants

`VirtualNode` and `VirtualGateway` provide `grantStreamAggregatedResources` methods that grant identities that are running
Envoy access to stream generated config from App Mesh.

```python
# mesh: appmesh.Mesh

gateway = appmesh.VirtualGateway(self, "testGateway", mesh=mesh)
envoy_user = iam.User(self, "envoyUser")

#
# This will grant `grantStreamAggregatedResources` ONLY for this gateway.
#
gateway.grant_stream_aggregated_resources(envoy_user)
```

## Adding Resources to shared meshes

A shared mesh allows resources created by different accounts to communicate with each other in the same mesh:

```python
# This is the ARN for the mesh from different AWS IAM account ID.
# Ensure mesh is properly shared with your account. For more details, see: https://github.com/aws/aws-cdk/issues/15404
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh"
shared_mesh = appmesh.Mesh.from_mesh_arn(self, "imported-mesh", arn)

# This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
appmesh.VirtualNode(self, "test-node",
    mesh=shared_mesh
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_e0a482dc,
    CfnTag as _CfnTag_95fbdc29,
    Construct as _Construct_e78e779f,
    Duration as _Duration_070aa057,
    IInspectable as _IInspectable_82c04a63,
    IResolvable as _IResolvable_a771d0ef,
    IResource as _IResource_8c1dbbbd,
    Resource as _Resource_abff4495,
    TagManager as _TagManager_0b7ab120,
    TreeInspector as _TreeInspector_1cd1894e,
)
from ..aws_acmpca import ICertificateAuthority as _ICertificateAuthority_7f5d51a5
from ..aws_certificatemanager import ICertificate as _ICertificate_c7bbdc16
from ..aws_iam import Grant as _Grant_bcb5eae7, IGrantable as _IGrantable_4c5a91d1
from ..aws_servicediscovery import IService as _IService_66c1fbd2


class AccessLog(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.AccessLog",
):
    '''(experimental) Configuration for Envoy Access logs for mesh endpoints.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        vpc = ec2.Vpc(self, "vpc")
        namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
            vpc=vpc,
            name="domain.local"
        )
        service = namespace.create_service("Svc")
        node = mesh.add_virtual_node("virtual-node",
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8081,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=cdk.Duration.seconds(5),  # minimum
                    path="/health-check-path",
                    timeout=cdk.Duration.seconds(2),  # minimum
                    unhealthy_threshold=2
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFilePath")
    @builtins.classmethod
    def from_file_path(cls, file_path: builtins.str) -> "AccessLog":
        '''(experimental) Path to a file to write access logs to.

        :param file_path: -

        :default: - no file based access logging

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dbf98a7c5c81056c36d2753a5a9b3f06b56b6ad5e536b828afa7e6ec68882179)
            check_type(argname="argument file_path", value=file_path, expected_type=type_hints["file_path"])
        return typing.cast("AccessLog", jsii.sinvoke(cls, "fromFilePath", [file_path]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "AccessLogConfig":
        '''(experimental) Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        ...


class _AccessLogProxy(AccessLog):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "AccessLogConfig":
        '''(experimental) Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9910a4f9ef4d61867cf62c73eafd96bae713d83728a524d64547c2339ca30e9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("AccessLogConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AccessLog).__jsii_proxy_class__ = lambda : _AccessLogProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.AccessLogConfig",
    jsii_struct_bases=[],
    name_mapping={
        "virtual_gateway_access_log": "virtualGatewayAccessLog",
        "virtual_node_access_log": "virtualNodeAccessLog",
    },
)
class AccessLogConfig:
    def __init__(
        self,
        *,
        virtual_gateway_access_log: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", typing.Dict[builtins.str, typing.Any]]] = None,
        virtual_node_access_log: typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) All Properties for Envoy Access logs for mesh endpoints.

        :param virtual_gateway_access_log: (experimental) VirtualGateway CFN configuration for Access Logging. Default: - no access logging
        :param virtual_node_access_log: (experimental) VirtualNode CFN configuration for Access Logging. Default: - no access logging

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            access_log_config = appmesh.AccessLogConfig(
                virtual_gateway_access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                        path="path",
            
                        # the properties below are optional
                        format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                ),
                virtual_node_access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                    file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                        path="path",
            
                        # the properties below are optional
                        format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                )
            )
        '''
        if isinstance(virtual_gateway_access_log, dict):
            virtual_gateway_access_log = CfnVirtualGateway.VirtualGatewayAccessLogProperty(**virtual_gateway_access_log)
        if isinstance(virtual_node_access_log, dict):
            virtual_node_access_log = CfnVirtualNode.AccessLogProperty(**virtual_node_access_log)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__13bee68e20e05ac56d7ea9aae80a1b466eecbd6606a78467ce0649f1cfa176a3)
            check_type(argname="argument virtual_gateway_access_log", value=virtual_gateway_access_log, expected_type=type_hints["virtual_gateway_access_log"])
            check_type(argname="argument virtual_node_access_log", value=virtual_node_access_log, expected_type=type_hints["virtual_node_access_log"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if virtual_gateway_access_log is not None:
            self._values["virtual_gateway_access_log"] = virtual_gateway_access_log
        if virtual_node_access_log is not None:
            self._values["virtual_node_access_log"] = virtual_node_access_log

    @builtins.property
    def virtual_gateway_access_log(
        self,
    ) -> typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"]:
        '''(experimental) VirtualGateway CFN configuration for Access Logging.

        :default: - no access logging

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway_access_log")
        return typing.cast(typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"], result)

    @builtins.property
    def virtual_node_access_log(
        self,
    ) -> typing.Optional["CfnVirtualNode.AccessLogProperty"]:
        '''(experimental) VirtualNode CFN configuration for Access Logging.

        :default: - no access logging

        :stability: experimental
        '''
        result = self._values.get("virtual_node_access_log")
        return typing.cast(typing.Optional["CfnVirtualNode.AccessLogProperty"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessLogConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Backend(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.Backend",
):
    '''(experimental) Contains static factory methods to create backends.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        # router: appmesh.VirtualRouter
        # service: cloudmap.Service
        
        
        node = appmesh.VirtualNode(self, "node",
            mesh=mesh,
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8080,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=cdk.Duration.seconds(5),
                    path="/ping",
                    timeout=cdk.Duration.seconds(2),
                    unhealthy_threshold=2
                ),
                timeout=cdk.aws_appmesh.HttpTimeout(
                    idle=cdk.Duration.seconds(5)
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
        
        virtual_service = appmesh.VirtualService(self, "service-1",
            virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router),
            virtual_service_name="service1.domain.local"
        )
        
        node.add_backend(appmesh.Backend.virtual_service(virtual_service))
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="virtualService")
    @builtins.classmethod
    def virtual_service(
        cls,
        virtual_service: "IVirtualService",
        *,
        tls_client_policy: typing.Optional[typing.Union["TlsClientPolicy", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "Backend":
        '''(experimental) Construct a Virtual Service backend.

        :param virtual_service: -
        :param tls_client_policy: (experimental) TLS properties for Client policy for the backend. Default: - none

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0b223487704a90ed9e39d794a715196bd0eda99e5400ba1a8b3c833613bbbdb)
            check_type(argname="argument virtual_service", value=virtual_service, expected_type=type_hints["virtual_service"])
        props = VirtualServiceBackendOptions(tls_client_policy=tls_client_policy)

        return typing.cast("Backend", jsii.sinvoke(cls, "virtualService", [virtual_service, props]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, _scope: _Construct_e78e779f) -> "BackendConfig":
        '''(experimental) Return backend config.

        :param _scope: -

        :stability: experimental
        '''
        ...


class _BackendProxy(Backend):
    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _Construct_e78e779f) -> "BackendConfig":
        '''(experimental) Return backend config.

        :param _scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dbd51eb3912be5ea8b5196b0275bb4400ec85bc0e5234e1f859cfcc1eec2a577)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
        return typing.cast("BackendConfig", jsii.invoke(self, "bind", [_scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Backend).__jsii_proxy_class__ = lambda : _BackendProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.BackendConfig",
    jsii_struct_bases=[],
    name_mapping={"virtual_service_backend": "virtualServiceBackend"},
)
class BackendConfig:
    def __init__(
        self,
        *,
        virtual_service_backend: typing.Union["CfnVirtualNode.BackendProperty", typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''(experimental) Properties for a backend.

        :param virtual_service_backend: (experimental) Config for a Virtual Service backend.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            backend_config = appmesh.BackendConfig(
                virtual_service_backend=appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
            
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )
            )
        '''
        if isinstance(virtual_service_backend, dict):
            virtual_service_backend = CfnVirtualNode.BackendProperty(**virtual_service_backend)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6f8edb5c5d7f65ad5f3ebca9e009151a34bf163525dd0834ddce54043deca4f)
            check_type(argname="argument virtual_service_backend", value=virtual_service_backend, expected_type=type_hints["virtual_service_backend"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "virtual_service_backend": virtual_service_backend,
        }

    @builtins.property
    def virtual_service_backend(self) -> "CfnVirtualNode.BackendProperty":
        '''(experimental) Config for a Virtual Service backend.

        :stability: experimental
        '''
        result = self._values.get("virtual_service_backend")
        assert result is not None, "Required property 'virtual_service_backend' is missing"
        return typing.cast("CfnVirtualNode.BackendProperty", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackendConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.BackendDefaults",
    jsii_struct_bases=[],
    name_mapping={"tls_client_policy": "tlsClientPolicy"},
)
class BackendDefaults:
    def __init__(
        self,
        *,
        tls_client_policy: typing.Optional[typing.Union["TlsClientPolicy", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) Represents the properties needed to define backend defaults.

        :param tls_client_policy: (experimental) TLS properties for Client policy for backend defaults. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # mesh: appmesh.Mesh
            # service: cloudmap.Service
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=8080,
                    health_check=appmesh.HealthCheck.http(
                        healthy_threshold=3,
                        interval=cdk.Duration.seconds(5),
                        path="/ping",
                        timeout=cdk.Duration.seconds(2),
                        unhealthy_threshold=2
                    ),
                    timeout=cdk.aws_appmesh.HttpTimeout(
                        idle=cdk.Duration.seconds(5)
                    )
                )],
                backend_defaults=cdk.aws_appmesh.BackendDefaults(
                    tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
                        validation=cdk.aws_appmesh.TlsValidation(
                            trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
                        )
                    )
                ),
                access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
            )
            
            cdk.Tags.of(node).add("Environment", "Dev")
        '''
        if isinstance(tls_client_policy, dict):
            tls_client_policy = TlsClientPolicy(**tls_client_policy)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b789fcbe985f7568977b41c486d350642433fe097423c8b56647ebf4efb2974)
            check_type(argname="argument tls_client_policy", value=tls_client_policy, expected_type=type_hints["tls_client_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if tls_client_policy is not None:
            self._values["tls_client_policy"] = tls_client_policy

    @builtins.property
    def tls_client_policy(self) -> typing.Optional["TlsClientPolicy"]:
        '''(experimental) TLS properties for Client policy for backend defaults.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls_client_policy")
        return typing.cast(typing.Optional["TlsClientPolicy"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackendDefaults(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnGatewayRoute(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnGatewayRoute",
):
    '''A CloudFormation ``AWS::AppMesh::GatewayRoute``.

    Creates a gateway route.

    A gateway route is attached to a virtual gateway and routes traffic to an existing virtual service. If a route matches a request, it can distribute traffic to a target virtual service.

    For more information about gateway routes, see `Gateway routes <https://docs.aws.amazon.com/app-mesh/latest/userguide/gateway-routes.html>`_ .

    :cloudformationResource: AWS::AppMesh::GatewayRoute
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_gateway_route = appmesh.CfnGatewayRoute(self, "MyCfnGatewayRoute",
            mesh_name="meshName",
            spec=appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
        
                            # the properties below are optional
                            port=123
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        port=123,
                        service_name="serviceName"
                    )
                ),
                http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
        
                            # the properties below are optional
                            port=123
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
        
                            # the properties below are optional
                            port=123
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                priority=123
            ),
            virtual_gateway_name="virtualGatewayName",
        
            # the properties below are optional
            gateway_route_name="gatewayRouteName",
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        virtual_gateway_name: builtins.str,
        gateway_route_name: typing.Optional[builtins.str] = None,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::GatewayRoute``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh that the resource resides in.
        :param spec: The specifications of the gateway route.
        :param virtual_gateway_name: The virtual gateway that the gateway route is associated with.
        :param gateway_route_name: The name of the gateway route.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the gateway route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf780facb7b401e0b5fbdde991a94efca636a0460056244c7c71f81a55ebb62d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGatewayRouteProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_gateway_name=virtual_gateway_name,
            gateway_route_name=gateway_route_name,
            mesh_owner=mesh_owner,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38a007ef9bee9cb7c57f11030a72220dc346634ca2b44b143cd63a4e43efdaad)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5b6efc742064ff1dac84efdbf81f065448419a73ed20a28a89d54e9e9e70510)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the gateway route.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrGatewayRouteName")
    def attr_gateway_route_name(self) -> builtins.str:
        '''The name of the gateway route.

        :cloudformationAttribute: GatewayRouteName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrGatewayRouteName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the gateway route resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the gateway route.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualGatewayName")
    def attr_virtual_gateway_name(self) -> builtins.str:
        '''The name of the virtual gateway that the gateway route is associated with.

        :cloudformationAttribute: VirtualGatewayName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualGatewayName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the gateway route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the resource resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__413c86367018437fe6932e29a093ee64b6e5687e4f716367a48b213a5935904f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef]:
        '''The specifications of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-spec
        '''
        return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d45abec7e0f26e52bd2d7b0e8f0b3e2fafe78dcbdbad855b464dade276f0ea5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @builtins.property
    @jsii.member(jsii_name="virtualGatewayName")
    def virtual_gateway_name(self) -> builtins.str:
        '''The virtual gateway that the gateway route is associated with.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-virtualgatewayname
        '''
        return typing.cast(builtins.str, jsii.get(self, "virtualGatewayName"))

    @virtual_gateway_name.setter
    def virtual_gateway_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3451d7ff11672d34058cf96e29324875a5049c1622c0295af84e7c602ce6821)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualGatewayName", value)

    @builtins.property
    @jsii.member(jsii_name="gatewayRouteName")
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-gatewayroutename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "gatewayRouteName"))

    @gateway_route_name.setter
    def gateway_route_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cfdbfcabcd24163a706a1e67a11b4f9d75393e05c01fadc281862aa33b6a26ff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "gatewayRouteName", value)

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__260d5eef8097950726a39fffa915b04f52dab0dcac869285a24fb633a3844193)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "suffix": "suffix"},
    )
    class GatewayRouteHostnameMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the gateway route host name to match.

            :param exact: The exact host name to match on.
            :param suffix: The specified ending characters of the host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_hostname_match_property = appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                    exact="exact",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8fd4e9496dbbeac66781e89d0ee02dde1efa460b15b1c8c7e800951079963ec6)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html#cfn-appmesh-gatewayroute-gatewayroutehostnamematch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The specified ending characters of the host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html#cfn-appmesh-gatewayroute-gatewayroutehostnamematch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteHostnameMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"default_target_hostname": "defaultTargetHostname"},
    )
    class GatewayRouteHostnameRewriteProperty:
        def __init__(
            self,
            *,
            default_target_hostname: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the gateway route host name to rewrite.

            :param default_target_hostname: The default target host name to write to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamerewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_hostname_rewrite_property = appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                    default_target_hostname="defaultTargetHostname"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__46a2fe6744b131af0e1b36ee9b832ebd06ad5b4692807bd3f975898dab903e4e)
                check_type(argname="argument default_target_hostname", value=default_target_hostname, expected_type=type_hints["default_target_hostname"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if default_target_hostname is not None:
                self._values["default_target_hostname"] = default_target_hostname

        @builtins.property
        def default_target_hostname(self) -> typing.Optional[builtins.str]:
            '''The default target host name to write to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamerewrite.html#cfn-appmesh-gatewayroute-gatewayroutehostnamerewrite-defaulttargethostname
            '''
            result = self._values.get("default_target_hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteHostnameRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class GatewayRouteMetadataMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the method header to be matched.

            :param exact: The exact method header to be matched on.
            :param prefix: The specified beginning characters of the method header to be matched on.
            :param range: An object that represents the range of values to match on.
            :param regex: The regex used to match the method header.
            :param suffix: The specified ending characters of the method header to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_metadata_match_property = appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4b45cf6adbc3a85f10dc33cc2c531282fbc6bd727b9a416b0a1f6d95718f67d8)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact method header to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The specified beginning characters of the method header to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the method header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The specified ending characters of the method header to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteMetadataMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"end": "end", "start": "start"},
    )
    class GatewayRouteRangeMatchProperty:
        def __init__(self, *, end: jsii.Number, start: jsii.Number) -> None:
            '''An object that represents the range of values to match on.

            The first character of the range is included in the range, though the last character is not. For example, if the range specified were 1-100, only values 1-99 would be matched.

            :param end: The end of the range.
            :param start: The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_range_match_property = appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                    end=123,
                    start=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__52a324ae04662c4c3cf8406843854fe3065fdecd45a9a3460cb9c851e1cc1ecf)
                check_type(argname="argument end", value=end, expected_type=type_hints["end"])
                check_type(argname="argument start", value=start, expected_type=type_hints["start"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "end": end,
                "start": start,
            }

        @builtins.property
        def end(self) -> jsii.Number:
            '''The end of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html#cfn-appmesh-gatewayroute-gatewayrouterangematch-end
            '''
            result = self._values.get("end")
            assert result is not None, "Required property 'end' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def start(self) -> jsii.Number:
            '''The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html#cfn-appmesh-gatewayroute-gatewayrouterangematch-start
            '''
            result = self._values.get("start")
            assert result is not None, "Required property 'start' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteRangeMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "grpc_route": "grpcRoute",
            "http2_route": "http2Route",
            "http_route": "httpRoute",
            "priority": "priority",
        },
    )
    class GatewayRouteSpecProperty:
        def __init__(
            self,
            *,
            grpc_route: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http2_route: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http_route: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            priority: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a gateway route specification.

            Specify one gateway route type.

            :param grpc_route: An object that represents the specification of a gRPC gateway route.
            :param http2_route: An object that represents the specification of an HTTP/2 gateway route.
            :param http_route: An object that represents the specification of an HTTP gateway route.
            :param priority: The ordering of the gateway routes spec.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_spec_property = appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                    grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
                
                                # the properties below are optional
                                port=123
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            port=123,
                            service_name="serviceName"
                        )
                    ),
                    http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
                
                                # the properties below are optional
                                port=123
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
                
                                # the properties below are optional
                                port=123
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    priority=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__223b610119eb225e58eb4a515952ae2f7d930bd8bfe641584a03a9ff0b329aa7)
                check_type(argname="argument grpc_route", value=grpc_route, expected_type=type_hints["grpc_route"])
                check_type(argname="argument http2_route", value=http2_route, expected_type=type_hints["http2_route"])
                check_type(argname="argument http_route", value=http_route, expected_type=type_hints["http_route"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc_route is not None:
                self._values["grpc_route"] = grpc_route
            if http2_route is not None:
                self._values["http2_route"] = http2_route
            if http_route is not None:
                self._values["http_route"] = http_route
            if priority is not None:
                self._values["priority"] = priority

        @builtins.property
        def grpc_route(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of a gRPC gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-grpcroute
            '''
            result = self._values.get("grpc_route")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2_route(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP/2 gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-http2route
            '''
            result = self._values.get("http2_route")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http_route(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-httproute
            '''
            result = self._values.get("http_route")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The ordering of the gateway routes spec.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteTargetProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service": "virtualService", "port": "port"},
    )
    class GatewayRouteTargetProperty:
        def __init__(
            self,
            *,
            virtual_service: typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a gateway route target.

            :param virtual_service: An object that represents a virtual service gateway route target.
            :param port: The port number of the gateway route target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_target_property = appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                    virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                        virtual_service_name="virtualServiceName"
                    ),
                
                    # the properties below are optional
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e500b17e60f488df4775a49a7cd52b4b7546c8d7ec035c88091fd85c0ebb3756)
                check_type(argname="argument virtual_service", value=virtual_service, expected_type=type_hints["virtual_service"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_service": virtual_service,
            }
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def virtual_service(
            self,
        ) -> typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", _IResolvable_a771d0ef]:
            '''An object that represents a virtual service gateway route target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html#cfn-appmesh-gatewayroute-gatewayroutetarget-virtualservice
            '''
            result = self._values.get("virtual_service")
            assert result is not None, "Required property 'virtual_service' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number of the gateway route target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html#cfn-appmesh-gatewayroute-gatewayroutetarget-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service_name": "virtualServiceName"},
    )
    class GatewayRouteVirtualServiceProperty:
        def __init__(self, *, virtual_service_name: builtins.str) -> None:
            '''An object that represents the virtual service that traffic is routed to.

            :param virtual_service_name: The name of the virtual service that traffic is routed to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutevirtualservice.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_virtual_service_property = appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                    virtual_service_name="virtualServiceName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2e71513a78ef8e5af14556eddde710b6489745698e24794dce8cf207078713ad)
                check_type(argname="argument virtual_service_name", value=virtual_service_name, expected_type=type_hints["virtual_service_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_service_name": virtual_service_name,
            }

        @builtins.property
        def virtual_service_name(self) -> builtins.str:
            '''The name of the virtual service that traffic is routed to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutevirtualservice.html#cfn-appmesh-gatewayroute-gatewayroutevirtualservice-virtualservicename
            '''
            result = self._values.get("virtual_service_name")
            assert result is not None, "Required property 'virtual_service_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteVirtualServiceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"target": "target", "rewrite": "rewrite"},
    )
    class GrpcGatewayRouteActionProperty:
        def __init__(
            self,
            *,
            target: typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            rewrite: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param target: An object that represents the target that traffic is routed to when a request matches the gateway route.
            :param rewrite: The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_action_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                    target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                        virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                            virtual_service_name="virtualServiceName"
                        ),
                
                        # the properties below are optional
                        port=123
                    ),
                
                    # the properties below are optional
                    rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                            default_target_hostname="defaultTargetHostname"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__99eec87e77681fcc588dc2d7f353675f9c8ab39944aeeb8862eab01685295cfe)
                check_type(argname="argument target", value=target, expected_type=type_hints["target"])
                check_type(argname="argument rewrite", value=rewrite, expected_type=type_hints["rewrite"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target": target,
            }
            if rewrite is not None:
                self._values["rewrite"] = rewrite

        @builtins.property
        def target(
            self,
        ) -> typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef]:
            '''An object that represents the target that traffic is routed to when a request matches the gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html#cfn-appmesh-gatewayroute-grpcgatewayrouteaction-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def rewrite(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", _IResolvable_a771d0ef]]:
            '''The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html#cfn-appmesh-gatewayroute-grpcgatewayrouteaction-rewrite
            '''
            result = self._values.get("rewrite")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hostname": "hostname",
            "metadata": "metadata",
            "port": "port",
            "service_name": "serviceName",
        },
    )
    class GrpcGatewayRouteMatchProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            metadata: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            port: typing.Optional[jsii.Number] = None,
            service_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param hostname: The gateway route host name to be matched on.
            :param metadata: The gateway route metadata to be matched on.
            :param port: The gateway route port to be matched on.
            :param service_name: The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_match_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                        exact="exact",
                        suffix="suffix"
                    ),
                    metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    port=123,
                    service_name="serviceName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4567ca72a13af6a87fe59f279d24ed389c8931d45092cf0b327c50a0397a07b2)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument metadata", value=metadata, expected_type=type_hints["metadata"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if metadata is not None:
                self._values["metadata"] = metadata
            if port is not None:
                self._values["port"] = port
            if service_name is not None:
                self._values["service_name"] = service_name

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]]:
            '''The gateway route host name to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", _IResolvable_a771d0ef]]]]:
            '''The gateway route metadata to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The gateway route port to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def service_name(self) -> typing.Optional[builtins.str]:
            '''The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-servicename
            '''
            result = self._values.get("service_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class GrpcGatewayRouteMetadataProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object representing the metadata of the gateway route.

            :param name: A name for the gateway route metadata.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: The criteria for determining a metadata match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_metadata_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__746e2f9c3ee08c3296353edb060b125c9fd8a82fe2aa4eaf658bc31c6daa54c8)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the gateway route metadata.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", _IResolvable_a771d0ef]]:
            '''The criteria for determining a metadata match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match"},
    )
    class GrpcGatewayRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            match: typing.Union[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents a gRPC gateway route.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
                
                            # the properties below are optional
                            port=123
                        ),
                
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        port=123,
                        service_name="serviceName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0344dd21b7d40aed37c37cee06f3012b186156e20d90080bca7a3c9c6e03227f)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html#cfn-appmesh-gatewayroute-grpcgatewayroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html#cfn-appmesh-gatewayroute-grpcgatewayroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname"},
    )
    class GrpcGatewayRouteRewriteProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the gateway route to rewrite.

            :param hostname: The host name of the gateway route to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouterewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_rewrite_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                        default_target_hostname="defaultTargetHostname"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a6edc76e5bb777a067041747d51ed2c962be687b781072c187a3466ecedae1ae)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]]:
            '''The host name of the gateway route to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouterewrite.html#cfn-appmesh-gatewayroute-grpcgatewayrouterewrite-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"target": "target", "rewrite": "rewrite"},
    )
    class HttpGatewayRouteActionProperty:
        def __init__(
            self,
            *,
            target: typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            rewrite: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param target: An object that represents the target that traffic is routed to when a request matches the gateway route.
            :param rewrite: The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_action_property = appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                    target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                        virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                            virtual_service_name="virtualServiceName"
                        ),
                
                        # the properties below are optional
                        port=123
                    ),
                
                    # the properties below are optional
                    rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                            default_target_hostname="defaultTargetHostname"
                        ),
                        path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                            exact="exact"
                        ),
                        prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                            default_prefix="defaultPrefix",
                            value="value"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__06119f492ee2bf56a2d6d1e65610818ff9105f70512298617b53ef3d61a382ea)
                check_type(argname="argument target", value=target, expected_type=type_hints["target"])
                check_type(argname="argument rewrite", value=rewrite, expected_type=type_hints["rewrite"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target": target,
            }
            if rewrite is not None:
                self._values["rewrite"] = rewrite

        @builtins.property
        def target(
            self,
        ) -> typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef]:
            '''An object that represents the target that traffic is routed to when a request matches the gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html#cfn-appmesh-gatewayroute-httpgatewayrouteaction-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def rewrite(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", _IResolvable_a771d0ef]]:
            '''The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html#cfn-appmesh-gatewayroute-httpgatewayrouteaction-rewrite
            '''
            result = self._values.get("rewrite")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class HttpGatewayRouteHeaderMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_header_match_property = appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__78c860795fa3908201ec01f0634a814fa170bf5669aec982d0dd12961c247b82)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteHeaderMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class HttpGatewayRouteHeaderProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the HTTP header in the gateway route.

            :param name: A name for the HTTP header in the gateway route that will be matched on.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: An object that represents the method and value to match with the header value sent in a request. Specify one match method.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_header_property = appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b317659df565f2c5b048ed32663db4c9f73bcf8df0e905b3faba9016bda7b2fe)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the HTTP header in the gateway route that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "headers": "headers",
            "hostname": "hostname",
            "method": "method",
            "path": "path",
            "port": "port",
            "prefix": "prefix",
            "query_parameters": "queryParameters",
        },
    )
    class HttpGatewayRouteMatchProperty:
        def __init__(
            self,
            *,
            headers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            hostname: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            method: typing.Optional[builtins.str] = None,
            path: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpPathMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            port: typing.Optional[jsii.Number] = None,
            prefix: typing.Optional[builtins.str] = None,
            query_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnGatewayRoute.QueryParameterProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param headers: The client request headers to match on.
            :param hostname: The host name to match on.
            :param method: The method to match on.
            :param path: The path to match on.
            :param port: The port number to match on.
            :param prefix: Specifies the path to match requests with. This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .
            :param query_parameters: The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_match_property = appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                    headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                        exact="exact",
                        suffix="suffix"
                    ),
                    method="method",
                    path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                        exact="exact",
                        regex="regex"
                    ),
                    port=123,
                    prefix="prefix",
                    query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                        name="name",
                
                        # the properties below are optional
                        match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                            exact="exact"
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__48dbe183da5c07dbcff7294cd8fa712b70d7a57912bf32fd7a331519b787443b)
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if headers is not None:
                self._values["headers"] = headers
            if hostname is not None:
                self._values["hostname"] = hostname
            if method is not None:
                self._values["method"] = method
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port
            if prefix is not None:
                self._values["prefix"] = prefix
            if query_parameters is not None:
                self._values["query_parameters"] = query_parameters

        @builtins.property
        def headers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", _IResolvable_a771d0ef]]]]:
            '''The client request headers to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]]:
            '''The host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The method to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]]:
            '''The path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''Specifies the path to match requests with.

            This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def query_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]]:
            '''The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-queryparameters
            '''
            result = self._values.get("query_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpGatewayRoutePathRewriteProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object that represents the path to rewrite.

            :param exact: The exact path to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutepathrewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_path_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                    exact="exact"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ee8f63370db71c4b75a032233a4950a1becd17483ccb184eb0f289e30f519777)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutepathrewrite.html#cfn-appmesh-gatewayroute-httpgatewayroutepathrewrite-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRoutePathRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"default_prefix": "defaultPrefix", "value": "value"},
    )
    class HttpGatewayRoutePrefixRewriteProperty:
        def __init__(
            self,
            *,
            default_prefix: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the beginning characters of the route to rewrite.

            :param default_prefix: The default prefix used to replace the incoming route prefix when rewritten.
            :param value: The value used to replace the incoming route prefix when rewritten.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_prefix_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                    default_prefix="defaultPrefix",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b4543a2e5808fdfe00cd1e8c5d9aab9e223b7e8691940711aa741a1c6e6537d3)
                check_type(argname="argument default_prefix", value=default_prefix, expected_type=type_hints["default_prefix"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if default_prefix is not None:
                self._values["default_prefix"] = default_prefix
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def default_prefix(self) -> typing.Optional[builtins.str]:
            '''The default prefix used to replace the incoming route prefix when rewritten.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouteprefixrewrite-defaultprefix
            '''
            result = self._values.get("default_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value used to replace the incoming route prefix when rewritten.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouteprefixrewrite-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRoutePrefixRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match"},
    )
    class HttpGatewayRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            match: typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents an HTTP gateway route.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_property = appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
                
                            # the properties below are optional
                            port=123
                        ),
                
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
                
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__aeb57c8bb6f2e4e19eba8cca5c9b4177b87f24c280531506f6f5519015af9853)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html#cfn-appmesh-gatewayroute-httpgatewayroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html#cfn-appmesh-gatewayroute-httpgatewayroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "path": "path", "prefix": "prefix"},
    )
    class HttpGatewayRouteRewriteProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            path: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            prefix: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object representing the gateway route to rewrite.

            :param hostname: The host name to rewrite.
            :param path: The path to rewrite.
            :param prefix: The specified beginning characters to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                        default_target_hostname="defaultTargetHostname"
                    ),
                    path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                        exact="exact"
                    ),
                    prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                        default_prefix="defaultPrefix",
                        value="value"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b27134f2126e68496f3d324fd8bc9350930b966316ca5ea0ab55a493ccbe1371)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if path is not None:
                self._values["path"] = path
            if prefix is not None:
                self._values["prefix"] = prefix

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]]:
            '''The host name to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", _IResolvable_a771d0ef]]:
            '''The path to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def prefix(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", _IResolvable_a771d0ef]]:
            '''The specified beginning characters to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpPathMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "regex": "regex"},
    )
    class HttpPathMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            regex: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the path to match in the request.

            :param exact: The exact path to match on.
            :param regex: The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_path_match_property = appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e82c8805a2d5b0ef2d90097981143965b511757d0cc7a49bb330edd755f3cf2f)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if regex is not None:
                self._values["regex"] = regex

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html#cfn-appmesh-gatewayroute-httppathmatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html#cfn-appmesh-gatewayroute-httppathmatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpPathMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpQueryParameterMatchProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object representing the query parameter to match.

            :param exact: The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpqueryparametermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_query_parameter_match_property = appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                    exact="exact"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__530b2d32b26fc4d3f7cb8af5b09761534abec7332e4992ccd64d461f30ae6467)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpqueryparametermatch.html#cfn-appmesh-gatewayroute-httpqueryparametermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpQueryParameterMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.QueryParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "match": "match"},
    )
    class QueryParameterProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            match: typing.Optional[typing.Union[typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the query parameter in the request.

            :param name: A name for the query parameter that will be matched on.
            :param match: The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                query_parameter_property = appmesh.CfnGatewayRoute.QueryParameterProperty(
                    name="name",
                
                    # the properties below are optional
                    match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                        exact="exact"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7d830576fd1486002fa2ba81586b3bf7641bb0888a5597a337a8ba5734f7e0bc)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the query parameter that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html#cfn-appmesh-gatewayroute-queryparameter-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]]:
            '''The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html#cfn-appmesh-gatewayroute-queryparameter-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnGatewayRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_gateway_name": "virtualGatewayName",
        "gateway_route_name": "gatewayRouteName",
        "mesh_owner": "meshOwner",
        "tags": "tags",
    },
)
class CfnGatewayRouteProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        virtual_gateway_name: builtins.str,
        gateway_route_name: typing.Optional[builtins.str] = None,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnGatewayRoute``.

        :param mesh_name: The name of the service mesh that the resource resides in.
        :param spec: The specifications of the gateway route.
        :param virtual_gateway_name: The virtual gateway that the gateway route is associated with.
        :param gateway_route_name: The name of the gateway route.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the gateway route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_gateway_route_props = appmesh.CfnGatewayRouteProps(
                mesh_name="meshName",
                spec=appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                    grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
            
                                # the properties below are optional
                                port=123
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            port=123,
                            service_name="serviceName"
                        )
                    ),
                    http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
            
                                # the properties below are optional
                                port=123
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
            
                                # the properties below are optional
                                port=123
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    priority=123
                ),
                virtual_gateway_name="virtualGatewayName",
            
                # the properties below are optional
                gateway_route_name="gatewayRouteName",
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3063570f823ddbf72b13a7b8c389d20bcf5d1b47960e451280d341eb4eb51b5e)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument virtual_gateway_name", value=virtual_gateway_name, expected_type=type_hints["virtual_gateway_name"])
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_gateway_name": virtual_gateway_name,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the resource resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, _IResolvable_a771d0ef]:
        '''The specifications of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def virtual_gateway_name(self) -> builtins.str:
        '''The virtual gateway that the gateway route is associated with.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-virtualgatewayname
        '''
        result = self._values.get("virtual_gateway_name")
        assert result is not None, "Required property 'virtual_gateway_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-gatewayroutename
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the gateway route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnMesh(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnMesh",
):
    '''A CloudFormation ``AWS::AppMesh::Mesh``.

    Creates a service mesh.

    A service mesh is a logical boundary for network traffic between services that are represented by resources within the mesh. After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.

    For more information about service meshes, see `Service meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html>`_ .

    :cloudformationResource: AWS::AppMesh::Mesh
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_mesh = appmesh.CfnMesh(self, "MyCfnMesh",
            mesh_name="meshName",
            spec=appmesh.CfnMesh.MeshSpecProperty(
                egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                    type="type"
                ),
                service_discovery=appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                    ip_preference="ipPreference"
                )
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: typing.Optional[builtins.str] = None,
        spec: typing.Optional[typing.Union[typing.Union["CfnMesh.MeshSpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::Mesh``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name to use for the service mesh.
        :param spec: The service mesh specification to apply.
        :param tags: Optional metadata that you can apply to the service mesh to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8753592c966aec0e85c467857f9599845428b4d2552ad984cb93434f7dd518f6)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnMeshProps(mesh_name=mesh_name, spec=spec, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__20708f21ad93cb8c2300eaba74573208e223bd4582a6022b10f13d51f8b4171a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__004e62ef6c821ddb805ae29a8717d5755db84bacc3ce3ae31a70194973aca4fd)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the mesh.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the mesh.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the service mesh to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the service mesh.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-meshname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3158f3cb46d2811e7fed4d42febb51395eb45e5e2101b5123f10ed6fa33b0936)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]]:
        '''The service mesh specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-spec
        '''
        return typing.cast(typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__826cfcb6c86341f43906e8409fe8e4d035ebda2cad58f6988429fc012025934e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnMesh.EgressFilterProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class EgressFilterProperty:
        def __init__(self, *, type: builtins.str) -> None:
            '''An object that represents the egress filter rules for a service mesh.

            :param type: The egress filter type. By default, the type is ``DROP_ALL`` , which allows egress only from virtual nodes to other defined resources in the service mesh (and any traffic to ``*.amazonaws.com`` for AWS API calls). You can set the egress filter type to ``ALLOW_ALL`` to allow egress to any endpoint inside or outside of the service mesh. .. epigraph:: If you specify any backends on a virtual node when using ``ALLOW_ALL`` , you must specifiy all egress for that virtual node as backends. Otherwise, ``ALLOW_ALL`` will no longer work for that virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                egress_filter_property = appmesh.CfnMesh.EgressFilterProperty(
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__674ab5cdd6e1652a08821cf8a8263e2ea7e56bdc6ee7c9e60fa848a22df484d7)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''The egress filter type.

            By default, the type is ``DROP_ALL`` , which allows egress only from virtual nodes to other defined resources in the service mesh (and any traffic to ``*.amazonaws.com`` for AWS API calls). You can set the egress filter type to ``ALLOW_ALL`` to allow egress to any endpoint inside or outside of the service mesh.
            .. epigraph::

               If you specify any backends on a virtual node when using ``ALLOW_ALL`` , you must specifiy all egress for that virtual node as backends. Otherwise, ``ALLOW_ALL`` will no longer work for that virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html#cfn-appmesh-mesh-egressfilter-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EgressFilterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnMesh.MeshServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={"ip_preference": "ipPreference"},
    )
    class MeshServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            ip_preference: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the service discovery information for a service mesh.

            :param ip_preference: The IP version to use to control traffic within the mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                mesh_service_discovery_property = appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                    ip_preference="ipPreference"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d66da9e26f61480ce47a614319bd095f857d4d18dd53db5c81e1e78ee4a44574)
                check_type(argname="argument ip_preference", value=ip_preference, expected_type=type_hints["ip_preference"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ip_preference is not None:
                self._values["ip_preference"] = ip_preference

        @builtins.property
        def ip_preference(self) -> typing.Optional[builtins.str]:
            '''The IP version to use to control traffic within the mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshservicediscovery.html#cfn-appmesh-mesh-meshservicediscovery-ippreference
            '''
            result = self._values.get("ip_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MeshServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnMesh.MeshSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "egress_filter": "egressFilter",
            "service_discovery": "serviceDiscovery",
        },
    )
    class MeshSpecProperty:
        def __init__(
            self,
            *,
            egress_filter: typing.Optional[typing.Union[typing.Union["CfnMesh.EgressFilterProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            service_discovery: typing.Optional[typing.Union[typing.Union["CfnMesh.MeshServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a service mesh.

            :param egress_filter: The egress filter rules for the service mesh.
            :param service_discovery: ``CfnMesh.MeshSpecProperty.ServiceDiscovery``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                mesh_spec_property = appmesh.CfnMesh.MeshSpecProperty(
                    egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                        type="type"
                    ),
                    service_discovery=appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                        ip_preference="ipPreference"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a591fd972de4dd17744b51d490dd41387c377fa2ac284f3a7a16a015eedc7e8b)
                check_type(argname="argument egress_filter", value=egress_filter, expected_type=type_hints["egress_filter"])
                check_type(argname="argument service_discovery", value=service_discovery, expected_type=type_hints["service_discovery"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if egress_filter is not None:
                self._values["egress_filter"] = egress_filter
            if service_discovery is not None:
                self._values["service_discovery"] = service_discovery

        @builtins.property
        def egress_filter(
            self,
        ) -> typing.Optional[typing.Union["CfnMesh.EgressFilterProperty", _IResolvable_a771d0ef]]:
            '''The egress filter rules for the service mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html#cfn-appmesh-mesh-meshspec-egressfilter
            '''
            result = self._values.get("egress_filter")
            return typing.cast(typing.Optional[typing.Union["CfnMesh.EgressFilterProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def service_discovery(
            self,
        ) -> typing.Optional[typing.Union["CfnMesh.MeshServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''``CfnMesh.MeshSpecProperty.ServiceDiscovery``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html#cfn-appmesh-mesh-meshspec-servicediscovery
            '''
            result = self._values.get("service_discovery")
            return typing.cast(typing.Optional[typing.Union["CfnMesh.MeshServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MeshSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnMeshProps",
    jsii_struct_bases=[],
    name_mapping={"mesh_name": "meshName", "spec": "spec", "tags": "tags"},
)
class CfnMeshProps:
    def __init__(
        self,
        *,
        mesh_name: typing.Optional[builtins.str] = None,
        spec: typing.Optional[typing.Union[typing.Union[CfnMesh.MeshSpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnMesh``.

        :param mesh_name: The name to use for the service mesh.
        :param spec: The service mesh specification to apply.
        :param tags: Optional metadata that you can apply to the service mesh to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_mesh_props = appmesh.CfnMeshProps(
                mesh_name="meshName",
                spec=appmesh.CfnMesh.MeshSpecProperty(
                    egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                        type="type"
                    ),
                    service_discovery=appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                        ip_preference="ipPreference"
                    )
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24626fe2bf957a7f3c54a3e05e925af2751e3d0b736e98cdfd5e55018e8e172d)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if mesh_name is not None:
            self._values["mesh_name"] = mesh_name
        if spec is not None:
            self._values["spec"] = spec
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the service mesh.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-meshname
        '''
        result = self._values.get("mesh_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Optional[typing.Union[CfnMesh.MeshSpecProperty, _IResolvable_a771d0ef]]:
        '''The service mesh specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-spec
        '''
        result = self._values.get("spec")
        return typing.cast(typing.Optional[typing.Union[CfnMesh.MeshSpecProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the service mesh to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMeshProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnRoute(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnRoute",
):
    '''A CloudFormation ``AWS::AppMesh::Route``.

    Creates a route that is associated with a virtual router.

    You can route several different protocols and define a retry policy for a route. Traffic can be routed to one or more virtual nodes.

    For more information about routes, see `Routes <https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html>`_ .

    :cloudformationResource: AWS::AppMesh::Route
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_route = appmesh.CfnRoute(self, "MyCfnRoute",
            mesh_name="meshName",
            spec=appmesh.CfnRoute.RouteSpecProperty(
                grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                    action=appmesh.CfnRoute.GrpcRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                        metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method_name="methodName",
                        port=123,
                        service_name="serviceName"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        grpc_retry_events=["grpcRetryEvents"],
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                http2_route=appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                http_route=appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                priority=123,
                tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                    action=appmesh.CfnRoute.TcpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
        
                    # the properties below are optional
                    match=appmesh.CfnRoute.TcpRouteMatchProperty(
                        port=123
                    ),
                    timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            ),
            virtual_router_name="virtualRouterName",
        
            # the properties below are optional
            mesh_owner="meshOwner",
            route_name="routeName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union["CfnRoute.RouteSpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        virtual_router_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        route_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::Route``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the route in.
        :param spec: The route specification to apply.
        :param virtual_router_name: The name of the virtual router in which to create the route. If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param route_name: The name to use for the route.
        :param tags: Optional metadata that you can apply to the route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__894b5726dea7f73893e4b48d76223de1423f43ef4f45623940970e57355a6780)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRouteProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_router_name=virtual_router_name,
            mesh_owner=mesh_owner,
            route_name=route_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c4faeed5af886ba4c7d200e80284f705b3f9a4ba0b6d5d4d13037c81d32e013)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed145b7f9200883cd98675cc45048b0a7f4e2748f4975990669a6b9c39740cdc)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the route.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the route resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrRouteName")
    def attr_route_name(self) -> builtins.str:
        '''The name of the route.

        :cloudformationAttribute: RouteName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRouteName"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the route.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualRouterName")
    def attr_virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router that the route is associated with.

        :cloudformationAttribute: VirtualRouterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualRouterName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the route in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__646964f64488fddacb7e51776b8a36e9c8a839ca124616a6e25d20a09f3d41e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(self) -> typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef]:
        '''The route specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-spec
        '''
        return typing.cast(typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4424fd11af1e420184af22c9637f0af5c51d1f2b607fde5a5d9154068863a16b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @builtins.property
    @jsii.member(jsii_name="virtualRouterName")
    def virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router in which to create the route.

        If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-virtualroutername
        '''
        return typing.cast(builtins.str, jsii.get(self, "virtualRouterName"))

    @virtual_router_name.setter
    def virtual_router_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55307044d84c4dd5b3359945f80c2a8a454dac4daad94cc1ba1a421d13b70c0b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualRouterName", value)

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b6400fe3ebe237c3d8dab0f29cbcafc0c67d9985da32cc33ed7a74cc64e4ebf7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value)

    @builtins.property
    @jsii.member(jsii_name="routeName")
    def route_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-routename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "routeName"))

    @route_name.setter
    def route_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__135bc2d092462720e64c33a1f897aedb1273996b1af91c760530664c44fcf605)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.DurationProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class DurationProperty:
        def __init__(self, *, unit: builtins.str, value: jsii.Number) -> None:
            '''An object that represents a duration of time.

            :param unit: A unit of time.
            :param value: A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                duration_property = appmesh.CfnRoute.DurationProperty(
                    unit="unit",
                    value=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__093a41f10abacb06a36760c3ab8a3aaf45ad36c02fe44b44a96672e799c67f23)
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "unit": unit,
                "value": value,
            }

        @builtins.property
        def unit(self) -> builtins.str:
            '''A unit of time.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html#cfn-appmesh-route-duration-unit
            '''
            result = self._values.get("unit")
            assert result is not None, "Required property 'unit' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> jsii.Number:
            '''A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html#cfn-appmesh-route-duration-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRetryPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_retries": "maxRetries",
            "per_retry_timeout": "perRetryTimeout",
            "grpc_retry_events": "grpcRetryEvents",
            "http_retry_events": "httpRetryEvents",
            "tcp_retry_events": "tcpRetryEvents",
        },
    )
    class GrpcRetryPolicyProperty:
        def __init__(
            self,
            *,
            max_retries: jsii.Number,
            per_retry_timeout: typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            grpc_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            http_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            tcp_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a retry policy.

            Specify at least one value for at least one of the types of ``RetryEvents`` , a value for ``maxRetries`` , and a value for ``perRetryTimeout`` . Both ``server-error`` and ``gateway-error`` under ``httpRetryEvents`` include the Envoy ``reset`` policy. For more information on the ``reset`` policy, see the `Envoy documentation <https://docs.aws.amazon.com/https://www.envoyproxy.io/docs/envoy/latest/configuration/http/http_filters/router_filter#x-envoy-retry-on>`_ .

            :param max_retries: The maximum number of retry attempts.
            :param per_retry_timeout: The timeout for each retry attempt.
            :param grpc_retry_events: Specify at least one of the valid values.
            :param http_retry_events: Specify at least one of the following values. - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511 - *gateway-error* – HTTP status codes 502, 503, and 504 - *client-error* – HTTP status code 409 - *stream-error* – Retry on refused stream
            :param tcp_retry_events: Specify a valid value. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_retry_policy_property = appmesh.CfnRoute.GrpcRetryPolicyProperty(
                    max_retries=123,
                    per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                
                    # the properties below are optional
                    grpc_retry_events=["grpcRetryEvents"],
                    http_retry_events=["httpRetryEvents"],
                    tcp_retry_events=["tcpRetryEvents"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__728e92b135074683e57d928e8b57d24b6e2aca4705b7e4c32e5a1e5236e6cd62)
                check_type(argname="argument max_retries", value=max_retries, expected_type=type_hints["max_retries"])
                check_type(argname="argument per_retry_timeout", value=per_retry_timeout, expected_type=type_hints["per_retry_timeout"])
                check_type(argname="argument grpc_retry_events", value=grpc_retry_events, expected_type=type_hints["grpc_retry_events"])
                check_type(argname="argument http_retry_events", value=http_retry_events, expected_type=type_hints["http_retry_events"])
                check_type(argname="argument tcp_retry_events", value=tcp_retry_events, expected_type=type_hints["tcp_retry_events"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_retries": max_retries,
                "per_retry_timeout": per_retry_timeout,
            }
            if grpc_retry_events is not None:
                self._values["grpc_retry_events"] = grpc_retry_events
            if http_retry_events is not None:
                self._values["http_retry_events"] = http_retry_events
            if tcp_retry_events is not None:
                self._values["tcp_retry_events"] = tcp_retry_events

        @builtins.property
        def max_retries(self) -> jsii.Number:
            '''The maximum number of retry attempts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-maxretries
            '''
            result = self._values.get("max_retries")
            assert result is not None, "Required property 'max_retries' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def per_retry_timeout(
            self,
        ) -> typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]:
            '''The timeout for each retry attempt.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-perretrytimeout
            '''
            result = self._values.get("per_retry_timeout")
            assert result is not None, "Required property 'per_retry_timeout' is missing"
            return typing.cast(typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def grpc_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the valid values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-grpcretryevents
            '''
            result = self._values.get("grpc_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def http_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the following values.

            - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
            - *gateway-error* – HTTP status codes 502, 503, and 504
            - *client-error* – HTTP status code 409
            - *stream-error* – Retry on refused stream

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-httpretryevents
            '''
            result = self._values.get("http_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tcp_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify a valid value.

            The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-tcpretryevents
            '''
            result = self._values.get("tcp_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRetryPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class GrpcRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnRoute.WeightedTargetProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_action_property = appmesh.CfnRoute.GrpcRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123,
                
                        # the properties below are optional
                        port=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__82c213986dbda24361eb73df0711bafecdf0ed9c54294c3ac0102b1ff12fefce)
                check_type(argname="argument weighted_targets", value=weighted_targets, expected_type=type_hints["weighted_targets"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcrouteaction.html#cfn-appmesh-route-grpcrouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "metadata": "metadata",
            "method_name": "methodName",
            "port": "port",
            "service_name": "serviceName",
        },
    )
    class GrpcRouteMatchProperty:
        def __init__(
            self,
            *,
            metadata: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnRoute.GrpcRouteMetadataProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            method_name: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
            service_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param metadata: An object that represents the data to match from the request.
            :param method_name: The method name to match from the request. If you specify a name, you must also specify a ``serviceName`` .
            :param port: The port number to match on.
            :param service_name: The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_match_property = appmesh.CfnRoute.GrpcRouteMatchProperty(
                    metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnRoute.MatchRangeProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    method_name="methodName",
                    port=123,
                    service_name="serviceName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2fb201c10bc16615122a033a652514aecde26218a84c158736e6ff100035b195)
                check_type(argname="argument metadata", value=metadata, expected_type=type_hints["metadata"])
                check_type(argname="argument method_name", value=method_name, expected_type=type_hints["method_name"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if metadata is not None:
                self._values["metadata"] = metadata
            if method_name is not None:
                self._values["method_name"] = method_name
            if port is not None:
                self._values["port"] = port
            if service_name is not None:
                self._values["service_name"] = service_name

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.GrpcRouteMetadataProperty", _IResolvable_a771d0ef]]]]:
            '''An object that represents the data to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.GrpcRouteMetadataProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def method_name(self) -> typing.Optional[builtins.str]:
            '''The method name to match from the request.

            If you specify a name, you must also specify a ``serviceName`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-methodname
            '''
            result = self._values.get("method_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def service_name(self) -> typing.Optional[builtins.str]:
            '''The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-servicename
            '''
            result = self._values.get("service_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class GrpcRouteMetadataMatchMethodProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[typing.Union["CfnRoute.MatchRangeProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the match method.

            Specify one of the match values.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_metadata_match_method_property = appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnRoute.MatchRangeProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a7a0bcde72cf3498b29dfdc096aefdbd9278703cb8b2e1c2869f04eee803a5da)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMetadataMatchMethodProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class GrpcRouteMetadataProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union[typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the match metadata for the route.

            :param name: The name of the route.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: An object that represents the data to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_metadata_property = appmesh.CfnRoute.GrpcRouteMetadataProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0a8e4bcdafaec3c8355103d85125d202c964f03bb02880f48b7e9bff51683c5d)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the data to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "match": "match",
            "retry_policy": "retryPolicy",
            "timeout": "timeout",
        },
    )
    class GrpcRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[typing.Union["CfnRoute.GrpcRouteActionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            match: typing.Union[typing.Union["CfnRoute.GrpcRouteMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            retry_policy: typing.Optional[typing.Union[typing.Union["CfnRoute.GrpcRetryPolicyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union[typing.Union["CfnRoute.GrpcTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a gRPC route type.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param retry_policy: An object that represents a retry policy.
            :param timeout: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_property = appmesh.CfnRoute.GrpcRouteProperty(
                    action=appmesh.CfnRoute.GrpcRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
                
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                        metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method_name="methodName",
                        port=123,
                        service_name="serviceName"
                    ),
                
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                
                        # the properties below are optional
                        grpc_retry_events=["grpcRetryEvents"],
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e8a2eb613b9508650b14e466e5b11b2525588eca4093aaedd58004cd6d452ebb)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
                check_type(argname="argument retry_policy", value=retry_policy, expected_type=type_hints["retry_policy"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }
            if retry_policy is not None:
                self._values["retry_policy"] = retry_policy
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnRoute.GrpcRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnRoute.GrpcRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnRoute.GrpcRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnRoute.GrpcRouteMatchProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def retry_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcRetryPolicyProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a retry policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-retrypolicy
            '''
            result = self._values.get("retry_policy")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcRetryPolicyProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class GrpcTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_timeout_property = appmesh.CfnRoute.GrpcTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8d2de8c485a4b151d46aba328d4e741af2a60c32973e784db7682afd2915992c)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html#cfn-appmesh-route-grpctimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html#cfn-appmesh-route-grpctimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HeaderMatchMethodProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class HeaderMatchMethodProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[typing.Union["CfnRoute.MatchRangeProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                header_match_method_property = appmesh.CfnRoute.HeaderMatchMethodProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnRoute.MatchRangeProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e71eabcc8834404fde65623654f1da316b2c9da78bf61036a9b694b2a7da89cd)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HeaderMatchMethodProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpPathMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "regex": "regex"},
    )
    class HttpPathMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            regex: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the path to match in the request.

            :param exact: The exact path to match on.
            :param regex: The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_path_match_property = appmesh.CfnRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b6218bfd573b7dbe1645d49f431a5f9c25c3b9df46e17b784c757d3ad0d9c4ff)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if regex is not None:
                self._values["regex"] = regex

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html#cfn-appmesh-route-httppathmatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html#cfn-appmesh-route-httppathmatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpPathMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpQueryParameterMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpQueryParameterMatchProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object representing the query parameter to match.

            :param exact: The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpqueryparametermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_query_parameter_match_property = appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                    exact="exact"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__de6d7a935962e3b2c718117bf73c886aa0380b2e125517904ddd17b17cf6421b)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpqueryparametermatch.html#cfn-appmesh-route-httpqueryparametermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpQueryParameterMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRetryPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_retries": "maxRetries",
            "per_retry_timeout": "perRetryTimeout",
            "http_retry_events": "httpRetryEvents",
            "tcp_retry_events": "tcpRetryEvents",
        },
    )
    class HttpRetryPolicyProperty:
        def __init__(
            self,
            *,
            max_retries: jsii.Number,
            per_retry_timeout: typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            http_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            tcp_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a retry policy.

            Specify at least one value for at least one of the types of ``RetryEvents`` , a value for ``maxRetries`` , and a value for ``perRetryTimeout`` . Both ``server-error`` and ``gateway-error`` under ``httpRetryEvents`` include the Envoy ``reset`` policy. For more information on the ``reset`` policy, see the `Envoy documentation <https://docs.aws.amazon.com/https://www.envoyproxy.io/docs/envoy/latest/configuration/http/http_filters/router_filter#x-envoy-retry-on>`_ .

            :param max_retries: The maximum number of retry attempts.
            :param per_retry_timeout: The timeout for each retry attempt.
            :param http_retry_events: Specify at least one of the following values. - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511 - *gateway-error* – HTTP status codes 502, 503, and 504 - *client-error* – HTTP status code 409 - *stream-error* – Retry on refused stream
            :param tcp_retry_events: Specify a valid value. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_retry_policy_property = appmesh.CfnRoute.HttpRetryPolicyProperty(
                    max_retries=123,
                    per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                
                    # the properties below are optional
                    http_retry_events=["httpRetryEvents"],
                    tcp_retry_events=["tcpRetryEvents"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__59d95d620d5edba868375dbdd0e4d71dd7876da047462621f1ddd6efb13f49d1)
                check_type(argname="argument max_retries", value=max_retries, expected_type=type_hints["max_retries"])
                check_type(argname="argument per_retry_timeout", value=per_retry_timeout, expected_type=type_hints["per_retry_timeout"])
                check_type(argname="argument http_retry_events", value=http_retry_events, expected_type=type_hints["http_retry_events"])
                check_type(argname="argument tcp_retry_events", value=tcp_retry_events, expected_type=type_hints["tcp_retry_events"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_retries": max_retries,
                "per_retry_timeout": per_retry_timeout,
            }
            if http_retry_events is not None:
                self._values["http_retry_events"] = http_retry_events
            if tcp_retry_events is not None:
                self._values["tcp_retry_events"] = tcp_retry_events

        @builtins.property
        def max_retries(self) -> jsii.Number:
            '''The maximum number of retry attempts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-maxretries
            '''
            result = self._values.get("max_retries")
            assert result is not None, "Required property 'max_retries' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def per_retry_timeout(
            self,
        ) -> typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]:
            '''The timeout for each retry attempt.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-perretrytimeout
            '''
            result = self._values.get("per_retry_timeout")
            assert result is not None, "Required property 'per_retry_timeout' is missing"
            return typing.cast(typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def http_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the following values.

            - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
            - *gateway-error* – HTTP status codes 502, 503, and 504
            - *client-error* – HTTP status code 409
            - *stream-error* – Retry on refused stream

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-httpretryevents
            '''
            result = self._values.get("http_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tcp_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify a valid value.

            The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-tcpretryevents
            '''
            result = self._values.get("tcp_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRetryPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class HttpRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnRoute.WeightedTargetProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_action_property = appmesh.CfnRoute.HttpRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123,
                
                        # the properties below are optional
                        port=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4d83ba1e6a9a34f588ec0878ea700fedd91c989504786d803d34a349fb8827fe)
                check_type(argname="argument weighted_targets", value=weighted_targets, expected_type=type_hints["weighted_targets"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteaction.html#cfn-appmesh-route-httprouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class HttpRouteHeaderProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union[typing.Union["CfnRoute.HeaderMatchMethodProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the HTTP header in the request.

            :param name: A name for the HTTP header in the client request that will be matched on.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: The ``HeaderMatchMethod`` object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_header_property = appmesh.CfnRoute.HttpRouteHeaderProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__554668a2808800af345f2b296f187abaa6e3735b3d422d550a764165bbe41934)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the HTTP header in the client request that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HeaderMatchMethodProperty", _IResolvable_a771d0ef]]:
            '''The ``HeaderMatchMethod`` object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HeaderMatchMethodProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "headers": "headers",
            "method": "method",
            "path": "path",
            "port": "port",
            "prefix": "prefix",
            "query_parameters": "queryParameters",
            "scheme": "scheme",
        },
    )
    class HttpRouteMatchProperty:
        def __init__(
            self,
            *,
            headers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnRoute.HttpRouteHeaderProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            method: typing.Optional[builtins.str] = None,
            path: typing.Optional[typing.Union[typing.Union["CfnRoute.HttpPathMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            port: typing.Optional[jsii.Number] = None,
            prefix: typing.Optional[builtins.str] = None,
            query_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnRoute.QueryParameterProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            scheme: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the requirements for a route to match HTTP requests for a virtual router.

            :param headers: The client request headers to match on.
            :param method: The client request method to match on. Specify only one.
            :param path: The client request path to match on.
            :param port: The port number to match on.
            :param prefix: Specifies the path to match requests with. This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .
            :param query_parameters: The client request query parameters to match on.
            :param scheme: The client request scheme to match on. Specify only one. Applicable only for HTTP2 routes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_match_property = appmesh.CfnRoute.HttpRouteMatchProperty(
                    headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnRoute.MatchRangeProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    method="method",
                    path=appmesh.CfnRoute.HttpPathMatchProperty(
                        exact="exact",
                        regex="regex"
                    ),
                    port=123,
                    prefix="prefix",
                    query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                        name="name",
                
                        # the properties below are optional
                        match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                            exact="exact"
                        )
                    )],
                    scheme="scheme"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__817efb57828e59ce0db3767ae0dd4ed37122a2523e01b1d8e501446cbd2e0715)
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
                check_type(argname="argument scheme", value=scheme, expected_type=type_hints["scheme"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if headers is not None:
                self._values["headers"] = headers
            if method is not None:
                self._values["method"] = method
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port
            if prefix is not None:
                self._values["prefix"] = prefix
            if query_parameters is not None:
                self._values["query_parameters"] = query_parameters
            if scheme is not None:
                self._values["scheme"] = scheme

        @builtins.property
        def headers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.HttpRouteHeaderProperty", _IResolvable_a771d0ef]]]]:
            '''The client request headers to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.HttpRouteHeaderProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The client request method to match on.

            Specify only one.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]]:
            '''The client request path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''Specifies the path to match requests with.

            This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def query_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]]:
            '''The client request query parameters to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-queryparameters
            '''
            result = self._values.get("query_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def scheme(self) -> typing.Optional[builtins.str]:
            '''The client request scheme to match on.

            Specify only one. Applicable only for HTTP2 routes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-scheme
            '''
            result = self._values.get("scheme")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "match": "match",
            "retry_policy": "retryPolicy",
            "timeout": "timeout",
        },
    )
    class HttpRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[typing.Union["CfnRoute.HttpRouteActionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            match: typing.Union[typing.Union["CfnRoute.HttpRouteMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            retry_policy: typing.Optional[typing.Union[typing.Union["CfnRoute.HttpRetryPolicyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union[typing.Union["CfnRoute.HttpTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents an HTTP or HTTP/2 route type.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param retry_policy: An object that represents a retry policy.
            :param timeout: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_property = appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
                
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
                
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
                
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2bbedc3585b0317750fad43018a0c82e0e5557a416ecab96668fa44c741ac261)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
                check_type(argname="argument retry_policy", value=retry_policy, expected_type=type_hints["retry_policy"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }
            if retry_policy is not None:
                self._values["retry_policy"] = retry_policy
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnRoute.HttpRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnRoute.HttpRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnRoute.HttpRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnRoute.HttpRouteMatchProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def retry_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpRetryPolicyProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a retry policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-retrypolicy
            '''
            result = self._values.get("retry_policy")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpRetryPolicyProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class HttpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_timeout_property = appmesh.CfnRoute.HttpTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9d23ebeeaffbbe0eef2bb299a591d0c785f89b81aca063937e311bb2362cb1c9)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html#cfn-appmesh-route-httptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html#cfn-appmesh-route-httptimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.MatchRangeProperty",
        jsii_struct_bases=[],
        name_mapping={"end": "end", "start": "start"},
    )
    class MatchRangeProperty:
        def __init__(self, *, end: jsii.Number, start: jsii.Number) -> None:
            '''An object that represents the range of values to match on.

            The first character of the range is included in the range, though the last character is not. For example, if the range specified were 1-100, only values 1-99 would be matched.

            :param end: The end of the range.
            :param start: The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                match_range_property = appmesh.CfnRoute.MatchRangeProperty(
                    end=123,
                    start=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5379870d298594250f05e078d6fbbe93eaf49458de0fa00a4785fbbea4b78846)
                check_type(argname="argument end", value=end, expected_type=type_hints["end"])
                check_type(argname="argument start", value=start, expected_type=type_hints["start"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "end": end,
                "start": start,
            }

        @builtins.property
        def end(self) -> jsii.Number:
            '''The end of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html#cfn-appmesh-route-matchrange-end
            '''
            result = self._values.get("end")
            assert result is not None, "Required property 'end' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def start(self) -> jsii.Number:
            '''The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html#cfn-appmesh-route-matchrange-start
            '''
            result = self._values.get("start")
            assert result is not None, "Required property 'start' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MatchRangeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.QueryParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "match": "match"},
    )
    class QueryParameterProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            match: typing.Optional[typing.Union[typing.Union["CfnRoute.HttpQueryParameterMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the query parameter in the request.

            :param name: A name for the query parameter that will be matched on.
            :param match: The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                query_parameter_property = appmesh.CfnRoute.QueryParameterProperty(
                    name="name",
                
                    # the properties below are optional
                    match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                        exact="exact"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f3892bf8706e2c469f7c0118fd7f982d488ba9a44dd8448d619255807ef15a49)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the query parameter that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html#cfn-appmesh-route-queryparameter-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]]:
            '''The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html#cfn-appmesh-route-queryparameter-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.RouteSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "grpc_route": "grpcRoute",
            "http2_route": "http2Route",
            "http_route": "httpRoute",
            "priority": "priority",
            "tcp_route": "tcpRoute",
        },
    )
    class RouteSpecProperty:
        def __init__(
            self,
            *,
            grpc_route: typing.Optional[typing.Union[typing.Union["CfnRoute.GrpcRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http2_route: typing.Optional[typing.Union[typing.Union["CfnRoute.HttpRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http_route: typing.Optional[typing.Union[typing.Union["CfnRoute.HttpRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            priority: typing.Optional[jsii.Number] = None,
            tcp_route: typing.Optional[typing.Union[typing.Union["CfnRoute.TcpRouteProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a route specification.

            Specify one route type.

            :param grpc_route: An object that represents the specification of a gRPC route.
            :param http2_route: An object that represents the specification of an HTTP/2 route.
            :param http_route: An object that represents the specification of an HTTP route.
            :param priority: The priority for the route. Routes are matched based on the specified value, where 0 is the highest priority.
            :param tcp_route: An object that represents the specification of a TCP route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                route_spec_property = appmesh.CfnRoute.RouteSpecProperty(
                    grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                        action=appmesh.CfnRoute.GrpcRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                            metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method_name="methodName",
                            port=123,
                            service_name="serviceName"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            grpc_retry_events=["grpcRetryEvents"],
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http2_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    priority=123,
                    tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                        action=appmesh.CfnRoute.TcpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                
                        # the properties below are optional
                        match=appmesh.CfnRoute.TcpRouteMatchProperty(
                            port=123
                        ),
                        timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f1f078bd18103a9ad156eeae97384512f184a2e0fcae1f83169b12c2c72dd477)
                check_type(argname="argument grpc_route", value=grpc_route, expected_type=type_hints["grpc_route"])
                check_type(argname="argument http2_route", value=http2_route, expected_type=type_hints["http2_route"])
                check_type(argname="argument http_route", value=http_route, expected_type=type_hints["http_route"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
                check_type(argname="argument tcp_route", value=tcp_route, expected_type=type_hints["tcp_route"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc_route is not None:
                self._values["grpc_route"] = grpc_route
            if http2_route is not None:
                self._values["http2_route"] = http2_route
            if http_route is not None:
                self._values["http_route"] = http_route
            if priority is not None:
                self._values["priority"] = priority
            if tcp_route is not None:
                self._values["tcp_route"] = tcp_route

        @builtins.property
        def grpc_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of a gRPC route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-grpcroute
            '''
            result = self._values.get("grpc_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP/2 route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-http2route
            '''
            result = self._values.get("http2_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-httproute
            '''
            result = self._values.get("http_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority for the route.

            Routes are matched based on the specified value, where 0 is the highest priority.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tcp_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.TcpRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of a TCP route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-tcproute
            '''
            result = self._values.get("tcp_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.TcpRouteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RouteSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class TcpRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnRoute.WeightedTargetProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcprouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_route_action_property = appmesh.CfnRoute.TcpRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123,
                
                        # the properties below are optional
                        port=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1a654b11b0590065c124e1b43f2b1e60ffe7a745f2b1c4d549ac10a5b59e684)
                check_type(argname="argument weighted_targets", value=weighted_targets, expected_type=type_hints["weighted_targets"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcprouteaction.html#cfn-appmesh-route-tcprouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port"},
    )
    class TcpRouteMatchProperty:
        def __init__(self, *, port: typing.Optional[jsii.Number] = None) -> None:
            '''An object representing the TCP route to match.

            :param port: The port number to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_route_match_property = appmesh.CfnRoute.TcpRouteMatchProperty(
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8a416bae18c5b5ac482a8c74740386c3732ee46284ec4b20fc98bb7245088b17)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproutematch.html#cfn-appmesh-route-tcproutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match", "timeout": "timeout"},
    )
    class TcpRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[typing.Union["CfnRoute.TcpRouteActionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            match: typing.Optional[typing.Union[typing.Union["CfnRoute.TcpRouteMatchProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union[typing.Union["CfnRoute.TcpTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a TCP route type.

            :param action: The action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param timeout: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_route_property = appmesh.CfnRoute.TcpRouteProperty(
                    action=appmesh.CfnRoute.TcpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
                
                            # the properties below are optional
                            port=123
                        )]
                    ),
                
                    # the properties below are optional
                    match=appmesh.CfnRoute.TcpRouteMatchProperty(
                        port=123
                    ),
                    timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a29ac2b143a6d3bed344ffb870d741837ea97a873685a09edcdcd6a9393acea7)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
            }
            if match is not None:
                self._values["match"] = match
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnRoute.TcpRouteActionProperty", _IResolvable_a771d0ef]:
            '''The action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnRoute.TcpRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.TcpRouteMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.TcpRouteMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.TcpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.TcpTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle"},
    )
    class TcpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_timeout_property = appmesh.CfnRoute.TcpTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__33a5dead751cba18d09ccb37c2d90ffba3209ed587df222b9933b8848aea69b9)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcptimeout.html#cfn-appmesh-route-tcptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.WeightedTargetProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_node": "virtualNode",
            "weight": "weight",
            "port": "port",
        },
    )
    class WeightedTargetProperty:
        def __init__(
            self,
            *,
            virtual_node: builtins.str,
            weight: jsii.Number,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a target and its relative weight.

            Traffic is distributed across targets according to their relative weight. For example, a weighted target with a relative weight of 50 receives five times as much traffic as one with a relative weight of 10. The total weight for all targets combined must be less than or equal to 100.

            :param virtual_node: The virtual node to associate with the weighted target.
            :param weight: The relative weight of the weighted target.
            :param port: The targeted port of the weighted object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                weighted_target_property = appmesh.CfnRoute.WeightedTargetProperty(
                    virtual_node="virtualNode",
                    weight=123,
                
                    # the properties below are optional
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cc588218688290d3f670395a3ef9b26720ccae479e6daeae0e843bf6d91370d4)
                check_type(argname="argument virtual_node", value=virtual_node, expected_type=type_hints["virtual_node"])
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_node": virtual_node,
                "weight": weight,
            }
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def virtual_node(self) -> builtins.str:
            '''The virtual node to associate with the weighted target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-virtualnode
            '''
            result = self._values.get("virtual_node")
            assert result is not None, "Required property 'virtual_node' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def weight(self) -> jsii.Number:
            '''The relative weight of the weighted target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-weight
            '''
            result = self._values.get("weight")
            assert result is not None, "Required property 'weight' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The targeted port of the weighted object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WeightedTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_router_name": "virtualRouterName",
        "mesh_owner": "meshOwner",
        "route_name": "routeName",
        "tags": "tags",
    },
)
class CfnRouteProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union[CfnRoute.RouteSpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        virtual_router_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        route_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRoute``.

        :param mesh_name: The name of the service mesh to create the route in.
        :param spec: The route specification to apply.
        :param virtual_router_name: The name of the virtual router in which to create the route. If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param route_name: The name to use for the route.
        :param tags: Optional metadata that you can apply to the route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_route_props = appmesh.CfnRouteProps(
                mesh_name="meshName",
                spec=appmesh.CfnRoute.RouteSpecProperty(
                    grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                        action=appmesh.CfnRoute.GrpcRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                            metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method_name="methodName",
                            port=123,
                            service_name="serviceName"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            grpc_retry_events=["grpcRetryEvents"],
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http2_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    priority=123,
                    tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                        action=appmesh.CfnRoute.TcpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
            
                        # the properties below are optional
                        match=appmesh.CfnRoute.TcpRouteMatchProperty(
                            port=123
                        ),
                        timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    )
                ),
                virtual_router_name="virtualRouterName",
            
                # the properties below are optional
                mesh_owner="meshOwner",
                route_name="routeName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4c77409832b9c9f864a25fc560cd7a5c3e6482b2378d7583b20b3397c0c4b867)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument virtual_router_name", value=virtual_router_name, expected_type=type_hints["virtual_router_name"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument route_name", value=route_name, expected_type=type_hints["route_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_router_name": virtual_router_name,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if route_name is not None:
            self._values["route_name"] = route_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the route in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(self) -> typing.Union[CfnRoute.RouteSpecProperty, _IResolvable_a771d0ef]:
        '''The route specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnRoute.RouteSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router in which to create the route.

        If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-virtualroutername
        '''
        result = self._values.get("virtual_router_name")
        assert result is not None, "Required property 'virtual_router_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def route_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-routename
        '''
        result = self._values.get("route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualGateway(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualGateway",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualGateway``.

    Creates a virtual gateway.

    A virtual gateway allows resources outside your mesh to communicate to resources that are inside your mesh. The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance. Unlike a virtual node, which represents an Envoy running with an application, a virtual gateway represents Envoy deployed by itself.

    For more information about virtual gateways, see `Virtual gateways <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html>`_ .

    :cloudformationResource: AWS::AppMesh::VirtualGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_gateway = appmesh.CfnVirtualGateway(self, "MyCfnVirtualGateway",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                    port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
        
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                            max_connections=123,
        
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                            max_requests=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
        
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
        
                        # the properties below are optional
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
        
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )],
        
                # the properties below are optional
                backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
        
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
        
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                ),
                logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                    access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                            path="path",
        
                            # the properties below are optional
                            format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                )
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_gateway_name="virtualGatewayName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_gateway_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh that the virtual gateway resides in.
        :param spec: The specifications of the virtual gateway.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual gateway to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_gateway_name: The name of the virtual gateway.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b28f249a86cdb2fe87c119204677d27e3c0f970de5fa5edfd69a2aadc059425)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualGatewayProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_gateway_name=virtual_gateway_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eeba42947244afd3fb0d6c4500c989855d3418121a41a7e8667a7f664c786311)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f47bd5b625f3d81b14c6582db3d361a26987cd1fb4e17ca9f64274fd12e7336)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual gateway.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual gateway.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualGatewayName")
    def attr_virtual_gateway_name(self) -> builtins.str:
        '''The name of the virtual gateway.

        :cloudformationAttribute: VirtualGatewayName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualGatewayName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual gateway to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__04d509ed46cc1f5774ed6cd06bb64eb20c2df113d92e59a5fd47a335762e5396)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef]:
        '''The specifications of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-spec
        '''
        return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d0a623f1503a251586680074840b5b5b592aa2316cd17e0495c8aa932de5905)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d23906487db9cca775ae326d4dd254330865750e85410c3d553f39fef5a06ed2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value)

    @builtins.property
    @jsii.member(jsii_name="virtualGatewayName")
    def virtual_gateway_name(self) -> typing.Optional[builtins.str]:
        '''The name of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-virtualgatewayname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualGatewayName"))

    @virtual_gateway_name.setter
    def virtual_gateway_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__323921fb56e29e97e1433e17710d422f03f649e0d904c294ed5171359d1c3f25)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualGatewayName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.JsonFormatRefProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class JsonFormatRefProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the key value pairs for the JSON.

            :param key: The specified key for the JSON.
            :param value: The specified value for the JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-jsonformatref.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                json_format_ref_property = appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6eedff34ff0dfcb8dec61d68a99e818b8afed80925ab6f13b056c3f97c4f4abb)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The specified key for the JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-jsonformatref.html#cfn-appmesh-virtualgateway-jsonformatref-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The specified value for the JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-jsonformatref.html#cfn-appmesh-virtualgateway-jsonformatref-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JsonFormatRefProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.LoggingFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"json": "json", "text": "text"},
    )
    class LoggingFormatProperty:
        def __init__(
            self,
            *,
            json: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualGateway.JsonFormatRefProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            text: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the format for the logs.

            :param json: The logging format for JSON.
            :param text: The logging format for text.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-loggingformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                logging_format_property = appmesh.CfnVirtualGateway.LoggingFormatProperty(
                    json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                        key="key",
                        value="value"
                    )],
                    text="text"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__de56f24981644f8fb3288a464ab1f948755f91e6f49ab9a7b0c74835c7b749d6)
                check_type(argname="argument json", value=json, expected_type=type_hints["json"])
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if json is not None:
                self._values["json"] = json
            if text is not None:
                self._values["text"] = text

        @builtins.property
        def json(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualGateway.JsonFormatRefProperty", _IResolvable_a771d0ef]]]]:
            '''The logging format for JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-loggingformat.html#cfn-appmesh-virtualgateway-loggingformat-json
            '''
            result = self._values.get("json")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualGateway.JsonFormatRefProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def text(self) -> typing.Optional[builtins.str]:
            '''The logging format for text.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-loggingformat.html#cfn-appmesh-virtualgateway-loggingformat-text
            '''
            result = self._values.get("text")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class SubjectAlternativeNameMatchersProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents the methods by which a subject alternative name on a peer Transport Layer Security (TLS) certificate can be matched.

            :param exact: The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenamematchers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_name_matchers_property = appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                    exact=["exact"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f5bcf84cfb0efd48fc9c6140193d41e471238504644aaf743146ba81ab58a903)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenamematchers.html#cfn-appmesh-virtualgateway-subjectalternativenamematchers-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNameMatchersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty",
        jsii_struct_bases=[],
        name_mapping={"match": "match"},
    )
    class SubjectAlternativeNamesProperty:
        def __init__(
            self,
            *,
            match: typing.Union[typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents the subject alternative names secured by the certificate.

            :param match: An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenames.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_names_property = appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                        exact=["exact"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1042f9022956bc495c86827e5aaf6e6d4816aedf389b44f607871d4794835072)
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "match": match,
            }

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenames.html#cfn-appmesh-virtualgateway-subjectalternativenames-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNamesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file"},
    )
    class VirtualGatewayAccessLogProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The access log configuration for a virtual gateway.

            :param file: The file object to send virtual gateway access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_access_log_property = appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                        path="path",
                
                        # the properties below are optional
                        format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__af044ad5d67cb92a5c229d99a620c69a511090b395e098438cd9ada6c5bc5ca5)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", _IResolvable_a771d0ef]]:
            '''The file object to send virtual gateway access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayaccesslog-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"client_policy": "clientPolicy"},
    )
    class VirtualGatewayBackendDefaultsProperty:
        def __init__(
            self,
            *,
            client_policy: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the default properties for a backend.

            :param client_policy: A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaybackenddefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_backend_defaults_property = appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0de65e6bf0d1ff4a24ba2c43aa74fb35e17fd4fcd0cce5dfb8927562baee0512)
                check_type(argname="argument client_policy", value=client_policy, expected_type=type_hints["client_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaybackenddefaults.html#cfn-appmesh-virtualgateway-virtualgatewaybackenddefaults-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayBackendDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"tls": "tls"},
    )
    class VirtualGatewayClientPolicyProperty:
        def __init__(
            self,
            *,
            tls: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a client policy.

            :param tls: A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_client_policy_property = appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                    certificate_authority_arns=["certificateAuthorityArns"]
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        ),
                
                        # the properties below are optional
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        enforce=False,
                        ports=[123]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__91649bf32fbff64f22b01fbf3e03a13ac43ce80793bfb8689122f0cbac5925a3)
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicy-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "validation": "validation",
            "certificate": "certificate",
            "enforce": "enforce",
            "ports": "ports",
        },
    )
    class VirtualGatewayClientPolicyTlsProperty:
        def __init__(
            self,
            *,
            validation: typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            certificate: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            enforce: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            ports: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[jsii.Number]]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) client policy.

            :param validation: A reference to an object that represents a Transport Layer Security (TLS) validation context.
            :param certificate: A reference to an object that represents a virtual gateway's client's Transport Layer Security (TLS) certificate.
            :param enforce: Whether the policy is enforced. The default is ``True`` , if a value isn't specified.
            :param ports: One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_client_policy_tls_property = appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                    validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                certificate_authority_arns=["certificateAuthorityArns"]
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    ),
                
                    # the properties below are optional
                    certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    enforce=False,
                    ports=[123]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a0287c04f0a09bbc4965a72f54e3b9368188106b4a2b8c7d5098721049fc5f73)
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument enforce", value=enforce, expected_type=type_hints["enforce"])
                check_type(argname="argument ports", value=ports, expected_type=type_hints["ports"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "validation": validation,
            }
            if certificate is not None:
                self._values["certificate"] = certificate
            if enforce is not None:
                self._values["enforce"] = enforce
            if ports is not None:
                self._values["ports"] = ports

        @builtins.property
        def validation(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", _IResolvable_a771d0ef]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-validation
            '''
            result = self._values.get("validation")
            assert result is not None, "Required property 'validation' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def certificate(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's client's Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-certificate
            '''
            result = self._values.get("certificate")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def enforce(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the policy is enforced.

            The default is ``True`` , if a value isn't specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-enforce
            '''
            result = self._values.get("enforce")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def ports(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]]:
            '''One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-ports
            '''
            result = self._values.get("ports")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientPolicyTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class VirtualGatewayClientTlsCertificateProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the virtual gateway's client's Transport Layer Security (TLS) certificate.

            :param file: An object that represents a local file certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .
            :param sds: A reference to an object that represents a virtual gateway's client's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_client_tls_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cc97126ded315125e422b98a6eea29d22e673e4afbd066f2308dcd43ca737f52)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewayclienttlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's client's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewayclienttlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2"},
    )
    class VirtualGatewayConnectionPoolProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http2: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the type of virtual gateway connection pool.

            Only one protocol is used at a time and should be the same protocol as the one chosen under port mapping.

            If not present the default value for ``maxPendingRequests`` is ``2147483647`` .

            :param grpc: An object that represents a type of connection pool.
            :param http: An object that represents a type of connection pool.
            :param http2: An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                    grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                        max_requests=123
                    ),
                    http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                        max_connections=123,
                
                        # the properties below are optional
                        max_pending_requests=123
                    ),
                    http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                        max_requests=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5ff03f4f6ddf2205b7ec5c1778ee2871df58b281d33928167963356c0340b79e)
                check_type(argname="argument grpc", value=grpc, expected_type=type_hints["grpc"])
                check_type(argname="argument http", value=http, expected_type=type_hints["http"])
                check_type(argname="argument http2", value=http2, expected_type=type_hints["http2"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path", "format": "format"},
    )
    class VirtualGatewayFileAccessLogProperty:
        def __init__(
            self,
            *,
            path: builtins.str,
            format: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.LoggingFormatProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents an access log file.

            :param path: The file path to write access logs to. You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.
            :param format: The specified format for the virtual gateway access logs. It can be either ``json_format`` or ``text_format`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_file_access_log_property = appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                    path="path",
                
                    # the properties below are optional
                    format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                        json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                            key="key",
                            value="value"
                        )],
                        text="text"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ce7b8c18f465c410a596e422857f7c14901d40df0c697f0c9de86306dabe563a)
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "path": path,
            }
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def path(self) -> builtins.str:
            '''The file path to write access logs to.

            You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayfileaccesslog-path
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def format(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.LoggingFormatProperty", _IResolvable_a771d0ef]]:
            '''The specified format for the virtual gateway access logs.

            It can be either ``json_format`` or ``text_format`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayfileaccesslog-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.LoggingFormatProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayFileAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualGatewayGrpcConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_grpc_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cd928303ffb3114f4715ed52005249d98c9cfd3a9bac9d8c1e13bb91f653edff)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayGrpcConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "healthy_threshold": "healthyThreshold",
            "interval_millis": "intervalMillis",
            "protocol": "protocol",
            "timeout_millis": "timeoutMillis",
            "unhealthy_threshold": "unhealthyThreshold",
            "path": "path",
            "port": "port",
        },
    )
    class VirtualGatewayHealthCheckPolicyProperty:
        def __init__(
            self,
            *,
            healthy_threshold: jsii.Number,
            interval_millis: jsii.Number,
            protocol: builtins.str,
            timeout_millis: jsii.Number,
            unhealthy_threshold: jsii.Number,
            path: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents the health check policy for a virtual gateway's listener.

            :param healthy_threshold: The number of consecutive successful health checks that must occur before declaring the listener healthy.
            :param interval_millis: The time period in milliseconds between each health check execution.
            :param protocol: The protocol for the health check request. If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .
            :param timeout_millis: The amount of time to wait when receiving a response from the health check, in milliseconds.
            :param unhealthy_threshold: The number of consecutive failed health checks that must occur before declaring a virtual gateway unhealthy.
            :param path: The destination path for the health check request. This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.
            :param port: The destination port for the health check request. This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_health_check_policy_property = appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
                
                    # the properties below are optional
                    path="path",
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a906d13da437d213977f480bf67613e9108c1d733a174042e2588587e8e866df)
                check_type(argname="argument healthy_threshold", value=healthy_threshold, expected_type=type_hints["healthy_threshold"])
                check_type(argname="argument interval_millis", value=interval_millis, expected_type=type_hints["interval_millis"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
                check_type(argname="argument timeout_millis", value=timeout_millis, expected_type=type_hints["timeout_millis"])
                check_type(argname="argument unhealthy_threshold", value=unhealthy_threshold, expected_type=type_hints["unhealthy_threshold"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "healthy_threshold": healthy_threshold,
                "interval_millis": interval_millis,
                "protocol": protocol,
                "timeout_millis": timeout_millis,
                "unhealthy_threshold": unhealthy_threshold,
            }
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def healthy_threshold(self) -> jsii.Number:
            '''The number of consecutive successful health checks that must occur before declaring the listener healthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-healthythreshold
            '''
            result = self._values.get("healthy_threshold")
            assert result is not None, "Required property 'healthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def interval_millis(self) -> jsii.Number:
            '''The time period in milliseconds between each health check execution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-intervalmillis
            '''
            result = self._values.get("interval_millis")
            assert result is not None, "Required property 'interval_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol for the health check request.

            If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def timeout_millis(self) -> jsii.Number:
            '''The amount of time to wait when receiving a response from the health check, in milliseconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-timeoutmillis
            '''
            result = self._values.get("timeout_millis")
            assert result is not None, "Required property 'timeout_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def unhealthy_threshold(self) -> jsii.Number:
            '''The number of consecutive failed health checks that must occur before declaring a virtual gateway unhealthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-unhealthythreshold
            '''
            result = self._values.get("unhealthy_threshold")
            assert result is not None, "Required property 'unhealthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The destination path for the health check request.

            This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The destination port for the health check request.

            This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHealthCheckPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualGatewayHttp2ConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttp2connectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_http2_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b4508bd000cb09164fb32cdb024398f143608fd2c9acbd8c79e7c6040111afaa)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttp2connectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttp2connectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHttp2ConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_connections": "maxConnections",
            "max_pending_requests": "maxPendingRequests",
        },
    )
    class VirtualGatewayHttpConnectionPoolProperty:
        def __init__(
            self,
            *,
            max_connections: jsii.Number,
            max_pending_requests: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.
            :param max_pending_requests: Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_http_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                    max_connections=123,
                
                    # the properties below are optional
                    max_pending_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__104420ac4c9654ac2d737aef555717a5b9351beb6b578fe8f4945d8b3f253431)
                check_type(argname="argument max_connections", value=max_connections, expected_type=type_hints["max_connections"])
                check_type(argname="argument max_pending_requests", value=max_pending_requests, expected_type=type_hints["max_pending_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_connections": max_connections,
            }
            if max_pending_requests is not None:
                self._values["max_pending_requests"] = max_pending_requests

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_pending_requests(self) -> typing.Optional[jsii.Number]:
            '''Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttpconnectionpool-maxpendingrequests
            '''
            result = self._values.get("max_pending_requests")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHttpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_mapping": "portMapping",
            "connection_pool": "connectionPool",
            "health_check": "healthCheck",
            "tls": "tls",
        },
    )
    class VirtualGatewayListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            connection_pool: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            health_check: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            tls: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener for a virtual gateway.

            :param port_mapping: The port mapping information for the listener.
            :param connection_pool: The connection pool information for the listener.
            :param health_check: The health check information for the listener.
            :param tls: A reference to an object that represents the Transport Layer Security (TLS) properties for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                    port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
                
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                            max_connections=123,
                
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                            max_requests=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
                
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
                
                        # the properties below are optional
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__34096b30c560e97c9dbc2d5f41bf2ac6401ea7c129760baf59765bf735b5b17c)
                check_type(argname="argument port_mapping", value=port_mapping, expected_type=type_hints["port_mapping"])
                check_type(argname="argument connection_pool", value=connection_pool, expected_type=type_hints["connection_pool"])
                check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_mapping": port_mapping,
            }
            if connection_pool is not None:
                self._values["connection_pool"] = connection_pool
            if health_check is not None:
                self._values["health_check"] = health_check
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", _IResolvable_a771d0ef]:
            '''The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def connection_pool(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''The connection pool information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-connectionpool
            '''
            result = self._values.get("connection_pool")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", _IResolvable_a771d0ef]]:
            '''The health check information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the Transport Layer Security (TLS) properties for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class VirtualGatewayListenerTlsAcmCertificateProperty:
        def __init__(self, *, certificate_arn: builtins.str) -> None:
            '''An object that represents an AWS Certificate Manager certificate.

            :param certificate_arn: The Amazon Resource Name (ARN) for the certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_acm_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                    certificate_arn="certificateArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__31dbbc2366dd0630fd49cbbd9b3653766504817aaf932457c0619ce83fc10404)
                check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_arn": certificate_arn,
            }

        @builtins.property
        def certificate_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) for the certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate-certificatearn
            '''
            result = self._values.get("certificate_arn")
            assert result is not None, "Required property 'certificate_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsAcmCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class VirtualGatewayListenerTlsCertificateProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) certificate.

            :param acm: A reference to an object that represents an AWS Certificate Manager certificate.
            :param file: A reference to an object that represents a local file certificate.
            :param sds: A reference to an object that represents a virtual gateway's listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                    acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                        certificate_arn="certificateArn"
                    ),
                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f275dc3b3fb467dd4969026294feb0726463b20eea0858b01b144f72bbb663a8)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a local file certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_chain": "certificateChain",
            "private_key": "privateKey",
        },
    )
    class VirtualGatewayListenerTlsFileCertificateProperty:
        def __init__(
            self,
            *,
            certificate_chain: builtins.str,
            private_key: builtins.str,
        ) -> None:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :param certificate_chain: The certificate chain for the certificate.
            :param private_key: The private key for a certificate stored on the file system of the mesh endpoint that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_file_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                    certificate_chain="certificateChain",
                    private_key="privateKey"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f0576d7748859785bf71d3470d1da2829dfdc60e949b32001d5dde54ec67dfe)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
                check_type(argname="argument private_key", value=private_key, expected_type=type_hints["private_key"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
                "private_key": private_key,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate chain for the certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def private_key(self) -> builtins.str:
            '''The private key for a certificate stored on the file system of the mesh endpoint that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate-privatekey
            '''
            result = self._values.get("private_key")
            assert result is not None, "Required property 'private_key' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsFileCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate": "certificate",
            "mode": "mode",
            "validation": "validation",
        },
    )
    class VirtualGatewayListenerTlsProperty:
        def __init__(
            self,
            *,
            certificate: typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            mode: builtins.str,
            validation: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the Transport Layer Security (TLS) properties for a listener.

            :param certificate: An object that represents a Transport Layer Security (TLS) certificate.
            :param mode: Specify one of the following modes. - ** STRICT – Listener only accepts connections with TLS enabled. - ** PERMISSIVE – Listener accepts connections with or without TLS enabled. - ** DISABLED – Listener only accepts connections without TLS.
            :param validation: A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                    certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                        acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                            certificate_arn="certificateArn"
                        ),
                        file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    mode="mode",
                
                    # the properties below are optional
                    validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e025f609eaa2312df97e8d153867a96f71ccee374ec98adc0798e2eb8ba56ae3)
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate": certificate,
                "mode": mode,
            }
            if validation is not None:
                self._values["validation"] = validation

        @builtins.property
        def certificate(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", _IResolvable_a771d0ef]:
            '''An object that represents a Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-certificate
            '''
            result = self._values.get("certificate")
            assert result is not None, "Required property 'certificate' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def mode(self) -> builtins.str:
            '''Specify one of the following modes.

            - ** STRICT – Listener only accepts connections with TLS enabled.
            - ** PERMISSIVE – Listener accepts connections with or without TLS enabled.
            - ** DISABLED – Listener only accepts connections without TLS.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-mode
            '''
            result = self._values.get("mode")
            assert result is not None, "Required property 'mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def validation(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-validation
            '''
            result = self._values.get("validation")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class VirtualGatewayListenerTlsSdsCertificateProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents the virtual gateway's listener's Secret Discovery Service certificate.The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_sds_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__05682c23ffdddfa9f60139a4771164cd7589b674af3beef04a2a81b7f1cafdef)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsSdsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class VirtualGatewayListenerTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a virtual gateway listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_validation_context_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fc9f808f5eaedbfa1d5e7766c395bc25be788bfc31ea90a4cdcc6ddcd55d4bd4)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a virtual gateway listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class VirtualGatewayListenerTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context trust.

            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_validation_context_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ab411f29a08e5595b8020b731f27cba97b9017b92e1c195e4407d3c8e6a9cbf9)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"access_log": "accessLog"},
    )
    class VirtualGatewayLoggingProperty:
        def __init__(
            self,
            *,
            access_log: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents logging information.

            :param access_log: The access log configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylogging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_logging_property = appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                    access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                            path="path",
                
                            # the properties below are optional
                            format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__765ba8ae5b6c4f1c3ecd6e92b974467fca1bcafc890c61e753947aca7091349f)
                check_type(argname="argument access_log", value=access_log, expected_type=type_hints["access_log"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_log is not None:
                self._values["access_log"] = access_log

        @builtins.property
        def access_log(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", _IResolvable_a771d0ef]]:
            '''The access log configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylogging.html#cfn-appmesh-virtualgateway-virtualgatewaylogging-accesslog
            '''
            result = self._values.get("access_log")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayLoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class VirtualGatewayPortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object that represents a port mapping.

            :param port: The port used for the port mapping. Specify one protocol.
            :param protocol: The protocol used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_port_mapping_property = appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1a781edd6a719067db60754fa0733665f2cec3b277c4364a0beebe4ae117d29d)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            Specify one protocol.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html#cfn-appmesh-virtualgateway-virtualgatewayportmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html#cfn-appmesh-virtualgateway-virtualgatewayportmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayPortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "listeners": "listeners",
            "backend_defaults": "backendDefaults",
            "logging": "logging",
        },
    )
    class VirtualGatewaySpecProperty:
        def __init__(
            self,
            *,
            listeners: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]],
            backend_defaults: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            logging: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a service mesh resource.

            :param listeners: The listeners that the mesh endpoint is expected to receive inbound traffic from. You can specify one listener.
            :param backend_defaults: A reference to an object that represents the defaults for backends.
            :param logging: An object that represents logging information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_spec_property = appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                    listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                        port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
                
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                                max_connections=123,
                
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                                max_requests=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
                
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
                
                            # the properties below are optional
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                
                    # the properties below are optional
                    backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                            tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                        access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                                path="path",
                
                                # the properties below are optional
                                format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d7819dac888ef9fe6f22f88b2c8392a1a27d8d26c53737707bfbd1f905fb8233)
                check_type(argname="argument listeners", value=listeners, expected_type=type_hints["listeners"])
                check_type(argname="argument backend_defaults", value=backend_defaults, expected_type=type_hints["backend_defaults"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "listeners": listeners,
            }
            if backend_defaults is not None:
                self._values["backend_defaults"] = backend_defaults
            if logging is not None:
                self._values["logging"] = logging

        @builtins.property
        def listeners(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", _IResolvable_a771d0ef]]]:
            '''The listeners that the mesh endpoint is expected to receive inbound traffic from.

            You can specify one listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-listeners
            '''
            result = self._values.get("listeners")
            assert result is not None, "Required property 'listeners' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", _IResolvable_a771d0ef]]], result)

        @builtins.property
        def backend_defaults(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the defaults for backends.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-backenddefaults
            '''
            result = self._values.get("backend_defaults")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", _IResolvable_a771d0ef]]:
            '''An object that represents logging information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewaySpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_authority_arns": "certificateAuthorityArns"},
    )
    class VirtualGatewayTlsValidationContextAcmTrustProperty:
        def __init__(
            self,
            *,
            certificate_authority_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :param certificate_authority_arns: One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_acm_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                    certificate_authority_arns=["certificateAuthorityArns"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c73ff5f801a66b6722a204c1e8a373ab1954fe9fd84530ea25d88cf829afda06)
                check_type(argname="argument certificate_authority_arns", value=certificate_authority_arns, expected_type=type_hints["certificate_authority_arns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_authority_arns": certificate_authority_arns,
            }

        @builtins.property
        def certificate_authority_arns(self) -> typing.List[builtins.str]:
            '''One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust-certificateauthorityarns
            '''
            result = self._values.get("certificate_authority_arns")
            assert result is not None, "Required property 'certificate_authority_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextAcmTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_chain": "certificateChain"},
    )
    class VirtualGatewayTlsValidationContextFileTrustProperty:
        def __init__(self, *, certificate_chain: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :param certificate_chain: The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_file_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                    certificate_chain="certificateChain"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9414b2e53854f606e3217bb58d178729adc6d2ae242b39d3e3afe1bbf6a95370)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextFileTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class VirtualGatewayTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                            certificate_authority_arns=["certificateAuthorityArns"]
                        ),
                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__00f82e22a84b0ac2e39756f579abdaef4dba486389d95dcb1732894bffbd7faf)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class VirtualGatewayTlsValidationContextSdsTrustProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret for a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_sds_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__75015bfd91ee4306ac60a11efebab6d8221830ebd0c02e711d0494d870ebdf67)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret for a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextSdsTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class VirtualGatewayTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust.

            :param acm: A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.
            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                        certificate_authority_arns=["certificateAuthorityArns"]
                    ),
                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__eedaee15e3597472473fbfc73db590783f48df467064adca2238f3ab765fe957)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_gateway_name": "virtualGatewayName",
    },
)
class CfnVirtualGatewayProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_gateway_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualGateway``.

        :param mesh_name: The name of the service mesh that the virtual gateway resides in.
        :param spec: The specifications of the virtual gateway.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual gateway to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_gateway_name: The name of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_gateway_props = appmesh.CfnVirtualGatewayProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                    listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                        port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
            
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                                max_connections=123,
            
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                                max_requests=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
            
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
            
                            # the properties below are optional
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
            
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
            
                    # the properties below are optional
                    backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                            tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                        access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                                path="path",
            
                                # the properties below are optional
                                format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    )
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_gateway_name="virtualGatewayName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f1856dae6e90e0fd16a8aa3b02626d03e58b34ce62844715809007a4fa131eb)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_gateway_name", value=virtual_gateway_name, expected_type=type_hints["virtual_gateway_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_gateway_name is not None:
            self._values["virtual_gateway_name"] = virtual_gateway_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, _IResolvable_a771d0ef]:
        '''The specifications of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual gateway to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def virtual_gateway_name(self) -> typing.Optional[builtins.str]:
        '''The name of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-virtualgatewayname
        '''
        result = self._values.get("virtual_gateway_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualNode(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualNode",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualNode``.

    Creates a virtual node within a service mesh.

    A virtual node acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment. When you create a virtual node, you can specify the service discovery information for your task group, and whether the proxy running in a task group will communicate with other proxies using Transport Layer Security (TLS).

    You define a ``listener`` for any inbound traffic that your virtual node expects. Any virtual service that your virtual node expects to communicate to is specified as a ``backend`` .

    The response metadata for your new virtual node contains the ``arn`` that is associated with the virtual node. Set this value to the full ARN; for example, ``arn:aws:appmesh:us-west-2:123456789012:myMesh/default/virtualNode/myApp`` ) as the ``APPMESH_RESOURCE_ARN`` environment variable for your task group's Envoy proxy container in your task definition or pod spec. This is then mapped to the ``node.id`` and ``node.cluster`` Envoy parameters.
    .. epigraph::

       By default, App Mesh uses the name of the resource you specified in ``APPMESH_RESOURCE_ARN`` when Envoy is referring to itself in metrics and traces. You can override this behavior by setting the ``APPMESH_RESOURCE_CLUSTER`` environment variable with your own name.

    For more information about virtual nodes, see `Virtual nodes <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html>`_ . You must be using ``1.15.0`` or later of the Envoy image when setting these variables. For more information about App Mesh Envoy variables, see `Envoy image <https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html>`_ in the AWS App Mesh User Guide.

    :cloudformationResource: AWS::AppMesh::VirtualNode
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_node = appmesh.CfnVirtualNode(self, "MyCfnVirtualNode",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
        
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
        
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                ),
                backends=[appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
        
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
        
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
        
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )],
                listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                    port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
        
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                            max_connections=123,
        
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                            max_requests=123
                        ),
                        tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                            max_connections=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
        
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                        base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        interval=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        max_ejection_percent=123,
                        max_server_errors=123
                    ),
                    timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                        grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                        certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
        
                        # the properties below are optional
                        validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
        
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )],
                logging=appmesh.CfnVirtualNode.LoggingProperty(
                    access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                        file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                            path="path",
        
                            # the properties below are optional
                            format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                ),
                service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                    aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                        namespace_name="namespaceName",
                        service_name="serviceName",
        
                        # the properties below are optional
                        attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                            key="key",
                            value="value"
                        )],
                        ip_preference="ipPreference"
                    ),
                    dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                        hostname="hostname",
        
                        # the properties below are optional
                        ip_preference="ipPreference",
                        response_type="responseType"
                    )
                )
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_node_name="virtualNodeName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_node_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualNode``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the virtual node in.
        :param spec: The virtual node specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual node to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_node_name: The name to use for the virtual node.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4842e2391f91264b5651295c5de791aaa9f8eeaabcbd280498988d9362871ac5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualNodeProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_node_name=virtual_node_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82a337ed97bd9c6b1f31dc77ca29e0d0b8e5325a56d39afdbed1f7744deb4ac2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f46aeaf7f5facf416c3d9b15f24590d3862a7510bde35ad592d1bc2a6d135714)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual node.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual node resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual node.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualNodeName")
    def attr_virtual_node_name(self) -> builtins.str:
        '''The name of the virtual node.

        :cloudformationAttribute: VirtualNodeName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualNodeName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual node to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual node in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0963c931d673293dfdae8d0d5366684b1a34fa9a942d9533801e502e2d7b9cc1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef]:
        '''The virtual node specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-spec
        '''
        return typing.cast(typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4cc07175adfc21c01536f8082df97f6dda2f43d7171c3657f37b4783b71578fd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb4b891596f217759454ef8f629ef32b055bd7f9f1d919df1de1769102965d1f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value)

    @builtins.property
    @jsii.member(jsii_name="virtualNodeName")
    def virtual_node_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual node.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-virtualnodename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualNodeName"))

    @virtual_node_name.setter
    def virtual_node_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9018a2a1209de40d147f8a2ffd57e36bdfaa0417e1983d3f79f6da0a04400d2c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualNodeName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.AccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file"},
    )
    class AccessLogProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.FileAccessLogProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the access logging information for a virtual node.

            :param file: The file object to send virtual node access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-accesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                access_log_property = appmesh.CfnVirtualNode.AccessLogProperty(
                    file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                        path="path",
                
                        # the properties below are optional
                        format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__029b2c8a419aa5d4108edff61df68210dc2ee2eaf2ed02e16349d468a415faea)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.FileAccessLogProperty", _IResolvable_a771d0ef]]:
            '''The file object to send virtual node access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-accesslog.html#cfn-appmesh-virtualnode-accesslog-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.FileAccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class AwsCloudMapInstanceAttributeProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the AWS Cloud Map attribute information for your virtual node.

            .. epigraph::

               AWS Cloud Map is not available in the eu-south-1 Region.

            :param key: The name of an AWS Cloud Map service instance attribute key. Any AWS Cloud Map service instance that contains the specified key and value is returned.
            :param value: The value of an AWS Cloud Map service instance attribute key. Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                aws_cloud_map_instance_attribute_property = appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a79c75a75e34f654ec2016d267ed8b40df39c0d5ae81317dc5db22ce05f9c570)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The name of an AWS Cloud Map service instance attribute key.

            Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html#cfn-appmesh-virtualnode-awscloudmapinstanceattribute-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value of an AWS Cloud Map service instance attribute key.

            Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html#cfn-appmesh-virtualnode-awscloudmapinstanceattribute-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsCloudMapInstanceAttributeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "namespace_name": "namespaceName",
            "service_name": "serviceName",
            "attributes": "attributes",
            "ip_preference": "ipPreference",
        },
    )
    class AwsCloudMapServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            namespace_name: builtins.str,
            service_name: builtins.str,
            attributes: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            ip_preference: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the AWS Cloud Map service discovery information for your virtual node.

            .. epigraph::

               AWS Cloud Map is not available in the eu-south-1 Region.

            :param namespace_name: The name of the AWS Cloud Map namespace to use.
            :param service_name: The name of the AWS Cloud Map service to use.
            :param attributes: A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance. Only instances that match all of the specified key/value pairs will be returned.
            :param ip_preference: The preferred IP version that this virtual node uses. Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                aws_cloud_map_service_discovery_property = appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                    namespace_name="namespaceName",
                    service_name="serviceName",
                
                    # the properties below are optional
                    attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                        key="key",
                        value="value"
                    )],
                    ip_preference="ipPreference"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b99bcec927ab1785f6bc191d604623dbf2bd66f4fcb1984b6f1235c82dc0f9f3)
                check_type(argname="argument namespace_name", value=namespace_name, expected_type=type_hints["namespace_name"])
                check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
                check_type(argname="argument attributes", value=attributes, expected_type=type_hints["attributes"])
                check_type(argname="argument ip_preference", value=ip_preference, expected_type=type_hints["ip_preference"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "namespace_name": namespace_name,
                "service_name": service_name,
            }
            if attributes is not None:
                self._values["attributes"] = attributes
            if ip_preference is not None:
                self._values["ip_preference"] = ip_preference

        @builtins.property
        def namespace_name(self) -> builtins.str:
            '''The name of the AWS Cloud Map namespace to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-namespacename
            '''
            result = self._values.get("namespace_name")
            assert result is not None, "Required property 'namespace_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def service_name(self) -> builtins.str:
            '''The name of the AWS Cloud Map service to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-servicename
            '''
            result = self._values.get("service_name")
            assert result is not None, "Required property 'service_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def attributes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", _IResolvable_a771d0ef]]]]:
            '''A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance.

            Only instances that match all of the specified key/value pairs will be returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-attributes
            '''
            result = self._values.get("attributes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def ip_preference(self) -> typing.Optional[builtins.str]:
            '''The preferred IP version that this virtual node uses.

            Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-ippreference
            '''
            result = self._values.get("ip_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsCloudMapServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.BackendDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"client_policy": "clientPolicy"},
    )
    class BackendDefaultsProperty:
        def __init__(
            self,
            *,
            client_policy: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ClientPolicyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the default properties for a backend.

            :param client_policy: A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backenddefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                backend_defaults_property = appmesh.CfnVirtualNode.BackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c3554eb6b4ded594d38f7b74bfbb69eaa0261d20675de34dac4bf24c7fe6204a)
                check_type(argname="argument client_policy", value=client_policy, expected_type=type_hints["client_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backenddefaults.html#cfn-appmesh-virtualnode-backenddefaults-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BackendDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.BackendProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service": "virtualService"},
    )
    class BackendProperty:
        def __init__(
            self,
            *,
            virtual_service: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the backends that a virtual node is expected to send outbound traffic to.

            :param virtual_service: Specifies a virtual service to use as a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backend.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                backend_property = appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
                
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dc1fea87b4b5d0a6e0c64b995201a04611d0c142cd902b2dd007f37e65daccd2)
                check_type(argname="argument virtual_service", value=virtual_service, expected_type=type_hints["virtual_service"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if virtual_service is not None:
                self._values["virtual_service"] = virtual_service

        @builtins.property
        def virtual_service(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", _IResolvable_a771d0ef]]:
            '''Specifies a virtual service to use as a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backend.html#cfn-appmesh-virtualnode-backend-virtualservice
            '''
            result = self._values.get("virtual_service")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BackendProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ClientPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"tls": "tls"},
    )
    class ClientPolicyProperty:
        def __init__(
            self,
            *,
            tls: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a client policy.

            :param tls: A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                client_policy_property = appmesh.CfnVirtualNode.ClientPolicyProperty(
                    tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                        validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                    certificate_authority_arns=["certificateAuthorityArns"]
                                ),
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        ),
                
                        # the properties below are optional
                        certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        enforce=False,
                        ports=[123]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__28359fcf9f2e613099e7d05f5c4fd6917ac100174cedc16de9ecc58f22ecc349)
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicy.html#cfn-appmesh-virtualnode-clientpolicy-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ClientPolicyTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "validation": "validation",
            "certificate": "certificate",
            "enforce": "enforce",
            "ports": "ports",
        },
    )
    class ClientPolicyTlsProperty:
        def __init__(
            self,
            *,
            validation: typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            certificate: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            enforce: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            ports: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[jsii.Number]]] = None,
        ) -> None:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :param validation: A reference to an object that represents a TLS validation context.
            :param certificate: A reference to an object that represents a client's TLS certificate.
            :param enforce: Whether the policy is enforced. The default is ``True`` , if a value isn't specified.
            :param ports: One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                client_policy_tls_property = appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                certificate_authority_arns=["certificateAuthorityArns"]
                            ),
                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    ),
                
                    # the properties below are optional
                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    enforce=False,
                    ports=[123]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9bc651729be38f98b3bb82ff6dfc2c41f702d2666f0029d25d160bb5e3c61864)
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument enforce", value=enforce, expected_type=type_hints["enforce"])
                check_type(argname="argument ports", value=ports, expected_type=type_hints["ports"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "validation": validation,
            }
            if certificate is not None:
                self._values["certificate"] = certificate
            if enforce is not None:
                self._values["enforce"] = enforce
            if ports is not None:
                self._values["ports"] = ports

        @builtins.property
        def validation(
            self,
        ) -> typing.Union["CfnVirtualNode.TlsValidationContextProperty", _IResolvable_a771d0ef]:
            '''A reference to an object that represents a TLS validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-validation
            '''
            result = self._values.get("validation")
            assert result is not None, "Required property 'validation' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.TlsValidationContextProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def certificate(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client's TLS certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-certificate
            '''
            result = self._values.get("certificate")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def enforce(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the policy is enforced.

            The default is ``True`` , if a value isn't specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-enforce
            '''
            result = self._values.get("enforce")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def ports(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]]:
            '''One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-ports
            '''
            result = self._values.get("ports")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientPolicyTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ClientTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class ClientTlsCertificateProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the client's certificate.

            :param file: An object that represents a local file certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .
            :param sds: A reference to an object that represents a client's TLS Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                client_tls_certificate_property = appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9c48126e3287836df5adbfb9602fdf5420d80b2843e879452f3bd08f00f2cd89)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html#cfn-appmesh-virtualnode-clienttlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client's TLS Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html#cfn-appmesh-virtualnode-clienttlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hostname": "hostname",
            "ip_preference": "ipPreference",
            "response_type": "responseType",
        },
    )
    class DnsServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            hostname: builtins.str,
            ip_preference: typing.Optional[builtins.str] = None,
            response_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the DNS service discovery information for your virtual node.

            :param hostname: Specifies the DNS service discovery hostname for the virtual node.
            :param ip_preference: The preferred IP version that this virtual node uses. Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.
            :param response_type: Specifies the DNS response type for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                dns_service_discovery_property = appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                    hostname="hostname",
                
                    # the properties below are optional
                    ip_preference="ipPreference",
                    response_type="responseType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8eae89440187bd0a03ded3a30b175df51ee853d5f942d3932dc3c430c432cfc1)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument ip_preference", value=ip_preference, expected_type=type_hints["ip_preference"])
                check_type(argname="argument response_type", value=response_type, expected_type=type_hints["response_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "hostname": hostname,
            }
            if ip_preference is not None:
                self._values["ip_preference"] = ip_preference
            if response_type is not None:
                self._values["response_type"] = response_type

        @builtins.property
        def hostname(self) -> builtins.str:
            '''Specifies the DNS service discovery hostname for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-hostname
            '''
            result = self._values.get("hostname")
            assert result is not None, "Required property 'hostname' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ip_preference(self) -> typing.Optional[builtins.str]:
            '''The preferred IP version that this virtual node uses.

            Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-ippreference
            '''
            result = self._values.get("ip_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def response_type(self) -> typing.Optional[builtins.str]:
            '''Specifies the DNS response type for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-responsetype
            '''
            result = self._values.get("response_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DnsServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.DurationProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class DurationProperty:
        def __init__(self, *, unit: builtins.str, value: jsii.Number) -> None:
            '''An object that represents a duration of time.

            :param unit: A unit of time.
            :param value: A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                duration_property = appmesh.CfnVirtualNode.DurationProperty(
                    unit="unit",
                    value=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b7c813c58f775555c1de88ba2209034a183a348d5d14bc4c65e6ad067af20547)
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "unit": unit,
                "value": value,
            }

        @builtins.property
        def unit(self) -> builtins.str:
            '''A unit of time.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html#cfn-appmesh-virtualnode-duration-unit
            '''
            result = self._values.get("unit")
            assert result is not None, "Required property 'unit' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> jsii.Number:
            '''A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html#cfn-appmesh-virtualnode-duration-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.FileAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path", "format": "format"},
    )
    class FileAccessLogProperty:
        def __init__(
            self,
            *,
            path: builtins.str,
            format: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.LoggingFormatProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents an access log file.

            :param path: The file path to write access logs to. You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk. .. epigraph:: The Envoy process must have write permissions to the path that you specify here. Otherwise, Envoy fails to bootstrap properly.
            :param format: The specified format for the logs. The format is either ``json_format`` or ``text_format`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                file_access_log_property = appmesh.CfnVirtualNode.FileAccessLogProperty(
                    path="path",
                
                    # the properties below are optional
                    format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                        json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                            key="key",
                            value="value"
                        )],
                        text="text"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3f5814e27605ddbbce2a24a764564164f32addfc169d650b40e42adcc40eaca4)
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "path": path,
            }
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def path(self) -> builtins.str:
            '''The file path to write access logs to.

            You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.
            .. epigraph::

               The Envoy process must have write permissions to the path that you specify here. Otherwise, Envoy fails to bootstrap properly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html#cfn-appmesh-virtualnode-fileaccesslog-path
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def format(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.LoggingFormatProperty", _IResolvable_a771d0ef]]:
            '''The specified format for the logs.

            The format is either ``json_format`` or ``text_format`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html#cfn-appmesh-virtualnode-fileaccesslog-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.LoggingFormatProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FileAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.GrpcTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class GrpcTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_timeout_property = appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__687ffcdfe55552696c0ba7746f5b8cedc6c928ab8e3f2a8381584a382f8edb19)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html#cfn-appmesh-virtualnode-grpctimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html#cfn-appmesh-virtualnode-grpctimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.HealthCheckProperty",
        jsii_struct_bases=[],
        name_mapping={
            "healthy_threshold": "healthyThreshold",
            "interval_millis": "intervalMillis",
            "protocol": "protocol",
            "timeout_millis": "timeoutMillis",
            "unhealthy_threshold": "unhealthyThreshold",
            "path": "path",
            "port": "port",
        },
    )
    class HealthCheckProperty:
        def __init__(
            self,
            *,
            healthy_threshold: jsii.Number,
            interval_millis: jsii.Number,
            protocol: builtins.str,
            timeout_millis: jsii.Number,
            unhealthy_threshold: jsii.Number,
            path: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents the health check policy for a virtual node's listener.

            :param healthy_threshold: The number of consecutive successful health checks that must occur before declaring listener healthy.
            :param interval_millis: The time period in milliseconds between each health check execution.
            :param protocol: The protocol for the health check request. If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .
            :param timeout_millis: The amount of time to wait when receiving a response from the health check, in milliseconds.
            :param unhealthy_threshold: The number of consecutive failed health checks that must occur before declaring a virtual node unhealthy.
            :param path: The destination path for the health check request. This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.
            :param port: The destination port for the health check request. This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                health_check_property = appmesh.CfnVirtualNode.HealthCheckProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
                
                    # the properties below are optional
                    path="path",
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3d43c4e6ce582f523ffeebbf2ebec9cbd3668e3187cedc5273e62f90b0838b5b)
                check_type(argname="argument healthy_threshold", value=healthy_threshold, expected_type=type_hints["healthy_threshold"])
                check_type(argname="argument interval_millis", value=interval_millis, expected_type=type_hints["interval_millis"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
                check_type(argname="argument timeout_millis", value=timeout_millis, expected_type=type_hints["timeout_millis"])
                check_type(argname="argument unhealthy_threshold", value=unhealthy_threshold, expected_type=type_hints["unhealthy_threshold"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "healthy_threshold": healthy_threshold,
                "interval_millis": interval_millis,
                "protocol": protocol,
                "timeout_millis": timeout_millis,
                "unhealthy_threshold": unhealthy_threshold,
            }
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def healthy_threshold(self) -> jsii.Number:
            '''The number of consecutive successful health checks that must occur before declaring listener healthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-healthythreshold
            '''
            result = self._values.get("healthy_threshold")
            assert result is not None, "Required property 'healthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def interval_millis(self) -> jsii.Number:
            '''The time period in milliseconds between each health check execution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-intervalmillis
            '''
            result = self._values.get("interval_millis")
            assert result is not None, "Required property 'interval_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol for the health check request.

            If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def timeout_millis(self) -> jsii.Number:
            '''The amount of time to wait when receiving a response from the health check, in milliseconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-timeoutmillis
            '''
            result = self._values.get("timeout_millis")
            assert result is not None, "Required property 'timeout_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def unhealthy_threshold(self) -> jsii.Number:
            '''The number of consecutive failed health checks that must occur before declaring a virtual node unhealthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-unhealthythreshold
            '''
            result = self._values.get("unhealthy_threshold")
            assert result is not None, "Required property 'unhealthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The destination path for the health check request.

            This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The destination port for the health check request.

            This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HealthCheckProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.HttpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class HttpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_timeout_property = appmesh.CfnVirtualNode.HttpTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bf5356c2431488f161968182b4ae22c49b9bddc44ea92f8c99141d4dc5bea2f1)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html#cfn-appmesh-virtualnode-httptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html#cfn-appmesh-virtualnode-httptimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.JsonFormatRefProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class JsonFormatRefProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the key value pairs for the JSON.

            :param key: The specified key for the JSON.
            :param value: The specified value for the JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-jsonformatref.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                json_format_ref_property = appmesh.CfnVirtualNode.JsonFormatRefProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__adafa7a8981ef7f783822b2de133325adc34d5e4855cd3d7fd27db69687a2cc5)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The specified key for the JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-jsonformatref.html#cfn-appmesh-virtualnode-jsonformatref-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The specified value for the JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-jsonformatref.html#cfn-appmesh-virtualnode-jsonformatref-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JsonFormatRefProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_mapping": "portMapping",
            "connection_pool": "connectionPool",
            "health_check": "healthCheck",
            "outlier_detection": "outlierDetection",
            "timeout": "timeout",
            "tls": "tls",
        },
    )
    class ListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union[typing.Union["CfnVirtualNode.PortMappingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            connection_pool: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            health_check: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.HealthCheckProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            outlier_detection: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.OutlierDetectionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            tls: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener for a virtual node.

            :param port_mapping: The port mapping information for the listener.
            :param connection_pool: The connection pool information for the listener.
            :param health_check: The health check information for the listener.
            :param outlier_detection: The outlier detection information for the listener.
            :param timeout: An object that represents timeouts for different protocols.
            :param tls: A reference to an object that represents the Transport Layer Security (TLS) properties for a listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_property = appmesh.CfnVirtualNode.ListenerProperty(
                    port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
                
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                            max_connections=123,
                
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                            max_requests=123
                        ),
                        tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                            max_connections=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
                
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                        base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        interval=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        max_ejection_percent=123,
                        max_server_errors=123
                    ),
                    timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                        grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                        certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
                
                        # the properties below are optional
                        validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6376e0a09903a69d1f307a07b60dcfcfd0e022c177d8ce4fba08d0737e0b01ea)
                check_type(argname="argument port_mapping", value=port_mapping, expected_type=type_hints["port_mapping"])
                check_type(argname="argument connection_pool", value=connection_pool, expected_type=type_hints["connection_pool"])
                check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
                check_type(argname="argument outlier_detection", value=outlier_detection, expected_type=type_hints["outlier_detection"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_mapping": port_mapping,
            }
            if connection_pool is not None:
                self._values["connection_pool"] = connection_pool
            if health_check is not None:
                self._values["health_check"] = health_check
            if outlier_detection is not None:
                self._values["outlier_detection"] = outlier_detection
            if timeout is not None:
                self._values["timeout"] = timeout
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union["CfnVirtualNode.PortMappingProperty", _IResolvable_a771d0ef]:
            '''The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.PortMappingProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def connection_pool(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''The connection pool information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-connectionpool
            '''
            result = self._values.get("connection_pool")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.HealthCheckProperty", _IResolvable_a771d0ef]]:
            '''The health check information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.HealthCheckProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def outlier_detection(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.OutlierDetectionProperty", _IResolvable_a771d0ef]]:
            '''The outlier detection information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-outlierdetection
            '''
            result = self._values.get("outlier_detection")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.OutlierDetectionProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents timeouts for different protocols.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the Transport Layer Security (TLS) properties for a listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2", "tcp": "tcp"},
    )
    class ListenerTimeoutProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.GrpcTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.HttpTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http2: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.HttpTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            tcp: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.TcpTimeoutProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents timeouts for different protocols.

            :param grpc: An object that represents types of timeouts.
            :param http: An object that represents types of timeouts.
            :param http2: An object that represents types of timeouts.
            :param tcp: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_timeout_property = appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                    grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__489aebf55866be85cf08a531baf0b2cb328bff433c34f241283eae96f7e545ae)
                check_type(argname="argument grpc", value=grpc, expected_type=type_hints["grpc"])
                check_type(argname="argument http", value=http, expected_type=type_hints["http"])
                check_type(argname="argument http2", value=http2, expected_type=type_hints["http2"])
                check_type(argname="argument tcp", value=tcp, expected_type=type_hints["tcp"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2
            if tcp is not None:
                self._values["tcp"] = tcp

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.GrpcTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.GrpcTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tcp(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TcpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-tcp
            '''
            result = self._values.get("tcp")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TcpTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class ListenerTlsAcmCertificateProperty:
        def __init__(self, *, certificate_arn: builtins.str) -> None:
            '''An object that represents an AWS Certificate Manager certificate.

            :param certificate_arn: The Amazon Resource Name (ARN) for the certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsacmcertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_acm_certificate_property = appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                    certificate_arn="certificateArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d7965133e418275cbc692e31487201b1027d69634fe1d803019f159d77db071c)
                check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_arn": certificate_arn,
            }

        @builtins.property
        def certificate_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) for the certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsacmcertificate.html#cfn-appmesh-virtualnode-listenertlsacmcertificate-certificatearn
            '''
            result = self._values.get("certificate_arn")
            assert result is not None, "Required property 'certificate_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsAcmCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class ListenerTlsCertificateProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) certificate.

            :param acm: A reference to an object that represents an AWS Certificate Manager certificate.
            :param file: A reference to an object that represents a local file certificate.
            :param sds: A reference to an object that represents a listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_certificate_property = appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                    acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                        certificate_arn="certificateArn"
                    ),
                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__17ec439d1051dd0bcb26feb86be6ca69e110ceaddab5b134db6ad26e5ee1eb4f)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a local file certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_chain": "certificateChain",
            "private_key": "privateKey",
        },
    )
    class ListenerTlsFileCertificateProperty:
        def __init__(
            self,
            *,
            certificate_chain: builtins.str,
            private_key: builtins.str,
        ) -> None:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :param certificate_chain: The certificate chain for the certificate.
            :param private_key: The private key for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_file_certificate_property = appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                    certificate_chain="certificateChain",
                    private_key="privateKey"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8e84a24d4b08b5f0ebd14d5e9527ee6519bf3513d380e1059a34a917a40aa2a1)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
                check_type(argname="argument private_key", value=private_key, expected_type=type_hints["private_key"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
                "private_key": private_key,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate chain for the certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html#cfn-appmesh-virtualnode-listenertlsfilecertificate-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def private_key(self) -> builtins.str:
            '''The private key for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html#cfn-appmesh-virtualnode-listenertlsfilecertificate-privatekey
            '''
            result = self._values.get("private_key")
            assert result is not None, "Required property 'private_key' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsFileCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate": "certificate",
            "mode": "mode",
            "validation": "validation",
        },
    )
    class ListenerTlsProperty:
        def __init__(
            self,
            *,
            certificate: typing.Union[typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            mode: builtins.str,
            validation: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the Transport Layer Security (TLS) properties for a listener.

            :param certificate: A reference to an object that represents a listener's Transport Layer Security (TLS) certificate.
            :param mode: Specify one of the following modes. - ** STRICT – Listener only accepts connections with TLS enabled. - ** PERMISSIVE – Listener accepts connections with or without TLS enabled. - ** DISABLED – Listener only accepts connections without TLS.
            :param validation: A reference to an object that represents a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_property = appmesh.CfnVirtualNode.ListenerTlsProperty(
                    certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                        acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                            certificate_arn="certificateArn"
                        ),
                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    mode="mode",
                
                    # the properties below are optional
                    validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__aa73e696c78232871e0c6d9ccad2237d2253f09e12429462a4c7d03d935c1505)
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate": certificate,
                "mode": mode,
            }
            if validation is not None:
                self._values["validation"] = validation

        @builtins.property
        def certificate(
            self,
        ) -> typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", _IResolvable_a771d0ef]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-certificate
            '''
            result = self._values.get("certificate")
            assert result is not None, "Required property 'certificate' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def mode(self) -> builtins.str:
            '''Specify one of the following modes.

            - ** STRICT – Listener only accepts connections with TLS enabled.
            - ** PERMISSIVE – Listener accepts connections with or without TLS enabled.
            - ** DISABLED – Listener only accepts connections without TLS.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-mode
            '''
            result = self._values.get("mode")
            assert result is not None, "Required property 'mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def validation(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-validation
            '''
            result = self._values.get("validation")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class ListenerTlsSdsCertificateProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents the listener's Secret Discovery Service certificate.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlssdscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_sds_certificate_property = appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0e654582246da7c5effb9008fc235dd4f8b7e0893dd2461532396c1747cd447b)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlssdscertificate.html#cfn-appmesh-virtualnode-listenertlssdscertificate-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsSdsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class ListenerTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_validation_context_property = appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__de23330412ac4dfe6dcdeacb50d33193b70f42e0f5f306ba7449d918fa0ed00a)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html#cfn-appmesh-virtualnode-listenertlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html#cfn-appmesh-virtualnode-listenertlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class ListenerTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) validation context trust.

            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_validation_context_trust_property = appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9526b4b61b33f40830239f41fef2a293d3566da24140193fb2495c968ae66889)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-listenertlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-listenertlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.LoggingFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"json": "json", "text": "text"},
    )
    class LoggingFormatProperty:
        def __init__(
            self,
            *,
            json: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualNode.JsonFormatRefProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            text: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the format for the logs.

            :param json: The logging format for JSON.
            :param text: The logging format for text.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-loggingformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                logging_format_property = appmesh.CfnVirtualNode.LoggingFormatProperty(
                    json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                        key="key",
                        value="value"
                    )],
                    text="text"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a1d71f31b6ca3e642dcf979459c9b0e82e0a07bfed27ba736e445d2f825f5e0d)
                check_type(argname="argument json", value=json, expected_type=type_hints["json"])
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if json is not None:
                self._values["json"] = json
            if text is not None:
                self._values["text"] = text

        @builtins.property
        def json(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.JsonFormatRefProperty", _IResolvable_a771d0ef]]]]:
            '''The logging format for JSON.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-loggingformat.html#cfn-appmesh-virtualnode-loggingformat-json
            '''
            result = self._values.get("json")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.JsonFormatRefProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def text(self) -> typing.Optional[builtins.str]:
            '''The logging format for text.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-loggingformat.html#cfn-appmesh-virtualnode-loggingformat-text
            '''
            result = self._values.get("text")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.LoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"access_log": "accessLog"},
    )
    class LoggingProperty:
        def __init__(
            self,
            *,
            access_log: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.AccessLogProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the logging information for a virtual node.

            :param access_log: The access log configuration for a virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-logging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                logging_property = appmesh.CfnVirtualNode.LoggingProperty(
                    access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                        file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                            path="path",
                
                            # the properties below are optional
                            format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a33073efb808fb3b3b14584be16c73416f8377f864dee580afb278ecd3c4511f)
                check_type(argname="argument access_log", value=access_log, expected_type=type_hints["access_log"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_log is not None:
                self._values["access_log"] = access_log

        @builtins.property
        def access_log(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", _IResolvable_a771d0ef]]:
            '''The access log configuration for a virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-logging.html#cfn-appmesh-virtualnode-logging-accesslog
            '''
            result = self._values.get("access_log")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.OutlierDetectionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base_ejection_duration": "baseEjectionDuration",
            "interval": "interval",
            "max_ejection_percent": "maxEjectionPercent",
            "max_server_errors": "maxServerErrors",
        },
    )
    class OutlierDetectionProperty:
        def __init__(
            self,
            *,
            base_ejection_duration: typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            interval: typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            max_ejection_percent: jsii.Number,
            max_server_errors: jsii.Number,
        ) -> None:
            '''An object that represents the outlier detection for a virtual node's listener.

            :param base_ejection_duration: The base amount of time for which a host is ejected.
            :param interval: The time interval between ejection sweep analysis.
            :param max_ejection_percent: Maximum percentage of hosts in load balancing pool for upstream service that can be ejected. Will eject at least one host regardless of the value.
            :param max_server_errors: Number of consecutive ``5xx`` errors required for ejection.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                outlier_detection_property = appmesh.CfnVirtualNode.OutlierDetectionProperty(
                    base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    interval=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    max_ejection_percent=123,
                    max_server_errors=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1536882caa67cf2258fb2e69b44e2f535792941e55f4abbc56d75f4c9fb7f22)
                check_type(argname="argument base_ejection_duration", value=base_ejection_duration, expected_type=type_hints["base_ejection_duration"])
                check_type(argname="argument interval", value=interval, expected_type=type_hints["interval"])
                check_type(argname="argument max_ejection_percent", value=max_ejection_percent, expected_type=type_hints["max_ejection_percent"])
                check_type(argname="argument max_server_errors", value=max_server_errors, expected_type=type_hints["max_server_errors"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "base_ejection_duration": base_ejection_duration,
                "interval": interval,
                "max_ejection_percent": max_ejection_percent,
                "max_server_errors": max_server_errors,
            }

        @builtins.property
        def base_ejection_duration(
            self,
        ) -> typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]:
            '''The base amount of time for which a host is ejected.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-baseejectionduration
            '''
            result = self._values.get("base_ejection_duration")
            assert result is not None, "Required property 'base_ejection_duration' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def interval(
            self,
        ) -> typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]:
            '''The time interval between ejection sweep analysis.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-interval
            '''
            result = self._values.get("interval")
            assert result is not None, "Required property 'interval' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def max_ejection_percent(self) -> jsii.Number:
            '''Maximum percentage of hosts in load balancing pool for upstream service that can be ejected.

            Will eject at least one host regardless of the value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-maxejectionpercent
            '''
            result = self._values.get("max_ejection_percent")
            assert result is not None, "Required property 'max_ejection_percent' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_server_errors(self) -> jsii.Number:
            '''Number of consecutive ``5xx`` errors required for ejection.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-maxservererrors
            '''
            result = self._values.get("max_server_errors")
            assert result is not None, "Required property 'max_server_errors' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OutlierDetectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class PortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object representing a virtual node or virtual router listener port mapping.

            :param port: The port used for the port mapping.
            :param protocol: The protocol used for the port mapping. Specify ``http`` , ``http2`` , ``grpc`` , or ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                port_mapping_property = appmesh.CfnVirtualNode.PortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ac31eefdedf302678e59883d355545c8d202519892dc04c30cc7e7fb0bbd7776)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html#cfn-appmesh-virtualnode-portmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            Specify ``http`` , ``http2`` , ``grpc`` , or ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html#cfn-appmesh-virtualnode-portmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={"aws_cloud_map": "awsCloudMap", "dns": "dns"},
    )
    class ServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            aws_cloud_map: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            dns: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the service discovery information for a virtual node.

            :param aws_cloud_map: Specifies any AWS Cloud Map information for the virtual node.
            :param dns: Specifies the DNS information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                service_discovery_property = appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                    aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                        namespace_name="namespaceName",
                        service_name="serviceName",
                
                        # the properties below are optional
                        attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                            key="key",
                            value="value"
                        )],
                        ip_preference="ipPreference"
                    ),
                    dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                        hostname="hostname",
                
                        # the properties below are optional
                        ip_preference="ipPreference",
                        response_type="responseType"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b0a72ffb70e02918febed43bfed9e163db1556e0a252f12b6fdedf7a9dae1cd8)
                check_type(argname="argument aws_cloud_map", value=aws_cloud_map, expected_type=type_hints["aws_cloud_map"])
                check_type(argname="argument dns", value=dns, expected_type=type_hints["dns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if aws_cloud_map is not None:
                self._values["aws_cloud_map"] = aws_cloud_map
            if dns is not None:
                self._values["dns"] = dns

        @builtins.property
        def aws_cloud_map(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''Specifies any AWS Cloud Map information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html#cfn-appmesh-virtualnode-servicediscovery-awscloudmap
            '''
            result = self._values.get("aws_cloud_map")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def dns(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''Specifies the DNS information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html#cfn-appmesh-virtualnode-servicediscovery-dns
            '''
            result = self._values.get("dns")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class SubjectAlternativeNameMatchersProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents the methods by which a subject alternative name on a peer Transport Layer Security (TLS) certificate can be matched.

            :param exact: The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenamematchers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_name_matchers_property = appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                    exact=["exact"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c43961468f3591a168b5762bd1c6f3c86bd47935fb29829509462e14cf053946)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenamematchers.html#cfn-appmesh-virtualnode-subjectalternativenamematchers-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNameMatchersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty",
        jsii_struct_bases=[],
        name_mapping={"match": "match"},
    )
    class SubjectAlternativeNamesProperty:
        def __init__(
            self,
            *,
            match: typing.Union[typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents the subject alternative names secured by the certificate.

            :param match: An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenames.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_names_property = appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                        exact=["exact"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9f9785c795e4ad629cde848bed518b50462bae73a531eae08d7fede2a1c0cf13)
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "match": match,
            }

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenames.html#cfn-appmesh-virtualnode-subjectalternativenames-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNamesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TcpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle"},
    )
    class TcpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tcptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_timeout_property = appmesh.CfnVirtualNode.TcpTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9656be65a93106e9782658bb3cbddff52b358e16fbe8e958efdf9a2c28dee3c0)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tcptimeout.html#cfn-appmesh-virtualnode-tcptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_authority_arns": "certificateAuthorityArns"},
    )
    class TlsValidationContextAcmTrustProperty:
        def __init__(
            self,
            *,
            certificate_authority_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :param certificate_authority_arns: One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextacmtrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_acm_trust_property = appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                    certificate_authority_arns=["certificateAuthorityArns"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e7cee40ffbcc0326b51688a5af3d38bc89a0918f4d7ddd49b23a316b4a7fa5df)
                check_type(argname="argument certificate_authority_arns", value=certificate_authority_arns, expected_type=type_hints["certificate_authority_arns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_authority_arns": certificate_authority_arns,
            }

        @builtins.property
        def certificate_authority_arns(self) -> typing.List[builtins.str]:
            '''One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextacmtrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextacmtrust-certificateauthorityarns
            '''
            result = self._values.get("certificate_authority_arns")
            assert result is not None, "Required property 'certificate_authority_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextAcmTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_chain": "certificateChain"},
    )
    class TlsValidationContextFileTrustProperty:
        def __init__(self, *, certificate_chain: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :param certificate_chain: The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextfiletrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_file_trust_property = appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                    certificate_chain="certificateChain"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0a1878939b5f1487135b6409daa256ed42a8c69e5be212cb63675a52cabc3e6d)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextfiletrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextfiletrust-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextFileTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class TlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents how the proxy will validate its peer during Transport Layer Security (TLS) negotiation.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a Transport Layer Security (TLS) validation context. If you don't specify SANs on the *terminating* mesh endpoint, the Envoy proxy for that node doesn't verify the SAN on a peer client certificate. If you don't specify SANs on the *originating* mesh endpoint, the SAN on the certificate provided by the terminating endpoint must match the mesh endpoint service discovery configuration. Since SPIRE vended certificates have a SPIFFE ID as a name, you must set the SAN since the name doesn't match the service discovery name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_property = appmesh.CfnVirtualNode.TlsValidationContextProperty(
                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                            certificate_authority_arns=["certificateAuthorityArns"]
                        ),
                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__58dc0dc924866e79c8d61c650410f5ce15a43e8c5ba52004e8655aca86496cfe)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html#cfn-appmesh-virtualnode-tlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a Transport Layer Security (TLS) validation context.

            If you don't specify SANs on the *terminating* mesh endpoint, the Envoy proxy for that node doesn't verify the SAN on a peer client certificate. If you don't specify SANs on the *originating* mesh endpoint, the SAN on the certificate provided by the terminating endpoint must match the mesh endpoint service discovery configuration. Since SPIRE vended certificates have a SPIFFE ID as a name, you must set the SAN since the name doesn't match the service discovery name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html#cfn-appmesh-virtualnode-tlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class TlsValidationContextSdsTrustProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret for a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextsdstrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_sds_trust_property = appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__077995053a7a0b6350cec5c670400b20b320f863529b55ddc7caa04d9eb7d90e)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret for a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextsdstrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextsdstrust-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextSdsTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class TlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust.

            :param acm: A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.
            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_trust_property = appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                        certificate_authority_arns=["certificateAuthorityArns"]
                    ),
                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f2272121b53711515405445d8a47126992c45e88c7ec10f60d23052c53efd66d)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2", "tcp": "tcp"},
    )
    class VirtualNodeConnectionPoolProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            http2: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            tcp: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the type of virtual node connection pool.

            Only one protocol is used at a time and should be the same protocol as the one chosen under port mapping.

            If not present the default value for ``maxPendingRequests`` is ``2147483647`` .

            :param grpc: An object that represents a type of connection pool.
            :param http: An object that represents a type of connection pool.
            :param http2: An object that represents a type of connection pool.
            :param tcp: An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                    grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                        max_requests=123
                    ),
                    http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                        max_connections=123,
                
                        # the properties below are optional
                        max_pending_requests=123
                    ),
                    http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                        max_requests=123
                    ),
                    tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                        max_connections=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0c505e81ff9400a5a3f257a81e1b2972458a60a3bd39ce722c4294719497c3d3)
                check_type(argname="argument grpc", value=grpc, expected_type=type_hints["grpc"])
                check_type(argname="argument http", value=http, expected_type=type_hints["http"])
                check_type(argname="argument http2", value=http2, expected_type=type_hints["http2"])
                check_type(argname="argument tcp", value=tcp, expected_type=type_hints["tcp"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2
            if tcp is not None:
                self._values["tcp"] = tcp

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tcp(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-tcp
            '''
            result = self._values.get("tcp")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualNodeGrpcConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodegrpcconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_grpc_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__792363c2ab4f796995dcde4ff05b60cc7586f6e5cead0e5b813e7b01fda75f51)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodegrpcconnectionpool.html#cfn-appmesh-virtualnode-virtualnodegrpcconnectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeGrpcConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualNodeHttp2ConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttp2connectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_http2_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3a4f18eb3dc4616b30b682db78673f2b5a38e9933418bb2b25e3c78339b0424e)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttp2connectionpool.html#cfn-appmesh-virtualnode-virtualnodehttp2connectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeHttp2ConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_connections": "maxConnections",
            "max_pending_requests": "maxPendingRequests",
        },
    )
    class VirtualNodeHttpConnectionPoolProperty:
        def __init__(
            self,
            *,
            max_connections: jsii.Number,
            max_pending_requests: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.
            :param max_pending_requests: Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_http_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                    max_connections=123,
                
                    # the properties below are optional
                    max_pending_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__09baf0ddc0fa0787766cb6944dd6f11936534365bd69ce0d88a1a1326df8319d)
                check_type(argname="argument max_connections", value=max_connections, expected_type=type_hints["max_connections"])
                check_type(argname="argument max_pending_requests", value=max_pending_requests, expected_type=type_hints["max_pending_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_connections": max_connections,
            }
            if max_pending_requests is not None:
                self._values["max_pending_requests"] = max_pending_requests

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodehttpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_pending_requests(self) -> typing.Optional[jsii.Number]:
            '''Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodehttpconnectionpool-maxpendingrequests
            '''
            result = self._values.get("max_pending_requests")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeHttpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "backend_defaults": "backendDefaults",
            "backends": "backends",
            "listeners": "listeners",
            "logging": "logging",
            "service_discovery": "serviceDiscovery",
        },
    )
    class VirtualNodeSpecProperty:
        def __init__(
            self,
            *,
            backend_defaults: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.BackendDefaultsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            backends: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualNode.BackendProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            listeners: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualNode.ListenerProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            logging: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.LoggingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            service_discovery: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a virtual node.

            :param backend_defaults: A reference to an object that represents the defaults for backends.
            :param backends: The backends that the virtual node is expected to send outbound traffic to.
            :param listeners: The listener that the virtual node is expected to receive inbound traffic from. You can specify one listener.
            :param logging: The inbound and outbound access logging information for the virtual node.
            :param service_discovery: The service discovery information for the virtual node. If your virtual node does not expect ingress traffic, you can omit this parameter. If you specify a ``listener`` , then you must specify service discovery information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_spec_property = appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                    backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    backends=[appmesh.CfnVirtualNode.BackendProperty(
                        virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                            virtual_service_name="virtualServiceName",
                
                            # the properties below are optional
                            client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                                tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                                certificate_authority_arns=["certificateAuthorityArns"]
                                            ),
                                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                                certificate_chain="certificateChain"
                                            ),
                                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                                secret_name="secretName"
                                            )
                                        ),
                
                                        # the properties below are optional
                                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                                exact=["exact"]
                                            )
                                        )
                                    ),
                
                                    # the properties below are optional
                                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                            certificate_chain="certificateChain",
                                            private_key="privateKey"
                                        ),
                                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                                    enforce=False,
                                    ports=[123]
                                )
                            )
                        )
                    )],
                    listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                        port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
                
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                                max_connections=123,
                
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                                max_requests=123
                            ),
                            tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                                max_connections=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
                
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                            base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            interval=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            max_ejection_percent=123,
                            max_server_errors=123
                        ),
                        timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                            grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            )
                        ),
                        tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                            certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
                
                            # the properties below are optional
                            validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                    logging=appmesh.CfnVirtualNode.LoggingProperty(
                        access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                            file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                                path="path",
                
                                # the properties below are optional
                                format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    ),
                    service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                        aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                            namespace_name="namespaceName",
                            service_name="serviceName",
                
                            # the properties below are optional
                            attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                                key="key",
                                value="value"
                            )],
                            ip_preference="ipPreference"
                        ),
                        dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                            hostname="hostname",
                
                            # the properties below are optional
                            ip_preference="ipPreference",
                            response_type="responseType"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6e4f284bb053780ba3420b896a308dde13ed2c0ff57675c75cc85e49f28928a1)
                check_type(argname="argument backend_defaults", value=backend_defaults, expected_type=type_hints["backend_defaults"])
                check_type(argname="argument backends", value=backends, expected_type=type_hints["backends"])
                check_type(argname="argument listeners", value=listeners, expected_type=type_hints["listeners"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
                check_type(argname="argument service_discovery", value=service_discovery, expected_type=type_hints["service_discovery"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if backend_defaults is not None:
                self._values["backend_defaults"] = backend_defaults
            if backends is not None:
                self._values["backends"] = backends
            if listeners is not None:
                self._values["listeners"] = listeners
            if logging is not None:
                self._values["logging"] = logging
            if service_discovery is not None:
                self._values["service_discovery"] = service_discovery

        @builtins.property
        def backend_defaults(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.BackendDefaultsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the defaults for backends.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-backenddefaults
            '''
            result = self._values.get("backend_defaults")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.BackendDefaultsProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def backends(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.BackendProperty", _IResolvable_a771d0ef]]]]:
            '''The backends that the virtual node is expected to send outbound traffic to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-backends
            '''
            result = self._values.get("backends")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.BackendProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def listeners(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.ListenerProperty", _IResolvable_a771d0ef]]]]:
            '''The listener that the virtual node is expected to receive inbound traffic from.

            You can specify one listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-listeners
            '''
            result = self._values.get("listeners")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.ListenerProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.LoggingProperty", _IResolvable_a771d0ef]]:
            '''The inbound and outbound access logging information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.LoggingProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def service_discovery(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''The service discovery information for the virtual node.

            If your virtual node does not expect ingress traffic, you can omit this parameter. If you specify a ``listener`` , then you must specify service discovery information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-servicediscovery
            '''
            result = self._values.get("service_discovery")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_connections": "maxConnections"},
    )
    class VirtualNodeTcpConnectionPoolProperty:
        def __init__(self, *, max_connections: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodetcpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_tcp_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                    max_connections=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bc2de85f618a24a1abb11631ef1a9e31cc02db09e7f378f6543fa9f2a3e32482)
                check_type(argname="argument max_connections", value=max_connections, expected_type=type_hints["max_connections"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_connections": max_connections,
            }

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodetcpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodetcpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeTcpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualServiceBackendProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_service_name": "virtualServiceName",
            "client_policy": "clientPolicy",
        },
    )
    class VirtualServiceBackendProperty:
        def __init__(
            self,
            *,
            virtual_service_name: builtins.str,
            client_policy: typing.Optional[typing.Union[typing.Union["CfnVirtualNode.ClientPolicyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a virtual service backend for a virtual node.

            :param virtual_service_name: The name of the virtual service that is acting as a virtual node backend.
            :param client_policy: A reference to an object that represents the client policy for a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_service_backend_property = appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                    virtual_service_name="virtualServiceName",
                
                    # the properties below are optional
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a811b2d1b78ecde5cb09c0cad5e911d3400ac80f6c85f6098757a51b420fbddb)
                check_type(argname="argument virtual_service_name", value=virtual_service_name, expected_type=type_hints["virtual_service_name"])
                check_type(argname="argument client_policy", value=client_policy, expected_type=type_hints["client_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_service_name": virtual_service_name,
            }
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def virtual_service_name(self) -> builtins.str:
            '''The name of the virtual service that is acting as a virtual node backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html#cfn-appmesh-virtualnode-virtualservicebackend-virtualservicename
            '''
            result = self._values.get("virtual_service_name")
            assert result is not None, "Required property 'virtual_service_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the client policy for a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html#cfn-appmesh-virtualnode-virtualservicebackend-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceBackendProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualNodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_node_name": "virtualNodeName",
    },
)
class CfnVirtualNodeProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_node_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualNode``.

        :param mesh_name: The name of the service mesh to create the virtual node in.
        :param spec: The virtual node specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual node to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_node_name: The name to use for the virtual node.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_node_props = appmesh.CfnVirtualNodeProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                    backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    backends=[appmesh.CfnVirtualNode.BackendProperty(
                        virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                            virtual_service_name="virtualServiceName",
            
                            # the properties below are optional
                            client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                                tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                                certificate_authority_arns=["certificateAuthorityArns"]
                                            ),
                                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                                certificate_chain="certificateChain"
                                            ),
                                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                                secret_name="secretName"
                                            )
                                        ),
            
                                        # the properties below are optional
                                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                                exact=["exact"]
                                            )
                                        )
                                    ),
            
                                    # the properties below are optional
                                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                            certificate_chain="certificateChain",
                                            private_key="privateKey"
                                        ),
                                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                                    enforce=False,
                                    ports=[123]
                                )
                            )
                        )
                    )],
                    listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                        port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
            
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                                max_connections=123,
            
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                                max_requests=123
                            ),
                            tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                                max_connections=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
            
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                            base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            interval=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            max_ejection_percent=123,
                            max_server_errors=123
                        ),
                        timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                            grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            )
                        ),
                        tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                            certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
            
                            # the properties below are optional
                            validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
            
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                    logging=appmesh.CfnVirtualNode.LoggingProperty(
                        access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                            file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                                path="path",
            
                                # the properties below are optional
                                format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    ),
                    service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                        aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                            namespace_name="namespaceName",
                            service_name="serviceName",
            
                            # the properties below are optional
                            attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                                key="key",
                                value="value"
                            )],
                            ip_preference="ipPreference"
                        ),
                        dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                            hostname="hostname",
            
                            # the properties below are optional
                            ip_preference="ipPreference",
                            response_type="responseType"
                        )
                    )
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_node_name="virtualNodeName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c4a192d8866760fef0af5549ec95c7fc03c2030ed0a5a18ac9b7c30abc6d7b9)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_node_name", value=virtual_node_name, expected_type=type_hints["virtual_node_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_node_name is not None:
            self._values["virtual_node_name"] = virtual_node_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual node in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, _IResolvable_a771d0ef]:
        '''The virtual node specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual node to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def virtual_node_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual node.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-virtualnodename
        '''
        result = self._values.get("virtual_node_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualNodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualRouter(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualRouter",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualRouter``.

    Creates a virtual router within a service mesh.

    Specify a ``listener`` for any inbound traffic that your virtual router receives. Create a virtual router for each protocol and port that you need to route. Virtual routers handle traffic for one or more virtual services within your mesh. After you create your virtual router, create and associate routes for your virtual router that direct incoming requests to different virtual nodes.

    For more information about virtual routers, see `Virtual routers <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html>`_ .

    :cloudformationResource: AWS::AppMesh::VirtualRouter
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_router = appmesh.CfnVirtualRouter(self, "MyCfnVirtualRouter",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                    port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    )
                )]
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_router_name="virtualRouterName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_router_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualRouter``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the virtual router in.
        :param spec: The virtual router specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual router to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_router_name: The name to use for the virtual router.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b580e671668ae16998e4ac83d27c638ef3aa71d184e426eb265d6c59eddbb3d5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualRouterProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_router_name=virtual_router_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f35a240ca4e601059cdf3441626e9301829b7e4e7c94e6c2c70d7733c9ab80f)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__987bc053769b53b07a55431de17b0fab714e5318bec9a9a092fd3cce476e2738)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual router.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual router resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual router.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualRouterName")
    def attr_virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router.

        :cloudformationAttribute: VirtualRouterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualRouterName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual router to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual router in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1928a019eea7816370de3f44d866550b1c229a72bcbfdc2dd9bb32e6509f7ab3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef]:
        '''The virtual router specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-spec
        '''
        return typing.cast(typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__082d5e7fa0667445ceefdb1ac102155a2e850acd4b4f27577006951bf3a5c520)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62fcbea1811d5d467a0da99d8caacd5456d3460de357f7ac63ac6f45c4d2dd17)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value)

    @builtins.property
    @jsii.member(jsii_name="virtualRouterName")
    def virtual_router_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual router.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-virtualroutername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualRouterName"))

    @virtual_router_name.setter
    def virtual_router_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__72e30628b94763f2421abcd9c0089fb2ce6e84b02c67dcb164a16d564c23bee3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualRouterName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualRouter.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class PortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object representing a virtual router listener port mapping.

            :param port: The port used for the port mapping.
            :param protocol: The protocol used for the port mapping. Specify one protocol.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                port_mapping_property = appmesh.CfnVirtualRouter.PortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c2340beef7949adeb0b8d8ba053869b985ff2c47c5c9b1990640590f69b0367e)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html#cfn-appmesh-virtualrouter-portmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            Specify one protocol.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html#cfn-appmesh-virtualrouter-portmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualRouter.VirtualRouterListenerProperty",
        jsii_struct_bases=[],
        name_mapping={"port_mapping": "portMapping"},
    )
    class VirtualRouterListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union[typing.Union["CfnVirtualRouter.PortMappingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents a virtual router listener.

            :param port_mapping: The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterlistener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_router_listener_property = appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                    port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fcd5137de67d4e67e5a2539a95278c705ea832dc885219e6fc675f1c7c0071ec)
                check_type(argname="argument port_mapping", value=port_mapping, expected_type=type_hints["port_mapping"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_mapping": port_mapping,
            }

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union["CfnVirtualRouter.PortMappingProperty", _IResolvable_a771d0ef]:
            '''The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterlistener.html#cfn-appmesh-virtualrouter-virtualrouterlistener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union["CfnVirtualRouter.PortMappingProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualRouter.VirtualRouterSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"listeners": "listeners"},
    )
    class VirtualRouterSpecProperty:
        def __init__(
            self,
            *,
            listeners: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the specification of a virtual router.

            :param listeners: The listeners that the virtual router is expected to receive inbound traffic from.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_router_spec_property = appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                    listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                        port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f792314a0fe37bd441404d59378e2dd049fc35dfe3f36f54d7dbb9fd588523b)
                check_type(argname="argument listeners", value=listeners, expected_type=type_hints["listeners"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "listeners": listeners,
            }

        @builtins.property
        def listeners(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", _IResolvable_a771d0ef]]]:
            '''The listeners that the virtual router is expected to receive inbound traffic from.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterspec.html#cfn-appmesh-virtualrouter-virtualrouterspec-listeners
            '''
            result = self._values.get("listeners")
            assert result is not None, "Required property 'listeners' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualRouterProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_router_name": "virtualRouterName",
    },
)
class CfnVirtualRouterProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_router_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualRouter``.

        :param mesh_name: The name of the service mesh to create the virtual router in.
        :param spec: The virtual router specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual router to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_router_name: The name to use for the virtual router.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_router_props = appmesh.CfnVirtualRouterProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                    listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                        port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        )
                    )]
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_router_name="virtualRouterName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa05fdcc677c5955a83b5295feaf8581be1f48a0f2dd583e8f462aaf987a2a29)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_router_name", value=virtual_router_name, expected_type=type_hints["virtual_router_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_router_name is not None:
            self._values["virtual_router_name"] = virtual_router_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual router in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, _IResolvable_a771d0ef]:
        '''The virtual router specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual router to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def virtual_router_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual router.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-virtualroutername
        '''
        result = self._values.get("virtual_router_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualRouterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualService(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualService",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualService``.

    Creates a virtual service within a service mesh.

    A virtual service is an abstraction of a real service that is provided by a virtual node directly or indirectly by means of a virtual router. Dependent services call your virtual service by its ``virtualServiceName`` , and those requests are routed to the virtual node or virtual router that is specified as the provider for the virtual service.

    For more information about virtual services, see `Virtual services <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html>`_ .

    :cloudformationResource: AWS::AppMesh::VirtualService
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_service = appmesh.CfnVirtualService(self, "MyCfnVirtualService",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                    virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                        virtual_node_name="virtualNodeName"
                    ),
                    virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                        virtual_router_name="virtualRouterName"
                    )
                )
            ),
            virtual_service_name="virtualServiceName",
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union["CfnVirtualService.VirtualServiceSpecProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        virtual_service_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualService``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the virtual service in.
        :param spec: The virtual service specification to apply.
        :param virtual_service_name: The name to use for the virtual service.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual service to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e530d12009e680d613f29fbeeef18b19b0fd16ae8d4715396266eb1446363ced)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualServiceProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_service_name=virtual_service_name,
            mesh_owner=mesh_owner,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4c77fcaaa415e236744aa65160f8a8a40fef0f3e4d40e902b80599cee7ebbe3e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__03ce11f71471665c4168a124b65a009d93887dcf17c54d2136fef8f31168f175)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual service.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual service resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual service.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualServiceName")
    def attr_virtual_service_name(self) -> builtins.str:
        '''The name of the virtual service.

        :cloudformationAttribute: VirtualServiceName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualServiceName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual service to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual service in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f161d55059b88e25342fab2a142d2058289603ae3167be64c4aa6473c7b15c82)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value)

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef]:
        '''The virtual service specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-spec
        '''
        return typing.cast(typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a518ed3675805ad10b4044cf0665bb55bec5fce845fba53115a19d0beaf89b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value)

    @builtins.property
    @jsii.member(jsii_name="virtualServiceName")
    def virtual_service_name(self) -> builtins.str:
        '''The name to use for the virtual service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-virtualservicename
        '''
        return typing.cast(builtins.str, jsii.get(self, "virtualServiceName"))

    @virtual_service_name.setter
    def virtual_service_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1092e52e038450607842b5ed1b65cbbd66b5f2aeaa1ebacb51b261851b71668a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualServiceName", value)

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5bdf69317cdf1bccaef67f8d5d62715158a2c01c70e4b46001a42ea36e6932c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_node_name": "virtualNodeName"},
    )
    class VirtualNodeServiceProviderProperty:
        def __init__(self, *, virtual_node_name: builtins.str) -> None:
            '''An object that represents a virtual node service provider.

            :param virtual_node_name: The name of the virtual node that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualnodeserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_service_provider_property = appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                    virtual_node_name="virtualNodeName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6d939c42f0d294c034a89db1b7cbd1b8f4db0c0b3290b03eca9dfe1473eeb149)
                check_type(argname="argument virtual_node_name", value=virtual_node_name, expected_type=type_hints["virtual_node_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_node_name": virtual_node_name,
            }

        @builtins.property
        def virtual_node_name(self) -> builtins.str:
            '''The name of the virtual node that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualnodeserviceprovider.html#cfn-appmesh-virtualservice-virtualnodeserviceprovider-virtualnodename
            '''
            result = self._values.get("virtual_node_name")
            assert result is not None, "Required property 'virtual_node_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_router_name": "virtualRouterName"},
    )
    class VirtualRouterServiceProviderProperty:
        def __init__(self, *, virtual_router_name: builtins.str) -> None:
            '''An object that represents a virtual node service provider.

            :param virtual_router_name: The name of the virtual router that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualrouterserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_router_service_provider_property = appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                    virtual_router_name="virtualRouterName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__139930255e41bd70ee56861200c8566fe37dcbfc73d2e63d13160eef5813b3ed)
                check_type(argname="argument virtual_router_name", value=virtual_router_name, expected_type=type_hints["virtual_router_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_router_name": virtual_router_name,
            }

        @builtins.property
        def virtual_router_name(self) -> builtins.str:
            '''The name of the virtual router that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualrouterserviceprovider.html#cfn-appmesh-virtualservice-virtualrouterserviceprovider-virtualroutername
            '''
            result = self._values.get("virtual_router_name")
            assert result is not None, "Required property 'virtual_router_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_node": "virtualNode",
            "virtual_router": "virtualRouter",
        },
    )
    class VirtualServiceProviderProperty:
        def __init__(
            self,
            *,
            virtual_node: typing.Optional[typing.Union[typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            virtual_router: typing.Optional[typing.Union[typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the provider for a virtual service.

            :param virtual_node: The virtual node associated with a virtual service.
            :param virtual_router: The virtual router associated with a virtual service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_service_provider_property = appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                    virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                        virtual_node_name="virtualNodeName"
                    ),
                    virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                        virtual_router_name="virtualRouterName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__55ebb64faf05de88e37a4402a88855f9b77e950785946800b4dfde8b3ed806be)
                check_type(argname="argument virtual_node", value=virtual_node, expected_type=type_hints["virtual_node"])
                check_type(argname="argument virtual_router", value=virtual_router, expected_type=type_hints["virtual_router"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if virtual_node is not None:
                self._values["virtual_node"] = virtual_node
            if virtual_router is not None:
                self._values["virtual_router"] = virtual_router

        @builtins.property
        def virtual_node(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", _IResolvable_a771d0ef]]:
            '''The virtual node associated with a virtual service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html#cfn-appmesh-virtualservice-virtualserviceprovider-virtualnode
            '''
            result = self._values.get("virtual_node")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def virtual_router(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", _IResolvable_a771d0ef]]:
            '''The virtual router associated with a virtual service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html#cfn-appmesh-virtualservice-virtualserviceprovider-virtualrouter
            '''
            result = self._values.get("virtual_router")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualServiceSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"provider": "provider"},
    )
    class VirtualServiceSpecProperty:
        def __init__(
            self,
            *,
            provider: typing.Optional[typing.Union[typing.Union["CfnVirtualService.VirtualServiceProviderProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a virtual service.

            :param provider: The App Mesh object that is acting as the provider for a virtual service. You can specify a single virtual node or virtual router.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualservicespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_service_spec_property = appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                    provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                        virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                            virtual_node_name="virtualNodeName"
                        ),
                        virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                            virtual_router_name="virtualRouterName"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b6806d509ea056ff6d296e09283a87f2969cde830bd7c62816a978b75e2652e9)
                check_type(argname="argument provider", value=provider, expected_type=type_hints["provider"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if provider is not None:
                self._values["provider"] = provider

        @builtins.property
        def provider(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualService.VirtualServiceProviderProperty", _IResolvable_a771d0ef]]:
            '''The App Mesh object that is acting as the provider for a virtual service.

            You can specify a single virtual node or virtual router.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualservicespec.html#cfn-appmesh-virtualservice-virtualservicespec-provider
            '''
            result = self._values.get("provider")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualService.VirtualServiceProviderProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualServiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_service_name": "virtualServiceName",
        "mesh_owner": "meshOwner",
        "tags": "tags",
    },
)
class CfnVirtualServiceProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[typing.Union[CfnVirtualService.VirtualServiceSpecProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        virtual_service_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualService``.

        :param mesh_name: The name of the service mesh to create the virtual service in.
        :param spec: The virtual service specification to apply.
        :param virtual_service_name: The name to use for the virtual service.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual service to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_service_props = appmesh.CfnVirtualServiceProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                    provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                        virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                            virtual_node_name="virtualNodeName"
                        ),
                        virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                            virtual_router_name="virtualRouterName"
                        )
                    )
                ),
                virtual_service_name="virtualServiceName",
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2adae3119dc94fa8ac3a3fd04984dbf06d4fd405f5741459fcfdfb81a6f5384)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument virtual_service_name", value=virtual_service_name, expected_type=type_hints["virtual_service_name"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_service_name": virtual_service_name,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual service in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualService.VirtualServiceSpecProperty, _IResolvable_a771d0ef]:
        '''The virtual service specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualService.VirtualServiceSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def virtual_service_name(self) -> builtins.str:
        '''The name to use for the virtual service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-virtualservicename
        '''
        result = self._values.get("virtual_service_name")
        assert result is not None, "Required property 'virtual_service_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual service to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CommonGatewayRouteSpecOptions",
    jsii_struct_bases=[],
    name_mapping={"priority": "priority"},
)
class CommonGatewayRouteSpecOptions:
    def __init__(self, *, priority: typing.Optional[jsii.Number] = None) -> None:
        '''(experimental) Base options for all gateway route specs.

        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            common_gateway_route_spec_options = appmesh.CommonGatewayRouteSpecOptions(
                priority=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ffa11c2b8b972175cac950cca89fab714e780027e8b58482374629fe37bf93aa)
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority

        :stability: experimental
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonGatewayRouteSpecOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_appmesh.DnsResponseType")
class DnsResponseType(enum.Enum):
    '''(experimental) Enum of DNS service discovery response type.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # A Virtual Node with a gRPC listener with a connection pool set
        # mesh: appmesh.Mesh
        
        node = appmesh.VirtualNode(self, "node",
            mesh=mesh,
            # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
            # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
            # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
            # By default, the response type is assumed to be LOAD_BALANCER
            service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
            listeners=[appmesh.VirtualNodeListener.http(
                port=80,
                connection_pool=cdk.aws_appmesh.HttpConnectionPool(
                    max_connections=100,
                    max_pending_requests=10
                )
            )]
        )
        
        # A Virtual Gateway with a gRPC listener with a connection pool set
        gateway = appmesh.VirtualGateway(self, "gateway",
            mesh=mesh,
            listeners=[appmesh.VirtualGatewayListener.grpc(
                port=8080,
                connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
                    max_requests=10
                )
            )],
            virtual_gateway_name="gateway"
        )
    '''

    LOAD_BALANCER = "LOAD_BALANCER"
    '''(experimental) DNS resolver returns a loadbalanced set of endpoints and the traffic would be sent to the given endpoints.

    It would not drain existing connections to other endpoints that are not part of this list.

    :stability: experimental
    '''
    ENDPOINTS = "ENDPOINTS"
    '''(experimental) DNS resolver is returning all the endpoints.

    This also means that if an endpoint is missing, it would drain the current connections to the missing endpoint.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "gateway_route_name": "gatewayRouteName",
        "virtual_gateway": "virtualGateway",
    },
)
class GatewayRouteAttributes:
    def __init__(
        self,
        *,
        gateway_route_name: builtins.str,
        virtual_gateway: "IVirtualGateway",
    ) -> None:
        '''(experimental) Interface with properties necessary to import a reusable GatewayRoute.

        :param gateway_route_name: (experimental) The name of the GatewayRoute.
        :param virtual_gateway: (experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            # virtual_gateway: appmesh.VirtualGateway
            
            gateway_route_attributes = appmesh.GatewayRouteAttributes(
                gateway_route_name="gatewayRouteName",
                virtual_gateway=virtual_gateway
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f83a986b376eb314d70984171b604c33553ab4b59999e2b2f45c13061f6d8b27)
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
            check_type(argname="argument virtual_gateway", value=virtual_gateway, expected_type=type_hints["virtual_gateway"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "gateway_route_name": gateway_route_name,
            "virtual_gateway": virtual_gateway,
        }

    @builtins.property
    def gateway_route_name(self) -> builtins.str:
        '''(experimental) The name of the GatewayRoute.

        :stability: experimental
        '''
        result = self._values.get("gateway_route_name")
        assert result is not None, "Required property 'gateway_route_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def virtual_gateway(self) -> "IVirtualGateway":
        '''(experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway")
        assert result is not None, "Required property 'virtual_gateway' is missing"
        return typing.cast("IVirtualGateway", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteBaseProps",
    jsii_struct_bases=[],
    name_mapping={"route_spec": "routeSpec", "gateway_route_name": "gatewayRouteName"},
)
class GatewayRouteBaseProps:
    def __init__(
        self,
        *,
        route_spec: "GatewayRouteSpec",
        gateway_route_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Basic configuration properties for a GatewayRoute.

        :param route_spec: (experimental) What protocol the route uses.
        :param gateway_route_name: (experimental) The name of the GatewayRoute. Default: - an automatically generated name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-grpc",
                route_spec=appmesh.GatewayRouteSpec.grpc(
                    route_target=virtual_service,
                    match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                        hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
                        # This disables the default rewrite to virtual service name and retain original request.
                        rewrite_request_hostname=False
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5836ca790536ab0d94b22c5ed4716fcd9e2c9ea98a045fd6e76f2ef5ed78f13f)
            check_type(argname="argument route_spec", value=route_spec, expected_type=type_hints["route_spec"])
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "route_spec": route_spec,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name

    @builtins.property
    def route_spec(self) -> "GatewayRouteSpec":
        '''(experimental) What protocol the route uses.

        :stability: experimental
        '''
        result = self._values.get("route_spec")
        assert result is not None, "Required property 'route_spec' is missing"
        return typing.cast("GatewayRouteSpec", result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the GatewayRoute.

        :default: - an automatically generated name

        :stability: experimental
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GatewayRouteHostnameMatch(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.GatewayRouteHostnameMatch",
):
    '''(experimental) Used to generate host name matching methods.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-grpc",
            route_spec=appmesh.GatewayRouteSpec.grpc(
                route_target=virtual_service,
                match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                    hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="endsWith")
    @builtins.classmethod
    def ends_with(cls, suffix: builtins.str) -> "GatewayRouteHostnameMatch":
        '''(experimental) The value of the host name with the given name must end with the specified characters.

        :param suffix: The specified ending characters of the host name to match on.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1151bcc08a91419e514e83bf9a9da4b5ea3509eaac11a8f6cf6a150a9dc24fab)
            check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
        return typing.cast("GatewayRouteHostnameMatch", jsii.sinvoke(cls, "endsWith", [suffix]))

    @jsii.member(jsii_name="exactly")
    @builtins.classmethod
    def exactly(cls, name: builtins.str) -> "GatewayRouteHostnameMatch":
        '''(experimental) The value of the host name must match the specified value exactly.

        :param name: The exact host name to match on.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eed3165c63ec1d94f751485e32a0d066dc9b16320860ae32932d758239dc51e9)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        return typing.cast("GatewayRouteHostnameMatch", jsii.sinvoke(cls, "exactly", [name]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteHostnameMatchConfig":
        '''(experimental) Returns the gateway route host name match configuration.

        :param scope: -

        :stability: experimental
        '''
        ...


class _GatewayRouteHostnameMatchProxy(GatewayRouteHostnameMatch):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteHostnameMatchConfig":
        '''(experimental) Returns the gateway route host name match configuration.

        :param scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__baf0a63bce82285fb8c2438348f817523a582a6391c4fa156d3d8d1f79fbdb3f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("GatewayRouteHostnameMatchConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GatewayRouteHostnameMatch).__jsii_proxy_class__ = lambda : _GatewayRouteHostnameMatchProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteHostnameMatchConfig",
    jsii_struct_bases=[],
    name_mapping={"hostname_match": "hostnameMatch"},
)
class GatewayRouteHostnameMatchConfig:
    def __init__(
        self,
        *,
        hostname_match: typing.Union[CfnGatewayRoute.GatewayRouteHostnameMatchProperty, typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''(experimental) Configuration for gateway route host name match.

        :param hostname_match: (experimental) GatewayRoute CFN configuration for host name match.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            gateway_route_hostname_match_config = appmesh.GatewayRouteHostnameMatchConfig(
                hostname_match=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                    exact="exact",
                    suffix="suffix"
                )
            )
        '''
        if isinstance(hostname_match, dict):
            hostname_match = CfnGatewayRoute.GatewayRouteHostnameMatchProperty(**hostname_match)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__799998f5b2be778d1a46aaf944d8e1e8016b6f31c7117dc78ba2fdf51723f3bc)
            check_type(argname="argument hostname_match", value=hostname_match, expected_type=type_hints["hostname_match"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "hostname_match": hostname_match,
        }

    @builtins.property
    def hostname_match(self) -> CfnGatewayRoute.GatewayRouteHostnameMatchProperty:
        '''(experimental) GatewayRoute CFN configuration for host name match.

        :stability: experimental
        '''
        result = self._values.get("hostname_match")
        assert result is not None, "Required property 'hostname_match' is missing"
        return typing.cast(CfnGatewayRoute.GatewayRouteHostnameMatchProperty, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteHostnameMatchConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteProps",
    jsii_struct_bases=[GatewayRouteBaseProps],
    name_mapping={
        "route_spec": "routeSpec",
        "gateway_route_name": "gatewayRouteName",
        "virtual_gateway": "virtualGateway",
    },
)
class GatewayRouteProps(GatewayRouteBaseProps):
    def __init__(
        self,
        *,
        route_spec: "GatewayRouteSpec",
        gateway_route_name: typing.Optional[builtins.str] = None,
        virtual_gateway: "IVirtualGateway",
    ) -> None:
        '''(experimental) Properties to define a new GatewayRoute.

        :param route_spec: (experimental) What protocol the route uses.
        :param gateway_route_name: (experimental) The name of the GatewayRoute. Default: - an automatically generated name
        :param virtual_gateway: (experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            # gateway_route_spec: appmesh.GatewayRouteSpec
            # virtual_gateway: appmesh.VirtualGateway
            
            gateway_route_props = appmesh.GatewayRouteProps(
                route_spec=gateway_route_spec,
                virtual_gateway=virtual_gateway,
            
                # the properties below are optional
                gateway_route_name="gatewayRouteName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a8a31dca201e49fe7bbde01388e63c746e9c04d68092eb868e9b0a2708e6c46)
            check_type(argname="argument route_spec", value=route_spec, expected_type=type_hints["route_spec"])
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
            check_type(argname="argument virtual_gateway", value=virtual_gateway, expected_type=type_hints["virtual_gateway"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "route_spec": route_spec,
            "virtual_gateway": virtual_gateway,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name

    @builtins.property
    def route_spec(self) -> "GatewayRouteSpec":
        '''(experimental) What protocol the route uses.

        :stability: experimental
        '''
        result = self._values.get("route_spec")
        assert result is not None, "Required property 'route_spec' is missing"
        return typing.cast("GatewayRouteSpec", result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the GatewayRoute.

        :default: - an automatically generated name

        :stability: experimental
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def virtual_gateway(self) -> "IVirtualGateway":
        '''(experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway")
        assert result is not None, "Required property 'virtual_gateway' is missing"
        return typing.cast("IVirtualGateway", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GatewayRouteSpec(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.GatewayRouteSpec",
):
    '''(experimental) Used to generate specs with different protocols for a GatewayRoute.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-grpc",
            route_spec=appmesh.GatewayRouteSpec.grpc(
                route_target=virtual_service,
                match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                    hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
                    # This disables the default rewrite to virtual service name and retain original request.
                    rewrite_request_hostname=False
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="grpc")
    @builtins.classmethod
    def grpc(
        cls,
        *,
        match: typing.Union["GrpcGatewayRouteMatch", typing.Dict[builtins.str, typing.Any]],
        route_target: "IVirtualService",
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''(experimental) Creates an gRPC Based GatewayRoute.

        :param match: (experimental) The criterion for determining a request match for this GatewayRoute.
        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        '''
        options = GrpcGatewayRouteSpecOptions(
            match=match, route_target=route_target, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "grpc", [options]))

    @jsii.member(jsii_name="http")
    @builtins.classmethod
    def http(
        cls,
        *,
        route_target: "IVirtualService",
        match: typing.Optional[typing.Union["HttpGatewayRouteMatch", typing.Dict[builtins.str, typing.Any]]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''(experimental) Creates an HTTP Based GatewayRoute.

        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param match: (experimental) The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        '''
        options = HttpGatewayRouteSpecOptions(
            route_target=route_target, match=match, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "http", [options]))

    @jsii.member(jsii_name="http2")
    @builtins.classmethod
    def http2(
        cls,
        *,
        route_target: "IVirtualService",
        match: typing.Optional[typing.Union["HttpGatewayRouteMatch", typing.Dict[builtins.str, typing.Any]]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''(experimental) Creates an HTTP2 Based GatewayRoute.

        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param match: (experimental) The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        '''
        options = HttpGatewayRouteSpecOptions(
            route_target=route_target, match=match, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "http2", [options]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteSpecConfig":
        '''(experimental) Called when the GatewayRouteSpec type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        ...


class _GatewayRouteSpecProxy(GatewayRouteSpec):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteSpecConfig":
        '''(experimental) Called when the GatewayRouteSpec type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__657023920c35675c379947c879d292b103b2a0f9a0f462392119768761517d38)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("GatewayRouteSpecConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GatewayRouteSpec).__jsii_proxy_class__ = lambda : _GatewayRouteSpecProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteSpecConfig",
    jsii_struct_bases=[],
    name_mapping={
        "grpc_spec_config": "grpcSpecConfig",
        "http2_spec_config": "http2SpecConfig",
        "http_spec_config": "httpSpecConfig",
        "priority": "priority",
    },
)
class GatewayRouteSpecConfig:
    def __init__(
        self,
        *,
        grpc_spec_config: typing.Optional[typing.Union[CfnGatewayRoute.GrpcGatewayRouteProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        http2_spec_config: typing.Optional[typing.Union[CfnGatewayRoute.HttpGatewayRouteProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        http_spec_config: typing.Optional[typing.Union[CfnGatewayRoute.HttpGatewayRouteProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) All Properties for GatewayRoute Specs.

        :param grpc_spec_config: (experimental) The spec for a grpc gateway route. Default: - no grpc spec
        :param http2_spec_config: (experimental) The spec for an http2 gateway route. Default: - no http2 spec
        :param http_spec_config: (experimental) The spec for an http gateway route. Default: - no http spec
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            gateway_route_spec_config = appmesh.GatewayRouteSpecConfig(
                grpc_spec_config=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
            
                            # the properties below are optional
                            port=123
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        port=123,
                        service_name="serviceName"
                    )
                ),
                http2_spec_config=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
            
                            # the properties below are optional
                            port=123
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
            
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                http_spec_config=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
            
                            # the properties below are optional
                            port=123
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
            
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                priority=123
            )
        '''
        if isinstance(grpc_spec_config, dict):
            grpc_spec_config = CfnGatewayRoute.GrpcGatewayRouteProperty(**grpc_spec_config)
        if isinstance(http2_spec_config, dict):
            http2_spec_config = CfnGatewayRoute.HttpGatewayRouteProperty(**http2_spec_config)
        if isinstance(http_spec_config, dict):
            http_spec_config = CfnGatewayRoute.HttpGatewayRouteProperty(**http_spec_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d931c774caf375c63c1b6324fac2ee00505bab2aa5cde208a7aa175cce7e655)
            check_type(argname="argument grpc_spec_config", value=grpc_spec_config, expected_type=type_hints["grpc_spec_config"])
            check_type(argname="argument http2_spec_config", value=http2_spec_config, expected_type=type_hints["http2_spec_config"])
            check_type(argname="argument http_spec_config", value=http_spec_config, expected_type=type_hints["http_spec_config"])
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if grpc_spec_config is not None:
            self._values["grpc_spec_config"] = grpc_spec_config
        if http2_spec_config is not None:
            self._values["http2_spec_config"] = http2_spec_config
        if http_spec_config is not None:
            self._values["http_spec_config"] = http_spec_config
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def grpc_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty]:
        '''(experimental) The spec for a grpc gateway route.

        :default: - no grpc spec

        :stability: experimental
        '''
        result = self._values.get("grpc_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty], result)

    @builtins.property
    def http2_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty]:
        '''(experimental) The spec for an http2 gateway route.

        :default: - no http2 spec

        :stability: experimental
        '''
        result = self._values.get("http2_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty], result)

    @builtins.property
    def http_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty]:
        '''(experimental) The spec for an http gateway route.

        :default: - no http spec

        :stability: experimental
        '''
        result = self._values.get("http_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty], result)

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority

        :stability: experimental
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteSpecConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcConnectionPool",
    jsii_struct_bases=[],
    name_mapping={"max_requests": "maxRequests"},
)
class GrpcConnectionPool:
    def __init__(self, *, max_requests: jsii.Number) -> None:
        '''(experimental) Connection pool properties for gRPC listeners.

        :param max_requests: (experimental) The maximum requests in the pool. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with a gRPC listener with a connection pool set
            # mesh: appmesh.Mesh
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
                # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
                # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
                # By default, the response type is assumed to be LOAD_BALANCER
                service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=80,
                    connection_pool=cdk.aws_appmesh.HttpConnectionPool(
                        max_connections=100,
                        max_pending_requests=10
                    )
                )]
            )
            
            # A Virtual Gateway with a gRPC listener with a connection pool set
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
                        max_requests=10
                    )
                )],
                virtual_gateway_name="gateway"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__190315858e31c843ae9f8362a0b03d6ee91c5790206676ca2ab16e37b099e5d7)
            check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "max_requests": max_requests,
        }

    @builtins.property
    def max_requests(self) -> jsii.Number:
        '''(experimental) The maximum requests in the pool.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("max_requests")
        assert result is not None, "Required property 'max_requests' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcConnectionPool(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcGatewayListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "port": "port",
        "tls": "tls",
    },
)
class GrpcGatewayListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[typing.Union[GrpcConnectionPool, typing.Dict[builtins.str, typing.Any]]] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        port: typing.Optional[jsii.Number] = None,
        tls: typing.Optional[typing.Union["ListenerTlsOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) Represents the properties needed to define GRPC Listeners for a VirtualGateway.

        :param connection_pool: (experimental) Connection pool for http listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param tls: (experimental) Represents the configuration for enabling TLS on a listener. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with listener TLS from an ACM provided certificate
            # cert: certificatemanager.Certificate
            # mesh: appmesh.Mesh
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.dns("node"),
                listeners=[appmesh.VirtualNodeListener.grpc(
                    port=80,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.acm(cert)
                    )
                )]
            )
            
            # A Virtual Gateway with listener TLS from a customer provided file certificate
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
                    )
                )],
                virtual_gateway_name="gateway"
            )
            
            # A Virtual Gateway with listener TLS from a SDS provided certificate
            gateway2 = appmesh.VirtualGateway(self, "gateway2",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.http2(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.sds("secrete_certificate")
                    )
                )],
                virtual_gateway_name="gateway2"
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = GrpcConnectionPool(**connection_pool)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f1114116fcfa518285e7f1673c23f7b0adc7fc2a2e49f012de7012c0a90d294f)
            check_type(argname="argument connection_pool", value=connection_pool, expected_type=type_hints["connection_pool"])
            check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if port is not None:
            self._values["port"] = port
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[GrpcConnectionPool]:
        '''(experimental) Connection pool for http listeners.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[GrpcConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check information for the listener.

        :default: - no healthcheck

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port to listen for connections on.

        :default: - 8080

        :stability: experimental
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''(experimental) Represents the configuration for enabling TLS on a listener.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls")
        return typing.cast(typing.Optional["ListenerTlsOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcGatewayListenerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcGatewayRouteMatch",
    jsii_struct_bases=[],
    name_mapping={
        "hostname": "hostname",
        "metadata": "metadata",
        "rewrite_request_hostname": "rewriteRequestHostname",
        "service_name": "serviceName",
    },
)
class GrpcGatewayRouteMatch:
    def __init__(
        self,
        *,
        hostname: typing.Optional[GatewayRouteHostnameMatch] = None,
        metadata: typing.Optional[typing.Sequence["HeaderMatch"]] = None,
        rewrite_request_hostname: typing.Optional[builtins.bool] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The criterion for determining a request match for this GatewayRoute.

        :param hostname: (experimental) Create host name based gRPC gateway route match. Default: - no matching on host name
        :param metadata: (experimental) Create metadata based gRPC gateway route match. All specified metadata must match for the route to match. Default: - no matching on metadata
        :param rewrite_request_hostname: (experimental) When ``true``, rewrites the original request received at the Virtual Gateway to the destination Virtual Service name. When ``false``, retains the original hostname from the request. Default: true
        :param service_name: (experimental) Create service name based gRPC gateway route match. Default: - no matching on service name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-grpc",
                route_spec=appmesh.GatewayRouteSpec.grpc(
                    route_target=virtual_s