'''
# Amazon API Gateway Construct Library

Amazon API Gateway is a fully managed service that makes it easy for developers
to publish, maintain, monitor, and secure APIs at any scale. Create an API to
access data, business logic, or functionality from your back-end services, such
as applications running on Amazon Elastic Compute Cloud (Amazon EC2), code
running on AWS Lambda, or any web application.

## Table of Contents

* [Defining APIs](#defining-apis)

  * [Breaking up Methods and Resources across Stacks](#breaking-up-methods-and-resources-across-stacks)
* [AWS Lambda-backed APIs](#aws-lambda-backed-apis)
* [AWS StepFunctions backed APIs](#aws-stepfunctions-backed-APIs)
* [Integration Targets](#integration-targets)
* [Usage Plan & API Keys](#usage-plan--api-keys)
* [Working with models](#working-with-models)
* [Default Integration and Method Options](#default-integration-and-method-options)
* [Proxy Routes](#proxy-routes)
* [Authorizers](#authorizers)

  * [IAM-based authorizer](#iam-based-authorizer)
  * [Lambda-based token authorizer](#lambda-based-token-authorizer)
  * [Lambda-based request authorizer](#lambda-based-request-authorizer)
  * [Cognito User Pools authorizer](#cognito-user-pools-authorizer)
* [Mutual TLS](#mutal-tls-mtls)
* [Deployments](#deployments)

  * [Deep dive: Invalidation of deployments](#deep-dive-invalidation-of-deployments)
* [Custom Domains](#custom-domains)
* [Access Logging](#access-logging)
* [Cross Origin Resource Sharing (CORS)](#cross-origin-resource-sharing-cors)
* [Endpoint Configuration](#endpoint-configuration)
* [Private Integrations](#private-integrations)
* [Gateway Response](#gateway-response)
* [OpenAPI Definition](#openapi-definition)

  * [Endpoint configuration](#endpoint-configuration)
* [Metrics](#metrics)
* [APIGateway v2](#apigateway-v2)

## Defining APIs

APIs are defined as a hierarchy of resources and methods. `addResource` and
`addMethod` can be used to build this hierarchy. The root resource is
`api.root`.

For example, the following code defines an API that includes the following HTTP
endpoints: `ANY /`, `GET /books`, `POST /books`, `GET /books/{book_id}`, `DELETE /books/{book_id}`.

```python
api = apigateway.RestApi(self, "books-api")

api.root.add_method("ANY")

books = api.root.add_resource("books")
books.add_method("GET")
books.add_method("POST")

book = books.add_resource("{book_id}")
book.add_method("GET")
book.add_method("DELETE")
```

## AWS Lambda-backed APIs

A very common practice is to use Amazon API Gateway with AWS Lambda as the
backend integration. The `LambdaRestApi` construct makes it easy:

The following code defines a REST API that routes all requests to the
specified AWS Lambda function:

```python
# backend: lambda.Function

apigateway.LambdaRestApi(self, "myapi",
    handler=backend
)
```

You can also supply `proxy: false`, in which case you will have to explicitly
define the API model:

```python
# backend: lambda.Function

api = apigateway.LambdaRestApi(self, "myapi",
    handler=backend,
    proxy=False
)

items = api.root.add_resource("items")
items.add_method("GET") # GET /items
items.add_method("POST") # POST /items

item = items.add_resource("{item}")
item.add_method("GET") # GET /items/{item}

# the default integration for methods is "handler", but one can
# customize this behavior per method or even a sub path.
item.add_method("DELETE", apigateway.HttpIntegration("http://amazon.com"))
```

## AWS StepFunctions backed APIs

You can use Amazon API Gateway with AWS Step Functions as the backend integration, specifically Synchronous Express Workflows.

The `StepFunctionsRestApi` only supports integration with Synchronous Express state machine. The `StepFunctionsRestApi` construct makes this easy by setting up input, output and error mapping.

The construct sets up an API endpoint and maps the `ANY` HTTP method and any calls to the API endpoint starts an express workflow execution for the underlying state machine.

Invoking the endpoint with any HTTP method (`GET`, `POST`, `PUT`, `DELETE`, ...) in the example below will send the request to the state machine as a new execution. On success, an HTTP code `200` is returned with the execution output as the Response Body.

If the execution fails, an HTTP `500` response is returned with the `error` and `cause` from the execution output as the Response Body. If the request is invalid (ex. bad execution input) HTTP code `400` is returned.

The response from the invocation contains only the `output` field from the
[StartSyncExecution](https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartSyncExecution.html#API_StartSyncExecution_ResponseSyntax) API.
In case of failures, the fields `error` and `cause` are returned as part of the response.
Other metadata such as billing details, AWS account ID and resource ARNs are not returned in the API response.

By default, a `prod` stage is provisioned.

In order to reduce the payload size sent to AWS Step Functions, `headers` are not forwarded to the Step Functions execution input. It is possible to choose whether `headers`,  `requestContext`, `path`, `querystring`, and `authorizer` are included or not. By default, `headers` are excluded in all requests.

More details about AWS Step Functions payload limit can be found at https://docs.aws.amazon.com/step-functions/latest/dg/limits-overview.html#service-limits-task-executions.

The following code defines a REST API that routes all requests to the specified AWS StepFunctions state machine:

```python
state_machine_definition = stepfunctions.Pass(self, "PassState")

state_machine = stepfunctions.StateMachine(self, "StateMachine",
    definition=state_machine_definition,
    state_machine_type=stepfunctions.StateMachineType.EXPRESS
)

apigateway.StepFunctionsRestApi(self, "StepFunctionsRestApi",
    deploy=True,
    state_machine=state_machine
)
```

When the REST API endpoint configuration above is invoked using POST, as follows -

```bash
curl -X POST -d '{ "customerId": 1 }' https://example.com/
```

AWS Step Functions will receive the request body in its input as follows:

```json
{
  "body": {
    "customerId": 1
  },
  "path": "/",
  "querystring": {}
}
```

When the endpoint is invoked at path '/users/5' using the HTTP GET method as below:

```bash
curl -X GET https://example.com/users/5?foo=bar
```

AWS Step Functions will receive the following execution input:

```json
{
  "body": {},
  "path": {
     "users": "5"
  },
  "querystring": {
    "foo": "bar"
  }
}
```

Additional information around the request such as the request context, authorizer context, and headers can be included as part of the input
forwarded to the state machine. The following example enables headers to be included in the input but not query string.

```python
apigateway.StepFunctionsRestApi(self, "StepFunctionsRestApi",
    state_machine=machine,
    headers=True,
    path=False,
    querystring=False,
    authorizer=False,
    request_context=apigateway.aws_apigateway.RequestContext(
        caller=True,
        user=True
    )
)
```

In such a case, when the endpoint is invoked as below:

```bash
curl -X GET https://example.com/
```

AWS Step Functions will receive the following execution input:

```json
{
  "headers": {
    "Accept": "...",
    "CloudFront-Forwarded-Proto": "...",
  },
  "requestContext": {
     "accountId": "...",
     "apiKey": "...",
  },
  "body": {}
}
```

### Breaking up Methods and Resources across Stacks

It is fairly common for REST APIs with a large number of Resources and Methods to hit the [CloudFormation
limit](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) of 500 resources per
stack.

To help with this, Resources and Methods for the same REST API can be re-organized across multiple stacks. A common
way to do this is to have a stack per Resource or groups of Resources, but this is not the only possible way.
The following example uses sets up two Resources '/pets' and '/books' in separate stacks using nested stacks:

```python
from monocdk.aws_apigateway import IntegrationResponse, MethodResponse, IntegrationResponse, MethodResponse
from monocdk import App, CfnOutput, NestedStack, NestedStackProps, Stack
from constructs import Construct
from monocdk.aws_apigateway import Deployment, Method, MockIntegration, PassthroughBehavior, RestApi, Stage

#
# This file showcases how to split up a RestApi's Resources and Methods across nested stacks.
#
# The root stack 'RootStack' first defines a RestApi.
# Two nested stacks BooksStack and PetsStack, create corresponding Resources '/books' and '/pets'.
# They are then deployed to a 'prod' Stage via a third nested stack - DeployStack.
#
# To verify this worked, go to the APIGateway
#

class RootStack(Stack):
    def __init__(self, scope):
        super().__init__(scope, "integ-restapi-import-RootStack")

        rest_api = RestApi(self, "RestApi",
            deploy=False
        )
        rest_api.root.add_method("ANY")

        pets_stack = PetsStack(self,
            rest_api_id=rest_api.rest_api_id,
            root_resource_id=rest_api.rest_api_root_resource_id
        )
        books_stack = BooksStack(self,
            rest_api_id=rest_api.rest_api_id,
            root_resource_id=rest_api.rest_api_root_resource_id
        )
        DeployStack(self,
            rest_api_id=rest_api.rest_api_id,
            methods=pets_stack.methods.concat(books_stack.methods)
        )

        CfnOutput(self, "PetsURL",
            value=f"https://{restApi.restApiId}.execute-api.{this.region}.amazonaws.com/prod/pets"
        )

        CfnOutput(self, "BooksURL",
            value=f"https://{restApi.restApiId}.execute-api.{this.region}.amazonaws.com/prod/books"
        )

class PetsStack(NestedStack):

    def __init__(self, scope, *, restApiId, rootResourceId, parameters=None, timeout=None, notificationArns=None, removalPolicy=None):
        super().__init__(scope, "integ-restapi-import-PetsStack", restApiId=restApiId, rootResourceId=rootResourceId, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy)

        api = RestApi.from_rest_api_attributes(self, "RestApi",
            rest_api_id=rest_api_id,
            root_resource_id=root_resource_id
        )

        method = api.root.add_resource("pets").add_method("GET", MockIntegration(
            integration_responses=[IntegrationResponse(
                status_code="200"
            )],
            passthrough_behavior=PassthroughBehavior.NEVER,
            request_templates={
                "application/json": "{ \"statusCode\": 200 }"
            }
        ),
            method_responses=[MethodResponse(status_code="200")]
        )

        self.methods.push(method)

class BooksStack(NestedStack):

    def __init__(self, scope, *, restApiId, rootResourceId, parameters=None, timeout=None, notificationArns=None, removalPolicy=None):
        super().__init__(scope, "integ-restapi-import-BooksStack", restApiId=restApiId, rootResourceId=rootResourceId, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy)

        api = RestApi.from_rest_api_attributes(self, "RestApi",
            rest_api_id=rest_api_id,
            root_resource_id=root_resource_id
        )

        method = api.root.add_resource("books").add_method("GET", MockIntegration(
            integration_responses=[IntegrationResponse(
                status_code="200"
            )],
            passthrough_behavior=PassthroughBehavior.NEVER,
            request_templates={
                "application/json": "{ \"statusCode\": 200 }"
            }
        ),
            method_responses=[MethodResponse(status_code="200")]
        )

        self.methods.push(method)

class DeployStack(NestedStack):
    def __init__(self, scope, *, restApiId, methods=None, parameters=None, timeout=None, notificationArns=None, removalPolicy=None):
        super().__init__(scope, "integ-restapi-import-DeployStack", restApiId=restApiId, methods=methods, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy)

        deployment = Deployment(self, "Deployment",
            api=RestApi.from_rest_api_id(self, "RestApi", rest_api_id)
        )
        if methods:
            for method in methods:
                deployment.node.add_dependency(method)
        Stage(self, "Stage", deployment=deployment)

RootStack(App())
```

## Integration Targets

Methods are associated with backend integrations, which are invoked when this
method is called. API Gateway supports the following integrations:

* `MockIntegration` - can be used to test APIs. This is the default
  integration if one is not specified.
* `LambdaIntegration` - can be used to invoke an AWS Lambda function.
* `AwsIntegration` - can be used to invoke arbitrary AWS service APIs.
* `HttpIntegration` - can be used to invoke HTTP endpoints.

The following example shows how to integrate the `GET /book/{book_id}` method to
an AWS Lambda function:

```python
# get_book_handler: lambda.Function
# book: apigateway.Resource


get_book_integration = apigateway.LambdaIntegration(get_book_handler)
book.add_method("GET", get_book_integration)
```

Integration options can be optionally be specified:

```python
# get_book_handler: lambda.Function
# get_book_integration: apigateway.LambdaIntegration


get_book_integration = apigateway.LambdaIntegration(get_book_handler,
    content_handling=apigateway.ContentHandling.CONVERT_TO_TEXT,  # convert to base64
    credentials_passthrough=True
)
```

Method options can optionally be specified when adding methods:

```python
# book: apigateway.Resource
# get_book_integration: apigateway.LambdaIntegration


book.add_method("GET", get_book_integration,
    authorization_type=apigateway.AuthorizationType.IAM,
    api_key_required=True
)
```

It is possible to also integrate with AWS services in a different region. The following code integrates with Amazon SQS in the
`eu-west-1` region.

```python
get_message_integration = apigateway.AwsIntegration(
    service="sqs",
    path="queueName",
    region="eu-west-1"
)
```

## Usage Plan & API Keys

A usage plan specifies who can access one or more deployed API stages and methods, and the rate at which they can be
accessed. The plan uses API keys to identify API clients and meters access to the associated API stages for each key.
Usage plans also allow configuring throttling limits and quota limits that are enforced on individual client API keys.

The following example shows how to create and asscociate a usage plan and an API key:

```python
# integration: apigateway.LambdaIntegration


api = apigateway.RestApi(self, "hello-api")

v1 = api.root.add_resource("v1")
echo = v1.add_resource("echo")
echo_method = echo.add_method("GET", integration, api_key_required=True)

plan = api.add_usage_plan("UsagePlan",
    name="Easy",
    throttle=apigateway.aws_apigateway.ThrottleSettings(
        rate_limit=10,
        burst_limit=2
    )
)

key = api.add_api_key("ApiKey")
plan.add_api_key(key)
```

To associate a plan to a given RestAPI stage:

```python
# plan: apigateway.UsagePlan
# api: apigateway.RestApi
# echo_method: apigateway.Method


plan.add_api_stage(
    stage=api.deployment_stage,
    throttle=[apigateway.aws_apigateway.ThrottlingPerMethod(
        method=echo_method,
        throttle=apigateway.aws_apigateway.ThrottleSettings(
            rate_limit=10,
            burst_limit=2
        )
    )
    ]
)
```

Existing usage plans can be imported into a CDK app using its id.

```python
imported_usage_plan = apigateway.UsagePlan.from_usage_plan_id(self, "imported-usage-plan", "<usage-plan-key-id>")
```

The name and value of the API Key can be specified at creation; if not
provided, a name and value will be automatically generated by API Gateway.

```python
# api: apigateway.RestApi

key = api.add_api_key("ApiKey",
    api_key_name="myApiKey1",
    value="MyApiKeyThatIsAtLeast20Characters"
)
```

Existing API keys can also be imported into a CDK app using its id.

```python
imported_key = apigateway.ApiKey.from_api_key_id(self, "imported-key", "<api-key-id>")
```

The "grant" methods can be used to give prepackaged sets of permissions to other resources. The
following code provides read permission to an API key.

```python
# imported_key: apigateway.ApiKey
# lambda_fn: lambda.Function

imported_key.grant_read(lambda_fn)
```

### ⚠️ Multiple API Keys

It is possible to specify multiple API keys for a given Usage Plan, by calling `usagePlan.addApiKey()`.

When using multiple API keys, a past bug of the CDK prevents API key associations to a Usage Plan to be deleted.
If the CDK app had the [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html) - `@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId` - enabled when the API
keys were created, then the app will not be affected by this bug.

If this is not the case, you will need to ensure that the CloudFormation [logical ids](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) of the API keys that are not
being deleted remain unchanged.
Make note of the logical ids of these API keys before removing any, and set it as part of the `addApiKey()` method:

```python
# usageplan: apigateway.UsagePlan
# api_key: apigateway.ApiKey


usageplan.add_api_key(api_key,
    override_logical_id="..."
)
```

### Rate Limited API Key

In scenarios where you need to create a single api key and configure rate limiting for it, you can use `RateLimitedApiKey`.
This construct lets you specify rate limiting properties which should be applied only to the api key being created.
The API key created has the specified rate limits, such as quota and throttles, applied.

The following example shows how to use a rate limited api key :

```python
# api: apigateway.RestApi


key = apigateway.RateLimitedApiKey(self, "rate-limited-api-key",
    customer_id="hello-customer",
    resources=[api],
    quota=apigateway.aws_apigateway.QuotaSettings(
        limit=10000,
        period=apigateway.Period.MONTH
    )
)
```

## Working with models

When you work with Lambda integrations that are not Proxy integrations, you
have to define your models and mappings for the request, response, and integration.

```python
hello = lambda_.Function(self, "hello",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="hello.handler",
    code=lambda_.Code.from_asset("lambda")
)

api = apigateway.RestApi(self, "hello-api")
resource = api.root.add_resource("v1")
```

You can define more parameters on the integration to tune the behavior of API Gateway

```python
# hello: lambda.Function


integration = apigateway.LambdaIntegration(hello,
    proxy=False,
    request_parameters={
        # You can define mapping parameters from your method to your integration
        # - Destination parameters (the key) are the integration parameters (used in mappings)
        # - Source parameters (the value) are the source request parameters or expressions
        # @see: https://docs.aws.amazon.com/apigateway/latest/developerguide/request-response-data-mappings.html
        "integration.request.querystring.who": "method.request.querystring.who"
    },
    allow_test_invoke=True,
    request_templates={
        # You can define a mapping that will build a payload for your integration, based
        #  on the integration parameters that you have specified
        # Check: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
        "application/json": JSON.stringify({"action": "sayHello", "poll_id": "$util.escapeJavaScript($input.params('who'))"})
    },
    # This parameter defines the behavior of the engine is no suitable response template is found
    passthrough_behavior=apigateway.PassthroughBehavior.NEVER,
    integration_responses=[apigateway.aws_apigateway.IntegrationResponse(
        # Successful response from the Lambda function, no filter defined
        #  - the selectionPattern filter only tests the error message
        # We will set the response status code to 200
        status_code="200",
        response_templates={
            # This template takes the "message" result from the Lambda function, and embeds it in a JSON response
            # Check https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
            "application/json": JSON.stringify({"state": "ok", "greeting": "$util.escapeJavaScript($input.body)"})
        },
        response_parameters={
            # We can map response parameters
            # - Destination parameters (the key) are the response parameters (used in mappings)
            # - Source parameters (the value) are the integration response parameters or expressions
            "method.response.header.Content-Type": "'application/json'",
            "method.response.header.Access-Control-Allow-Origin": "'*'",
            "method.response.header.Access-Control-Allow-Credentials": "'true'"
        }
    ), apigateway.aws_apigateway.IntegrationResponse(
        # For errors, we check if the error message is not empty, get the error data
        selection_pattern="(\n|.)+",
        # We will set the response status code to 200
        status_code="400",
        response_templates={
            "application/json": JSON.stringify({"state": "error", "message": "$util.escapeJavaScript($input.path('$.errorMessage'))"})
        },
        response_parameters={
            "method.response.header.Content-Type": "'application/json'",
            "method.response.header.Access-Control-Allow-Origin": "'*'",
            "method.response.header.Access-Control-Allow-Credentials": "'true'"
        }
    )
    ]
)
```

You can define models for your responses (and requests)

```python
# api: apigateway.RestApi


# We define the JSON Schema for the transformed valid response
response_model = api.add_model("ResponseModel",
    content_type="application/json",
    model_name="ResponseModel",
    schema=apigateway.aws_apigateway.JsonSchema(
        schema=apigateway.JsonSchemaVersion.DRAFT4,
        title="pollResponse",
        type=apigateway.JsonSchemaType.OBJECT,
        properties={
            "state": apigateway.aws_apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
            "greeting": apigateway.aws_apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
        }
    )
)

# We define the JSON Schema for the transformed error response
error_response_model = api.add_model("ErrorResponseModel",
    content_type="application/json",
    model_name="ErrorResponseModel",
    schema=apigateway.aws_apigateway.JsonSchema(
        schema=apigateway.JsonSchemaVersion.DRAFT4,
        title="errorResponse",
        type=apigateway.JsonSchemaType.OBJECT,
        properties={
            "state": apigateway.aws_apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
            "message": apigateway.aws_apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
        }
    )
)
```

And reference all on your method definition.

```python
# integration: apigateway.LambdaIntegration
# resource: apigateway.Resource
# response_model: apigateway.Model
# error_response_model: apigateway.Model


resource.add_method("GET", integration,
    # We can mark the parameters as required
    request_parameters={
        "method.request.querystring.who": True
    },
    # we can set request validator options like below
    request_validator_options=apigateway.aws_apigateway.RequestValidatorOptions(
        request_validator_name="test-validator",
        validate_request_body=True,
        validate_request_parameters=False
    ),
    method_responses=[apigateway.aws_apigateway.MethodResponse(
        # Successful response from the integration
        status_code="200",
        # Define what parameters are allowed or not
        response_parameters={
            "method.response.header.Content-Type": True,
            "method.response.header.Access-Control-Allow-Origin": True,
            "method.response.header.Access-Control-Allow-Credentials": True
        },
        # Validate the schema on the response
        response_models={
            "application/json": response_model
        }
    ), apigateway.aws_apigateway.MethodResponse(
        # Same thing for the error responses
        status_code="400",
        response_parameters={
            "method.response.header.Content-Type": True,
            "method.response.header.Access-Control-Allow-Origin": True,
            "method.response.header.Access-Control-Allow-Credentials": True
        },
        response_models={
            "application/json": error_response_model
        }
    )
    ]
)
```

Specifying `requestValidatorOptions` automatically creates the RequestValidator construct with the given options.
However, if you have your RequestValidator already initialized or imported, use the `requestValidator` option instead.

## Default Integration and Method Options

The `defaultIntegration` and `defaultMethodOptions` properties can be used to
configure a default integration at any resource level. These options will be
used when defining method under this resource (recursively) with undefined
integration or options.

> If not defined, the default integration is `MockIntegration`. See reference
> documentation for default method options.

The following example defines the `booksBackend` integration as a default
integration. This means that all API methods that do not explicitly define an
integration will be routed to this AWS Lambda function.

```python
# books_backend: apigateway.LambdaIntegration

api = apigateway.RestApi(self, "books",
    default_integration=books_backend
)

books = api.root.add_resource("books")
books.add_method("GET") # integrated with `booksBackend`
books.add_method("POST") # integrated with `booksBackend`

book = books.add_resource("{book_id}")
book.add_method("GET")
```

A Method can be configured with authorization scopes. Authorization scopes are
used in conjunction with an [authorizer that uses Amazon Cognito user
pools](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool).
Read more about authorization scopes
[here](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes).

Authorization scopes for a Method can be configured using the `authorizationScopes` property as shown below -

```python
# books: apigateway.Resource


books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorization_type=apigateway.AuthorizationType.COGNITO,
    authorization_scopes=["Scope1", "Scope2"]
)
```

## Proxy Routes

The `addProxy` method can be used to install a greedy `{proxy+}` resource
on a path. By default, this also installs an `"ANY"` method:

```python
# resource: apigateway.Resource
# handler: lambda.Function

proxy = resource.add_proxy(
    default_integration=apigateway.LambdaIntegration(handler),

    # "false" will require explicitly adding methods on the `proxy` resource
    any_method=True
)
```

## Authorizers

API Gateway [supports several different authorization types](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-to-api.html)
that can be used for controlling access to your REST APIs.

### IAM-based authorizer

The following CDK code provides 'execute-api' permission to an IAM user, via IAM policies, for the 'GET' method on the `books` resource:

```python
# books: apigateway.Resource
# iam_user: iam.User


get_books = books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorization_type=apigateway.AuthorizationType.IAM
)

iam_user.attach_inline_policy(iam.Policy(self, "AllowBooks",
    statements=[
        iam.PolicyStatement(
            actions=["execute-api:Invoke"],
            effect=iam.Effect.ALLOW,
            resources=[get_books.method_arn]
        )
    ]
))
```

### Lambda-based token authorizer

API Gateway also allows [lambda functions to be used as authorizers](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html).

This module provides support for token-based Lambda authorizers. When a client makes a request to an API's methods configured with such
an authorizer, API Gateway calls the Lambda authorizer, which takes the caller's identity as input and returns an IAM policy as output.
A token-based Lambda authorizer (also called a token authorizer) receives the caller's identity in a bearer token, such as
a JSON Web Token (JWT) or an OAuth token.

API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
The event object that the handler is called with contains the `authorizationToken` and the `methodArn` from the request to the
API Gateway endpoint. The handler is expected to return the `principalId` (i.e. the client identifier) and a `policyDocument` stating
what the client is authorizer to perform.
See [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html) for a detailed specification on
inputs and outputs of the Lambda handler.

The following code attaches a token-based Lambda authorizer to the 'GET' Method of the Book resource:

```python
# auth_fn: lambda.Function
# books: apigateway.Resource


auth = apigateway.TokenAuthorizer(self, "booksAuthorizer",
    handler=auth_fn
)

books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorizer=auth
)
```

A full working example is shown below.

!cdk-integ pragma:ignore-assets

```python
from monocdk.aws_apigateway import IntegrationResponse, MethodResponse
import path as path
import monocdk.aws_lambda as lambda_
from monocdk import App, Stack
from monocdk.aws_apigateway import MockIntegration, PassthroughBehavior, RestApi, TokenAuthorizer

#
# Stack verification steps:
# * `curl -s -o /dev/null -w "%{http_code}" <url>` should return 401
# * `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: deny' <url>` should return 403
# * `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: allow' <url>` should return 200
#

app = App()
stack = Stack(app, "TokenAuthorizerInteg")

authorizer_fn = lambda_.Function(stack, "MyAuthorizerFunction",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=lambda_.AssetCode.from_asset(path.join(__dirname, "integ.token-authorizer.handler"))
)

restapi = RestApi(stack, "MyRestApi")

authorizer = TokenAuthorizer(stack, "MyAuthorizer",
    handler=authorizer_fn
)

restapi.root.add_method("ANY", MockIntegration(
    integration_responses=[IntegrationResponse(status_code="200")
    ],
    passthrough_behavior=PassthroughBehavior.NEVER,
    request_templates={
        "application/json": "{ \"statusCode\": 200 }"
    }
),
    method_responses=[MethodResponse(status_code="200")
    ],
    authorizer=authorizer
)
```

By default, the `TokenAuthorizer` looks for the authorization token in the request header with the key 'Authorization'. This can,
however, be modified by changing the `identitySource` property.

Authorizers can also be passed via the `defaultMethodOptions` property within the `RestApi` construct or the `Method` construct. Unless
explicitly overridden, the specified defaults will be applied across all `Method`s across the `RestApi` or across all `Resource`s,
depending on where the defaults were specified.

### Lambda-based request authorizer

This module provides support for request-based Lambda authorizers. When a client makes a request to an API's methods configured with such
an authorizer, API Gateway calls the Lambda authorizer, which takes specified parts of the request, known as identity sources,
as input and returns an IAM policy as output. A request-based Lambda authorizer (also called a request authorizer) receives
the identity sources in a series of values pulled from the request, from the headers, stage variables, query strings, and the context.

API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
The event object that the handler is called with contains the body of the request and the `methodArn` from the request to the
API Gateway endpoint. The handler is expected to return the `principalId` (i.e. the client identifier) and a `policyDocument` stating
what the client is authorizer to perform.
See [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html) for a detailed specification on
inputs and outputs of the Lambda handler.

The following code attaches a request-based Lambda authorizer to the 'GET' Method of the Book resource:

```python
# auth_fn: lambda.Function
# books: apigateway.Resource


auth = apigateway.RequestAuthorizer(self, "booksAuthorizer",
    handler=auth_fn,
    identity_sources=[apigateway.IdentitySource.header("Authorization")]
)

books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorizer=auth
)
```

A full working example is shown below.

!cdk-integ pragma:ignore-assets

```python
from monocdk.aws_apigateway import IntegrationResponse, MethodResponse
import path as path
import monocdk.aws_lambda as lambda_
from monocdk import App, Stack
from monocdk.aws_apigateway import MockIntegration, PassthroughBehavior, RestApi
from monocdk.aws_apigateway import RequestAuthorizer
from monocdk.aws_apigateway import IdentitySource

# Against the RestApi endpoint from the stack output, run
# `curl -s -o /dev/null -w "%{http_code}" <url>` should return 401
# `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: deny' <url>?allow=yes` should return 403
# `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: allow' <url>?allow=yes` should return 200

app = App()
stack = Stack(app, "RequestAuthorizerInteg")

authorizer_fn = lambda_.Function(stack, "MyAuthorizerFunction",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=lambda_.AssetCode.from_asset(path.join(__dirname, "integ.request-authorizer.handler"))
)

restapi = RestApi(stack, "MyRestApi")

authorizer = RequestAuthorizer(stack, "MyAuthorizer",
    handler=authorizer_fn,
    identity_sources=[IdentitySource.header("Authorization"), IdentitySource.query_string("allow")]
)

restapi.root.add_method("ANY", MockIntegration(
    integration_responses=[IntegrationResponse(status_code="200")
    ],
    passthrough_behavior=PassthroughBehavior.NEVER,
    request_templates={
        "application/json": "{ \"statusCode\": 200 }"
    }
),
    method_responses=[MethodResponse(status_code="200")
    ],
    authorizer=authorizer
)
```

By default, the `RequestAuthorizer` does not pass any kind of information from the request. This can,
however, be modified by changing the `identitySource` property, and is required when specifying a value for caching.

Authorizers can also be passed via the `defaultMethodOptions` property within the `RestApi` construct or the `Method` construct. Unless
explicitly overridden, the specified defaults will be applied across all `Method`s across the `RestApi` or across all `Resource`s,
depending on where the defaults were specified.

### Cognito User Pools authorizer

API Gateway also allows [Amazon Cognito user pools as authorizer](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html)

The following snippet configures a Cognito user pool as an authorizer:

```python
# books: apigateway.Resource
user_pool = cognito.UserPool(self, "UserPool")

auth = apigateway.CognitoUserPoolsAuthorizer(self, "booksAuthorizer",
    cognito_user_pools=[user_pool]
)
books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorizer=auth,
    authorization_type=apigateway.AuthorizationType.COGNITO
)
```

## Mutual TLS (mTLS)

Mutual TLS can be configured to limit access to your API based by using client certificates instead of (or as an extension of) using authorization headers.

```python
# acm: Any


apigateway.DomainName(self, "domain-name",
    domain_name="example.com",
    certificate=acm.Certificate.from_certificate_arn(self, "cert", "arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d"),
    mtls=apigateway.aws_apigateway.MTLSConfig(
        bucket=s3.Bucket(self, "bucket"),
        key="truststore.pem",
        version="version"
    )
)
```

Instructions for configuring your trust store can be found [here](https://aws.amazon.com/blogs/compute/introducing-mutual-tls-authentication-for-amazon-api-gateway/).

## Deployments

By default, the `RestApi` construct will automatically create an API Gateway
[Deployment](https://docs.aws.amazon.com/apigateway/api-reference/resource/deployment/) and a "prod" [Stage](https://docs.aws.amazon.com/apigateway/api-reference/resource/stage/) which represent the API configuration you
defined in your CDK app. This means that when you deploy your app, your API will
be have open access from the internet via the stage URL.

The URL of your API can be obtained from the attribute `restApi.url`, and is
also exported as an `Output` from your stack, so it's printed when you `cdk deploy` your app:

```console
$ cdk deploy
...
books.booksapiEndpointE230E8D5 = https://6lyktd4lpk.execute-api.us-east-1.amazonaws.com/prod/
```

To disable this behavior, you can set `{ deploy: false }` when creating your
API. This means that the API will not be deployed and a stage will not be
created for it. You will need to manually define a `apigateway.Deployment` and
`apigateway.Stage` resources.

Use the `deployOptions` property to customize the deployment options of your
API.

The following example will configure API Gateway to emit logs and data traces to
AWS CloudWatch for all API calls:

> By default, an IAM role will be created and associated with API Gateway to
> allow it to write logs and metrics to AWS CloudWatch unless `cloudWatchRole` is
> set to `false`.

```python
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.aws_apigateway.StageOptions(
        logging_level=apigateway.MethodLoggingLevel.INFO,
        data_trace_enabled=True
    )
)
```

### Deep dive: Invalidation of deployments

API Gateway deployments are an immutable snapshot of the API. This means that we
want to automatically create a new deployment resource every time the API model
defined in our CDK app changes.

In order to achieve that, the AWS CloudFormation logical ID of the
`AWS::ApiGateway::Deployment` resource is dynamically calculated by hashing the
API configuration (resources, methods). This means that when the configuration
changes (i.e. a resource or method are added, configuration is changed), a new
logical ID will be assigned to the deployment resource. This will cause
CloudFormation to create a new deployment resource.

By default, old deployments are *deleted*. You can set `retainDeployments: true`
to allow users revert the stage to an old deployment manually.

## Custom Domains

To associate an API with a custom domain, use the `domainName` configuration when
you define your API:

```python
# acm_certificate_for_example_com: Any


api = apigateway.RestApi(self, "MyDomain",
    domain_name=apigateway.aws_apigateway.DomainNameOptions(
        domain_name="example.com",
        certificate=acm_certificate_for_example_com
    )
)
```

This will define a `DomainName` resource for you, along with a `BasePathMapping`
from the root of the domain to the deployment stage of the API. This is a common
set up.

To route domain traffic to an API Gateway API, use Amazon Route 53 to create an
alias record. An alias record is a Route 53 extension to DNS. It's similar to a
CNAME record, but you can create an alias record both for the root domain, such
as `example.com`, and for subdomains, such as `www.example.com`. (You can create
CNAME records only for subdomains.)

```python
import monocdk as route53
import monocdk as targets

# api: apigateway.RestApi
# hosted_zone_for_example_com: Any


route53.ARecord(self, "CustomDomainAliasRecord",
    zone=hosted_zone_for_example_com,
    target=route53.RecordTarget.from_alias(targets.ApiGateway(api))
)
```

You can also define a `DomainName` resource directly in order to customize the default behavior:

```python
# acm_certificate_for_example_com: Any


apigateway.DomainName(self, "custom-domain",
    domain_name="example.com",
    certificate=acm_certificate_for_example_com,
    endpoint_type=apigateway.EndpointType.EDGE,  # default is REGIONAL
    security_policy=apigateway.SecurityPolicy.TLS_1_2
)
```

Once you have a domain, you can map base paths of the domain to APIs.
The following example will map the URL [https://example.com/go-to-api1](https://example.com/go-to-api1)
to the `api1` API and [https://example.com/boom](https://example.com/boom) to the `api2` API.

```python
# domain: apigateway.DomainName
# api1: apigateway.RestApi
# api2: apigateway.RestApi


domain.add_base_path_mapping(api1, base_path="go-to-api1")
domain.add_base_path_mapping(api2, base_path="boom")
```

You can specify the API `Stage` to which this base path URL will map to. By default, this will be the
`deploymentStage` of the `RestApi`.

```python
# domain: apigateway.DomainName
# restapi: apigateway.RestApi


beta_deploy = apigateway.Deployment(self, "beta-deployment",
    api=restapi
)
beta_stage = apigateway.Stage(self, "beta-stage",
    deployment=beta_deploy
)
domain.add_base_path_mapping(restapi, base_path="api/beta", stage=beta_stage)
```

If you don't specify `basePath`, all URLs under this domain will be mapped
to the API, and you won't be able to map another API to the same domain:

```python
# domain: apigateway.DomainName
# api: apigateway.RestApi

domain.add_base_path_mapping(api)
```

This can also be achieved through the `mapping` configuration when defining the
domain as demonstrated above.

If you wish to setup this domain with an Amazon Route53 alias, use the `targets.ApiGatewayDomain`:

```python
# hosted_zone_for_example_com: Any
# domain_name: apigateway.DomainName

import monocdk as route53
import monocdk as targets


route53.ARecord(self, "CustomDomainAliasRecord",
    zone=hosted_zone_for_example_com,
    target=route53.RecordTarget.from_alias(targets.ApiGatewayDomain(domain_name))
)
```

## Access Logging

Access logging creates logs every time an API method is accessed. Access logs can have information on
who has accessed the API, how the caller accessed the API and what responses were generated.
Access logs are configured on a Stage of the RestApi.
Access logs can be expressed in a format of your choosing, and can contain any access details, with a
minimum that it must include the 'requestId'. The list of  variables that can be expressed in the access
log can be found
[here](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference).
Read more at [Setting Up CloudWatch API Logging in API
Gateway](https://docs.aws.amazon.com/apigateway/latest/developerguide/set-up-logging.html)

```python
# production stage
prd_log_group = logs.LogGroup(self, "PrdLogs")
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.aws_apigateway.StageOptions(
        access_log_destination=apigateway.LogGroupLogDestination(prd_log_group),
        access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
    )
)
deployment = apigateway.Deployment(self, "Deployment", api=api)

# development stage
dev_log_group = logs.LogGroup(self, "DevLogs")
apigateway.Stage(self, "dev",
    deployment=deployment,
    access_log_destination=apigateway.LogGroupLogDestination(dev_log_group),
    access_log_format=apigateway.AccessLogFormat.json_with_standard_fields(
        caller=False,
        http_method=True,
        ip=True,
        protocol=True,
        request_time=True,
        resource_path=True,
        response_length=True,
        status=True,
        user=True
    )
)
```

The following code will generate the access log in the [CLF format](https://en.wikipedia.org/wiki/Common_Log_Format).

```python
log_group = logs.LogGroup(self, "ApiGatewayAccessLogs")
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.aws_apigateway.StageOptions(
        access_log_destination=apigateway.LogGroupLogDestination(log_group),
        access_log_format=apigateway.AccessLogFormat.clf()
    )
)
```

You can also configure your own access log format by using the `AccessLogFormat.custom()` API.
`AccessLogField` provides commonly used fields. The following code configures access log to contain.

```python
log_group = logs.LogGroup(self, "ApiGatewayAccessLogs")
apigateway.RestApi(self, "books",
    deploy_options=apigateway.aws_apigateway.StageOptions(
        access_log_destination=apigateway.LogGroupLogDestination(log_group),
        access_log_format=apigateway.AccessLogFormat.custom(f"{apigateway.AccessLogField.contextRequestId()} {apigateway.AccessLogField.contextErrorMessage()} {apigateway.AccessLogField.contextErrorMessageString()}")
    )
)
```

You can use the `methodOptions` property to configure
[default method throttling](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html#apigateway-api-level-throttling-in-usage-plan)
for a stage. The following snippet configures the a stage that accepts
100 requests per minute, allowing burst up to 200 requests per minute.

```python
api = apigateway.RestApi(self, "books")
deployment = apigateway.Deployment(self, "my-deployment", api=api)
stage = apigateway.Stage(self, "my-stage",
    deployment=deployment,
    method_options={
        "/*/*": apigateway.aws_apigateway.MethodDeploymentOptions( # This special path applies to all resource paths and all HTTP methods
            throttling_rate_limit=100,
            throttling_burst_limit=200)
    }
)
```

Configuring `methodOptions` on the `deployOptions` of `RestApi` will set the
throttling behaviors on the default stage that is automatically created.

```python
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.aws_apigateway.StageOptions(
        method_options={
            "/*/*": apigateway.aws_apigateway.MethodDeploymentOptions( # This special path applies to all resource paths and all HTTP methods
                throttling_rate_limit=100,
                throttling_burst_limit=1000)
        }
    )
)
```

## Cross Origin Resource Sharing (CORS)

[Cross-Origin Resource Sharing (CORS)](https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS) is a mechanism
that uses additional HTTP headers to tell browsers to give a web application
running at one origin, access to selected resources from a different origin. A
web application executes a cross-origin HTTP request when it requests a resource
that has a different origin (domain, protocol, or port) from its own.

You can add the CORS [preflight](https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS#Preflighted_requests) OPTIONS
HTTP method to any API resource via the `defaultCorsPreflightOptions` option or by calling the `addCorsPreflight` on a specific resource.

The following example will enable CORS for all methods and all origins on all resources of the API:

```python
apigateway.RestApi(self, "api",
    default_cors_preflight_options=apigateway.aws_apigateway.CorsOptions(
        allow_origins=apigateway.Cors.ALL_ORIGINS,
        allow_methods=apigateway.Cors.ALL_METHODS
    )
)
```

The following example will add an OPTIONS method to the `myResource` API resource, which
only allows GET and PUT HTTP requests from the origin [https://amazon.com.](https://amazon.com.)

```python
# my_resource: apigateway.Resource


my_resource.add_cors_preflight(
    allow_origins=["https://amazon.com"],
    allow_methods=["GET", "PUT"]
)
```

See the
[`CorsOptions`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-apigateway.CorsOptions.html)
API reference for a detailed list of supported configuration options.

You can specify defaults this at the resource level, in which case they will be applied to the entire resource sub-tree:

```python
# resource: apigateway.Resource


subtree = resource.add_resource("subtree",
    default_cors_preflight_options=apigateway.aws_apigateway.CorsOptions(
        allow_origins=["https://amazon.com"]
    )
)
```

This means that all resources under `subtree` (inclusive) will have a preflight
OPTIONS added to them.

See [#906](https://github.com/aws/aws-cdk/issues/906) for a list of CORS
features which are not yet supported.

## Endpoint Configuration

API gateway allows you to specify an
[API Endpoint Type](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-endpoint-types.html).
To define an endpoint type for the API gateway, use `endpointConfiguration` property:

```python
api = apigateway.RestApi(self, "api",
    endpoint_configuration=apigateway.aws_apigateway.EndpointConfiguration(
        types=[apigateway.EndpointType.EDGE]
    )
)
```

You can also create an association between your Rest API and a VPC endpoint. By doing so,
API Gateway will generate a new
Route53 Alias DNS record which you can use to invoke your private APIs. More info can be found
[here](https://docs.aws.amazon.com/apigateway/latest/developerguide/associate-private-api-with-vpc-endpoint.html).

Here is an example:

```python
# some_endpoint: ec2.IVpcEndpoint


api = apigateway.RestApi(self, "api",
    endpoint_configuration=apigateway.aws_apigateway.EndpointConfiguration(
        types=[apigateway.EndpointType.PRIVATE],
        vpc_endpoints=[some_endpoint]
    )
)
```

By performing this association, we can invoke the API gateway using the following format:

```plaintext
https://{rest-api-id}-{vpce-id}.execute-api.{region}.amazonaws.com/{stage}
```

## Private Integrations

A private integration makes it simple to expose HTTP/HTTPS resources behind an
Amazon VPC for access by clients outside of the VPC. The private integration uses
an API Gateway resource of `VpcLink` to encapsulate connections between API
Gateway and targeted VPC resources.
The `VpcLink` is then attached to the `Integration` of a specific API Gateway
Method. The following code sets up a private integration with a network load
balancer -

```python
import monocdk as elbv2


vpc = ec2.Vpc(self, "VPC")
nlb = elbv2.NetworkLoadBalancer(self, "NLB",
    vpc=vpc
)
link = apigateway.VpcLink(self, "link",
    targets=[nlb]
)

integration = apigateway.Integration(
    type=apigateway.IntegrationType.HTTP_PROXY,
    options=elbv2.aws_apigateway.IntegrationOptions(
        connection_type=apigateway.ConnectionType.VPC_LINK,
        vpc_link=link
    )
)
```

The uri for the private integration, in the case of a VpcLink, will be set to the DNS name of
the VPC Link's NLB. If the VPC Link has multiple NLBs or the VPC Link is imported or the DNS
name cannot be determined for any other reason, the user is expected to specify the `uri`
property.

Any existing `VpcLink` resource can be imported into the CDK app via the `VpcLink.fromVpcLinkId()`.

```python
awesome_link = apigateway.VpcLink.from_vpc_link_id(self, "awesome-vpc-link", "us-east-1_oiuR12Abd")
```

## Gateway response

If the Rest API fails to process an incoming request, it returns to the client an error response without forwarding the
request to the integration backend. API Gateway has a set of standard response messages that are sent to the client for
each type of error. These error responses can be configured on the Rest API. The list of Gateway responses that can be
configured can be found [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html).
Learn more about [Gateway
Responses](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-gatewayResponse-definition.html).

The following code configures a Gateway Response when the response is 'access denied':

```python
api = apigateway.RestApi(self, "books-api")
api.add_gateway_response("test-response",
    type=apigateway.ResponseType.ACCESS_DENIED,
    status_code="500",
    response_headers={
        "Access-Control-Allow-Origin": "test.com",
        "test-key": "test-value"
    },
    templates={
        "application/json": "{ \"message\": $context.error.messageString, \"statusCode\": \"488\", \"type\": \"$context.error.responseType\" }"
    }
)
```

## OpenAPI Definition

CDK supports creating a REST API by importing an OpenAPI definition file. It currently supports OpenAPI v2.0 and OpenAPI
v3.0 definition files. Read more about [Configuring a REST API using
OpenAPI](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-import-api.html).

The following code creates a REST API using an external OpenAPI definition JSON file -

```python
# integration: apigateway.Integration


api = apigateway.SpecRestApi(self, "books-api",
    api_definition=apigateway.ApiDefinition.from_asset("path-to-file.json")
)

books_resource = api.root.add_resource("books")
books_resource.add_method("GET", integration)
```

It is possible to use the `addResource()` API to define additional API Gateway Resources.

**Note:** Deployment will fail if a Resource of the same name is already defined in the Open API specification.

**Note:** Any default properties configured, such as `defaultIntegration`, `defaultMethodOptions`, etc. will only be
applied to Resources and Methods defined in the CDK, and not the ones defined in the spec. Use the [API Gateway
extensions to OpenAPI](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions.html)
to configure these.

There are a number of limitations in using OpenAPI definitions in API Gateway. Read the [Amazon API Gateway important
notes for REST APIs](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-known-issues.html#api-gateway-known-issues-rest-apis)
for more details.

**Note:** When starting off with an OpenAPI definition using `SpecRestApi`, it is not possible to configure some
properties that can be configured directly in the OpenAPI specification file. This is to prevent people duplication
of these properties and potential confusion.

### Endpoint configuration

By default, `SpecRestApi` will create an edge optimized endpoint.

This can be modified as shown below:

```python
# api_definition: apigateway.ApiDefinition


api = apigateway.SpecRestApi(self, "ExampleRestApi",
    api_definition=api_definition,
    endpoint_types=[apigateway.EndpointType.PRIVATE]
)
```

**Note:** For private endpoints you will still need to provide the
[`x-amazon-apigateway-policy`](https://docs.aws.amazon.com/apigateway/latest/developerguide/openapi-extensions-policy.html) and
[`x-amazon-apigateway-endpoint-configuration`](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-endpoint-configuration.html)
in your openApi file.

## Metrics

The API Gateway service sends metrics around the performance of Rest APIs to Amazon CloudWatch.
These metrics can be referred to using the metric APIs available on the `RestApi` construct.
The APIs with the `metric` prefix can be used to get reference to specific metrics for this API. For example,
the method below refers to the client side errors metric for this API.

```python
api = apigateway.RestApi(self, "my-api")
client_error_metric = api.metric_client_error()
```

## APIGateway v2

APIGateway v2 APIs are now moved to its own package named `aws-apigatewayv2`. For backwards compatibility, existing
APIGateway v2 "CFN resources" (such as `CfnApi`) that were previously exported as part of this package, are still
exported from here and have been marked deprecated. However, updates to these CloudFormation resources, such as new
properties and new resource types will not be available.

Move to using `aws-apigatewayv2` to get the latest APIs and updates.

---


This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    AssetHashType as _AssetHashType_49193809,
    BundlingOptions as _BundlingOptions_ab115a99,
    CfnResource as _CfnResource_e0a482dc,
    CfnTag as _CfnTag_95fbdc29,
    Construct as _Construct_e78e779f,
    Duration as _Duration_070aa057,
    IInspectable as _IInspectable_82c04a63,
    IResolvable as _IResolvable_a771d0ef,
    IResource as _IResource_8c1dbbbd,
    IgnoreMode as _IgnoreMode_31d8bf46,
    Resource as _Resource_abff4495,
    ResourceProps as _ResourceProps_9b554c0f,
    SymlinkFollowMode as _SymlinkFollowMode_abf4527a,
    TagManager as _TagManager_0b7ab120,
    TreeInspector as _TreeInspector_1cd1894e,
)
from ..assets import FollowMode as _FollowMode_98b05cc5
from ..aws_certificatemanager import ICertificate as _ICertificate_c7bbdc16
from ..aws_cloudwatch import (
    Metric as _Metric_5b2b8e58,
    MetricOptions as _MetricOptions_1c185ae8,
    Unit as _Unit_113c79f9,
)
from ..aws_cognito import IUserPool as _IUserPool_5e500460
from ..aws_ec2 import IVpcEndpoint as _IVpcEndpoint_8b8ad2f1
from ..aws_elasticloadbalancingv2 import (
    INetworkLoadBalancer as _INetworkLoadBalancer_ead0b7fa
)
from ..aws_iam import (
    Grant as _Grant_bcb5eae7,
    IGrantable as _IGrantable_4c5a91d1,
    IRole as _IRole_59af6f50,
    PolicyDocument as _PolicyDocument_b5de5177,
)
from ..aws_lambda import IFunction as _IFunction_6e14f09e
from ..aws_logs import ILogGroup as _ILogGroup_846e17a0
from ..aws_s3 import IBucket as _IBucket_73486e29
from ..aws_s3_assets import AssetOptions as _AssetOptions_bd2996da
from ..aws_stepfunctions import IStateMachine as _IStateMachine_269a89c4


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.AccessLogDestinationConfig",
    jsii_struct_bases=[],
    name_mapping={"destination_arn": "destinationArn"},
)
class AccessLogDestinationConfig:
    def __init__(self, *, destination_arn: builtins.str) -> None:
        '''(experimental) Options when binding a log destination to a RestApi Stage.

        :param destination_arn: (experimental) The Amazon Resource Name (ARN) of the destination resource.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            access_log_destination_config = apigateway.AccessLogDestinationConfig(
                destination_arn="destinationArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__579228476a84e972e11aa02be0af11ebc258f4f4014dc6d51c5c30aa8eb71764)
            check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "destination_arn": destination_arn,
        }

    @builtins.property
    def destination_arn(self) -> builtins.str:
        '''(experimental) The Amazon Resource Name (ARN) of the destination resource.

        :stability: experimental
        '''
        result = self._values.get("destination_arn")
        assert result is not None, "Required property 'destination_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessLogDestinationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AccessLogField(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.AccessLogField",
):
    '''(experimental) $context variables that can be used to customize access log pattern.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        apigateway.AccessLogFormat.custom(JSON.stringify({
            "request_id": apigateway.AccessLogField.context_request_id(),
            "source_ip": apigateway.AccessLogField.context_identity_source_ip(),
            "method": apigateway.AccessLogField.context_http_method(),
            "user_context": {
                "sub": apigateway.AccessLogField.context_authorizer_claims("sub"),
                "email": apigateway.AccessLogField.context_authorizer_claims("email")
            }
        }))
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="contextAccountId")
    @builtins.classmethod
    def context_account_id(cls) -> builtins.str:
        '''(experimental) The API owner's AWS account ID.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAccountId", []))

    @jsii.member(jsii_name="contextApiId")
    @builtins.classmethod
    def context_api_id(cls) -> builtins.str:
        '''(experimental) The identifier API Gateway assigns to your API.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextApiId", []))

    @jsii.member(jsii_name="contextAuthorizer")
    @builtins.classmethod
    def context_authorizer(cls, property: builtins.str) -> builtins.str:
        '''(experimental) The stringified value of the specified key-value pair of the ``context`` map returned from an API Gateway Lambda authorizer function.

        :param property: key of the context map.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed35bff7c9206d0da2e09757d02edc26ae7f9cd6b848873f20204765fc192992)
            check_type(argname="argument property", value=property, expected_type=type_hints["property"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizer", [property]))

    @jsii.member(jsii_name="contextAuthorizerClaims")
    @builtins.classmethod
    def context_authorizer_claims(cls, property: builtins.str) -> builtins.str:
        '''(experimental) A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.

        :param property: A property key of the claims.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__934bc71788e6a8a4b301295941def62adc8e781b2bd1045e960358df3d6ee4eb)
            check_type(argname="argument property", value=property, expected_type=type_hints["property"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerClaims", [property]))

    @jsii.member(jsii_name="contextAuthorizerIntegrationLatency")
    @builtins.classmethod
    def context_authorizer_integration_latency(cls) -> builtins.str:
        '''(experimental) The authorizer latency in ms.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerIntegrationLatency", []))

    @jsii.member(jsii_name="contextAuthorizerPrincipalId")
    @builtins.classmethod
    def context_authorizer_principal_id(cls) -> builtins.str:
        '''(experimental) The principal user identification associated with the token sent by the client and returned from an API Gateway Lambda authorizer (formerly known as a custom authorizer).

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerPrincipalId", []))

    @jsii.member(jsii_name="contextAwsEndpointRequestId")
    @builtins.classmethod
    def context_aws_endpoint_request_id(cls) -> builtins.str:
        '''(experimental) The AWS endpoint's request ID.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAwsEndpointRequestId", []))

    @jsii.member(jsii_name="contextDomainName")
    @builtins.classmethod
    def context_domain_name(cls) -> builtins.str:
        '''(experimental) The full domain name used to invoke the API.

        This should be the same as the incoming ``Host`` header.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextDomainName", []))

    @jsii.member(jsii_name="contextDomainPrefix")
    @builtins.classmethod
    def context_domain_prefix(cls) -> builtins.str:
        '''(experimental) The first label of the ``$context.domainName``. This is often used as a caller/customer identifier.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextDomainPrefix", []))

    @jsii.member(jsii_name="contextErrorMessage")
    @builtins.classmethod
    def context_error_message(cls) -> builtins.str:
        '''(experimental) A string containing an API Gateway error message.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorMessage", []))

    @jsii.member(jsii_name="contextErrorMessageString")
    @builtins.classmethod
    def context_error_message_string(cls) -> builtins.str:
        '''(experimental) The quoted value of $context.error.message, namely "$context.error.message".

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorMessageString", []))

    @jsii.member(jsii_name="contextErrorResponseType")
    @builtins.classmethod
    def context_error_response_type(cls) -> builtins.str:
        '''(experimental) A type of GatewayResponse.

        This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
        which is not processed by the Velocity Template Language engine, and in access logging.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorResponseType", []))

    @jsii.member(jsii_name="contextErrorValidationErrorString")
    @builtins.classmethod
    def context_error_validation_error_string(cls) -> builtins.str:
        '''(experimental) A string containing a detailed validation error message.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorValidationErrorString", []))

    @jsii.member(jsii_name="contextExtendedRequestId")
    @builtins.classmethod
    def context_extended_request_id(cls) -> builtins.str:
        '''(experimental) The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextExtendedRequestId", []))

    @jsii.member(jsii_name="contextHttpMethod")
    @builtins.classmethod
    def context_http_method(cls) -> builtins.str:
        '''(experimental) The HTTP method used.

        Valid values include: ``DELETE``, ``GET``, ``HEAD``, ``OPTIONS``, ``PATCH``, ``POST``, and ``PUT``.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextHttpMethod", []))

    @jsii.member(jsii_name="contextIdentityAccountId")
    @builtins.classmethod
    def context_identity_account_id(cls) -> builtins.str:
        '''(experimental) The AWS account ID associated with the request.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityAccountId", []))

    @jsii.member(jsii_name="contextIdentityApiKey")
    @builtins.classmethod
    def context_identity_api_key(cls) -> builtins.str:
        '''(experimental) For API methods that require an API key, this variable is the API key associated with the method request.

        For methods that don't require an API key, this variable is

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityApiKey", []))

    @jsii.member(jsii_name="contextIdentityApiKeyId")
    @builtins.classmethod
    def context_identity_api_key_id(cls) -> builtins.str:
        '''(experimental) The API key ID associated with an API request that requires an API key.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityApiKeyId", []))

    @jsii.member(jsii_name="contextIdentityCaller")
    @builtins.classmethod
    def context_identity_caller(cls) -> builtins.str:
        '''(experimental) The principal identifier of the caller making the request.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCaller", []))

    @jsii.member(jsii_name="contextIdentityCognitoAuthenticationProvider")
    @builtins.classmethod
    def context_identity_cognito_authentication_provider(cls) -> builtins.str:
        '''(experimental) The Amazon Cognito authentication provider used by the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoAuthenticationProvider", []))

    @jsii.member(jsii_name="contextIdentityCognitoAuthenticationType")
    @builtins.classmethod
    def context_identity_cognito_authentication_type(cls) -> builtins.str:
        '''(experimental) The Amazon Cognito authentication type of the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoAuthenticationType", []))

    @jsii.member(jsii_name="contextIdentityCognitoIdentityId")
    @builtins.classmethod
    def context_identity_cognito_identity_id(cls) -> builtins.str:
        '''(experimental) The Amazon Cognito identity ID of the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoIdentityId", []))

    @jsii.member(jsii_name="contextIdentityCognitoIdentityPoolId")
    @builtins.classmethod
    def context_identity_cognito_identity_pool_id(cls) -> builtins.str:
        '''(experimental) The Amazon Cognito identity pool ID of the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoIdentityPoolId", []))

    @jsii.member(jsii_name="contextIdentityPrincipalOrgId")
    @builtins.classmethod
    def context_identity_principal_org_id(cls) -> builtins.str:
        '''(experimental) The AWS organization ID.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityPrincipalOrgId", []))

    @jsii.member(jsii_name="contextIdentitySourceIp")
    @builtins.classmethod
    def context_identity_source_ip(cls) -> builtins.str:
        '''(experimental) The source IP address of the TCP connection making the request to API Gateway.

        Warning: You should not trust this value if there is any chance that the ``X-Forwarded-For`` header could be forged.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentitySourceIp", []))

    @jsii.member(jsii_name="contextIdentityUser")
    @builtins.classmethod
    def context_identity_user(cls) -> builtins.str:
        '''(experimental) The principal identifier of the user making the request.

        Used in Lambda authorizers.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityUser", []))

    @jsii.member(jsii_name="contextIdentityUserAgent")
    @builtins.classmethod
    def context_identity_user_agent(cls) -> builtins.str:
        '''(experimental) The User-Agent header of the API caller.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityUserAgent", []))

    @jsii.member(jsii_name="contextIdentityUserArn")
    @builtins.classmethod
    def context_identity_user_arn(cls) -> builtins.str:
        '''(experimental) The Amazon Resource Name (ARN) of the effective user identified after authentication.

        :see: https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityUserArn", []))

    @jsii.member(jsii_name="contextIntegrationLatency")
    @builtins.classmethod
    def context_integration_latency(cls) -> builtins.str:
        '''(experimental) The integration latency in ms.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIntegrationLatency", []))

    @jsii.member(jsii_name="contextIntegrationStatus")
    @builtins.classmethod
    def context_integration_status(cls) -> builtins.str:
        '''(experimental) For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda, not from the backend Lambda function.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIntegrationStatus", []))

    @jsii.member(jsii_name="contextPath")
    @builtins.classmethod
    def context_path(cls) -> builtins.str:
        '''(experimental) The request path.

        For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
        this value is /{stage}/root/child.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextPath", []))

    @jsii.member(jsii_name="contextProtocol")
    @builtins.classmethod
    def context_protocol(cls) -> builtins.str:
        '''(experimental) The request protocol, for example, HTTP/1.1.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextProtocol", []))

    @jsii.member(jsii_name="contextRequestId")
    @builtins.classmethod
    def context_request_id(cls) -> builtins.str:
        '''(experimental) The ID that API Gateway assigns to the API request.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestId", []))

    @jsii.member(jsii_name="contextRequestOverrideHeader")
    @builtins.classmethod
    def context_request_override_header(cls, header_name: builtins.str) -> builtins.str:
        '''(experimental) The request header override.

        If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.

        :param header_name: -

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9d3c37fe17790170114e354b89fb1a6a2439f7db515f0397da62f0ab74361233)
            check_type(argname="argument header_name", value=header_name, expected_type=type_hints["header_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestOverrideHeader", [header_name]))

    @jsii.member(jsii_name="contextRequestOverridePath")
    @builtins.classmethod
    def context_request_override_path(cls, path_name: builtins.str) -> builtins.str:
        '''(experimental) The request path override.

        If this parameter is defined,
        it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.

        :param path_name: -

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7848936c65b170888f53842e7fb8ac53e51626931f68598fae97261d538ba4f5)
            check_type(argname="argument path_name", value=path_name, expected_type=type_hints["path_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestOverridePath", [path_name]))

    @jsii.member(jsii_name="contextRequestOverrideQuerystring")
    @builtins.classmethod
    def context_request_override_querystring(
        cls,
        querystring_name: builtins.str,
    ) -> builtins.str:
        '''(experimental) The request query string override.

        If this parameter is defined, it contains the request query strings to be used instead
        of the URL Query String Parameters that are defined in the Integration Request pane.

        :param querystring_name: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__97379d68bcae716234bbc72cec753c0a4c4da619bdc4b105a51f916c9b4343fd)
            check_type(argname="argument querystring_name", value=querystring_name, expected_type=type_hints["querystring_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestOverrideQuerystring", [querystring_name]))

    @jsii.member(jsii_name="contextRequestTime")
    @builtins.classmethod
    def context_request_time(cls) -> builtins.str:
        '''(experimental) The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestTime", []))

    @jsii.member(jsii_name="contextRequestTimeEpoch")
    @builtins.classmethod
    def context_request_time_epoch(cls) -> builtins.str:
        '''(experimental) The Epoch-formatted request time.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestTimeEpoch", []))

    @jsii.member(jsii_name="contextResourceId")
    @builtins.classmethod
    def context_resource_id(cls) -> builtins.str:
        '''(experimental) The identifier that API Gateway assigns to your resource.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResourceId", []))

    @jsii.member(jsii_name="contextResourcePath")
    @builtins.classmethod
    def context_resource_path(cls) -> builtins.str:
        '''(experimental) The path to your resource.

        For example, for the non-proxy request URI of ``https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child``,
        The $context.resourcePath value is ``/root/child``.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResourcePath", []))

    @jsii.member(jsii_name="contextResponseLatency")
    @builtins.classmethod
    def context_response_latency(cls) -> builtins.str:
        '''(experimental) The response latency in ms.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseLatency", []))

    @jsii.member(jsii_name="contextResponseLength")
    @builtins.classmethod
    def context_response_length(cls) -> builtins.str:
        '''(experimental) The response payload length.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseLength", []))

    @jsii.member(jsii_name="contextResponseOverrideHeader")
    @builtins.classmethod
    def context_response_override_header(
        cls,
        header_name: builtins.str,
    ) -> builtins.str:
        '''(experimental) The response header override.

        If this parameter is defined, it contains the header to be returned instead of the Response header
        that is defined as the Default mapping in the Integration Response pane.

        :param header_name: -

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d55533871ba6cb8bdb2e8cb92a2bed3f35ac6b0a253d4bef1145fc5b21d4514)
            check_type(argname="argument header_name", value=header_name, expected_type=type_hints["header_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseOverrideHeader", [header_name]))

    @jsii.member(jsii_name="contextResponseOverrideStatus")
    @builtins.classmethod
    def context_response_override_status(cls) -> builtins.str:
        '''(experimental) The response status code override.

        If this parameter is defined, it contains the status code to be returned instead of the Method response status
        that is defined as the Default mapping in the Integration Response pane.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseOverrideStatus", []))

    @jsii.member(jsii_name="contextStage")
    @builtins.classmethod
    def context_stage(cls) -> builtins.str:
        '''(experimental) The deployment stage of the API request (for example, ``Beta`` or ``Prod``).

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextStage", []))

    @jsii.member(jsii_name="contextStatus")
    @builtins.classmethod
    def context_status(cls) -> builtins.str:
        '''(experimental) The method response status.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextStatus", []))

    @jsii.member(jsii_name="contextWafResponseCode")
    @builtins.classmethod
    def context_waf_response_code(cls) -> builtins.str:
        '''(experimental) The response received from AWS WAF: ``WAF_ALLOW`` or ``WAF_BLOCK``.

        Will not be set if the stage is not associated with a web ACL.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWafResponseCode", []))

    @jsii.member(jsii_name="contextWebaclArn")
    @builtins.classmethod
    def context_webacl_arn(cls) -> builtins.str:
        '''(experimental) The complete ARN of the web ACL that is used to decide whether to allow or block the request.

        Will not be set if the stage is not associated with a web ACL.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWebaclArn", []))

    @jsii.member(jsii_name="contextXrayTraceId")
    @builtins.classmethod
    def context_xray_trace_id(cls) -> builtins.str:
        '''(experimental) The trace ID for the X-Ray trace.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextXrayTraceId", []))


class AccessLogFormat(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.AccessLogFormat",
):
    '''(experimental) factory methods for access log format.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        log_group = logs.LogGroup(self, "ApiGatewayAccessLogs")
        apigateway.RestApi(self, "books",
            deploy_options=apigateway.aws_apigateway.StageOptions(
                access_log_destination=apigateway.LogGroupLogDestination(log_group),
                access_log_format=apigateway.AccessLogFormat.custom(f"{apigateway.AccessLogField.contextRequestId()} {apigateway.AccessLogField.contextErrorMessage()} {apigateway.AccessLogField.contextErrorMessageString()}")
            )
        )
    '''

    @jsii.member(jsii_name="clf")
    @builtins.classmethod
    def clf(cls) -> "AccessLogFormat":
        '''(experimental) Generate Common Log Format.

        :stability: experimental
        '''
        return typing.cast("AccessLogFormat", jsii.sinvoke(cls, "clf", []))

    @jsii.member(jsii_name="custom")
    @builtins.classmethod
    def custom(cls, format: builtins.str) -> "AccessLogFormat":
        '''(experimental) Custom log format.

        You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.

        :param format: -

        :stability: experimental

        Example::

            apigateway.AccessLogFormat.custom(JSON.stringify({
                "request_id": apigateway.AccessLogField.context_request_id(),
                "source_ip": apigateway.AccessLogField.context_identity_source_ip(),
                "method": apigateway.AccessLogField.context_http_method(),
                "user_context": {
                    "sub": apigateway.AccessLogField.context_authorizer_claims("sub"),
                    "email": apigateway.AccessLogField.context_authorizer_claims("email")
                }
            }))
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51044cb61695fb46ff0341182c24ef641b78d7a6f8a26b4519ac1147610a6b3b)
            check_type(argname="argument format", value=format, expected_type=type_hints["format"])
        return typing.cast("AccessLogFormat", jsii.sinvoke(cls, "custom", [format]))

    @jsii.member(jsii_name="jsonWithStandardFields")
    @builtins.classmethod
    def json_with_standard_fields(
        cls,
        *,
        caller: builtins.bool,
        http_method: builtins.bool,
        ip: builtins.bool,
        protocol: builtins.bool,
        request_time: builtins.bool,
        resource_path: builtins.bool,
        response_length: builtins.bool,
        status: builtins.bool,
        user: builtins.bool,
    ) -> "AccessLogFormat":
        '''(experimental) Access log will be produced in the JSON format with a set of fields most useful in the access log.

        All fields are turned on by default with the
        option to turn off specific fields.

        :param caller: (experimental) If this flag is enabled, the principal identifier of the caller will be output to the log.
        :param http_method: (experimental) If this flag is enabled, the http method will be output to the log.
        :param ip: (experimental) If this flag is enabled, the source IP of request will be output to the log.
        :param protocol: (experimental) If this flag is enabled, the request protocol will be output to the log.
        :param request_time: (experimental) If this flag is enabled, the CLF-formatted request time((dd/MMM/yyyy:HH:mm:ss +-hhmm) will be output to the log.
        :param resource_path: (experimental) If this flag is enabled, the path to your resource will be output to the log.
        :param response_length: (experimental) If this flag is enabled, the response payload length will be output to the log.
        :param status: (experimental) If this flag is enabled, the method response status will be output to the log.
        :param user: (experimental) If this flag is enabled, the principal identifier of the user will be output to the log.

        :stability: experimental
        '''
        fields = JsonWithStandardFieldProps(
            caller=caller,
            http_method=http_method,
            ip=ip,
            protocol=protocol,
            request_time=request_time,
            resource_path=resource_path,
            response_length=response_length,
            status=status,
            user=user,
        )

        return typing.cast("AccessLogFormat", jsii.sinvoke(cls, "jsonWithStandardFields", [fields]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''(experimental) Output a format string to be used with CloudFormation.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.AddApiKeyOptions",
    jsii_struct_bases=[],
    name_mapping={"override_logical_id": "overrideLogicalId"},
)
class AddApiKeyOptions:
    def __init__(
        self,
        *,
        override_logical_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Options to the UsagePlan.addApiKey() method.

        :param override_logical_id: (experimental) Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource. Default: - autogenerated by the CDK

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # usageplan: apigateway.UsagePlan
            # api_key: apigateway.ApiKey
            
            
            usageplan.add_api_key(api_key,
                override_logical_id="..."
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__39b39963ed4290bba89d8f3c44fb3766ae49a911e5bcc88cc244373f2ae1e85f)
            check_type(argname="argument override_logical_id", value=override_logical_id, expected_type=type_hints["override_logical_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if override_logical_id is not None:
            self._values["override_logical_id"] = override_logical_id

    @builtins.property
    def override_logical_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource.

        :default: - autogenerated by the CDK

        :stability: experimental
        '''
        result = self._values.get("override_logical_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddApiKeyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ApiDefinition(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_apigateway.ApiDefinition",
):
    '''(experimental) Represents an OpenAPI definition asset.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # integration: apigateway.Integration
        
        
        api = apigateway.SpecRestApi(self, "books-api",
            api_definition=apigateway.ApiDefinition.from_asset("path-to-file.json")
        )
        
        books_resource = api.root.add_resource("books")
        books_resource.add_method("GET", integration)
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromAsset")
    @builtins.classmethod
    def from_asset(
        cls,
        file: builtins.str,
        *,
        readers: typing.Optional[typing.Sequence[_IGrantable_4c5a91d1]] = None,
        source_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_49193809] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_ab115a99, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "AssetApiDefinition":
        '''(experimental) Loads the API specification from a local disk asset.

        :param file: -
        :param readers: (experimental) A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_hash: (deprecated) Custom hash to use when identifying the specific version of the asset. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the source hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the source hash, you will need to make sure it is updated every time the source changes, or otherwise it is possible that some deployments will not be invalidated. Default: - automatically calculate source hash based on the contents of the source file or directory.
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param asset_hash: (experimental) Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: (experimental) Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: (experimental) Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3a6718016bf1138f14971183db4c789621a36dbab1cc1e6411bed24308be896)
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
        options = _AssetOptions_bd2996da(
            readers=readers,
            source_hash=source_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
        )

        return typing.cast("AssetApiDefinition", jsii.sinvoke(cls, "fromAsset", [file, options]))

    @jsii.member(jsii_name="fromBucket")
    @builtins.classmethod
    def from_bucket(
        cls,
        bucket: _IBucket_73486e29,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> "S3ApiDefinition":
        '''(experimental) Creates an API definition from a specification file in an S3 bucket.

        :param bucket: -
        :param key: -
        :param object_version: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__88c95b2650449b625fab7207386216a017433b5475b53f957adebc6f31305037)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
        return typing.cast("S3ApiDefinition", jsii.sinvoke(cls, "fromBucket", [bucket, key, object_version]))

    @jsii.member(jsii_name="fromInline")
    @builtins.classmethod
    def from_inline(cls, definition: typing.Any) -> "InlineApiDefinition":
        '''(experimental) Create an API definition from an inline object.

        The inline object must follow the
        schema of OpenAPI 2.0 or OpenAPI 3.0

        :param definition: -

        :stability: experimental

        Example::

            apigateway.ApiDefinition.from_inline({
                "openapi": "3.0.2",
                "paths": {
                    "/pets": {
                        "get": {
                            "responses": {
                                "200": {
                                    "content": {
                                        "application/json": {
                                            "schema": {
                                                "$ref": "#/components/schemas/Empty"
                                            }
                                        }
                                    }
                                }
                            },
                            "x-amazon-apigateway-integration": {
                                "responses": {
                                    "default": {
                                        "status_code": "200"
                                    }
                                },
                                "request_templates": {
                                    "application/json": "{\"statusCode\": 200}"
                                },
                                "passthrough_behavior": "when_no_match",
                                "type": "mock"
                            }
                        }
                    }
                },
                "components": {
                    "schemas": {
                        "Empty": {
                            "title": "Empty Schema",
                            "type": "object"
                        }
                    }
                }
            })
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76d1437d52860cce1914c0830c99f1d985e398d5b9e913b6aa6d14b68d48dd4d)
            check_type(argname="argument definition", value=definition, expected_type=type_hints["definition"])
        return typing.cast("InlineApiDefinition", jsii.sinvoke(cls, "fromInline", [definition]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "ApiDefinitionConfig":
        '''(experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="bindAfterCreate")
    def bind_after_create(
        self,
        _scope: _Construct_e78e779f,
        _rest_api: "IRestApi",
    ) -> None:
        '''(experimental) Called after the CFN RestApi resource has been created to allow the Api Definition to bind to it.

        Specifically it's required to allow assets to add
        metadata for tooling like SAM CLI to be able to find their origins.

        :param _scope: -
        :param _rest_api: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__12514b74567cd303fd4d179745a2ce5e43a6adb04b14b867ba7f77a1ab9bc76d)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
            check_type(argname="argument _rest_api", value=_rest_api, expected_type=type_hints["_rest_api"])
        return typing.cast(None, jsii.invoke(self, "bindAfterCreate", [_scope, _rest_api]))


class _ApiDefinitionProxy(ApiDefinition):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "ApiDefinitionConfig":
        '''(experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0eb289a2ea0947d5ddf097a20d78850c9ea62addcac2a63184fd3f82f332556e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("ApiDefinitionConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, ApiDefinition).__jsii_proxy_class__ = lambda : _ApiDefinitionProxy


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.ApiDefinitionConfig",
    jsii_struct_bases=[],
    name_mapping={
        "inline_definition": "inlineDefinition",
        "s3_location": "s3Location",
    },
)
class ApiDefinitionConfig:
    def __init__(
        self,
        *,
        inline_definition: typing.Any = None,
        s3_location: typing.Optional[typing.Union["ApiDefinitionS3Location", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) Post-Binding Configuration for a CDK construct.

        :param inline_definition: (experimental) Inline specification (mutually exclusive with ``s3Location``). Default: - API definition is not defined inline
        :param s3_location: (experimental) The location of the specification in S3 (mutually exclusive with ``inlineDefinition``). Default: - API definition is not an S3 location

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # inline_definition: Any
            
            api_definition_config = apigateway.ApiDefinitionConfig(
                inline_definition=inline_definition,
                s3_location=apigateway.ApiDefinitionS3Location(
                    bucket="bucket",
                    key="key",
            
                    # the properties below are optional
                    version="version"
                )
            )
        '''
        if isinstance(s3_location, dict):
            s3_location = ApiDefinitionS3Location(**s3_location)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__15551a74e6e44c9b574acd9236c91b5170f659f6d852768464095111f3839396)
            check_type(argname="argument inline_definition", value=inline_definition, expected_type=type_hints["inline_definition"])
            check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if inline_definition is not None:
            self._values["inline_definition"] = inline_definition
        if s3_location is not None:
            self._values["s3_location"] = s3_location

    @builtins.property
    def inline_definition(self) -> typing.Any:
        '''(experimental) Inline specification (mutually exclusive with ``s3Location``).

        :default: - API definition is not defined inline

        :stability: experimental
        '''
        result = self._values.get("inline_definition")
        return typing.cast(typing.Any, result)

    @builtins.property
    def s3_location(self) -> typing.Optional["ApiDefinitionS3Location"]:
        '''(experimental) The location of the specification in S3 (mutually exclusive with ``inlineDefinition``).

        :default: - API definition is not an S3 location

        :stability: experimental
        '''
        result = self._values.get("s3_location")
        return typing.cast(typing.Optional["ApiDefinitionS3Location"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiDefinitionConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.ApiDefinitionS3Location",
    jsii_struct_bases=[],
    name_mapping={"bucket": "bucket", "key": "key", "version": "version"},
)
class ApiDefinitionS3Location:
    def __init__(
        self,
        *,
        bucket: builtins.str,
        key: builtins.str,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) S3 location of the API definition file.

        :param bucket: (experimental) The S3 bucket.
        :param key: (experimental) The S3 key.
        :param version: (experimental) An optional version. Default: - latest version

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            api_definition_s3_location = apigateway.ApiDefinitionS3Location(
                bucket="bucket",
                key="key",
            
                # the properties below are optional
                version="version"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3777ba7e54eb3d51f5df6f49920b008012ac53c05a98586e337c7c70281739e0)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "bucket": bucket,
            "key": key,
        }
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def bucket(self) -> builtins.str:
        '''(experimental) The S3 bucket.

        :stability: experimental
        '''
        result = self._values.get("bucket")
        assert result is not None, "Required property 'bucket' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def key(self) -> builtins.str:
        '''(experimental) The S3 key.

        :stability: experimental
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional version.

        :default: - latest version

        :stability: experimental
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiDefinitionS3Location(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_apigateway.ApiKeySourceType")
class ApiKeySourceType(enum.Enum):
    '''
    :stability: experimental
    '''

    HEADER = "HEADER"
    '''(experimental) To read the API key from the ``X-API-Key`` header of a request.

    :stability: experimental
    '''
    AUTHORIZER = "AUTHORIZER"
    '''(experimental) To read the API key from the ``UsageIdentifierKey`` from a custom authorizer.

    :stability: experimental
    '''


class AssetApiDefinition(
    ApiDefinition,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.AssetApiDefinition",
):
    '''(experimental) OpenAPI specification from a local file.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import monocdk as monocdk
        from monocdk import assets
        from monocdk import aws_apigateway as apigateway
        from monocdk import aws_iam as iam
        
        # docker_image: monocdk.DockerImage
        # grantable: iam.IGrantable
        # local_bundling: monocdk.ILocalBundling
        
        asset_api_definition = apigateway.AssetApiDefinition("path",
            asset_hash="assetHash",
            asset_hash_type=monocdk.AssetHashType.SOURCE,
            bundling=monocdk.BundlingOptions(
                image=docker_image,
        
                # the properties below are optional
                command=["command"],
                entrypoint=["entrypoint"],
                environment={
                    "environment_key": "environment"
                },
                local=local_bundling,
                output_type=monocdk.BundlingOutput.ARCHIVED,
                security_opt="securityOpt",
                user="user",
                volumes=[monocdk.DockerVolume(
                    container_path="containerPath",
                    host_path="hostPath",
        
                    # the properties below are optional
                    consistency=monocdk.DockerVolumeConsistency.CONSISTENT
                )],
                working_directory="workingDirectory"
            ),
            exclude=["exclude"],
            follow=assets.FollowMode.NEVER,
            follow_symlinks=monocdk.SymlinkFollowMode.NEVER,
            ignore_mode=monocdk.IgnoreMode.GLOB,
            readers=[grantable],
            source_hash="sourceHash"
        )
    '''

    def __init__(
        self,
        path: builtins.str,
        *,
        readers: typing.Optional[typing.Sequence[_IGrantable_4c5a91d1]] = None,
        source_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_49193809] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_ab115a99, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param path: -
        :param readers: (experimental) A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_hash: (deprecated) Custom hash to use when identifying the specific version of the asset. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the source hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the source hash, you will need to make sure it is updated every time the source changes, or otherwise it is possible that some deployments will not be invalidated. Default: - automatically calculate source hash based on the contents of the source file or directory.
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param asset_hash: (experimental) Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: (experimental) Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: (experimental) Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30af661538662fe3644c46c410c71713023096631edefda7a0a9b1f60da5f434)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = _AssetOptions_bd2996da(
            readers=readers,
            source_hash=source_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
        )

        jsii.create(self.__class__, self, [path, options])

    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> ApiDefinitionConfig:
        '''(experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3d5200adaa1f6929da6026c14d603f27ce539bebe8fe927f1e4a2a80bdc4283)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(ApiDefinitionConfig, jsii.invoke(self, "bind", [scope]))

    @jsii.member(jsii_name="bindAfterCreate")
    def bind_after_create(
        self,
        scope: _Construct_e78e779f,
        rest_api: "IRestApi",
    ) -> None:
        '''(experimental) Called after the CFN RestApi resource has been created to allow the Api Definition to bind to it.

        Specifically it's required to allow assets to add
        metadata for tooling like SAM CLI to be able to find their origins.

        :param scope: -
        :param rest_api: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f89f43d598a33f3fcab4a475e270afa9623863250150c8581e69c15932ec3a1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument rest_api", value=rest_api, expected_type=type_hints["rest_api"])
        return typing.cast(None, jsii.invoke(self, "bindAfterCreate", [scope, rest_api]))


@jsii.enum(jsii_type="monocdk.aws_apigateway.AuthorizationType")
class AuthorizationType(enum.Enum):
    '''
    :stability: experimental
    :exampleMetadata: infused

    Example::

        # books: apigateway.Resource
        user_pool = cognito.UserPool(self, "UserPool")
        
        auth = apigateway.CognitoUserPoolsAuthorizer(self, "booksAuthorizer",
            cognito_user_pools=[user_pool]
        )
        books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
            authorizer=auth,
            authorization_type=apigateway.AuthorizationType.COGNITO
        )
    '''

    NONE = "NONE"
    '''(experimental) Open access.

    :stability: experimental
    '''
    IAM = "IAM"
    '''(experimental) Use AWS IAM permissions.

    :stability: experimental
    '''
    CUSTOM = "CUSTOM"
    '''(experimental) Use a custom authorizer.

    :stability: experimental
    '''
    COGNITO = "COGNITO"
    '''(experimental) Use an AWS Cognito user pool.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.AwsIntegrationProps",
    jsii_struct_bases=[],
    name_mapping={
        "service": "service",
        "action": "action",
        "action_parameters": "actionParameters",
        "integration_http_method": "integrationHttpMethod",
        "options": "options",
        "path": "path",
        "proxy": "proxy",
        "region": "region",
        "subdomain": "subdomain",
    },
)
class AwsIntegrationProps:
    def __init__(
        self,
        *,
        service: builtins.str,
        action: typing.Optional[builtins.str] = None,
        action_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        integration_http_method: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union["IntegrationOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        path: typing.Optional[builtins.str] = None,
        proxy: typing.Optional[builtins.bool] = None,
        region: typing.Optional[builtins.str] = None,
        subdomain: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param service: (experimental) The name of the integrated AWS service (e.g. ``s3``).
        :param action: (experimental) The AWS action to perform in the integration. Use ``actionParams`` to specify key-value params for the action. Mutually exclusive with ``path``.
        :param action_parameters: (experimental) Parameters for the action. ``action`` must be set, and ``path`` must be undefined. The action params will be URL encoded.
        :param integration_http_method: (experimental) The integration's HTTP method type. Default: POST
        :param options: (experimental) Integration options, such as content handling, request/response mapping, etc.
        :param path: (experimental) The path to use for path-base APIs. For example, for S3 GET, you can set path to ``bucket/key``. For lambda, you can set path to ``2015-03-31/functions/${function-arn}/invocations`` Mutually exclusive with the ``action`` options.
        :param proxy: (experimental) Use AWS_PROXY integration. Default: false
        :param region: (experimental) The region of the integrated AWS service. Default: - same region as the stack
        :param subdomain: (experimental) A designated subdomain supported by certain AWS service for fast host-name lookup.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            get_message_integration = apigateway.AwsIntegration(
                service="sqs",
                path="queueName",
                region="eu-west-1"
            )
        '''
        if isinstance(options, dict):
            options = IntegrationOptions(**options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8476a055b9a2080237b76846f7f2123936c7deaa835314f517161cdde8987a93)
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument action_parameters", value=action_parameters, expected_type=type_hints["action_parameters"])
            check_type(argname="argument integration_http_method", value=integration_http_method, expected_type=type_hints["integration_http_method"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
            check_type(argname="argument proxy", value=proxy, expected_type=type_hints["proxy"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
            check_type(argname="argument subdomain", value=subdomain, expected_type=type_hints["subdomain"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "service": service,
        }
        if action is not None:
            self._values["action"] = action
        if action_parameters is not None:
            self._values["action_parameters"] = action_parameters
        if integration_http_method is not None:
            self._values["integration_http_method"] = integration_http_method
        if options is not None:
            self._values["options"] = options
        if path is not None:
            self._values["path"] = path
        if proxy is not None:
            self._values["proxy"] = proxy
        if region is not None:
            self._values["region"] = region
        if subdomain is not None:
            self._values["subdomain"] = subdomain

    @builtins.property
    def service(self) -> builtins.str:
        '''(experimental) The name of the integrated AWS service (e.g. ``s3``).

        :stability: experimental
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def action(self) -> typing.Optional[builtins.str]:
        '''(experimental) The AWS action to perform in the integration.

        Use ``actionParams`` to specify key-value params for the action.

        Mutually exclusive with ``path``.

        :stability: experimental
        '''
        result = self._values.get("action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def action_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) Parameters for the action.

        ``action`` must be set, and ``path`` must be undefined.
        The action params will be URL encoded.

        :stability: experimental
        '''
        result = self._values.get("action_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def integration_http_method(self) -> typing.Optional[builtins.str]:
        '''(experimental) The integration's HTTP method type.

        :default: POST

        :stability: experimental
        '''
        result = self._values.get("integration_http_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(self) -> typing.Optional["IntegrationOptions"]:
        '''(experimental) Integration options, such as content handling, request/response mapping, etc.

        :stability: experimental
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional["IntegrationOptions"], result)

    @builtins.property
    def path(self) -> typing.Optional[builtins.str]:
        '''(experimental) The path to use for path-base APIs.

        For example, for S3 GET, you can set path to ``bucket/key``.
        For lambda, you can set path to ``2015-03-31/functions/${function-arn}/invocations``

        Mutually exclusive with the ``action`` options.

        :stability: experimental
        '''
        result = self._values.get("path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def proxy(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Use AWS_PROXY integration.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("proxy")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''(experimental) The region of the integrated AWS service.

        :default: - same region as the stack

        :stability: experimental
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def subdomain(self) -> typing.Optional[builtins.str]:
        '''(experimental) A designated subdomain supported by certain AWS service for fast host-name lookup.

        :stability: experimental
        '''
        result = self._values.get("subdomain")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsIntegrationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BasePathMapping(
    _Resource_abff4495,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.BasePathMapping",
):
    '''(experimental) This resource creates a base path that clients who call your API must use in the invocation URL.

    Unless you're importing a domain with ``DomainName.fromDomainNameAttributes()``,
    you can use ``DomainName.addBasePathMapping()`` to define mappings.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # domain_name: apigateway.DomainName
        # rest_api: apigateway.RestApi
        # stage: apigateway.Stage
        
        base_path_mapping = apigateway.BasePathMapping(self, "MyBasePathMapping",
            domain_name=domain_name,
            rest_api=rest_api,
        
            # the properties below are optional
            base_path="basePath",
            stage=stage
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        domain_name: "IDomainName",
        rest_api: "IRestApi",
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional["Stage"] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param domain_name: (experimental) The DomainName to associate with this base path mapping.
        :param rest_api: (experimental) The RestApi resource to target.
        :param base_path: (experimental) The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``). If this is undefined, no additional mappings will be allowed on this domain name.
        :param stage: (experimental) The Deployment stage of API [disable-awslint:ref-via-interface]. Default: - map to deploymentStage of restApi otherwise stage needs to pass in URL

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ab54be9e79afe3074ce222e2cfac7fbbd1420bd5200b431945e81c21832b8bc)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BasePathMappingProps(
            domain_name=domain_name,
            rest_api=rest_api,
            base_path=base_path,
            stage=stage,
        )

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.BasePathMappingOptions",
    jsii_struct_bases=[],
    name_mapping={"base_path": "basePath", "stage": "stage"},
)
class BasePathMappingOptions:
    def __init__(
        self,
        *,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional["Stage"] = None,
    ) -> None:
        '''
        :param base_path: (experimental) The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``). If this is undefined, no additional mappings will be allowed on this domain name.
        :param stage: (experimental) The Deployment stage of API [disable-awslint:ref-via-interface]. Default: - map to deploymentStage of restApi otherwise stage needs to pass in URL

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # domain: apigateway.DomainName
            # api1: apigateway.RestApi
            # api2: apigateway.RestApi
            
            
            domain.add_base_path_mapping(api1, base_path="go-to-api1")
            domain.add_base_path_mapping(api2, base_path="boom")
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63bb0024fb0c14be50e7e81b368609c3df74066983dbc82d000c1f9f3043d0ed)
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if base_path is not None:
            self._values["base_path"] = base_path
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        :default:

        - map requests from the domain root (e.g. ``example.com``). If this
        is undefined, no additional mappings will be allowed on this domain name.

        :stability: experimental
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional["Stage"]:
        '''(experimental) The Deployment stage of API [disable-awslint:ref-via-interface].

        :default: - map to deploymentStage of restApi otherwise stage needs to pass in URL

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional["Stage"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasePathMappingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.BasePathMappingProps",
    jsii_struct_bases=[BasePathMappingOptions],
    name_mapping={
        "base_path": "basePath",
        "stage": "stage",
        "domain_name": "domainName",
        "rest_api": "restApi",
    },
)
class BasePathMappingProps(BasePathMappingOptions):
    def __init__(
        self,
        *,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional["Stage"] = None,
        domain_name: "IDomainName",
        rest_api: "IRestApi",
    ) -> None:
        '''
        :param base_path: (experimental) The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``). If this is undefined, no additional mappings will be allowed on this domain name.
        :param stage: (experimental) The Deployment stage of API [disable-awslint:ref-via-interface]. Default: - map to deploymentStage of restApi otherwise stage needs to pass in URL
        :param domain_name: (experimental) The DomainName to associate with this base path mapping.
        :param rest_api: (experimental) The RestApi resource to target.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # domain_name: apigateway.DomainName
            # rest_api: apigateway.RestApi
            # stage: apigateway.Stage
            
            base_path_mapping_props = apigateway.BasePathMappingProps(
                domain_name=domain_name,
                rest_api=rest_api,
            
                # the properties below are optional
                base_path="basePath",
                stage=stage
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__839dc27a22f649b144a708491d8fea767d33ce2456719bdb658855fe3acaccf6)
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument rest_api", value=rest_api, expected_type=type_hints["rest_api"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
            "rest_api": rest_api,
        }
        if base_path is not None:
            self._values["base_path"] = base_path
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        :default:

        - map requests from the domain root (e.g. ``example.com``). If this
        is undefined, no additional mappings will be allowed on this domain name.

        :stability: experimental
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional["Stage"]:
        '''(experimental) The Deployment stage of API [disable-awslint:ref-via-interface].

        :default: - map to deploymentStage of restApi otherwise stage needs to pass in URL

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional["Stage"], result)

    @builtins.property
    def domain_name(self) -> "IDomainName":
        '''(experimental) The DomainName to associate with this base path mapping.

        :stability: experimental
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast("IDomainName", result)

    @builtins.property
    def rest_api(self) -> "IRestApi":
        '''(experimental) The RestApi resource to target.

        :stability: experimental
        '''
        result = self._values.get("rest_api")
        assert result is not None, "Required property 'rest_api' is missing"
        return typing.cast("IRestApi", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasePathMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnAccount(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnAccount",
):
    '''A CloudFormation ``AWS::ApiGateway::Account``.

    The ``AWS::ApiGateway::Account`` resource specifies the IAM role that Amazon API Gateway uses to write API logs to Amazon CloudWatch Logs.
    .. epigraph::

       If an API Gateway resource has never been created in your AWS account , you must add a dependency on another API Gateway resource, such as an `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ or `AWS::ApiGateway::ApiKey <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html>`_ resource.

       If an API Gateway resource has been created in your AWS account , no dependency is required (even if the resource was deleted).

    :cloudformationResource: AWS::ApiGateway::Account
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_account = apigateway.CfnAccount(self, "MyCfnAccount",
            cloud_watch_role_arn="cloudWatchRoleArn"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        cloud_watch_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Account``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cloud_watch_role_arn: The Amazon Resource Name (ARN) of an IAM role that has write access to CloudWatch Logs in your account.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f58ff43f4af3fa951727d48db7b195b5628fb6e3acdd9562e9eec761a27534f0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAccountProps(cloud_watch_role_arn=cloud_watch_role_arn)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c31d8b591c6ec3aada04ae69f09a5a9738015b570407417827f2ae65e804686)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e66434bcda750f2164c706b4804a63f2a5a50db9edd72c12cd82f7c366165da0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the account.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="cloudWatchRoleArn")
    def cloud_watch_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of an IAM role that has write access to CloudWatch Logs in your account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html#cfn-apigateway-account-cloudwatchrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cloudWatchRoleArn"))

    @cloud_watch_role_arn.setter
    def cloud_watch_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67fcafd56de4c40cff0f6464ebab94d58da3187926b7e74be8c84a0e7171b57b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cloudWatchRoleArn", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnAccountProps",
    jsii_struct_bases=[],
    name_mapping={"cloud_watch_role_arn": "cloudWatchRoleArn"},
)
class CfnAccountProps:
    def __init__(
        self,
        *,
        cloud_watch_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnAccount``.

        :param cloud_watch_role_arn: The Amazon Resource Name (ARN) of an IAM role that has write access to CloudWatch Logs in your account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_account_props = apigateway.CfnAccountProps(
                cloud_watch_role_arn="cloudWatchRoleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50b91449188716c0847cc3038ef58b363d986b6f743f39c5949484f111db8daa)
            check_type(argname="argument cloud_watch_role_arn", value=cloud_watch_role_arn, expected_type=type_hints["cloud_watch_role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cloud_watch_role_arn is not None:
            self._values["cloud_watch_role_arn"] = cloud_watch_role_arn

    @builtins.property
    def cloud_watch_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of an IAM role that has write access to CloudWatch Logs in your account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html#cfn-apigateway-account-cloudwatchrolearn
        '''
        result = self._values.get("cloud_watch_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAccountProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnApiKey(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnApiKey",
):
    '''A CloudFormation ``AWS::ApiGateway::ApiKey``.

    The ``AWS::ApiGateway::ApiKey`` resource creates a unique key that you can distribute to clients who are executing API Gateway ``Method`` resources that require an API key. To specify which API key clients must use, map the API key with the ``RestApi`` and ``Stage`` resources that include the methods that require a key.

    :cloudformationResource: AWS::ApiGateway::ApiKey
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_api_key = apigateway.CfnApiKey(self, "MyCfnApiKey",
            customer_id="customerId",
            description="description",
            enabled=False,
            generate_distinct_id=False,
            name="name",
            stage_keys=[apigateway.CfnApiKey.StageKeyProperty(
                rest_api_id="restApiId",
                stage_name="stageName"
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            value="value"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        customer_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        generate_distinct_id: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        name: typing.Optional[builtins.str] = None,
        stage_keys: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnApiKey.StageKeyProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        value: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::ApiKey``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param customer_id: An AWS Marketplace customer identifier to use when integrating with the AWS SaaS Marketplace.
        :param description: A description of the purpose of the API key.
        :param enabled: Indicates whether the API key can be used by clients.
        :param generate_distinct_id: Specifies whether the key identifier is distinct from the created API key value. This parameter is deprecated and should not be used.
        :param name: A name for the API key. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param stage_keys: A list of stages to associate with this API key.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the API key.
        :param value: The value of the API key. Must be at least 20 characters long.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9c0cd60bcbd6b3e2e070a6535ebd1fc6e4bde0a2065fcf1ce76aaa0fc9a37af)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnApiKeyProps(
            customer_id=customer_id,
            description=description,
            enabled=enabled,
            generate_distinct_id=generate_distinct_id,
            name=name,
            stage_keys=stage_keys,
            tags=tags,
            value=value,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c0fdc77f58578fe625b35cb80b2392631ed8009b6ad02ce4d7d912b901018e67)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f20ea624d4a824da61efd586f5fdecdb7fa64c4c062715075720d2bda3ed6aa6)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrApiKeyId")
    def attr_api_key_id(self) -> builtins.str:
        '''The ID for the API key.

        For example: ``abc123`` .

        :cloudformationAttribute: APIKeyId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrApiKeyId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="customerId")
    def customer_id(self) -> typing.Optional[builtins.str]:
        '''An AWS Marketplace customer identifier to use when integrating with the AWS SaaS Marketplace.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-customerid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "customerId"))

    @customer_id.setter
    def customer_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f292b94d1592f835ee99b7f96dae4070282a130d2e25351fa29eee428a4a9a03)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customerId", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the purpose of the API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9cf09ca1f59f8a75e4ac0db3fc511a9ff85b7bc288898cc02fa219f9dbe17e7a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether the API key can be used by clients.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-enabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "enabled"))

    @enabled.setter
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe22673f48e12a940e37be9f31db355a592d27cb6d6381c0f9453e5a5aa3fbbe)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabled", value)

    @builtins.property
    @jsii.member(jsii_name="generateDistinctId")
    def generate_distinct_id(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether the key identifier is distinct from the created API key value.

        This parameter is deprecated and should not be used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-generatedistinctid
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "generateDistinctId"))

    @generate_distinct_id.setter
    def generate_distinct_id(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4203e0d503166b417d47401a1942207d65d043ee016c592d61d5dcdbfd5512ee)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "generateDistinctId", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the API key.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .
        .. epigraph::

           If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd37c2fc3ba25e3e9e55da5e4e319463f31a5f0d05513368c5aafb695b92107f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="stageKeys")
    def stage_keys(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnApiKey.StageKeyProperty", _IResolvable_a771d0ef]]]]:
        '''A list of stages to associate with this API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-stagekeys
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnApiKey.StageKeyProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "stageKeys"))

    @stage_keys.setter
    def stage_keys(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnApiKey.StageKeyProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d8deccf0110e9c138901c09978fa93b85a6338b2f97ddf0482ffcee902bb53a0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageKeys", value)

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> typing.Optional[builtins.str]:
        '''The value of the API key.

        Must be at least 20 characters long.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-value
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "value"))

    @value.setter
    def value(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c48132c5891f41dbba4857c0e3728742fb75fd94aa0273adf34bb2cc225d81ea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "value", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnApiKey.StageKeyProperty",
        jsii_struct_bases=[],
        name_mapping={"rest_api_id": "restApiId", "stage_name": "stageName"},
    )
    class StageKeyProperty:
        def __init__(
            self,
            *,
            rest_api_id: typing.Optional[builtins.str] = None,
            stage_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``StageKey`` is a property of the `AWS::ApiGateway::ApiKey <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html>`_ resource that specifies the stage to associate with the API key. This association allows only clients with the key to make requests to methods in that stage.

            :param rest_api_id: The ID of a ``RestApi`` resource that includes the stage with which you want to associate the API key.
            :param stage_name: The name of the stage with which to associate the API key. The stage must be included in the ``RestApi`` resource that you specified in the ``RestApiId`` property.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-apikey-stagekey.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                stage_key_property = apigateway.CfnApiKey.StageKeyProperty(
                    rest_api_id="restApiId",
                    stage_name="stageName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d8a61aa33f0542c6892a4ae2d13a231fab0ffbd5b26849ace7eddd5f3541e1b6)
                check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
                check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if rest_api_id is not None:
                self._values["rest_api_id"] = rest_api_id
            if stage_name is not None:
                self._values["stage_name"] = stage_name

        @builtins.property
        def rest_api_id(self) -> typing.Optional[builtins.str]:
            '''The ID of a ``RestApi`` resource that includes the stage with which you want to associate the API key.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-apikey-stagekey.html#cfn-apigateway-apikey-stagekey-restapiid
            '''
            result = self._values.get("rest_api_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def stage_name(self) -> typing.Optional[builtins.str]:
            '''The name of the stage with which to associate the API key.

            The stage must be included in the ``RestApi`` resource that you specified in the ``RestApiId`` property.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-apikey-stagekey.html#cfn-apigateway-apikey-stagekey-stagename
            '''
            result = self._values.get("stage_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StageKeyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnApiKeyProps",
    jsii_struct_bases=[],
    name_mapping={
        "customer_id": "customerId",
        "description": "description",
        "enabled": "enabled",
        "generate_distinct_id": "generateDistinctId",
        "name": "name",
        "stage_keys": "stageKeys",
        "tags": "tags",
        "value": "value",
    },
)
class CfnApiKeyProps:
    def __init__(
        self,
        *,
        customer_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        generate_distinct_id: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        name: typing.Optional[builtins.str] = None,
        stage_keys: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union[CfnApiKey.StageKeyProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        value: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnApiKey``.

        :param customer_id: An AWS Marketplace customer identifier to use when integrating with the AWS SaaS Marketplace.
        :param description: A description of the purpose of the API key.
        :param enabled: Indicates whether the API key can be used by clients.
        :param generate_distinct_id: Specifies whether the key identifier is distinct from the created API key value. This parameter is deprecated and should not be used.
        :param name: A name for the API key. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param stage_keys: A list of stages to associate with this API key.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the API key.
        :param value: The value of the API key. Must be at least 20 characters long.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_api_key_props = apigateway.CfnApiKeyProps(
                customer_id="customerId",
                description="description",
                enabled=False,
                generate_distinct_id=False,
                name="name",
                stage_keys=[apigateway.CfnApiKey.StageKeyProperty(
                    rest_api_id="restApiId",
                    stage_name="stageName"
                )],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                value="value"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a67571db14897d27dbd1d800bd1fafa84aabf2b5cc150382b1f22388788a5512)
            check_type(argname="argument customer_id", value=customer_id, expected_type=type_hints["customer_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument generate_distinct_id", value=generate_distinct_id, expected_type=type_hints["generate_distinct_id"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument stage_keys", value=stage_keys, expected_type=type_hints["stage_keys"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if customer_id is not None:
            self._values["customer_id"] = customer_id
        if description is not None:
            self._values["description"] = description
        if enabled is not None:
            self._values["enabled"] = enabled
        if generate_distinct_id is not None:
            self._values["generate_distinct_id"] = generate_distinct_id
        if name is not None:
            self._values["name"] = name
        if stage_keys is not None:
            self._values["stage_keys"] = stage_keys
        if tags is not None:
            self._values["tags"] = tags
        if value is not None:
            self._values["value"] = value

    @builtins.property
    def customer_id(self) -> typing.Optional[builtins.str]:
        '''An AWS Marketplace customer identifier to use when integrating with the AWS SaaS Marketplace.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-customerid
        '''
        result = self._values.get("customer_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the purpose of the API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether the API key can be used by clients.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-enabled
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def generate_distinct_id(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether the key identifier is distinct from the created API key value.

        This parameter is deprecated and should not be used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-generatedistinctid
        '''
        result = self._values.get("generate_distinct_id")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the API key.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .
        .. epigraph::

           If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage_keys(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnApiKey.StageKeyProperty, _IResolvable_a771d0ef]]]]:
        '''A list of stages to associate with this API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-stagekeys
        '''
        result = self._values.get("stage_keys")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnApiKey.StageKeyProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def value(self) -> typing.Optional[builtins.str]:
        '''The value of the API key.

        Must be at least 20 characters long.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-value
        '''
        result = self._values.get("value")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnApiKeyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnApiMappingV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnApiMappingV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::ApiMapping``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::ApiMapping
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_api_mapping_v2 = apigateway.CfnApiMappingV2(self, "MyCfnApiMappingV2",
            api_id="apiId",
            domain_name="domainName",
            stage="stage",
        
            # the properties below are optional
            api_mapping_key="apiMappingKey"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        domain_name: builtins.str,
        stage: builtins.str,
        api_mapping_key: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::ApiMapping``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiId``.
        :param domain_name: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.DomainName``.
        :param stage: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.Stage``.
        :param api_mapping_key: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiMappingKey``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fead44bc5bd7d917e2253ede6409307da8e0d344565350bd5b76729a787705c0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnApiMappingV2Props(
            api_id=api_id,
            domain_name=domain_name,
            stage=stage,
            api_mapping_key=api_mapping_key,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4c4548626b7fe982fe93e1ca4d89a9947c84aeaaa32b9dc7fda66e33b4f7e7a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a58720824b3ff6c6788609d9fe0deb012340b52c73f023dfe40c269e43edc9dd)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66154b4256156e1171f66f9994397580c74a738125731ba5cbbf4d91d60d219c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.DomainName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-domainname
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24910ebb0c3dbc0e2f18f97ddbcd18c5a3f667e7c1495330e04939b5f3c4343e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value)

    @builtins.property
    @jsii.member(jsii_name="stage")
    def stage(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.Stage``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-stage
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "stage"))

    @stage.setter
    def stage(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__347a83b8cc194a6dae2a360ced814ed1698d889256e288ab1ecc9002abe766af)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stage", value)

    @builtins.property
    @jsii.member(jsii_name="apiMappingKey")
    def api_mapping_key(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiMappingKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-apimappingkey
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "apiMappingKey"))

    @api_mapping_key.setter
    def api_mapping_key(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0682bcec15177fa4227d30835c1be0048040eb9dbde5775101f274704f2f5682)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiMappingKey", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnApiMappingV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "domain_name": "domainName",
        "stage": "stage",
        "api_mapping_key": "apiMappingKey",
    },
)
class CfnApiMappingV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        domain_name: builtins.str,
        stage: builtins.str,
        api_mapping_key: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::ApiMapping``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiId``.
        :param domain_name: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.DomainName``.
        :param stage: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.Stage``.
        :param api_mapping_key: (deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiMappingKey``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_api_mapping_v2_props = apigateway.CfnApiMappingV2Props(
                api_id="apiId",
                domain_name="domainName",
                stage="stage",
            
                # the properties below are optional
                api_mapping_key="apiMappingKey"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6ab32c3984ce957bbdc271b0317110ff90fa8e968849ab7b4a57f2a79101d32)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument api_mapping_key", value=api_mapping_key, expected_type=type_hints["api_mapping_key"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "domain_name": domain_name,
            "stage": stage,
        }
        if api_mapping_key is not None:
            self._values["api_mapping_key"] = api_mapping_key

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.DomainName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-domainname
        :stability: deprecated
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def stage(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.Stage``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-stage
        :stability: deprecated
        '''
        result = self._values.get("stage")
        assert result is not None, "Required property 'stage' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_mapping_key(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::ApiMapping.ApiMappingKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-apimapping.html#cfn-apigatewayv2-apimapping-apimappingkey
        :stability: deprecated
        '''
        result = self._values.get("api_mapping_key")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnApiMappingV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnApiV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnApiV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Api``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Api
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # body: Any
        # tags: Any
        
        cfn_api_v2 = apigateway.CfnApiV2(self, "MyCfnApiV2",
            api_key_selection_expression="apiKeySelectionExpression",
            base_path="basePath",
            body=body,
            body_s3_location=apigateway.CfnApiV2.BodyS3LocationProperty(
                bucket="bucket",
                etag="etag",
                key="key",
                version="version"
            ),
            cors_configuration=apigateway.CfnApiV2.CorsProperty(
                allow_credentials=False,
                allow_headers=["allowHeaders"],
                allow_methods=["allowMethods"],
                allow_origins=["allowOrigins"],
                expose_headers=["exposeHeaders"],
                max_age=123
            ),
            credentials_arn="credentialsArn",
            description="description",
            disable_schema_validation=False,
            fail_on_warnings=False,
            name="name",
            protocol_type="protocolType",
            route_key="routeKey",
            route_selection_expression="routeSelectionExpression",
            tags=tags,
            target="target",
            version="version"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_key_selection_expression: typing.Optional[builtins.str] = None,
        base_path: typing.Optional[builtins.str] = None,
        body: typing.Any = None,
        body_s3_location: typing.Optional[typing.Union[typing.Union["CfnApiV2.BodyS3LocationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        cors_configuration: typing.Optional[typing.Union[typing.Union["CfnApiV2.CorsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        credentials_arn: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        disable_schema_validation: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        fail_on_warnings: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        name: typing.Optional[builtins.str] = None,
        protocol_type: typing.Optional[builtins.str] = None,
        route_key: typing.Optional[builtins.str] = None,
        route_selection_expression: typing.Optional[builtins.str] = None,
        tags: typing.Any = None,
        target: typing.Optional[builtins.str] = None,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Api``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_key_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Api.ApiKeySelectionExpression``.
        :param base_path: (deprecated) ``AWS::ApiGatewayV2::Api.BasePath``.
        :param body: (deprecated) ``AWS::ApiGatewayV2::Api.Body``.
        :param body_s3_location: (deprecated) ``AWS::ApiGatewayV2::Api.BodyS3Location``.
        :param cors_configuration: (deprecated) ``AWS::ApiGatewayV2::Api.CorsConfiguration``.
        :param credentials_arn: (deprecated) ``AWS::ApiGatewayV2::Api.CredentialsArn``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Api.Description``.
        :param disable_schema_validation: (deprecated) ``AWS::ApiGatewayV2::Api.DisableSchemaValidation``.
        :param fail_on_warnings: (deprecated) ``AWS::ApiGatewayV2::Api.FailOnWarnings``.
        :param name: (deprecated) ``AWS::ApiGatewayV2::Api.Name``.
        :param protocol_type: (deprecated) ``AWS::ApiGatewayV2::Api.ProtocolType``.
        :param route_key: (deprecated) ``AWS::ApiGatewayV2::Api.RouteKey``.
        :param route_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Api.RouteSelectionExpression``.
        :param tags: (deprecated) ``AWS::ApiGatewayV2::Api.Tags``.
        :param target: (deprecated) ``AWS::ApiGatewayV2::Api.Target``.
        :param version: (deprecated) ``AWS::ApiGatewayV2::Api.Version``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9df9ef62549fe091b14a26236e5d52f4e5d644107f2db3f97778e598e49a52c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnApiV2Props(
            api_key_selection_expression=api_key_selection_expression,
            base_path=base_path,
            body=body,
            body_s3_location=body_s3_location,
            cors_configuration=cors_configuration,
            credentials_arn=credentials_arn,
            description=description,
            disable_schema_validation=disable_schema_validation,
            fail_on_warnings=fail_on_warnings,
            name=name,
            protocol_type=protocol_type,
            route_key=route_key,
            route_selection_expression=route_selection_expression,
            tags=tags,
            target=target,
            version=version,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc89583ffc629f05649a5b6d5eeefb71efe7f8774f56e9212d5063e863825411)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52878e40e8957a66a0f8d6846586ae970a5b53f640804c52a7874efd453b8216)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-tags
        :stability: deprecated
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="body")
    def body(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Body``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-body
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "body"))

    @body.setter
    def body(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76b11372b812a4e87d4766c12088e3cd898f09c24c048884f894333aff72fd97)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "body", value)

    @builtins.property
    @jsii.member(jsii_name="apiKeySelectionExpression")
    def api_key_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.ApiKeySelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-apikeyselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "apiKeySelectionExpression"))

    @api_key_selection_expression.setter
    def api_key_selection_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9cd7c472dd6248f3722113b177ab70517a3d13feb3de8f3c8f28e8b923d02bc4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiKeySelectionExpression", value)

    @builtins.property
    @jsii.member(jsii_name="basePath")
    def base_path(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.BasePath``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-basepath
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "basePath"))

    @base_path.setter
    def base_path(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a9b9715e71ccc16bcdde96f2a3fc7203a28f4caaf6372a753a9d9e7976c7af2c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "basePath", value)

    @builtins.property
    @jsii.member(jsii_name="bodyS3Location")
    def body_s3_location(
        self,
    ) -> typing.Optional[typing.Union["CfnApiV2.BodyS3LocationProperty", _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.BodyS3Location``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-bodys3location
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union["CfnApiV2.BodyS3LocationProperty", _IResolvable_a771d0ef]], jsii.get(self, "bodyS3Location"))

    @body_s3_location.setter
    def body_s3_location(
        self,
        value: typing.Optional[typing.Union["CfnApiV2.BodyS3LocationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa3d3eaf202619386ad671703c3bcf3fef80f132263e0fd9c365f123495215c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bodyS3Location", value)

    @builtins.property
    @jsii.member(jsii_name="corsConfiguration")
    def cors_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnApiV2.CorsProperty", _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.CorsConfiguration``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-corsconfiguration
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union["CfnApiV2.CorsProperty", _IResolvable_a771d0ef]], jsii.get(self, "corsConfiguration"))

    @cors_configuration.setter
    def cors_configuration(
        self,
        value: typing.Optional[typing.Union["CfnApiV2.CorsProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__598c36ebe4fe0fbe31f838e167cf53b6d0a13216c73fa6aab1763a2d1547b849)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "corsConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="credentialsArn")
    def credentials_arn(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.CredentialsArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-credentialsarn
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "credentialsArn"))

    @credentials_arn.setter
    def credentials_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf83010327c266a2e7e295116b74d15dd852569042b39f9b3e4e773521a0077b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "credentialsArn", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-description
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c5ed3ea969832955f68bec65f9aad9e2947464682621d366eab7cbabdbe2e2cc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="disableSchemaValidation")
    def disable_schema_validation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.DisableSchemaValidation``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-disableschemavalidation
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "disableSchemaValidation"))

    @disable_schema_validation.setter
    def disable_schema_validation(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27d68e1d57f5c4bee4a2519cbbde05f8dc098ade7be7cde31c650f3c3d567ac2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "disableSchemaValidation", value)

    @builtins.property
    @jsii.member(jsii_name="failOnWarnings")
    def fail_on_warnings(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.FailOnWarnings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-failonwarnings
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "failOnWarnings"))

    @fail_on_warnings.setter
    def fail_on_warnings(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce97ab5409d76c9996b780e9f71cb0030add99decabb07bdedba1adedc347b2b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "failOnWarnings", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Name``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-name
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7714185b36f31d452529c8b1a7d221200f6b7ffec62718f9b1d27186cae17c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="protocolType")
    def protocol_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.ProtocolType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-protocoltype
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "protocolType"))

    @protocol_type.setter
    def protocol_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__045b8fe9271d638fc6d8e11533aa8136c6e7fa5d0b3a05de4dd0cdf9ffa25e83)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "protocolType", value)

    @builtins.property
    @jsii.member(jsii_name="routeKey")
    def route_key(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.RouteKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-routekey
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "routeKey"))

    @route_key.setter
    def route_key(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__720d98932b6c53ae9b00464bbcf3f49d07441d06d81d1cda525b07a147d05a6b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeKey", value)

    @builtins.property
    @jsii.member(jsii_name="routeSelectionExpression")
    def route_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.RouteSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-routeselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "routeSelectionExpression"))

    @route_selection_expression.setter
    def route_selection_expression(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8062483c79f61d901ac301ff40ef585771d07e3b3854905187621fb3f0552fa9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeSelectionExpression", value)

    @builtins.property
    @jsii.member(jsii_name="target")
    def target(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Target``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-target
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "target"))

    @target.setter
    def target(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ec6304f0d7340167b4664098cba78e17c9db9ba27c006b2fd35eee6548c9081)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "target", value)

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Version``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-version
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "version"))

    @version.setter
    def version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__44c590891cfd168f504f0d340930911e3d7ebf6fe3225f7ba6b92aa764d0d5ef)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "version", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnApiV2.BodyS3LocationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "bucket": "bucket",
            "etag": "etag",
            "key": "key",
            "version": "version",
        },
    )
    class BodyS3LocationProperty:
        def __init__(
            self,
            *,
            bucket: typing.Optional[builtins.str] = None,
            etag: typing.Optional[builtins.str] = None,
            key: typing.Optional[builtins.str] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param bucket: (deprecated) ``CfnApiV2.BodyS3LocationProperty.Bucket``.
            :param etag: (deprecated) ``CfnApiV2.BodyS3LocationProperty.Etag``.
            :param key: (deprecated) ``CfnApiV2.BodyS3LocationProperty.Key``.
            :param version: (deprecated) ``CfnApiV2.BodyS3LocationProperty.Version``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-bodys3location.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                body_s3_location_property = apigateway.CfnApiV2.BodyS3LocationProperty(
                    bucket="bucket",
                    etag="etag",
                    key="key",
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__09052bf0dfb57a4fa9644953bb1a4d51869427c0966520d289977e1476257efd)
                check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
                check_type(argname="argument etag", value=etag, expected_type=type_hints["etag"])
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if bucket is not None:
                self._values["bucket"] = bucket
            if etag is not None:
                self._values["etag"] = etag
            if key is not None:
                self._values["key"] = key
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def bucket(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnApiV2.BodyS3LocationProperty.Bucket``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-bodys3location.html#cfn-apigatewayv2-api-bodys3location-bucket
            :stability: deprecated
            '''
            result = self._values.get("bucket")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def etag(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnApiV2.BodyS3LocationProperty.Etag``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-bodys3location.html#cfn-apigatewayv2-api-bodys3location-etag
            :stability: deprecated
            '''
            result = self._values.get("etag")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnApiV2.BodyS3LocationProperty.Key``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-bodys3location.html#cfn-apigatewayv2-api-bodys3location-key
            :stability: deprecated
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnApiV2.BodyS3LocationProperty.Version``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-bodys3location.html#cfn-apigatewayv2-api-bodys3location-version
            :stability: deprecated
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BodyS3LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnApiV2.CorsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allow_credentials": "allowCredentials",
            "allow_headers": "allowHeaders",
            "allow_methods": "allowMethods",
            "allow_origins": "allowOrigins",
            "expose_headers": "exposeHeaders",
            "max_age": "maxAge",
        },
    )
    class CorsProperty:
        def __init__(
            self,
            *,
            allow_credentials: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            allow_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            allow_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            allow_origins: typing.Optional[typing.Sequence[builtins.str]] = None,
            expose_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            max_age: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param allow_credentials: (deprecated) ``CfnApiV2.CorsProperty.AllowCredentials``.
            :param allow_headers: (deprecated) ``CfnApiV2.CorsProperty.AllowHeaders``.
            :param allow_methods: (deprecated) ``CfnApiV2.CorsProperty.AllowMethods``.
            :param allow_origins: (deprecated) ``CfnApiV2.CorsProperty.AllowOrigins``.
            :param expose_headers: (deprecated) ``CfnApiV2.CorsProperty.ExposeHeaders``.
            :param max_age: (deprecated) ``CfnApiV2.CorsProperty.MaxAge``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                cors_property = apigateway.CfnApiV2.CorsProperty(
                    allow_credentials=False,
                    allow_headers=["allowHeaders"],
                    allow_methods=["allowMethods"],
                    allow_origins=["allowOrigins"],
                    expose_headers=["exposeHeaders"],
                    max_age=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__416336bd7def1cdd3bb72a189585e945f2fa65b0342c04c10a6a63be003c7968)
                check_type(argname="argument allow_credentials", value=allow_credentials, expected_type=type_hints["allow_credentials"])
                check_type(argname="argument allow_headers", value=allow_headers, expected_type=type_hints["allow_headers"])
                check_type(argname="argument allow_methods", value=allow_methods, expected_type=type_hints["allow_methods"])
                check_type(argname="argument allow_origins", value=allow_origins, expected_type=type_hints["allow_origins"])
                check_type(argname="argument expose_headers", value=expose_headers, expected_type=type_hints["expose_headers"])
                check_type(argname="argument max_age", value=max_age, expected_type=type_hints["max_age"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allow_credentials is not None:
                self._values["allow_credentials"] = allow_credentials
            if allow_headers is not None:
                self._values["allow_headers"] = allow_headers
            if allow_methods is not None:
                self._values["allow_methods"] = allow_methods
            if allow_origins is not None:
                self._values["allow_origins"] = allow_origins
            if expose_headers is not None:
                self._values["expose_headers"] = expose_headers
            if max_age is not None:
                self._values["max_age"] = max_age

        @builtins.property
        def allow_credentials(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''(deprecated) ``CfnApiV2.CorsProperty.AllowCredentials``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html#cfn-apigatewayv2-api-cors-allowcredentials
            :stability: deprecated
            '''
            result = self._values.get("allow_credentials")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def allow_headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(deprecated) ``CfnApiV2.CorsProperty.AllowHeaders``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html#cfn-apigatewayv2-api-cors-allowheaders
            :stability: deprecated
            '''
            result = self._values.get("allow_headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allow_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(deprecated) ``CfnApiV2.CorsProperty.AllowMethods``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html#cfn-apigatewayv2-api-cors-allowmethods
            :stability: deprecated
            '''
            result = self._values.get("allow_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allow_origins(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(deprecated) ``CfnApiV2.CorsProperty.AllowOrigins``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html#cfn-apigatewayv2-api-cors-alloworigins
            :stability: deprecated
            '''
            result = self._values.get("allow_origins")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def expose_headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(deprecated) ``CfnApiV2.CorsProperty.ExposeHeaders``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html#cfn-apigatewayv2-api-cors-exposeheaders
            :stability: deprecated
            '''
            result = self._values.get("expose_headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def max_age(self) -> typing.Optional[jsii.Number]:
            '''(deprecated) ``CfnApiV2.CorsProperty.MaxAge``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-api-cors.html#cfn-apigatewayv2-api-cors-maxage
            :stability: deprecated
            '''
            result = self._values.get("max_age")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CorsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnApiV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_key_selection_expression": "apiKeySelectionExpression",
        "base_path": "basePath",
        "body": "body",
        "body_s3_location": "bodyS3Location",
        "cors_configuration": "corsConfiguration",
        "credentials_arn": "credentialsArn",
        "description": "description",
        "disable_schema_validation": "disableSchemaValidation",
        "fail_on_warnings": "failOnWarnings",
        "name": "name",
        "protocol_type": "protocolType",
        "route_key": "routeKey",
        "route_selection_expression": "routeSelectionExpression",
        "tags": "tags",
        "target": "target",
        "version": "version",
    },
)
class CfnApiV2Props:
    def __init__(
        self,
        *,
        api_key_selection_expression: typing.Optional[builtins.str] = None,
        base_path: typing.Optional[builtins.str] = None,
        body: typing.Any = None,
        body_s3_location: typing.Optional[typing.Union[typing.Union[CfnApiV2.BodyS3LocationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        cors_configuration: typing.Optional[typing.Union[typing.Union[CfnApiV2.CorsProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        credentials_arn: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        disable_schema_validation: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        fail_on_warnings: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        name: typing.Optional[builtins.str] = None,
        protocol_type: typing.Optional[builtins.str] = None,
        route_key: typing.Optional[builtins.str] = None,
        route_selection_expression: typing.Optional[builtins.str] = None,
        tags: typing.Any = None,
        target: typing.Optional[builtins.str] = None,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Api``.

        :param api_key_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Api.ApiKeySelectionExpression``.
        :param base_path: (deprecated) ``AWS::ApiGatewayV2::Api.BasePath``.
        :param body: (deprecated) ``AWS::ApiGatewayV2::Api.Body``.
        :param body_s3_location: (deprecated) ``AWS::ApiGatewayV2::Api.BodyS3Location``.
        :param cors_configuration: (deprecated) ``AWS::ApiGatewayV2::Api.CorsConfiguration``.
        :param credentials_arn: (deprecated) ``AWS::ApiGatewayV2::Api.CredentialsArn``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Api.Description``.
        :param disable_schema_validation: (deprecated) ``AWS::ApiGatewayV2::Api.DisableSchemaValidation``.
        :param fail_on_warnings: (deprecated) ``AWS::ApiGatewayV2::Api.FailOnWarnings``.
        :param name: (deprecated) ``AWS::ApiGatewayV2::Api.Name``.
        :param protocol_type: (deprecated) ``AWS::ApiGatewayV2::Api.ProtocolType``.
        :param route_key: (deprecated) ``AWS::ApiGatewayV2::Api.RouteKey``.
        :param route_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Api.RouteSelectionExpression``.
        :param tags: (deprecated) ``AWS::ApiGatewayV2::Api.Tags``.
        :param target: (deprecated) ``AWS::ApiGatewayV2::Api.Target``.
        :param version: (deprecated) ``AWS::ApiGatewayV2::Api.Version``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # body: Any
            # tags: Any
            
            cfn_api_v2_props = apigateway.CfnApiV2Props(
                api_key_selection_expression="apiKeySelectionExpression",
                base_path="basePath",
                body=body,
                body_s3_location=apigateway.CfnApiV2.BodyS3LocationProperty(
                    bucket="bucket",
                    etag="etag",
                    key="key",
                    version="version"
                ),
                cors_configuration=apigateway.CfnApiV2.CorsProperty(
                    allow_credentials=False,
                    allow_headers=["allowHeaders"],
                    allow_methods=["allowMethods"],
                    allow_origins=["allowOrigins"],
                    expose_headers=["exposeHeaders"],
                    max_age=123
                ),
                credentials_arn="credentialsArn",
                description="description",
                disable_schema_validation=False,
                fail_on_warnings=False,
                name="name",
                protocol_type="protocolType",
                route_key="routeKey",
                route_selection_expression="routeSelectionExpression",
                tags=tags,
                target="target",
                version="version"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__819ebb1d44cdf644eb4b8277ca4931e8ee62daf68dbbfe9df286cdc7ce7a8daf)
            check_type(argname="argument api_key_selection_expression", value=api_key_selection_expression, expected_type=type_hints["api_key_selection_expression"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument body_s3_location", value=body_s3_location, expected_type=type_hints["body_s3_location"])
            check_type(argname="argument cors_configuration", value=cors_configuration, expected_type=type_hints["cors_configuration"])
            check_type(argname="argument credentials_arn", value=credentials_arn, expected_type=type_hints["credentials_arn"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument disable_schema_validation", value=disable_schema_validation, expected_type=type_hints["disable_schema_validation"])
            check_type(argname="argument fail_on_warnings", value=fail_on_warnings, expected_type=type_hints["fail_on_warnings"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument protocol_type", value=protocol_type, expected_type=type_hints["protocol_type"])
            check_type(argname="argument route_key", value=route_key, expected_type=type_hints["route_key"])
            check_type(argname="argument route_selection_expression", value=route_selection_expression, expected_type=type_hints["route_selection_expression"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if api_key_selection_expression is not None:
            self._values["api_key_selection_expression"] = api_key_selection_expression
        if base_path is not None:
            self._values["base_path"] = base_path
        if body is not None:
            self._values["body"] = body
        if body_s3_location is not None:
            self._values["body_s3_location"] = body_s3_location
        if cors_configuration is not None:
            self._values["cors_configuration"] = cors_configuration
        if credentials_arn is not None:
            self._values["credentials_arn"] = credentials_arn
        if description is not None:
            self._values["description"] = description
        if disable_schema_validation is not None:
            self._values["disable_schema_validation"] = disable_schema_validation
        if fail_on_warnings is not None:
            self._values["fail_on_warnings"] = fail_on_warnings
        if name is not None:
            self._values["name"] = name
        if protocol_type is not None:
            self._values["protocol_type"] = protocol_type
        if route_key is not None:
            self._values["route_key"] = route_key
        if route_selection_expression is not None:
            self._values["route_selection_expression"] = route_selection_expression
        if tags is not None:
            self._values["tags"] = tags
        if target is not None:
            self._values["target"] = target
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def api_key_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.ApiKeySelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-apikeyselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("api_key_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.BasePath``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-basepath
        :stability: deprecated
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def body(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Body``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-body
        :stability: deprecated
        '''
        result = self._values.get("body")
        return typing.cast(typing.Any, result)

    @builtins.property
    def body_s3_location(
        self,
    ) -> typing.Optional[typing.Union[CfnApiV2.BodyS3LocationProperty, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.BodyS3Location``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-bodys3location
        :stability: deprecated
        '''
        result = self._values.get("body_s3_location")
        return typing.cast(typing.Optional[typing.Union[CfnApiV2.BodyS3LocationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def cors_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnApiV2.CorsProperty, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.CorsConfiguration``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-corsconfiguration
        :stability: deprecated
        '''
        result = self._values.get("cors_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnApiV2.CorsProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def credentials_arn(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.CredentialsArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-credentialsarn
        :stability: deprecated
        '''
        result = self._values.get("credentials_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-description
        :stability: deprecated
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def disable_schema_validation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.DisableSchemaValidation``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-disableschemavalidation
        :stability: deprecated
        '''
        result = self._values.get("disable_schema_validation")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def fail_on_warnings(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.FailOnWarnings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-failonwarnings
        :stability: deprecated
        '''
        result = self._values.get("fail_on_warnings")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Name``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-name
        :stability: deprecated
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def protocol_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.ProtocolType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-protocoltype
        :stability: deprecated
        '''
        result = self._values.get("protocol_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def route_key(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.RouteKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-routekey
        :stability: deprecated
        '''
        result = self._values.get("route_key")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def route_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.RouteSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-routeselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("route_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-tags
        :stability: deprecated
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Any, result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Target``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-target
        :stability: deprecated
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Api.Version``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-api.html#cfn-apigatewayv2-api-version
        :stability: deprecated
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnApiV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnAuthorizer(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnAuthorizer",
):
    '''A CloudFormation ``AWS::ApiGateway::Authorizer``.

    The ``AWS::ApiGateway::Authorizer`` resource creates an authorization layer that API Gateway activates for methods that have authorization enabled. API Gateway activates the authorizer when a client calls those methods.

    :cloudformationResource: AWS::ApiGateway::Authorizer
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_authorizer = apigateway.CfnAuthorizer(self, "MyCfnAuthorizer",
            name="name",
            rest_api_id="restApiId",
            type="type",
        
            # the properties below are optional
            authorizer_credentials="authorizerCredentials",
            authorizer_result_ttl_in_seconds=123,
            authorizer_uri="authorizerUri",
            auth_type="authType",
            identity_source="identitySource",
            identity_validation_expression="identityValidationExpression",
            provider_arns=["providerArns"]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        name: builtins.str,
        rest_api_id: builtins.str,
        type: builtins.str,
        authorizer_credentials: typing.Optional[builtins.str] = None,
        authorizer_result_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        authorizer_uri: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[builtins.str] = None,
        identity_source: typing.Optional[builtins.str] = None,
        identity_validation_expression: typing.Optional[builtins.str] = None,
        provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Authorizer``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param name: The name of the authorizer.
        :param rest_api_id: The ID of the ``RestApi`` resource that API Gateway creates the authorizer in.
        :param type: The type of authorizer. Valid values include:. - ``TOKEN`` : A custom authorizer that uses a Lambda function. - ``COGNITO_USER_POOLS`` : An authorizer that uses Amazon Cognito user pools. - ``REQUEST`` : An authorizer that uses a Lambda function using incoming request parameters.
        :param authorizer_credentials: The credentials that are required for the authorizer. To specify an IAM role that API Gateway assumes, specify the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.
        :param authorizer_result_ttl_in_seconds: The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches authorizer results. If you specify a value greater than 0, API Gateway caches the authorizer responses. By default, API Gateway sets this property to 300. The maximum value is 3600, or 1 hour.
        :param authorizer_uri: The authorizer's Uniform Resource Identifier (URI). If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a Lambda function URI that has the form ``arn:aws:apigateway: *region* :lambda:path/ *path*`` . The path usually has the form /2015-03-31/functions/ *LambdaFunctionARN* /invocations.
        :param auth_type: An optional customer-defined field that's used in OpenApi imports and exports without functional impact.
        :param identity_source: The source of the identity in an incoming request. If you specify ``TOKEN`` or ``COGNITO_USER_POOLS`` for the ``Type`` property, this property is required. Specify a header mapping expression using the form ``method.request.header. *name*`` , where *name* is the name of a custom authorization header that clients submit as part of their requests. If you specify ``REQUEST`` for the ``Type`` property, this property is required when authorization caching is enabled. Specify a comma-separated string of one or more mapping expressions of the specified request parameter using the form ``method.request.parameter. *name*`` . For supported parameter types, see `Configure Lambda Authorizer Using the API Gateway Console <https://docs.aws.amazon.com/apigateway/latest/developerguide/configure-api-gateway-lambda-authorization-with-console.html>`_ in the *API Gateway Developer Guide* .
        :param identity_validation_expression: A validation expression for the incoming identity. If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a regular expression. API Gateway uses the expression to attempt to match the incoming client token, and proceeds if the token matches. If the token doesn't match, API Gateway responds with a 401 (unauthorized request) error code.
        :param provider_arns: A list of the Amazon Cognito user pool Amazon Resource Names (ARNs) to associate with this authorizer. Required if you specify ``COGNITO_USER_POOLS`` as the authorizer ``Type`` . For more information, see `Use Amazon Cognito User Pools <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool>`_ in the *API Gateway Developer Guide* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b9e9f99238daf040f92e6cf0c890bd68b16dc3f8601e5a0f07bd6c7b9d1fec58)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAuthorizerProps(
            name=name,
            rest_api_id=rest_api_id,
            type=type,
            authorizer_credentials=authorizer_credentials,
            authorizer_result_ttl_in_seconds=authorizer_result_ttl_in_seconds,
            authorizer_uri=authorizer_uri,
            auth_type=auth_type,
            identity_source=identity_source,
            identity_validation_expression=identity_validation_expression,
            provider_arns=provider_arns,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf37a10408f89a4604d79d0afd05e1d1cbc43a2382ed9dd41e71ad52cf5b4af4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__935ed4680ecbc7c53175cf4b34427ead89204d32bc2760f6173d8ae086dcf389)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAuthorizerId")
    def attr_authorizer_id(self) -> builtins.str:
        '''The ID for the authorizer.

        For example: ``abc123`` .

        :cloudformationAttribute: AuthorizerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAuthorizerId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the authorizer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-name
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a3e4cff560ce4af7eaa9b1cd6cb39ee906cbc27f4cfaebba429e9b6932f54e3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of the ``RestApi`` resource that API Gateway creates the authorizer in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d14093c3d4ee157d0132610acd9dd857a70c3e2602dfdd65b6796e3f34230537)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of authorizer. Valid values include:.

        - ``TOKEN`` : A custom authorizer that uses a Lambda function.
        - ``COGNITO_USER_POOLS`` : An authorizer that uses Amazon Cognito user pools.
        - ``REQUEST`` : An authorizer that uses a Lambda function using incoming request parameters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-type
        '''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8cca46adfbf5221d9c45ec2d281229958482ab92f5db988ed0641f3d2f2c5da4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerCredentials")
    def authorizer_credentials(self) -> typing.Optional[builtins.str]:
        '''The credentials that are required for the authorizer.

        To specify an IAM role that API Gateway assumes, specify the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizercredentials
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerCredentials"))

    @authorizer_credentials.setter
    def authorizer_credentials(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33566414e9b15625ab23d56608ba951d91abfec25dca986f59bc0ef3ef021869)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerCredentials", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerResultTtlInSeconds")
    def authorizer_result_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches authorizer results.

        If you specify a value greater than 0, API Gateway caches the authorizer responses. By default, API Gateway sets this property to 300. The maximum value is 3600, or 1 hour.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizerresultttlinseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "authorizerResultTtlInSeconds"))

    @authorizer_result_ttl_in_seconds.setter
    def authorizer_result_ttl_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__01b2ec8e4a3ed5c83fc9441ea5ed688a8d867880ba0c829b81e39c321a98992e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerResultTtlInSeconds", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerUri")
    def authorizer_uri(self) -> typing.Optional[builtins.str]:
        '''The authorizer's Uniform Resource Identifier (URI).

        If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a Lambda function URI that has the form ``arn:aws:apigateway: *region* :lambda:path/ *path*`` . The path usually has the form /2015-03-31/functions/ *LambdaFunctionARN* /invocations.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizeruri
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerUri"))

    @authorizer_uri.setter
    def authorizer_uri(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9141685741efa272befefee16582c11ec3b03a7aa72299a6a3b73369190b8bac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerUri", value)

    @builtins.property
    @jsii.member(jsii_name="authType")
    def auth_type(self) -> typing.Optional[builtins.str]:
        '''An optional customer-defined field that's used in OpenApi imports and exports without functional impact.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authType"))

    @auth_type.setter
    def auth_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bbb1f421ee050bbe66deb4220281d97f2c44f83e040194691e59b345eeedbf70)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authType", value)

    @builtins.property
    @jsii.member(jsii_name="identitySource")
    def identity_source(self) -> typing.Optional[builtins.str]:
        '''The source of the identity in an incoming request.

        If you specify ``TOKEN`` or ``COGNITO_USER_POOLS`` for the ``Type`` property, this property is required. Specify a header mapping expression using the form ``method.request.header. *name*`` , where *name* is the name of a custom authorization header that clients submit as part of their requests.

        If you specify ``REQUEST`` for the ``Type`` property, this property is required when authorization caching is enabled. Specify a comma-separated string of one or more mapping expressions of the specified request parameter using the form ``method.request.parameter. *name*`` . For supported parameter types, see `Configure Lambda Authorizer Using the API Gateway Console <https://docs.aws.amazon.com/apigateway/latest/developerguide/configure-api-gateway-lambda-authorization-with-console.html>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identitysource
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identitySource"))

    @identity_source.setter
    def identity_source(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0bf32956dc21da7f41edb5b8e8a4e40db0838412fde845ed3daef9f82aff793)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identitySource", value)

    @builtins.property
    @jsii.member(jsii_name="identityValidationExpression")
    def identity_validation_expression(self) -> typing.Optional[builtins.str]:
        '''A validation expression for the incoming identity.

        If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a regular expression. API Gateway uses the expression to attempt to match the incoming client token, and proceeds if the token matches. If the token doesn't match, API Gateway responds with a 401 (unauthorized request) error code.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identityvalidationexpression
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identityValidationExpression"))

    @identity_validation_expression.setter
    def identity_validation_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be3627cf0587bda6a520ac3628914e728c01364e594f303aefaf5ea78c39b556)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityValidationExpression", value)

    @builtins.property
    @jsii.member(jsii_name="providerArns")
    def provider_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the Amazon Cognito user pool Amazon Resource Names (ARNs) to associate with this authorizer.

        Required if you specify ``COGNITO_USER_POOLS`` as the authorizer ``Type`` . For more information, see `Use Amazon Cognito User Pools <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-providerarns
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "providerArns"))

    @provider_arns.setter
    def provider_arns(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dac23541d28c1e297b23adefcfe326bb49e08bdb24698149362f7568fac32584)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "providerArns", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnAuthorizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "name": "name",
        "rest_api_id": "restApiId",
        "type": "type",
        "authorizer_credentials": "authorizerCredentials",
        "authorizer_result_ttl_in_seconds": "authorizerResultTtlInSeconds",
        "authorizer_uri": "authorizerUri",
        "auth_type": "authType",
        "identity_source": "identitySource",
        "identity_validation_expression": "identityValidationExpression",
        "provider_arns": "providerArns",
    },
)
class CfnAuthorizerProps:
    def __init__(
        self,
        *,
        name: builtins.str,
        rest_api_id: builtins.str,
        type: builtins.str,
        authorizer_credentials: typing.Optional[builtins.str] = None,
        authorizer_result_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        authorizer_uri: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[builtins.str] = None,
        identity_source: typing.Optional[builtins.str] = None,
        identity_validation_expression: typing.Optional[builtins.str] = None,
        provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAuthorizer``.

        :param name: The name of the authorizer.
        :param rest_api_id: The ID of the ``RestApi`` resource that API Gateway creates the authorizer in.
        :param type: The type of authorizer. Valid values include:. - ``TOKEN`` : A custom authorizer that uses a Lambda function. - ``COGNITO_USER_POOLS`` : An authorizer that uses Amazon Cognito user pools. - ``REQUEST`` : An authorizer that uses a Lambda function using incoming request parameters.
        :param authorizer_credentials: The credentials that are required for the authorizer. To specify an IAM role that API Gateway assumes, specify the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.
        :param authorizer_result_ttl_in_seconds: The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches authorizer results. If you specify a value greater than 0, API Gateway caches the authorizer responses. By default, API Gateway sets this property to 300. The maximum value is 3600, or 1 hour.
        :param authorizer_uri: The authorizer's Uniform Resource Identifier (URI). If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a Lambda function URI that has the form ``arn:aws:apigateway: *region* :lambda:path/ *path*`` . The path usually has the form /2015-03-31/functions/ *LambdaFunctionARN* /invocations.
        :param auth_type: An optional customer-defined field that's used in OpenApi imports and exports without functional impact.
        :param identity_source: The source of the identity in an incoming request. If you specify ``TOKEN`` or ``COGNITO_USER_POOLS`` for the ``Type`` property, this property is required. Specify a header mapping expression using the form ``method.request.header. *name*`` , where *name* is the name of a custom authorization header that clients submit as part of their requests. If you specify ``REQUEST`` for the ``Type`` property, this property is required when authorization caching is enabled. Specify a comma-separated string of one or more mapping expressions of the specified request parameter using the form ``method.request.parameter. *name*`` . For supported parameter types, see `Configure Lambda Authorizer Using the API Gateway Console <https://docs.aws.amazon.com/apigateway/latest/developerguide/configure-api-gateway-lambda-authorization-with-console.html>`_ in the *API Gateway Developer Guide* .
        :param identity_validation_expression: A validation expression for the incoming identity. If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a regular expression. API Gateway uses the expression to attempt to match the incoming client token, and proceeds if the token matches. If the token doesn't match, API Gateway responds with a 401 (unauthorized request) error code.
        :param provider_arns: A list of the Amazon Cognito user pool Amazon Resource Names (ARNs) to associate with this authorizer. Required if you specify ``COGNITO_USER_POOLS`` as the authorizer ``Type`` . For more information, see `Use Amazon Cognito User Pools <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_authorizer_props = apigateway.CfnAuthorizerProps(
                name="name",
                rest_api_id="restApiId",
                type="type",
            
                # the properties below are optional
                authorizer_credentials="authorizerCredentials",
                authorizer_result_ttl_in_seconds=123,
                authorizer_uri="authorizerUri",
                auth_type="authType",
                identity_source="identitySource",
                identity_validation_expression="identityValidationExpression",
                provider_arns=["providerArns"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0877799b9d004fd2e0f95e59c679965e569f81e00017f4499fbaeda95f24b11a)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument authorizer_credentials", value=authorizer_credentials, expected_type=type_hints["authorizer_credentials"])
            check_type(argname="argument authorizer_result_ttl_in_seconds", value=authorizer_result_ttl_in_seconds, expected_type=type_hints["authorizer_result_ttl_in_seconds"])
            check_type(argname="argument authorizer_uri", value=authorizer_uri, expected_type=type_hints["authorizer_uri"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument identity_source", value=identity_source, expected_type=type_hints["identity_source"])
            check_type(argname="argument identity_validation_expression", value=identity_validation_expression, expected_type=type_hints["identity_validation_expression"])
            check_type(argname="argument provider_arns", value=provider_arns, expected_type=type_hints["provider_arns"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "name": name,
            "rest_api_id": rest_api_id,
            "type": type,
        }
        if authorizer_credentials is not None:
            self._values["authorizer_credentials"] = authorizer_credentials
        if authorizer_result_ttl_in_seconds is not None:
            self._values["authorizer_result_ttl_in_seconds"] = authorizer_result_ttl_in_seconds
        if authorizer_uri is not None:
            self._values["authorizer_uri"] = authorizer_uri
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if identity_source is not None:
            self._values["identity_source"] = identity_source
        if identity_validation_expression is not None:
            self._values["identity_validation_expression"] = identity_validation_expression
        if provider_arns is not None:
            self._values["provider_arns"] = provider_arns

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the authorizer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The ID of the ``RestApi`` resource that API Gateway creates the authorizer in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of authorizer. Valid values include:.

        - ``TOKEN`` : A custom authorizer that uses a Lambda function.
        - ``COGNITO_USER_POOLS`` : An authorizer that uses Amazon Cognito user pools.
        - ``REQUEST`` : An authorizer that uses a Lambda function using incoming request parameters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def authorizer_credentials(self) -> typing.Optional[builtins.str]:
        '''The credentials that are required for the authorizer.

        To specify an IAM role that API Gateway assumes, specify the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizercredentials
        '''
        result = self._values.get("authorizer_credentials")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_result_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches authorizer results.

        If you specify a value greater than 0, API Gateway caches the authorizer responses. By default, API Gateway sets this property to 300. The maximum value is 3600, or 1 hour.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizerresultttlinseconds
        '''
        result = self._values.get("authorizer_result_ttl_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def authorizer_uri(self) -> typing.Optional[builtins.str]:
        '''The authorizer's Uniform Resource Identifier (URI).

        If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a Lambda function URI that has the form ``arn:aws:apigateway: *region* :lambda:path/ *path*`` . The path usually has the form /2015-03-31/functions/ *LambdaFunctionARN* /invocations.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizeruri
        '''
        result = self._values.get("authorizer_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[builtins.str]:
        '''An optional customer-defined field that's used in OpenApi imports and exports without functional impact.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authtype
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_source(self) -> typing.Optional[builtins.str]:
        '''The source of the identity in an incoming request.

        If you specify ``TOKEN`` or ``COGNITO_USER_POOLS`` for the ``Type`` property, this property is required. Specify a header mapping expression using the form ``method.request.header. *name*`` , where *name* is the name of a custom authorization header that clients submit as part of their requests.

        If you specify ``REQUEST`` for the ``Type`` property, this property is required when authorization caching is enabled. Specify a comma-separated string of one or more mapping expressions of the specified request parameter using the form ``method.request.parameter. *name*`` . For supported parameter types, see `Configure Lambda Authorizer Using the API Gateway Console <https://docs.aws.amazon.com/apigateway/latest/developerguide/configure-api-gateway-lambda-authorization-with-console.html>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identitysource
        '''
        result = self._values.get("identity_source")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_validation_expression(self) -> typing.Optional[builtins.str]:
        '''A validation expression for the incoming identity.

        If you specify ``TOKEN`` for the authorizer's ``Type`` property, specify a regular expression. API Gateway uses the expression to attempt to match the incoming client token, and proceeds if the token matches. If the token doesn't match, API Gateway responds with a 401 (unauthorized request) error code.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identityvalidationexpression
        '''
        result = self._values.get("identity_validation_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provider_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the Amazon Cognito user pool Amazon Resource Names (ARNs) to associate with this authorizer.

        Required if you specify ``COGNITO_USER_POOLS`` as the authorizer ``Type`` . For more information, see `Use Amazon Cognito User Pools <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-providerarns
        '''
        result = self._values.get("provider_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAuthorizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnAuthorizerV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnAuthorizerV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Authorizer``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Authorizer
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_authorizer_v2 = apigateway.CfnAuthorizerV2(self, "MyCfnAuthorizerV2",
            api_id="apiId",
            authorizer_type="authorizerType",
            identity_source=["identitySource"],
            name="name",
        
            # the properties below are optional
            authorizer_credentials_arn="authorizerCredentialsArn",
            authorizer_result_ttl_in_seconds=123,
            authorizer_uri="authorizerUri",
            identity_validation_expression="identityValidationExpression",
            jwt_configuration=apigateway.CfnAuthorizerV2.JWTConfigurationProperty(
                audience=["audience"],
                issuer="issuer"
            )
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        authorizer_type: builtins.str,
        identity_source: typing.Sequence[builtins.str],
        name: builtins.str,
        authorizer_credentials_arn: typing.Optional[builtins.str] = None,
        authorizer_result_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        authorizer_uri: typing.Optional[builtins.str] = None,
        identity_validation_expression: typing.Optional[builtins.str] = None,
        jwt_configuration: typing.Optional[typing.Union[typing.Union["CfnAuthorizerV2.JWTConfigurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Authorizer``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Authorizer.ApiId``.
        :param authorizer_type: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerType``.
        :param identity_source: (deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentitySource``.
        :param name: (deprecated) ``AWS::ApiGatewayV2::Authorizer.Name``.
        :param authorizer_credentials_arn: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerCredentialsArn``.
        :param authorizer_result_ttl_in_seconds: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerResultTtlInSeconds``.
        :param authorizer_uri: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerUri``.
        :param identity_validation_expression: (deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentityValidationExpression``.
        :param jwt_configuration: (deprecated) ``AWS::ApiGatewayV2::Authorizer.JwtConfiguration``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2860bf47c2213699cdf45f18ef893a309429c8d74c756aa383e7bce3d70f47e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAuthorizerV2Props(
            api_id=api_id,
            authorizer_type=authorizer_type,
            identity_source=identity_source,
            name=name,
            authorizer_credentials_arn=authorizer_credentials_arn,
            authorizer_result_ttl_in_seconds=authorizer_result_ttl_in_seconds,
            authorizer_uri=authorizer_uri,
            identity_validation_expression=identity_validation_expression,
            jwt_configuration=jwt_configuration,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__233db25805e635d06ab75fb07d5956f119d6f788b7b319583c6f0648c4b7d603)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__960dc8af8d67ec1aa9800df6da1fafd4c64d649a041f3ceeeec648eb10b8fe82)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__86c491bbf6d9a1552c92554929755b1c3681796fb1a8a4f0a2a9465edca97aad)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerType")
    def authorizer_type(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizertype
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "authorizerType"))

    @authorizer_type.setter
    def authorizer_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f85d0953520b4ce15528bb32cd6728deade0cc65b9f66491efdea1f80891b1e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerType", value)

    @builtins.property
    @jsii.member(jsii_name="identitySource")
    def identity_source(self) -> typing.List[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentitySource``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-identitysource
        :stability: deprecated
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "identitySource"))

    @identity_source.setter
    def identity_source(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9bd03bc635633e7cac7686d9057a774d52d00f175e52125396bee1c0bc457f26)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identitySource", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.Name``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-name
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ddb6f64973808c6fe31705c1cb9d34bb096030a495e6fc934e862fb1e26f446)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerCredentialsArn")
    def authorizer_credentials_arn(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerCredentialsArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizercredentialsarn
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerCredentialsArn"))

    @authorizer_credentials_arn.setter
    def authorizer_credentials_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb82610c6139c05542ad01bae76d1ee27f8cde77f2690a7ca88643bb90950a52)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerCredentialsArn", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerResultTtlInSeconds")
    def authorizer_result_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerResultTtlInSeconds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizerresultttlinseconds
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "authorizerResultTtlInSeconds"))

    @authorizer_result_ttl_in_seconds.setter
    def authorizer_result_ttl_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a43d6ba321e1f74af3a18d2e612ca0419b060f73a632e07a760b89b4700cf15)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerResultTtlInSeconds", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerUri")
    def authorizer_uri(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerUri``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizeruri
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerUri"))

    @authorizer_uri.setter
    def authorizer_uri(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d308f6a4507e6c9ad02fb94957d4a93fbb2703206a2fff340e9ce44c3099b84)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerUri", value)

    @builtins.property
    @jsii.member(jsii_name="identityValidationExpression")
    def identity_validation_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentityValidationExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-identityvalidationexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identityValidationExpression"))

    @identity_validation_expression.setter
    def identity_validation_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42cb696678a725596a8fce442420ba2813cd1aea482a5df9b648be222296e4d7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityValidationExpression", value)

    @builtins.property
    @jsii.member(jsii_name="jwtConfiguration")
    def jwt_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnAuthorizerV2.JWTConfigurationProperty", _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.JwtConfiguration``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-jwtconfiguration
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union["CfnAuthorizerV2.JWTConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "jwtConfiguration"))

    @jwt_configuration.setter
    def jwt_configuration(
        self,
        value: typing.Optional[typing.Union["CfnAuthorizerV2.JWTConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__577f2879f9baf49e9238ff2dfdaf81bde82f4866c77bf3478210dde38a079652)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "jwtConfiguration", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnAuthorizerV2.JWTConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"audience": "audience", "issuer": "issuer"},
    )
    class JWTConfigurationProperty:
        def __init__(
            self,
            *,
            audience: typing.Optional[typing.Sequence[builtins.str]] = None,
            issuer: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param audience: (deprecated) ``CfnAuthorizerV2.JWTConfigurationProperty.Audience``.
            :param issuer: (deprecated) ``CfnAuthorizerV2.JWTConfigurationProperty.Issuer``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-authorizer-jwtconfiguration.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                j_wTConfiguration_property = apigateway.CfnAuthorizerV2.JWTConfigurationProperty(
                    audience=["audience"],
                    issuer="issuer"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__22a6927850b8021a5c13501e9a9d612e716b4675f7e09d0cb26d7805a987bf00)
                check_type(argname="argument audience", value=audience, expected_type=type_hints["audience"])
                check_type(argname="argument issuer", value=issuer, expected_type=type_hints["issuer"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if audience is not None:
                self._values["audience"] = audience
            if issuer is not None:
                self._values["issuer"] = issuer

        @builtins.property
        def audience(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(deprecated) ``CfnAuthorizerV2.JWTConfigurationProperty.Audience``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-authorizer-jwtconfiguration.html#cfn-apigatewayv2-authorizer-jwtconfiguration-audience
            :stability: deprecated
            '''
            result = self._values.get("audience")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def issuer(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnAuthorizerV2.JWTConfigurationProperty.Issuer``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-authorizer-jwtconfiguration.html#cfn-apigatewayv2-authorizer-jwtconfiguration-issuer
            :stability: deprecated
            '''
            result = self._values.get("issuer")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JWTConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnAuthorizerV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "authorizer_type": "authorizerType",
        "identity_source": "identitySource",
        "name": "name",
        "authorizer_credentials_arn": "authorizerCredentialsArn",
        "authorizer_result_ttl_in_seconds": "authorizerResultTtlInSeconds",
        "authorizer_uri": "authorizerUri",
        "identity_validation_expression": "identityValidationExpression",
        "jwt_configuration": "jwtConfiguration",
    },
)
class CfnAuthorizerV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        authorizer_type: builtins.str,
        identity_source: typing.Sequence[builtins.str],
        name: builtins.str,
        authorizer_credentials_arn: typing.Optional[builtins.str] = None,
        authorizer_result_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        authorizer_uri: typing.Optional[builtins.str] = None,
        identity_validation_expression: typing.Optional[builtins.str] = None,
        jwt_configuration: typing.Optional[typing.Union[typing.Union[CfnAuthorizerV2.JWTConfigurationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Authorizer``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Authorizer.ApiId``.
        :param authorizer_type: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerType``.
        :param identity_source: (deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentitySource``.
        :param name: (deprecated) ``AWS::ApiGatewayV2::Authorizer.Name``.
        :param authorizer_credentials_arn: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerCredentialsArn``.
        :param authorizer_result_ttl_in_seconds: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerResultTtlInSeconds``.
        :param authorizer_uri: (deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerUri``.
        :param identity_validation_expression: (deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentityValidationExpression``.
        :param jwt_configuration: (deprecated) ``AWS::ApiGatewayV2::Authorizer.JwtConfiguration``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_authorizer_v2_props = apigateway.CfnAuthorizerV2Props(
                api_id="apiId",
                authorizer_type="authorizerType",
                identity_source=["identitySource"],
                name="name",
            
                # the properties below are optional
                authorizer_credentials_arn="authorizerCredentialsArn",
                authorizer_result_ttl_in_seconds=123,
                authorizer_uri="authorizerUri",
                identity_validation_expression="identityValidationExpression",
                jwt_configuration=apigateway.CfnAuthorizerV2.JWTConfigurationProperty(
                    audience=["audience"],
                    issuer="issuer"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__99a8563441f4eb084d3af4203b1e0e6d78ebb11453c0d8c9cfbe130be5ba93d1)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument authorizer_type", value=authorizer_type, expected_type=type_hints["authorizer_type"])
            check_type(argname="argument identity_source", value=identity_source, expected_type=type_hints["identity_source"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument authorizer_credentials_arn", value=authorizer_credentials_arn, expected_type=type_hints["authorizer_credentials_arn"])
            check_type(argname="argument authorizer_result_ttl_in_seconds", value=authorizer_result_ttl_in_seconds, expected_type=type_hints["authorizer_result_ttl_in_seconds"])
            check_type(argname="argument authorizer_uri", value=authorizer_uri, expected_type=type_hints["authorizer_uri"])
            check_type(argname="argument identity_validation_expression", value=identity_validation_expression, expected_type=type_hints["identity_validation_expression"])
            check_type(argname="argument jwt_configuration", value=jwt_configuration, expected_type=type_hints["jwt_configuration"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "authorizer_type": authorizer_type,
            "identity_source": identity_source,
            "name": name,
        }
        if authorizer_credentials_arn is not None:
            self._values["authorizer_credentials_arn"] = authorizer_credentials_arn
        if authorizer_result_ttl_in_seconds is not None:
            self._values["authorizer_result_ttl_in_seconds"] = authorizer_result_ttl_in_seconds
        if authorizer_uri is not None:
            self._values["authorizer_uri"] = authorizer_uri
        if identity_validation_expression is not None:
            self._values["identity_validation_expression"] = identity_validation_expression
        if jwt_configuration is not None:
            self._values["jwt_configuration"] = jwt_configuration

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def authorizer_type(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizertype
        :stability: deprecated
        '''
        result = self._values.get("authorizer_type")
        assert result is not None, "Required property 'authorizer_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def identity_source(self) -> typing.List[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentitySource``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-identitysource
        :stability: deprecated
        '''
        result = self._values.get("identity_source")
        assert result is not None, "Required property 'identity_source' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.Name``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-name
        :stability: deprecated
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def authorizer_credentials_arn(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerCredentialsArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizercredentialsarn
        :stability: deprecated
        '''
        result = self._values.get("authorizer_credentials_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_result_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerResultTtlInSeconds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizerresultttlinseconds
        :stability: deprecated
        '''
        result = self._values.get("authorizer_result_ttl_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def authorizer_uri(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.AuthorizerUri``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-authorizeruri
        :stability: deprecated
        '''
        result = self._values.get("authorizer_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_validation_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.IdentityValidationExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-identityvalidationexpression
        :stability: deprecated
        '''
        result = self._values.get("identity_validation_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def jwt_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnAuthorizerV2.JWTConfigurationProperty, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Authorizer.JwtConfiguration``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-authorizer.html#cfn-apigatewayv2-authorizer-jwtconfiguration
        :stability: deprecated
        '''
        result = self._values.get("jwt_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnAuthorizerV2.JWTConfigurationProperty, _IResolvable_a771d0ef]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAuthorizerV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnBasePathMapping(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnBasePathMapping",
):
    '''A CloudFormation ``AWS::ApiGateway::BasePathMapping``.

    The ``AWS::ApiGateway::BasePathMapping`` resource creates a base path that clients who call your API must use in the invocation URL.

    :cloudformationResource: AWS::ApiGateway::BasePathMapping
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_base_path_mapping = apigateway.CfnBasePathMapping(self, "MyCfnBasePathMapping",
            domain_name="domainName",
        
            # the properties below are optional
            base_path="basePath",
            id="id",
            rest_api_id="restApiId",
            stage="stage"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id_: builtins.str,
        *,
        domain_name: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        id: typing.Optional[builtins.str] = None,
        rest_api_id: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::BasePathMapping``.

        :param scope: - scope in which this resource is defined.
        :param id_: - scoped id of the resource.
        :param domain_name: The ``DomainName`` of an `AWS::ApiGateway::DomainName <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html>`_ resource.
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name.
        :param id: ``AWS::ApiGateway::BasePathMapping.Id``.
        :param rest_api_id: The ID of the API.
        :param stage: The name of the API's stage.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__65f985ae6105ca9cbc7b2572276f8b8c86afbfccfb2e4e349642ead3d151be4a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id_", value=id_, expected_type=type_hints["id_"])
        props = CfnBasePathMappingProps(
            domain_name=domain_name,
            base_path=base_path,
            id=id,
            rest_api_id=rest_api_id,
            stage=stage,
        )

        jsii.create(self.__class__, self, [scope, id_, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4220b3f93181a6bc79e2c723a6859fef64b003393d8e58d32ec883525089afe2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00ddc8360ad0382d7fbff79636c50ffee0029a8a4b9e4023be89a627825fddfe)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''The ``DomainName`` of an `AWS::ApiGateway::DomainName <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html>`_ resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-domainname
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__799160d51682af8517b08bbbd156fc0cc8f99c131cadfeca67477b11a68500e3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value)

    @builtins.property
    @jsii.member(jsii_name="basePath")
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide in the URL after the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-basepath
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "basePath"))

    @base_path.setter
    def base_path(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2764723761301e4dffa88c91b3eb64222673cc5f9a6f3fbc2a13328a4e18fe58)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "basePath", value)

    @builtins.property
    @jsii.member(jsii_name="id")
    def id(self) -> typing.Optional[builtins.str]:
        '''``AWS::ApiGateway::BasePathMapping.Id``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-id
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "id"))

    @id.setter
    def id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e4b962836413d593142ae93ec7b54ba8fc8bb70966717dfa818b5ab7c8d8a0f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "id", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-restapiid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__12261f17083edf57d41b2e0a6749ab17b8b1891d6ab1dd18b05a2f617ee0091f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="stage")
    def stage(self) -> typing.Optional[builtins.str]:
        '''The name of the API's stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-stage
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stage"))

    @stage.setter
    def stage(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3c597e987e52581a07744eadff9695210f4907f01cfcf539418c1ed89de70ff7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stage", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnBasePathMappingProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "base_path": "basePath",
        "id": "id",
        "rest_api_id": "restApiId",
        "stage": "stage",
    },
)
class CfnBasePathMappingProps:
    def __init__(
        self,
        *,
        domain_name: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        id: typing.Optional[builtins.str] = None,
        rest_api_id: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnBasePathMapping``.

        :param domain_name: The ``DomainName`` of an `AWS::ApiGateway::DomainName <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html>`_ resource.
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name.
        :param id: ``AWS::ApiGateway::BasePathMapping.Id``.
        :param rest_api_id: The ID of the API.
        :param stage: The name of the API's stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_base_path_mapping_props = apigateway.CfnBasePathMappingProps(
                domain_name="domainName",
            
                # the properties below are optional
                base_path="basePath",
                id="id",
                rest_api_id="restApiId",
                stage="stage"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7981e8bc46fdbc88a81a45984979d68b84b53cdf95b815deb284de868863c723)
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
        }
        if base_path is not None:
            self._values["base_path"] = base_path
        if id is not None:
            self._values["id"] = id
        if rest_api_id is not None:
            self._values["rest_api_id"] = rest_api_id
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The ``DomainName`` of an `AWS::ApiGateway::DomainName <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html>`_ resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-domainname
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide in the URL after the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-basepath
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def id(self) -> typing.Optional[builtins.str]:
        '''``AWS::ApiGateway::BasePathMapping.Id``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-id
        '''
        result = self._values.get("id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def rest_api_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-restapiid
        '''
        result = self._values.get("rest_api_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''The name of the API's stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-stage
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnBasePathMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnClientCertificate(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnClientCertificate",
):
    '''A CloudFormation ``AWS::ApiGateway::ClientCertificate``.

    The ``AWS::ApiGateway::ClientCertificate`` resource creates a client certificate that API Gateway uses to configure client-side SSL authentication for sending requests to the integration endpoint.

    :cloudformationResource: AWS::ApiGateway::ClientCertificate
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_client_certificate = apigateway.CfnClientCertificate(self, "MyCfnClientCertificate",
            description="description",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::ClientCertificate``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: A description of the client certificate.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the client certificate.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c48ba671d9ce578591b5433bbe7f95ce6287170bafb1da8d7109e4bbd6b2e94)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClientCertificateProps(description=description, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d05ea53fb54dbd33b32c9fe6928a96eaae134b96aaf6dd8faa51bc197cb878c)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a2ad23a30f181e46ba70961ff999516466d6b72eb68a61fd4f0d0c8eb934e9d)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrClientCertificateId")
    def attr_client_certificate_id(self) -> builtins.str:
        '''The ID for the client certificate.

        For example: ``abc123`` .

        :cloudformationAttribute: ClientCertificateId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrClientCertificateId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the client certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html#cfn-apigateway-clientcertificate-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the client certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html#cfn-apigateway-clientcertificate-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0bf80f628c60029c344e2abf94231a67bd85945e6fc4cc350d5a115df49c9bb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnClientCertificateProps",
    jsii_struct_bases=[],
    name_mapping={"description": "description", "tags": "tags"},
)
class CfnClientCertificateProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientCertificate``.

        :param description: A description of the client certificate.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the client certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_client_certificate_props = apigateway.CfnClientCertificateProps(
                description="description",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95668a163346f2724f5cfb82125e4d920c76aa9ef950b35025722dcbd09d8d31)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the client certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html#cfn-apigateway-clientcertificate-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the client certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html#cfn-apigateway-clientcertificate-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientCertificateProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnDeployment(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnDeployment",
):
    '''A CloudFormation ``AWS::ApiGateway::Deployment``.

    The ``AWS::ApiGateway::Deployment`` resource deploys an API Gateway ``RestApi`` resource to a stage so that clients can call the API over the internet. The stage acts as an environment.

    :cloudformationResource: AWS::ApiGateway::Deployment
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_deployment = apigateway.CfnDeployment(self, "MyCfnDeployment",
            rest_api_id="restApiId",
        
            # the properties below are optional
            deployment_canary_settings=apigateway.CfnDeployment.DeploymentCanarySettingsProperty(
                percent_traffic=123,
                stage_variable_overrides={
                    "stage_variable_overrides_key": "stageVariableOverrides"
                },
                use_stage_cache=False
            ),
            description="description",
            stage_description=apigateway.CfnDeployment.StageDescriptionProperty(
                access_log_setting=apigateway.CfnDeployment.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                ),
                cache_cluster_enabled=False,
                cache_cluster_size="cacheClusterSize",
                cache_data_encrypted=False,
                cache_ttl_in_seconds=123,
                caching_enabled=False,
                canary_setting=apigateway.CfnDeployment.CanarySettingProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                ),
                client_certificate_id="clientCertificateId",
                data_trace_enabled=False,
                description="description",
                documentation_version="documentationVersion",
                logging_level="loggingLevel",
                method_settings=[apigateway.CfnDeployment.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )],
                metrics_enabled=False,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                throttling_burst_limit=123,
                throttling_rate_limit=123,
                tracing_enabled=False,
                variables={
                    "variables_key": "variables"
                }
            ),
            stage_name="stageName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        deployment_canary_settings: typing.Optional[typing.Union[typing.Union["CfnDeployment.DeploymentCanarySettingsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        description: typing.Optional[builtins.str] = None,
        stage_description: typing.Optional[typing.Union[typing.Union["CfnDeployment.StageDescriptionProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Deployment``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param rest_api_id: The ID of the ``RestApi`` resource to deploy.
        :param deployment_canary_settings: Specifies settings for the canary deployment.
        :param description: A description of the purpose of the API Gateway deployment.
        :param stage_description: Configures the stage that API Gateway creates with this deployment.
        :param stage_name: A name for the stage that API Gateway creates with this deployment. Use only alphanumeric characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63ac287db5c0c19b8860fda40d692912b9b4a5c8020d337eb8429ec5f94d024a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDeploymentProps(
            rest_api_id=rest_api_id,
            deployment_canary_settings=deployment_canary_settings,
            description=description,
            stage_description=stage_description,
            stage_name=stage_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4735762988eb79baa711e08777fa32e62fc24162934e4ae9b7f4d2db17cc319d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9476bb86c210706b41a4c75504f45bd8dd3ebb03a6903403f79b7bff775b3cbf)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDeploymentId")
    def attr_deployment_id(self) -> builtins.str:
        '''The ID for the deployment.

        For example: ``abc123`` .

        :cloudformationAttribute: DeploymentId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDeploymentId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of the ``RestApi`` resource to deploy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e88bd45a2f3ef0811fabf775455b72d993b70763144709f18118b6a5653fe70d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="deploymentCanarySettings")
    def deployment_canary_settings(
        self,
    ) -> typing.Optional[typing.Union["CfnDeployment.DeploymentCanarySettingsProperty", _IResolvable_a771d0ef]]:
        '''Specifies settings for the canary deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-deploymentcanarysettings
        '''
        return typing.cast(typing.Optional[typing.Union["CfnDeployment.DeploymentCanarySettingsProperty", _IResolvable_a771d0ef]], jsii.get(self, "deploymentCanarySettings"))

    @deployment_canary_settings.setter
    def deployment_canary_settings(
        self,
        value: typing.Optional[typing.Union["CfnDeployment.DeploymentCanarySettingsProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de5e13ed72692a10e71078c9ade8e3ab29036266684361f9a52903dfddddfa30)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentCanarySettings", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the purpose of the API Gateway deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e046aa03e699ebba8eebc774cb753c09c1033845d19a9e6c76847a85a4ab9bf7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="stageDescription")
    def stage_description(
        self,
    ) -> typing.Optional[typing.Union["CfnDeployment.StageDescriptionProperty", _IResolvable_a771d0ef]]:
        '''Configures the stage that API Gateway creates with this deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-stagedescription
        '''
        return typing.cast(typing.Optional[typing.Union["CfnDeployment.StageDescriptionProperty", _IResolvable_a771d0ef]], jsii.get(self, "stageDescription"))

    @stage_description.setter
    def stage_description(
        self,
        value: typing.Optional[typing.Union["CfnDeployment.StageDescriptionProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ac5a15943a29de77767c1685c0aabb4111e1b234a8fe1b0eea5935d7cfefac5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageDescription", value)

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''A name for the stage that API Gateway creates with this deployment.

        Use only alphanumeric characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-stagename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @stage_name.setter
    def stage_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3153b644f4c6f881e7cde8031846531c26e785abe33df5b50c1dee1c10e70d1d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDeployment.AccessLogSettingProperty",
        jsii_struct_bases=[],
        name_mapping={"destination_arn": "destinationArn", "format": "format"},
    )
    class AccessLogSettingProperty:
        def __init__(
            self,
            *,
            destination_arn: typing.Optional[builtins.str] = None,
            format: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``AccessLogSetting`` property type specifies settings for logging access in this stage.

            ``AccessLogSetting`` is a property of the `StageDescription <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html>`_ property type.

            :param destination_arn: The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs. If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` .
            :param format: A single line format of the access logs of data, as specified by selected `$context variables <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference>`_ . The format must include at least ``$context.requestId`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-accesslogsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                access_log_setting_property = apigateway.CfnDeployment.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5787ad3e7d215f0d5485f5e4de4a769eabe35d6395451cb280a4870488b26d1d)
                check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination_arn is not None:
                self._values["destination_arn"] = destination_arn
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def destination_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs.

            If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-accesslogsetting.html#cfn-apigateway-deployment-accesslogsetting-destinationarn
            '''
            result = self._values.get("destination_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def format(self) -> typing.Optional[builtins.str]:
            '''A single line format of the access logs of data, as specified by selected `$context variables <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference>`_ . The format must include at least ``$context.requestId`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-accesslogsetting.html#cfn-apigateway-deployment-accesslogsetting-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDeployment.CanarySettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "percent_traffic": "percentTraffic",
            "stage_variable_overrides": "stageVariableOverrides",
            "use_stage_cache": "useStageCache",
        },
    )
    class CanarySettingProperty:
        def __init__(
            self,
            *,
            percent_traffic: typing.Optional[jsii.Number] = None,
            stage_variable_overrides: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            use_stage_cache: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The ``CanarySetting`` property type specifies settings for the canary deployment in this stage.

            ``CanarySetting`` is a property of the `StageDescription <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html>`_ property type.

            :param percent_traffic: The percent (0-100) of traffic diverted to a canary deployment.
            :param stage_variable_overrides: Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary. These stage variables are represented as a string-to-string map between stage variable names and their values.
            :param use_stage_cache: Whether the canary deployment uses the stage cache or not.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                canary_setting_property = apigateway.CfnDeployment.CanarySettingProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d550c07b5ceb2b97a7a7cc797389a03fdddafa96c9ea28afb4a754cf931875a4)
                check_type(argname="argument percent_traffic", value=percent_traffic, expected_type=type_hints["percent_traffic"])
                check_type(argname="argument stage_variable_overrides", value=stage_variable_overrides, expected_type=type_hints["stage_variable_overrides"])
                check_type(argname="argument use_stage_cache", value=use_stage_cache, expected_type=type_hints["use_stage_cache"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if percent_traffic is not None:
                self._values["percent_traffic"] = percent_traffic
            if stage_variable_overrides is not None:
                self._values["stage_variable_overrides"] = stage_variable_overrides
            if use_stage_cache is not None:
                self._values["use_stage_cache"] = use_stage_cache

        @builtins.property
        def percent_traffic(self) -> typing.Optional[jsii.Number]:
            '''The percent (0-100) of traffic diverted to a canary deployment.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html#cfn-apigateway-deployment-canarysetting-percenttraffic
            '''
            result = self._values.get("percent_traffic")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stage_variable_overrides(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary.

            These stage variables are represented as a string-to-string map between stage variable names and their values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html#cfn-apigateway-deployment-canarysetting-stagevariableoverrides
            '''
            result = self._values.get("stage_variable_overrides")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def use_stage_cache(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the canary deployment uses the stage cache or not.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html#cfn-apigateway-deployment-canarysetting-usestagecache
            '''
            result = self._values.get("use_stage_cache")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CanarySettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDeployment.DeploymentCanarySettingsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "percent_traffic": "percentTraffic",
            "stage_variable_overrides": "stageVariableOverrides",
            "use_stage_cache": "useStageCache",
        },
    )
    class DeploymentCanarySettingsProperty:
        def __init__(
            self,
            *,
            percent_traffic: typing.Optional[jsii.Number] = None,
            stage_variable_overrides: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            use_stage_cache: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The ``DeploymentCanarySettings`` property type specifies settings for the canary deployment.

            :param percent_traffic: The percentage (0-100) of traffic diverted to a canary deployment.
            :param stage_variable_overrides: Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary. These stage variables are represented as a string-to-string map between stage variable names and their values. Duplicates are not allowed.
            :param use_stage_cache: Whether the canary deployment uses the stage cache.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                deployment_canary_settings_property = apigateway.CfnDeployment.DeploymentCanarySettingsProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3ac03e50bc772886e27d7d6640f504abc92ba2cc4fd94b40aedc0c15ba3863d8)
                check_type(argname="argument percent_traffic", value=percent_traffic, expected_type=type_hints["percent_traffic"])
                check_type(argname="argument stage_variable_overrides", value=stage_variable_overrides, expected_type=type_hints["stage_variable_overrides"])
                check_type(argname="argument use_stage_cache", value=use_stage_cache, expected_type=type_hints["use_stage_cache"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if percent_traffic is not None:
                self._values["percent_traffic"] = percent_traffic
            if stage_variable_overrides is not None:
                self._values["stage_variable_overrides"] = stage_variable_overrides
            if use_stage_cache is not None:
                self._values["use_stage_cache"] = use_stage_cache

        @builtins.property
        def percent_traffic(self) -> typing.Optional[jsii.Number]:
            '''The percentage (0-100) of traffic diverted to a canary deployment.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html#cfn-apigateway-deployment-deploymentcanarysettings-percenttraffic
            '''
            result = self._values.get("percent_traffic")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stage_variable_overrides(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary.

            These stage variables are represented as a string-to-string map between stage variable names and their values.

            Duplicates are not allowed.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html#cfn-apigateway-deployment-deploymentcanarysettings-stagevariableoverrides
            '''
            result = self._values.get("stage_variable_overrides")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def use_stage_cache(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the canary deployment uses the stage cache.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html#cfn-apigateway-deployment-deploymentcanarysettings-usestagecache
            '''
            result = self._values.get("use_stage_cache")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentCanarySettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDeployment.MethodSettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cache_data_encrypted": "cacheDataEncrypted",
            "cache_ttl_in_seconds": "cacheTtlInSeconds",
            "caching_enabled": "cachingEnabled",
            "data_trace_enabled": "dataTraceEnabled",
            "http_method": "httpMethod",
            "logging_level": "loggingLevel",
            "metrics_enabled": "metricsEnabled",
            "resource_path": "resourcePath",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
        },
    )
    class MethodSettingProperty:
        def __init__(
            self,
            *,
            cache_data_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            cache_ttl_in_seconds: typing.Optional[jsii.Number] = None,
            caching_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            http_method: typing.Optional[builtins.str] = None,
            logging_level: typing.Optional[builtins.str] = None,
            metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            resource_path: typing.Optional[builtins.str] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``MethodSetting`` property type configures settings for all methods in a stage.

            The ``MethodSettings`` property of the `Amazon API Gateway Deployment StageDescription <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html>`_ property type contains a list of ``MethodSetting`` property types.

            :param cache_data_encrypted: Indicates whether the cached responses are encrypted.
            :param cache_ttl_in_seconds: The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.
            :param caching_enabled: Indicates whether responses are cached and returned for requests. You must enable a cache cluster on the stage to cache responses. For more information, see `Enable API Gateway Caching in a Stage to Enhance API Performance <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ in the *API Gateway Developer Guide* .
            :param data_trace_enabled: Indicates whether data trace logging is enabled for methods in the stage. API Gateway pushes these logs to Amazon CloudWatch Logs.
            :param http_method: The HTTP method.
            :param logging_level: The logging level for this method. For valid values, see the ``loggingLevel`` property of the `Stage <https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html#loggingLevel>`_ resource in the *Amazon API Gateway API Reference* .
            :param metrics_enabled: Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.
            :param resource_path: The resource path for this method. Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ).
            :param throttling_burst_limit: The number of burst requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account . For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param throttling_rate_limit: The number of steady-state requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account . For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                method_setting_property = apigateway.CfnDeployment.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5bf36568151f4876389a7d0c74c26c2f4e984b92d65b5365bf278d5aaafca4d2)
                check_type(argname="argument cache_data_encrypted", value=cache_data_encrypted, expected_type=type_hints["cache_data_encrypted"])
                check_type(argname="argument cache_ttl_in_seconds", value=cache_ttl_in_seconds, expected_type=type_hints["cache_ttl_in_seconds"])
                check_type(argname="argument caching_enabled", value=caching_enabled, expected_type=type_hints["caching_enabled"])
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument metrics_enabled", value=metrics_enabled, expected_type=type_hints["metrics_enabled"])
                check_type(argname="argument resource_path", value=resource_path, expected_type=type_hints["resource_path"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cache_data_encrypted is not None:
                self._values["cache_data_encrypted"] = cache_data_encrypted
            if cache_ttl_in_seconds is not None:
                self._values["cache_ttl_in_seconds"] = cache_ttl_in_seconds
            if caching_enabled is not None:
                self._values["caching_enabled"] = caching_enabled
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if http_method is not None:
                self._values["http_method"] = http_method
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if metrics_enabled is not None:
                self._values["metrics_enabled"] = metrics_enabled
            if resource_path is not None:
                self._values["resource_path"] = resource_path
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit

        @builtins.property
        def cache_data_encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether the cached responses are encrypted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-cachedataencrypted
            '''
            result = self._values.get("cache_data_encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def cache_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
            '''The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-cachettlinseconds
            '''
            result = self._values.get("cache_ttl_in_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def caching_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether responses are cached and returned for requests.

            You must enable a cache cluster on the stage to cache responses. For more information, see `Enable API Gateway Caching in a Stage to Enhance API Performance <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-cachingenabled
            '''
            result = self._values.get("caching_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether data trace logging is enabled for methods in the stage.

            API Gateway pushes these logs to Amazon CloudWatch Logs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-datatraceenabled
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def http_method(self) -> typing.Optional[builtins.str]:
            '''The HTTP method.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-httpmethod
            '''
            result = self._values.get("http_method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''The logging level for this method.

            For valid values, see the ``loggingLevel`` property of the `Stage <https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html#loggingLevel>`_ resource in the *Amazon API Gateway API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-logginglevel
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-metricsenabled
            '''
            result = self._values.get("metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def resource_path(self) -> typing.Optional[builtins.str]:
            '''The resource path for this method.

            Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-resourcepath
            '''
            result = self._values.get("resource_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''The number of burst requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account .

            For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-throttlingburstlimit
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The number of steady-state requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account .

            For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-throttlingratelimit
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MethodSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDeployment.StageDescriptionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "access_log_setting": "accessLogSetting",
            "cache_cluster_enabled": "cacheClusterEnabled",
            "cache_cluster_size": "cacheClusterSize",
            "cache_data_encrypted": "cacheDataEncrypted",
            "cache_ttl_in_seconds": "cacheTtlInSeconds",
            "caching_enabled": "cachingEnabled",
            "canary_setting": "canarySetting",
            "client_certificate_id": "clientCertificateId",
            "data_trace_enabled": "dataTraceEnabled",
            "description": "description",
            "documentation_version": "documentationVersion",
            "logging_level": "loggingLevel",
            "method_settings": "methodSettings",
            "metrics_enabled": "metricsEnabled",
            "tags": "tags",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
            "tracing_enabled": "tracingEnabled",
            "variables": "variables",
        },
    )
    class StageDescriptionProperty:
        def __init__(
            self,
            *,
            access_log_setting: typing.Optional[typing.Union[typing.Union["CfnDeployment.AccessLogSettingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            cache_cluster_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            cache_cluster_size: typing.Optional[builtins.str] = None,
            cache_data_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            cache_ttl_in_seconds: typing.Optional[jsii.Number] = None,
            caching_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            canary_setting: typing.Optional[typing.Union[typing.Union["CfnDeployment.CanarySettingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
            client_certificate_id: typing.Optional[builtins.str] = None,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            description: typing.Optional[builtins.str] = None,
            documentation_version: typing.Optional[builtins.str] = None,
            logging_level: typing.Optional[builtins.str] = None,
            method_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnDeployment.MethodSettingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
            tracing_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            variables: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        ) -> None:
            '''``StageDescription`` is a property of the `AWS::ApiGateway::Deployment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html>`_ resource that configures a deployment stage.

            :param access_log_setting: Specifies settings for logging access in this stage.
            :param cache_cluster_enabled: Indicates whether cache clustering is enabled for the stage.
            :param cache_cluster_size: The size of the stage's cache cluster.
            :param cache_data_encrypted: Indicates whether the cached responses are encrypted.
            :param cache_ttl_in_seconds: The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.
            :param caching_enabled: Indicates whether responses are cached and returned for requests. You must enable a cache cluster on the stage to cache responses. For more information, see `Enable API Gateway Caching in a Stage to Enhance API Performance <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ in the *API Gateway Developer Guide* .
            :param canary_setting: Specifies settings for the canary deployment in this stage.
            :param client_certificate_id: The identifier of the client certificate that API Gateway uses to call your integration endpoints in the stage.
            :param data_trace_enabled: Indicates whether data trace logging is enabled for methods in the stage. API Gateway pushes these logs to Amazon CloudWatch Logs.
            :param description: A description of the purpose of the stage.
            :param documentation_version: The version identifier of the API documentation snapshot.
            :param logging_level: The logging level for this method. For valid values, see the ``loggingLevel`` property of the `Stage <https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html#loggingLevel>`_ resource in the *Amazon API Gateway API Reference* .
            :param method_settings: Configures settings for all of the stage's methods.
            :param metrics_enabled: Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.
            :param tags: An array of arbitrary tags (key-value pairs) to associate with the stage.
            :param throttling_burst_limit: The target request burst rate limit. This allows more requests through for a period of time than the target rate limit. For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param throttling_rate_limit: The target request steady-state rate limit. For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param tracing_enabled: Specifies whether active tracing with X-ray is enabled for this stage. For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .
            :param variables: A map that defines the stage variables. Variable names must consist of alphanumeric characters, and the values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                stage_description_property = apigateway.CfnDeployment.StageDescriptionProperty(
                    access_log_setting=apigateway.CfnDeployment.AccessLogSettingProperty(
                        destination_arn="destinationArn",
                        format="format"
                    ),
                    cache_cluster_enabled=False,
                    cache_cluster_size="cacheClusterSize",
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    canary_setting=apigateway.CfnDeployment.CanarySettingProperty(
                        percent_traffic=123,
                        stage_variable_overrides={
                            "stage_variable_overrides_key": "stageVariableOverrides"
                        },
                        use_stage_cache=False
                    ),
                    client_certificate_id="clientCertificateId",
                    data_trace_enabled=False,
                    description="description",
                    documentation_version="documentationVersion",
                    logging_level="loggingLevel",
                    method_settings=[apigateway.CfnDeployment.MethodSettingProperty(
                        cache_data_encrypted=False,
                        cache_ttl_in_seconds=123,
                        caching_enabled=False,
                        data_trace_enabled=False,
                        http_method="httpMethod",
                        logging_level="loggingLevel",
                        metrics_enabled=False,
                        resource_path="resourcePath",
                        throttling_burst_limit=123,
                        throttling_rate_limit=123
                    )],
                    metrics_enabled=False,
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )],
                    throttling_burst_limit=123,
                    throttling_rate_limit=123,
                    tracing_enabled=False,
                    variables={
                        "variables_key": "variables"
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e329089f2cf1bdd2f3b8687af01d6ee5894d0141f324b7fcdf897d14065688e2)
                check_type(argname="argument access_log_setting", value=access_log_setting, expected_type=type_hints["access_log_setting"])
                check_type(argname="argument cache_cluster_enabled", value=cache_cluster_enabled, expected_type=type_hints["cache_cluster_enabled"])
                check_type(argname="argument cache_cluster_size", value=cache_cluster_size, expected_type=type_hints["cache_cluster_size"])
                check_type(argname="argument cache_data_encrypted", value=cache_data_encrypted, expected_type=type_hints["cache_data_encrypted"])
                check_type(argname="argument cache_ttl_in_seconds", value=cache_ttl_in_seconds, expected_type=type_hints["cache_ttl_in_seconds"])
                check_type(argname="argument caching_enabled", value=caching_enabled, expected_type=type_hints["caching_enabled"])
                check_type(argname="argument canary_setting", value=canary_setting, expected_type=type_hints["canary_setting"])
                check_type(argname="argument client_certificate_id", value=client_certificate_id, expected_type=type_hints["client_certificate_id"])
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument documentation_version", value=documentation_version, expected_type=type_hints["documentation_version"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument method_settings", value=method_settings, expected_type=type_hints["method_settings"])
                check_type(argname="argument metrics_enabled", value=metrics_enabled, expected_type=type_hints["metrics_enabled"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
                check_type(argname="argument tracing_enabled", value=tracing_enabled, expected_type=type_hints["tracing_enabled"])
                check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_log_setting is not None:
                self._values["access_log_setting"] = access_log_setting
            if cache_cluster_enabled is not None:
                self._values["cache_cluster_enabled"] = cache_cluster_enabled
            if cache_cluster_size is not None:
                self._values["cache_cluster_size"] = cache_cluster_size
            if cache_data_encrypted is not None:
                self._values["cache_data_encrypted"] = cache_data_encrypted
            if cache_ttl_in_seconds is not None:
                self._values["cache_ttl_in_seconds"] = cache_ttl_in_seconds
            if caching_enabled is not None:
                self._values["caching_enabled"] = caching_enabled
            if canary_setting is not None:
                self._values["canary_setting"] = canary_setting
            if client_certificate_id is not None:
                self._values["client_certificate_id"] = client_certificate_id
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if description is not None:
                self._values["description"] = description
            if documentation_version is not None:
                self._values["documentation_version"] = documentation_version
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if method_settings is not None:
                self._values["method_settings"] = method_settings
            if metrics_enabled is not None:
                self._values["metrics_enabled"] = metrics_enabled
            if tags is not None:
                self._values["tags"] = tags
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit
            if tracing_enabled is not None:
                self._values["tracing_enabled"] = tracing_enabled
            if variables is not None:
                self._values["variables"] = variables

        @builtins.property
        def access_log_setting(
            self,
        ) -> typing.Optional[typing.Union["CfnDeployment.AccessLogSettingProperty", _IResolvable_a771d0ef]]:
            '''Specifies settings for logging access in this stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-accesslogsetting
            '''
            result = self._values.get("access_log_setting")
            return typing.cast(typing.Optional[typing.Union["CfnDeployment.AccessLogSettingProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def cache_cluster_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether cache clustering is enabled for the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cacheclusterenabled
            '''
            result = self._values.get("cache_cluster_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def cache_cluster_size(self) -> typing.Optional[builtins.str]:
            '''The size of the stage's cache cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cacheclustersize
            '''
            result = self._values.get("cache_cluster_size")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cache_data_encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether the cached responses are encrypted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cachedataencrypted
            '''
            result = self._values.get("cache_data_encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def cache_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
            '''The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cachettlinseconds
            '''
            result = self._values.get("cache_ttl_in_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def caching_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether responses are cached and returned for requests.

            You must enable a cache cluster on the stage to cache responses. For more information, see `Enable API Gateway Caching in a Stage to Enhance API Performance <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cachingenabled
            '''
            result = self._values.get("caching_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def canary_setting(
            self,
        ) -> typing.Optional[typing.Union["CfnDeployment.CanarySettingProperty", _IResolvable_a771d0ef]]:
            '''Specifies settings for the canary deployment in this stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-canarysetting
            '''
            result = self._values.get("canary_setting")
            return typing.cast(typing.Optional[typing.Union["CfnDeployment.CanarySettingProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def client_certificate_id(self) -> typing.Optional[builtins.str]:
            '''The identifier of the client certificate that API Gateway uses to call your integration endpoints in the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-clientcertificateid
            '''
            result = self._values.get("client_certificate_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether data trace logging is enabled for methods in the stage.

            API Gateway pushes these logs to Amazon CloudWatch Logs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-datatraceenabled
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A description of the purpose of the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def documentation_version(self) -> typing.Optional[builtins.str]:
            '''The version identifier of the API documentation snapshot.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-documentationversion
            '''
            result = self._values.get("documentation_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''The logging level for this method.

            For valid values, see the ``loggingLevel`` property of the `Stage <https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html#loggingLevel>`_ resource in the *Amazon API Gateway API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-logginglevel
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def method_settings(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnDeployment.MethodSettingProperty", _IResolvable_a771d0ef]]]]:
            '''Configures settings for all of the stage's methods.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-methodsettings
            '''
            result = self._values.get("method_settings")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnDeployment.MethodSettingProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-metricsenabled
            '''
            result = self._values.get("metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
            '''An array of arbitrary tags (key-value pairs) to associate with the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''The target request burst rate limit.

            This allows more requests through for a period of time than the target rate limit. For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-throttlingburstlimit
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The target request steady-state rate limit.

            For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-throttlingratelimit
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tracing_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specifies whether active tracing with X-ray is enabled for this stage.

            For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-tracingenabled
            '''
            result = self._values.get("tracing_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def variables(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''A map that defines the stage variables.

            Variable names must consist of alphanumeric characters, and the values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-variables
            '''
            result = self._values.get("variables")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StageDescriptionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnDeploymentProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "deployment_canary_settings": "deploymentCanarySettings",
        "description": "description",
        "stage_description": "stageDescription",
        "stage_name": "stageName",
    },
)
class CfnDeploymentProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        deployment_canary_settings: typing.Optional[typing.Union[typing.Union[CfnDeployment.DeploymentCanarySettingsProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        description: typing.Optional[builtins.str] = None,
        stage_description: typing.Optional[typing.Union[typing.Union[CfnDeployment.StageDescriptionProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnDeployment``.

        :param rest_api_id: The ID of the ``RestApi`` resource to deploy.
        :param deployment_canary_settings: Specifies settings for the canary deployment.
        :param description: A description of the purpose of the API Gateway deployment.
        :param stage_description: Configures the stage that API Gateway creates with this deployment.
        :param stage_name: A name for the stage that API Gateway creates with this deployment. Use only alphanumeric characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_deployment_props = apigateway.CfnDeploymentProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                deployment_canary_settings=apigateway.CfnDeployment.DeploymentCanarySettingsProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                ),
                description="description",
                stage_description=apigateway.CfnDeployment.StageDescriptionProperty(
                    access_log_setting=apigateway.CfnDeployment.AccessLogSettingProperty(
                        destination_arn="destinationArn",
                        format="format"
                    ),
                    cache_cluster_enabled=False,
                    cache_cluster_size="cacheClusterSize",
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    canary_setting=apigateway.CfnDeployment.CanarySettingProperty(
                        percent_traffic=123,
                        stage_variable_overrides={
                            "stage_variable_overrides_key": "stageVariableOverrides"
                        },
                        use_stage_cache=False
                    ),
                    client_certificate_id="clientCertificateId",
                    data_trace_enabled=False,
                    description="description",
                    documentation_version="documentationVersion",
                    logging_level="loggingLevel",
                    method_settings=[apigateway.CfnDeployment.MethodSettingProperty(
                        cache_data_encrypted=False,
                        cache_ttl_in_seconds=123,
                        caching_enabled=False,
                        data_trace_enabled=False,
                        http_method="httpMethod",
                        logging_level="loggingLevel",
                        metrics_enabled=False,
                        resource_path="resourcePath",
                        throttling_burst_limit=123,
                        throttling_rate_limit=123
                    )],
                    metrics_enabled=False,
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )],
                    throttling_burst_limit=123,
                    throttling_rate_limit=123,
                    tracing_enabled=False,
                    variables={
                        "variables_key": "variables"
                    }
                ),
                stage_name="stageName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0e4670b407f041f3e584e5379ccc097084ad9b9bd6a97b41065948920bda299)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument deployment_canary_settings", value=deployment_canary_settings, expected_type=type_hints["deployment_canary_settings"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument stage_description", value=stage_description, expected_type=type_hints["stage_description"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if deployment_canary_settings is not None:
            self._values["deployment_canary_settings"] = deployment_canary_settings
        if description is not None:
            self._values["description"] = description
        if stage_description is not None:
            self._values["stage_description"] = stage_description
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The ID of the ``RestApi`` resource to deploy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deployment_canary_settings(
        self,
    ) -> typing.Optional[typing.Union[CfnDeployment.DeploymentCanarySettingsProperty, _IResolvable_a771d0ef]]:
        '''Specifies settings for the canary deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-deploymentcanarysettings
        '''
        result = self._values.get("deployment_canary_settings")
        return typing.cast(typing.Optional[typing.Union[CfnDeployment.DeploymentCanarySettingsProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the purpose of the API Gateway deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage_description(
        self,
    ) -> typing.Optional[typing.Union[CfnDeployment.StageDescriptionProperty, _IResolvable_a771d0ef]]:
        '''Configures the stage that API Gateway creates with this deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-stagedescription
        '''
        result = self._values.get("stage_description")
        return typing.cast(typing.Optional[typing.Union[CfnDeployment.StageDescriptionProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''A name for the stage that API Gateway creates with this deployment.

        Use only alphanumeric characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-stagename
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDeploymentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnDeploymentV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnDeploymentV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Deployment``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Deployment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_deployment_v2 = apigateway.CfnDeploymentV2(self, "MyCfnDeploymentV2",
            api_id="apiId",
        
            # the properties below are optional
            description="description",
            stage_name="stageName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Deployment``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Deployment.ApiId``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Deployment.Description``.
        :param stage_name: (deprecated) ``AWS::ApiGatewayV2::Deployment.StageName``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__993e8b713195bb9d12a2c6d3ab7259da4d13ac34924c1577ef1e00eb71e010c5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDeploymentV2Props(
            api_id=api_id, description=description, stage_name=stage_name
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7494e804ef32b229e598cda382606e87d03895d51c3c746fa1193319f78febcc)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3924505dfb309cba7ba388b121c6fe3b15aff2fe9073d1a1d2dc28a3fc617880)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Deployment.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html#cfn-apigatewayv2-deployment-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4239102686161dea651166a1e67b5655ef16a9c3df5ffa21771479c74df7524)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Deployment.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html#cfn-apigatewayv2-deployment-description
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb0358751eac06aef731e85e6ed69d2ee6a469f311017345f6b58f23152f20f3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Deployment.StageName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html#cfn-apigatewayv2-deployment-stagename
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @stage_name.setter
    def stage_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29c46cdd8a8305cf0457432c8dd415e6d1db31f39ad2f67d3b2e4172023b0081)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageName", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnDeploymentV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "description": "description",
        "stage_name": "stageName",
    },
)
class CfnDeploymentV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Deployment``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Deployment.ApiId``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Deployment.Description``.
        :param stage_name: (deprecated) ``AWS::ApiGatewayV2::Deployment.StageName``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_deployment_v2_props = apigateway.CfnDeploymentV2Props(
                api_id="apiId",
            
                # the properties below are optional
                description="description",
                stage_name="stageName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a308192c0e7813a3b211ad9a83cb584b58a908c9c084ad83c370841fce50f15c)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
        }
        if description is not None:
            self._values["description"] = description
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Deployment.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html#cfn-apigatewayv2-deployment-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Deployment.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html#cfn-apigatewayv2-deployment-description
        :stability: deprecated
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Deployment.StageName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-deployment.html#cfn-apigatewayv2-deployment-stagename
        :stability: deprecated
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDeploymentV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnDocumentationPart(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnDocumentationPart",
):
    '''A CloudFormation ``AWS::ApiGateway::DocumentationPart``.

    The ``AWS::ApiGateway::DocumentationPart`` resource creates a documentation part for an API. For more information, see `Representation of API Documentation in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-documenting-api-content-representation.html>`_ in the *API Gateway Developer Guide* .

    :cloudformationResource: AWS::ApiGateway::DocumentationPart
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_documentation_part = apigateway.CfnDocumentationPart(self, "MyCfnDocumentationPart",
            location=apigateway.CfnDocumentationPart.LocationProperty(
                method="method",
                name="name",
                path="path",
                status_code="statusCode",
                type="type"
            ),
            properties="properties",
            rest_api_id="restApiId"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        location: typing.Union[typing.Union["CfnDocumentationPart.LocationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        properties: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::DocumentationPart``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param location: The location of the API entity that the documentation applies to.
        :param properties: The documentation content map of the targeted API entity.
        :param rest_api_id: The identifier of the targeted API entity.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6116f121c34f4c1d282b2c4fa18e4e6f5b4866d0055da2f0cd8c1ec690089dba)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDocumentationPartProps(
            location=location, properties=properties, rest_api_id=rest_api_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bd375f8ec1bfac24f201033d052160c7a04ba0fe23eb4b5bc41456d7f15a2722)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42ac3e3cd08383863c29a549fba4285daf91c3fcc1b64dd48755779c990db570)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDocumentationPartId")
    def attr_documentation_part_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: DocumentationPartId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDocumentationPartId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="location")
    def location(
        self,
    ) -> typing.Union["CfnDocumentationPart.LocationProperty", _IResolvable_a771d0ef]:
        '''The location of the API entity that the documentation applies to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-location
        '''
        return typing.cast(typing.Union["CfnDocumentationPart.LocationProperty", _IResolvable_a771d0ef], jsii.get(self, "location"))

    @location.setter
    def location(
        self,
        value: typing.Union["CfnDocumentationPart.LocationProperty", _IResolvable_a771d0ef],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48f5b1786490eb81a1b4fd737c0137cac39c6aca979ec65c74a2a441367c30db)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "location", value)

    @builtins.property
    @jsii.member(jsii_name="properties")
    def properties(self) -> builtins.str:
        '''The documentation content map of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-properties
        '''
        return typing.cast(builtins.str, jsii.get(self, "properties"))

    @properties.setter
    def properties(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e54ce96fbd07312db8c0631477dc2f77d5251761a2b707713a1697cf31e0fab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "properties", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__225b479260e96b034d2cc4ab9b7e0cc15f7d8630f8b90dbc3f6a017ab98ed786)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDocumentationPart.LocationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "method": "method",
            "name": "name",
            "path": "path",
            "status_code": "statusCode",
            "type": "type",
        },
    )
    class LocationProperty:
        def __init__(
            self,
            *,
            method: typing.Optional[builtins.str] = None,
            name: typing.Optional[builtins.str] = None,
            path: typing.Optional[builtins.str] = None,
            status_code: typing.Optional[builtins.str] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Location`` property specifies the location of the Amazon API Gateway API entity that the documentation applies to.

            ``Location`` is a property of the `AWS::ApiGateway::DocumentationPart <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html>`_ resource.
            .. epigraph::

               For more information about each property, including constraints and valid values, see `DocumentationPart <https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationPartLocation.html>`_ in the *Amazon API Gateway REST API Reference* .

            :param method: The HTTP verb of a method.
            :param name: The name of the targeted API entity.
            :param path: The URL path of the target.
            :param status_code: The HTTP status code of a response.
            :param type: The type of API entity that the documentation content applies to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                location_property = apigateway.CfnDocumentationPart.LocationProperty(
                    method="method",
                    name="name",
                    path="path",
                    status_code="statusCode",
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__876b01f833ce4a0b45bc5eb8e638e9b306fc05131b8b1424a4d3df72d0b29be2)
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if method is not None:
                self._values["method"] = method
            if name is not None:
                self._values["name"] = name
            if path is not None:
                self._values["path"] = path
            if status_code is not None:
                self._values["status_code"] = status_code
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The HTTP verb of a method.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the targeted API entity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The URL path of the target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def status_code(self) -> typing.Optional[builtins.str]:
            '''The HTTP status code of a response.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-statuscode
            '''
            result = self._values.get("status_code")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The type of API entity that the documentation content applies to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnDocumentationPartProps",
    jsii_struct_bases=[],
    name_mapping={
        "location": "location",
        "properties": "properties",
        "rest_api_id": "restApiId",
    },
)
class CfnDocumentationPartProps:
    def __init__(
        self,
        *,
        location: typing.Union[typing.Union[CfnDocumentationPart.LocationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef],
        properties: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnDocumentationPart``.

        :param location: The location of the API entity that the documentation applies to.
        :param properties: The documentation content map of the targeted API entity.
        :param rest_api_id: The identifier of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_documentation_part_props = apigateway.CfnDocumentationPartProps(
                location=apigateway.CfnDocumentationPart.LocationProperty(
                    method="method",
                    name="name",
                    path="path",
                    status_code="statusCode",
                    type="type"
                ),
                properties="properties",
                rest_api_id="restApiId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e7e65adb9b4d68726aa44269e7bdb4f3bbec96223ec20c93d34754a7bd43d25)
            check_type(argname="argument location", value=location, expected_type=type_hints["location"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "location": location,
            "properties": properties,
            "rest_api_id": rest_api_id,
        }

    @builtins.property
    def location(
        self,
    ) -> typing.Union[CfnDocumentationPart.LocationProperty, _IResolvable_a771d0ef]:
        '''The location of the API entity that the documentation applies to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-location
        '''
        result = self._values.get("location")
        assert result is not None, "Required property 'location' is missing"
        return typing.cast(typing.Union[CfnDocumentationPart.LocationProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def properties(self) -> builtins.str:
        '''The documentation content map of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-properties
        '''
        result = self._values.get("properties")
        assert result is not None, "Required property 'properties' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDocumentationPartProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnDocumentationVersion(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnDocumentationVersion",
):
    '''A CloudFormation ``AWS::ApiGateway::DocumentationVersion``.

    The ``AWS::ApiGateway::DocumentationVersion`` resource creates a snapshot of the documentation for an API. For more information, see `Representation of API Documentation in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-documenting-api-content-representation.html>`_ in the *API Gateway Developer Guide* .

    :cloudformationResource: AWS::ApiGateway::DocumentationVersion
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_documentation_version = apigateway.CfnDocumentationVersion(self, "MyCfnDocumentationVersion",
            documentation_version="documentationVersion",
            rest_api_id="restApiId",
        
            # the properties below are optional
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        documentation_version: builtins.str,
        rest_api_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::DocumentationVersion``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param documentation_version: The version identifier of the API documentation snapshot.
        :param rest_api_id: The identifier of the API.
        :param description: The description of the API documentation snapshot.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0debb164b3328238345d831c80367b11ffd2b213dd81b1dc2a00b6844d21ff48)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDocumentationVersionProps(
            documentation_version=documentation_version,
            rest_api_id=rest_api_id,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0703fe4730cce4e9b1051eb66e51f9c6dc18bf90735f430f80abfe34fe592f17)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__efec204f5610b7b019e171b1ef614ec04e25ce1a567d17bff730481d424a1127)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="documentationVersion")
    def documentation_version(self) -> builtins.str:
        '''The version identifier of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-documentationversion
        '''
        return typing.cast(builtins.str, jsii.get(self, "documentationVersion"))

    @documentation_version.setter
    def documentation_version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76bd0bdc9c3cce701be2b7e44e8fcf49b8fcef9ff9845cf3c8875b5fa5bd7999)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "documentationVersion", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4f692a6f6092a9f8f1bcfc64f3b72e367dcbe5064e84ae542c1ab98e34af722)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d72c090c55acb20c0d698605c28bbb74d2f1c3de6c7b23677caa1eb9b8968e1f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnDocumentationVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "documentation_version": "documentationVersion",
        "rest_api_id": "restApiId",
        "description": "description",
    },
)
class CfnDocumentationVersionProps:
    def __init__(
        self,
        *,
        documentation_version: builtins.str,
        rest_api_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnDocumentationVersion``.

        :param documentation_version: The version identifier of the API documentation snapshot.
        :param rest_api_id: The identifier of the API.
        :param description: The description of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_documentation_version_props = apigateway.CfnDocumentationVersionProps(
                documentation_version="documentationVersion",
                rest_api_id="restApiId",
            
                # the properties below are optional
                description="description"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a083c70d554477f414308f1eb44fb0ff662070962f8c07ef74a7f84fab2e3c51)
            check_type(argname="argument documentation_version", value=documentation_version, expected_type=type_hints["documentation_version"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "documentation_version": documentation_version,
            "rest_api_id": rest_api_id,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def documentation_version(self) -> builtins.str:
        '''The version identifier of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-documentationversion
        '''
        result = self._values.get("documentation_version")
        assert result is not None, "Required property 'documentation_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDocumentationVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnDomainName(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnDomainName",
):
    '''A CloudFormation ``AWS::ApiGateway::DomainName``.

    The ``AWS::ApiGateway::DomainName`` resource specifies a custom domain name for your API in API Gateway.

    You can use a custom domain name to provide a URL that's more intuitive and easier to recall. For more information about using custom domain names, see `Set up Custom Domain Name for an API in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html>`_ in the *API Gateway Developer Guide* .

    :cloudformationResource: AWS::ApiGateway::DomainName
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_domain_name = apigateway.CfnDomainName(self, "MyCfnDomainName",
            certificate_arn="certificateArn",
            domain_name="domainName",
            endpoint_configuration=apigateway.CfnDomainName.EndpointConfigurationProperty(
                types=["types"]
            ),
            mutual_tls_authentication=apigateway.CfnDomainName.MutualTlsAuthenticationProperty(
                truststore_uri="truststoreUri",
                truststore_version="truststoreVersion"
            ),
            ownership_verification_certificate_arn="ownershipVerificationCertificateArn",
            regional_certificate_arn="regionalCertificateArn",
            security_policy="securityPolicy",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        certificate_arn: typing.Optional[builtins.str] = None,
        domain_name: typing.Optional[builtins.str] = None,
        endpoint_configuration: typing.Optional[typing.Union[typing.Union["CfnDomainName.EndpointConfigurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        mutual_tls_authentication: typing.Optional[typing.Union[typing.Union["CfnDomainName.MutualTlsAuthenticationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ownership_verification_certificate_arn: typing.Optional[builtins.str] = None,
        regional_certificate_arn: typing.Optional[builtins.str] = None,
        security_policy: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::DomainName``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param certificate_arn: The reference to an AWS -managed certificate for use by the edge-optimized endpoint for this domain name. AWS Certificate Manager is the only supported source. For requirements and additional information about setting up certificates, see `Get Certificates Ready in AWS Certificate Manager <https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html#how-to-custom-domains-prerequisites>`_ in the *API Gateway Developer Guide* .
        :param domain_name: The custom domain name for your API. Uppercase letters are not supported.
        :param endpoint_configuration: A list of the endpoint types of the domain name.
        :param mutual_tls_authentication: The mutual TLS authentication configuration for a custom domain name.
        :param ownership_verification_certificate_arn: The ARN of the public certificate issued by ACM to validate ownership of your custom domain. Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.
        :param regional_certificate_arn: The reference to an AWS -managed certificate for use by the regional endpoint for the domain name. AWS Certificate Manager is the only supported source.
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this domain name. Valid values include ``TLS_1_0`` and ``TLS_1_2`` .
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the domain name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bde87b5fdca7f7aaaaf84e56066398a7fe3551b04d124044eded860b96038662)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDomainNameProps(
            certificate_arn=certificate_arn,
            domain_name=domain_name,
            endpoint_configuration=endpoint_configuration,
            mutual_tls_authentication=mutual_tls_authentication,
            ownership_verification_certificate_arn=ownership_verification_certificate_arn,
            regional_certificate_arn=regional_certificate_arn,
            security_policy=security_policy,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8809be9546a7c3f422fd3bcee0df08bbfd1daadbd88ee53a2dad5ad11ae066f)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09c0e7bba40e0b8bf19fbcb45ae98f42e984e6abc07fcf01ed0e514247f606a9)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDistributionDomainName")
    def attr_distribution_domain_name(self) -> builtins.str:
        '''The Amazon CloudFront distribution domain name that's mapped to the custom domain name.

        This is only applicable for endpoints whose type is ``EDGE`` .

        Example: ``d111111abcdef8.cloudfront.net``

        :cloudformationAttribute: DistributionDomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDistributionDomainName"))

    @builtins.property
    @jsii.member(jsii_name="attrDistributionHostedZoneId")
    def attr_distribution_hosted_zone_id(self) -> builtins.str:
        '''The region-agnostic Amazon Route 53 Hosted Zone ID of the edge-optimized endpoint.

        The only valid value is ``Z2FDTNDATAQYW2`` for all regions.

        :cloudformationAttribute: DistributionHostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDistributionHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="attrRegionalDomainName")
    def attr_regional_domain_name(self) -> builtins.str:
        '''The domain name associated with the regional endpoint for this custom domain name.

        You set up this association by adding a DNS record that points the custom domain name to this regional domain name.

        :cloudformationAttribute: RegionalDomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRegionalDomainName"))

    @builtins.property
    @jsii.member(jsii_name="attrRegionalHostedZoneId")
    def attr_regional_hosted_zone_id(self) -> builtins.str:
        '''The region-specific Amazon Route 53 Hosted Zone ID of the regional endpoint.

        :cloudformationAttribute: RegionalHostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRegionalHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate for use by the edge-optimized endpoint for this domain name.

        AWS Certificate Manager is the only supported source. For requirements and additional information about setting up certificates, see `Get Certificates Ready in AWS Certificate Manager <https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html#how-to-custom-domains-prerequisites>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-certificatearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf8d918eb0d249d7c53e843c51a3ed111bed03a9dbd7635e80d0a2b4a4be6710)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateArn", value)

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''The custom domain name for your API.

        Uppercase letters are not supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-domainname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d982bfb3f940709ec82fe80dea8108b2513a89a3fd63ab842f420056b212134)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value)

    @builtins.property
    @jsii.member(jsii_name="endpointConfiguration")
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnDomainName.EndpointConfigurationProperty", _IResolvable_a771d0ef]]:
        '''A list of the endpoint types of the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-endpointconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnDomainName.EndpointConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "endpointConfiguration"))

    @endpoint_configuration.setter
    def endpoint_configuration(
        self,
        value: typing.Optional[typing.Union["CfnDomainName.EndpointConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1efb8ce0ddfdb61065a407d2d01d8f402efc12ced1c4d1a5e80fc26fcbf1fd56)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpointConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="mutualTlsAuthentication")
    def mutual_tls_authentication(
        self,
    ) -> typing.Optional[typing.Union["CfnDomainName.MutualTlsAuthenticationProperty", _IResolvable_a771d0ef]]:
        '''The mutual TLS authentication configuration for a custom domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-mutualtlsauthentication
        '''
        return typing.cast(typing.Optional[typing.Union["CfnDomainName.MutualTlsAuthenticationProperty", _IResolvable_a771d0ef]], jsii.get(self, "mutualTlsAuthentication"))

    @mutual_tls_authentication.setter
    def mutual_tls_authentication(
        self,
        value: typing.Optional[typing.Union["CfnDomainName.MutualTlsAuthenticationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d838d49dd2ac0253c21ef38f20acb4b68859d2bff39bdc571dc05142f8d722e6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "mutualTlsAuthentication", value)

    @builtins.property
    @jsii.member(jsii_name="ownershipVerificationCertificateArn")
    def ownership_verification_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the public certificate issued by ACM to validate ownership of your custom domain.

        Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-ownershipverificationcertificatearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ownershipVerificationCertificateArn"))

    @ownership_verification_certificate_arn.setter
    def ownership_verification_certificate_arn(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2160fd45324c43cb71d67d0b2e404f0965aa63fca41beaa2d7a4e8e27617c87a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ownershipVerificationCertificateArn", value)

    @builtins.property
    @jsii.member(jsii_name="regionalCertificateArn")
    def regional_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate for use by the regional endpoint for the domain name.

        AWS Certificate Manager is the only supported source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-regionalcertificatearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "regionalCertificateArn"))

    @regional_certificate_arn.setter
    def regional_certificate_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ebe873c0b4c8808a2949f4e04bf08074096067518e98ea67936ebfd0fa232995)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "regionalCertificateArn", value)

    @builtins.property
    @jsii.member(jsii_name="securityPolicy")
    def security_policy(self) -> typing.Optional[builtins.str]:
        '''The Transport Layer Security (TLS) version + cipher suite for this domain name.

        Valid values include ``TLS_1_0`` and ``TLS_1_2`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-securitypolicy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "securityPolicy"))

    @security_policy.setter
    def security_policy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75afdb4dd191961bbc6d953485a5ad0aa378366d791d7f192badace2044cc8ed)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "securityPolicy", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDomainName.EndpointConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"types": "types"},
    )
    class EndpointConfigurationProperty:
        def __init__(
            self,
            *,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``EndpointConfiguration`` property type specifies the endpoint types of an Amazon API Gateway domain name.

            ``EndpointConfiguration`` is a property of the `AWS::ApiGateway::DomainName <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html>`_ resource.

            :param types: A list of endpoint types of an API or its custom domain name. For an edge-optimized API and its custom domain name, the endpoint type is ``EDGE`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-endpointconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                endpoint_configuration_property = apigateway.CfnDomainName.EndpointConfigurationProperty(
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__36c411396c8c0551a824e0c6d34b79359042cdb65c803c1329ef6d329f19b488)
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of endpoint types of an API or its custom domain name.

            For an edge-optimized API and its custom domain name, the endpoint type is ``EDGE`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-endpointconfiguration.html#cfn-apigateway-domainname-endpointconfiguration-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDomainName.MutualTlsAuthenticationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "truststore_uri": "truststoreUri",
            "truststore_version": "truststoreVersion",
        },
    )
    class MutualTlsAuthenticationProperty:
        def __init__(
            self,
            *,
            truststore_uri: typing.Optional[builtins.str] = None,
            truststore_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''If specified, API Gateway performs two-way authentication between the client and the server.

            Clients must present a trusted certificate to access your API.

            :param truststore_uri: An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example, ``s3:// bucket-name / key-name`` . The truststore can contain certificates from public or private certificate authorities. To update the truststore, upload a new version to S3, and then update your custom domain name to use the new version. To update the truststore, you must have permissions to access the S3 object.
            :param truststore_version: The version of the S3 object that contains your truststore. To specify a version, you must have versioning enabled for the S3 bucket.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-mutualtlsauthentication.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                mutual_tls_authentication_property = apigateway.CfnDomainName.MutualTlsAuthenticationProperty(
                    truststore_uri="truststoreUri",
                    truststore_version="truststoreVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8eb274bcae6d59c283999588a689a3bf25654086b40226be71eeab697f2a5818)
                check_type(argname="argument truststore_uri", value=truststore_uri, expected_type=type_hints["truststore_uri"])
                check_type(argname="argument truststore_version", value=truststore_version, expected_type=type_hints["truststore_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if truststore_uri is not None:
                self._values["truststore_uri"] = truststore_uri
            if truststore_version is not None:
                self._values["truststore_version"] = truststore_version

        @builtins.property
        def truststore_uri(self) -> typing.Optional[builtins.str]:
            '''An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example, ``s3:// bucket-name / key-name`` .

            The truststore can contain certificates from public or private certificate authorities. To update the truststore, upload a new version to S3, and then update your custom domain name to use the new version. To update the truststore, you must have permissions to access the S3 object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-mutualtlsauthentication.html#cfn-apigateway-domainname-mutualtlsauthentication-truststoreuri
            '''
            result = self._values.get("truststore_uri")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def truststore_version(self) -> typing.Optional[builtins.str]:
            '''The version of the S3 object that contains your truststore.

            To specify a version, you must have versioning enabled for the S3 bucket.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-mutualtlsauthentication.html#cfn-apigateway-domainname-mutualtlsauthentication-truststoreversion
            '''
            result = self._values.get("truststore_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MutualTlsAuthenticationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnDomainNameProps",
    jsii_struct_bases=[],
    name_mapping={
        "certificate_arn": "certificateArn",
        "domain_name": "domainName",
        "endpoint_configuration": "endpointConfiguration",
        "mutual_tls_authentication": "mutualTlsAuthentication",
        "ownership_verification_certificate_arn": "ownershipVerificationCertificateArn",
        "regional_certificate_arn": "regionalCertificateArn",
        "security_policy": "securityPolicy",
        "tags": "tags",
    },
)
class CfnDomainNameProps:
    def __init__(
        self,
        *,
        certificate_arn: typing.Optional[builtins.str] = None,
        domain_name: typing.Optional[builtins.str] = None,
        endpoint_configuration: typing.Optional[typing.Union[typing.Union[CfnDomainName.EndpointConfigurationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        mutual_tls_authentication: typing.Optional[typing.Union[typing.Union[CfnDomainName.MutualTlsAuthenticationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        ownership_verification_certificate_arn: typing.Optional[builtins.str] = None,
        regional_certificate_arn: typing.Optional[builtins.str] = None,
        security_policy: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDomainName``.

        :param certificate_arn: The reference to an AWS -managed certificate for use by the edge-optimized endpoint for this domain name. AWS Certificate Manager is the only supported source. For requirements and additional information about setting up certificates, see `Get Certificates Ready in AWS Certificate Manager <https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html#how-to-custom-domains-prerequisites>`_ in the *API Gateway Developer Guide* .
        :param domain_name: The custom domain name for your API. Uppercase letters are not supported.
        :param endpoint_configuration: A list of the endpoint types of the domain name.
        :param mutual_tls_authentication: The mutual TLS authentication configuration for a custom domain name.
        :param ownership_verification_certificate_arn: The ARN of the public certificate issued by ACM to validate ownership of your custom domain. Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.
        :param regional_certificate_arn: The reference to an AWS -managed certificate for use by the regional endpoint for the domain name. AWS Certificate Manager is the only supported source.
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this domain name. Valid values include ``TLS_1_0`` and ``TLS_1_2`` .
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_domain_name_props = apigateway.CfnDomainNameProps(
                certificate_arn="certificateArn",
                domain_name="domainName",
                endpoint_configuration=apigateway.CfnDomainName.EndpointConfigurationProperty(
                    types=["types"]
                ),
                mutual_tls_authentication=apigateway.CfnDomainName.MutualTlsAuthenticationProperty(
                    truststore_uri="truststoreUri",
                    truststore_version="truststoreVersion"
                ),
                ownership_verification_certificate_arn="ownershipVerificationCertificateArn",
                regional_certificate_arn="regionalCertificateArn",
                security_policy="securityPolicy",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf1a720d5749a43860a38686b5c96d98b18993dd4cb61c9ca56b386337887a9e)
            check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument endpoint_configuration", value=endpoint_configuration, expected_type=type_hints["endpoint_configuration"])
            check_type(argname="argument mutual_tls_authentication", value=mutual_tls_authentication, expected_type=type_hints["mutual_tls_authentication"])
            check_type(argname="argument ownership_verification_certificate_arn", value=ownership_verification_certificate_arn, expected_type=type_hints["ownership_verification_certificate_arn"])
            check_type(argname="argument regional_certificate_arn", value=regional_certificate_arn, expected_type=type_hints["regional_certificate_arn"])
            check_type(argname="argument security_policy", value=security_policy, expected_type=type_hints["security_policy"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if certificate_arn is not None:
            self._values["certificate_arn"] = certificate_arn
        if domain_name is not None:
            self._values["domain_name"] = domain_name
        if endpoint_configuration is not None:
            self._values["endpoint_configuration"] = endpoint_configuration
        if mutual_tls_authentication is not None:
            self._values["mutual_tls_authentication"] = mutual_tls_authentication
        if ownership_verification_certificate_arn is not None:
            self._values["ownership_verification_certificate_arn"] = ownership_verification_certificate_arn
        if regional_certificate_arn is not None:
            self._values["regional_certificate_arn"] = regional_certificate_arn
        if security_policy is not None:
            self._values["security_policy"] = security_policy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate for use by the edge-optimized endpoint for this domain name.

        AWS Certificate Manager is the only supported source. For requirements and additional information about setting up certificates, see `Get Certificates Ready in AWS Certificate Manager <https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html#how-to-custom-domains-prerequisites>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-certificatearn
        '''
        result = self._values.get("certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''The custom domain name for your API.

        Uppercase letters are not supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-domainname
        '''
        result = self._values.get("domain_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnDomainName.EndpointConfigurationProperty, _IResolvable_a771d0ef]]:
        '''A list of the endpoint types of the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-endpointconfiguration
        '''
        result = self._values.get("endpoint_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnDomainName.EndpointConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def mutual_tls_authentication(
        self,
    ) -> typing.Optional[typing.Union[CfnDomainName.MutualTlsAuthenticationProperty, _IResolvable_a771d0ef]]:
        '''The mutual TLS authentication configuration for a custom domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-mutualtlsauthentication
        '''
        result = self._values.get("mutual_tls_authentication")
        return typing.cast(typing.Optional[typing.Union[CfnDomainName.MutualTlsAuthenticationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def ownership_verification_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the public certificate issued by ACM to validate ownership of your custom domain.

        Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-ownershipverificationcertificatearn
        '''
        result = self._values.get("ownership_verification_certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def regional_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate for use by the regional endpoint for the domain name.

        AWS Certificate Manager is the only supported source.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-regionalcertificatearn
        '''
        result = self._values.get("regional_certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_policy(self) -> typing.Optional[builtins.str]:
        '''The Transport Layer Security (TLS) version + cipher suite for this domain name.

        Valid values include ``TLS_1_0`` and ``TLS_1_2`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-securitypolicy
        '''
        result = self._values.get("security_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the domain name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDomainNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnDomainNameV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnDomainNameV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::DomainName``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::DomainName
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # tags: Any
        
        cfn_domain_name_v2 = apigateway.CfnDomainNameV2(self, "MyCfnDomainNameV2",
            domain_name="domainName",
        
            # the properties below are optional
            domain_name_configurations=[apigateway.CfnDomainNameV2.DomainNameConfigurationProperty(
                certificate_arn="certificateArn",
                certificate_name="certificateName",
                endpoint_type="endpointType"
            )],
            tags=tags
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        domain_name: builtins.str,
        domain_name_configurations: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnDomainNameV2.DomainNameConfigurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        tags: typing.Any = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::DomainName``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param domain_name: (deprecated) ``AWS::ApiGatewayV2::DomainName.DomainName``.
        :param domain_name_configurations: (deprecated) ``AWS::ApiGatewayV2::DomainName.DomainNameConfigurations``.
        :param tags: (deprecated) ``AWS::ApiGatewayV2::DomainName.Tags``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__89ba937b3c5ea155d210088dda8a3d71e4ac4b15d0f778d86b05bf336a48f10b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDomainNameV2Props(
            domain_name=domain_name,
            domain_name_configurations=domain_name_configurations,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5a660fde144e6ee78b79aea720a381dc571626fc02839a52e8867331c99cbe3)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6dfaad0b55e74e5047b3c8b4772c48ecbf86e302721f739eaa7a240d32e6ccf9)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrRegionalDomainName")
    def attr_regional_domain_name(self) -> builtins.str:
        '''
        :stability: deprecated
        :cloudformationAttribute: RegionalDomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRegionalDomainName"))

    @builtins.property
    @jsii.member(jsii_name="attrRegionalHostedZoneId")
    def attr_regional_hosted_zone_id(self) -> builtins.str:
        '''
        :stability: deprecated
        :cloudformationAttribute: RegionalHostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRegionalHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''(deprecated) ``AWS::ApiGatewayV2::DomainName.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html#cfn-apigatewayv2-domainname-tags
        :stability: deprecated
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::DomainName.DomainName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html#cfn-apigatewayv2-domainname-domainname
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e951eec8ce8c55ef9bcc861a75d70faea9521bc68fa3a2bcaad8628c755a030)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value)

    @builtins.property
    @jsii.member(jsii_name="domainNameConfigurations")
    def domain_name_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnDomainNameV2.DomainNameConfigurationProperty", _IResolvable_a771d0ef]]]]:
        '''(deprecated) ``AWS::ApiGatewayV2::DomainName.DomainNameConfigurations``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html#cfn-apigatewayv2-domainname-domainnameconfigurations
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnDomainNameV2.DomainNameConfigurationProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "domainNameConfigurations"))

    @domain_name_configurations.setter
    def domain_name_configurations(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnDomainNameV2.DomainNameConfigurationProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5950692b6478ee166c3c5d28ef25a6ae839fd492f2e5dffc1875e0a15f245a7d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainNameConfigurations", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnDomainNameV2.DomainNameConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_arn": "certificateArn",
            "certificate_name": "certificateName",
            "endpoint_type": "endpointType",
        },
    )
    class DomainNameConfigurationProperty:
        def __init__(
            self,
            *,
            certificate_arn: typing.Optional[builtins.str] = None,
            certificate_name: typing.Optional[builtins.str] = None,
            endpoint_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param certificate_arn: (deprecated) ``CfnDomainNameV2.DomainNameConfigurationProperty.CertificateArn``.
            :param certificate_name: (deprecated) ``CfnDomainNameV2.DomainNameConfigurationProperty.CertificateName``.
            :param endpoint_type: (deprecated) ``CfnDomainNameV2.DomainNameConfigurationProperty.EndpointType``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-domainname-domainnameconfiguration.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                domain_name_configuration_property = apigateway.CfnDomainNameV2.DomainNameConfigurationProperty(
                    certificate_arn="certificateArn",
                    certificate_name="certificateName",
                    endpoint_type="endpointType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__11337be0ae885f297a68195708c2cc320272585224c52d5e2cc29808c2d0ca0c)
                check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
                check_type(argname="argument certificate_name", value=certificate_name, expected_type=type_hints["certificate_name"])
                check_type(argname="argument endpoint_type", value=endpoint_type, expected_type=type_hints["endpoint_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if certificate_arn is not None:
                self._values["certificate_arn"] = certificate_arn
            if certificate_name is not None:
                self._values["certificate_name"] = certificate_name
            if endpoint_type is not None:
                self._values["endpoint_type"] = endpoint_type

        @builtins.property
        def certificate_arn(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnDomainNameV2.DomainNameConfigurationProperty.CertificateArn``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-domainname-domainnameconfiguration.html#cfn-apigatewayv2-domainname-domainnameconfiguration-certificatearn
            :stability: deprecated
            '''
            result = self._values.get("certificate_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def certificate_name(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnDomainNameV2.DomainNameConfigurationProperty.CertificateName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-domainname-domainnameconfiguration.html#cfn-apigatewayv2-domainname-domainnameconfiguration-certificatename
            :stability: deprecated
            '''
            result = self._values.get("certificate_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def endpoint_type(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnDomainNameV2.DomainNameConfigurationProperty.EndpointType``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-domainname-domainnameconfiguration.html#cfn-apigatewayv2-domainname-domainnameconfiguration-endpointtype
            :stability: deprecated
            '''
            result = self._values.get("endpoint_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DomainNameConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnDomainNameV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "domain_name_configurations": "domainNameConfigurations",
        "tags": "tags",
    },
)
class CfnDomainNameV2Props:
    def __init__(
        self,
        *,
        domain_name: builtins.str,
        domain_name_configurations: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union[CfnDomainNameV2.DomainNameConfigurationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        tags: typing.Any = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::DomainName``.

        :param domain_name: (deprecated) ``AWS::ApiGatewayV2::DomainName.DomainName``.
        :param domain_name_configurations: (deprecated) ``AWS::ApiGatewayV2::DomainName.DomainNameConfigurations``.
        :param tags: (deprecated) ``AWS::ApiGatewayV2::DomainName.Tags``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # tags: Any
            
            cfn_domain_name_v2_props = apigateway.CfnDomainNameV2Props(
                domain_name="domainName",
            
                # the properties below are optional
                domain_name_configurations=[apigateway.CfnDomainNameV2.DomainNameConfigurationProperty(
                    certificate_arn="certificateArn",
                    certificate_name="certificateName",
                    endpoint_type="endpointType"
                )],
                tags=tags
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d9b35396471735f222ea2fe66cde3a16f8aba4ae1f041775daea83e6d6958cd3)
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument domain_name_configurations", value=domain_name_configurations, expected_type=type_hints["domain_name_configurations"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
        }
        if domain_name_configurations is not None:
            self._values["domain_name_configurations"] = domain_name_configurations
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::DomainName.DomainName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html#cfn-apigatewayv2-domainname-domainname
        :stability: deprecated
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnDomainNameV2.DomainNameConfigurationProperty, _IResolvable_a771d0ef]]]]:
        '''(deprecated) ``AWS::ApiGatewayV2::DomainName.DomainNameConfigurations``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html#cfn-apigatewayv2-domainname-domainnameconfigurations
        :stability: deprecated
        '''
        result = self._values.get("domain_name_configurations")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnDomainNameV2.DomainNameConfigurationProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def tags(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::DomainName.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-domainname.html#cfn-apigatewayv2-domainname-tags
        :stability: deprecated
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDomainNameV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnGatewayResponse(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnGatewayResponse",
):
    '''A CloudFormation ``AWS::ApiGateway::GatewayResponse``.

    The ``AWS::ApiGateway::GatewayResponse`` resource creates a gateway response for your API. For more information, see `API Gateway Responses <https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html#api-gateway-gatewayResponse-definition>`_ in the *API Gateway Developer Guide* .

    :cloudformationResource: AWS::ApiGateway::GatewayResponse
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_gateway_response = apigateway.CfnGatewayResponse(self, "MyCfnGatewayResponse",
            response_type="responseType",
            rest_api_id="restApiId",
        
            # the properties below are optional
            response_parameters={
                "response_parameters_key": "responseParameters"
            },
            response_templates={
                "response_templates_key": "responseTemplates"
            },
            status_code="statusCode"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        response_type: builtins.str,
        rest_api_id: builtins.str,
        response_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        response_templates: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        status_code: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::GatewayResponse``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param response_type: The response type. For valid values, see `GatewayResponse <https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html>`_ in the *API Gateway API Reference* .
        :param rest_api_id: The identifier of the API.
        :param response_parameters: The response parameters (paths, query strings, and headers) for the response. Duplicates not allowed.
        :param response_templates: The response templates for the response. Duplicates not allowed.
        :param status_code: The HTTP status code for the response.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77eac94ccb084d7b68df7875b029dee2f6d7089fbf42105484edf66e4fe028d8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGatewayResponseProps(
            response_type=response_type,
            rest_api_id=rest_api_id,
            response_parameters=response_parameters,
            response_templates=response_templates,
            status_code=status_code,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e40f4d7070a189bf49aa7068e432f6e46162846cb712f6f90699a61478fde16)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53442ff2f8479cf0b5842d5c1737862599aa6b63b28f9c090e866222eb283594)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the gateway response.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="responseType")
    def response_type(self) -> builtins.str:
        '''The response type.

        For valid values, see `GatewayResponse <https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html>`_ in the *API Gateway API Reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responsetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "responseType"))

    @response_type.setter
    def response_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5afe13179945368d639942a28d817b675e32d410e3be4302bdeb2851b5be006)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseType", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b5d469ecc2c3903195d0cee96a7aeecb23546283b45f808137d088295039aa8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="responseParameters")
    def response_parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''The response parameters (paths, query strings, and headers) for the response.

        Duplicates not allowed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responseparameters
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], jsii.get(self, "responseParameters"))

    @response_parameters.setter
    def response_parameters(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__999eb2c7f37e5a2e136ec80032e08a368ecaa946f237e37e97e252988ae745a8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseParameters", value)

    @builtins.property
    @jsii.member(jsii_name="responseTemplates")
    def response_templates(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''The response templates for the response.

        Duplicates not allowed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responsetemplates
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], jsii.get(self, "responseTemplates"))

    @response_templates.setter
    def response_templates(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27ada1012634c474404cb2750af8c37610cf45cf83832755e117754d5eb9a917)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseTemplates", value)

    @builtins.property
    @jsii.member(jsii_name="statusCode")
    def status_code(self) -> typing.Optional[builtins.str]:
        '''The HTTP status code for the response.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-statuscode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "statusCode"))

    @status_code.setter
    def status_code(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__03b84577ef997d9a07f8a21fcff10724e36499f693367a51d9ae427c3b2207d5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "statusCode", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnGatewayResponseProps",
    jsii_struct_bases=[],
    name_mapping={
        "response_type": "responseType",
        "rest_api_id": "restApiId",
        "response_parameters": "responseParameters",
        "response_templates": "responseTemplates",
        "status_code": "statusCode",
    },
)
class CfnGatewayResponseProps:
    def __init__(
        self,
        *,
        response_type: builtins.str,
        rest_api_id: builtins.str,
        response_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        response_templates: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        status_code: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnGatewayResponse``.

        :param response_type: The response type. For valid values, see `GatewayResponse <https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html>`_ in the *API Gateway API Reference* .
        :param rest_api_id: The identifier of the API.
        :param response_parameters: The response parameters (paths, query strings, and headers) for the response. Duplicates not allowed.
        :param response_templates: The response templates for the response. Duplicates not allowed.
        :param status_code: The HTTP status code for the response.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_gateway_response_props = apigateway.CfnGatewayResponseProps(
                response_type="responseType",
                rest_api_id="restApiId",
            
                # the properties below are optional
                response_parameters={
                    "response_parameters_key": "responseParameters"
                },
                response_templates={
                    "response_templates_key": "responseTemplates"
                },
                status_code="statusCode"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4c1fcec3340db3288a9f8a2a279a58f01bc0a3db234a933e4b9e1590950b304c)
            check_type(argname="argument response_type", value=response_type, expected_type=type_hints["response_type"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
            check_type(argname="argument response_templates", value=response_templates, expected_type=type_hints["response_templates"])
            check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "response_type": response_type,
            "rest_api_id": rest_api_id,
        }
        if response_parameters is not None:
            self._values["response_parameters"] = response_parameters
        if response_templates is not None:
            self._values["response_templates"] = response_templates
        if status_code is not None:
            self._values["status_code"] = status_code

    @builtins.property
    def response_type(self) -> builtins.str:
        '''The response type.

        For valid values, see `GatewayResponse <https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html>`_ in the *API Gateway API Reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responsetype
        '''
        result = self._values.get("response_type")
        assert result is not None, "Required property 'response_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def response_parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''The response parameters (paths, query strings, and headers) for the response.

        Duplicates not allowed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responseparameters
        '''
        result = self._values.get("response_parameters")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

    @builtins.property
    def response_templates(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''The response templates for the response.

        Duplicates not allowed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responsetemplates
        '''
        result = self._values.get("response_templates")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

    @builtins.property
    def status_code(self) -> typing.Optional[builtins.str]:
        '''The HTTP status code for the response.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-statuscode
        '''
        result = self._values.get("status_code")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayResponseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnIntegrationResponseV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnIntegrationResponseV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::IntegrationResponse``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::IntegrationResponse
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # response_parameters: Any
        # response_templates: Any
        
        cfn_integration_response_v2 = apigateway.CfnIntegrationResponseV2(self, "MyCfnIntegrationResponseV2",
            api_id="apiId",
            integration_id="integrationId",
            integration_response_key="integrationResponseKey",
        
            # the properties below are optional
            content_handling_strategy="contentHandlingStrategy",
            response_parameters=response_parameters,
            response_templates=response_templates,
            template_selection_expression="templateSelectionExpression"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        integration_id: builtins.str,
        integration_response_key: builtins.str,
        content_handling_strategy: typing.Optional[builtins.str] = None,
        response_parameters: typing.Any = None,
        response_templates: typing.Any = None,
        template_selection_expression: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::IntegrationResponse``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ApiId``.
        :param integration_id: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationId``.
        :param integration_response_key: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationResponseKey``.
        :param content_handling_strategy: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ContentHandlingStrategy``.
        :param response_parameters: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseParameters``.
        :param response_templates: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseTemplates``.
        :param template_selection_expression: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.TemplateSelectionExpression``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__88f3e10fc034fa791e3b84ab4a13c32bbcb6a425491f68961463b74f5f74334a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIntegrationResponseV2Props(
            api_id=api_id,
            integration_id=integration_id,
            integration_response_key=integration_response_key,
            content_handling_strategy=content_handling_strategy,
            response_parameters=response_parameters,
            response_templates=response_templates,
            template_selection_expression=template_selection_expression,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__af758a62b5bec54498b2bed279c588c25927ee5db32c2242d1978ce2a86908ca)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57b81832afabc603310e7f1b35c004fc1d0c8568e2b4bcac9015aa5e3e6ab917)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc7289c1598544b09947be321a3cc689731c39e9a06d1ac4d4bc4c16161e9cbc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="integrationId")
    def integration_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-integrationid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "integrationId"))

    @integration_id.setter
    def integration_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e88a9c95b904de5b8ec2fd34ab263550c25bd32594feb41d80c810dc58a00dda)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integrationId", value)

    @builtins.property
    @jsii.member(jsii_name="integrationResponseKey")
    def integration_response_key(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationResponseKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-integrationresponsekey
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "integrationResponseKey"))

    @integration_response_key.setter
    def integration_response_key(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dba09266f5e62b66f103f9fa1cba9034ca47e3a279b0a2e7bd47ae3086323654)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integrationResponseKey", value)

    @builtins.property
    @jsii.member(jsii_name="responseParameters")
    def response_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-responseparameters
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "responseParameters"))

    @response_parameters.setter
    def response_parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a531e80f5223544e5189de853361f9119a4086cbf847ca82a23a739431b1ca9f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseParameters", value)

    @builtins.property
    @jsii.member(jsii_name="responseTemplates")
    def response_templates(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseTemplates``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-responsetemplates
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "responseTemplates"))

    @response_templates.setter
    def response_templates(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c39a31e7c86eb0696badf162d7e6e57cc781e3d9bf43e2568083a5550d27e57)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseTemplates", value)

    @builtins.property
    @jsii.member(jsii_name="contentHandlingStrategy")
    def content_handling_strategy(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ContentHandlingStrategy``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-contenthandlingstrategy
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "contentHandlingStrategy"))

    @content_handling_strategy.setter
    def content_handling_strategy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__adada7b7d4bfc7a3d8b344e45c9e44760980e51a47d12ee857551495cf578656)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "contentHandlingStrategy", value)

    @builtins.property
    @jsii.member(jsii_name="templateSelectionExpression")
    def template_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.TemplateSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-templateselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "templateSelectionExpression"))

    @template_selection_expression.setter
    def template_selection_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0a8d08fc21c6212c3d292d8e98aca086e3a5de8f6d10ea9a644789703853a690)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "templateSelectionExpression", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnIntegrationResponseV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "integration_id": "integrationId",
        "integration_response_key": "integrationResponseKey",
        "content_handling_strategy": "contentHandlingStrategy",
        "response_parameters": "responseParameters",
        "response_templates": "responseTemplates",
        "template_selection_expression": "templateSelectionExpression",
    },
)
class CfnIntegrationResponseV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        integration_id: builtins.str,
        integration_response_key: builtins.str,
        content_handling_strategy: typing.Optional[builtins.str] = None,
        response_parameters: typing.Any = None,
        response_templates: typing.Any = None,
        template_selection_expression: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::IntegrationResponse``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ApiId``.
        :param integration_id: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationId``.
        :param integration_response_key: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationResponseKey``.
        :param content_handling_strategy: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ContentHandlingStrategy``.
        :param response_parameters: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseParameters``.
        :param response_templates: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseTemplates``.
        :param template_selection_expression: (deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.TemplateSelectionExpression``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # response_parameters: Any
            # response_templates: Any
            
            cfn_integration_response_v2_props = apigateway.CfnIntegrationResponseV2Props(
                api_id="apiId",
                integration_id="integrationId",
                integration_response_key="integrationResponseKey",
            
                # the properties below are optional
                content_handling_strategy="contentHandlingStrategy",
                response_parameters=response_parameters,
                response_templates=response_templates,
                template_selection_expression="templateSelectionExpression"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4dda921787dab9e87b67b0f58a3a28a4bea7118a0b47d154b2f746c47264367c)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument integration_id", value=integration_id, expected_type=type_hints["integration_id"])
            check_type(argname="argument integration_response_key", value=integration_response_key, expected_type=type_hints["integration_response_key"])
            check_type(argname="argument content_handling_strategy", value=content_handling_strategy, expected_type=type_hints["content_handling_strategy"])
            check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
            check_type(argname="argument response_templates", value=response_templates, expected_type=type_hints["response_templates"])
            check_type(argname="argument template_selection_expression", value=template_selection_expression, expected_type=type_hints["template_selection_expression"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "integration_id": integration_id,
            "integration_response_key": integration_response_key,
        }
        if content_handling_strategy is not None:
            self._values["content_handling_strategy"] = content_handling_strategy
        if response_parameters is not None:
            self._values["response_parameters"] = response_parameters
        if response_templates is not None:
            self._values["response_templates"] = response_templates
        if template_selection_expression is not None:
            self._values["template_selection_expression"] = template_selection_expression

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def integration_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-integrationid
        :stability: deprecated
        '''
        result = self._values.get("integration_id")
        assert result is not None, "Required property 'integration_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def integration_response_key(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.IntegrationResponseKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-integrationresponsekey
        :stability: deprecated
        '''
        result = self._values.get("integration_response_key")
        assert result is not None, "Required property 'integration_response_key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def content_handling_strategy(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ContentHandlingStrategy``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-contenthandlingstrategy
        :stability: deprecated
        '''
        result = self._values.get("content_handling_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def response_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-responseparameters
        :stability: deprecated
        '''
        result = self._values.get("response_parameters")
        return typing.cast(typing.Any, result)

    @builtins.property
    def response_templates(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.ResponseTemplates``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-responsetemplates
        :stability: deprecated
        '''
        result = self._values.get("response_templates")
        return typing.cast(typing.Any, result)

    @builtins.property
    def template_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::IntegrationResponse.TemplateSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integrationresponse.html#cfn-apigatewayv2-integrationresponse-templateselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("template_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIntegrationResponseV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnIntegrationV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnIntegrationV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Integration``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Integration
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # request_parameters: Any
        # request_templates: Any
        
        cfn_integration_v2 = apigateway.CfnIntegrationV2(self, "MyCfnIntegrationV2",
            api_id="apiId",
            integration_type="integrationType",
        
            # the properties below are optional
            connection_type="connectionType",
            content_handling_strategy="contentHandlingStrategy",
            credentials_arn="credentialsArn",
            description="description",
            integration_method="integrationMethod",
            integration_uri="integrationUri",
            passthrough_behavior="passthroughBehavior",
            payload_format_version="payloadFormatVersion",
            request_parameters=request_parameters,
            request_templates=request_templates,
            template_selection_expression="templateSelectionExpression",
            timeout_in_millis=123
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        integration_type: builtins.str,
        connection_type: typing.Optional[builtins.str] = None,
        content_handling_strategy: typing.Optional[builtins.str] = None,
        credentials_arn: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        integration_method: typing.Optional[builtins.str] = None,
        integration_uri: typing.Optional[builtins.str] = None,
        passthrough_behavior: typing.Optional[builtins.str] = None,
        payload_format_version: typing.Optional[builtins.str] = None,
        request_parameters: typing.Any = None,
        request_templates: typing.Any = None,
        template_selection_expression: typing.Optional[builtins.str] = None,
        timeout_in_millis: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Integration``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Integration.ApiId``.
        :param integration_type: (deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationType``.
        :param connection_type: (deprecated) ``AWS::ApiGatewayV2::Integration.ConnectionType``.
        :param content_handling_strategy: (deprecated) ``AWS::ApiGatewayV2::Integration.ContentHandlingStrategy``.
        :param credentials_arn: (deprecated) ``AWS::ApiGatewayV2::Integration.CredentialsArn``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Integration.Description``.
        :param integration_method: (deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationMethod``.
        :param integration_uri: (deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationUri``.
        :param passthrough_behavior: (deprecated) ``AWS::ApiGatewayV2::Integration.PassthroughBehavior``.
        :param payload_format_version: (deprecated) ``AWS::ApiGatewayV2::Integration.PayloadFormatVersion``.
        :param request_parameters: (deprecated) ``AWS::ApiGatewayV2::Integration.RequestParameters``.
        :param request_templates: (deprecated) ``AWS::ApiGatewayV2::Integration.RequestTemplates``.
        :param template_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Integration.TemplateSelectionExpression``.
        :param timeout_in_millis: (deprecated) ``AWS::ApiGatewayV2::Integration.TimeoutInMillis``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__26216a1ac48365f0954b98f413df358a875f99f8a0e431396bab4ea4369bcaef)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIntegrationV2Props(
            api_id=api_id,
            integration_type=integration_type,
            connection_type=connection_type,
            content_handling_strategy=content_handling_strategy,
            credentials_arn=credentials_arn,
            description=description,
            integration_method=integration_method,
            integration_uri=integration_uri,
            passthrough_behavior=passthrough_behavior,
            payload_format_version=payload_format_version,
            request_parameters=request_parameters,
            request_templates=request_templates,
            template_selection_expression=template_selection_expression,
            timeout_in_millis=timeout_in_millis,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35f350dbab7598c5808128a55a16dbca4618fe50a8498d803ef5d7c4d0b9385d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9c8ade491ea612ac63455772a658be3b333213d662462365633dcf79a9a9c9b0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__982013b8f1d4262b87a91b467cec9ce5046d5645da4d1a2ab55b46565292aaf4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="integrationType")
    def integration_type(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-integrationtype
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "integrationType"))

    @integration_type.setter
    def integration_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb756b303d3fe05a763fc72afbc40b84bd5841ffb7056877a69c6220e1d50d6c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integrationType", value)

    @builtins.property
    @jsii.member(jsii_name="requestParameters")
    def request_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.RequestParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-requestparameters
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "requestParameters"))

    @request_parameters.setter
    def request_parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee665c1bb8837b733bf6675ca8f42fe063cd12db624c49d60537e368800b9a12)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestParameters", value)

    @builtins.property
    @jsii.member(jsii_name="requestTemplates")
    def request_templates(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.RequestTemplates``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-requesttemplates
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "requestTemplates"))

    @request_templates.setter
    def request_templates(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a2ba3abc0805de649cfeb38315f219c9aa797a163d817d36d60c0ae792a952e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestTemplates", value)

    @builtins.property
    @jsii.member(jsii_name="connectionType")
    def connection_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.ConnectionType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-connectiontype
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "connectionType"))

    @connection_type.setter
    def connection_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a0b656ec7c4881baf07ac2ca0b7a387217634ccf33f2ffff7df29794ab73bd5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "connectionType", value)

    @builtins.property
    @jsii.member(jsii_name="contentHandlingStrategy")
    def content_handling_strategy(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.ContentHandlingStrategy``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-contenthandlingstrategy
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "contentHandlingStrategy"))

    @content_handling_strategy.setter
    def content_handling_strategy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c95c7fd7497e77915bffa983ae64edf3f349e6afa603cc87902d5deee42bfc8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "contentHandlingStrategy", value)

    @builtins.property
    @jsii.member(jsii_name="credentialsArn")
    def credentials_arn(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.CredentialsArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-credentialsarn
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "credentialsArn"))

    @credentials_arn.setter
    def credentials_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__86486d800bbd37759c75151f058945a43465a5443917bebd5d87243dfb7540fd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "credentialsArn", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-description
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b380b692de77b117f02f3d14057d4f96d295bb7bcdcb50f252b55ec2b7af93b2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="integrationMethod")
    def integration_method(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationMethod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-integrationmethod
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "integrationMethod"))

    @integration_method.setter
    def integration_method(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00dfa60a543281e4ef50ad067dec2c0799b7c0e8fba9cb369b9ccffde7b107f2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integrationMethod", value)

    @builtins.property
    @jsii.member(jsii_name="integrationUri")
    def integration_uri(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationUri``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-integrationuri
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "integrationUri"))

    @integration_uri.setter
    def integration_uri(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ff866f398f5e655518ad0be4b82d1569fc5f0bb5ca6ed4ec654325fdc81ae24)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integrationUri", value)

    @builtins.property
    @jsii.member(jsii_name="passthroughBehavior")
    def passthrough_behavior(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.PassthroughBehavior``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-passthroughbehavior
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "passthroughBehavior"))

    @passthrough_behavior.setter
    def passthrough_behavior(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea187b22d3325ec48dd2a0f2e703cae4a5c3402c054ed58a2ed59fa7adcab63c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "passthroughBehavior", value)

    @builtins.property
    @jsii.member(jsii_name="payloadFormatVersion")
    def payload_format_version(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.PayloadFormatVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-payloadformatversion
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "payloadFormatVersion"))

    @payload_format_version.setter
    def payload_format_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e1e17538d04a3a2dabab0f43f04c3f211b3052fb53ebc115568cff7baf6c5d4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "payloadFormatVersion", value)

    @builtins.property
    @jsii.member(jsii_name="templateSelectionExpression")
    def template_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.TemplateSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-templateselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "templateSelectionExpression"))

    @template_selection_expression.setter
    def template_selection_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98dacd7f47f355e282899bcc93c390dcc8dd9c154ad0ce9e97056703461194bc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "templateSelectionExpression", value)

    @builtins.property
    @jsii.member(jsii_name="timeoutInMillis")
    def timeout_in_millis(self) -> typing.Optional[jsii.Number]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.TimeoutInMillis``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-timeoutinmillis
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "timeoutInMillis"))

    @timeout_in_millis.setter
    def timeout_in_millis(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__07f9823f36bc15d099ae4db5148ae7aca63ce21d3c55dea5c3d16c63f138c5f3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timeoutInMillis", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnIntegrationV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "integration_type": "integrationType",
        "connection_type": "connectionType",
        "content_handling_strategy": "contentHandlingStrategy",
        "credentials_arn": "credentialsArn",
        "description": "description",
        "integration_method": "integrationMethod",
        "integration_uri": "integrationUri",
        "passthrough_behavior": "passthroughBehavior",
        "payload_format_version": "payloadFormatVersion",
        "request_parameters": "requestParameters",
        "request_templates": "requestTemplates",
        "template_selection_expression": "templateSelectionExpression",
        "timeout_in_millis": "timeoutInMillis",
    },
)
class CfnIntegrationV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        integration_type: builtins.str,
        connection_type: typing.Optional[builtins.str] = None,
        content_handling_strategy: typing.Optional[builtins.str] = None,
        credentials_arn: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        integration_method: typing.Optional[builtins.str] = None,
        integration_uri: typing.Optional[builtins.str] = None,
        passthrough_behavior: typing.Optional[builtins.str] = None,
        payload_format_version: typing.Optional[builtins.str] = None,
        request_parameters: typing.Any = None,
        request_templates: typing.Any = None,
        template_selection_expression: typing.Optional[builtins.str] = None,
        timeout_in_millis: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Integration``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Integration.ApiId``.
        :param integration_type: (deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationType``.
        :param connection_type: (deprecated) ``AWS::ApiGatewayV2::Integration.ConnectionType``.
        :param content_handling_strategy: (deprecated) ``AWS::ApiGatewayV2::Integration.ContentHandlingStrategy``.
        :param credentials_arn: (deprecated) ``AWS::ApiGatewayV2::Integration.CredentialsArn``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Integration.Description``.
        :param integration_method: (deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationMethod``.
        :param integration_uri: (deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationUri``.
        :param passthrough_behavior: (deprecated) ``AWS::ApiGatewayV2::Integration.PassthroughBehavior``.
        :param payload_format_version: (deprecated) ``AWS::ApiGatewayV2::Integration.PayloadFormatVersion``.
        :param request_parameters: (deprecated) ``AWS::ApiGatewayV2::Integration.RequestParameters``.
        :param request_templates: (deprecated) ``AWS::ApiGatewayV2::Integration.RequestTemplates``.
        :param template_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Integration.TemplateSelectionExpression``.
        :param timeout_in_millis: (deprecated) ``AWS::ApiGatewayV2::Integration.TimeoutInMillis``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # request_parameters: Any
            # request_templates: Any
            
            cfn_integration_v2_props = apigateway.CfnIntegrationV2Props(
                api_id="apiId",
                integration_type="integrationType",
            
                # the properties below are optional
                connection_type="connectionType",
                content_handling_strategy="contentHandlingStrategy",
                credentials_arn="credentialsArn",
                description="description",
                integration_method="integrationMethod",
                integration_uri="integrationUri",
                passthrough_behavior="passthroughBehavior",
                payload_format_version="payloadFormatVersion",
                request_parameters=request_parameters,
                request_templates=request_templates,
                template_selection_expression="templateSelectionExpression",
                timeout_in_millis=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48d227a7afea5477733e87cd46cf0c68c5a37445ed89a2d976f0b36793cf1349)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument integration_type", value=integration_type, expected_type=type_hints["integration_type"])
            check_type(argname="argument connection_type", value=connection_type, expected_type=type_hints["connection_type"])
            check_type(argname="argument content_handling_strategy", value=content_handling_strategy, expected_type=type_hints["content_handling_strategy"])
            check_type(argname="argument credentials_arn", value=credentials_arn, expected_type=type_hints["credentials_arn"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument integration_method", value=integration_method, expected_type=type_hints["integration_method"])
            check_type(argname="argument integration_uri", value=integration_uri, expected_type=type_hints["integration_uri"])
            check_type(argname="argument passthrough_behavior", value=passthrough_behavior, expected_type=type_hints["passthrough_behavior"])
            check_type(argname="argument payload_format_version", value=payload_format_version, expected_type=type_hints["payload_format_version"])
            check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
            check_type(argname="argument request_templates", value=request_templates, expected_type=type_hints["request_templates"])
            check_type(argname="argument template_selection_expression", value=template_selection_expression, expected_type=type_hints["template_selection_expression"])
            check_type(argname="argument timeout_in_millis", value=timeout_in_millis, expected_type=type_hints["timeout_in_millis"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "integration_type": integration_type,
        }
        if connection_type is not None:
            self._values["connection_type"] = connection_type
        if content_handling_strategy is not None:
            self._values["content_handling_strategy"] = content_handling_strategy
        if credentials_arn is not None:
            self._values["credentials_arn"] = credentials_arn
        if description is not None:
            self._values["description"] = description
        if integration_method is not None:
            self._values["integration_method"] = integration_method
        if integration_uri is not None:
            self._values["integration_uri"] = integration_uri
        if passthrough_behavior is not None:
            self._values["passthrough_behavior"] = passthrough_behavior
        if payload_format_version is not None:
            self._values["payload_format_version"] = payload_format_version
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if request_templates is not None:
            self._values["request_templates"] = request_templates
        if template_selection_expression is not None:
            self._values["template_selection_expression"] = template_selection_expression
        if timeout_in_millis is not None:
            self._values["timeout_in_millis"] = timeout_in_millis

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def integration_type(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-integrationtype
        :stability: deprecated
        '''
        result = self._values.get("integration_type")
        assert result is not None, "Required property 'integration_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connection_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.ConnectionType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-connectiontype
        :stability: deprecated
        '''
        result = self._values.get("connection_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def content_handling_strategy(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.ContentHandlingStrategy``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-contenthandlingstrategy
        :stability: deprecated
        '''
        result = self._values.get("content_handling_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials_arn(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.CredentialsArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-credentialsarn
        :stability: deprecated
        '''
        result = self._values.get("credentials_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-description
        :stability: deprecated
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_method(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationMethod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-integrationmethod
        :stability: deprecated
        '''
        result = self._values.get("integration_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_uri(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.IntegrationUri``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-integrationuri
        :stability: deprecated
        '''
        result = self._values.get("integration_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def passthrough_behavior(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.PassthroughBehavior``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-passthroughbehavior
        :stability: deprecated
        '''
        result = self._values.get("passthrough_behavior")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def payload_format_version(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.PayloadFormatVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-payloadformatversion
        :stability: deprecated
        '''
        result = self._values.get("payload_format_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.RequestParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-requestparameters
        :stability: deprecated
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Any, result)

    @builtins.property
    def request_templates(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.RequestTemplates``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-requesttemplates
        :stability: deprecated
        '''
        result = self._values.get("request_templates")
        return typing.cast(typing.Any, result)

    @builtins.property
    def template_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.TemplateSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-templateselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("template_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout_in_millis(self) -> typing.Optional[jsii.Number]:
        '''(deprecated) ``AWS::ApiGatewayV2::Integration.TimeoutInMillis``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-integration.html#cfn-apigatewayv2-integration-timeoutinmillis
        :stability: deprecated
        '''
        result = self._values.get("timeout_in_millis")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIntegrationV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnMethod(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnMethod",
):
    '''A CloudFormation ``AWS::ApiGateway::Method``.

    The ``AWS::ApiGateway::Method`` resource creates API Gateway methods that define the parameters and body that clients must send in their requests.

    :cloudformationResource: AWS::ApiGateway::Method
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_method = apigateway.CfnMethod(self, "MyCfnMethod",
            http_method="httpMethod",
            resource_id="resourceId",
            rest_api_id="restApiId",
        
            # the properties below are optional
            api_key_required=False,
            authorization_scopes=["authorizationScopes"],
            authorization_type="authorizationType",
            authorizer_id="authorizerId",
            integration=apigateway.CfnMethod.IntegrationProperty(
                cache_key_parameters=["cacheKeyParameters"],
                cache_namespace="cacheNamespace",
                connection_id="connectionId",
                connection_type="connectionType",
                content_handling="contentHandling",
                credentials="credentials",
                integration_http_method="integrationHttpMethod",
                integration_responses=[apigateway.CfnMethod.IntegrationResponseProperty(
                    status_code="statusCode",
        
                    # the properties below are optional
                    content_handling="contentHandling",
                    response_parameters={
                        "response_parameters_key": "responseParameters"
                    },
                    response_templates={
                        "response_templates_key": "responseTemplates"
                    },
                    selection_pattern="selectionPattern"
                )],
                passthrough_behavior="passthroughBehavior",
                request_parameters={
                    "request_parameters_key": "requestParameters"
                },
                request_templates={
                    "request_templates_key": "requestTemplates"
                },
                timeout_in_millis=123,
                type="type",
                uri="uri"
            ),
            method_responses=[apigateway.CfnMethod.MethodResponseProperty(
                status_code="statusCode",
        
                # the properties below are optional
                response_models={
                    "response_models_key": "responseModels"
                },
                response_parameters={
                    "response_parameters_key": False
                }
            )],
            operation_name="operationName",
            request_models={
                "request_models_key": "requestModels"
            },
            request_parameters={
                "request_parameters_key": False
            },
            request_validator_id="requestValidatorId"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        http_method: builtins.str,
        resource_id: builtins.str,
        rest_api_id: builtins.str,
        api_key_required: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[builtins.str] = None,
        authorizer_id: typing.Optional[builtins.str] = None,
        integration: typing.Optional[typing.Union[typing.Union["CfnMethod.IntegrationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        method_responses: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnMethod.MethodResponseProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        request_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]] = None,
        request_validator_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Method``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param http_method: The HTTP method that clients use to call this method.
        :param resource_id: The ID of an API Gateway `resource <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html>`_ . For root resource methods, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .
        :param rest_api_id: The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which API Gateway creates the method.
        :param api_key_required: Indicates whether the method requires clients to submit a valid API key.
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes match a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.
        :param authorization_type: The method's authorization type. This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* . .. epigraph:: If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.
        :param authorizer_id: The identifier of the `authorizer <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html>`_ to use on this method. If you specify this property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for the ``AuthorizationType`` property.
        :param integration: The backend system that the method calls when it receives a request.
        :param method_responses: The responses that can be sent to the client who calls the method.
        :param operation_name: A friendly operation name for the method. For example, you can assign the ``OperationName`` of ``ListPets`` for the ``GET /pets`` method.
        :param request_models: The resources that are used for the request's content type. Specify request models as key-value pairs (string-to-string mapping), with a content type as the key and a ``Model`` resource name as the value. To use the same model regardless of the content type, specify ``$default`` as the key.
        :param request_parameters: The request parameters that API Gateway accepts. Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format ``method.request. *location* . *name*`` , where the location is querystring, path, or header, and *name* is a valid, unique parameter name.
        :param request_validator_id: The ID of the associated request validator.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__36ffcc494ce8281636126d001b6511cf64465cd0836959d1014ef6a72ffaf64b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnMethodProps(
            http_method=http_method,
            resource_id=resource_id,
            rest_api_id=rest_api_id,
            api_key_required=api_key_required,
            authorization_scopes=authorization_scopes,
            authorization_type=authorization_type,
            authorizer_id=authorizer_id,
            integration=integration,
            method_responses=method_responses,
            operation_name=operation_name,
            request_models=request_models,
            request_parameters=request_parameters,
            request_validator_id=request_validator_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e519c6aa95e122ff49fe3157e57a8ce053b852ffc835466bf3990fff666c92f)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b22edc4d1f4d5b7af5dc09b48e9ea207955d458803976f471042ec94949d3d5)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="httpMethod")
    def http_method(self) -> builtins.str:
        '''The HTTP method that clients use to call this method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-httpmethod
        '''
        return typing.cast(builtins.str, jsii.get(self, "httpMethod"))

    @http_method.setter
    def http_method(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f4f23a25bb931c59f63fb8826f41eecef040511ff5dbd8682c4cfe70737d616)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "httpMethod", value)

    @builtins.property
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''The ID of an API Gateway `resource <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html>`_ . For root resource methods, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-resourceid
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceId"))

    @resource_id.setter
    def resource_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__97383762ba41e2d7fa00b1ec9c8a7479e71e665e83ee41400d15dcedb124d87f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resourceId", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which API Gateway creates the method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f05eddf3004dcc246274c986c7c504035c0100721902db315d7e99ed68636dd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="apiKeyRequired")
    def api_key_required(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether the method requires clients to submit a valid API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-apikeyrequired
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "apiKeyRequired"))

    @api_key_required.setter
    def api_key_required(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a26d94bfda70c13009ab6840746a2fb58eb5478fb4d0dc17e58e2b7abbb615a2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiKeyRequired", value)

    @builtins.property
    @jsii.member(jsii_name="authorizationScopes")
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of authorization scopes configured on the method.

        The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes match a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "authorizationScopes"))

    @authorization_scopes.setter
    def authorization_scopes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bdfac8366d475ab99bfea94aa0de93c3dea517e5a199d4a69e71aeed5f40f62d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizationScopes", value)

    @builtins.property
    @jsii.member(jsii_name="authorizationType")
    def authorization_type(self) -> typing.Optional[builtins.str]:
        '''The method's authorization type.

        This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* .
        .. epigraph::

           If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizationType"))

    @authorization_type.setter
    def authorization_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__39421fa2523e949b0da4e0a49522b8cb54f6fd28f5b1d79bae9f7da1ae2916ff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizationType", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerId")
    def authorizer_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of the `authorizer <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html>`_ to use on this method. If you specify this property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for the ``AuthorizationType`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizerid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerId"))

    @authorizer_id.setter
    def authorizer_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__322f2af378a4181fa69b3cffe1ca9784b04b496a283cf838c5b676b39a218c03)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerId", value)

    @builtins.property
    @jsii.member(jsii_name="integration")
    def integration(
        self,
    ) -> typing.Optional[typing.Union["CfnMethod.IntegrationProperty", _IResolvable_a771d0ef]]:
        '''The backend system that the method calls when it receives a request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-integration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnMethod.IntegrationProperty", _IResolvable_a771d0ef]], jsii.get(self, "integration"))

    @integration.setter
    def integration(
        self,
        value: typing.Optional[typing.Union["CfnMethod.IntegrationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__88eee5d34aaa956c488a4f4ead278ce202ba323d38c0f270bef031b562ec622a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integration", value)

    @builtins.property
    @jsii.member(jsii_name="methodResponses")
    def method_responses(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnMethod.MethodResponseProperty", _IResolvable_a771d0ef]]]]:
        '''The responses that can be sent to the client who calls the method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-methodresponses
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnMethod.MethodResponseProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "methodResponses"))

    @method_responses.setter
    def method_responses(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnMethod.MethodResponseProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cce180bab6f0af88653e94fd4f661ab3302302c205c167b281634f5904b258f5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "methodResponses", value)

    @builtins.property
    @jsii.member(jsii_name="operationName")
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''A friendly operation name for the method.

        For example, you can assign the ``OperationName`` of ``ListPets`` for the ``GET /pets`` method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-operationname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "operationName"))

    @operation_name.setter
    def operation_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76b27ac3666d6d4fd58dd39ec55461c9b70d2b948582fa479e28a6dd1b1bbdcb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "operationName", value)

    @builtins.property
    @jsii.member(jsii_name="requestModels")
    def request_models(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''The resources that are used for the request's content type.

        Specify request models as key-value pairs (string-to-string mapping), with a content type as the key and a ``Model`` resource name as the value. To use the same model regardless of the content type, specify ``$default`` as the key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestmodels
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], jsii.get(self, "requestModels"))

    @request_models.setter
    def request_models(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__04a67d72de199d616812afe6cbcb61cf37707c1ca69329f6e08f71ac68085ab3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestModels", value)

    @builtins.property
    @jsii.member(jsii_name="requestParameters")
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]]:
        '''The request parameters that API Gateway accepts.

        Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format ``method.request. *location* . *name*`` , where the location is querystring, path, or header, and *name* is a valid, unique parameter name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestparameters
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]], jsii.get(self, "requestParameters"))

    @request_parameters.setter
    def request_parameters(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a01ac30e87d2b4c091633e5cb0eea53752034d7ee43622f2636b6059ae0f7e04)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestParameters", value)

    @builtins.property
    @jsii.member(jsii_name="requestValidatorId")
    def request_validator_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the associated request validator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestvalidatorid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "requestValidatorId"))

    @request_validator_id.setter
    def request_validator_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fd00593c4ec4b0a6c27f23287af3acdf8bc088b87ed52b587a9ba4625ecae3d9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestValidatorId", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnMethod.IntegrationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cache_key_parameters": "cacheKeyParameters",
            "cache_namespace": "cacheNamespace",
            "connection_id": "connectionId",
            "connection_type": "connectionType",
            "content_handling": "contentHandling",
            "credentials": "credentials",
            "integration_http_method": "integrationHttpMethod",
            "integration_responses": "integrationResponses",
            "passthrough_behavior": "passthroughBehavior",
            "request_parameters": "requestParameters",
            "request_templates": "requestTemplates",
            "timeout_in_millis": "timeoutInMillis",
            "type": "type",
            "uri": "uri",
        },
    )
    class IntegrationProperty:
        def __init__(
            self,
            *,
            cache_key_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
            cache_namespace: typing.Optional[builtins.str] = None,
            connection_id: typing.Optional[builtins.str] = None,
            connection_type: typing.Optional[builtins.str] = None,
            content_handling: typing.Optional[builtins.str] = None,
            credentials: typing.Optional[builtins.str] = None,
            integration_http_method: typing.Optional[builtins.str] = None,
            integration_responses: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnMethod.IntegrationResponseProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
            passthrough_behavior: typing.Optional[builtins.str] = None,
            request_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            request_templates: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            timeout_in_millis: typing.Optional[jsii.Number] = None,
            type: typing.Optional[builtins.str] = None,
            uri: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``Integration`` is a property of the `AWS::ApiGateway::Method <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html>`_ resource that specifies information about the target backend that a method calls.

            :param cache_key_parameters: A list of request parameters whose values API Gateway caches. For cases where the integration type allows for RequestParameters to be set, these parameters must also be specified in `RequestParameters <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestparameters>`_ to be supported in ``CacheKeyParameters`` .
            :param cache_namespace: An API-specific tag group of related cached parameters.
            :param connection_id: The ID of the ``VpcLink`` used for the integration when ``connectionType=VPC_LINK`` , otherwise undefined.
            :param connection_type: The type of the network connection to the integration endpoint. The valid value is ``INTERNET`` for connections through the public routable internet or ``VPC_LINK`` for private connections between API Gateway and a network load balancer in a VPC. The default value is ``INTERNET`` .
            :param content_handling: Specifies how to handle request payload content type conversions. Valid values are:. - ``CONVERT_TO_BINARY`` : Converts a request payload from a base64-encoded string to a binary blob. - ``CONVERT_TO_TEXT`` : Converts a request payload from a binary blob to a base64-encoded string. If this property isn't defined, the request payload is passed through from the method request to the integration request without modification, provided that the ``PassthroughBehaviors`` property is configured to support payload pass-through.
            :param credentials: The credentials that are required for the integration. To specify an AWS Identity and Access Management (IAM) role that API Gateway assumes, specify the role's Amazon Resource Name (ARN). To require that the caller's identity be passed through from the request, specify arn:aws:iam::*:user/*. To use resource-based permissions on the AWS Lambda (Lambda) function, don't specify this property. Use the `AWS::Lambda::Permission <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html>`_ resource to permit API Gateway to call the function. For more information, see `Allow Amazon API Gateway to Invoke a Lambda Function <https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#access-control-resource-based-example-apigateway-invoke-function>`_ in the *AWS Lambda Developer Guide* .
            :param integration_http_method: The integration's HTTP method type. For the ``Type`` property, if you specify ``MOCK`` , this property is optional. For all other types, you must specify this property.
            :param integration_responses: The response that API Gateway provides after a method's backend completes processing a request. API Gateway intercepts the response from the backend so that you can control how API Gateway surfaces backend responses. For example, you can map the backend status codes to codes that you define.
            :param passthrough_behavior: Indicates when API Gateway passes requests to the targeted backend. This behavior depends on the request's ``Content-Type`` header and whether you defined a mapping template for it. For more information and valid values, see the `passthroughBehavior <https://docs.aws.amazon.com/apigateway/latest/api/API_PutIntegration.html#passthroughBehavior>`_ field in the *API Gateway API Reference* .
            :param request_parameters: The request parameters that API Gateway sends with the backend request. Specify request parameters as key-value pairs (string-to-string mappings), with a destination as the key and a source as the value. Specify the destination by using the following pattern ``integration.request. *location* . *name*`` , where *location* is query string, path, or header, and *name* is a valid, unique parameter name. The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on their destination in the request.
            :param request_templates: A map of Apache Velocity templates that are applied on the request payload. The template that API Gateway uses is based on the value of the ``Content-Type`` header that's sent by the client. The content type value is the key, and the template is the value (specified as a string), such as the following snippet: ``"application/json": "{\\n \\"statusCode\\": 200\\n}"`` For more information about templates, see `API Gateway Mapping Template and Access Logging Variable Reference <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html>`_ in the *API Gateway Developer Guide* .
            :param timeout_in_millis: Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29 seconds.
            :param type: The type of backend that your method is running, such as ``HTTP`` or ``MOCK`` . For all of the valid values, see the `type <https://docs.aws.amazon.com/apigateway/latest/api/API_Integration.html#type>`_ property for the ``Integration`` resource in the *Amazon API Gateway REST API Reference* .
            :param uri: The Uniform Resource Identifier (URI) for the integration. If you specify ``HTTP`` for the ``Type`` property, specify the API endpoint URL. If you specify ``MOCK`` for the ``Type`` property, don't specify this property. If you specify ``AWS`` for the ``Type`` property, specify an AWS service that follows this form: arn:aws:apigateway: *region* : *subdomain* . *service|service* : *path|action* / *service_api* . For example, a Lambda function URI follows this form: arn:aws:apigateway: *region* :lambda:path/ *path* . The path is usually in the form /2015-03-31/functions/ *LambdaFunctionARN* /invocations. For more information, see the ``uri`` property of the `Integration <https://docs.aws.amazon.com/apigateway/latest/api/API_Integration.html>`_ resource in the Amazon API Gateway REST API Reference. If you specified ``HTTP`` or ``AWS`` for the ``Type`` property, you must specify this property.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                integration_property = apigateway.CfnMethod.IntegrationProperty(
                    cache_key_parameters=["cacheKeyParameters"],
                    cache_namespace="cacheNamespace",
                    connection_id="connectionId",
                    connection_type="connectionType",
                    content_handling="contentHandling",
                    credentials="credentials",
                    integration_http_method="integrationHttpMethod",
                    integration_responses=[apigateway.CfnMethod.IntegrationResponseProperty(
                        status_code="statusCode",
                
                        # the properties below are optional
                        content_handling="contentHandling",
                        response_parameters={
                            "response_parameters_key": "responseParameters"
                        },
                        response_templates={
                            "response_templates_key": "responseTemplates"
                        },
                        selection_pattern="selectionPattern"
                    )],
                    passthrough_behavior="passthroughBehavior",
                    request_parameters={
                        "request_parameters_key": "requestParameters"
                    },
                    request_templates={
                        "request_templates_key": "requestTemplates"
                    },
                    timeout_in_millis=123,
                    type="type",
                    uri="uri"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a7fbf4e789d8342e74d34b1a358a1eb692bb6933c9d226699c2f7f31f11d0907)
                check_type(argname="argument cache_key_parameters", value=cache_key_parameters, expected_type=type_hints["cache_key_parameters"])
                check_type(argname="argument cache_namespace", value=cache_namespace, expected_type=type_hints["cache_namespace"])
                check_type(argname="argument connection_id", value=connection_id, expected_type=type_hints["connection_id"])
                check_type(argname="argument connection_type", value=connection_type, expected_type=type_hints["connection_type"])
                check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
                check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
                check_type(argname="argument integration_http_method", value=integration_http_method, expected_type=type_hints["integration_http_method"])
                check_type(argname="argument integration_responses", value=integration_responses, expected_type=type_hints["integration_responses"])
                check_type(argname="argument passthrough_behavior", value=passthrough_behavior, expected_type=type_hints["passthrough_behavior"])
                check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
                check_type(argname="argument request_templates", value=request_templates, expected_type=type_hints["request_templates"])
                check_type(argname="argument timeout_in_millis", value=timeout_in_millis, expected_type=type_hints["timeout_in_millis"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cache_key_parameters is not None:
                self._values["cache_key_parameters"] = cache_key_parameters
            if cache_namespace is not None:
                self._values["cache_namespace"] = cache_namespace
            if connection_id is not None:
                self._values["connection_id"] = connection_id
            if connection_type is not None:
                self._values["connection_type"] = connection_type
            if content_handling is not None:
                self._values["content_handling"] = content_handling
            if credentials is not None:
                self._values["credentials"] = credentials
            if integration_http_method is not None:
                self._values["integration_http_method"] = integration_http_method
            if integration_responses is not None:
                self._values["integration_responses"] = integration_responses
            if passthrough_behavior is not None:
                self._values["passthrough_behavior"] = passthrough_behavior
            if request_parameters is not None:
                self._values["request_parameters"] = request_parameters
            if request_templates is not None:
                self._values["request_templates"] = request_templates
            if timeout_in_millis is not None:
                self._values["timeout_in_millis"] = timeout_in_millis
            if type is not None:
                self._values["type"] = type
            if uri is not None:
                self._values["uri"] = uri

        @builtins.property
        def cache_key_parameters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of request parameters whose values API Gateway caches.

            For cases where the integration type allows for RequestParameters to be set, these parameters must also be specified in `RequestParameters <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestparameters>`_ to be supported in ``CacheKeyParameters`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-cachekeyparameters
            '''
            result = self._values.get("cache_key_parameters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cache_namespace(self) -> typing.Optional[builtins.str]:
            '''An API-specific tag group of related cached parameters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-cachenamespace
            '''
            result = self._values.get("cache_namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def connection_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the ``VpcLink`` used for the integration when ``connectionType=VPC_LINK`` , otherwise undefined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-connectionid
            '''
            result = self._values.get("connection_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def connection_type(self) -> typing.Optional[builtins.str]:
            '''The type of the network connection to the integration endpoint.

            The valid value is ``INTERNET`` for connections through the public routable internet or ``VPC_LINK`` for private connections between API Gateway and a network load balancer in a VPC. The default value is ``INTERNET`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-connectiontype
            '''
            result = self._values.get("connection_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def content_handling(self) -> typing.Optional[builtins.str]:
            '''Specifies how to handle request payload content type conversions. Valid values are:.

            - ``CONVERT_TO_BINARY`` : Converts a request payload from a base64-encoded string to a binary blob.
            - ``CONVERT_TO_TEXT`` : Converts a request payload from a binary blob to a base64-encoded string.

            If this property isn't defined, the request payload is passed through from the method request to the integration request without modification, provided that the ``PassthroughBehaviors`` property is configured to support payload pass-through.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-contenthandling
            '''
            result = self._values.get("content_handling")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def credentials(self) -> typing.Optional[builtins.str]:
            '''The credentials that are required for the integration.

            To specify an AWS Identity and Access Management (IAM) role that API Gateway assumes, specify the role's Amazon Resource Name (ARN). To require that the caller's identity be passed through from the request, specify arn:aws:iam::*:user/*.

            To use resource-based permissions on the AWS Lambda (Lambda) function, don't specify this property. Use the `AWS::Lambda::Permission <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html>`_ resource to permit API Gateway to call the function. For more information, see `Allow Amazon API Gateway to Invoke a Lambda Function <https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#access-control-resource-based-example-apigateway-invoke-function>`_ in the *AWS Lambda Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-credentials
            '''
            result = self._values.get("credentials")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def integration_http_method(self) -> typing.Optional[builtins.str]:
            '''The integration's HTTP method type.

            For the ``Type`` property, if you specify ``MOCK`` , this property is optional. For all other types, you must specify this property.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-integrationhttpmethod
            '''
            result = self._values.get("integration_http_method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def integration_responses(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnMethod.IntegrationResponseProperty", _IResolvable_a771d0ef]]]]:
            '''The response that API Gateway provides after a method's backend completes processing a request.

            API Gateway intercepts the response from the backend so that you can control how API Gateway surfaces backend responses. For example, you can map the backend status codes to codes that you define.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-integrationresponses
            '''
            result = self._values.get("integration_responses")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnMethod.IntegrationResponseProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def passthrough_behavior(self) -> typing.Optional[builtins.str]:
            '''Indicates when API Gateway passes requests to the targeted backend.

            This behavior depends on the request's ``Content-Type`` header and whether you defined a mapping template for it.

            For more information and valid values, see the `passthroughBehavior <https://docs.aws.amazon.com/apigateway/latest/api/API_PutIntegration.html#passthroughBehavior>`_ field in the *API Gateway API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-passthroughbehavior
            '''
            result = self._values.get("passthrough_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def request_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''The request parameters that API Gateway sends with the backend request.

            Specify request parameters as key-value pairs (string-to-string mappings), with a destination as the key and a source as the value.

            Specify the destination by using the following pattern ``integration.request. *location* . *name*`` , where *location* is query string, path, or header, and *name* is a valid, unique parameter name.

            The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on their destination in the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-requestparameters
            '''
            result = self._values.get("request_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def request_templates(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''A map of Apache Velocity templates that are applied on the request payload.

            The template that API Gateway uses is based on the value of the ``Content-Type`` header that's sent by the client. The content type value is the key, and the template is the value (specified as a string), such as the following snippet:

            ``"application/json": "{\\n \\"statusCode\\": 200\\n}"``

            For more information about templates, see `API Gateway Mapping Template and Access Logging Variable Reference <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-requesttemplates
            '''
            result = self._values.get("request_templates")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def timeout_in_millis(self) -> typing.Optional[jsii.Number]:
            '''Custom timeout between 50 and 29,000 milliseconds.

            The default value is 29,000 milliseconds or 29 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-timeoutinmillis
            '''
            result = self._values.get("timeout_in_millis")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The type of backend that your method is running, such as ``HTTP`` or ``MOCK`` .

            For all of the valid values, see the `type <https://docs.aws.amazon.com/apigateway/latest/api/API_Integration.html#type>`_ property for the ``Integration`` resource in the *Amazon API Gateway REST API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def uri(self) -> typing.Optional[builtins.str]:
            '''The Uniform Resource Identifier (URI) for the integration.

            If you specify ``HTTP`` for the ``Type`` property, specify the API endpoint URL.

            If you specify ``MOCK`` for the ``Type`` property, don't specify this property.

            If you specify ``AWS`` for the ``Type`` property, specify an AWS service that follows this form: arn:aws:apigateway: *region* : *subdomain* . *service|service* : *path|action* / *service_api* . For example, a Lambda function URI follows this form: arn:aws:apigateway: *region* :lambda:path/ *path* . The path is usually in the form /2015-03-31/functions/ *LambdaFunctionARN* /invocations. For more information, see the ``uri`` property of the `Integration <https://docs.aws.amazon.com/apigateway/latest/api/API_Integration.html>`_ resource in the Amazon API Gateway REST API Reference.

            If you specified ``HTTP`` or ``AWS`` for the ``Type`` property, you must specify this property.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html#cfn-apigateway-method-integration-uri
            '''
            result = self._values.get("uri")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IntegrationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnMethod.IntegrationResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "status_code": "statusCode",
            "content_handling": "contentHandling",
            "response_parameters": "responseParameters",
            "response_templates": "responseTemplates",
            "selection_pattern": "selectionPattern",
        },
    )
    class IntegrationResponseProperty:
        def __init__(
            self,
            *,
            status_code: builtins.str,
            content_handling: typing.Optional[builtins.str] = None,
            response_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            response_templates: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            selection_pattern: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``IntegrationResponse`` is a property of the `Amazon API Gateway Method Integration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html>`_ property type that specifies the response that API Gateway sends after a method's backend finishes processing a request.

            :param status_code: The status code that API Gateway uses to map the integration response to a `MethodResponse <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html>`_ status code.
            :param content_handling: Specifies how to handle request payload content type conversions. Valid values are:. - ``CONVERT_TO_BINARY`` : Converts a request payload from a base64-encoded string to a binary blob. - ``CONVERT_TO_TEXT`` : Converts a request payload from a binary blob to a base64-encoded string. If this property isn't defined, the request payload is passed through from the method request to the integration request without modification.
            :param response_parameters: The response parameters from the backend response that API Gateway sends to the method response. Specify response parameters as key-value pairs ( `string-to-string mappings <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html>`_ ). Use the destination as the key and the source as the value: - The destination must be an existing response parameter in the `MethodResponse <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html>`_ property. - The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on the destination specified in the request. For more information about templates, see `API Gateway Mapping Template and Access Logging Variable Reference <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html>`_ in the *API Gateway Developer Guide* .
            :param response_templates: The templates that are used to transform the integration response body. Specify templates as key-value pairs (string-to-string mappings), with a content type as the key and a template as the value. For more information, see `API Gateway Mapping Template and Access Logging Variable Reference <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html>`_ in the *API Gateway Developer Guide* .
            :param selection_pattern: A `regular expression <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-regexes.html>`_ that specifies which error strings or status codes from the backend map to the integration response.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                integration_response_property = apigateway.CfnMethod.IntegrationResponseProperty(
                    status_code="statusCode",
                
                    # the properties below are optional
                    content_handling="contentHandling",
                    response_parameters={
                        "response_parameters_key": "responseParameters"
                    },
                    response_templates={
                        "response_templates_key": "responseTemplates"
                    },
                    selection_pattern="selectionPattern"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bc5e87ba872865eefba0bd59de14442c6d188e1b91eb779af8572e886375d6bf)
                check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
                check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
                check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
                check_type(argname="argument response_templates", value=response_templates, expected_type=type_hints["response_templates"])
                check_type(argname="argument selection_pattern", value=selection_pattern, expected_type=type_hints["selection_pattern"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "status_code": status_code,
            }
            if content_handling is not None:
                self._values["content_handling"] = content_handling
            if response_parameters is not None:
                self._values["response_parameters"] = response_parameters
            if response_templates is not None:
                self._values["response_templates"] = response_templates
            if selection_pattern is not None:
                self._values["selection_pattern"] = selection_pattern

        @builtins.property
        def status_code(self) -> builtins.str:
            '''The status code that API Gateway uses to map the integration response to a `MethodResponse <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html>`_ status code.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html#cfn-apigateway-method-integration-integrationresponse-statuscode
            '''
            result = self._values.get("status_code")
            assert result is not None, "Required property 'status_code' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def content_handling(self) -> typing.Optional[builtins.str]:
            '''Specifies how to handle request payload content type conversions. Valid values are:.

            - ``CONVERT_TO_BINARY`` : Converts a request payload from a base64-encoded string to a binary blob.
            - ``CONVERT_TO_TEXT`` : Converts a request payload from a binary blob to a base64-encoded string.

            If this property isn't defined, the request payload is passed through from the method request to the integration request without modification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html#cfn-apigateway-method-integrationresponse-contenthandling
            '''
            result = self._values.get("content_handling")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def response_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''The response parameters from the backend response that API Gateway sends to the method response.

            Specify response parameters as key-value pairs ( `string-to-string mappings <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html>`_ ).

            Use the destination as the key and the source as the value:

            - The destination must be an existing response parameter in the `MethodResponse <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html>`_ property.
            - The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on the destination specified in the request.

            For more information about templates, see `API Gateway Mapping Template and Access Logging Variable Reference <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html#cfn-apigateway-method-integration-integrationresponse-responseparameters
            '''
            result = self._values.get("response_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def response_templates(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''The templates that are used to transform the integration response body.

            Specify templates as key-value pairs (string-to-string mappings), with a content type as the key and a template as the value. For more information, see `API Gateway Mapping Template and Access Logging Variable Reference <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html#cfn-apigateway-method-integration-integrationresponse-responsetemplates
            '''
            result = self._values.get("response_templates")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def selection_pattern(self) -> typing.Optional[builtins.str]:
            '''A `regular expression <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-regexes.html>`_ that specifies which error strings or status codes from the backend map to the integration response.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html#cfn-apigateway-method-integration-integrationresponse-selectionpattern
            '''
            result = self._values.get("selection_pattern")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IntegrationResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnMethod.MethodResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "status_code": "statusCode",
            "response_models": "responseModels",
            "response_parameters": "responseParameters",
        },
    )
    class MethodResponseProperty:
        def __init__(
            self,
            *,
            status_code: builtins.str,
            response_models: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            response_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''``MethodResponse`` is a property of the `AWS::ApiGateway::Method <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html>`_ resource that defines the responses that can be sent to the client that calls a method.

            :param status_code: The method response's status code, which you map to an `IntegrationResponse <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html>`_ .
            :param response_models: The resources used for the response's content type. Specify response models as key-value pairs (string-to-string maps), with a content type as the key and a `Model <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html>`_ resource name as the value.
            :param response_parameters: Response parameters that API Gateway sends to the client that called a method. Specify response parameters as key-value pairs (string-to-Boolean maps), with a destination as the key and a Boolean as the value. Specify the destination using the following pattern: ``method.response.header. *name*`` , where *name* is a valid, unique header name. The Boolean specifies whether a parameter is required.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                method_response_property = apigateway.CfnMethod.MethodResponseProperty(
                    status_code="statusCode",
                
                    # the properties below are optional
                    response_models={
                        "response_models_key": "responseModels"
                    },
                    response_parameters={
                        "response_parameters_key": False
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e034a695da833353688529408ac334dbf9128a84976f1c4aa28df2b4682f3ae1)
                check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
                check_type(argname="argument response_models", value=response_models, expected_type=type_hints["response_models"])
                check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "status_code": status_code,
            }
            if response_models is not None:
                self._values["response_models"] = response_models
            if response_parameters is not None:
                self._values["response_parameters"] = response_parameters

        @builtins.property
        def status_code(self) -> builtins.str:
            '''The method response's status code, which you map to an `IntegrationResponse <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration-integrationresponse.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html#cfn-apigateway-method-methodresponse-statuscode
            '''
            result = self._values.get("status_code")
            assert result is not None, "Required property 'status_code' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def response_models(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''The resources used for the response's content type.

            Specify response models as key-value pairs (string-to-string maps), with a content type as the key and a `Model <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html>`_ resource name as the value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html#cfn-apigateway-method-methodresponse-responsemodels
            '''
            result = self._values.get("response_models")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def response_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]]:
            '''Response parameters that API Gateway sends to the client that called a method.

            Specify response parameters as key-value pairs (string-to-Boolean maps), with a destination as the key and a Boolean as the value. Specify the destination using the following pattern: ``method.response.header. *name*`` , where *name* is a valid, unique header name. The Boolean specifies whether a parameter is required.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-methodresponse.html#cfn-apigateway-method-methodresponse-responseparameters
            '''
            result = self._values.get("response_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MethodResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnMethodProps",
    jsii_struct_bases=[],
    name_mapping={
        "http_method": "httpMethod",
        "resource_id": "resourceId",
        "rest_api_id": "restApiId",
        "api_key_required": "apiKeyRequired",
        "authorization_scopes": "authorizationScopes",
        "authorization_type": "authorizationType",
        "authorizer_id": "authorizerId",
        "integration": "integration",
        "method_responses": "methodResponses",
        "operation_name": "operationName",
        "request_models": "requestModels",
        "request_parameters": "requestParameters",
        "request_validator_id": "requestValidatorId",
    },
)
class CfnMethodProps:
    def __init__(
        self,
        *,
        http_method: builtins.str,
        resource_id: builtins.str,
        rest_api_id: builtins.str,
        api_key_required: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[builtins.str] = None,
        authorizer_id: typing.Optional[builtins.str] = None,
        integration: typing.Optional[typing.Union[typing.Union[CfnMethod.IntegrationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        method_responses: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union[CfnMethod.MethodResponseProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        request_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]] = None,
        request_validator_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnMethod``.

        :param http_method: The HTTP method that clients use to call this method.
        :param resource_id: The ID of an API Gateway `resource <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html>`_ . For root resource methods, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .
        :param rest_api_id: The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which API Gateway creates the method.
        :param api_key_required: Indicates whether the method requires clients to submit a valid API key.
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes match a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.
        :param authorization_type: The method's authorization type. This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* . .. epigraph:: If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.
        :param authorizer_id: The identifier of the `authorizer <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html>`_ to use on this method. If you specify this property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for the ``AuthorizationType`` property.
        :param integration: The backend system that the method calls when it receives a request.
        :param method_responses: The responses that can be sent to the client who calls the method.
        :param operation_name: A friendly operation name for the method. For example, you can assign the ``OperationName`` of ``ListPets`` for the ``GET /pets`` method.
        :param request_models: The resources that are used for the request's content type. Specify request models as key-value pairs (string-to-string mapping), with a content type as the key and a ``Model`` resource name as the value. To use the same model regardless of the content type, specify ``$default`` as the key.
        :param request_parameters: The request parameters that API Gateway accepts. Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format ``method.request. *location* . *name*`` , where the location is querystring, path, or header, and *name* is a valid, unique parameter name.
        :param request_validator_id: The ID of the associated request validator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_method_props = apigateway.CfnMethodProps(
                http_method="httpMethod",
                resource_id="resourceId",
                rest_api_id="restApiId",
            
                # the properties below are optional
                api_key_required=False,
                authorization_scopes=["authorizationScopes"],
                authorization_type="authorizationType",
                authorizer_id="authorizerId",
                integration=apigateway.CfnMethod.IntegrationProperty(
                    cache_key_parameters=["cacheKeyParameters"],
                    cache_namespace="cacheNamespace",
                    connection_id="connectionId",
                    connection_type="connectionType",
                    content_handling="contentHandling",
                    credentials="credentials",
                    integration_http_method="integrationHttpMethod",
                    integration_responses=[apigateway.CfnMethod.IntegrationResponseProperty(
                        status_code="statusCode",
            
                        # the properties below are optional
                        content_handling="contentHandling",
                        response_parameters={
                            "response_parameters_key": "responseParameters"
                        },
                        response_templates={
                            "response_templates_key": "responseTemplates"
                        },
                        selection_pattern="selectionPattern"
                    )],
                    passthrough_behavior="passthroughBehavior",
                    request_parameters={
                        "request_parameters_key": "requestParameters"
                    },
                    request_templates={
                        "request_templates_key": "requestTemplates"
                    },
                    timeout_in_millis=123,
                    type="type",
                    uri="uri"
                ),
                method_responses=[apigateway.CfnMethod.MethodResponseProperty(
                    status_code="statusCode",
            
                    # the properties below are optional
                    response_models={
                        "response_models_key": "responseModels"
                    },
                    response_parameters={
                        "response_parameters_key": False
                    }
                )],
                operation_name="operationName",
                request_models={
                    "request_models_key": "requestModels"
                },
                request_parameters={
                    "request_parameters_key": False
                },
                request_validator_id="requestValidatorId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__455d93d1c9465809fbb81b7d186f5b7f2f98524df623c547d9da7f2e0b1e979c)
            check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
            check_type(argname="argument resource_id", value=resource_id, expected_type=type_hints["resource_id"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument api_key_required", value=api_key_required, expected_type=type_hints["api_key_required"])
            check_type(argname="argument authorization_scopes", value=authorization_scopes, expected_type=type_hints["authorization_scopes"])
            check_type(argname="argument authorization_type", value=authorization_type, expected_type=type_hints["authorization_type"])
            check_type(argname="argument authorizer_id", value=authorizer_id, expected_type=type_hints["authorizer_id"])
            check_type(argname="argument integration", value=integration, expected_type=type_hints["integration"])
            check_type(argname="argument method_responses", value=method_responses, expected_type=type_hints["method_responses"])
            check_type(argname="argument operation_name", value=operation_name, expected_type=type_hints["operation_name"])
            check_type(argname="argument request_models", value=request_models, expected_type=type_hints["request_models"])
            check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
            check_type(argname="argument request_validator_id", value=request_validator_id, expected_type=type_hints["request_validator_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "http_method": http_method,
            "resource_id": resource_id,
            "rest_api_id": rest_api_id,
        }
        if api_key_required is not None:
            self._values["api_key_required"] = api_key_required
        if authorization_scopes is not None:
            self._values["authorization_scopes"] = authorization_scopes
        if authorization_type is not None:
            self._values["authorization_type"] = authorization_type
        if authorizer_id is not None:
            self._values["authorizer_id"] = authorizer_id
        if integration is not None:
            self._values["integration"] = integration
        if method_responses is not None:
            self._values["method_responses"] = method_responses
        if operation_name is not None:
            self._values["operation_name"] = operation_name
        if request_models is not None:
            self._values["request_models"] = request_models
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if request_validator_id is not None:
            self._values["request_validator_id"] = request_validator_id

    @builtins.property
    def http_method(self) -> builtins.str:
        '''The HTTP method that clients use to call this method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-httpmethod
        '''
        result = self._values.get("http_method")
        assert result is not None, "Required property 'http_method' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def resource_id(self) -> builtins.str:
        '''The ID of an API Gateway `resource <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html>`_ . For root resource methods, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-resourceid
        '''
        result = self._values.get("resource_id")
        assert result is not None, "Required property 'resource_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which API Gateway creates the method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_key_required(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether the method requires clients to submit a valid API key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-apikeyrequired
        '''
        result = self._values.get("api_key_required")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of authorization scopes configured on the method.

        The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes match a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes
        '''
        result = self._values.get("authorization_scopes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def authorization_type(self) -> typing.Optional[builtins.str]:
        '''The method's authorization type.

        This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* .
        .. epigraph::

           If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationtype
        '''
        result = self._values.get("authorization_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of the `authorizer <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html>`_ to use on this method. If you specify this property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for the ``AuthorizationType`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizerid
        '''
        result = self._values.get("authorizer_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration(
        self,
    ) -> typing.Optional[typing.Union[CfnMethod.IntegrationProperty, _IResolvable_a771d0ef]]:
        '''The backend system that the method calls when it receives a request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-integration
        '''
        result = self._values.get("integration")
        return typing.cast(typing.Optional[typing.Union[CfnMethod.IntegrationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def method_responses(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnMethod.MethodResponseProperty, _IResolvable_a771d0ef]]]]:
        '''The responses that can be sent to the client who calls the method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-methodresponses
        '''
        result = self._values.get("method_responses")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnMethod.MethodResponseProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''A friendly operation name for the method.

        For example, you can assign the ``OperationName`` of ``ListPets`` for the ``GET /pets`` method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-operationname
        '''
        result = self._values.get("operation_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_models(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''The resources that are used for the request's content type.

        Specify request models as key-value pairs (string-to-string mapping), with a content type as the key and a ``Model`` resource name as the value. To use the same model regardless of the content type, specify ``$default`` as the key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestmodels
        '''
        result = self._values.get("request_models")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

    @builtins.property
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]]:
        '''The request parameters that API Gateway accepts.

        Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format ``method.request. *location* . *name*`` , where the location is querystring, path, or header, and *name* is a valid, unique parameter name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestparameters
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def request_validator_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the associated request validator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestvalidatorid
        '''
        result = self._values.get("request_validator_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMethodProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnModel(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnModel",
):
    '''A CloudFormation ``AWS::ApiGateway::Model``.

    The ``AWS::ApiGateway::Model`` resource defines the structure of a request or response payload for an API method.

    :cloudformationResource: AWS::ApiGateway::Model
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # schema: Any
        
        cfn_model = apigateway.CfnModel(self, "MyCfnModel",
            rest_api_id="restApiId",
        
            # the properties below are optional
            content_type="contentType",
            description="description",
            name="name",
            schema=schema
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        schema: typing.Any = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Model``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param rest_api_id: The ID of a REST API with which to associate this model.
        :param content_type: The content type for the model.
        :param description: A description that identifies this model.
        :param name: A name for the model. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param schema: The schema to use to transform data to one or more output formats. Specify null ( ``{}`` ) if you don't want to specify a schema.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74de826aad14d77bcf079bf63dadb53868aec01293f40515161a0d90850ffc52)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnModelProps(
            rest_api_id=rest_api_id,
            content_type=content_type,
            description=description,
            name=name,
            schema=schema,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0a6190ef14229b68010b5808feea58e1cdebf79344617fb4c40889fa6622592a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__acca8e397653fe7635e1db72a1c722fd803a9824819ff69ca75be039159b4772)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of a REST API with which to associate this model.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e64369264a1acd3e5efc312fd0d9cbf514cdae657a457522610d40e46a8d864)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="schema")
    def schema(self) -> typing.Any:
        '''The schema to use to transform data to one or more output formats.

        Specify null ( ``{}`` ) if you don't want to specify a schema.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-schema
        '''
        return typing.cast(typing.Any, jsii.get(self, "schema"))

    @schema.setter
    def schema(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f62cda7b2e7b533ef4587a616ea3c763939caa133511b0c8c1300feb30a3cdf3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schema", value)

    @builtins.property
    @jsii.member(jsii_name="contentType")
    def content_type(self) -> typing.Optional[builtins.str]:
        '''The content type for the model.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-contenttype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "contentType"))

    @content_type.setter
    def content_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55427aa8025b4793d9cdf9dd8cb89c875a6f6ecdfe14dd500e3b1716cf1ba6fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "contentType", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description that identifies this model.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b36c3d1403afcf38f354308805d57f6dec7cfe578eba85108c16f5e1635d54a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the model.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .
        .. epigraph::

           If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c0ab9ea8e604fcf8cc391b56324ed158e1f907181b08c4d19258cea4a65087f7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnModelProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "content_type": "contentType",
        "description": "description",
        "name": "name",
        "schema": "schema",
    },
)
class CfnModelProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        schema: typing.Any = None,
    ) -> None:
        '''Properties for defining a ``CfnModel``.

        :param rest_api_id: The ID of a REST API with which to associate this model.
        :param content_type: The content type for the model.
        :param description: A description that identifies this model.
        :param name: A name for the model. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param schema: The schema to use to transform data to one or more output formats. Specify null ( ``{}`` ) if you don't want to specify a schema.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # schema: Any
            
            cfn_model_props = apigateway.CfnModelProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                content_type="contentType",
                description="description",
                name="name",
                schema=schema
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9dc921f463cf838f7ef61e10066ff3467fbdd95bc2d511d17ef8ae7628010eb1)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if content_type is not None:
            self._values["content_type"] = content_type
        if description is not None:
            self._values["description"] = description
        if name is not None:
            self._values["name"] = name
        if schema is not None:
            self._values["schema"] = schema

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The ID of a REST API with which to associate this model.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''The content type for the model.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-contenttype
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description that identifies this model.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the model.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .
        .. epigraph::

           If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def schema(self) -> typing.Any:
        '''The schema to use to transform data to one or more output formats.

        Specify null ( ``{}`` ) if you don't want to specify a schema.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-schema
        '''
        result = self._values.get("schema")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnModelProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnModelV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnModelV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Model``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Model
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # schema: Any
        
        cfn_model_v2 = apigateway.CfnModelV2(self, "MyCfnModelV2",
            api_id="apiId",
            name="name",
            schema=schema,
        
            # the properties below are optional
            content_type="contentType",
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        name: builtins.str,
        schema: typing.Any,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Model``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Model.ApiId``.
        :param name: (deprecated) ``AWS::ApiGatewayV2::Model.Name``.
        :param schema: (deprecated) ``AWS::ApiGatewayV2::Model.Schema``.
        :param content_type: (deprecated) ``AWS::ApiGatewayV2::Model.ContentType``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Model.Description``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e6a4b98a8f8689b7d015970c44d8b771c331c464bf9fdfac09525130002c6652)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnModelV2Props(
            api_id=api_id,
            name=name,
            schema=schema,
            content_type=content_type,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5614bcc4bfe3ec9353d390eedbb871d40062bef6c8c38191c4c1aaf4d25491b7)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f665a32f57ab643c00747bb23372a85436192ae705eafd1bedc83aceeeaffff)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df597a2eb560bc20f94897a605cedf3b1c972e8c79ef76b1eb1f4b94c5b54bbe)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.Name``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-name
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afa1cf6dc41124d356252d7c223f8d26a821ee880d87f805a4e9b52b322612a0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="schema")
    def schema(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.Schema``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-schema
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "schema"))

    @schema.setter
    def schema(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8848db1d771eccf3ab090a8f4b1435ba198e963547fad32a73cda3f58eb55459)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schema", value)

    @builtins.property
    @jsii.member(jsii_name="contentType")
    def content_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.ContentType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-contenttype
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "contentType"))

    @content_type.setter
    def content_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__186eec014e56076e69be4f284aeaf3899a070e4e545aae6ceba4999da99f7803)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "contentType", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-description
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5084f01b500298a9e2f57b5100bf927e2aa313b8262d6823e9c13ebe2311bf87)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnModelV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "name": "name",
        "schema": "schema",
        "content_type": "contentType",
        "description": "description",
    },
)
class CfnModelV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        name: builtins.str,
        schema: typing.Any,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Model``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Model.ApiId``.
        :param name: (deprecated) ``AWS::ApiGatewayV2::Model.Name``.
        :param schema: (deprecated) ``AWS::ApiGatewayV2::Model.Schema``.
        :param content_type: (deprecated) ``AWS::ApiGatewayV2::Model.ContentType``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Model.Description``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # schema: Any
            
            cfn_model_v2_props = apigateway.CfnModelV2Props(
                api_id="apiId",
                name="name",
                schema=schema,
            
                # the properties below are optional
                content_type="contentType",
                description="description"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfece9780a0c31572f060515db2358504cebe7f88f404a968e795bc62b5f690f)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "name": name,
            "schema": schema,
        }
        if content_type is not None:
            self._values["content_type"] = content_type
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.Name``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-name
        :stability: deprecated
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def schema(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.Schema``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-schema
        :stability: deprecated
        '''
        result = self._values.get("schema")
        assert result is not None, "Required property 'schema' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.ContentType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-contenttype
        :stability: deprecated
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Model.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-model.html#cfn-apigatewayv2-model-description
        :stability: deprecated
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnModelV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnRequestValidator(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnRequestValidator",
):
    '''A CloudFormation ``AWS::ApiGateway::RequestValidator``.

    The ``AWS::ApiGateway::RequestValidator`` resource sets up basic validation rules for incoming requests to your API. For more information, see `Enable Basic Request Validation for an API in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-method-request-validation.html>`_ in the *API Gateway Developer Guide* .

    :cloudformationResource: AWS::ApiGateway::RequestValidator
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_request_validator = apigateway.CfnRequestValidator(self, "MyCfnRequestValidator",
            rest_api_id="restApiId",
        
            # the properties below are optional
            name="name",
            validate_request_body=False,
            validate_request_parameters=False
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        name: typing.Optional[builtins.str] = None,
        validate_request_body: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        validate_request_parameters: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::RequestValidator``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param rest_api_id: The identifier of the targeted API entity.
        :param name: The name of this request validator.
        :param validate_request_body: Indicates whether to validate the request body according to the configured schema for the targeted API and method.
        :param validate_request_parameters: Indicates whether to validate request parameters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa983eaa0728de3c93293714e821d4992dc8583aa89669531b9897503ac81cc2)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRequestValidatorProps(
            rest_api_id=rest_api_id,
            name=name,
            validate_request_body=validate_request_body,
            validate_request_parameters=validate_request_parameters,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8521bf829b78748800853bbaafe3ece70d90423c656c4a8014dacb39512a44c0)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2797fa4f9b632b9b7ac4489bb7daf21e6ced42e999817d0970fb5d66495ead56)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrRequestValidatorId")
    def attr_request_validator_id(self) -> builtins.str:
        '''The ID for the request validator.

        For example: ``abc123`` .

        :cloudformationAttribute: RequestValidatorId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRequestValidatorId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eb75ed1004913f3654350b357d0f95dd8a6acb8c940230640609dc8f025cf102)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this request validator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__97cb7f711f4839bbf6386e9254ab8a50a6b0771e8939b31dc23a13cc9f5be2c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="validateRequestBody")
    def validate_request_body(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether to validate the request body according to the configured schema for the targeted API and method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-validaterequestbody
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "validateRequestBody"))

    @validate_request_body.setter
    def validate_request_body(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a6a192348ccbb483ad5548d49042126e883c5db04a8994fab6856dba9be4e9a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validateRequestBody", value)

    @builtins.property
    @jsii.member(jsii_name="validateRequestParameters")
    def validate_request_parameters(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether to validate request parameters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-validaterequestparameters
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "validateRequestParameters"))

    @validate_request_parameters.setter
    def validate_request_parameters(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3a40ab49fdfb25719dcf4fb88dafcb79a2e9105d43da60b0623ce9acb56cf92)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validateRequestParameters", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnRequestValidatorProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "name": "name",
        "validate_request_body": "validateRequestBody",
        "validate_request_parameters": "validateRequestParameters",
    },
)
class CfnRequestValidatorProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        name: typing.Optional[builtins.str] = None,
        validate_request_body: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        validate_request_parameters: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRequestValidator``.

        :param rest_api_id: The identifier of the targeted API entity.
        :param name: The name of this request validator.
        :param validate_request_body: Indicates whether to validate the request body according to the configured schema for the targeted API and method.
        :param validate_request_parameters: Indicates whether to validate request parameters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_request_validator_props = apigateway.CfnRequestValidatorProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                name="name",
                validate_request_body=False,
                validate_request_parameters=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4984176963cf9dc18dc7225ebc20e2993f43b5a33b32527f6dceb43342849979)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument validate_request_body", value=validate_request_body, expected_type=type_hints["validate_request_body"])
            check_type(argname="argument validate_request_parameters", value=validate_request_parameters, expected_type=type_hints["validate_request_parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if name is not None:
            self._values["name"] = name
        if validate_request_body is not None:
            self._values["validate_request_body"] = validate_request_body
        if validate_request_parameters is not None:
            self._values["validate_request_parameters"] = validate_request_parameters

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The identifier of the targeted API entity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this request validator.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def validate_request_body(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether to validate the request body according to the configured schema for the targeted API and method.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-validaterequestbody
        '''
        result = self._values.get("validate_request_body")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def validate_request_parameters(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether to validate request parameters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-validaterequestparameters
        '''
        result = self._values.get("validate_request_parameters")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRequestValidatorProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnResource(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnResource",
):
    '''A CloudFormation ``AWS::ApiGateway::Resource``.

    The ``AWS::ApiGateway::Resource`` resource creates a resource in an API.

    :cloudformationResource: AWS::ApiGateway::Resource
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_resource = apigateway.CfnResource(self, "MyCfnResource",
            parent_id="parentId",
            path_part="pathPart",
            rest_api_id="restApiId"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        parent_id: builtins.str,
        path_part: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Resource``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param parent_id: If you want to create a child resource, the ID of the parent resource. For resources without a parent, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .
        :param path_part: A path name for the resource.
        :param rest_api_id: The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which you want to create this resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d80a7bef28348a41fc4ae24fce45fb6b1b608b76438a753229887dc1c818fca8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnResourceProps(
            parent_id=parent_id, path_part=path_part, rest_api_id=rest_api_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63b96f4343abf4fba3a29b9e5bacb19596c1f7d991d8601931571529217fa9ca)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e51cde8f8d1741bb297bdcb8406588a210081ecbb588b5fa2fb89364d959b072)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceId")
    def attr_resource_id(self) -> builtins.str:
        '''The ID for the resource.

        For example: ``abc123`` .

        :cloudformationAttribute: ResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="parentId")
    def parent_id(self) -> builtins.str:
        '''If you want to create a child resource, the ID of the parent resource.

        For resources without a parent, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-parentid
        '''
        return typing.cast(builtins.str, jsii.get(self, "parentId"))

    @parent_id.setter
    def parent_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5ce75edea3894015df3261624e453bb5eefd13be45df631bd4f0ba39be7291d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parentId", value)

    @builtins.property
    @jsii.member(jsii_name="pathPart")
    def path_part(self) -> builtins.str:
        '''A path name for the resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-pathpart
        '''
        return typing.cast(builtins.str, jsii.get(self, "pathPart"))

    @path_part.setter
    def path_part(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7cc4c503d0536d71d0aae889714685315d05b451b169117e6985b2cb35d0a58)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "pathPart", value)

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which you want to create this resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb9a9f62166c6ac73723654f471c6b7fd5fd4d6bfe4f8cddc0b1eecc624b6045)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnResourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "parent_id": "parentId",
        "path_part": "pathPart",
        "rest_api_id": "restApiId",
    },
)
class CfnResourceProps:
    def __init__(
        self,
        *,
        parent_id: builtins.str,
        path_part: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnResource``.

        :param parent_id: If you want to create a child resource, the ID of the parent resource. For resources without a parent, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .
        :param path_part: A path name for the resource.
        :param rest_api_id: The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which you want to create this resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_resource_props = apigateway.CfnResourceProps(
                parent_id="parentId",
                path_part="pathPart",
                rest_api_id="restApiId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__381d3f80881590d09ef79e91a2eb320cc97df4f39818ce78f0c817b40e946be8)
            check_type(argname="argument parent_id", value=parent_id, expected_type=type_hints["parent_id"])
            check_type(argname="argument path_part", value=path_part, expected_type=type_hints["path_part"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "parent_id": parent_id,
            "path_part": path_part,
            "rest_api_id": rest_api_id,
        }

    @builtins.property
    def parent_id(self) -> builtins.str:
        '''If you want to create a child resource, the ID of the parent resource.

        For resources without a parent, specify the ``RestApi`` root resource ID, such as ``{ "Fn::GetAtt": ["MyRestApi", "RootResourceId"] }`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-parentid
        '''
        result = self._values.get("parent_id")
        assert result is not None, "Required property 'parent_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def path_part(self) -> builtins.str:
        '''A path name for the resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-pathpart
        '''
        result = self._values.get("path_part")
        assert result is not None, "Required property 'path_part' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The ID of the `RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource in which you want to create this resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnRestApi(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnRestApi",
):
    '''A CloudFormation ``AWS::ApiGateway::RestApi``.

    The ``AWS::ApiGateway::RestApi`` resource creates a REST API. For more information, see `restapi:create <https://docs.aws.amazon.com/apigateway/latest/api/API_CreateRestApi.html>`_ in the *Amazon API Gateway REST API Reference* .
    .. epigraph::

       On January 1, 2016, the Swagger Specification was donated to the `OpenAPI initiative <https://docs.aws.amazon.com/https://www.openapis.org/>`_ , becoming the foundation of the OpenAPI Specification.

    :cloudformationResource: AWS::ApiGateway::RestApi
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # body: Any
        # policy: Any
        
        cfn_rest_api = apigateway.CfnRestApi(self, "MyCfnRestApi",
            api_key_source_type="apiKeySourceType",
            binary_media_types=["binaryMediaTypes"],
            body=body,
            body_s3_location=apigateway.CfnRestApi.S3LocationProperty(
                bucket="bucket",
                e_tag="eTag",
                key="key",
                version="version"
            ),
            clone_from="cloneFrom",
            description="description",
            disable_execute_api_endpoint=False,
            endpoint_configuration=apigateway.CfnRestApi.EndpointConfigurationProperty(
                types=["types"],
                vpc_endpoint_ids=["vpcEndpointIds"]
            ),
            fail_on_warnings=False,
            minimum_compression_size=123,
            mode="mode",
            name="name",
            parameters={
                "parameters_key": "parameters"
            },
            policy=policy,
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_key_source_type: typing.Optional[builtins.str] = None,
        binary_media_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        body: typing.Any = None,
        body_s3_location: typing.Optional[typing.Union[typing.Union["CfnRestApi.S3LocationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        clone_from: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        disable_execute_api_endpoint: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        endpoint_configuration: typing.Optional[typing.Union[typing.Union["CfnRestApi.EndpointConfigurationProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        fail_on_warnings: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        minimum_compression_size: typing.Optional[jsii.Number] = None,
        mode: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        policy: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::RestApi``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_key_source_type: The source of the API key for metering requests according to a usage plan. Valid values are:. - ``HEADER`` to read the API key from the ``X-API-Key`` header of a request. - ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a Lambda authorizer.
        :param binary_media_types: The list of binary media types that are supported by the ``RestApi`` resource. Use ``~1`` instead of ``/`` in the media types, for example ``image~1png`` or ``application~1octet-stream`` . By default, ``RestApi`` supports only UTF-8-encoded text payloads. Duplicates are not allowed. For more information, see `Enable Support for Binary Payloads in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-payload-encodings.html>`_ in the *API Gateway Developer Guide* .
        :param body: An OpenAPI specification that defines a set of RESTful APIs in JSON format. For YAML templates, you can also provide the specification in YAML format.
        :param body_s3_location: The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.
        :param clone_from: The ID of the ``RestApi`` resource that you want to clone.
        :param description: A description of the ``RestApi`` resource.
        :param disable_execute_api_endpoint: Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint. By default, clients can invoke your API with the default https://{api_id}.execute-api.{region}.amazonaws.com endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint.
        :param endpoint_configuration: A list of the endpoint types of the API. Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.
        :param fail_on_warnings: Indicates whether to roll back the resource if a warning occurs while API Gateway is creating the ``RestApi`` resource.
        :param minimum_compression_size: A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API. When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.
        :param mode: This property applies only when you use OpenAPI to define your REST API. The ``Mode`` determines how API Gateway handles resource updates. Valid values are ``overwrite`` or ``merge`` . For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged. For ``merge`` , the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API. Use ``merge`` to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties. If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . Otherwise, the default value is ``merge`` .
        :param name: A name for the ``RestApi`` resource.
        :param parameters: Custom header parameters for the request.
        :param policy: A policy document that contains the permissions for the ``RestApi`` resource. To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the API.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d697550d14875b497e7c9057d17e071683ae97bb57c996e9bec5e3a177cd70a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRestApiProps(
            api_key_source_type=api_key_source_type,
            binary_media_types=binary_media_types,
            body=body,
            body_s3_location=body_s3_location,
            clone_from=clone_from,
            description=description,
            disable_execute_api_endpoint=disable_execute_api_endpoint,
            endpoint_configuration=endpoint_configuration,
            fail_on_warnings=fail_on_warnings,
            minimum_compression_size=minimum_compression_size,
            mode=mode,
            name=name,
            parameters=parameters,
            policy=policy,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0fac1524a313a3f06bed65df1b8dbf113db8b97ee08eb60baad6ec1bfd47f124)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f9ca871e11a91e4867421996ab9e25ed98a4cb04b903f371c476e20f863cf507)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrRootResourceId")
    def attr_root_resource_id(self) -> builtins.str:
        '''The root resource ID for a ``RestApi`` resource, such as ``a0bc123d4e`` .

        :cloudformationAttribute: RootResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRootResourceId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="body")
    def body(self) -> typing.Any:
        '''An OpenAPI specification that defines a set of RESTful APIs in JSON format.

        For YAML templates, you can also provide the specification in YAML format.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-body
        '''
        return typing.cast(typing.Any, jsii.get(self, "body"))

    @body.setter
    def body(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d8cec714f14aa2b1ad9179cde90a718c94c7f7d280b2f9c386487c5da7399c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "body", value)

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> typing.Any:
        '''A policy document that contains the permissions for the ``RestApi`` resource.

        To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-policy
        '''
        return typing.cast(typing.Any, jsii.get(self, "policy"))

    @policy.setter
    def policy(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__680790b48d1ca0e5d06a6976f6507b5481ff362318dcc55cb5997b486aa65f1e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "policy", value)

    @builtins.property
    @jsii.member(jsii_name="apiKeySourceType")
    def api_key_source_type(self) -> typing.Optional[builtins.str]:
        '''The source of the API key for metering requests according to a usage plan. Valid values are:.

        - ``HEADER`` to read the API key from the ``X-API-Key`` header of a request.
        - ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a Lambda authorizer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-apikeysourcetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "apiKeySourceType"))

    @api_key_source_type.setter
    def api_key_source_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c16cb75f2d83631b97c1e997e88e8ff253e9188a8063cb8bc3b26e3220fa6e6e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiKeySourceType", value)

    @builtins.property
    @jsii.member(jsii_name="binaryMediaTypes")
    def binary_media_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of binary media types that are supported by the ``RestApi`` resource.

        Use ``~1`` instead of ``/`` in the media types, for example ``image~1png`` or ``application~1octet-stream`` . By default, ``RestApi`` supports only UTF-8-encoded text payloads. Duplicates are not allowed. For more information, see `Enable Support for Binary Payloads in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-payload-encodings.html>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-binarymediatypes
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "binaryMediaTypes"))

    @binary_media_types.setter
    def binary_media_types(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__241a965af17604d16edf6cb458c72a11551fa7d137f23dbcf8d3be884f974142)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "binaryMediaTypes", value)

    @builtins.property
    @jsii.member(jsii_name="bodyS3Location")
    def body_s3_location(
        self,
    ) -> typing.Optional[typing.Union["CfnRestApi.S3LocationProperty", _IResolvable_a771d0ef]]:
        '''The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-bodys3location
        '''
        return typing.cast(typing.Optional[typing.Union["CfnRestApi.S3LocationProperty", _IResolvable_a771d0ef]], jsii.get(self, "bodyS3Location"))

    @body_s3_location.setter
    def body_s3_location(
        self,
        value: typing.Optional[typing.Union["CfnRestApi.S3LocationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5f4d664416697097d3c56bb62ff68737a981043da82c6d8841e35dab72fac89)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bodyS3Location", value)

    @builtins.property
    @jsii.member(jsii_name="cloneFrom")
    def clone_from(self) -> typing.Optional[builtins.str]:
        '''The ID of the ``RestApi`` resource that you want to clone.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-clonefrom
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cloneFrom"))

    @clone_from.setter
    def clone_from(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__534ab394ae14e8113ac2ec17d7411e4234d1450cb8ab9faf8a998a7fd98f4935)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cloneFrom", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the ``RestApi`` resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b5153bbdfc7d99acc22a91e7e4347b8651f69cf5cac8139ddb69d99024ace28)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="disableExecuteApiEndpoint")
    def disable_execute_api_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint.

        By default, clients can invoke your API with the default https://{api_id}.execute-api.{region}.amazonaws.com endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-disableexecuteapiendpoint
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "disableExecuteApiEndpoint"))

    @disable_execute_api_endpoint.setter
    def disable_execute_api_endpoint(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__88a8c503994dbb42232495efae254009b7d99857d86d398b64c218b169a76c5c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "disableExecuteApiEndpoint", value)

    @builtins.property
    @jsii.member(jsii_name="endpointConfiguration")
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnRestApi.EndpointConfigurationProperty", _IResolvable_a771d0ef]]:
        '''A list of the endpoint types of the API.

        Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-endpointconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnRestApi.EndpointConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "endpointConfiguration"))

    @endpoint_configuration.setter
    def endpoint_configuration(
        self,
        value: typing.Optional[typing.Union["CfnRestApi.EndpointConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__19d313b7fe9de16c477ee0b1ba53f6f30924b3d002e5e7d3a6dcdc087c3913fa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpointConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="failOnWarnings")
    def fail_on_warnings(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether to roll back the resource if a warning occurs while API Gateway is creating the ``RestApi`` resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-failonwarnings
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "failOnWarnings"))

    @fail_on_warnings.setter
    def fail_on_warnings(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ec3abffa17bb2c21fb9f91f84285373d41c9201507634fc31b1ce4691976d14)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "failOnWarnings", value)

    @builtins.property
    @jsii.member(jsii_name="minimumCompressionSize")
    def minimum_compression_size(self) -> typing.Optional[jsii.Number]:
        '''A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API.

        When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-minimumcompressionsize
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minimumCompressionSize"))

    @minimum_compression_size.setter
    def minimum_compression_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ffc6ac11a92f7e403d2da1e62a5d0e507e3e5cd806ca9670fe57bbaa1d3c5c2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minimumCompressionSize", value)

    @builtins.property
    @jsii.member(jsii_name="mode")
    def mode(self) -> typing.Optional[builtins.str]:
        '''This property applies only when you use OpenAPI to define your REST API.

        The ``Mode`` determines how API Gateway handles resource updates.

        Valid values are ``overwrite`` or ``merge`` .

        For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged.

        For ``merge`` , the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API. Use ``merge`` to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties.

        If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . Otherwise, the default value is ``merge`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-mode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "mode"))

    @mode.setter
    def mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e15ad557abcde97cec557eca0bdf0bb0970899554a4c75f62797073d8d701a07)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "mode", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the ``RestApi`` resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22d4a3dfcfa8a17ac12cbee20b3f3e801c354a77c26238e6acf8607be4385a9f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''Custom header parameters for the request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-parameters
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c87aebb9bf86ebf91f4c1652a7a89ec197c82099285b143debc03e72b907dfdb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnRestApi.EndpointConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"types": "types", "vpc_endpoint_ids": "vpcEndpointIds"},
    )
    class EndpointConfigurationProperty:
        def __init__(
            self,
            *,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
            vpc_endpoint_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``EndpointConfiguration`` property type specifies the endpoint types of a REST API.

            ``EndpointConfiguration`` is a property of the `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource.

            :param types: A list of endpoint types of an API or its custom domain name. Valid values include:. - ``EDGE`` : For an edge-optimized API and its custom domain name. - ``REGIONAL`` : For a regional API and its custom domain name. - ``PRIVATE`` : For a private API.
            :param vpc_endpoint_ids: A list of VPC endpoint IDs of an API ( `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ ) against which to create Route53 ALIASes. It is only supported for ``PRIVATE`` endpoint type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                endpoint_configuration_property = apigateway.CfnRestApi.EndpointConfigurationProperty(
                    types=["types"],
                    vpc_endpoint_ids=["vpcEndpointIds"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1ab6bcd675a8f4cd6da603c4cc6fce0bf348ce7fae4bec6aebf242108492c45d)
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
                check_type(argname="argument vpc_endpoint_ids", value=vpc_endpoint_ids, expected_type=type_hints["vpc_endpoint_ids"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if types is not None:
                self._values["types"] = types
            if vpc_endpoint_ids is not None:
                self._values["vpc_endpoint_ids"] = vpc_endpoint_ids

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of endpoint types of an API or its custom domain name. Valid values include:.

            - ``EDGE`` : For an edge-optimized API and its custom domain name.
            - ``REGIONAL`` : For a regional API and its custom domain name.
            - ``PRIVATE`` : For a private API.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html#cfn-apigateway-restapi-endpointconfiguration-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def vpc_endpoint_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VPC endpoint IDs of an API ( `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ ) against which to create Route53 ALIASes. It is only supported for ``PRIVATE`` endpoint type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html#cfn-apigateway-restapi-endpointconfiguration-vpcendpointids
            '''
            result = self._values.get("vpc_endpoint_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnRestApi.S3LocationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "bucket": "bucket",
            "e_tag": "eTag",
            "key": "key",
            "version": "version",
        },
    )
    class S3LocationProperty:
        def __init__(
            self,
            *,
            bucket: typing.Optional[builtins.str] = None,
            e_tag: typing.Optional[builtins.str] = None,
            key: typing.Optional[builtins.str] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``S3Location`` is a property of the `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource that specifies the Amazon S3 location of a OpenAPI (formerly Swagger) file that defines a set of RESTful APIs in JSON or YAML.

            .. epigraph::

               On January 1, 2016, the Swagger Specification was donated to the `OpenAPI initiative <https://docs.aws.amazon.com/https://www.openapis.org/>`_ , becoming the foundation of the OpenAPI Specification.

            :param bucket: The name of the S3 bucket where the OpenAPI file is stored.
            :param e_tag: The Amazon S3 ETag (a file checksum) of the OpenAPI file. If you don't specify a value, API Gateway skips ETag validation of your OpenAPI file.
            :param key: The file name of the OpenAPI file (Amazon S3 object name).
            :param version: For versioning-enabled buckets, a specific version of the OpenAPI file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                s3_location_property = apigateway.CfnRestApi.S3LocationProperty(
                    bucket="bucket",
                    e_tag="eTag",
                    key="key",
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__29682aee8f7a2423860708dd24f971508b766184358746182d41f653c8547530)
                check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
                check_type(argname="argument e_tag", value=e_tag, expected_type=type_hints["e_tag"])
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if bucket is not None:
                self._values["bucket"] = bucket
            if e_tag is not None:
                self._values["e_tag"] = e_tag
            if key is not None:
                self._values["key"] = key
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def bucket(self) -> typing.Optional[builtins.str]:
            '''The name of the S3 bucket where the OpenAPI file is stored.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-bucket
            '''
            result = self._values.get("bucket")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def e_tag(self) -> typing.Optional[builtins.str]:
            '''The Amazon S3 ETag (a file checksum) of the OpenAPI file.

            If you don't specify a value, API Gateway skips ETag validation of your OpenAPI file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-etag
            '''
            result = self._values.get("e_tag")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''The file name of the OpenAPI file (Amazon S3 object name).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''For versioning-enabled buckets, a specific version of the OpenAPI file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnRestApiProps",
    jsii_struct_bases=[],
    name_mapping={
        "api_key_source_type": "apiKeySourceType",
        "binary_media_types": "binaryMediaTypes",
        "body": "body",
        "body_s3_location": "bodyS3Location",
        "clone_from": "cloneFrom",
        "description": "description",
        "disable_execute_api_endpoint": "disableExecuteApiEndpoint",
        "endpoint_configuration": "endpointConfiguration",
        "fail_on_warnings": "failOnWarnings",
        "minimum_compression_size": "minimumCompressionSize",
        "mode": "mode",
        "name": "name",
        "parameters": "parameters",
        "policy": "policy",
        "tags": "tags",
    },
)
class CfnRestApiProps:
    def __init__(
        self,
        *,
        api_key_source_type: typing.Optional[builtins.str] = None,
        binary_media_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        body: typing.Any = None,
        body_s3_location: typing.Optional[typing.Union[typing.Union[CfnRestApi.S3LocationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        clone_from: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        disable_execute_api_endpoint: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        endpoint_configuration: typing.Optional[typing.Union[typing.Union[CfnRestApi.EndpointConfigurationProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        fail_on_warnings: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        minimum_compression_size: typing.Optional[jsii.Number] = None,
        mode: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
        policy: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRestApi``.

        :param api_key_source_type: The source of the API key for metering requests according to a usage plan. Valid values are:. - ``HEADER`` to read the API key from the ``X-API-Key`` header of a request. - ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a Lambda authorizer.
        :param binary_media_types: The list of binary media types that are supported by the ``RestApi`` resource. Use ``~1`` instead of ``/`` in the media types, for example ``image~1png`` or ``application~1octet-stream`` . By default, ``RestApi`` supports only UTF-8-encoded text payloads. Duplicates are not allowed. For more information, see `Enable Support for Binary Payloads in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-payload-encodings.html>`_ in the *API Gateway Developer Guide* .
        :param body: An OpenAPI specification that defines a set of RESTful APIs in JSON format. For YAML templates, you can also provide the specification in YAML format.
        :param body_s3_location: The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.
        :param clone_from: The ID of the ``RestApi`` resource that you want to clone.
        :param description: A description of the ``RestApi`` resource.
        :param disable_execute_api_endpoint: Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint. By default, clients can invoke your API with the default https://{api_id}.execute-api.{region}.amazonaws.com endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint.
        :param endpoint_configuration: A list of the endpoint types of the API. Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.
        :param fail_on_warnings: Indicates whether to roll back the resource if a warning occurs while API Gateway is creating the ``RestApi`` resource.
        :param minimum_compression_size: A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API. When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.
        :param mode: This property applies only when you use OpenAPI to define your REST API. The ``Mode`` determines how API Gateway handles resource updates. Valid values are ``overwrite`` or ``merge`` . For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged. For ``merge`` , the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API. Use ``merge`` to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties. If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . Otherwise, the default value is ``merge`` .
        :param name: A name for the ``RestApi`` resource.
        :param parameters: Custom header parameters for the request.
        :param policy: A policy document that contains the permissions for the ``RestApi`` resource. To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # body: Any
            # policy: Any
            
            cfn_rest_api_props = apigateway.CfnRestApiProps(
                api_key_source_type="apiKeySourceType",
                binary_media_types=["binaryMediaTypes"],
                body=body,
                body_s3_location=apigateway.CfnRestApi.S3LocationProperty(
                    bucket="bucket",
                    e_tag="eTag",
                    key="key",
                    version="version"
                ),
                clone_from="cloneFrom",
                description="description",
                disable_execute_api_endpoint=False,
                endpoint_configuration=apigateway.CfnRestApi.EndpointConfigurationProperty(
                    types=["types"],
                    vpc_endpoint_ids=["vpcEndpointIds"]
                ),
                fail_on_warnings=False,
                minimum_compression_size=123,
                mode="mode",
                name="name",
                parameters={
                    "parameters_key": "parameters"
                },
                policy=policy,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7520883bbbade40f2bbb2fb3dd43c8c789c3b376fbfe6d6056f8e97b8646db9b)
            check_type(argname="argument api_key_source_type", value=api_key_source_type, expected_type=type_hints["api_key_source_type"])
            check_type(argname="argument binary_media_types", value=binary_media_types, expected_type=type_hints["binary_media_types"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument body_s3_location", value=body_s3_location, expected_type=type_hints["body_s3_location"])
            check_type(argname="argument clone_from", value=clone_from, expected_type=type_hints["clone_from"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument disable_execute_api_endpoint", value=disable_execute_api_endpoint, expected_type=type_hints["disable_execute_api_endpoint"])
            check_type(argname="argument endpoint_configuration", value=endpoint_configuration, expected_type=type_hints["endpoint_configuration"])
            check_type(argname="argument fail_on_warnings", value=fail_on_warnings, expected_type=type_hints["fail_on_warnings"])
            check_type(argname="argument minimum_compression_size", value=minimum_compression_size, expected_type=type_hints["minimum_compression_size"])
            check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if api_key_source_type is not None:
            self._values["api_key_source_type"] = api_key_source_type
        if binary_media_types is not None:
            self._values["binary_media_types"] = binary_media_types
        if body is not None:
            self._values["body"] = body
        if body_s3_location is not None:
            self._values["body_s3_location"] = body_s3_location
        if clone_from is not None:
            self._values["clone_from"] = clone_from
        if description is not None:
            self._values["description"] = description
        if disable_execute_api_endpoint is not None:
            self._values["disable_execute_api_endpoint"] = disable_execute_api_endpoint
        if endpoint_configuration is not None:
            self._values["endpoint_configuration"] = endpoint_configuration
        if fail_on_warnings is not None:
            self._values["fail_on_warnings"] = fail_on_warnings
        if minimum_compression_size is not None:
            self._values["minimum_compression_size"] = minimum_compression_size
        if mode is not None:
            self._values["mode"] = mode
        if name is not None:
            self._values["name"] = name
        if parameters is not None:
            self._values["parameters"] = parameters
        if policy is not None:
            self._values["policy"] = policy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def api_key_source_type(self) -> typing.Optional[builtins.str]:
        '''The source of the API key for metering requests according to a usage plan. Valid values are:.

        - ``HEADER`` to read the API key from the ``X-API-Key`` header of a request.
        - ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a Lambda authorizer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-apikeysourcetype
        '''
        result = self._values.get("api_key_source_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def binary_media_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of binary media types that are supported by the ``RestApi`` resource.

        Use ``~1`` instead of ``/`` in the media types, for example ``image~1png`` or ``application~1octet-stream`` . By default, ``RestApi`` supports only UTF-8-encoded text payloads. Duplicates are not allowed. For more information, see `Enable Support for Binary Payloads in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-payload-encodings.html>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-binarymediatypes
        '''
        result = self._values.get("binary_media_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def body(self) -> typing.Any:
        '''An OpenAPI specification that defines a set of RESTful APIs in JSON format.

        For YAML templates, you can also provide the specification in YAML format.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-body
        '''
        result = self._values.get("body")
        return typing.cast(typing.Any, result)

    @builtins.property
    def body_s3_location(
        self,
    ) -> typing.Optional[typing.Union[CfnRestApi.S3LocationProperty, _IResolvable_a771d0ef]]:
        '''The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-bodys3location
        '''
        result = self._values.get("body_s3_location")
        return typing.cast(typing.Optional[typing.Union[CfnRestApi.S3LocationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def clone_from(self) -> typing.Optional[builtins.str]:
        '''The ID of the ``RestApi`` resource that you want to clone.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-clonefrom
        '''
        result = self._values.get("clone_from")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the ``RestApi`` resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def disable_execute_api_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint.

        By default, clients can invoke your API with the default https://{api_id}.execute-api.{region}.amazonaws.com endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-disableexecuteapiendpoint
        '''
        result = self._values.get("disable_execute_api_endpoint")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnRestApi.EndpointConfigurationProperty, _IResolvable_a771d0ef]]:
        '''A list of the endpoint types of the API.

        Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-endpointconfiguration
        '''
        result = self._values.get("endpoint_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnRestApi.EndpointConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def fail_on_warnings(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether to roll back the resource if a warning occurs while API Gateway is creating the ``RestApi`` resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-failonwarnings
        '''
        result = self._values.get("fail_on_warnings")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def minimum_compression_size(self) -> typing.Optional[jsii.Number]:
        '''A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API.

        When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-minimumcompressionsize
        '''
        result = self._values.get("minimum_compression_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def mode(self) -> typing.Optional[builtins.str]:
        '''This property applies only when you use OpenAPI to define your REST API.

        The ``Mode`` determines how API Gateway handles resource updates.

        Valid values are ``overwrite`` or ``merge`` .

        For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged.

        For ``merge`` , the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API. Use ``merge`` to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties.

        If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . Otherwise, the default value is ``merge`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-mode
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the ``RestApi`` resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''Custom header parameters for the request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

    @builtins.property
    def policy(self) -> typing.Any:
        '''A policy document that contains the permissions for the ``RestApi`` resource.

        To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-policy
        '''
        result = self._values.get("policy")
        return typing.cast(typing.Any, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the API.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRestApiProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnRouteResponseV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnRouteResponseV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::RouteResponse``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::RouteResponse
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # response_models: Any
        # response_parameters: Any
        
        cfn_route_response_v2 = apigateway.CfnRouteResponseV2(self, "MyCfnRouteResponseV2",
            api_id="apiId",
            route_id="routeId",
            route_response_key="routeResponseKey",
        
            # the properties below are optional
            model_selection_expression="modelSelectionExpression",
            response_models=response_models,
            response_parameters=response_parameters
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        route_id: builtins.str,
        route_response_key: builtins.str,
        model_selection_expression: typing.Optional[builtins.str] = None,
        response_models: typing.Any = None,
        response_parameters: typing.Any = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::RouteResponse``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ApiId``.
        :param route_id: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteId``.
        :param route_response_key: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteResponseKey``.
        :param model_selection_expression: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ModelSelectionExpression``.
        :param response_models: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseModels``.
        :param response_parameters: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseParameters``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8efaa8b7065530b0e9e3ead904ff4b62bc19db792bfcf85c305d943a0b293f78)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRouteResponseV2Props(
            api_id=api_id,
            route_id=route_id,
            route_response_key=route_response_key,
            model_selection_expression=model_selection_expression,
            response_models=response_models,
            response_parameters=response_parameters,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb60302fc4d4cd34f5009963fc4adea672e9f2829cb63da81e0e162f3a9ee7b0)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d53f7e958cee94c0b107935a0779d25ad84d5b66439fcdb1eb3901d2218a07e7)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc797c8970c9b8989a145247f602139e79719e64829b12bd88c23eb9242e3500)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="responseModels")
    def response_models(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseModels``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-responsemodels
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "responseModels"))

    @response_models.setter
    def response_models(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6ebb0c952a6459ac60c1c23d8e57ec9bd66e785e14415189b21ed56c2893a5a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseModels", value)

    @builtins.property
    @jsii.member(jsii_name="responseParameters")
    def response_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-responseparameters
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "responseParameters"))

    @response_parameters.setter
    def response_parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__92020040c50f80f0424fb645a40a0d529aa4e6a35fd20a9b42ebf55863422934)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseParameters", value)

    @builtins.property
    @jsii.member(jsii_name="routeId")
    def route_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-routeid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "routeId"))

    @route_id.setter
    def route_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2124aad57bffc06957109bae3b4f39d6fc062ee8643d4335b91e05ede3f0dff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeId", value)

    @builtins.property
    @jsii.member(jsii_name="routeResponseKey")
    def route_response_key(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteResponseKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-routeresponsekey
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "routeResponseKey"))

    @route_response_key.setter
    def route_response_key(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cdd464ccdc428cdaf89b00222782f82e9bec12b3c0c6de2c0be016aaeeb95441)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeResponseKey", value)

    @builtins.property
    @jsii.member(jsii_name="modelSelectionExpression")
    def model_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ModelSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-modelselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "modelSelectionExpression"))

    @model_selection_expression.setter
    def model_selection_expression(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef4757a34428ddff45f7fd72ed7a5034b7246a619bf15b4b033c7284522018e5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "modelSelectionExpression", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnRouteResponseV2.ParameterConstraintsProperty",
        jsii_struct_bases=[],
        name_mapping={"required": "required"},
    )
    class ParameterConstraintsProperty:
        def __init__(
            self,
            *,
            required: typing.Union[builtins.bool, _IResolvable_a771d0ef],
        ) -> None:
            '''
            :param required: (deprecated) ``CfnRouteResponseV2.ParameterConstraintsProperty.Required``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-routeresponse-parameterconstraints.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                parameter_constraints_property = apigateway.CfnRouteResponseV2.ParameterConstraintsProperty(
                    required=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7a373edb566fa0c04322a145279074e09ce9009e21c18b3145dfeee99eb8ee6a)
                check_type(argname="argument required", value=required, expected_type=type_hints["required"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "required": required,
            }

        @builtins.property
        def required(self) -> typing.Union[builtins.bool, _IResolvable_a771d0ef]:
            '''(deprecated) ``CfnRouteResponseV2.ParameterConstraintsProperty.Required``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-routeresponse-parameterconstraints.html#cfn-apigatewayv2-routeresponse-parameterconstraints-required
            :stability: deprecated
            '''
            result = self._values.get("required")
            assert result is not None, "Required property 'required' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ParameterConstraintsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnRouteResponseV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "route_id": "routeId",
        "route_response_key": "routeResponseKey",
        "model_selection_expression": "modelSelectionExpression",
        "response_models": "responseModels",
        "response_parameters": "responseParameters",
    },
)
class CfnRouteResponseV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        route_id: builtins.str,
        route_response_key: builtins.str,
        model_selection_expression: typing.Optional[builtins.str] = None,
        response_models: typing.Any = None,
        response_parameters: typing.Any = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::RouteResponse``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ApiId``.
        :param route_id: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteId``.
        :param route_response_key: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteResponseKey``.
        :param model_selection_expression: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ModelSelectionExpression``.
        :param response_models: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseModels``.
        :param response_parameters: (deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseParameters``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # response_models: Any
            # response_parameters: Any
            
            cfn_route_response_v2_props = apigateway.CfnRouteResponseV2Props(
                api_id="apiId",
                route_id="routeId",
                route_response_key="routeResponseKey",
            
                # the properties below are optional
                model_selection_expression="modelSelectionExpression",
                response_models=response_models,
                response_parameters=response_parameters
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__796c9a687d72fb4c362af7fab45cd74ca07f737d1c5fa13d7d414c7652c3ea92)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument route_id", value=route_id, expected_type=type_hints["route_id"])
            check_type(argname="argument route_response_key", value=route_response_key, expected_type=type_hints["route_response_key"])
            check_type(argname="argument model_selection_expression", value=model_selection_expression, expected_type=type_hints["model_selection_expression"])
            check_type(argname="argument response_models", value=response_models, expected_type=type_hints["response_models"])
            check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "route_id": route_id,
            "route_response_key": route_response_key,
        }
        if model_selection_expression is not None:
            self._values["model_selection_expression"] = model_selection_expression
        if response_models is not None:
            self._values["response_models"] = response_models
        if response_parameters is not None:
            self._values["response_parameters"] = response_parameters

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def route_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-routeid
        :stability: deprecated
        '''
        result = self._values.get("route_id")
        assert result is not None, "Required property 'route_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def route_response_key(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.RouteResponseKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-routeresponsekey
        :stability: deprecated
        '''
        result = self._values.get("route_response_key")
        assert result is not None, "Required property 'route_response_key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def model_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ModelSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-modelselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("model_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def response_models(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseModels``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-responsemodels
        :stability: deprecated
        '''
        result = self._values.get("response_models")
        return typing.cast(typing.Any, result)

    @builtins.property
    def response_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::RouteResponse.ResponseParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-routeresponse.html#cfn-apigatewayv2-routeresponse-responseparameters
        :stability: deprecated
        '''
        result = self._values.get("response_parameters")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteResponseV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnRouteV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnRouteV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Route``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Route
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # request_models: Any
        # request_parameters: Any
        
        cfn_route_v2 = apigateway.CfnRouteV2(self, "MyCfnRouteV2",
            api_id="apiId",
            route_key="routeKey",
        
            # the properties below are optional
            api_key_required=False,
            authorization_scopes=["authorizationScopes"],
            authorization_type="authorizationType",
            authorizer_id="authorizerId",
            model_selection_expression="modelSelectionExpression",
            operation_name="operationName",
            request_models=request_models,
            request_parameters=request_parameters,
            route_response_selection_expression="routeResponseSelectionExpression",
            target="target"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        route_key: builtins.str,
        api_key_required: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[builtins.str] = None,
        authorizer_id: typing.Optional[builtins.str] = None,
        model_selection_expression: typing.Optional[builtins.str] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Any = None,
        request_parameters: typing.Any = None,
        route_response_selection_expression: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Route``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Route.ApiId``.
        :param route_key: (deprecated) ``AWS::ApiGatewayV2::Route.RouteKey``.
        :param api_key_required: (deprecated) ``AWS::ApiGatewayV2::Route.ApiKeyRequired``.
        :param authorization_scopes: (deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationScopes``.
        :param authorization_type: (deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationType``.
        :param authorizer_id: (deprecated) ``AWS::ApiGatewayV2::Route.AuthorizerId``.
        :param model_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Route.ModelSelectionExpression``.
        :param operation_name: (deprecated) ``AWS::ApiGatewayV2::Route.OperationName``.
        :param request_models: (deprecated) ``AWS::ApiGatewayV2::Route.RequestModels``.
        :param request_parameters: (deprecated) ``AWS::ApiGatewayV2::Route.RequestParameters``.
        :param route_response_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Route.RouteResponseSelectionExpression``.
        :param target: (deprecated) ``AWS::ApiGatewayV2::Route.Target``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42b7ab8eb017b5947109058e53c1735450045b6370f33a7a710de3a13ab316e0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRouteV2Props(
            api_id=api_id,
            route_key=route_key,
            api_key_required=api_key_required,
            authorization_scopes=authorization_scopes,
            authorization_type=authorization_type,
            authorizer_id=authorizer_id,
            model_selection_expression=model_selection_expression,
            operation_name=operation_name,
            request_models=request_models,
            request_parameters=request_parameters,
            route_response_selection_expression=route_response_selection_expression,
            target=target,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb5851651c66ce2ccc70ce75a76c7ce7c9a1625bf0d43cb3151321aa288c2665)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c61aef51d74cdbab460b45586fcaf02cfad28fbb4481328742fb29c4d62f7f06)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76a186f72690297538a79a70963a2eefa6856ecb34cfbd742828ace5c1804581)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="requestModels")
    def request_models(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RequestModels``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-requestmodels
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "requestModels"))

    @request_models.setter
    def request_models(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eff34cdab1eb729c6f0d2ad77e07606fc472c8e3b4c75a4957db9708d8a2b869)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestModels", value)

    @builtins.property
    @jsii.member(jsii_name="requestParameters")
    def request_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RequestParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-requestparameters
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "requestParameters"))

    @request_parameters.setter
    def request_parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__85a451268dec362b1fc4a8ca0d2db40e8fc488c38ccbeab481b686f59ff6c84b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestParameters", value)

    @builtins.property
    @jsii.member(jsii_name="routeKey")
    def route_key(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RouteKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-routekey
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "routeKey"))

    @route_key.setter
    def route_key(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__355860e7c9888686819b6a91fe69dc365fd9b485ab614737b206846f30628873)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeKey", value)

    @builtins.property
    @jsii.member(jsii_name="apiKeyRequired")
    def api_key_required(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.ApiKeyRequired``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-apikeyrequired
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "apiKeyRequired"))

    @api_key_required.setter
    def api_key_required(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c37f90bcb37a7d458f3a7591f1d029acc39f607e8fe29875cfa62929312b42d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiKeyRequired", value)

    @builtins.property
    @jsii.member(jsii_name="authorizationScopes")
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationScopes``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-authorizationscopes
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "authorizationScopes"))

    @authorization_scopes.setter
    def authorization_scopes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6596eb8144eef591ce8db6dd9fc52808f561717a3a98655c0b88206076f3ac0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizationScopes", value)

    @builtins.property
    @jsii.member(jsii_name="authorizationType")
    def authorization_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-authorizationtype
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizationType"))

    @authorization_type.setter
    def authorization_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__638fa8088f61252091347910654659d4976acb29fcc1522faba0e0eaf2e42638)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizationType", value)

    @builtins.property
    @jsii.member(jsii_name="authorizerId")
    def authorizer_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.AuthorizerId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-authorizerid
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerId"))

    @authorizer_id.setter
    def authorizer_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e9c5442d8fbf066dde6f09f854800af46f14f55d14a0e935b9fe25ba0f13ac60)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerId", value)

    @builtins.property
    @jsii.member(jsii_name="modelSelectionExpression")
    def model_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.ModelSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-modelselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "modelSelectionExpression"))

    @model_selection_expression.setter
    def model_selection_expression(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe6aa6cf92e30fabe7ebe4275bd5bccafe4372f62e7f47f4dd4dfcf05d3e9865)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "modelSelectionExpression", value)

    @builtins.property
    @jsii.member(jsii_name="operationName")
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.OperationName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-operationname
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "operationName"))

    @operation_name.setter
    def operation_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7b5cf6f5d1dfad8830a92ad9eea2a3f6e00e3b530241f74757837ca0273ad49)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "operationName", value)

    @builtins.property
    @jsii.member(jsii_name="routeResponseSelectionExpression")
    def route_response_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RouteResponseSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-routeresponseselectionexpression
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "routeResponseSelectionExpression"))

    @route_response_selection_expression.setter
    def route_response_selection_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4cbb97f376349beb29987c52870776f4dd06b1b9c45ae56bef1dde99e6c5f295)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeResponseSelectionExpression", value)

    @builtins.property
    @jsii.member(jsii_name="target")
    def target(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.Target``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-target
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "target"))

    @target.setter
    def target(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4c6ab608f66546ac52210d2868ecc0d8bdc98b0bfa11464997b7c82c040d0bd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "target", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnRouteV2.ParameterConstraintsProperty",
        jsii_struct_bases=[],
        name_mapping={"required": "required"},
    )
    class ParameterConstraintsProperty:
        def __init__(
            self,
            *,
            required: typing.Union[builtins.bool, _IResolvable_a771d0ef],
        ) -> None:
            '''
            :param required: (deprecated) ``CfnRouteV2.ParameterConstraintsProperty.Required``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-route-parameterconstraints.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                parameter_constraints_property = apigateway.CfnRouteV2.ParameterConstraintsProperty(
                    required=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8015258496b6428364f72fcab97a8c9da8fbf076120c41f30acd143d7366ce39)
                check_type(argname="argument required", value=required, expected_type=type_hints["required"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "required": required,
            }

        @builtins.property
        def required(self) -> typing.Union[builtins.bool, _IResolvable_a771d0ef]:
            '''(deprecated) ``CfnRouteV2.ParameterConstraintsProperty.Required``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-route-parameterconstraints.html#cfn-apigatewayv2-route-parameterconstraints-required
            :stability: deprecated
            '''
            result = self._values.get("required")
            assert result is not None, "Required property 'required' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ParameterConstraintsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnRouteV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "route_key": "routeKey",
        "api_key_required": "apiKeyRequired",
        "authorization_scopes": "authorizationScopes",
        "authorization_type": "authorizationType",
        "authorizer_id": "authorizerId",
        "model_selection_expression": "modelSelectionExpression",
        "operation_name": "operationName",
        "request_models": "requestModels",
        "request_parameters": "requestParameters",
        "route_response_selection_expression": "routeResponseSelectionExpression",
        "target": "target",
    },
)
class CfnRouteV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        route_key: builtins.str,
        api_key_required: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[builtins.str] = None,
        authorizer_id: typing.Optional[builtins.str] = None,
        model_selection_expression: typing.Optional[builtins.str] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Any = None,
        request_parameters: typing.Any = None,
        route_response_selection_expression: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Route``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Route.ApiId``.
        :param route_key: (deprecated) ``AWS::ApiGatewayV2::Route.RouteKey``.
        :param api_key_required: (deprecated) ``AWS::ApiGatewayV2::Route.ApiKeyRequired``.
        :param authorization_scopes: (deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationScopes``.
        :param authorization_type: (deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationType``.
        :param authorizer_id: (deprecated) ``AWS::ApiGatewayV2::Route.AuthorizerId``.
        :param model_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Route.ModelSelectionExpression``.
        :param operation_name: (deprecated) ``AWS::ApiGatewayV2::Route.OperationName``.
        :param request_models: (deprecated) ``AWS::ApiGatewayV2::Route.RequestModels``.
        :param request_parameters: (deprecated) ``AWS::ApiGatewayV2::Route.RequestParameters``.
        :param route_response_selection_expression: (deprecated) ``AWS::ApiGatewayV2::Route.RouteResponseSelectionExpression``.
        :param target: (deprecated) ``AWS::ApiGatewayV2::Route.Target``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # request_models: Any
            # request_parameters: Any
            
            cfn_route_v2_props = apigateway.CfnRouteV2Props(
                api_id="apiId",
                route_key="routeKey",
            
                # the properties below are optional
                api_key_required=False,
                authorization_scopes=["authorizationScopes"],
                authorization_type="authorizationType",
                authorizer_id="authorizerId",
                model_selection_expression="modelSelectionExpression",
                operation_name="operationName",
                request_models=request_models,
                request_parameters=request_parameters,
                route_response_selection_expression="routeResponseSelectionExpression",
                target="target"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a45af45da5fea9c53d4e21fa4c4145f5ad2992b0c2168edd22e5edacdf673830)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument route_key", value=route_key, expected_type=type_hints["route_key"])
            check_type(argname="argument api_key_required", value=api_key_required, expected_type=type_hints["api_key_required"])
            check_type(argname="argument authorization_scopes", value=authorization_scopes, expected_type=type_hints["authorization_scopes"])
            check_type(argname="argument authorization_type", value=authorization_type, expected_type=type_hints["authorization_type"])
            check_type(argname="argument authorizer_id", value=authorizer_id, expected_type=type_hints["authorizer_id"])
            check_type(argname="argument model_selection_expression", value=model_selection_expression, expected_type=type_hints["model_selection_expression"])
            check_type(argname="argument operation_name", value=operation_name, expected_type=type_hints["operation_name"])
            check_type(argname="argument request_models", value=request_models, expected_type=type_hints["request_models"])
            check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
            check_type(argname="argument route_response_selection_expression", value=route_response_selection_expression, expected_type=type_hints["route_response_selection_expression"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "route_key": route_key,
        }
        if api_key_required is not None:
            self._values["api_key_required"] = api_key_required
        if authorization_scopes is not None:
            self._values["authorization_scopes"] = authorization_scopes
        if authorization_type is not None:
            self._values["authorization_type"] = authorization_type
        if authorizer_id is not None:
            self._values["authorizer_id"] = authorizer_id
        if model_selection_expression is not None:
            self._values["model_selection_expression"] = model_selection_expression
        if operation_name is not None:
            self._values["operation_name"] = operation_name
        if request_models is not None:
            self._values["request_models"] = request_models
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if route_response_selection_expression is not None:
            self._values["route_response_selection_expression"] = route_response_selection_expression
        if target is not None:
            self._values["target"] = target

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def route_key(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RouteKey``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-routekey
        :stability: deprecated
        '''
        result = self._values.get("route_key")
        assert result is not None, "Required property 'route_key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_key_required(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.ApiKeyRequired``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-apikeyrequired
        :stability: deprecated
        '''
        result = self._values.get("api_key_required")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationScopes``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-authorizationscopes
        :stability: deprecated
        '''
        result = self._values.get("authorization_scopes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def authorization_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.AuthorizationType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-authorizationtype
        :stability: deprecated
        '''
        result = self._values.get("authorization_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.AuthorizerId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-authorizerid
        :stability: deprecated
        '''
        result = self._values.get("authorizer_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def model_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.ModelSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-modelselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("model_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.OperationName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-operationname
        :stability: deprecated
        '''
        result = self._values.get("operation_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_models(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RequestModels``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-requestmodels
        :stability: deprecated
        '''
        result = self._values.get("request_models")
        return typing.cast(typing.Any, result)

    @builtins.property
    def request_parameters(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RequestParameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-requestparameters
        :stability: deprecated
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Any, result)

    @builtins.property
    def route_response_selection_expression(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.RouteResponseSelectionExpression``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-routeresponseselectionexpression
        :stability: deprecated
        '''
        result = self._values.get("route_response_selection_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Route.Target``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-route.html#cfn-apigatewayv2-route-target
        :stability: deprecated
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnStage(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnStage",
):
    '''A CloudFormation ``AWS::ApiGateway::Stage``.

    The ``AWS::ApiGateway::Stage`` resource creates a stage for a deployment.

    :cloudformationResource: AWS::ApiGateway::Stage
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_stage = apigateway.CfnStage(self, "MyCfnStage",
            rest_api_id="restApiId",
        
            # the properties below are optional
            access_log_setting=apigateway.CfnStage.AccessLogSettingProperty(
                destination_arn="destinationArn",
                format="format"
            ),
            cache_cluster_enabled=False,
            cache_cluster_size="cacheClusterSize",
            canary_setting=apigateway.CfnStage.CanarySettingProperty(
                deployment_id="deploymentId",
                percent_traffic=123,
                stage_variable_overrides={
                    "stage_variable_overrides_key": "stageVariableOverrides"
                },
                use_stage_cache=False
            ),
            client_certificate_id="clientCertificateId",
            deployment_id="deploymentId",
            description="description",
            documentation_version="documentationVersion",
            method_settings=[apigateway.CfnStage.MethodSettingProperty(
                cache_data_encrypted=False,
                cache_ttl_in_seconds=123,
                caching_enabled=False,
                data_trace_enabled=False,
                http_method="httpMethod",
                logging_level="loggingLevel",
                metrics_enabled=False,
                resource_path="resourcePath",
                throttling_burst_limit=123,
                throttling_rate_limit=123
            )],
            stage_name="stageName",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            tracing_enabled=False,
            variables={
                "variables_key": "variables"
            }
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        access_log_setting: typing.Optional[typing.Union[typing.Union["CfnStage.AccessLogSettingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        cache_cluster_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        cache_cluster_size: typing.Optional[builtins.str] = None,
        canary_setting: typing.Optional[typing.Union[typing.Union["CfnStage.CanarySettingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        client_certificate_id: typing.Optional[builtins.str] = None,
        deployment_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        documentation_version: typing.Optional[builtins.str] = None,
        method_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnStage.MethodSettingProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        stage_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        tracing_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        variables: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::Stage``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param rest_api_id: The ID of the ``RestApi`` resource that you're deploying with this stage.
        :param access_log_setting: Specifies settings for logging access in this stage.
        :param cache_cluster_enabled: Indicates whether cache clustering is enabled for the stage.
        :param cache_cluster_size: The stage's cache cluster size.
        :param canary_setting: Specifies settings for the canary deployment in this stage.
        :param client_certificate_id: The ID of the client certificate that API Gateway uses to call your integration endpoints in the stage.
        :param deployment_id: The ID of the deployment that the stage is associated with. This parameter is required to create a stage.
        :param description: A description of the stage.
        :param documentation_version: The version ID of the API documentation snapshot.
        :param method_settings: Settings for all methods in the stage.
        :param stage_name: The name of the stage, which API Gateway uses as the first path segment in the invoked Uniform Resource Identifier (URI).
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the stage.
        :param tracing_enabled: Specifies whether active X-Ray tracing is enabled for this stage. For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .
        :param variables: A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value. Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2bf3b6dfe18f7d10d577d4d54b3646c743e8e06306bfe50c56951394c581159)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnStageProps(
            rest_api_id=rest_api_id,
            access_log_setting=access_log_setting,
            cache_cluster_enabled=cache_cluster_enabled,
            cache_cluster_size=cache_cluster_size,
            canary_setting=canary_setting,
            client_certificate_id=client_certificate_id,
            deployment_id=deployment_id,
            description=description,
            documentation_version=documentation_version,
            method_settings=method_settings,
            stage_name=stage_name,
            tags=tags,
            tracing_enabled=tracing_enabled,
            variables=variables,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d64766fb6adec216a19dbecc6615301fe6b5d0b13ad83632322a37ba08a29519)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69ac01537bbe3b4757fdb14bf15acbdc31851b131c475ca7a0557970f1698e3e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of the ``RestApi`` resource that you're deploying with this stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-restapiid
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5021c72a9c5515dd182ca53255447cd8f56ca754769ea95ee03e86f0d4b8a7dd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value)

    @builtins.property
    @jsii.member(jsii_name="accessLogSetting")
    def access_log_setting(
        self,
    ) -> typing.Optional[typing.Union["CfnStage.AccessLogSettingProperty", _IResolvable_a771d0ef]]:
        '''Specifies settings for logging access in this stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-accesslogsetting
        '''
        return typing.cast(typing.Optional[typing.Union["CfnStage.AccessLogSettingProperty", _IResolvable_a771d0ef]], jsii.get(self, "accessLogSetting"))

    @access_log_setting.setter
    def access_log_setting(
        self,
        value: typing.Optional[typing.Union["CfnStage.AccessLogSettingProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f946a8e854222a51a9d0b49e8b0925571424b9031d75a61b41de3e94421015ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessLogSetting", value)

    @builtins.property
    @jsii.member(jsii_name="cacheClusterEnabled")
    def cache_cluster_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether cache clustering is enabled for the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-cacheclusterenabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "cacheClusterEnabled"))

    @cache_cluster_enabled.setter
    def cache_cluster_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27b6ef09973d100ec383a4d4afb096002c8c4ffbd5fd11e5230c6cac86e269cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cacheClusterEnabled", value)

    @builtins.property
    @jsii.member(jsii_name="cacheClusterSize")
    def cache_cluster_size(self) -> typing.Optional[builtins.str]:
        '''The stage's cache cluster size.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-cacheclustersize
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cacheClusterSize"))

    @cache_cluster_size.setter
    def cache_cluster_size(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf7439212c0cca4809bd21419ccd25419ee466644ee22759933b83ac3cb3446d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cacheClusterSize", value)

    @builtins.property
    @jsii.member(jsii_name="canarySetting")
    def canary_setting(
        self,
    ) -> typing.Optional[typing.Union["CfnStage.CanarySettingProperty", _IResolvable_a771d0ef]]:
        '''Specifies settings for the canary deployment in this stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-canarysetting
        '''
        return typing.cast(typing.Optional[typing.Union["CfnStage.CanarySettingProperty", _IResolvable_a771d0ef]], jsii.get(self, "canarySetting"))

    @canary_setting.setter
    def canary_setting(
        self,
        value: typing.Optional[typing.Union["CfnStage.CanarySettingProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__827c209592292ef2f32a95539e5fefecf71c6cfeea6b9ec9fb683765aee62896)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "canarySetting", value)

    @builtins.property
    @jsii.member(jsii_name="clientCertificateId")
    def client_certificate_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the client certificate that API Gateway uses to call your integration endpoints in the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-clientcertificateid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clientCertificateId"))

    @client_certificate_id.setter
    def client_certificate_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__104f0ed345f8ad83ace823533173f76fe007d503c937fa989ac397102d7b8a32)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientCertificateId", value)

    @builtins.property
    @jsii.member(jsii_name="deploymentId")
    def deployment_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the deployment that the stage is associated with.

        This parameter is required to create a stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-deploymentid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deploymentId"))

    @deployment_id.setter
    def deployment_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16f82256f174f90834fcf66047a95302a40c0cf7cc54e61ee67a2e70f2726a6d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentId", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2045dc94d44615a0634b8a8249aa7a857857b3e14962ae0de0804a1ba774b3df)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="documentationVersion")
    def documentation_version(self) -> typing.Optional[builtins.str]:
        '''The version ID of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-documentationversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "documentationVersion"))

    @documentation_version.setter
    def documentation_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16c96dfe7962b4865246f86b3786dd86740750b3ff6af633463261b2b735a31d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "documentationVersion", value)

    @builtins.property
    @jsii.member(jsii_name="methodSettings")
    def method_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnStage.MethodSettingProperty", _IResolvable_a771d0ef]]]]:
        '''Settings for all methods in the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-methodsettings
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnStage.MethodSettingProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "methodSettings"))

    @method_settings.setter
    def method_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnStage.MethodSettingProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a23c0a57f77c452635b103bcb61d17fcea473d5f3d52f3289cbf6ea956c79cc9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "methodSettings", value)

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the stage, which API Gateway uses as the first path segment in the invoked Uniform Resource Identifier (URI).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-stagename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @stage_name.setter
    def stage_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8baaf29c67665894354e68d63133a330415f0dcbf07a7da1634c3cdb2e98b425)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageName", value)

    @builtins.property
    @jsii.member(jsii_name="tracingEnabled")
    def tracing_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether active X-Ray tracing is enabled for this stage.

        For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-tracingenabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "tracingEnabled"))

    @tracing_enabled.setter
    def tracing_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__298ce72cf0059a948e16e9c72d21ec5b0057d49a03c80205a924fb8f0d0f377c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tracingEnabled", value)

    @builtins.property
    @jsii.member(jsii_name="variables")
    def variables(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value.

        Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-variables
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], jsii.get(self, "variables"))

    @variables.setter
    def variables(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__730594d0c5863b0e91c924ac52bcb21caa0326262521f374133b66ee0c6e4aca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "variables", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnStage.AccessLogSettingProperty",
        jsii_struct_bases=[],
        name_mapping={"destination_arn": "destinationArn", "format": "format"},
    )
    class AccessLogSettingProperty:
        def __init__(
            self,
            *,
            destination_arn: typing.Optional[builtins.str] = None,
            format: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``AccessLogSetting`` property type specifies settings for logging access in this stage.

            ``AccessLogSetting`` is a property of the `AWS::ApiGateway::Stage <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html>`_ resource.

            :param destination_arn: The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs. If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` . This parameter is required to enable access logging.
            :param format: A single line format of the access logs of data, as specified by selected `$context variables <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference>`_ . The format must include at least ``$context.requestId`` . This parameter is required to enable access logging.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-accesslogsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                access_log_setting_property = apigateway.CfnStage.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__317c72bf207407d861f932d433b56355267c742be1f8d5cb978682f748ce14eb)
                check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination_arn is not None:
                self._values["destination_arn"] = destination_arn
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def destination_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs.

            If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` . This parameter is required to enable access logging.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-accesslogsetting.html#cfn-apigateway-stage-accesslogsetting-destinationarn
            '''
            result = self._values.get("destination_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def format(self) -> typing.Optional[builtins.str]:
            '''A single line format of the access logs of data, as specified by selected `$context variables <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference>`_ . The format must include at least ``$context.requestId`` . This parameter is required to enable access logging.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-accesslogsetting.html#cfn-apigateway-stage-accesslogsetting-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnStage.CanarySettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "deployment_id": "deploymentId",
            "percent_traffic": "percentTraffic",
            "stage_variable_overrides": "stageVariableOverrides",
            "use_stage_cache": "useStageCache",
        },
    )
    class CanarySettingProperty:
        def __init__(
            self,
            *,
            deployment_id: typing.Optional[builtins.str] = None,
            percent_traffic: typing.Optional[jsii.Number] = None,
            stage_variable_overrides: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            use_stage_cache: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The ``CanarySetting`` property type specifies settings for the canary deployment in this stage.

            ``CanarySetting`` is a property of the `AWS::ApiGateway::Stage <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html>`_ resource.

            :param deployment_id: The identifier of the deployment that the stage points to.
            :param percent_traffic: The percentage (0-100) of traffic diverted to a canary deployment.
            :param stage_variable_overrides: Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary. These stage variables are represented as a string-to-string map between stage variable names and their values. Duplicates are not allowed.
            :param use_stage_cache: Whether the canary deployment uses the stage cache or not.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                canary_setting_property = apigateway.CfnStage.CanarySettingProperty(
                    deployment_id="deploymentId",
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cdd13602c062e2efe1009cf02ec0b0f336f8551dd4044955686937b4edb98c76)
                check_type(argname="argument deployment_id", value=deployment_id, expected_type=type_hints["deployment_id"])
                check_type(argname="argument percent_traffic", value=percent_traffic, expected_type=type_hints["percent_traffic"])
                check_type(argname="argument stage_variable_overrides", value=stage_variable_overrides, expected_type=type_hints["stage_variable_overrides"])
                check_type(argname="argument use_stage_cache", value=use_stage_cache, expected_type=type_hints["use_stage_cache"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if deployment_id is not None:
                self._values["deployment_id"] = deployment_id
            if percent_traffic is not None:
                self._values["percent_traffic"] = percent_traffic
            if stage_variable_overrides is not None:
                self._values["stage_variable_overrides"] = stage_variable_overrides
            if use_stage_cache is not None:
                self._values["use_stage_cache"] = use_stage_cache

        @builtins.property
        def deployment_id(self) -> typing.Optional[builtins.str]:
            '''The identifier of the deployment that the stage points to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-deploymentid
            '''
            result = self._values.get("deployment_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def percent_traffic(self) -> typing.Optional[jsii.Number]:
            '''The percentage (0-100) of traffic diverted to a canary deployment.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-percenttraffic
            '''
            result = self._values.get("percent_traffic")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stage_variable_overrides(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary.

            These stage variables are represented as a string-to-string map between stage variable names and their values.

            Duplicates are not allowed.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-stagevariableoverrides
            '''
            result = self._values.get("stage_variable_overrides")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def use_stage_cache(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the canary deployment uses the stage cache or not.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-usestagecache
            '''
            result = self._values.get("use_stage_cache")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CanarySettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnStage.MethodSettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cache_data_encrypted": "cacheDataEncrypted",
            "cache_ttl_in_seconds": "cacheTtlInSeconds",
            "caching_enabled": "cachingEnabled",
            "data_trace_enabled": "dataTraceEnabled",
            "http_method": "httpMethod",
            "logging_level": "loggingLevel",
            "metrics_enabled": "metricsEnabled",
            "resource_path": "resourcePath",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
        },
    )
    class MethodSettingProperty:
        def __init__(
            self,
            *,
            cache_data_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            cache_ttl_in_seconds: typing.Optional[jsii.Number] = None,
            caching_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            http_method: typing.Optional[builtins.str] = None,
            logging_level: typing.Optional[builtins.str] = None,
            metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            resource_path: typing.Optional[builtins.str] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``MethodSetting`` property type configures settings for all methods in a stage.

            The ``MethodSettings`` property of the ``AWS::ApiGateway::Stage`` resource contains a list of ``MethodSetting`` property types.

            :param cache_data_encrypted: Indicates whether the cached responses are encrypted.
            :param cache_ttl_in_seconds: The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.
            :param caching_enabled: Indicates whether responses are cached and returned for requests. You must enable a cache cluster on the stage to cache responses.
            :param data_trace_enabled: Indicates whether data trace logging is enabled for methods in the stage. API Gateway pushes these logs to Amazon CloudWatch Logs.
            :param http_method: The HTTP method. To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .
            :param logging_level: The logging level for this method. For valid values, see the ``loggingLevel`` property of the `Stage <https://docs.aws.amazon.com/apigateway/latest/api/API_MethodSetting.html#apigw-Type-MethodSetting-loggingLevel>`_ resource in the *Amazon API Gateway API Reference* .
            :param metrics_enabled: Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.
            :param resource_path: The resource path for this method. Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ). To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .
            :param throttling_burst_limit: The number of burst requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account . For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param throttling_rate_limit: The number of steady-state requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account . For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                method_setting_property = apigateway.CfnStage.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d1ec731889579d77af8bc7427d63162d1e77c817c43699b8bcbb8900ecfdee7c)
                check_type(argname="argument cache_data_encrypted", value=cache_data_encrypted, expected_type=type_hints["cache_data_encrypted"])
                check_type(argname="argument cache_ttl_in_seconds", value=cache_ttl_in_seconds, expected_type=type_hints["cache_ttl_in_seconds"])
                check_type(argname="argument caching_enabled", value=caching_enabled, expected_type=type_hints["caching_enabled"])
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument metrics_enabled", value=metrics_enabled, expected_type=type_hints["metrics_enabled"])
                check_type(argname="argument resource_path", value=resource_path, expected_type=type_hints["resource_path"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cache_data_encrypted is not None:
                self._values["cache_data_encrypted"] = cache_data_encrypted
            if cache_ttl_in_seconds is not None:
                self._values["cache_ttl_in_seconds"] = cache_ttl_in_seconds
            if caching_enabled is not None:
                self._values["caching_enabled"] = caching_enabled
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if http_method is not None:
                self._values["http_method"] = http_method
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if metrics_enabled is not None:
                self._values["metrics_enabled"] = metrics_enabled
            if resource_path is not None:
                self._values["resource_path"] = resource_path
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit

        @builtins.property
        def cache_data_encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether the cached responses are encrypted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-cachedataencrypted
            '''
            result = self._values.get("cache_data_encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def cache_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
            '''The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-cachettlinseconds
            '''
            result = self._values.get("cache_ttl_in_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def caching_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether responses are cached and returned for requests.

            You must enable a cache cluster on the stage to cache responses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-cachingenabled
            '''
            result = self._values.get("caching_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether data trace logging is enabled for methods in the stage.

            API Gateway pushes these logs to Amazon CloudWatch Logs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-datatraceenabled
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def http_method(self) -> typing.Optional[builtins.str]:
            '''The HTTP method.

            To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-httpmethod
            '''
            result = self._values.get("http_method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''The logging level for this method.

            For valid values, see the ``loggingLevel`` property of the `Stage <https://docs.aws.amazon.com/apigateway/latest/api/API_MethodSetting.html#apigw-Type-MethodSetting-loggingLevel>`_ resource in the *Amazon API Gateway API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-logginglevel
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-metricsenabled
            '''
            result = self._values.get("metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def resource_path(self) -> typing.Optional[builtins.str]:
            '''The resource path for this method.

            Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ). To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-resourcepath
            '''
            result = self._values.get("resource_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''The number of burst requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account .

            For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-throttlingburstlimit
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The number of steady-state requests per second that API Gateway permits across all APIs, stages, and methods in your AWS account .

            For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-throttlingratelimit
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MethodSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnStageProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "access_log_setting": "accessLogSetting",
        "cache_cluster_enabled": "cacheClusterEnabled",
        "cache_cluster_size": "cacheClusterSize",
        "canary_setting": "canarySetting",
        "client_certificate_id": "clientCertificateId",
        "deployment_id": "deploymentId",
        "description": "description",
        "documentation_version": "documentationVersion",
        "method_settings": "methodSettings",
        "stage_name": "stageName",
        "tags": "tags",
        "tracing_enabled": "tracingEnabled",
        "variables": "variables",
    },
)
class CfnStageProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        access_log_setting: typing.Optional[typing.Union[typing.Union[CfnStage.AccessLogSettingProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        cache_cluster_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        cache_cluster_size: typing.Optional[builtins.str] = None,
        canary_setting: typing.Optional[typing.Union[typing.Union[CfnStage.CanarySettingProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        client_certificate_id: typing.Optional[builtins.str] = None,
        deployment_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        documentation_version: typing.Optional[builtins.str] = None,
        method_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union[CfnStage.MethodSettingProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        stage_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        tracing_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        variables: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnStage``.

        :param rest_api_id: The ID of the ``RestApi`` resource that you're deploying with this stage.
        :param access_log_setting: Specifies settings for logging access in this stage.
        :param cache_cluster_enabled: Indicates whether cache clustering is enabled for the stage.
        :param cache_cluster_size: The stage's cache cluster size.
        :param canary_setting: Specifies settings for the canary deployment in this stage.
        :param client_certificate_id: The ID of the client certificate that API Gateway uses to call your integration endpoints in the stage.
        :param deployment_id: The ID of the deployment that the stage is associated with. This parameter is required to create a stage.
        :param description: A description of the stage.
        :param documentation_version: The version ID of the API documentation snapshot.
        :param method_settings: Settings for all methods in the stage.
        :param stage_name: The name of the stage, which API Gateway uses as the first path segment in the invoked Uniform Resource Identifier (URI).
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the stage.
        :param tracing_enabled: Specifies whether active X-Ray tracing is enabled for this stage. For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .
        :param variables: A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value. Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_stage_props = apigateway.CfnStageProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                access_log_setting=apigateway.CfnStage.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                ),
                cache_cluster_enabled=False,
                cache_cluster_size="cacheClusterSize",
                canary_setting=apigateway.CfnStage.CanarySettingProperty(
                    deployment_id="deploymentId",
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                ),
                client_certificate_id="clientCertificateId",
                deployment_id="deploymentId",
                description="description",
                documentation_version="documentationVersion",
                method_settings=[apigateway.CfnStage.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )],
                stage_name="stageName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                tracing_enabled=False,
                variables={
                    "variables_key": "variables"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__273e1da73653ce957f96bbfa0d3f124b04b54c31ebe33ed78cd19aceb57c8a52)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument access_log_setting", value=access_log_setting, expected_type=type_hints["access_log_setting"])
            check_type(argname="argument cache_cluster_enabled", value=cache_cluster_enabled, expected_type=type_hints["cache_cluster_enabled"])
            check_type(argname="argument cache_cluster_size", value=cache_cluster_size, expected_type=type_hints["cache_cluster_size"])
            check_type(argname="argument canary_setting", value=canary_setting, expected_type=type_hints["canary_setting"])
            check_type(argname="argument client_certificate_id", value=client_certificate_id, expected_type=type_hints["client_certificate_id"])
            check_type(argname="argument deployment_id", value=deployment_id, expected_type=type_hints["deployment_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument documentation_version", value=documentation_version, expected_type=type_hints["documentation_version"])
            check_type(argname="argument method_settings", value=method_settings, expected_type=type_hints["method_settings"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument tracing_enabled", value=tracing_enabled, expected_type=type_hints["tracing_enabled"])
            check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if access_log_setting is not None:
            self._values["access_log_setting"] = access_log_setting
        if cache_cluster_enabled is not None:
            self._values["cache_cluster_enabled"] = cache_cluster_enabled
        if cache_cluster_size is not None:
            self._values["cache_cluster_size"] = cache_cluster_size
        if canary_setting is not None:
            self._values["canary_setting"] = canary_setting
        if client_certificate_id is not None:
            self._values["client_certificate_id"] = client_certificate_id
        if deployment_id is not None:
            self._values["deployment_id"] = deployment_id
        if description is not None:
            self._values["description"] = description
        if documentation_version is not None:
            self._values["documentation_version"] = documentation_version
        if method_settings is not None:
            self._values["method_settings"] = method_settings
        if stage_name is not None:
            self._values["stage_name"] = stage_name
        if tags is not None:
            self._values["tags"] = tags
        if tracing_enabled is not None:
            self._values["tracing_enabled"] = tracing_enabled
        if variables is not None:
            self._values["variables"] = variables

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The ID of the ``RestApi`` resource that you're deploying with this stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_log_setting(
        self,
    ) -> typing.Optional[typing.Union[CfnStage.AccessLogSettingProperty, _IResolvable_a771d0ef]]:
        '''Specifies settings for logging access in this stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-accesslogsetting
        '''
        result = self._values.get("access_log_setting")
        return typing.cast(typing.Optional[typing.Union[CfnStage.AccessLogSettingProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def cache_cluster_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Indicates whether cache clustering is enabled for the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-cacheclusterenabled
        '''
        result = self._values.get("cache_cluster_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def cache_cluster_size(self) -> typing.Optional[builtins.str]:
        '''The stage's cache cluster size.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-cacheclustersize
        '''
        result = self._values.get("cache_cluster_size")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def canary_setting(
        self,
    ) -> typing.Optional[typing.Union[CfnStage.CanarySettingProperty, _IResolvable_a771d0ef]]:
        '''Specifies settings for the canary deployment in this stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-canarysetting
        '''
        result = self._values.get("canary_setting")
        return typing.cast(typing.Optional[typing.Union[CfnStage.CanarySettingProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def client_certificate_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the client certificate that API Gateway uses to call your integration endpoints in the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-clientcertificateid
        '''
        result = self._values.get("client_certificate_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deployment_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the deployment that the stage is associated with.

        This parameter is required to create a stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-deploymentid
        '''
        result = self._values.get("deployment_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def documentation_version(self) -> typing.Optional[builtins.str]:
        '''The version ID of the API documentation snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-documentationversion
        '''
        result = self._values.get("documentation_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def method_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnStage.MethodSettingProperty, _IResolvable_a771d0ef]]]]:
        '''Settings for all methods in the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-methodsettings
        '''
        result = self._values.get("method_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnStage.MethodSettingProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the stage, which API Gateway uses as the first path segment in the invoked Uniform Resource Identifier (URI).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-stagename
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the stage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def tracing_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether active X-Ray tracing is enabled for this stage.

        For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-tracingenabled
        '''
        result = self._values.get("tracing_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def variables(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
        '''A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value.

        Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-variables
        '''
        result = self._values.get("variables")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnStageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnStageV2(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnStageV2",
):
    '''(deprecated) A CloudFormation ``AWS::ApiGatewayV2::Stage``.

    :deprecated: moved to package aws-apigatewayv2

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html
    :stability: deprecated
    :cloudformationResource: AWS::ApiGatewayV2::Stage
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        # route_settings: Any
        # stage_variables: Any
        # tags: Any
        
        cfn_stage_v2 = apigateway.CfnStageV2(self, "MyCfnStageV2",
            api_id="apiId",
            stage_name="stageName",
        
            # the properties below are optional
            access_log_settings=apigateway.CfnStageV2.AccessLogSettingsProperty(
                destination_arn="destinationArn",
                format="format"
            ),
            auto_deploy=False,
            client_certificate_id="clientCertificateId",
            default_route_settings=apigateway.CfnStageV2.RouteSettingsProperty(
                data_trace_enabled=False,
                detailed_metrics_enabled=False,
                logging_level="loggingLevel",
                throttling_burst_limit=123,
                throttling_rate_limit=123
            ),
            deployment_id="deploymentId",
            description="description",
            route_settings=route_settings,
            stage_variables=stage_variables,
            tags=tags
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_id: builtins.str,
        stage_name: builtins.str,
        access_log_settings: typing.Optional[typing.Union[typing.Union["CfnStageV2.AccessLogSettingsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        auto_deploy: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        client_certificate_id: typing.Optional[builtins.str] = None,
        default_route_settings: typing.Optional[typing.Union[typing.Union["CfnStageV2.RouteSettingsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        deployment_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        route_settings: typing.Any = None,
        stage_variables: typing.Any = None,
        tags: typing.Any = None,
    ) -> None:
        '''(deprecated) Create a new ``AWS::ApiGatewayV2::Stage``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Stage.ApiId``.
        :param stage_name: (deprecated) ``AWS::ApiGatewayV2::Stage.StageName``.
        :param access_log_settings: (deprecated) ``AWS::ApiGatewayV2::Stage.AccessLogSettings``.
        :param auto_deploy: (deprecated) ``AWS::ApiGatewayV2::Stage.AutoDeploy``.
        :param client_certificate_id: (deprecated) ``AWS::ApiGatewayV2::Stage.ClientCertificateId``.
        :param default_route_settings: (deprecated) ``AWS::ApiGatewayV2::Stage.DefaultRouteSettings``.
        :param deployment_id: (deprecated) ``AWS::ApiGatewayV2::Stage.DeploymentId``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Stage.Description``.
        :param route_settings: (deprecated) ``AWS::ApiGatewayV2::Stage.RouteSettings``.
        :param stage_variables: (deprecated) ``AWS::ApiGatewayV2::Stage.StageVariables``.
        :param tags: (deprecated) ``AWS::ApiGatewayV2::Stage.Tags``.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4582fd7fd449730065f7188bb892f26e9944a725f1f9a660f187206fda025311)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnStageV2Props(
            api_id=api_id,
            stage_name=stage_name,
            access_log_settings=access_log_settings,
            auto_deploy=auto_deploy,
            client_certificate_id=client_certificate_id,
            default_route_settings=default_route_settings,
            deployment_id=deployment_id,
            description=description,
            route_settings=route_settings,
            stage_variables=stage_variables,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''(deprecated) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a87e611923f9cb5d5934d410d78916a1a493d8a1b0a6cbbd2d1d7cee2534d0c)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9af3585bf05c55153d8964356e731392b09f0a06a816c121edc066b388ad1016)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''(deprecated) The CloudFormation resource type name for this resource class.

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-tags
        :stability: deprecated
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="apiId")
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-apiid
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiId"))

    @api_id.setter
    def api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc90477cfce2145694f25c88a3b96175a36cac01ac05b75153a83d0bd6ddbde0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiId", value)

    @builtins.property
    @jsii.member(jsii_name="routeSettings")
    def route_settings(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.RouteSettings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-routesettings
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "routeSettings"))

    @route_settings.setter
    def route_settings(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6e0742b4a9bbf51d028dff7f4c6010eeda58ebba4c5f7b77a3a810d7309032a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeSettings", value)

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.StageName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-stagename
        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.get(self, "stageName"))

    @stage_name.setter
    def stage_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0adf8cb53ee32aa9a4ef0aa972554d2625785eda611b42d1c9c670ea0df490bf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageName", value)

    @builtins.property
    @jsii.member(jsii_name="stageVariables")
    def stage_variables(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.StageVariables``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-stagevariables
        :stability: deprecated
        '''
        return typing.cast(typing.Any, jsii.get(self, "stageVariables"))

    @stage_variables.setter
    def stage_variables(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1cdfb97faaf4dddff73516bfc90ff3aafc167ea0002a28a1325b139a40dae70e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageVariables", value)

    @builtins.property
    @jsii.member(jsii_name="accessLogSettings")
    def access_log_settings(
        self,
    ) -> typing.Optional[typing.Union["CfnStageV2.AccessLogSettingsProperty", _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.AccessLogSettings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-accesslogsettings
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union["CfnStageV2.AccessLogSettingsProperty", _IResolvable_a771d0ef]], jsii.get(self, "accessLogSettings"))

    @access_log_settings.setter
    def access_log_settings(
        self,
        value: typing.Optional[typing.Union["CfnStageV2.AccessLogSettingsProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f4145c558c891ef783b6ed059cc337f137ed51e8226b5be7ac4ef3181482b6e8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessLogSettings", value)

    @builtins.property
    @jsii.member(jsii_name="autoDeploy")
    def auto_deploy(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.AutoDeploy``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-autodeploy
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "autoDeploy"))

    @auto_deploy.setter
    def auto_deploy(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42de7eb773bca183fc48364f348f55c51b76879a3878d9613b60df3411431139)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoDeploy", value)

    @builtins.property
    @jsii.member(jsii_name="clientCertificateId")
    def client_certificate_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.ClientCertificateId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-clientcertificateid
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clientCertificateId"))

    @client_certificate_id.setter
    def client_certificate_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__111c87f9a905d9127e5c0e7d3aed45776f3191c6dcea8a5278084ce6fbf7f23c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientCertificateId", value)

    @builtins.property
    @jsii.member(jsii_name="defaultRouteSettings")
    def default_route_settings(
        self,
    ) -> typing.Optional[typing.Union["CfnStageV2.RouteSettingsProperty", _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.DefaultRouteSettings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-defaultroutesettings
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[typing.Union["CfnStageV2.RouteSettingsProperty", _IResolvable_a771d0ef]], jsii.get(self, "defaultRouteSettings"))

    @default_route_settings.setter
    def default_route_settings(
        self,
        value: typing.Optional[typing.Union["CfnStageV2.RouteSettingsProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__71771ab3ff5d528a4b11143e944bf8f6980b256a044b3aa05b560078638a1cd8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultRouteSettings", value)

    @builtins.property
    @jsii.member(jsii_name="deploymentId")
    def deployment_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.DeploymentId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-deploymentid
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deploymentId"))

    @deployment_id.setter
    def deployment_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f3dc2c74c3331ad70a5cb9a6160b01eadd03970b3898a300b0df72f4319d1278)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentId", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-description
        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe7567bf480a7114c512a95cdbfff844d2b3d6cdc76dbf7be5e0252b106dec89)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnStageV2.AccessLogSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"destination_arn": "destinationArn", "format": "format"},
    )
    class AccessLogSettingsProperty:
        def __init__(
            self,
            *,
            destination_arn: typing.Optional[builtins.str] = None,
            format: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param destination_arn: (deprecated) ``CfnStageV2.AccessLogSettingsProperty.DestinationArn``.
            :param format: (deprecated) ``CfnStageV2.AccessLogSettingsProperty.Format``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-accesslogsettings.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                access_log_settings_property = apigateway.CfnStageV2.AccessLogSettingsProperty(
                    destination_arn="destinationArn",
                    format="format"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cc262678393f99ae2f7d2e22cfd43797ee5555398ca53a1107952ac483da68bc)
                check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination_arn is not None:
                self._values["destination_arn"] = destination_arn
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def destination_arn(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnStageV2.AccessLogSettingsProperty.DestinationArn``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-accesslogsettings.html#cfn-apigatewayv2-stage-accesslogsettings-destinationarn
            :stability: deprecated
            '''
            result = self._values.get("destination_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def format(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnStageV2.AccessLogSettingsProperty.Format``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-accesslogsettings.html#cfn-apigatewayv2-stage-accesslogsettings-format
            :stability: deprecated
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnStageV2.RouteSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "data_trace_enabled": "dataTraceEnabled",
            "detailed_metrics_enabled": "detailedMetricsEnabled",
            "logging_level": "loggingLevel",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
        },
    )
    class RouteSettingsProperty:
        def __init__(
            self,
            *,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            detailed_metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            logging_level: typing.Optional[builtins.str] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param data_trace_enabled: (deprecated) ``CfnStageV2.RouteSettingsProperty.DataTraceEnabled``.
            :param detailed_metrics_enabled: (deprecated) ``CfnStageV2.RouteSettingsProperty.DetailedMetricsEnabled``.
            :param logging_level: (deprecated) ``CfnStageV2.RouteSettingsProperty.LoggingLevel``.
            :param throttling_burst_limit: (deprecated) ``CfnStageV2.RouteSettingsProperty.ThrottlingBurstLimit``.
            :param throttling_rate_limit: (deprecated) ``CfnStageV2.RouteSettingsProperty.ThrottlingRateLimit``.

            :deprecated: moved to package aws-apigatewayv2

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-routesettings.html
            :stability: deprecated
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                route_settings_property = apigateway.CfnStageV2.RouteSettingsProperty(
                    data_trace_enabled=False,
                    detailed_metrics_enabled=False,
                    logging_level="loggingLevel",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e00e9c5eb6f6b2b4609bebac14c087bc6fc071c8e58ca5b13e24febd869b336f)
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument detailed_metrics_enabled", value=detailed_metrics_enabled, expected_type=type_hints["detailed_metrics_enabled"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if detailed_metrics_enabled is not None:
                self._values["detailed_metrics_enabled"] = detailed_metrics_enabled
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''(deprecated) ``CfnStageV2.RouteSettingsProperty.DataTraceEnabled``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-routesettings.html#cfn-apigatewayv2-stage-routesettings-datatraceenabled
            :stability: deprecated
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def detailed_metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''(deprecated) ``CfnStageV2.RouteSettingsProperty.DetailedMetricsEnabled``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-routesettings.html#cfn-apigatewayv2-stage-routesettings-detailedmetricsenabled
            :stability: deprecated
            '''
            result = self._values.get("detailed_metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''(deprecated) ``CfnStageV2.RouteSettingsProperty.LoggingLevel``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-routesettings.html#cfn-apigatewayv2-stage-routesettings-logginglevel
            :stability: deprecated
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''(deprecated) ``CfnStageV2.RouteSettingsProperty.ThrottlingBurstLimit``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-routesettings.html#cfn-apigatewayv2-stage-routesettings-throttlingburstlimit
            :stability: deprecated
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''(deprecated) ``CfnStageV2.RouteSettingsProperty.ThrottlingRateLimit``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigatewayv2-stage-routesettings.html#cfn-apigatewayv2-stage-routesettings-throttlingratelimit
            :stability: deprecated
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RouteSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnStageV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "stage_name": "stageName",
        "access_log_settings": "accessLogSettings",
        "auto_deploy": "autoDeploy",
        "client_certificate_id": "clientCertificateId",
        "default_route_settings": "defaultRouteSettings",
        "deployment_id": "deploymentId",
        "description": "description",
        "route_settings": "routeSettings",
        "stage_variables": "stageVariables",
        "tags": "tags",
    },
)
class CfnStageV2Props:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        stage_name: builtins.str,
        access_log_settings: typing.Optional[typing.Union[typing.Union[CfnStageV2.AccessLogSettingsProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        auto_deploy: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        client_certificate_id: typing.Optional[builtins.str] = None,
        default_route_settings: typing.Optional[typing.Union[typing.Union[CfnStageV2.RouteSettingsProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        deployment_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        route_settings: typing.Any = None,
        stage_variables: typing.Any = None,
        tags: typing.Any = None,
    ) -> None:
        '''(deprecated) Properties for defining a ``AWS::ApiGatewayV2::Stage``.

        :param api_id: (deprecated) ``AWS::ApiGatewayV2::Stage.ApiId``.
        :param stage_name: (deprecated) ``AWS::ApiGatewayV2::Stage.StageName``.
        :param access_log_settings: (deprecated) ``AWS::ApiGatewayV2::Stage.AccessLogSettings``.
        :param auto_deploy: (deprecated) ``AWS::ApiGatewayV2::Stage.AutoDeploy``.
        :param client_certificate_id: (deprecated) ``AWS::ApiGatewayV2::Stage.ClientCertificateId``.
        :param default_route_settings: (deprecated) ``AWS::ApiGatewayV2::Stage.DefaultRouteSettings``.
        :param deployment_id: (deprecated) ``AWS::ApiGatewayV2::Stage.DeploymentId``.
        :param description: (deprecated) ``AWS::ApiGatewayV2::Stage.Description``.
        :param route_settings: (deprecated) ``AWS::ApiGatewayV2::Stage.RouteSettings``.
        :param stage_variables: (deprecated) ``AWS::ApiGatewayV2::Stage.StageVariables``.
        :param tags: (deprecated) ``AWS::ApiGatewayV2::Stage.Tags``.

        :deprecated: moved to package aws-apigatewayv2

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html
        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            # route_settings: Any
            # stage_variables: Any
            # tags: Any
            
            cfn_stage_v2_props = apigateway.CfnStageV2Props(
                api_id="apiId",
                stage_name="stageName",
            
                # the properties below are optional
                access_log_settings=apigateway.CfnStageV2.AccessLogSettingsProperty(
                    destination_arn="destinationArn",
                    format="format"
                ),
                auto_deploy=False,
                client_certificate_id="clientCertificateId",
                default_route_settings=apigateway.CfnStageV2.RouteSettingsProperty(
                    data_trace_enabled=False,
                    detailed_metrics_enabled=False,
                    logging_level="loggingLevel",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                ),
                deployment_id="deploymentId",
                description="description",
                route_settings=route_settings,
                stage_variables=stage_variables,
                tags=tags
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0fc3e43e47f77925e040a8d7557f2bdb72fb35f978ea667d7d6cd954ceadec5)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            check_type(argname="argument access_log_settings", value=access_log_settings, expected_type=type_hints["access_log_settings"])
            check_type(argname="argument auto_deploy", value=auto_deploy, expected_type=type_hints["auto_deploy"])
            check_type(argname="argument client_certificate_id", value=client_certificate_id, expected_type=type_hints["client_certificate_id"])
            check_type(argname="argument default_route_settings", value=default_route_settings, expected_type=type_hints["default_route_settings"])
            check_type(argname="argument deployment_id", value=deployment_id, expected_type=type_hints["deployment_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument route_settings", value=route_settings, expected_type=type_hints["route_settings"])
            check_type(argname="argument stage_variables", value=stage_variables, expected_type=type_hints["stage_variables"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "stage_name": stage_name,
        }
        if access_log_settings is not None:
            self._values["access_log_settings"] = access_log_settings
        if auto_deploy is not None:
            self._values["auto_deploy"] = auto_deploy
        if client_certificate_id is not None:
            self._values["client_certificate_id"] = client_certificate_id
        if default_route_settings is not None:
            self._values["default_route_settings"] = default_route_settings
        if deployment_id is not None:
            self._values["deployment_id"] = deployment_id
        if description is not None:
            self._values["description"] = description
        if route_settings is not None:
            self._values["route_settings"] = route_settings
        if stage_variables is not None:
            self._values["stage_variables"] = stage_variables
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.ApiId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-apiid
        :stability: deprecated
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def stage_name(self) -> builtins.str:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.StageName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-stagename
        :stability: deprecated
        '''
        result = self._values.get("stage_name")
        assert result is not None, "Required property 'stage_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_log_settings(
        self,
    ) -> typing.Optional[typing.Union[CfnStageV2.AccessLogSettingsProperty, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.AccessLogSettings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-accesslogsettings
        :stability: deprecated
        '''
        result = self._values.get("access_log_settings")
        return typing.cast(typing.Optional[typing.Union[CfnStageV2.AccessLogSettingsProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def auto_deploy(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.AutoDeploy``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-autodeploy
        :stability: deprecated
        '''
        result = self._values.get("auto_deploy")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def client_certificate_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.ClientCertificateId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-clientcertificateid
        :stability: deprecated
        '''
        result = self._values.get("client_certificate_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_route_settings(
        self,
    ) -> typing.Optional[typing.Union[CfnStageV2.RouteSettingsProperty, _IResolvable_a771d0ef]]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.DefaultRouteSettings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-defaultroutesettings
        :stability: deprecated
        '''
        result = self._values.get("default_route_settings")
        return typing.cast(typing.Optional[typing.Union[CfnStageV2.RouteSettingsProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def deployment_id(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.DeploymentId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-deploymentid
        :stability: deprecated
        '''
        result = self._values.get("deployment_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-description
        :stability: deprecated
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def route_settings(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.RouteSettings``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-routesettings
        :stability: deprecated
        '''
        result = self._values.get("route_settings")
        return typing.cast(typing.Any, result)

    @builtins.property
    def stage_variables(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.StageVariables``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-stagevariables
        :stability: deprecated
        '''
        result = self._values.get("stage_variables")
        return typing.cast(typing.Any, result)

    @builtins.property
    def tags(self) -> typing.Any:
        '''(deprecated) ``AWS::ApiGatewayV2::Stage.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigatewayv2-stage.html#cfn-apigatewayv2-stage-tags
        :stability: deprecated
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnStageV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnUsagePlan(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnUsagePlan",
):
    '''A CloudFormation ``AWS::ApiGateway::UsagePlan``.

    The ``AWS::ApiGateway::UsagePlan`` resource creates a usage plan for deployed APIs. A usage plan sets a target for the throttling and quota limits on individual client API keys. For more information, see `Creating and Using API Usage Plans in Amazon API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html>`_ in the *API Gateway Developer Guide* .

    In some cases clients can exceed the targets that you set. Don’t rely on usage plans to control costs. Consider using `AWS Budgets <https://docs.aws.amazon.com/cost-management/latest/userguide/budgets-managing-costs.html>`_ to monitor costs and `AWS WAF <https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html>`_ to manage API requests.

    :cloudformationResource: AWS::ApiGateway::UsagePlan
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_usage_plan = apigateway.CfnUsagePlan(self, "MyCfnUsagePlan",
            api_stages=[apigateway.CfnUsagePlan.ApiStageProperty(
                api_id="apiId",
                stage="stage",
                throttle={
                    "throttle_key": apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                        burst_limit=123,
                        rate_limit=123
                    )
                }
            )],
            description="description",
            quota=apigateway.CfnUsagePlan.QuotaSettingsProperty(
                limit=123,
                offset=123,
                period="period"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            throttle=apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                burst_limit=123,
                rate_limit=123
            ),
            usage_plan_name="usagePlanName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        api_stages: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union["CfnUsagePlan.ApiStageProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        description: typing.Optional[builtins.str] = None,
        quota: typing.Optional[typing.Union[typing.Union["CfnUsagePlan.QuotaSettingsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        throttle: typing.Optional[typing.Union[typing.Union["CfnUsagePlan.ThrottleSettingsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        usage_plan_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::UsagePlan``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param api_stages: The API stages to associate with this usage plan.
        :param description: A description of the usage plan.
        :param quota: Configures the number of requests that users can make within a given interval.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the usage plan.
        :param throttle: Configures the overall request rate (average requests per second) and burst capacity.
        :param usage_plan_name: A name for the usage plan.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3de18e5ccd0bd14a18ad11095251954e5442a1fb3b662374d1c551ad3ec1568b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUsagePlanProps(
            api_stages=api_stages,
            description=description,
            quota=quota,
            tags=tags,
            throttle=throttle,
            usage_plan_name=usage_plan_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__863541a65c9fbfb37f819a62dddd4a7370cef0238bc33b623dbc3a51ee1439f1)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__510457fe22edd86c875d64272465943efc8c2926b5a04f40461d5102c13418bd)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the usage plan.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="apiStages")
    def api_stages(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnUsagePlan.ApiStageProperty", _IResolvable_a771d0ef]]]]:
        '''The API stages to associate with this usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-apistages
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnUsagePlan.ApiStageProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "apiStages"))

    @api_stages.setter
    def api_stages(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnUsagePlan.ApiStageProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54c685a5337a2f7124e8293778aa56d4ff2e10dd35141f13e9ae4a12df1171ab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiStages", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5dbe389d26a6129020ef52c3e1c4f71be2bdcfce04352797eba5a2abd7853219)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="quota")
    def quota(
        self,
    ) -> typing.Optional[typing.Union["CfnUsagePlan.QuotaSettingsProperty", _IResolvable_a771d0ef]]:
        '''Configures the number of requests that users can make within a given interval.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-quota
        '''
        return typing.cast(typing.Optional[typing.Union["CfnUsagePlan.QuotaSettingsProperty", _IResolvable_a771d0ef]], jsii.get(self, "quota"))

    @quota.setter
    def quota(
        self,
        value: typing.Optional[typing.Union["CfnUsagePlan.QuotaSettingsProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__775accdcb6128ef0170e9b4645f0cf34027dabbaf80684a58779b672c92f5f51)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "quota", value)

    @builtins.property
    @jsii.member(jsii_name="throttle")
    def throttle(
        self,
    ) -> typing.Optional[typing.Union["CfnUsagePlan.ThrottleSettingsProperty", _IResolvable_a771d0ef]]:
        '''Configures the overall request rate (average requests per second) and burst capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-throttle
        '''
        return typing.cast(typing.Optional[typing.Union["CfnUsagePlan.ThrottleSettingsProperty", _IResolvable_a771d0ef]], jsii.get(self, "throttle"))

    @throttle.setter
    def throttle(
        self,
        value: typing.Optional[typing.Union["CfnUsagePlan.ThrottleSettingsProperty", _IResolvable_a771d0ef]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29eeb8da8467e6389f286d44fcf3ee81785d577b8177d0dfabfe4fcc99888ad3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "throttle", value)

    @builtins.property
    @jsii.member(jsii_name="usagePlanName")
    def usage_plan_name(self) -> typing.Optional[builtins.str]:
        '''A name for the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-usageplanname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "usagePlanName"))

    @usage_plan_name.setter
    def usage_plan_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b70a89c247f9054d9fbc99310dead1307e6f947b4e08bb34c812afb00a4539ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "usagePlanName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnUsagePlan.ApiStageProperty",
        jsii_struct_bases=[],
        name_mapping={"api_id": "apiId", "stage": "stage", "throttle": "throttle"},
    )
    class ApiStageProperty:
        def __init__(
            self,
            *,
            api_id: typing.Optional[builtins.str] = None,
            stage: typing.Optional[builtins.str] = None,
            throttle: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union[typing.Union["CfnUsagePlan.ThrottleSettingsProperty", typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''``ApiStage`` is a property of the `AWS::ApiGateway::UsagePlan <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html>`_ resource that specifies which stages and APIs to associate with a usage plan.

            :param api_id: The ID of an API that is in the specified ``Stage`` property that you want to associate with the usage plan.
            :param stage: The name of the stage to associate with the usage plan.
            :param throttle: Map containing method-level throttling information for an API stage in a usage plan. The key for the map is the path and method for which to configure custom throttling, for example, "/pets/GET". Duplicates are not allowed.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                api_stage_property = apigateway.CfnUsagePlan.ApiStageProperty(
                    api_id="apiId",
                    stage="stage",
                    throttle={
                        "throttle_key": apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                            burst_limit=123,
                            rate_limit=123
                        )
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0958aed2dcb919096aabfce696649a346087a6d033f60d8f24078f8120d6d6eb)
                check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
                check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
                check_type(argname="argument throttle", value=throttle, expected_type=type_hints["throttle"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if api_id is not None:
                self._values["api_id"] = api_id
            if stage is not None:
                self._values["stage"] = stage
            if throttle is not None:
                self._values["throttle"] = throttle

        @builtins.property
        def api_id(self) -> typing.Optional[builtins.str]:
            '''The ID of an API that is in the specified ``Stage`` property that you want to associate with the usage plan.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html#cfn-apigateway-usageplan-apistage-apiid
            '''
            result = self._values.get("api_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def stage(self) -> typing.Optional[builtins.str]:
            '''The name of the stage to associate with the usage plan.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html#cfn-apigateway-usageplan-apistage-stage
            '''
            result = self._values.get("stage")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union["CfnUsagePlan.ThrottleSettingsProperty", _IResolvable_a771d0ef]]]]:
            '''Map containing method-level throttling information for an API stage in a usage plan.

            The key for the map is the path and method for which to configure custom throttling, for example, "/pets/GET".

            Duplicates are not allowed.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html#cfn-apigateway-usageplan-apistage-throttle
            '''
            result = self._values.get("throttle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, typing.Union["CfnUsagePlan.ThrottleSettingsProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ApiStageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnUsagePlan.QuotaSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"limit": "limit", "offset": "offset", "period": "period"},
    )
    class QuotaSettingsProperty:
        def __init__(
            self,
            *,
            limit: typing.Optional[jsii.Number] = None,
            offset: typing.Optional[jsii.Number] = None,
            period: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``QuotaSettings`` is a property of the `AWS::ApiGateway::UsagePlan <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html>`_ resource that specifies a target for the maximum number of requests users can make to your REST APIs.

            In some cases clients can exceed the targets that you set. Don’t rely on usage plans to control costs. Consider using `AWS Budgets <https://docs.aws.amazon.com/cost-management/latest/userguide/budgets-managing-costs.html>`_ to monitor costs and `AWS WAF <https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html>`_ to manage API requests.

            :param limit: The target maximum number of requests that can be made in a given time period.
            :param offset: The day that a time period starts. For example, with a time period of ``WEEK`` , an offset of ``0`` starts on Sunday, and an offset of ``1`` starts on Monday.
            :param period: The time period for which the target maximum limit of requests applies, such as ``DAY`` or ``WEEK`` . For valid values, see the period property for the `UsagePlan <https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlan.html>`_ resource in the *Amazon API Gateway REST API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                quota_settings_property = apigateway.CfnUsagePlan.QuotaSettingsProperty(
                    limit=123,
                    offset=123,
                    period="period"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__eb40c5b17023ef0a96ffaa25371ca3809f7e31a88b5a65293c6da17096a37046)
                check_type(argname="argument limit", value=limit, expected_type=type_hints["limit"])
                check_type(argname="argument offset", value=offset, expected_type=type_hints["offset"])
                check_type(argname="argument period", value=period, expected_type=type_hints["period"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if limit is not None:
                self._values["limit"] = limit
            if offset is not None:
                self._values["offset"] = offset
            if period is not None:
                self._values["period"] = period

        @builtins.property
        def limit(self) -> typing.Optional[jsii.Number]:
            '''The target maximum number of requests that can be made in a given time period.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html#cfn-apigateway-usageplan-quotasettings-limit
            '''
            result = self._values.get("limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def offset(self) -> typing.Optional[jsii.Number]:
            '''The day that a time period starts.

            For example, with a time period of ``WEEK`` , an offset of ``0`` starts on Sunday, and an offset of ``1`` starts on Monday.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html#cfn-apigateway-usageplan-quotasettings-offset
            '''
            result = self._values.get("offset")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def period(self) -> typing.Optional[builtins.str]:
            '''The time period for which the target maximum limit of requests applies, such as ``DAY`` or ``WEEK`` .

            For valid values, see the period property for the `UsagePlan <https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlan.html>`_ resource in the *Amazon API Gateway REST API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html#cfn-apigateway-usageplan-quotasettings-period
            '''
            result = self._values.get("period")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QuotaSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_apigateway.CfnUsagePlan.ThrottleSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"burst_limit": "burstLimit", "rate_limit": "rateLimit"},
    )
    class ThrottleSettingsProperty:
        def __init__(
            self,
            *,
            burst_limit: typing.Optional[jsii.Number] = None,
            rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``ThrottleSettings`` is a property of the `AWS::ApiGateway::UsagePlan <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html>`_ resource that specifies the overall request rate (average requests per second) and burst capacity when users call your REST APIs.

            :param burst_limit: The API target request burst rate limit. This allows more requests through for a period of time than the target rate limit. For more information about request throttling, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param rate_limit: The API target request steady-state rate limit. For more information about request throttling, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-throttlesettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_apigateway as apigateway
                
                throttle_settings_property = apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                    burst_limit=123,
                    rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7d85a9ed9fcd4f2eaa9a05f906b507e6da6213524f486f3cebfdcf42aef7b2b5)
                check_type(argname="argument burst_limit", value=burst_limit, expected_type=type_hints["burst_limit"])
                check_type(argname="argument rate_limit", value=rate_limit, expected_type=type_hints["rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if burst_limit is not None:
                self._values["burst_limit"] = burst_limit
            if rate_limit is not None:
                self._values["rate_limit"] = rate_limit

        @builtins.property
        def burst_limit(self) -> typing.Optional[jsii.Number]:
            '''The API target request burst rate limit.

            This allows more requests through for a period of time than the target rate limit. For more information about request throttling, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-throttlesettings.html#cfn-apigateway-usageplan-throttlesettings-burstlimit
            '''
            result = self._values.get("burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The API target request steady-state rate limit.

            For more information about request throttling, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-throttlesettings.html#cfn-apigateway-usageplan-throttlesettings-ratelimit
            '''
            result = self._values.get("rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ThrottleSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_82c04a63)
class CfnUsagePlanKey(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnUsagePlanKey",
):
    '''A CloudFormation ``AWS::ApiGateway::UsagePlanKey``.

    The ``AWS::ApiGateway::UsagePlanKey`` resource associates an API key with a usage plan. This association determines which users the usage plan is applied to.

    :cloudformationResource: AWS::ApiGateway::UsagePlanKey
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_usage_plan_key = apigateway.CfnUsagePlanKey(self, "MyCfnUsagePlanKey",
            key_id="keyId",
            key_type="keyType",
            usage_plan_id="usagePlanId"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        key_id: builtins.str,
        key_type: builtins.str,
        usage_plan_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::UsagePlanKey``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param key_id: The ID of the usage plan key.
        :param key_type: The type of usage plan key. Currently, the only valid key type is ``API_KEY`` .
        :param usage_plan_id: The ID of the usage plan.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__291c5fe585f3b59367c15bd1df385c4c7e7fcf8d810604dfd2bf512c8679b01e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUsagePlanKeyProps(
            key_id=key_id, key_type=key_type, usage_plan_id=usage_plan_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2276b9256f38a7cf1733d34a4566a600e3987ff424d5f873da20999893c0c832)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb95da083c6f83be32c298fc75f41e8dfdd14c6e003ff59acdf1c8e692b3be1b)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="keyId")
    def key_id(self) -> builtins.str:
        '''The ID of the usage plan key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-keyid
        '''
        return typing.cast(builtins.str, jsii.get(self, "keyId"))

    @key_id.setter
    def key_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__89cafe4a9a4704b0580422ee60dc7eab0a897d00ccdaf1151b60e345d19bf08f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "keyId", value)

    @builtins.property
    @jsii.member(jsii_name="keyType")
    def key_type(self) -> builtins.str:
        '''The type of usage plan key.

        Currently, the only valid key type is ``API_KEY`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-keytype
        '''
        return typing.cast(builtins.str, jsii.get(self, "keyType"))

    @key_type.setter
    def key_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b73f74dc09b8f8534048acf0cc97bcc6ae286333a02ae79ecd6260e78db79109)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "keyType", value)

    @builtins.property
    @jsii.member(jsii_name="usagePlanId")
    def usage_plan_id(self) -> builtins.str:
        '''The ID of the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-usageplanid
        '''
        return typing.cast(builtins.str, jsii.get(self, "usagePlanId"))

    @usage_plan_id.setter
    def usage_plan_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a65d63b10f667215e06ec66cdb0f61e71b60676d914a983a9ccc47ca9695bd5d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "usagePlanId", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnUsagePlanKeyProps",
    jsii_struct_bases=[],
    name_mapping={
        "key_id": "keyId",
        "key_type": "keyType",
        "usage_plan_id": "usagePlanId",
    },
)
class CfnUsagePlanKeyProps:
    def __init__(
        self,
        *,
        key_id: builtins.str,
        key_type: builtins.str,
        usage_plan_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnUsagePlanKey``.

        :param key_id: The ID of the usage plan key.
        :param key_type: The type of usage plan key. Currently, the only valid key type is ``API_KEY`` .
        :param usage_plan_id: The ID of the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_usage_plan_key_props = apigateway.CfnUsagePlanKeyProps(
                key_id="keyId",
                key_type="keyType",
                usage_plan_id="usagePlanId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__456d3f561f2bd9b5357b668df8a338746258b05faa6083f9d601e789c9a90d97)
            check_type(argname="argument key_id", value=key_id, expected_type=type_hints["key_id"])
            check_type(argname="argument key_type", value=key_type, expected_type=type_hints["key_type"])
            check_type(argname="argument usage_plan_id", value=usage_plan_id, expected_type=type_hints["usage_plan_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key_id": key_id,
            "key_type": key_type,
            "usage_plan_id": usage_plan_id,
        }

    @builtins.property
    def key_id(self) -> builtins.str:
        '''The ID of the usage plan key.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-keyid
        '''
        result = self._values.get("key_id")
        assert result is not None, "Required property 'key_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def key_type(self) -> builtins.str:
        '''The type of usage plan key.

        Currently, the only valid key type is ``API_KEY`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-keytype
        '''
        result = self._values.get("key_type")
        assert result is not None, "Required property 'key_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def usage_plan_id(self) -> builtins.str:
        '''The ID of the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-usageplanid
        '''
        result = self._values.get("usage_plan_id")
        assert result is not None, "Required property 'usage_plan_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUsagePlanKeyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnUsagePlanProps",
    jsii_struct_bases=[],
    name_mapping={
        "api_stages": "apiStages",
        "description": "description",
        "quota": "quota",
        "tags": "tags",
        "throttle": "throttle",
        "usage_plan_name": "usagePlanName",
    },
)
class CfnUsagePlanProps:
    def __init__(
        self,
        *,
        api_stages: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[typing.Union[CfnUsagePlan.ApiStageProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]]]] = None,
        description: typing.Optional[builtins.str] = None,
        quota: typing.Optional[typing.Union[typing.Union[CfnUsagePlan.QuotaSettingsProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
        throttle: typing.Optional[typing.Union[typing.Union[CfnUsagePlan.ThrottleSettingsProperty, typing.Dict[builtins.str, typing.Any]], _IResolvable_a771d0ef]] = None,
        usage_plan_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUsagePlan``.

        :param api_stages: The API stages to associate with this usage plan.
        :param description: A description of the usage plan.
        :param quota: Configures the number of requests that users can make within a given interval.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the usage plan.
        :param throttle: Configures the overall request rate (average requests per second) and burst capacity.
        :param usage_plan_name: A name for the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_apigateway as apigateway
            
            cfn_usage_plan_props = apigateway.CfnUsagePlanProps(
                api_stages=[apigateway.CfnUsagePlan.ApiStageProperty(
                    api_id="apiId",
                    stage="stage",
                    throttle={
                        "throttle_key": apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                            burst_limit=123,
                            rate_limit=123
                        )
                    }
                )],
                description="description",
                quota=apigateway.CfnUsagePlan.QuotaSettingsProperty(
                    limit=123,
                    offset=123,
                    period="period"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                throttle=apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                    burst_limit=123,
                    rate_limit=123
                ),
                usage_plan_name="usagePlanName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1fcf01f4f4c7380662d43f0b70be9b0550f2223a4665fffc0ae7a25714ee9f65)
            check_type(argname="argument api_stages", value=api_stages, expected_type=type_hints["api_stages"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument quota", value=quota, expected_type=type_hints["quota"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument throttle", value=throttle, expected_type=type_hints["throttle"])
            check_type(argname="argument usage_plan_name", value=usage_plan_name, expected_type=type_hints["usage_plan_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if api_stages is not None:
            self._values["api_stages"] = api_stages
        if description is not None:
            self._values["description"] = description
        if quota is not None:
            self._values["quota"] = quota
        if tags is not None:
            self._values["tags"] = tags
        if throttle is not None:
            self._values["throttle"] = throttle
        if usage_plan_name is not None:
            self._values["usage_plan_name"] = usage_plan_name

    @builtins.property
    def api_stages(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnUsagePlan.ApiStageProperty, _IResolvable_a771d0ef]]]]:
        '''The API stages to associate with this usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-apistages
        '''
        result = self._values.get("api_stages")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnUsagePlan.ApiStageProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def quota(
        self,
    ) -> typing.Optional[typing.Union[CfnUsagePlan.QuotaSettingsProperty, _IResolvable_a771d0ef]]:
        '''Configures the number of requests that users can make within a given interval.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-quota
        '''
        result = self._values.get("quota")
        return typing.cast(typing.Optional[typing.Union[CfnUsagePlan.QuotaSettingsProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def throttle(
        self,
    ) -> typing.Optional[typing.Union[CfnUsagePlan.ThrottleSettingsProperty, _IResolvable_a771d0ef]]:
        '''Configures the overall request rate (average requests per second) and burst capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-throttle
        '''
        result = self._values.get("throttle")
        return typing.cast(typing.Optional[typing.Union[CfnUsagePlan.ThrottleSettingsProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def usage_plan_name(self) -> typing.Optional[builtins.str]:
        '''A name for the usage plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-usageplanname
        '''
        result = self._values.get("usage_plan_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUsagePlanProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVpcLink(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_apigateway.CfnVpcLink",
):
    '''A CloudFormation ``AWS::ApiGateway::VpcLink``.

    The ``AWS::ApiGateway::VpcLink`` resource creates an API Gateway VPC link for a REST API to access resources in an Amazon Virtual Private Cloud (VPC). For more information, see `vpclink:create <https://docs.aws.amazon.com/apigateway/latest/api/API_CreateVpcLink.html>`_ in the ``Amazon API Gateway REST API Reference`` .

    :cloudformationResource: AWS::ApiGateway::VpcLink
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_apigateway as apigateway
        
        cfn_vpc_link = apigateway.CfnVpcLink(self, "MyCfnVpcLink",
            name="name",
            target_arns=["targetArns"],
        
            # the properties below are optional
            description="description",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        name: builtins.str,
        target_arns: typing.Sequence[builtins.str],
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ApiGateway::VpcLink``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param name: A name for the VPC link.
        :param target_arns: The ARN of network load balancer of the VPC targeted by the VPC link. The network load balancer must be owned by the same AWS account of the API owner.
        :param description: A description of the VPC link.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the VPC link.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__efeed08fd836ae8f40c05162a99eaa25df4ccfd557f86e1c03be6a68570c77e9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVpcLinkProps(
            name=name, target_arns=target_arns, description=description, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b573a2381d7b4c215570fabff0936d5ec7ba33e5330a33708da9b9eddcc97ea3)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__12dc17bc7821374ef725575549abde2b2f574dde75fa0cd891e6bf7f810e6c14)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''An array of arbitrary tags (key-value pairs) to associate with the VPC link.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''A name for the VPC link.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-name
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7d157a82436733cdaea2959eb45321d4f62193b283afb1d56c21073a13ea3657)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="targetArns")
    def target_arns(self) -> typing.List[builtins.str]:
        '''The ARN of network load balancer of the VPC targeted by the VPC link.

        The network load balancer must be owned by the same AWS account of the API owner.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-targetarns
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "targetArns"))

    @target_arns.setter
    def target_arns(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e480c3c7fc2b1d2d4a3acb50a110781395208ceff0dfebc57ff9283befdf28a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetArns", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the VPC link.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__af1c2c1506d5359491c1d8832665802c5016a047517acef7fdeb055ae76e9993)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="monocdk.aws_apigateway.CfnVpcLinkProps",
    jsii_struct_bases=[],
    name_mapping={
        "name": "name",
        "target_arns": "targetArns",
        "description": "description",
        "tags": "tags",
    },
)
class CfnVpcLinkProps:
    def __init__(
        self,
        *,
        name: builtins.str,
        target_arns: typing.Sequence[builtins.str],
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_95fbdc29, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVpcLink``.

        :param name: A name for the VPC link.
        :param target_arns: The ARN of network load balancer of the VPC targeted by the VPC link. The network load balancer must be owned by the same AWS account of the API owner.
        :param description: A description of the VPC lin