# -*- coding: utf-8 -*-
#
#   DIMP : Decentralized Instant Messaging Protocol
#
#                                Written in 2019 by Moky <albert.moky@gmail.com>
#
# ==============================================================================
# MIT License
#
# Copyright (c) 2019 Albert Moky
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# ==============================================================================

from typing import Optional, Union, Any, Dict

from mkm.crypto import base64_encode, base64_decode
from mkm.crypto import SymmetricKey

from dkd import ContentType, BaseContent

from ..protocol import FileContent, ImageContent, AudioContent, VideoContent


class BaseFileContent(BaseContent, FileContent):
    """
        File Message Content
        ~~~~~~~~~~~~~~~~~~~~

        data format: {
            type : 0x10,
            sn   : 123,

            URL      : "http://", // upload to CDN
            data     : "...",     // if (!URL) base64_encode(fileContent)
            filename : "..."
        }
    """

    def __init__(self, content: Optional[Dict[str, Any]] = None,
                 msg_type: Union[int, ContentType] = 0,
                 filename: Optional[str] = None, data: Union[bytes, str, None] = None):
        if content is None and msg_type == 0:
            msg_type = ContentType.FILE
        super().__init__(content=content, msg_type=msg_type)
        # file name
        if filename is not None:
            self['filename'] = filename
        # file data (encoded or binary)
        if data is None:
            self.__attachment = None
        elif isinstance(data, bytes):
            self.__attachment = data
            self['data'] = base64_encode(data=data)
        else:
            assert isinstance(data, str), 'encode file data error: %s' % data
            self.__attachment = None
            self['data'] = data
        # symmetric key for decryption
        self.__password = None

    @property  # Override
    def url(self) -> Optional[str]:
        return self.get('URL')

    @url.setter  # Override
    def url(self, string: str):
        if string is None:
            self.pop('URL', None)
        else:
            self['URL'] = string

    @property  # Override
    def data(self) -> Optional[bytes]:
        if self.__attachment is None:
            base64 = self.get('data')
            if base64 is not None:
                self.__attachment = base64_decode(base64)
        return self.__attachment

    @data.setter  # Override
    def data(self, attachment: bytes):
        if attachment is None:
            self.pop('data', None)
        else:
            self['data'] = base64_encode(attachment)
        self.__attachment = attachment

    @property  # Override
    def filename(self) -> Optional[str]:
        return self.get('filename')

    @filename.setter  # Override
    def filename(self, string: str):
        if string is None:
            self.pop('filename', None)
        else:
            self['filename'] = string

    @property  # Override
    def password(self) -> Optional[SymmetricKey]:
        if self.__password is None:
            key = self.get('password')
            self.__password = SymmetricKey.parse(key=key)
        return self.__password

    @password.setter  # Override
    def password(self, key: SymmetricKey):
        if key is None:
            self.pop('password', None)
        else:
            self['password'] = key.dictionary
        self.__password = key


class ImageFileContent(BaseFileContent, ImageContent):
    """
        Image Message Content
        ~~~~~~~~~~~~~~~~~~~~~

        data format: {
            type : 0x12,
            sn   : 123,

            URL       : "http://", // upload to CDN
            data      : "...",     // if (!URL) base64_encode(image)
            thumbnail : "...",     // base64_encode(smallImage)
            filename  : "..."
        }
    """

    def __init__(self, content: Optional[Dict[str, Any]] = None,
                 filename: Optional[str] = None, data: Union[bytes, str, None] = None):
        msg_type = ContentType.IMAGE if content is None else 0
        super().__init__(content=content, msg_type=msg_type, filename=filename, data=data)
        # lazy load
        self.__thumbnail = None

    @property  # Override
    def thumbnail(self) -> Optional[bytes]:
        if self.__thumbnail is None:
            base64 = self.get('thumbnail')
            if base64 is not None:
                self.__thumbnail = base64_decode(base64)
        return self.__thumbnail

    @thumbnail.setter  # Override
    def thumbnail(self, small_image: bytes):
        if small_image is None:
            self.pop('thumbnail', None)
        else:
            self['thumbnail'] = base64_encode(small_image)
        self.__thumbnail = small_image


class AudioFileContent(BaseFileContent, AudioContent):
    """
        Audio Message Content
        ~~~~~~~~~~~~~~~~~~~~~

        data format: {
            type : 0x14,
            sn   : 123,

            URL      : "http://", // upload to CDN
            data     : "...",     // if (!URL) base64_encode(audio)
            text     : "...",     // Automatic Speech Recognition
            filename : "..."
        }
    """

    def __init__(self, content: Optional[Dict[str, Any]] = None,
                 filename: Optional[str] = None, data: Union[bytes, str, None] = None):
        msg_type = ContentType.AUDIO if content is None else 0
        super().__init__(content=content, msg_type=msg_type, filename=filename, data=data)

    @property  # Override
    def text(self) -> Optional[bytes]:
        return self.get('text')

    @text.setter  # Override
    def text(self, string: str):
        if string is None:
            self.pop('text', None)
        else:
            self['text'] = string


class VideoFileContent(BaseFileContent, VideoContent):
    """
        Video Message Content
        ~~~~~~~~~~~~~~~~~~~~~

        data format: {
            type : 0x16,
            sn   : 123,

            URL      : "http://", // upload to CDN
            data     : "...",     // if (!URL) base64_encode(video)
            snapshot : "...",     // base64_encode(smallImage)
            filename : "..."
        }
    """

    def __init__(self, content: Optional[Dict[str, Any]] = None,
                 filename: Optional[str] = None, data: Union[bytes, str, None] = None):
        msg_type = ContentType.VIDEO if content is None else 0
        super().__init__(content=content, msg_type=msg_type, filename=filename, data=data)
        # lazy load
        self.__snapshot = None

    @property  # Override
    def snapshot(self) -> Optional[bytes]:
        if self.__snapshot is None:
            base64 = self.get('snapshot')
            if base64 is not None:
                self.__snapshot = base64_decode(base64)
        return self.__snapshot

    @snapshot.setter  # Override
    def snapshot(self, small_image: bytes):
        if small_image is None:
            self.pop('snapshot', None)
        else:
            self['snapshot'] = base64_encode(small_image)
        self.__snapshot = small_image
