"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFakeCertificate = exports.getTestVpc = exports.generateIntegStackName = exports.CreateScrapBucket = exports.fakeEcrRepoArn = void 0;
// Imports
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
const utils_1 = require("../lib/utils");
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
exports.fakeEcrRepoArn = 'arn:aws:ecr:us-east-1:123456789012:repository/fake-repo';
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: core_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    utils_1.addCfnSuppressRules(scriptBucket, [
        {
            id: "W51",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W35",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W41",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct",
        }
    ]);
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
// Helper Functions
function getTestVpc(stack, publicFacing = true) {
    return vpc_helper_1.buildVpc(stack, {
        defaultVpcProps: publicFacing ?
            vpc_defaults_1.DefaultPublicPrivateVpcProps() :
            vpc_defaults_1.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
    });
}
exports.getTestVpc = getTestVpc;
function getFakeCertificate(scope, id) {
    return acm.Certificate.fromCertificateArn(scope, id, "arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012");
}
exports.getFakeCertificate = getFakeCertificate;
//# sourceMappingURL=data:application/json;base64,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