"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    var _a;
    if (props === null || props === void 0 ? void 0 : props.existingVpc) {
        return props === null || props === void 0 ? void 0 : props.existingVpc;
    }
    let cumulativeProps = props === null || props === void 0 ? void 0 : props.defaultVpcProps;
    if (props === null || props === void 0 ? void 0 : props.userVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.userVpcProps);
    }
    if (props === null || props === void 0 ? void 0 : props.constructVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.constructVpcProps);
    }
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        utils_1.addCfnSuppressRules(cfnSubnet, [
            {
                id: 'W33',
                reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
            }
        ]);
    });
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = (_a = flowLog.logGroup) === null || _a === void 0 ? void 0 : _a.node.defaultChild;
    utils_1.addCfnSuppressRules(cfnLogGroup, [
        {
            id: 'W84',
            reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
        }
    ]);
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEP_FUNCTIONS"] = "STEP_FUNCTIONS";
    ServiceEndpointTypes["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypes["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypes["SSM"] = "SSM";
    ServiceEndpointTypes["ECR_API"] = "ECR_API";
    ServiceEndpointTypes["ECR_DKR"] = "ECR_DKR";
    ServiceEndpointTypes["EVENTS"] = "CLOUDWATCH_EVENTS";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.STEP_FUNCTIONS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.STEP_FUNCTIONS,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: ServiceEndpointTypes.SAGEMAKER_RUNTIME,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: ServiceEndpointTypes.SECRETS_MANAGER,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: ServiceEndpointTypes.SSM,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SSM,
    },
    {
        endpointName: ServiceEndpointTypes.ECR_API,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR
    },
    {
        endpointName: ServiceEndpointTypes.ECR_DKR,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
    },
    {
        endpointName: ServiceEndpointTypes.EVENTS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS
    }
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (!vpc.node.children.some((child) => child.node.id === interfaceTag)) {
        const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
        if (!service) {
            throw new Error("Unsupported Service sent to AddServiceEndpoint");
        }
        if (service.endpointType === EndpointTypes.GATEWAY) {
            vpc.addGatewayEndpoint(interfaceTag, {
                service: service.endpointGatewayService,
            });
        }
        if (service.endpointType === EndpointTypes.INTERFACE) {
            const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, `${scope.node.id}-${service.endpointName}`, {
                vpc,
                allowAllOutbound: true,
            }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
            vpc.addInterfaceEndpoint(interfaceTag, {
                service: service.endpointInterfaceService,
                securityGroups: [endpointDefaultSecurityGroup],
            });
        }
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
//# sourceMappingURL=data:application/json;base64,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