"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.consolidateProps = exports.addCfnSuppressRules = exports.generateResourceName = exports.printWarning = exports.overrideProps = void 0;
const deepmerge = require("deepmerge");
const override_warning_service_1 = require("./override-warning-service");
const log = require("npmlog");
const crypto = require("crypto");
const cdk = require("@aws-cdk/core");
function isObject(val) {
    return val != null && typeof val === 'object'
        && Object.prototype.toString.call(val) === '[object Object]';
}
function isPlainObject(o) {
    if (Array.isArray(o) === true) {
        return true;
    }
    if (isObject(o) === false) {
        return false;
    }
    // If has modified constructor
    const ctor = o.constructor;
    if (typeof ctor !== 'function') {
        return false;
    }
    // If has modified prototype
    const prot = ctor.prototype;
    if (isObject(prot) === false) {
        return false;
    }
    // If constructor does not have an Object-specific method
    if (prot.hasOwnProperty('isPrototypeOf') === false) {
        return false;
    }
    // Most likely a plain Object
    return true;
}
function combineMerge(target, source) {
    return target.concat(source);
}
function overwriteMerge(target, source) {
    target = source;
    return target;
}
function overrideProps(DefaultProps, userProps, concatArray = false) {
    // Notify the user via console output if defaults are overridden
    const overrideWarningsEnabled = (process.env.overrideWarningsEnabled !== 'false');
    if (overrideWarningsEnabled) {
        override_warning_service_1.flagOverriddenDefaults(DefaultProps, userProps);
    }
    // Override the sensible defaults with user provided props
    if (concatArray) {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: combineMerge,
            isMergeableObject: isPlainObject
        });
    }
    else {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: overwriteMerge,
            isMergeableObject: isPlainObject
        });
    }
}
exports.overrideProps = overrideProps;
function printWarning(message) {
    // Style the log output
    log.prefixStyle.bold = true;
    log.prefixStyle.fg = 'red';
    log.enableColor();
    log.warn('AWS_SOLUTIONS_CONSTRUCTS_WARNING: ', message);
}
exports.printWarning = printWarning;
/**
 * @summary Creates a resource name in the style of the CDK (string+hash)
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if neccessary) + a hash of the full concatenated parts
 *
 * This is based upon this discussion - https://github.com/aws/aws-cdk/issues/1424
 */
function generateResourceName(parts, maxLength) {
    const hashLength = 12;
    const maxPartLength = Math.floor((maxLength - hashLength) / parts.length);
    const sha256 = crypto.createHash("sha256");
    let finalName = '';
    parts.forEach((part) => {
        sha256.update(part);
        finalName += removeNonAlphanumeric(part.slice(0, maxPartLength));
    });
    const hash = sha256.digest("hex").slice(0, hashLength);
    finalName += hash;
    return finalName.toLowerCase();
}
exports.generateResourceName = generateResourceName;
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Adds CFN NAG suppress rules to the CDK resource.
 * @param resource The CDK resource
 * @param rules The CFN NAG suppress rules
 */
function addCfnSuppressRules(resource, rules) {
    var _a, _b, _c;
    if (resource instanceof cdk.Resource) {
        resource = resource.node.defaultChild;
    }
    if ((_b = (_a = resource.cfnOptions.metadata) === null || _a === void 0 ? void 0 : _a.cfn_nag) === null || _b === void 0 ? void 0 : _b.rules_to_suppress) {
        (_c = resource.cfnOptions.metadata) === null || _c === void 0 ? void 0 : _c.cfn_nag.rules_to_suppress.push(...rules);
    }
    else {
        resource.addMetadata('cfn_nag', {
            rules_to_suppress: rules
        });
    }
}
exports.addCfnSuppressRules = addCfnSuppressRules;
/**
 * Creates the props to be used to instantiate a CDK L2 construct within a Solutions Construct
 *
 * @param defaultProps The default props to be used by the construct
 * @param clientProps Optional properties passed in from the client in the props object
 * @param constructProps Optional properties required by the construct for the construct to work (override any other values)
 * @returns The properties to use - all values prioritized:
 *  1) constructProps value
 *  2) clientProps value
 *  3) defaultProps value
 */
function consolidateProps(defaultProps, clientProps, constructProps) {
    let result = defaultProps;
    if (clientProps) {
        result = overrideProps(result, clientProps);
    }
    if (constructProps) {
        result = overrideProps(result, constructProps);
    }
    return result;
}
exports.consolidateProps = consolidateProps;
//# sourceMappingURL=data:application/json;base64,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