"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSagemakerEndpoint = exports.createSagemakerEndpointConfig = exports.createSagemakerModel = exports.deploySagemakerEndpoint = exports.BuildSagemakerEndpoint = exports.buildSagemakerNotebook = void 0;
const sagemaker = require("@aws-cdk/aws-sagemaker");
const ec2 = require("@aws-cdk/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(_role, props) {
    var _a, _b;
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            'sagemaker:CreateTrainingJob',
            'sagemaker:DescribeTrainingJob',
            'sagemaker:CreateModel',
            'sagemaker:DescribeModel',
            'sagemaker:DeleteModel',
            'sagemaker:CreateEndpoint',
            'sagemaker:CreateEndpointConfig',
            'sagemaker:DescribeEndpoint',
            'sagemaker:DescribeEndpointConfig',
            'sagemaker:DeleteEndpoint',
            'sagemaker:DeleteEndpointConfig',
            'sagemaker:InvokeEndpoint',
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents',
        ],
    }));
    // To place the Sagemaker endpoint in a VPC
    if (props && props.vpc) {
        _role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:CreateNetworkInterfacePermission',
                'ec2:DeleteNetworkInterface',
                'ec2:DeleteNetworkInterfacePermission',
                'ec2:DescribeNetworkInterfaces',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
                'ec2:DescribeVpcs',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
            ],
        }));
    }
    // To create a Sagemaker model using Bring-Your-Own-Model (BYOM) algorith image
    // The image URL is specified in the modelProps
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:ecr:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:repository/*`],
        actions: [
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:DescribeRepositories',
            'ecr:DescribeImages',
            'ecr:BatchGetImage',
        ],
    }));
    // Add GetAuthorizationToken (it can not be bound to resources other than *)
    _role.addToPolicy(new iam.PolicyStatement({
        resources: ['*'],
        actions: ['ecr:GetAuthorizationToken'],
    }));
    // add permission to use Elastic Inference accelerator
    if (props && props.endpointConfigProps) {
        // Get the acceleratorType, if any
        const acceleratorType = ((_a = props.endpointConfigProps) === null || _a === void 0 ? void 0 : _a.productionVariants)[0].acceleratorType;
        if (acceleratorType !== undefined) {
            _role.addToPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['elastic-inference:Connect'],
            }));
        }
    }
    // add kms permissions
    _role.addToPolicy(new iam.PolicyStatement({
        // the kmsKeyId in the endpointConfigProps can be any of the following formats:
        // Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
        // Key ARN: arn:aws:kms:<region>:<accountID>:key/1234abcd-12ab-34cd-56ef-1234567890ab
        // Alias name: alias/ExampleAlias
        // Alias name ARN: arn:aws:kms:<region>:<accountID>:alias/ExampleAlias
        // the key is used to encrypt/decrypt data captured by the Sagemaker endpoint and stored in S3 bucket
        resources: [
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:key/*`,
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:alias/*`,
        ],
        actions: ['kms:Encrypt', 'kms:Decrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*', 'kms:DescribeKey'],
    }));
    // Add S3 permissions to get Model artifact, put data capture files, etc.
    _role.addToPolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject', 's3:ListBucket'],
        resources: ['arn:aws:s3:::*'],
    }));
    // Grant GetRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:GetRole'],
    }));
    // Grant PassRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:PassRole'],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
        },
    }));
    // Add CFN NAG uppress to allow for "Resource": "*" for ENI access in VPC,
    // ECR authorization token for custom model images, and elastic inference
    // Add CFN NAG for Complex Role because Sagmaker needs permissions to access several services
    const roleDefaultPolicy = (_b = _role.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
    utils_1.addCfnSuppressRules(roleDefaultPolicy, [
        {
            id: 'W12',
            reason: `Sagemaker needs the following minimum required permissions to access ENIs in a VPC, ECR for custom model images, and elastic inference.`,
        },
        {
            id: 'W76',
            reason: 'Complex role becuase Sagemaker needs permissions to access several services',
        }
    ]);
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined &&
                ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, 'SecurityGroup', {
                    vpc: vpcInstance,
                    allowAllOutbound: false,
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [
                    securityGroup.securityGroupId,
                ]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        if (props.sagemakerNotebookProps) {
            sagemakerNotebookProps = utils_1.overrideProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        }
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, 'SagemakerNotebook', sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
function BuildSagemakerEndpoint(scope, props) {
    /** Conditional Sagemaker endpoint creation */
    if (!props.existingSagemakerEndpointObj) {
        if (props.modelProps) {
            /** return [endpoint, endpointConfig, model] */
            return deploySagemakerEndpoint(scope, props);
        }
        else {
            throw Error('Either existingSagemakerEndpointObj or at least modelProps is required');
        }
    }
    else {
        /** Otherwise, return [endpoint] */
        return [props.existingSagemakerEndpointObj];
    }
}
exports.BuildSagemakerEndpoint = BuildSagemakerEndpoint;
function deploySagemakerEndpoint(scope, props) {
    let model;
    let endpointConfig;
    let endpoint;
    let sagemakerRole;
    // Create Sagemaker's model, endpointConfig, and endpoint
    if (props.modelProps) {
        // Check if the client has provided executionRoleArn
        if (props.modelProps.executionRoleArn) {
            sagemakerRole = iam.Role.fromRoleArn(scope, 'SagemakerRoleCustomer', props.modelProps.executionRoleArn);
        }
        else {
            // Create the Sagemaker Role
            sagemakerRole = new iam.Role(scope, 'SagemakerRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            });
            // Add required permissions
            addPermissions(sagemakerRole, props);
        }
        // Create Sagemaker Model
        model = createSagemakerModel(scope, props.modelProps, sagemakerRole, props.vpc);
        // Create Sagemaker EndpointConfig
        endpointConfig = createSagemakerEndpointConfig(scope, model.attrModelName, props.endpointConfigProps);
        // Add dependency on model
        endpointConfig.addDependsOn(model);
        // Create Sagemaker Endpoint
        endpoint = createSagemakerEndpoint(scope, endpointConfig.attrEndpointConfigName, props.endpointProps);
        // Add dependency on EndpointConfig
        endpoint.addDependsOn(endpointConfig);
        return [endpoint, endpointConfig, model];
    }
    else {
        throw Error('You need to provide at least modelProps to create Sagemaker Endpoint');
    }
}
exports.deploySagemakerEndpoint = deploySagemakerEndpoint;
function createSagemakerModel(scope, modelProps, role, vpc) {
    let finalModelProps;
    let primaryContainer;
    let vpcConfig;
    let model;
    if (vpc) {
        const modelDefaultSecurityGroup = new ec2.SecurityGroup(scope, 'ReplaceModelDefaultSecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        // Allow https traffic from within the VPC
        modelDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
        const cfnSecurityGroup = modelDefaultSecurityGroup.node.findChild('Resource');
        utils_1.addCfnSuppressRules(cfnSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
        // Throw an error if the VPC does not contain private or isolated subnets
        if (vpc.privateSubnets.length === 0 && vpc.isolatedSubnets.length === 0) {
            throw Error('VPC must contain private or isolated subnets to deploy the Sagemaker endpoint in a vpc');
        }
        vpcConfig = {
            // default SubnetType.PRIVATE (or ISOLATED or PUBLIC if there are no PRIVATE subnets)
            // So, private subnets will be used if provided by customer. Otherwise, use the default isolated subnets,
            subnets: vpc.selectSubnets({
                onePerAz: true,
            }).subnetIds,
            securityGroupIds: [modelDefaultSecurityGroup.securityGroupId],
        };
    }
    if (modelProps.primaryContainer) {
        // Get user provided Model's primary container
        primaryContainer = modelProps.primaryContainer;
        // Get default Model props
        finalModelProps = sagemaker_defaults_1.DefaultSagemakerModelProps(role.roleArn, primaryContainer, vpcConfig);
        // Override default model properties
        finalModelProps = utils_1.overrideProps(finalModelProps, modelProps);
        // Create the Sagemaker's Model
        model = new sagemaker.CfnModel(scope, 'SagemakerModel', finalModelProps);
        // Add dependency on the Sagemaker's role
        model.node.addDependency(role);
        return model;
    }
    else {
        throw Error('You need to provide at least primaryContainer to create Sagemaker Model');
    }
}
exports.createSagemakerModel = createSagemakerModel;
function createSagemakerEndpointConfig(scope, modelName, endpointConfigProps) {
    let finalEndpointConfigProps;
    let kmsKeyId;
    let endpointConfig;
    // Create encryption key if one is not provided
    if (endpointConfigProps && endpointConfigProps.kmsKeyId) {
        kmsKeyId = endpointConfigProps.kmsKeyId;
    }
    else {
        kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
    }
    // Get default EndpointConfig props
    finalEndpointConfigProps = sagemaker_defaults_1.DefaultSagemakerEndpointConfigProps(modelName, kmsKeyId);
    // Overwrite default EndpointConfig properties
    if (endpointConfigProps) {
        finalEndpointConfigProps = utils_1.overrideProps(finalEndpointConfigProps, endpointConfigProps);
    }
    // Create the Sagemaker's EndpointConfig
    endpointConfig = new sagemaker.CfnEndpointConfig(scope, 'SagemakerEndpointConfig', finalEndpointConfigProps);
    return endpointConfig;
}
exports.createSagemakerEndpointConfig = createSagemakerEndpointConfig;
function createSagemakerEndpoint(scope, endpointConfigName, endpointProps) {
    let finalEndpointProps;
    let endpoint;
    // Get default Endpoint props
    finalEndpointProps = sagemaker_defaults_1.DefaultSagemakerEndpointProps(endpointConfigName);
    // Overwrite default Endpoint properties
    if (endpointProps) {
        finalEndpointProps = utils_1.overrideProps(finalEndpointProps, endpointProps);
    }
    // Create the Sagemaker's Endpoint
    endpoint = new sagemaker.CfnEndpoint(scope, 'SagemakerEndpoint', finalEndpointProps);
    return endpoint;
}
exports.createSagemakerEndpoint = createSagemakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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