"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.buildS3Bucket = exports.createAlbLoggingBucket = exports.createLoggingBucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
function createLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: _reason
        }
    ]);
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function createAlbLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: "This is a log bucket for an Application Load Balancer"
        }
    ]);
    return loggingBucket;
}
exports.createAlbLoggingBucket = createAlbLoggingBucket;
function buildS3Bucket(scope, props, bucketId) {
    var _a, _b, _c;
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: aws_s3_1.StorageClass.GLACIER,
                    transitionAfter: core_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let customBucketProps;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    // If logging S3 access logs is enabled/undefined and an existing bucket object is not provided
    if (props.logS3AccessLogs !== false && !((_a = props.bucketProps) === null || _a === void 0 ? void 0 : _a.serverAccessLogsBucket)) {
        // Create the Logging Bucket
        let loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
        if (props.loggingBucketProps) {
            // User provided logging bucket props
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, props.loggingBucketProps);
        }
        else if ((_b = props.bucketProps) === null || _b === void 0 ? void 0 : _b.removalPolicy) {
            // If the client explicitly specified a removal policy for the main bucket,
            // then replicate that policy on the logging bucket
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, { removalPolicy: props.bucketProps.removalPolicy });
        }
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, loggingBucketProps);
    }
    // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
    if (((_c = props.bucketProps) === null || _c === void 0 ? void 0 : _c.versioned) === undefined || props.bucketProps.versioned) {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
    }
    else {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    customBucketProps = props.bucketProps ? utils_1.overrideProps(customBucketProps, props.bucketProps) : customBucketProps;
    const s3Bucket = new s3.Bucket(scope, _bucketId, customBucketProps);
    return [s3Bucket, loggingBucket];
}
exports.buildS3Bucket = buildS3Bucket;
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    utils_1.addCfnSuppressRules(fnResource, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    utils_1.addCfnSuppressRules(policyResource, [
        {
            id: 'W12',
            reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
        }
    ]);
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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