"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        });
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    utils_1.addCfnSuppressRules(lambdafunction, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        utils_1.addCfnSuppressRules(cfnLambdafunctionDefPolicy, [
            {
                id: 'W12',
                reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
            }
        ]);
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
//# sourceMappingURL=data:application/json;base64,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