"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            if (props.glueJobProps.glueVersion === '2.0' && props.glueJobProps.maxCapacity) {
                throw Error('Cannot set "MaxCapacity" with GlueVersion 2.0 or higher. Use "NumberOfWorkers" and "WorkerType". ' +
                    'Refer the API documentation https://docs.aws.amazon.com/glue/latest/webapi/API_Job.html for more details');
            }
            if (props.glueJobProps.maxCapacity && (props.glueJobProps.numberOfWorkers || props.glueJobProps.workerType)) {
                throw Error('Cannot set MaxCapacity and "WorkerType" or  "NumberOfWorkers". If using glueVersion 2.0 or beyond, ' +
                    'it is recommended to use "WorkerType" or  "NumberOfWorkers"');
            }
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${core_1.Aws.PARTITION}:kms:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps, _glueSecurityConfigName, _jobArgumentsList), glueJobProps);
    let _scriptLocation;
    if (isJobCommandProperty(_newGlueJobProps.command)) {
        if (_newGlueJobProps.command.scriptLocation) {
            _scriptLocation = _newGlueJobProps.command.scriptLocation;
        }
        else {
            throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
        }
    }
    const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
    _scriptBucketLocation.grantRead(_jobRole);
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole, _outputLocation];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    const splitString = s3Url.slice('s3://'.length);
    return `arn:${core_1.Aws.PARTITION}:s3:::${splitString}`;
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name ||
        command.pythonVersion ||
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2x1ZS1qb2ItaGVscGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZ2x1ZS1qb2ItaGVscGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7O0FBRUgsMENBQTBDO0FBQzFDLDhDQUFrRztBQUNsRyw0Q0FBK0Q7QUFDL0Qsd0NBQTREO0FBQzVELGdDQUFnQztBQUNoQyxtQ0FBd0M7QUFFeEM7OztHQUdHO0FBQ0gsSUFBWSxhQUVYO0FBRkQsV0FBWSxhQUFhO0lBQ3ZCLDBCQUFTLENBQUE7QUFDWCxDQUFDLEVBRlcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFFeEI7QUFzREQsU0FBZ0IsWUFBWSxDQUFDLEtBQWdCLEVBQUUsS0FBd0I7SUFDckUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUU7UUFDekIsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxXQUFXLEtBQUssS0FBSyxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO2dCQUM5RSxNQUFNLEtBQUssQ0FBQyxtR0FBbUc7b0JBQy9HLDBHQUEwRyxDQUFDLENBQUM7YUFDN0c7WUFFRCxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsV0FBVyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxlQUFlLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRTtnQkFDM0csTUFBTSxLQUFLLENBQUMscUdBQXFHO29CQUNqSCw2REFBNkQsQ0FBQyxDQUFDO2FBQ2hFO1lBRUQsT0FBTyxhQUFhLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxZQUFZLEVBQUUsS0FBSyxDQUFDLFFBQVMsRUFBRSxLQUFLLENBQUMsS0FBTSxFQUFFLEtBQUssQ0FBQyxlQUFnQixDQUFDLENBQUM7U0FDeEc7YUFBTTtZQUNMLE1BQU0sS0FBSyxDQUFDLG1EQUFtRCxDQUFDLENBQUM7U0FDbEU7S0FDRjtTQUFNO1FBQ0wsT0FBTyxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUUsY0FBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztLQUNuRztBQUNILENBQUM7QUFwQkQsb0NBb0JDO0FBRUQsU0FBZ0IsYUFBYSxDQUFDLEtBQWdCLEVBQUUsWUFBOEIsRUFBRSxRQUEwQixFQUFFLEtBQW9CLEVBQzlILGVBQW1DO0lBRW5DLElBQUksdUJBQStCLENBQUM7SUFFcEMsSUFBSSxZQUFZLENBQUMscUJBQXFCLEtBQUssU0FBUyxFQUFFO1FBQ3BELHVCQUF1QixHQUFHLHNCQUFzQixDQUFDO1FBQ2pELE1BQU0sV0FBVyxHQUFHLE9BQU8sVUFBRyxDQUFDLFNBQVMsUUFBUSxVQUFHLENBQUMsTUFBTSxJQUFJLFVBQUcsQ0FBQyxVQUFVLGlCQUFpQixDQUFDO1FBRTlGLElBQUksSUFBSSxDQUFDLHdCQUF3QixDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtZQUM3RCxJQUFJLEVBQUUsdUJBQXVCO1lBQzdCLHVCQUF1QixFQUFFO2dCQUN2QixzQkFBc0IsRUFBRTtvQkFDdEIsMEJBQTBCLEVBQUUsU0FBUztvQkFDckMsU0FBUyxFQUFFLFdBQVc7aUJBQ3ZCO2dCQUNELGFBQWEsRUFBRSxDQUFDO3dCQUNkLGdCQUFnQixFQUFFLFFBQVE7cUJBQzNCLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztLQUNKO1NBQU07UUFDTCx1QkFBdUIsR0FBRyxZQUFZLENBQUMscUJBQXFCLENBQUM7S0FDOUQ7SUFFRCxNQUFNLGNBQWMsR0FBRyxJQUFJLGdCQUFNLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtRQUNwRCxVQUFVLEVBQUU7WUFDVixJQUFJLHlCQUFlLENBQUM7Z0JBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7Z0JBQ3BCLE9BQU8sRUFBRSxDQUFFLHFCQUFxQixFQUFFLHNCQUFzQixFQUFFLG1CQUFtQixDQUFFO2dCQUMvRSxTQUFTLEVBQUUsQ0FBRSxPQUFPLFVBQUcsQ0FBQyxTQUFTLFNBQVMsVUFBRyxDQUFDLE1BQU0sSUFBSSxVQUFHLENBQUMsVUFBVSx3QkFBd0IsQ0FBRTthQUNqRyxDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7SUFFSCxJQUFJLFFBQWUsQ0FBQztJQUNwQixJQUFJLFlBQVksQ0FBQyxJQUFJLEVBQUU7UUFDckIsUUFBUSxHQUFHLGNBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDbEU7U0FBTTtRQUNMLFFBQVEsR0FBRyxRQUFRLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDOUM7SUFFRCxjQUFjLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBRXRDLElBQUksZUFBb0MsQ0FBQztJQUN6QyxJQUFJLGVBQWUsS0FBSyxTQUFTLElBQUksZUFBZSxDQUFDLGFBQWEsS0FBSyxhQUFhLENBQUMsRUFBRSxFQUFFO1FBQ3ZGLElBQUksZUFBZSxDQUFDLHNCQUFzQixLQUFLLFNBQVMsRUFBRTtZQUN4RCxlQUFlLEdBQUcsQ0FBRSxlQUFlLENBQUMsc0JBQXNCLEVBQUUsU0FBUyxDQUFFLENBQUM7U0FDekU7YUFBTTtZQUNMLGVBQWUsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxFQUFFLFdBQVcsRUFBRSxlQUFlLENBQUMsaUJBQWlCLEVBQUUsQ0FBRSxDQUFDO1NBQ3RHO0tBQ0Y7U0FBTTtRQUNMLGVBQWUsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUNyRDtJQUVELGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7SUFFNUMsTUFBTSxpQkFBaUIsR0FBRztRQUN4QixrQkFBa0IsRUFBRyxJQUFJO1FBQ3pCLG9DQUFvQyxFQUFHLElBQUk7UUFDM0MsaUJBQWlCLEVBQUUsUUFBUSxDQUFDLEdBQUc7UUFDL0IsY0FBYyxFQUFFLEtBQUssQ0FBQyxHQUFHO1FBQ3pCLEdBQUcsQ0FBQyxDQUFDLGVBQWUsS0FBSyxTQUFTLElBQUksQ0FBQyxlQUFlLElBQUksZUFBZSxDQUFDLGFBQWEsS0FBSyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDNUcsRUFBRSxlQUFlLEVBQUcsU0FBUyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxVQUFVLEVBQUUsQ0FBQztRQUN6RSxHQUFHLFlBQVksQ0FBQyxnQkFBZ0I7S0FDakMsQ0FBQztJQUVGLE1BQU0sZ0JBQWdCLEdBQXFCLHFCQUFhLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLFFBQVMsRUFBRSxZQUFZLEVBQzNHLHVCQUF1QixFQUFFLGlCQUFpQixDQUFDLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFFN0QsSUFBSSxlQUF1QixDQUFDO0lBQzVCLElBQUksb0JBQW9CLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLEVBQUU7UUFDbEQsSUFBSSxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFO1lBQzNDLGVBQWUsR0FBRyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO1NBQzNEO2FBQU07WUFDTCxNQUFNLEtBQUssQ0FBQywyRkFBMkYsQ0FBQyxDQUFDO1NBQzFHO0tBQ0Y7SUFFRCxNQUFNLHFCQUFxQixHQUFZLGVBQU0sQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLGlCQUFpQixDQUFDLGVBQWdCLENBQUMsQ0FBQyxDQUFDO0lBQzFILHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUUxQyxNQUFNLFFBQVEsR0FBZ0IsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUN4RixPQUFPLENBQUMsUUFBUSxFQUFFLFFBQVEsRUFBRSxlQUFlLENBQUMsQ0FBQztBQUMvQyxDQUFDO0FBcEZELHNDQW9GQztBQUVEOzs7OztHQUtHO0FBQ0gsU0FBZ0IsaUJBQWlCLENBQUMsS0FBZ0I7SUFDaEQsT0FBTyxJQUFJLGNBQUksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1FBQ2hDLFNBQVMsRUFBRSxJQUFJLDBCQUFnQixDQUFDLG9CQUFvQixDQUFDO1FBQ3JELFdBQVcsRUFBRSxrRUFBa0U7S0FDaEYsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUxELDhDQUtDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixlQUFlLENBQUMsS0FBZ0IsRUFBRSxRQUEwQixFQUFFLFVBQStCLEVBQzNHLFdBQTZDLEVBQUUsVUFBbUIsRUFBRSxVQUFnQjtJQUNwRixPQUFPLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDN0UsUUFBUSxDQUFDLHFCQUFxQixDQUFDLFFBQVEsRUFBRSxXQUFZLEVBQUUsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUM7QUFDcEYsQ0FBQztBQUpELDBDQUlDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsU0FBZ0Isa0JBQWtCLENBQUMsS0FBZ0IsRUFBRyxhQUFxQztJQUN6RixNQUFNLGNBQWMsR0FBMEIsQ0FBQyxhQUFhLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLHFCQUFhLENBQUMsUUFBUSxDQUFDLHdCQUF3QixFQUFFLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUMvSSxRQUFRLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztJQUN0QyxPQUFPLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxDQUFDLENBQUM7QUFDN0QsQ0FBQztBQUpELGdEQUlDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQVMsaUJBQWlCLENBQUMsS0FBYTtJQUN0QyxNQUFNLFdBQVcsR0FBVyxLQUFLLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUN4RCxPQUFPLE9BQU8sVUFBRyxDQUFDLFNBQVMsU0FBUyxXQUFXLEVBQUUsQ0FBQztBQUNwRCxDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQVMsb0JBQW9CLENBQUMsT0FBcUQ7SUFDakYsSUFBSyxPQUEwQyxDQUFDLElBQUk7UUFDakQsT0FBMEMsQ0FBQyxhQUFhO1FBQ3hELE9BQTBDLENBQUMsY0FBYyxFQUFFO1FBQzVELE9BQU8sSUFBSSxDQUFDO0tBQ2I7U0FBTTtRQUNMLFFBQVEsQ0FBQyxZQUFZLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUNyRSxPQUFPLEtBQUssQ0FBQztLQUNkO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIyIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgZ2x1ZSBmcm9tICdAYXdzLWNkay9hd3MtZ2x1ZSc7XG5pbXBvcnQgeyBFZmZlY3QsIElSb2xlLCBQb2xpY3ksIFBvbGljeVN0YXRlbWVudCwgUm9sZSwgU2VydmljZVByaW5jaXBhbCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQnVja2V0LCBCdWNrZXRQcm9wcywgSUJ1Y2tldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBBd3MsIENvbnN0cnVjdCwgSVJlc29sdmFibGUgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJy4uLyc7XG5pbXBvcnQgeyBvdmVycmlkZVByb3BzIH0gZnJvbSAnLi91dGlscyc7XG5cbi8qKlxuICogRW51bWVyYXRpb24gb2YgZGF0YSBzdG9yZSB0eXBlcyB0aGF0IGNvdWxkIGluY2x1ZGUgUzMsIER5bmFtb0RCLCBEb2N1bWVudERCLCBSRFMgb3IgUmVkc2hpZnQuIEN1cnJlbnRcbiAqIGNvbnN0cnVjdCBpbXBsZW1lbnRhdGlvbiBvbmx5IHN1cHBvcnRzIFMzLCBidXQgcG90ZW50aWFsIHRvIGFkZCBvdGhlciBvdXRwdXQgdHlwZXMgaW4gdGhlIGZ1dHVyZVxuICovXG5leHBvcnQgZW51bSBTaW5rU3RvcmVUeXBlIHtcbiAgUzMgPSAnUzMnXG59XG5cbi8qKlxuICogSW50ZXJmYWNlIHRvIGRlZmluZSBwb3RlbnRpYWwgb3V0cHV0cyB0byBhbGxvdyB0aGUgY29uc3RydWN0IGRlZmluZSBhZGRpdGlvbmFsIG91dHB1dCBkZXN0aW5hdGlvbnMgZm9yIEVUTFxuICogdHJhbnNmb3JtYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTaW5rRGF0YVN0b3JlUHJvcHMge1xuICAvKipcbiAgICogU2luayBkYXRhIHN0b3JlIHR5cGVcbiAgICovXG4gIHJlYWRvbmx5IGRhdGFzdG9yZVR5cGU6IFNpbmtTdG9yZVR5cGU7XG4gIC8qKlxuICAgKiBUaGUgb3V0cHV0IFMzIGxvY2F0aW9uIHdoZXJlIHRoZSBkYXRhIHNob3VsZCBiZSB3cml0dGVuLiBUaGUgcHJvdmlkZWQgUzMgYnVja2V0IHdpbGwgYmUgdXNlZCB0byBwYXNzXG4gICAqIHRoZSBvdXRwdXQgbG9jYXRpb24gdG8gdGhlIGV0bCBzY3JpcHQgYXMgYW4gYXJndW1lbnQgdG8gdGhlIEFXUyBHbHVlIGpvYi5cbiAgICpcbiAgICogSWYgbm8gbG9jYXRpb24gaXMgcHJvdmlkZWQsIGl0IHdpbGwgY2hlY2sgaWYgQG91dHB1dEJ1Y2tldFByb3BzIGFyZSBwcm92aWRlZC4gSWYgbm90IGl0IHdpbGwgY3JlYXRlIGEgbmV3XG4gICAqIGJ1Y2tldCBpZiB0aGUgQGRhdGFzdG9yZVR5cGUgaXMgUzMuXG4gICAqXG4gICAqIFRoZSBhcmd1bWVudCBrZXkgaXMgYG91dHB1dF9wYXRoYC4gVGhlIHZhbHVlIG9mIHRoZSBhcmd1bWVudCBjYW4gYmUgcmV0cmlldmUgaW4gdGhlIHB5dGhvbiBzY3JpcHRcbiAgICogYXMgZm9sbG93czpcbiAgICogIGdldFJlc29sdmVkT3B0aW9ucyhzeXMuYXJndiwgW1wiSk9CX05BTUVcIiwgXCJvdXRwdXRfcGF0aFwiLCA8b3RoZXIgYXJndW1lbnRzIHRoYXQgYXJlIHBhc3NlZD4gXSlcbiAgICogIG91dHB1dF9wYXRoID0gYXJnc1tcIm91dHB1dF9wYXRoXCJdXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ1MzT3V0cHV0QnVja2V0PzogQnVja2V0XG4gIC8qKlxuICAgKiBJZiBAZXhpc3RpbmdTM091dHB1dEJVY2tldCBpcyBwcm92aWRlZCwgdGhpcyBwYXJhbWV0ZXIgaXMgaWdub3JlZC4gSWYgdGhpcyBwYXJhbWV0ZXIgaXMgbm90IHByb3ZpZGVkLFxuICAgKiB0aGUgY29uc3RydWN0IHdpbGwgY3JlYXRlIGEgbmV3IGJ1Y2tldCBpZiB0aGUgQGRhdGFzdG9yZVR5cGUgaXMgUzMuXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRCdWNrZXRQcm9wcz86IEJ1Y2tldFByb3BzO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkR2x1ZUpvYlByb3BzIHtcbiAgLyoqXG4gICAqIEdsdWUgRVRMIGpvYiBwcm9wZXJ0aWVzLlxuICAgKi9cbiAgcmVhZG9ubHkgZ2x1ZUpvYlByb3BzPzogZ2x1ZS5DZm5Kb2JQcm9wcyB8IGFueVxuICAvKipcbiAgICogRXhpc3RpbmcgaW5zdGFuY2Ugb2YgdGhlIFMzIGJ1Y2tldCBvYmplY3QsIGlmIHRoaXMgaXMgc2V0IHRoZW4gdGhlIHNjcmlwdCBsb2NhdGlvbiBpcyBpZ25vcmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdDZm5Kb2I/OiBnbHVlLkNmbkpvYjtcbiAgLyoqXG4gICAqIEFXUyBHbHVlIHRhYmxlXG4gICAqL1xuICByZWFkb25seSB0YWJsZTogZ2x1ZS5DZm5UYWJsZTtcbiAgLyoqXG4gICAqIEFXUyBHbHVlIGRhdGFiYXNlXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZTogZ2x1ZS5DZm5EYXRhYmFzZTtcbiAgLyoqXG4gICAqIE91dHB1dCBzdG9yYWdlIG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IG91dHB1dERhdGFTdG9yZT86IFNpbmtEYXRhU3RvcmVQcm9wc1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRHbHVlSm9iKHNjb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBCdWlsZEdsdWVKb2JQcm9wcyk6IFtnbHVlLkNmbkpvYiwgSVJvbGUsIFtCdWNrZXQsIChCdWNrZXQgfCB1bmRlZmluZWQpP10/XSB7XG4gIGlmICghcHJvcHMuZXhpc3RpbmdDZm5Kb2IpIHtcbiAgICBpZiAocHJvcHMuZ2x1ZUpvYlByb3BzKSB7XG4gICAgICBpZiAocHJvcHMuZ2x1ZUpvYlByb3BzLmdsdWVWZXJzaW9uID09PSAnMi4wJyAmJiBwcm9wcy5nbHVlSm9iUHJvcHMubWF4Q2FwYWNpdHkpIHtcbiAgICAgICAgdGhyb3cgRXJyb3IoJ0Nhbm5vdCBzZXQgXCJNYXhDYXBhY2l0eVwiIHdpdGggR2x1ZVZlcnNpb24gMi4wIG9yIGhpZ2hlci4gVXNlIFwiTnVtYmVyT2ZXb3JrZXJzXCIgYW5kIFwiV29ya2VyVHlwZVwiLiAnICtcbiAgICAgICAgJ1JlZmVyIHRoZSBBUEkgZG9jdW1lbnRhdGlvbiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ2x1ZS9sYXRlc3Qvd2ViYXBpL0FQSV9Kb2IuaHRtbCBmb3IgbW9yZSBkZXRhaWxzJyk7XG4gICAgICB9XG5cbiAgICAgIGlmIChwcm9wcy5nbHVlSm9iUHJvcHMubWF4Q2FwYWNpdHkgJiYgKHByb3BzLmdsdWVKb2JQcm9wcy5udW1iZXJPZldvcmtlcnMgfHwgcHJvcHMuZ2x1ZUpvYlByb3BzLndvcmtlclR5cGUpKSB7XG4gICAgICAgIHRocm93IEVycm9yKCdDYW5ub3Qgc2V0IE1heENhcGFjaXR5IGFuZCBcIldvcmtlclR5cGVcIiBvciAgXCJOdW1iZXJPZldvcmtlcnNcIi4gSWYgdXNpbmcgZ2x1ZVZlcnNpb24gMi4wIG9yIGJleW9uZCwgJyArXG4gICAgICAgICdpdCBpcyByZWNvbW1lbmRlZCB0byB1c2UgXCJXb3JrZXJUeXBlXCIgb3IgIFwiTnVtYmVyT2ZXb3JrZXJzXCInKTtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIGRlcGxveUdsdWVKb2Ioc2NvcGUsIHByb3BzLmdsdWVKb2JQcm9wcywgcHJvcHMuZGF0YWJhc2UhLCBwcm9wcy50YWJsZSEsIHByb3BzLm91dHB1dERhdGFTdG9yZSEpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBFcnJvcignRWl0aGVyIGdsdWVKb2JQcm9wcyBvciBleGlzdGluZ0NmbkpvYiBpcyByZXF1aXJlZCcpO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gW3Byb3BzLmV4aXN0aW5nQ2ZuSm9iLCBSb2xlLmZyb21Sb2xlQXJuKHNjb3BlLCAnRXhpc3RpbmdSb2xlJywgcHJvcHMuZXhpc3RpbmdDZm5Kb2Iucm9sZSldO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBkZXBsb3lHbHVlSm9iKHNjb3BlOiBDb25zdHJ1Y3QsIGdsdWVKb2JQcm9wczogZ2x1ZS5DZm5Kb2JQcm9wcywgZGF0YWJhc2U6IGdsdWUuQ2ZuRGF0YWJhc2UsIHRhYmxlOiBnbHVlLkNmblRhYmxlLFxuICBvdXRwdXREYXRhU3RvcmU6IFNpbmtEYXRhU3RvcmVQcm9wcyk6IFtnbHVlLkNmbkpvYiwgSVJvbGUsIFtCdWNrZXQsIChCdWNrZXQgfCB1bmRlZmluZWQpP11dIHtcblxuICBsZXQgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWU6IHN0cmluZztcblxuICBpZiAoZ2x1ZUpvYlByb3BzLnNlY3VyaXR5Q29uZmlndXJhdGlvbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWUgPSAnRVRMSm9iU2VjdXJpdHlDb25maWcnO1xuICAgIGNvbnN0IF9nbHVlS01TS2V5ID0gYGFybjoke0F3cy5QQVJUSVRJT059Omttczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OmFsaWFzL2F3cy9nbHVlYDtcblxuICAgIG5ldyBnbHVlLkNmblNlY3VyaXR5Q29uZmlndXJhdGlvbihzY29wZSwgJ0dsdWVTZWN1cml0eUNvbmZpZycsIHtcbiAgICAgIG5hbWU6IF9nbHVlU2VjdXJpdHlDb25maWdOYW1lLFxuICAgICAgZW5jcnlwdGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgam9iQm9va21hcmtzRW5jcnlwdGlvbjoge1xuICAgICAgICAgIGpvYkJvb2ttYXJrc0VuY3J5cHRpb25Nb2RlOiAnQ1NFLUtNUycsXG4gICAgICAgICAga21zS2V5QXJuOiBfZ2x1ZUtNU0tleVxuICAgICAgICB9LFxuICAgICAgICBzM0VuY3J5cHRpb25zOiBbe1xuICAgICAgICAgIHMzRW5jcnlwdGlvbk1vZGU6ICdTU0UtUzMnXG4gICAgICAgIH1dXG4gICAgICB9XG4gICAgfSk7XG4gIH0gZWxzZSB7XG4gICAgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWUgPSBnbHVlSm9iUHJvcHMuc2VjdXJpdHlDb25maWd1cmF0aW9uO1xuICB9XG5cbiAgY29uc3QgX2dsdWVKb2JQb2xpY3kgPSBuZXcgUG9saWN5KHNjb3BlLCAnTG9nUG9saWN5Jywge1xuICAgIHN0YXRlbWVudHM6IFtcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWyAnbG9nczpDcmVhdGVMb2dHcm91cCcsICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsICdsb2dzOlB1dExvZ0V2ZW50cycgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbIGBhcm46JHtBd3MuUEFSVElUSU9OfTpsb2dzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06bG9nLWdyb3VwOi9hd3MtZ2x1ZS8qYCBdXG4gICAgICB9KVxuICAgIF1cbiAgfSk7XG5cbiAgbGV0IF9qb2JSb2xlOiBJUm9sZTtcbiAgaWYgKGdsdWVKb2JQcm9wcy5yb2xlKSB7XG4gICAgX2pvYlJvbGUgPSBSb2xlLmZyb21Sb2xlQXJuKHNjb3BlLCAnSm9iUm9sZScsIGdsdWVKb2JQcm9wcy5yb2xlKTtcbiAgfSBlbHNlIHtcbiAgICBfam9iUm9sZSA9IGRlZmF1bHRzLmNyZWF0ZUdsdWVKb2JSb2xlKHNjb3BlKTtcbiAgfVxuXG4gIF9nbHVlSm9iUG9saWN5LmF0dGFjaFRvUm9sZShfam9iUm9sZSk7XG5cbiAgbGV0IF9vdXRwdXRMb2NhdGlvbjogWyBCdWNrZXQsIEJ1Y2tldD8gXTtcbiAgaWYgKG91dHB1dERhdGFTdG9yZSAhPT0gdW5kZWZpbmVkICYmIG91dHB1dERhdGFTdG9yZS5kYXRhc3RvcmVUeXBlID09PSBTaW5rU3RvcmVUeXBlLlMzKSB7XG4gICAgaWYgKG91dHB1dERhdGFTdG9yZS5leGlzdGluZ1MzT3V0cHV0QnVja2V0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIF9vdXRwdXRMb2NhdGlvbiA9IFsgb3V0cHV0RGF0YVN0b3JlLmV4aXN0aW5nUzNPdXRwdXRCdWNrZXQsIHVuZGVmaW5lZCBdO1xuICAgIH0gZWxzZSB7XG4gICAgICBfb3V0cHV0TG9jYXRpb24gPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHNjb3BlLCB7IGJ1Y2tldFByb3BzOiBvdXRwdXREYXRhU3RvcmUub3V0cHV0QnVja2V0UHJvcHMgfSApO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICBfb3V0cHV0TG9jYXRpb24gPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHNjb3BlLCB7fSk7XG4gIH1cblxuICBfb3V0cHV0TG9jYXRpb25bMF0uZ3JhbnRSZWFkV3JpdGUoX2pvYlJvbGUpO1xuXG4gIGNvbnN0IF9qb2JBcmd1bWVudHNMaXN0ID0ge1xuICAgIFwiLS1lbmFibGUtbWV0cmljc1wiIDogdHJ1ZSxcbiAgICBcIi0tZW5hYmxlLWNvbnRpbnVvdXMtY2xvdWR3YXRjaC1sb2dcIiA6IHRydWUsXG4gICAgXCItLWRhdGFiYXNlX25hbWVcIjogZGF0YWJhc2UucmVmLFxuICAgIFwiLS10YWJsZV9uYW1lXCI6IHRhYmxlLnJlZixcbiAgICAuLi4oKG91dHB1dERhdGFTdG9yZSA9PT0gdW5kZWZpbmVkIHx8IChvdXRwdXREYXRhU3RvcmUgJiYgb3V0cHV0RGF0YVN0b3JlLmRhdGFzdG9yZVR5cGUgPT09IFNpbmtTdG9yZVR5cGUuUzMpKSAmJlxuICAgICAgeyAnLS1vdXRwdXRfcGF0aCcgOiBgczNhOi8vJHtfb3V0cHV0TG9jYXRpb25bMF0uYnVja2V0TmFtZX0vb3V0cHV0L2AgfSksXG4gICAgLi4uZ2x1ZUpvYlByb3BzLmRlZmF1bHRBcmd1bWVudHNcbiAgfTtcblxuICBjb25zdCBfbmV3R2x1ZUpvYlByb3BzOiBnbHVlLkNmbkpvYlByb3BzID0gb3ZlcnJpZGVQcm9wcyhkZWZhdWx0cy5EZWZhdWx0R2x1ZUpvYlByb3BzKF9qb2JSb2xlISwgZ2x1ZUpvYlByb3BzLFxuICAgIF9nbHVlU2VjdXJpdHlDb25maWdOYW1lLCBfam9iQXJndW1lbnRzTGlzdCksIGdsdWVKb2JQcm9wcyk7XG5cbiAgbGV0IF9zY3JpcHRMb2NhdGlvbjogc3RyaW5nO1xuICBpZiAoaXNKb2JDb21tYW5kUHJvcGVydHkoX25ld0dsdWVKb2JQcm9wcy5jb21tYW5kKSkge1xuICAgIGlmIChfbmV3R2x1ZUpvYlByb3BzLmNvbW1hbmQuc2NyaXB0TG9jYXRpb24pIHtcbiAgICAgIF9zY3JpcHRMb2NhdGlvbiA9IF9uZXdHbHVlSm9iUHJvcHMuY29tbWFuZC5zY3JpcHRMb2NhdGlvbjtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgRXJyb3IoJ1NjcmlwdCBsb2NhdGlvbiBoYXMgdG8gYmUgcHJvdmlkZWQgYXMgYW4gczMgVXJsIGxvY2F0aW9uLiBTY3JpcHQgbG9jYXRpb24gY2Fubm90IGJlIGVtcHR5Jyk7XG4gICAgfVxuICB9XG5cbiAgY29uc3QgX3NjcmlwdEJ1Y2tldExvY2F0aW9uOiBJQnVja2V0ID0gQnVja2V0LmZyb21CdWNrZXRBcm4oc2NvcGUsICdTY3JpcHRMb2NhaXRvbicsIGdldFMzQXJuZnJvbVMzVXJsKF9zY3JpcHRMb2NhdGlvbiEpKTtcbiAgX3NjcmlwdEJ1Y2tldExvY2F0aW9uLmdyYW50UmVhZChfam9iUm9sZSk7XG5cbiAgY29uc3QgX2dsdWVKb2I6IGdsdWUuQ2ZuSm9iID0gbmV3IGdsdWUuQ2ZuSm9iKHNjb3BlLCAnS2luZXNpc0VUTEpvYicsIF9uZXdHbHVlSm9iUHJvcHMpO1xuICByZXR1cm4gW19nbHVlSm9iLCBfam9iUm9sZSwgX291dHB1dExvY2F0aW9uXTtcbn1cblxuLyoqXG4gKiBUaGlzIGlzIGEgaGVscGVyIG1ldGhvZCB0byBjcmVhdGUgdGhlIFJvbGUgcmVxdWlyZWQgZm9yIHRoZSBHbHVlIEpvYi4gSWYgYSByb2xlIGlzIGFscmVhZHkgY3JlYXRlZCB0aGVuIHRoaXNcbiAqIG1ldGhvZCBpcyBub3QgcmVxdWlyZWQgdG8gYmUgY2FsbGVkLlxuICpcbiAqIEBwYXJhbSBzY29wZSAtIFRoZSBBV1MgQ29uc3RydWN0IHVuZGVyIHdoaWNoIHRoZSByb2xlIGlzIHRvIGJlIGNyZWF0ZWRcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUdsdWVKb2JSb2xlKHNjb3BlOiBDb25zdHJ1Y3QpOiBSb2xlIHtcbiAgcmV0dXJuIG5ldyBSb2xlKHNjb3BlLCAnSm9iUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdnbHVlLmFtYXpvbmF3cy5jb20nKSxcbiAgICBkZXNjcmlwdGlvbjogJ1NlcnZpY2Ugcm9sZSB0aGF0IEdsdWUgY3VzdG9tIEVUTCBqb2JzIHdpbGwgYXNzdW1lIGZvciBleGV1Y3Rpb24nLFxuICB9KTtcbn1cblxuLyoqXG4gKiBUaGlzIG1ldGhvZCBjcmVhdGVzIGFuIEFXUyBHbHVlIHRhYmxlLiBUaGUgbWV0aG9kIGlzIGNhbGxlZCB3aGVuIGFuIGV4aXN0aW5nIEdsdWUgdGFibGUgaXMgbm90IHByb3ZpZGVkXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVHbHVlVGFibGUoc2NvcGU6IENvbnN0cnVjdCwgZGF0YWJhc2U6IGdsdWUuQ2ZuRGF0YWJhc2UsIHRhYmxlUHJvcHM/OiBnbHVlLkNmblRhYmxlUHJvcHMsXG4gIGZpZWxkU2NoZW1hPzogZ2x1ZS5DZm5UYWJsZS5Db2x1bW5Qcm9wZXJ0eSBbXSwgc291cmNlVHlwZT86IHN0cmluZywgcGFyYW1ldGVycz86IGFueSk6IGdsdWUuQ2ZuVGFibGUge1xuICByZXR1cm4gZGVmYXVsdHMuRGVmYXVsdEdsdWVUYWJsZShzY29wZSwgdGFibGVQcm9wcyAhPT0gdW5kZWZpbmVkID8gdGFibGVQcm9wcyA6XG4gICAgZGVmYXVsdHMuRGVmYXVsdEdsdWVUYWJsZVByb3BzKGRhdGFiYXNlLCBmaWVsZFNjaGVtYSEsIHNvdXJjZVR5cGUsIHBhcmFtZXRlcnMpKTtcbn1cblxuLyoqXG4gKiBUaGlzIG1ldGhvZCBjcmVhdGVzIGFuIEFXUyBHbHVlIGRhdGFiYXNlLiBUaGUgbWV0aG9kIGlzIG9ubHkgY2FsbGVkIHdpdGggYW4gZXhpc3RpbmcgR2x1ZSBkYXRhYmFzZSB0eXBlIGlzIG5vdCBwcm92aWRlZC5cbiAqIFRoZSBtZXRob2QgdXNlcyB0aGUgdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsIHByb3BzIGZvciB0aGUgR2x1ZSBkYXRhYmFzZVxuICpcbiAqIEBwYXJhbSBzY29wZVxuICogQHBhcmFtIGRhdGFiYXNlUHJvcHNcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUdsdWVEYXRhYmFzZShzY29wZTogQ29uc3RydWN0LCAgZGF0YWJhc2VQcm9wcz86IGdsdWUuQ2ZuRGF0YWJhc2VQcm9wcyk6IGdsdWUuQ2ZuRGF0YWJhc2Uge1xuICBjb25zdCBfbWVyZ2VkREJQcm9wczogZ2x1ZS5DZm5EYXRhYmFzZVByb3BzID0gKGRhdGFiYXNlUHJvcHMgIT09IHVuZGVmaW5lZCkgPyBvdmVycmlkZVByb3BzKGRlZmF1bHRzLkRlZmF1bHRHbHVlRGF0YWJhc2VQcm9wcygpLCBkYXRhYmFzZVByb3BzKSA6XG4gICAgZGVmYXVsdHMuRGVmYXVsdEdsdWVEYXRhYmFzZVByb3BzKCk7XG4gIHJldHVybiBkZWZhdWx0cy5EZWZhdWx0R2x1ZURhdGFiYXNlKHNjb3BlLCBfbWVyZ2VkREJQcm9wcyk7XG59XG5cbi8qKlxuICogQSB1dGlsaXR5IG1ldGhvZCB0byBnZW5lcmF0ZSB0aGUgUzMgQXJuIGZyb20gYW4gUzMgVXJsLlxuICpcbiAqIEBwYXJhbSBzM1VybFxuICovXG5mdW5jdGlvbiBnZXRTM0FybmZyb21TM1VybChzM1VybDogc3RyaW5nKTogc3RyaW5nIHtcbiAgY29uc3Qgc3BsaXRTdHJpbmc6IHN0cmluZyA9IHMzVXJsLnNsaWNlKCdzMzovLycubGVuZ3RoKTtcbiAgcmV0dXJuIGBhcm46JHtBd3MuUEFSVElUSU9OfTpzMzo6OiR7c3BsaXRTdHJpbmd9YDtcbn1cblxuLyoqXG4gKiBBIHV0aWxpdHkgbWV0aG9kIHRvIHR5cGUgY2hlY2sgQ2ZuSm9iLkpvYkNvbW1hbmRQcm9wZXJ0eSB0eXBlLlxuICpcbiAqIEBwYXJhbSBjb21tYW5kXG4gKi9cbmZ1bmN0aW9uIGlzSm9iQ29tbWFuZFByb3BlcnR5KGNvbW1hbmQ6IGdsdWUuQ2ZuSm9iLkpvYkNvbW1hbmRQcm9wZXJ0eSB8IElSZXNvbHZhYmxlKTogY29tbWFuZCBpcyBnbHVlLkNmbkpvYi5Kb2JDb21tYW5kUHJvcGVydHkge1xuICBpZiAoKGNvbW1hbmQgYXMgZ2x1ZS5DZm5Kb2IuSm9iQ29tbWFuZFByb3BlcnR5KS5uYW1lIHx8XG4gICAgKGNvbW1hbmQgYXMgZ2x1ZS5DZm5Kb2IuSm9iQ29tbWFuZFByb3BlcnR5KS5weXRob25WZXJzaW9uIHx8XG4gICAgKGNvbW1hbmQgYXMgZ2x1ZS5DZm5Kb2IuSm9iQ29tbWFuZFByb3BlcnR5KS5zY3JpcHRMb2NhdGlvbikge1xuICAgIHJldHVybiB0cnVlO1xuICB9IGVsc2Uge1xuICAgIGRlZmF1bHRzLnByaW50V2FybmluZygnY29tbWFuZCBub3Qgb2YgdHlwZSBKb2JDb21tYW5kUHJvcGVydHkgdHlwZScpO1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxufSJdfQ==