"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cdk = require("@aws-cdk/core");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    utils_1.addCfnSuppressRules(cfDistribution, [
        {
            id: 'W70',
            reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
        }
    ]);
    return cfDistribution;
}
// Cloudfront function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultCloudfrontFunction(scope) {
    // generate a stable unique id for the cloudfront function and use it
    // both for the function name and the logical id of the function so if
    // it is changed the function will be recreated.
    // see https://github.com/aws/aws-cdk/issues/15523
    const functionId = `SetHttpSecurityHeaders${scope.node.addr}`;
    return new cloudfront.Function(scope, "SetHttpSecurityHeaders", {
        functionName: functionId,
        code: cloudfront.FunctionCode.fromInline("function handler(event) { var response = event.response; \
      var headers = response.headers; \
      headers['strict-transport-security'] = { value: 'max-age=63072000; includeSubdomains; preload'}; \
      headers['content-security-policy'] = { value: \"default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self'; object-src 'none'\"}; \
      headers['x-content-type-options'] = { value: 'nosniff'}; \
      headers['x-frame-options'] = {value: 'DENY'}; \
      headers['x-xss-protection'] = {value: '1; mode=block'}; \
      return response; \
    }")
    });
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, httpSecurityHeaders, cloudfrontFunction);
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders = true, originPath, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, httpSecurityHeaders, originPath, cloudfrontFunction);
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    // the lack of a bucketPolicy means the bucket was imported from outside the stack so the lack of cfn_nag suppression is not an issue
    if (bucketPolicy) {
        utils_1.addCfnSuppressRules(bucketPolicy, [
            {
                id: 'F16',
                reason: `Public website bucket policy requires a wildcard principal`
            }
        ]);
    }
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    let originRequestPolicy;
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, httpSecurityHeaders, cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.customHeaders, cloudfrontFunction);
    let cfprops;
    if (cloudFrontDistributionProps) {
        cfprops = utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false);
    }
    else {
        cfprops = defaultprops;
    }
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, loggingBucket, originRequestPolicy, cloudfrontFunction];
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
function getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps) {
    const isLoggingDisabled = (cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.enableLogging) === false;
    const userSuppliedLogBucket = cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.logBucket;
    if (userSuppliedLogBucket && cloudFrontLoggingBucketProps) {
        throw Error('Either cloudFrontDistributionProps.logBucket or cloudFrontLoggingBucketProps can be set.');
    }
    return isLoggingDisabled
        ? undefined
        : userSuppliedLogBucket !== null && userSuppliedLogBucket !== void 0 ? userSuppliedLogBucket : s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket', cloudFrontLoggingBucketProps ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), cloudFrontLoggingBucketProps) : s3_bucket_defaults_1.DefaultS3Props());
}
function getCloudfrontFunction(httpSecurityHeaders, scope) {
    return httpSecurityHeaders ? defaultCloudfrontFunction(scope) : undefined;
}
//# sourceMappingURL=data:application/json;base64,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