"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    utils_1.addCfnSuppressRules(deployment, [
        {
            id: 'W45',
            reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
        }
    ]);
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, { ...apiGatewayProps, cloudWatchRole: false });
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    let cwRole;
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    const plan = _api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        // Configure Usage Plan with API Key
        const key = _api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, { ...apiGatewayProps, cloudWatchRole: false });
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    let cwRole;
    // Configure API access logging
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    const plan = _api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        // Configure Usage Plan with API Key
        const key = _api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    if (params.awsIntegrationProps) {
        const overridenProps = utils_1.overrideProps(baseProps, params.awsIntegrationProps);
        apiGatewayIntegration = new api.AwsIntegration(overridenProps);
    }
    else {
        apiGatewayIntegration = new api.AwsIntegration(baseProps);
    }
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    if (params.methodOptions) {
        const overridenProps = utils_1.overrideProps(defaultMethodOptions, params.methodOptions);
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    }
    else {
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, defaultMethodOptions);
    }
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpZ2F0ZXdheS1oZWxwZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGlnYXRld2F5LWhlbHBlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUtILHFDQUFxQztBQUNyQywrQ0FBK0M7QUFDL0Msd0NBQXdDO0FBQ3hDLHFEQUFxRDtBQUNyRCwrRUFBOEQ7QUFDOUQsbUNBQTZEO0FBSzdEOzs7O0dBSUc7QUFDSCxTQUFTLDZCQUE2QixDQUFDLEtBQWdCLEVBQUUsSUFBaUI7O0lBQ3hFLHVEQUF1RDtJQUN2RCxNQUFNLHFCQUFxQixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsNkJBQTZCLEVBQUU7UUFDL0UsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDO1FBQy9ELGNBQWMsRUFBRTtZQUNkLGlDQUFpQyxFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztnQkFDeEQsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUNuQyxPQUFPLEVBQUU7NEJBQ1AscUJBQXFCOzRCQUNyQixzQkFBc0I7NEJBQ3RCLHdCQUF3Qjs0QkFDeEIseUJBQXlCOzRCQUN6QixtQkFBbUI7NEJBQ25CLG1CQUFtQjs0QkFDbkIsc0JBQXNCO3lCQUN2Qjt3QkFDRCxTQUFTLEVBQUUsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUyxTQUFTLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUM7cUJBQ3ZGLENBQUMsQ0FBQzthQUNKLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztJQUNELG9GQUFvRjtJQUN0RixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQW1CLENBQUM7SUFDakUsTUFBTSxVQUFVLEdBQW1CLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUU7UUFDbkYsaUJBQWlCLEVBQUUscUJBQXFCLENBQUMsT0FBTztLQUNqRCxDQUFDLENBQUM7SUFDSCxVQUFVLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBRWhDLG9DQUFvQztJQUNwQyxNQUFNLFVBQVUsR0FBc0IsTUFBQSxJQUFJLENBQUMsZ0JBQWdCLDBDQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFzQixDQUFDO0lBQzdHLDJCQUFtQixDQUFDLFVBQVUsRUFBRTtRQUM5QjtZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLDZJQUE2STtTQUN0SjtLQUNGLENBQUMsQ0FBQztJQUVILHFCQUFxQjtJQUNyQixPQUFPLHFCQUFxQixDQUFDO0FBQy9CLENBQUM7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsc0JBQXNCLENBQUMsS0FBZ0IsRUFBRSxzQkFBOEMsRUFDOUYsZUFBd0M7O0lBRXhDLDZHQUE2RztJQUM3RyxJQUFJLGVBQWUsYUFBZixlQUFlLHVCQUFmLGVBQWUsQ0FBRSxhQUFhLEVBQUU7UUFDbEMsTUFBTSxLQUFLLENBQUMsZ0hBQWdILENBQUMsQ0FBQztLQUMvSDtJQUVELHdCQUF3QjtJQUN4QixJQUFJLElBQWlCLENBQUM7SUFDdEIsSUFBSSxlQUFlLEVBQUU7UUFDbkIsd0VBQXdFO1FBQ3hFLE1BQU0sZ0JBQWdCLEdBQUcscUJBQWEsQ0FBQyxzQkFBc0IsRUFBRSxFQUFFLEdBQUcsZUFBZSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQzlHLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0tBQ3hFO1NBQU07UUFDTCxtRUFBbUU7UUFDbkUsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFLHNCQUFzQixDQUFDLENBQUM7S0FDOUU7SUFDRCwrQkFBK0I7SUFDL0IsSUFBSSxNQUFNLENBQUM7SUFFWCxJQUFJLENBQUEsZUFBZSxhQUFmLGVBQWUsdUJBQWYsZUFBZSxDQUFFLGNBQWMsTUFBSyxLQUFLLEVBQUU7UUFDN0MsTUFBTSxHQUFHLDZCQUE2QixDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztLQUNyRDtJQUVELHVCQUF1QjtJQUN2QixNQUFNLGNBQWMsR0FBdUI7UUFDekMsU0FBUyxFQUFFLENBQUM7Z0JBQ1YsR0FBRyxFQUFFLElBQUk7Z0JBQ1QsS0FBSyxFQUFFLElBQUksQ0FBQyxlQUFlO2FBQzVCLENBQUM7S0FDSCxDQUFDO0lBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsY0FBYyxDQUFDLENBQUM7SUFFNUQsb0ZBQW9GO0lBQ3BGLElBQUksT0FBQSxlQUFlLGFBQWYsZUFBZSx1QkFBZixlQUFlLENBQUUsb0JBQW9CLDBDQUFFLGNBQWMsTUFBSyxJQUFJLEVBQUU7UUFDbEUsb0NBQW9DO1FBQ3BDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUNyQjtJQUVELDZCQUE2QjtJQUM3QixPQUFPLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0FBQ3hCLENBQUM7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsZ0JBQWdCLENBQUMsS0FBZ0IsRUFBRSxzQkFBd0MsRUFDbEYsZUFBa0M7O0lBRWxDLDZHQUE2RztJQUM3RyxJQUFJLGVBQWUsYUFBZixlQUFlLHVCQUFmLGVBQWUsQ0FBRSxhQUFhLEVBQUU7UUFDbEMsTUFBTSxLQUFLLENBQUMsZ0hBQWdILENBQUMsQ0FBQztLQUMvSDtJQUVELGlCQUFpQjtJQUNqQixJQUFJLElBQWlCLENBQUM7SUFDdEIsSUFBSSxlQUFlLEVBQUU7UUFDbkIsd0VBQXdFO1FBQ3hFLE1BQU0sZ0JBQWdCLEdBQUcscUJBQWEsQ0FBQyxzQkFBc0IsRUFBRSxFQUFFLEdBQUcsZUFBZSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQzlHLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0tBQzVEO1NBQU07UUFDTCxtRUFBbUU7UUFDbkUsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7S0FDbEU7SUFFRCxJQUFJLE1BQU0sQ0FBQztJQUVYLCtCQUErQjtJQUMvQixJQUFJLENBQUEsZUFBZSxhQUFmLGVBQWUsdUJBQWYsZUFBZSxDQUFFLGNBQWMsTUFBSyxLQUFLLEVBQUU7UUFDN0MsTUFBTSxHQUFHLDZCQUE2QixDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztLQUNyRDtJQUVELHVCQUF1QjtJQUN2QixNQUFNLGNBQWMsR0FBdUI7UUFDekMsU0FBUyxFQUFFLENBQUM7Z0JBQ1YsR0FBRyxFQUFFLElBQUk7Z0JBQ1QsS0FBSyxFQUFFLElBQUksQ0FBQyxlQUFlO2FBQzVCLENBQUM7S0FDSCxDQUFDO0lBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsY0FBYyxDQUFDLENBQUM7SUFFNUQsb0ZBQW9GO0lBQ3BGLElBQUksT0FBQSxlQUFlLGFBQWYsZUFBZSx1QkFBZixlQUFlLENBQUUsb0JBQW9CLDBDQUFFLGNBQWMsTUFBSyxJQUFJLEVBQUU7UUFDbEUsb0NBQW9DO1FBQ3BDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUNyQjtJQUVELDZCQUE2QjtJQUM3QixPQUFPLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0FBQ3hCLENBQUM7QUFFRDs7Ozs7R0FLRztBQUNILFNBQWdCLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsa0JBQW1DLEVBQ3ZGLGVBQXdDLEVBQUUsYUFBa0M7SUFDNUUsb0RBQW9EO0lBQ3BELE1BQU0sUUFBUSxHQUFHLDJDQUFhLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBRTFFLE1BQU0sWUFBWSxHQUFHLFdBQVcsQ0FBQywrQkFBK0IsQ0FBQyxrQkFBa0IsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUMvRixNQUFNLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxHQUFHLHNCQUFzQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDMUYsT0FBTyxDQUFDLE9BQU8sRUFBRSxTQUFTLEVBQUUsUUFBUSxDQUFFLENBQUM7QUFDekMsQ0FBQztBQVJELGtEQVFDO0FBRUQ7Ozs7O0dBS0c7QUFDSCxTQUFnQixxQkFBcUIsQ0FBQyxLQUFnQixFQUFFLGtCQUFtQyxFQUN6RixlQUF3QyxFQUFFLGFBQWtDO0lBQzVFLG9EQUFvRDtJQUNwRCxNQUFNLFFBQVEsR0FBRywyQ0FBYSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUUxRSxNQUFNLFlBQVksR0FBRyxXQUFXLENBQUMsaUNBQWlDLENBQUMsa0JBQWtCLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDakcsTUFBTSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsR0FBRyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBQzFGLE9BQU8sQ0FBQyxPQUFPLEVBQUUsU0FBUyxFQUFFLFFBQVEsQ0FBRSxDQUFDO0FBQ3pDLENBQUM7QUFSRCxzREFRQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixhQUFhLENBQUMsS0FBZ0IsRUFBRSxlQUFrQyxFQUNoRixhQUFrQztJQUNsQyxvREFBb0Q7SUFDcEQsTUFBTSxRQUFRLEdBQUcsMkNBQWEsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUUsYUFBYSxDQUFDLENBQUM7SUFFMUUsTUFBTSxZQUFZLEdBQUcsV0FBVyxDQUFDLHlCQUF5QixDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3JFLE1BQU0sQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLEdBQUcsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxlQUFlLENBQUMsQ0FBQztJQUNwRixPQUFPLENBQUMsT0FBTyxFQUFFLFNBQVMsRUFBRSxRQUFRLENBQUUsQ0FBQztBQUN6QyxDQUFDO0FBUkQsc0NBUUM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBZ0IsZUFBZSxDQUFDLEtBQWdCLEVBQUUsZUFBa0MsRUFDbEYsYUFBa0M7SUFDbEMsb0RBQW9EO0lBQ3BELE1BQU0sUUFBUSxHQUFHLDJDQUFhLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBRTFFLE1BQU0sWUFBWSxHQUFHLFdBQVcsQ0FBQywyQkFBMkIsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUN2RSxNQUFNLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxHQUFHLGdCQUFnQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDcEYsT0FBTyxDQUFDLE9BQU8sRUFBRSxTQUFTLEVBQUUsUUFBUSxDQUFFLENBQUM7QUFDekMsQ0FBQztBQVJELDBDQVFDO0FBaUJELFNBQWdCLDJCQUEyQixDQUFDLE1BQThDO0lBQ3hGLElBQUksU0FBUyxHQUE0QjtRQUN2QyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU87UUFDdkIscUJBQXFCLEVBQUUsTUFBTTtRQUM3QixPQUFPLEVBQUU7WUFDUCxtQkFBbUIsRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMsS0FBSztZQUNsRCxlQUFlLEVBQUUsTUFBTSxDQUFDLGNBQWM7WUFDdEMsaUJBQWlCLEVBQUU7Z0JBQ2pCLHlDQUF5QyxFQUFFLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLG9CQUFvQjthQUMxRztZQUNELGdCQUFnQixFQUFFO2dCQUNoQixrQkFBa0IsRUFBRSxNQUFNLENBQUMsZUFBZTthQUMzQztZQUNELG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxVQUFVLEVBQUUsS0FBSztpQkFDbEI7Z0JBQ0Q7b0JBQ0UsVUFBVSxFQUFFLEtBQUs7b0JBQ2pCLGlCQUFpQixFQUFFO3dCQUNqQixXQUFXLEVBQUUsT0FBTztxQkFDckI7b0JBQ0QsZ0JBQWdCLEVBQUUsS0FBSztpQkFDeEI7YUFDRjtTQUNGO0tBQ0YsQ0FBQztJQUVGLElBQUksVUFBVSxDQUFDO0lBRWYsSUFBSSxNQUFNLENBQUMsTUFBTSxFQUFFO1FBQ2pCLFVBQVUsR0FBRztZQUNYLE1BQU0sRUFBRSxNQUFNLENBQUMsTUFBTTtTQUN0QixDQUFDO0tBQ0g7U0FBTSxJQUFJLE1BQU0sQ0FBQyxJQUFJLEVBQUU7UUFDdEIsVUFBVSxHQUFHO1lBQ1gsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJO1NBQ2xCLENBQUM7S0FDSDtTQUFNO1FBQ0wsTUFBTSxLQUFLLENBQUMsbUNBQW1DLENBQUMsQ0FBQztLQUNsRDtJQUVELHdDQUF3QztJQUN4QyxTQUFTLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDakQsSUFBSSxxQkFBcUIsQ0FBQztJQUMxQixJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsRUFBRTtRQUM5QixNQUFNLGNBQWMsR0FBRyxxQkFBYSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM1RSxxQkFBcUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLENBQUM7S0FDaEU7U0FBTTtRQUNMLHFCQUFxQixHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztLQUMzRDtJQUVELE1BQU0sb0JBQW9CLEdBQUc7UUFDM0IsZUFBZSxFQUFFO1lBQ2Y7Z0JBQ0UsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLGtCQUFrQixFQUFFO29CQUNsQixxQ0FBcUMsRUFBRSxJQUFJO2lCQUM1QzthQUNGO1lBQ0Q7Z0JBQ0UsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLGtCQUFrQixFQUFFO29CQUNsQixxQ0FBcUMsRUFBRSxJQUFJO2lCQUM1QzthQUNGO1NBQ0Y7UUFDRCxnQkFBZ0IsRUFBRSxNQUFNLENBQUMsZ0JBQWdCO1FBQ3pDLGFBQWEsRUFBRSxNQUFNLENBQUMsWUFBWTtLQUNuQyxDQUFDO0lBRUYsSUFBSSxTQUFTLENBQUM7SUFDZCwrQkFBK0I7SUFDL0IsSUFBSSxNQUFNLENBQUMsYUFBYSxFQUFFO1FBQ3hCLE1BQU0sY0FBYyxHQUFJLHFCQUFhLENBQUMsb0JBQW9CLEVBQUUsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ2xGLFNBQVMsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLHFCQUFxQixFQUFFLGNBQWMsQ0FBQyxDQUFDO0tBQ25HO1NBQU07UUFDTCxTQUFTLEdBQUcsTUFBTSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO0tBQ3pHO0lBQ0QsT0FBTyxTQUFTLENBQUM7QUFDbkIsQ0FBQztBQWhGRCxrRUFnRkMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMiBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgYXBpIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGFwaURlZmF1bHRzIGZyb20gJy4vYXBpZ2F0ZXdheS1kZWZhdWx0cyc7XG5pbXBvcnQgeyBidWlsZExvZ0dyb3VwIH0gZnJvbSAnLi9jbG91ZHdhdGNoLWxvZy1ncm91cC1oZWxwZXInO1xuaW1wb3J0IHsgb3ZlcnJpZGVQcm9wcywgYWRkQ2ZuU3VwcHJlc3NSdWxlcyB9IGZyb20gJy4vdXRpbHMnO1xuaW1wb3J0IHsgSVJvbGUgfSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbi8vIE5vdGU6IFRvIGVuc3VyZSBDREt2MiBjb21wYXRpYmlsaXR5LCBrZWVwIHRoZSBpbXBvcnQgc3RhdGVtZW50IGZvciBDb25zdHJ1Y3Qgc2VwYXJhdGVcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4vKipcbiAqIENyZWF0ZSBhbmQgY29uZmlndXJlcyBhY2Nlc3MgbG9nZ2luZyBmb3IgQVBJIEdhdGV3YXkgcmVzb3VyY2VzLlxuICogQHBhcmFtIHNjb3BlIC0gdGhlIGNvbnN0cnVjdCB0byB3aGljaCB0aGUgYWNjZXNzIGxvZ2dpbmcgY2FwYWJpbGl0aWVzIHNob3VsZCBiZSBhdHRhY2hlZCB0by5cbiAqIEBwYXJhbSBfYXBpIC0gYW4gZXhpc3RpbmcgYXBpLlJlc3RBcGkgb3IgYXBpLkxhbWJkYVJlc3RBcGkuXG4gKi9cbmZ1bmN0aW9uIGNvbmZpZ3VyZUNsb3Vkd2F0Y2hSb2xlRm9yQXBpKHNjb3BlOiBDb25zdHJ1Y3QsIF9hcGk6IGFwaS5SZXN0QXBpKTogaWFtLlJvbGUge1xuICAvLyBTZXR1cCB0aGUgSUFNIFJvbGUgZm9yIEFQSSBHYXRld2F5IENsb3VkV2F0Y2ggYWNjZXNzXG4gIGNvbnN0IHJlc3RBcGlDbG91ZHdhdGNoUm9sZSA9IG5ldyBpYW0uUm9sZShzY29wZSwgJ0xhbWJkYVJlc3RBcGlDbG91ZFdhdGNoUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnYXBpZ2F0ZXdheS5hbWF6b25hd3MuY29tJyksXG4gICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgIExhbWJkYVJlc3RBcGlDbG91ZFdhdGNoUm9sZVBvbGljeTogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgIHN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nR3JvdXAnLFxuICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICdsb2dzOkRlc2NyaWJlTG9nR3JvdXBzJyxcbiAgICAgICAgICAgICdsb2dzOkRlc2NyaWJlTG9nU3RyZWFtcycsXG4gICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgJ2xvZ3M6R2V0TG9nRXZlbnRzJyxcbiAgICAgICAgICAgICdsb2dzOkZpbHRlckxvZ0V2ZW50cydcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogW2Bhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06bG9nczoke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06KmBdXG4gICAgICAgIH0pXVxuICAgICAgfSlcbiAgICB9XG4gIH0pO1xuICAgIC8vIENyZWF0ZSBhbmQgY29uZmlndXJlIEFXUzo6QXBpR2F0ZXdheTo6QWNjb3VudCB3aXRoIENsb3VkV2F0Y2ggUm9sZSBmb3IgQXBpR2F0ZXdheVxuICBjb25zdCBDZm5BcGkgPSBfYXBpLm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGFwaS5DZm5SZXN0QXBpO1xuICBjb25zdCBjZm5BY2NvdW50OiBhcGkuQ2ZuQWNjb3VudCA9IG5ldyBhcGkuQ2ZuQWNjb3VudChzY29wZSwgJ0xhbWJkYVJlc3RBcGlBY2NvdW50Jywge1xuICAgIGNsb3VkV2F0Y2hSb2xlQXJuOiByZXN0QXBpQ2xvdWR3YXRjaFJvbGUucm9sZUFyblxuICB9KTtcbiAgY2ZuQWNjb3VudC5hZGREZXBlbmRzT24oQ2ZuQXBpKTtcblxuICAvLyBTdXBwcmVzcyBDZm4gTmFnIHdhcm5pbmcgZm9yIEFQSUdcbiAgY29uc3QgZGVwbG95bWVudDogYXBpLkNmbkRlcGxveW1lbnQgPSBfYXBpLmxhdGVzdERlcGxveW1lbnQ/Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGFwaS5DZm5EZXBsb3ltZW50O1xuICBhZGRDZm5TdXBwcmVzc1J1bGVzKGRlcGxveW1lbnQsIFtcbiAgICB7XG4gICAgICBpZDogJ1c0NScsXG4gICAgICByZWFzb246IGBBcGlHYXRld2F5IGhhcyBBY2Nlc3NMb2dnaW5nIGVuYWJsZWQgaW4gQVdTOjpBcGlHYXRld2F5OjpTdGFnZSByZXNvdXJjZSwgYnV0IGNmbl9uYWcgY2hlY2tlcyBmb3IgaXQgaW4gQVdTOjpBcGlHYXRld2F5OjpEZXBsb3ltZW50IHJlc291cmNlYFxuICAgIH1cbiAgXSk7XG5cbiAgLy8gUmV0dXJuIHRoZSBDVyBSb2xlXG4gIHJldHVybiByZXN0QXBpQ2xvdWR3YXRjaFJvbGU7XG59XG5cbi8qKlxuICogQ3JlYXRlcyBhbmQgY29uZmlndXJlcyBhbiBhcGkuTGFtYmRhUmVzdEFwaS5cbiAqIEBwYXJhbSBzY29wZSAtIHRoZSBjb25zdHJ1Y3QgdG8gd2hpY2ggdGhlIExhbWJkYVJlc3RBcGkgc2hvdWxkIGJlIGF0dGFjaGVkIHRvLlxuICogQHBhcmFtIGRlZmF1bHRBcGlHYXRld2F5UHJvcHMgLSB0aGUgZGVmYXVsdCBwcm9wZXJ0aWVzIGZvciB0aGUgTGFtYmRhUmVzdEFwaS5cbiAqIEBwYXJhbSBhcGlHYXRld2F5UHJvcHMgLSAob3B0aW9uYWwpIHVzZXItc3BlY2lmaWVkIHByb3BlcnRpZXMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcGVydGllcy5cbiAqL1xuZnVuY3Rpb24gY29uZmlndXJlTGFtYmRhUmVzdEFwaShzY29wZTogQ29uc3RydWN0LCBkZWZhdWx0QXBpR2F0ZXdheVByb3BzOiBhcGkuTGFtYmRhUmVzdEFwaVByb3BzLFxuICBhcGlHYXRld2F5UHJvcHM/OiBhcGkuTGFtYmRhUmVzdEFwaVByb3BzKTogW2FwaS5SZXN0QXBpLCBpYW0uUm9sZSB8IHVuZGVmaW5lZF0ge1xuXG4gIC8vIEFQSSBHYXRld2F5IGRvZXNuJ3QgYWxsb3cgYm90aCBlbmRwb2ludFR5cGVzIGFuZCBlbmRwb2ludENvbmZpZ3VyYXRpb24sIGNoZWNrIHdoZXRoZXIgZW5kUG9pbnRUeXBlcyBleGlzdHNcbiAgaWYgKGFwaUdhdGV3YXlQcm9wcz8uZW5kcG9pbnRUeXBlcykge1xuICAgIHRocm93IEVycm9yKCdTb2x1dGlvbnMgQ29uc3RydWN0cyBpbnRlcm5hbGx5IHVzZXMgZW5kcG9pbnRDb25maWd1cmF0aW9uLCB1c2UgZW5kcG9pbnRDb25maWd1cmF0aW9uIGluc3RlYWQgb2YgZW5kcG9pbnRUeXBlcycpO1xuICB9XG5cbiAgLy8gRGVmaW5lIHRoZSBBUEkgb2JqZWN0XG4gIGxldCBfYXBpOiBhcGkuUmVzdEFwaTtcbiAgaWYgKGFwaUdhdGV3YXlQcm9wcykge1xuICAgIC8vIElmIHByb3BlcnR5IG92ZXJyaWRlcyBoYXZlIGJlZW4gcHJvdmlkZWQsIGluY29ycG9yYXRlIHRoZW0gYW5kIGRlcGxveVxuICAgIGNvbnN0IF9hcGlHYXRld2F5UHJvcHMgPSBvdmVycmlkZVByb3BzKGRlZmF1bHRBcGlHYXRld2F5UHJvcHMsIHsgLi4uYXBpR2F0ZXdheVByb3BzLCBjbG91ZFdhdGNoUm9sZTogZmFsc2UgfSk7XG4gICAgX2FwaSA9IG5ldyBhcGkuTGFtYmRhUmVzdEFwaShzY29wZSwgJ0xhbWJkYVJlc3RBcGknLCBfYXBpR2F0ZXdheVByb3BzKTtcbiAgfSBlbHNlIHtcbiAgICAvLyBJZiBubyBwcm9wZXJ0eSBvdmVycmlkZXMsIGRlcGxveSB1c2luZyB0aGUgZGVmYXVsdCBjb25maWd1cmF0aW9uXG4gICAgX2FwaSA9IG5ldyBhcGkuTGFtYmRhUmVzdEFwaShzY29wZSwgJ0xhbWJkYVJlc3RBcGknLCBkZWZhdWx0QXBpR2F0ZXdheVByb3BzKTtcbiAgfVxuICAvLyBDb25maWd1cmUgQVBJIGFjY2VzcyBsb2dnaW5nXG4gIGxldCBjd1JvbGU7XG5cbiAgaWYgKGFwaUdhdGV3YXlQcm9wcz8uY2xvdWRXYXRjaFJvbGUgIT09IGZhbHNlKSB7XG4gICAgY3dSb2xlID0gY29uZmlndXJlQ2xvdWR3YXRjaFJvbGVGb3JBcGkoc2NvcGUsIF9hcGkpO1xuICB9XG5cbiAgLy8gQ29uZmlndXJlIFVzYWdlIFBsYW5cbiAgY29uc3QgdXNhZ2VQbGFuUHJvcHM6IGFwaS5Vc2FnZVBsYW5Qcm9wcyA9IHtcbiAgICBhcGlTdGFnZXM6IFt7XG4gICAgICBhcGk6IF9hcGksXG4gICAgICBzdGFnZTogX2FwaS5kZXBsb3ltZW50U3RhZ2VcbiAgICB9XVxuICB9O1xuXG4gIGNvbnN0IHBsYW4gPSBfYXBpLmFkZFVzYWdlUGxhbignVXNhZ2VQbGFuJywgdXNhZ2VQbGFuUHJvcHMpO1xuXG4gIC8vIElmIHJlcXVpcmVBcGlLZXkgcGFyYW0gaXMgc2V0IHRvIHRydWUsIGNyZWF0ZSBhIGFwaSBrZXkgJiBhc3NvY2lhdGUgdG8gVXNhZ2UgUGxhblxuICBpZiAoYXBpR2F0ZXdheVByb3BzPy5kZWZhdWx0TWV0aG9kT3B0aW9ucz8uYXBpS2V5UmVxdWlyZWQgPT09IHRydWUpIHtcbiAgICAvLyBDb25maWd1cmUgVXNhZ2UgUGxhbiB3aXRoIEFQSSBLZXlcbiAgICBjb25zdCBrZXkgPSBfYXBpLmFkZEFwaUtleSgnQXBpS2V5Jyk7XG4gICAgcGxhbi5hZGRBcGlLZXkoa2V5KTtcbiAgfVxuXG4gIC8vIFJldHVybiB0aGUgQVBJIGFuZCBDVyBSb2xlXG4gIHJldHVybiBbX2FwaSwgY3dSb2xlXTtcbn1cblxuLyoqXG4gKiBDcmVhdGVzIGFuZCBjb25maWd1cmVzIGFuIGFwaS5SZXN0QXBpLlxuICogQHBhcmFtIHNjb3BlIC0gdGhlIGNvbnN0cnVjdCB0byB3aGljaCB0aGUgUmVzdEFwaSBzaG91bGQgYmUgYXR0YWNoZWQgdG8uXG4gKiBAcGFyYW0gZGVmYXVsdEFwaUdhdGV3YXlQcm9wcyAtIHRoZSBkZWZhdWx0IHByb3BlcnRpZXMgZm9yIHRoZSBSZXN0QXBpLlxuICogQHBhcmFtIGFwaUdhdGV3YXlQcm9wcyAtIChvcHRpb25hbCkgdXNlci1zcGVjaWZpZWQgcHJvcGVydGllcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wZXJ0aWVzLlxuICovXG5mdW5jdGlvbiBjb25maWd1cmVSZXN0QXBpKHNjb3BlOiBDb25zdHJ1Y3QsIGRlZmF1bHRBcGlHYXRld2F5UHJvcHM6IGFwaS5SZXN0QXBpUHJvcHMsXG4gIGFwaUdhdGV3YXlQcm9wcz86IGFwaS5SZXN0QXBpUHJvcHMpOiBbYXBpLlJlc3RBcGksIGlhbS5Sb2xlIHwgdW5kZWZpbmVkXSB7XG5cbiAgLy8gQVBJIEdhdGV3YXkgZG9lc24ndCBhbGxvdyBib3RoIGVuZHBvaW50VHlwZXMgYW5kIGVuZHBvaW50Q29uZmlndXJhdGlvbiwgY2hlY2sgd2hldGhlciBlbmRQb2ludFR5cGVzIGV4aXN0c1xuICBpZiAoYXBpR2F0ZXdheVByb3BzPy5lbmRwb2ludFR5cGVzKSB7XG4gICAgdGhyb3cgRXJyb3IoJ1NvbHV0aW9ucyBDb25zdHJ1Y3RzIGludGVybmFsbHkgdXNlcyBlbmRwb2ludENvbmZpZ3VyYXRpb24sIHVzZSBlbmRwb2ludENvbmZpZ3VyYXRpb24gaW5zdGVhZCBvZiBlbmRwb2ludFR5cGVzJyk7XG4gIH1cblxuICAvLyBEZWZpbmUgdGhlIEFQSVxuICBsZXQgX2FwaTogYXBpLlJlc3RBcGk7XG4gIGlmIChhcGlHYXRld2F5UHJvcHMpIHtcbiAgICAvLyBJZiBwcm9wZXJ0eSBvdmVycmlkZXMgaGF2ZSBiZWVuIHByb3ZpZGVkLCBpbmNvcnBvcmF0ZSB0aGVtIGFuZCBkZXBsb3lcbiAgICBjb25zdCBfYXBpR2F0ZXdheVByb3BzID0gb3ZlcnJpZGVQcm9wcyhkZWZhdWx0QXBpR2F0ZXdheVByb3BzLCB7IC4uLmFwaUdhdGV3YXlQcm9wcywgY2xvdWRXYXRjaFJvbGU6IGZhbHNlIH0pO1xuICAgIF9hcGkgPSBuZXcgYXBpLlJlc3RBcGkoc2NvcGUsICdSZXN0QXBpJywgX2FwaUdhdGV3YXlQcm9wcyk7XG4gIH0gZWxzZSB7XG4gICAgLy8gSWYgbm8gcHJvcGVydHkgb3ZlcnJpZGVzLCBkZXBsb3kgdXNpbmcgdGhlIGRlZmF1bHQgY29uZmlndXJhdGlvblxuICAgIF9hcGkgPSBuZXcgYXBpLlJlc3RBcGkoc2NvcGUsICdSZXN0QXBpJywgZGVmYXVsdEFwaUdhdGV3YXlQcm9wcyk7XG4gIH1cblxuICBsZXQgY3dSb2xlO1xuXG4gIC8vIENvbmZpZ3VyZSBBUEkgYWNjZXNzIGxvZ2dpbmdcbiAgaWYgKGFwaUdhdGV3YXlQcm9wcz8uY2xvdWRXYXRjaFJvbGUgIT09IGZhbHNlKSB7XG4gICAgY3dSb2xlID0gY29uZmlndXJlQ2xvdWR3YXRjaFJvbGVGb3JBcGkoc2NvcGUsIF9hcGkpO1xuICB9XG5cbiAgLy8gQ29uZmlndXJlIFVzYWdlIFBsYW5cbiAgY29uc3QgdXNhZ2VQbGFuUHJvcHM6IGFwaS5Vc2FnZVBsYW5Qcm9wcyA9IHtcbiAgICBhcGlTdGFnZXM6IFt7XG4gICAgICBhcGk6IF9hcGksXG4gICAgICBzdGFnZTogX2FwaS5kZXBsb3ltZW50U3RhZ2VcbiAgICB9XVxuICB9O1xuXG4gIGNvbnN0IHBsYW4gPSBfYXBpLmFkZFVzYWdlUGxhbignVXNhZ2VQbGFuJywgdXNhZ2VQbGFuUHJvcHMpO1xuXG4gIC8vIElmIHJlcXVpcmVBcGlLZXkgcGFyYW0gaXMgc2V0IHRvIHRydWUsIGNyZWF0ZSBhIGFwaSBrZXkgJiBhc3NvY2lhdGUgdG8gVXNhZ2UgUGxhblxuICBpZiAoYXBpR2F0ZXdheVByb3BzPy5kZWZhdWx0TWV0aG9kT3B0aW9ucz8uYXBpS2V5UmVxdWlyZWQgPT09IHRydWUpIHtcbiAgICAvLyBDb25maWd1cmUgVXNhZ2UgUGxhbiB3aXRoIEFQSSBLZXlcbiAgICBjb25zdCBrZXkgPSBfYXBpLmFkZEFwaUtleSgnQXBpS2V5Jyk7XG4gICAgcGxhbi5hZGRBcGlLZXkoa2V5KTtcbiAgfVxuXG4gIC8vIFJldHVybiB0aGUgQVBJIGFuZCBDVyBSb2xlXG4gIHJldHVybiBbX2FwaSwgY3dSb2xlXTtcbn1cblxuLyoqXG4gKiBCdWlsZHMgYW5kIHJldHVybnMgYSBnbG9iYWwgYXBpLlJlc3RBcGkgZGVzaWduZWQgdG8gYmUgdXNlZCB3aXRoIGFuIEFXUyBMYW1iZGEgZnVuY3Rpb24uXG4gKiBAcGFyYW0gc2NvcGUgLSB0aGUgY29uc3RydWN0IHRvIHdoaWNoIHRoZSBSZXN0QXBpIHNob3VsZCBiZSBhdHRhY2hlZCB0by5cbiAqIEBwYXJhbSBfZXhpc3RpbmdMYW1iZGFPYmogLSBhbiBleGlzdGluZyBBV1MgTGFtYmRhIGZ1bmN0aW9uLlxuICogQHBhcmFtIGFwaUdhdGV3YXlQcm9wcyAtIChvcHRpb25hbCkgdXNlci1zcGVjaWZpZWQgcHJvcGVydGllcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wZXJ0aWVzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gR2xvYmFsTGFtYmRhUmVzdEFwaShzY29wZTogQ29uc3RydWN0LCBfZXhpc3RpbmdMYW1iZGFPYmo6IGxhbWJkYS5GdW5jdGlvbixcbiAgYXBpR2F0ZXdheVByb3BzPzogYXBpLkxhbWJkYVJlc3RBcGlQcm9wcywgbG9nR3JvdXBQcm9wcz86IGxvZ3MuTG9nR3JvdXBQcm9wcyk6IFthcGkuUmVzdEFwaSwgaWFtLlJvbGUgfCB1bmRlZmluZWQsIGxvZ3MuTG9nR3JvdXBdIHtcbiAgLy8gQ29uZmlndXJlIGxvZyBncm91cCBmb3IgQVBJIEdhdGV3YXkgQWNjZXNzTG9nZ2luZ1xuICBjb25zdCBsb2dHcm91cCA9IGJ1aWxkTG9nR3JvdXAoc2NvcGUsICdBcGlBY2Nlc3NMb2dHcm91cCcsIGxvZ0dyb3VwUHJvcHMpO1xuXG4gIGNvbnN0IGRlZmF1bHRQcm9wcyA9IGFwaURlZmF1bHRzLkRlZmF1bHRHbG9iYWxMYW1iZGFSZXN0QXBpUHJvcHMoX2V4aXN0aW5nTGFtYmRhT2JqLCBsb2dHcm91cCk7XG4gIGNvbnN0IFtyZXN0QXBpLCBhcGlDV1JvbGVdID0gY29uZmlndXJlTGFtYmRhUmVzdEFwaShzY29wZSwgZGVmYXVsdFByb3BzLCBhcGlHYXRld2F5UHJvcHMpO1xuICByZXR1cm4gW3Jlc3RBcGksIGFwaUNXUm9sZSwgbG9nR3JvdXAgXTtcbn1cblxuLyoqXG4gKiBCdWlsZHMgYW5kIHJldHVybnMgYSByZWdpb25hbCBhcGkuUmVzdEFwaSBkZXNpZ25lZCB0byBiZSB1c2VkIHdpdGggYW4gQVdTIExhbWJkYSBmdW5jdGlvbi5cbiAqIEBwYXJhbSBzY29wZSAtIHRoZSBjb25zdHJ1Y3QgdG8gd2hpY2ggdGhlIFJlc3RBcGkgc2hvdWxkIGJlIGF0dGFjaGVkIHRvLlxuICogQHBhcmFtIF9leGlzdGluZ0xhbWJkYU9iaiAtIGFuIGV4aXN0aW5nIEFXUyBMYW1iZGEgZnVuY3Rpb24uXG4gKiBAcGFyYW0gYXBpR2F0ZXdheVByb3BzIC0gKG9wdGlvbmFsKSB1c2VyLXNwZWNpZmllZCBwcm9wZXJ0aWVzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BlcnRpZXMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBSZWdpb25hbExhbWJkYVJlc3RBcGkoc2NvcGU6IENvbnN0cnVjdCwgX2V4aXN0aW5nTGFtYmRhT2JqOiBsYW1iZGEuRnVuY3Rpb24sXG4gIGFwaUdhdGV3YXlQcm9wcz86IGFwaS5MYW1iZGFSZXN0QXBpUHJvcHMsIGxvZ0dyb3VwUHJvcHM/OiBsb2dzLkxvZ0dyb3VwUHJvcHMpOiBbYXBpLlJlc3RBcGksIGlhbS5Sb2xlIHwgdW5kZWZpbmVkLCBsb2dzLkxvZ0dyb3VwXSB7XG4gIC8vIENvbmZpZ3VyZSBsb2cgZ3JvdXAgZm9yIEFQSSBHYXRld2F5IEFjY2Vzc0xvZ2dpbmdcbiAgY29uc3QgbG9nR3JvdXAgPSBidWlsZExvZ0dyb3VwKHNjb3BlLCAnQXBpQWNjZXNzTG9nR3JvdXAnLCBsb2dHcm91cFByb3BzKTtcblxuICBjb25zdCBkZWZhdWx0UHJvcHMgPSBhcGlEZWZhdWx0cy5EZWZhdWx0UmVnaW9uYWxMYW1iZGFSZXN0QXBpUHJvcHMoX2V4aXN0aW5nTGFtYmRhT2JqLCBsb2dHcm91cCk7XG4gIGNvbnN0IFtyZXN0QXBpLCBhcGlDV1JvbGVdID0gY29uZmlndXJlTGFtYmRhUmVzdEFwaShzY29wZSwgZGVmYXVsdFByb3BzLCBhcGlHYXRld2F5UHJvcHMpO1xuICByZXR1cm4gW3Jlc3RBcGksIGFwaUNXUm9sZSwgbG9nR3JvdXAgXTtcbn1cblxuLyoqXG4gKiBCdWlsZHMgYW5kIHJldHVybnMgYSBzdGFuZGFyZCBhcGkuUmVzdEFwaS5cbiAqIEBwYXJhbSBzY29wZSAtIHRoZSBjb25zdHJ1Y3QgdG8gd2hpY2ggdGhlIFJlc3RBcGkgc2hvdWxkIGJlIGF0dGFjaGVkIHRvLlxuICogQHBhcmFtIGFwaUdhdGV3YXlQcm9wcyAtIChvcHRpb25hbCkgdXNlci1zcGVjaWZpZWQgcHJvcGVydGllcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wZXJ0aWVzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gR2xvYmFsUmVzdEFwaShzY29wZTogQ29uc3RydWN0LCBhcGlHYXRld2F5UHJvcHM/OiBhcGkuUmVzdEFwaVByb3BzLFxuICBsb2dHcm91cFByb3BzPzogbG9ncy5Mb2dHcm91cFByb3BzKTogW2FwaS5SZXN0QXBpLCBpYW0uUm9sZSB8IHVuZGVmaW5lZCwgbG9ncy5Mb2dHcm91cF0ge1xuICAvLyBDb25maWd1cmUgbG9nIGdyb3VwIGZvciBBUEkgR2F0ZXdheSBBY2Nlc3NMb2dnaW5nXG4gIGNvbnN0IGxvZ0dyb3VwID0gYnVpbGRMb2dHcm91cChzY29wZSwgJ0FwaUFjY2Vzc0xvZ0dyb3VwJywgbG9nR3JvdXBQcm9wcyk7XG5cbiAgY29uc3QgZGVmYXVsdFByb3BzID0gYXBpRGVmYXVsdHMuRGVmYXVsdEdsb2JhbFJlc3RBcGlQcm9wcyhsb2dHcm91cCk7XG4gIGNvbnN0IFtyZXN0QXBpLCBhcGlDV1JvbGVdID0gY29uZmlndXJlUmVzdEFwaShzY29wZSwgZGVmYXVsdFByb3BzLCBhcGlHYXRld2F5UHJvcHMpO1xuICByZXR1cm4gW3Jlc3RBcGksIGFwaUNXUm9sZSwgbG9nR3JvdXAgXTtcbn1cblxuLyoqXG4gKiBCdWlsZHMgYW5kIHJldHVybnMgYSBSZWdpb25hbCBhcGkuUmVzdEFwaS5cbiAqIEBwYXJhbSBzY29wZSAtIHRoZSBjb25zdHJ1Y3QgdG8gd2hpY2ggdGhlIFJlc3RBcGkgc2hvdWxkIGJlIGF0dGFjaGVkIHRvLlxuICogQHBhcmFtIGFwaUdhdGV3YXlQcm9wcyAtIChvcHRpb25hbCkgdXNlci1zcGVjaWZpZWQgcHJvcGVydGllcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wZXJ0aWVzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gUmVnaW9uYWxSZXN0QXBpKHNjb3BlOiBDb25zdHJ1Y3QsIGFwaUdhdGV3YXlQcm9wcz86IGFwaS5SZXN0QXBpUHJvcHMsXG4gIGxvZ0dyb3VwUHJvcHM/OiBsb2dzLkxvZ0dyb3VwUHJvcHMpOiBbYXBpLlJlc3RBcGksIGlhbS5Sb2xlIHwgdW5kZWZpbmVkLCBsb2dzLkxvZ0dyb3VwXSB7XG4gIC8vIENvbmZpZ3VyZSBsb2cgZ3JvdXAgZm9yIEFQSSBHYXRld2F5IEFjY2Vzc0xvZ2dpbmdcbiAgY29uc3QgbG9nR3JvdXAgPSBidWlsZExvZ0dyb3VwKHNjb3BlLCAnQXBpQWNjZXNzTG9nR3JvdXAnLCBsb2dHcm91cFByb3BzKTtcblxuICBjb25zdCBkZWZhdWx0UHJvcHMgPSBhcGlEZWZhdWx0cy5EZWZhdWx0UmVnaW9uYWxSZXN0QXBpUHJvcHMobG9nR3JvdXApO1xuICBjb25zdCBbcmVzdEFwaSwgYXBpQ1dSb2xlXSA9IGNvbmZpZ3VyZVJlc3RBcGkoc2NvcGUsIGRlZmF1bHRQcm9wcywgYXBpR2F0ZXdheVByb3BzKTtcbiAgcmV0dXJuIFtyZXN0QXBpLCBhcGlDV1JvbGUsIGxvZ0dyb3VwIF07XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQWRkUHJveHlNZXRob2RUb0FwaVJlc291cmNlSW5wdXRQYXJhbXMge1xuICAgIHJlYWRvbmx5IHNlcnZpY2U6IHN0cmluZyxcbiAgICByZWFkb25seSBhY3Rpb24/OiBzdHJpbmcsXG4gICAgcmVhZG9ubHkgcGF0aD86IHN0cmluZyxcbiAgICByZWFkb25seSBhcGlSZXNvdXJjZTogYXBpLklSZXNvdXJjZSxcbiAgICByZWFkb25seSBhcGlNZXRob2Q6IHN0cmluZyxcbiAgICByZWFkb25seSBhcGlHYXRld2F5Um9sZTogSVJvbGUsXG4gICAgcmVhZG9ubHkgcmVxdWVzdFRlbXBsYXRlOiBzdHJpbmcsXG4gICAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBzdHJpbmcsXG4gICAgcmVhZG9ubHkgcmVxdWVzdFZhbGlkYXRvcj86IGFwaS5JUmVxdWVzdFZhbGlkYXRvcixcbiAgICByZWFkb25seSByZXF1ZXN0TW9kZWw/OiB7IFtjb250ZW50VHlwZTogc3RyaW5nXTogYXBpLklNb2RlbDsgfSxcbiAgICByZWFkb25seSBhd3NJbnRlZ3JhdGlvblByb3BzPzogYXBpLkF3c0ludGVncmF0aW9uUHJvcHMgfCBhbnksXG4gICAgcmVhZG9ubHkgbWV0aG9kT3B0aW9ucz86IGFwaS5NZXRob2RPcHRpb25zXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhZGRQcm94eU1ldGhvZFRvQXBpUmVzb3VyY2UocGFyYW1zOiBBZGRQcm94eU1ldGhvZFRvQXBpUmVzb3VyY2VJbnB1dFBhcmFtcyk6IGFwaS5NZXRob2Qge1xuICBsZXQgYmFzZVByb3BzOiBhcGkuQXdzSW50ZWdyYXRpb25Qcm9wcyA9IHtcbiAgICBzZXJ2aWNlOiBwYXJhbXMuc2VydmljZSxcbiAgICBpbnRlZ3JhdGlvbkh0dHBNZXRob2Q6IFwiUE9TVFwiLFxuICAgIG9wdGlvbnM6IHtcbiAgICAgIHBhc3N0aHJvdWdoQmVoYXZpb3I6IGFwaS5QYXNzdGhyb3VnaEJlaGF2aW9yLk5FVkVSLFxuICAgICAgY3JlZGVudGlhbHNSb2xlOiBwYXJhbXMuYXBpR2F0ZXdheVJvbGUsXG4gICAgICByZXF1ZXN0UGFyYW1ldGVyczoge1xuICAgICAgICBcImludGVncmF0aW9uLnJlcXVlc3QuaGVhZGVyLkNvbnRlbnQtVHlwZVwiOiBwYXJhbXMuY29udGVudFR5cGUgPyBwYXJhbXMuY29udGVudFR5cGUgOiBcIidhcHBsaWNhdGlvbi9qc29uJ1wiXG4gICAgICB9LFxuICAgICAgcmVxdWVzdFRlbXBsYXRlczoge1xuICAgICAgICBcImFwcGxpY2F0aW9uL2pzb25cIjogcGFyYW1zLnJlcXVlc3RUZW1wbGF0ZVxuICAgICAgfSxcbiAgICAgIGludGVncmF0aW9uUmVzcG9uc2VzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBcIjIwMFwiXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBcIjUwMFwiLFxuICAgICAgICAgIHJlc3BvbnNlVGVtcGxhdGVzOiB7XG4gICAgICAgICAgICBcInRleHQvaHRtbFwiOiBcIkVycm9yXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIHNlbGVjdGlvblBhdHRlcm46IFwiNTAwXCJcbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfTtcblxuICBsZXQgZXh0cmFQcm9wcztcblxuICBpZiAocGFyYW1zLmFjdGlvbikge1xuICAgIGV4dHJhUHJvcHMgPSB7XG4gICAgICBhY3Rpb246IHBhcmFtcy5hY3Rpb25cbiAgICB9O1xuICB9IGVsc2UgaWYgKHBhcmFtcy5wYXRoKSB7XG4gICAgZXh0cmFQcm9wcyA9IHtcbiAgICAgIHBhdGg6IHBhcmFtcy5wYXRoXG4gICAgfTtcbiAgfSBlbHNlIHtcbiAgICB0aHJvdyBFcnJvcignRWl0aGVyIGFjdGlvbiBvciBwYXRoIGlzIHJlcXVpcmVkJyk7XG4gIH1cblxuICAvLyBTZXR1cCB0aGUgQVBJIEdhdGV3YXkgQVdTIEludGVncmF0aW9uXG4gIGJhc2VQcm9wcyA9IE9iamVjdC5hc3NpZ24oYmFzZVByb3BzLCBleHRyYVByb3BzKTtcbiAgbGV0IGFwaUdhdGV3YXlJbnRlZ3JhdGlvbjtcbiAgaWYgKHBhcmFtcy5hd3NJbnRlZ3JhdGlvblByb3BzKSB7XG4gICAgY29uc3Qgb3ZlcnJpZGVuUHJvcHMgPSBvdmVycmlkZVByb3BzKGJhc2VQcm9wcywgcGFyYW1zLmF3c0ludGVncmF0aW9uUHJvcHMpO1xuICAgIGFwaUdhdGV3YXlJbnRlZ3JhdGlvbiA9IG5ldyBhcGkuQXdzSW50ZWdyYXRpb24ob3ZlcnJpZGVuUHJvcHMpO1xuICB9IGVsc2Uge1xuICAgIGFwaUdhdGV3YXlJbnRlZ3JhdGlvbiA9IG5ldyBhcGkuQXdzSW50ZWdyYXRpb24oYmFzZVByb3BzKTtcbiAgfVxuXG4gIGNvbnN0IGRlZmF1bHRNZXRob2RPcHRpb25zID0ge1xuICAgIG1ldGhvZFJlc3BvbnNlczogW1xuICAgICAge1xuICAgICAgICBzdGF0dXNDb2RlOiBcIjIwMFwiLFxuICAgICAgICByZXNwb25zZVBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBcIm1ldGhvZC5yZXNwb25zZS5oZWFkZXIuQ29udGVudC1UeXBlXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgc3RhdHVzQ29kZTogXCI1MDBcIixcbiAgICAgICAgcmVzcG9uc2VQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgXCJtZXRob2QucmVzcG9uc2UuaGVhZGVyLkNvbnRlbnQtVHlwZVwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICB9XG4gICAgXSxcbiAgICByZXF1ZXN0VmFsaWRhdG9yOiBwYXJhbXMucmVxdWVzdFZhbGlkYXRvcixcbiAgICByZXF1ZXN0TW9kZWxzOiBwYXJhbXMucmVxdWVzdE1vZGVsXG4gIH07XG5cbiAgbGV0IGFwaU1ldGhvZDtcbiAgLy8gU2V0dXAgdGhlIEFQSSBHYXRld2F5IG1ldGhvZFxuICBpZiAocGFyYW1zLm1ldGhvZE9wdGlvbnMpIHtcbiAgICBjb25zdCBvdmVycmlkZW5Qcm9wcyA9ICBvdmVycmlkZVByb3BzKGRlZmF1bHRNZXRob2RPcHRpb25zLCBwYXJhbXMubWV0aG9kT3B0aW9ucyk7XG4gICAgYXBpTWV0aG9kID0gcGFyYW1zLmFwaVJlc291cmNlLmFkZE1ldGhvZChwYXJhbXMuYXBpTWV0aG9kLCBhcGlHYXRld2F5SW50ZWdyYXRpb24sIG92ZXJyaWRlblByb3BzKTtcbiAgfSBlbHNlIHtcbiAgICBhcGlNZXRob2QgPSBwYXJhbXMuYXBpUmVzb3VyY2UuYWRkTWV0aG9kKHBhcmFtcy5hcGlNZXRob2QsIGFwaUdhdGV3YXlJbnRlZ3JhdGlvbiwgZGVmYXVsdE1ldGhvZE9wdGlvbnMpO1xuICB9XG4gIHJldHVybiBhcGlNZXRob2Q7XG59Il19