"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckAlbProps = exports.GetActiveListener = exports.AddFargateTarget = exports.AddLambdaTarget = exports.AddListener = exports.ObtainAlb = void 0;
const elb = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const elbt = require("@aws-cdk/aws-elasticloadbalancingv2-targets");
const utils_1 = require("./utils");
const alb_defaults_1 = require("./alb-defaults");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
//  Returns the correct ALB Load Balancer to use in this construct, either an existing
//  one provided as an argument or create  new one otherwise.
function ObtainAlb(scope, id, vpc, publicApi, existingLoadBalancerObj, loadBalancerProps, logAccessLogs, loggingBucketProps) {
    let loadBalancer;
    if (existingLoadBalancerObj) {
        loadBalancer = existingLoadBalancerObj;
    }
    else {
        const consolidatedProps = loadBalancerProps
            ? utils_1.overrideProps(loadBalancerProps, { vpc, internetFacing: publicApi })
            : { vpc, internetFacing: publicApi };
        loadBalancer = new elb.ApplicationLoadBalancer(scope, `${id}-alb`, consolidatedProps);
        if (logAccessLogs === undefined || logAccessLogs === true) {
            const consolidatedLoggingBucketProps = loggingBucketProps
                ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultLoggingBucketProps(), loggingBucketProps)
                : s3_bucket_defaults_1.DefaultLoggingBucketProps();
            const loggingBucket = s3_bucket_helper_1.createAlbLoggingBucket(scope, id, consolidatedLoggingBucketProps);
            loadBalancer.logAccessLogs(loggingBucket);
        }
    }
    return loadBalancer;
}
exports.ObtainAlb = ObtainAlb;
function AddListener(scope, id, loadBalancer, listenerProps) {
    var _a;
    let consolidatedListenerProps;
    consolidatedListenerProps = utils_1.overrideProps(alb_defaults_1.DefaultListenerProps(loadBalancer), listenerProps);
    //   create the listener
    const listener = new elb.ApplicationListener(scope, `${id}-listener`, consolidatedListenerProps);
    loadBalancer.listeners.push(listener);
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTP) {
        // This will use core.printWarning in the actual construct
        utils_1.printWarning("AWS recommends encrypting traffic to an Application Load Balancer using HTTPS.");
        if (((_a = listenerProps.certificates) === null || _a === void 0 ? void 0 : _a.length) > 0) {
            throw new Error("HTTP listeners cannot use a certificate");
        }
    }
    else {
        if (!listenerProps.certificates || listenerProps.certificates.length === 0) {
            throw new Error("A listener using HTTPS protocol requires a certificate");
        }
        listener.addCertificates(`${id}-cert`, listenerProps.certificates);
    }
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTPS) {
        const opt = {
            port: "443",
            protocol: "HTTPS",
        };
        const httpListener = new elb.ApplicationListener(scope, `${id}-redirect`, {
            loadBalancer,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            defaultAction: aws_elasticloadbalancingv2_1.ListenerAction.redirect(opt),
        });
        loadBalancer.listeners.push(httpListener);
    }
    return listener;
}
exports.AddListener = AddListener;
// Creates a Target Group for Lambda functions and adds the
// provided functions as a target to that group. Then adds
// the new Target Group to the provided Listener.
function AddLambdaTarget(scope, id, currentListener, lambdaFunction, ruleProps, targetProps) {
    //  Create the target and assign it to a new target group
    const lambdaTarget = new elbt.LambdaTarget(lambdaFunction);
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, {
        targets: [lambdaTarget],
        targetGroupName: targetProps ? targetProps.targetGroupName : undefined,
        healthCheck: targetProps ? targetProps.healthCheck : undefined
    });
    // The interface AddRuleProps includes conditions and priority, combine that
    // with targetGroups and we can assemble AddApplicationTargetGroupProps
    if (ruleProps) {
        const consolidatedTargetProps = utils_1.overrideProps(ruleProps, { targetGroups: [newTargetGroup] });
        currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    }
    else {
        currentListener.addTargetGroups("targets", {
            targetGroups: [newTargetGroup],
        });
    }
    newTargetGroup.setAttribute('stickiness.enabled', undefined);
    return newTargetGroup;
}
exports.AddLambdaTarget = AddLambdaTarget;
function AddFargateTarget(scope, id, currentListener, fargateService, ruleProps, targetProps) {
    if ((targetProps === null || targetProps === void 0 ? void 0 : targetProps.protocol) !== elb.ApplicationProtocol.HTTPS) {
        utils_1.printWarning('AWS recommends using HTTPS protocol for Target Groups in production applications');
    }
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, targetProps);
    // The interface AddRuleProps includes conditions and priority, combine that
    // with targetGroups and we can assemble an AddApplicationTargetGroupProps object
    if (ruleProps) {
        const consolidatedTargetProps = utils_1.overrideProps(ruleProps, { targetGroups: [newTargetGroup] });
        currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    }
    else {
        currentListener.addTargetGroups(`${id}-targets`, {
            targetGroups: [newTargetGroup],
        });
    }
    newTargetGroup.addTarget(fargateService);
    return newTargetGroup;
}
exports.AddFargateTarget = AddFargateTarget;
// Looks for the listener associated with Target Groups
// If there is a single listener, this returns it whether it is HTTP or HTTPS
// If there are 2 listeners, it finds the HTTPS listener (we assume the HTTP listener redirects to HTTPS)
function GetActiveListener(listeners) {
    let listener;
    if (listeners.length === 0) {
        throw new Error(`There are no listeners in the ALB`);
    }
    if (listeners.length === 1) {
        listener = listeners[0];
    }
    else {
        listener = listeners.find(i => i.node.children[0].protocol === "HTTPS");
    }
    return listener;
}
exports.GetActiveListener = GetActiveListener;
function CheckAlbProps(props) {
    var _a, _b;
    let errorMessages = '';
    let errorFound = false;
    if ((_a = props.listenerProps) === null || _a === void 0 ? void 0 : _a.certificateArns) {
        errorMessages += "certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates\n";
        errorFound = true;
    }
    if (((props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length === 0) ||
        !props.existingLoadBalancerObj) &&
        !props.listenerProps) {
        errorMessages += "When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP\n";
        errorFound = true;
    }
    if (props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length > 0 &&
        props.listenerProps) {
        errorFound = true;
        errorMessages += "This load balancer already has a listener, listenerProps may not be specified\n";
    }
    if (props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length > 0 &&
        !props.ruleProps) {
        errorFound = true;
        errorMessages += "When adding a second target to an existing listener, there must be rules provided\n";
    }
    // Check construct specific invalid inputs
    if (props.existingLoadBalancerObj && !props.existingVpc) {
        errorFound = true;
        errorMessages += "An existing ALB is already in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use.\n";
    }
    if ((_b = props.loadBalancerProps) === null || _b === void 0 ? void 0 : _b.vpc) {
        errorFound = true;
        errorMessages += 'Specify any existing VPC at the construct level, not within loadBalancerProps.\n';
    }
    if (props.existingLoadBalancerObj) {
        utils_1.printWarning("The public/private property of an existing ALB must match the props.publicApi setting provided.");
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckAlbProps = CheckAlbProps;
//# sourceMappingURL=data:application/json;base64,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