"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
module.exports = {
    'Tasks can add permissions to the execution role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource']
                        })],
                })
            }
        });
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: task
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: "resource:Everything",
                        Effect: "Allow",
                        Resource: "resource"
                    }
                ],
            }
        }));
        test.done();
    },
    'Tasks hidden inside a Parallel state are also included'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [
                        new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource']
                        })
                    ]
                })
            }
        });
        const para = new stepfunctions.Parallel(stack, 'Para');
        para.branch(task);
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: para
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: "resource:Everything",
                        Effect: "Allow",
                        Resource: "resource"
                    }
                ],
            }
        }));
        test.done();
    },
    'Task should render InputPath / Parameters / OutputPath correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            inputPath: "$",
            outputPath: "$.state",
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    parameters: {
                        "input.$": "$",
                        "stringArgument": "inital-task",
                        "numberArgument": 123,
                        "booleanArgument": true,
                        "arrayArgument": ["a", "b", "c"]
                    }
                })
            }
        });
        // WHEN
        const taskState = task.toStateJson();
        // THEN
        test.deepEqual(taskState, { End: true,
            Retry: undefined,
            Catch: undefined,
            InputPath: '$',
            Parameters: { 'input.$': '$',
                'stringArgument': 'inital-task',
                'numberArgument': 123,
                'booleanArgument': true,
                'arrayArgument': ['a', 'b', 'c'] },
            OutputPath: '$.state',
            Type: 'Task',
            Comment: undefined,
            Resource: 'resource',
            ResultPath: undefined,
            TimeoutSeconds: undefined,
            HeartbeatSeconds: undefined
        });
        test.done();
    },
    'Task combines taskobject parameters with direct parameters'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            inputPath: "$",
            outputPath: "$.state",
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    parameters: {
                        a: "aa",
                    }
                })
            },
            parameters: {
                b: "bb"
            }
        });
        // WHEN
        const taskState = task.toStateJson();
        // THEN
        test.deepEqual(taskState, { End: true,
            Retry: undefined,
            Catch: undefined,
            InputPath: '$',
            Parameters: { a: 'aa',
                b: 'bb', },
            OutputPath: '$.state',
            Type: 'Task',
            Comment: undefined,
            Resource: 'resource',
            ResultPath: undefined,
            TimeoutSeconds: undefined,
            HeartbeatSeconds: undefined
        });
        test.done();
    },
    'Created state machine can grant start execution to a role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({ resourceArn: 'resource' })
            }
        });
        const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
            definition: task
        });
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
        });
        // WHEN
        stateMachine.grantStartExecution(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'states:StartExecution',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'StateMachine2E01A3A5'
                        }
                    }
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'RoleDefaultPolicy5FFB7DAB',
            Roles: [
                {
                    Ref: 'Role1ABCC5F0'
                }
            ]
        }));
        test.done();
    },
    'Imported state machine can grant start execution to a role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const stateMachineArn = 'arn:aws:states:::my-state-machine';
        const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
        });
        // WHEN
        stateMachine.grantStartExecution(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'states:StartExecution',
                        Effect: 'Allow',
                        Resource: stateMachineArn
                    }
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'RoleDefaultPolicy5FFB7DAB',
            Roles: [
                {
                    Ref: 'Role1ABCC5F0'
                }
            ]
        }));
        test.done();
    },
    'Pass should render InputPath / Parameters / OutputPath correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Pass(stack, 'Pass', {
            inputPath: "$",
            outputPath: "$.state",
            parameters: {
                "input.$": "$",
                "stringArgument": "inital-task",
                "numberArgument": 123,
                "booleanArgument": true,
                "arrayArgument": ["a", "b", "c"]
            }
        });
        // WHEN
        const taskState = task.toStateJson();
        // THEN
        test.deepEqual(taskState, { End: true,
            InputPath: '$',
            OutputPath: '$.state',
            Parameters: { 'input.$': '$',
                'stringArgument': 'inital-task',
                'numberArgument': 123,
                'booleanArgument': true,
                'arrayArgument': ['a', 'b', 'c'] },
            Type: 'Pass',
            Comment: undefined,
            Result: undefined,
            ResultPath: undefined,
        });
        test.done();
    },
    'State machines must depend on their roles'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [
                        new iam.PolicyStatement({
                            resources: ['resource'],
                            actions: ["lambda:InvokeFunction"],
                        })
                    ],
                })
            }
        });
        new stepfunctions.StateMachine(stack, 'StateMachine', {
            definition: task
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::StepFunctions::StateMachine', {
            DependsOn: [
                'StateMachineRoleDefaultPolicyDF1E6607',
                'StateMachineRoleB840431D'
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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