"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,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