"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        }
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers) {
    if (obj === undefined) {
        return undefined;
    }
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers);
        }
    }
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers);
            }
            return value;
        })
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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