# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mongodb_odm', 'mongodb_odm.utils']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.10.7,<2.0.0', 'pymongo[srv]>=4.3.3,<5.0.0']

setup_kwargs = {
    'name': 'mongodb-odm',
    'version': '0.2.1',
    'description': 'MongoDB-ODM, NOSQL databases in Python, designed for simplicity, compatibility, and robustness.',
    'long_description': '# MongoDB-ODM\n\n<p align="center">\n    <em>MongoDB-ODM, NOSQL databases in Python, designed for simplicity, compatibility, and robustness.</em>\n</p>\n\n<p align="center">\n\n<a href="https://github.com/nayan32biswas/mongodb-odm/actions/workflows/test.yml" target="_blank">\n    <img src="https://github.com/nayan32biswas/mongodb-odm/actions/workflows/test.yml/badge.svg?branch=main&event=push" alt="Test">\n</a>\n<a href="https://coverage-badge.samuelcolvin.workers.dev/redirect/nayan32biswas/mongodb-odm" target="_blank">\n    <img src="https://coverage-badge.samuelcolvin.workers.dev/nayan32biswas/mongodb-odm.svg" alt="Coverage">\n<br />\n<a href="https://pypi.org/project/mongodb-odm/" target="_blank">\n    <img alt="PyPI" src="https://img.shields.io/pypi/v/mongodb-odm?color=%2334D058&label=PyPi%20Package">\n</a>\n<a href="https://pypi.org/project/mongodb-odm/" target="_blank">\n    <img alt="PyPI - Python Version" src="https://img.shields.io/pypi/pyversions/mongodb-odm?color=%2334D058">\n</a>\n\n</p>\n\n---\n\n**Documentation**: <a href="https://mongodb-odm.readthedocs.io" target="_blank">https://mongodb-odm.readthedocs.io</a>\n\n**PyPi**: <a href="https://pypi.org/project/mongodb-odm" target="_blank">https://pypi.org/project/mongodb-odm</a>\n\n**Repository**: <a href="https://github.com/nayan32biswas/mongodb-odm" target="_blank">https://github.com/nayan32biswas/mongodb-odm</a>\n\n---\n\n## Introduction\n\nThe purpose of this module is to provide easy access to the database with the python object feature with **MongoDB** and **PyMongo**. With PyMongo that was very easy to make spelling mistakes in a collection name when you are doing database operation. This module provides you with minimal ODM with a modeling feature so that you don’t have to look up the MongoDB dashboard(Mongo Compass) to know about field names or data types.\n\n**MongoDB-ODM** is based on Python type annotations, and powered by <a href="https://pymongo.readthedocs.io/en/stable/" class="external-link" target="_blank">PyMongo</a> and <a href="https://docs.pydantic.dev/" class="external-link" target="_blank">Pydantic</a>.\n\nThe key features are:\n\n- **Intuitive to write**: Great editor support. Completion everywhere. Less time debugging. Designed to be easy to use and learn. Less time reading docs.\n- **Easy to use**: It has sensible defaults and does a lot of work underneath to simplify the code you write.\n- **Compatible**: It is designed to be compatible with **FastAPI**, **Pydantic**, and **PyMongo**.\n- **Extensible**: You have all the power of **PyMongo** and **Pydantic** underneath.\n- **Short**: Minimize code duplication. A single type annotation does a lot of work. No need to duplicate models in **PyMongo** and Pydantic.\n\n---\n\n## Requirement\n\n**MongoDB-ODM** will work on <a href="https://www.python.org/downloads/" class="external-link" target="_blank">Python 3.8 and above</a>\n\nThis **MongoDB-ODM** is built on top of **PyMongo** and **Pydantic**. Those packages are required and will auto-install while **MongoDB-ODM** was installed.\n\n## Installation\n\n```console\n$ pip install mongodb-odm\n```\n\n## Example\n\n### Define model\n\n```Python\nimport os\nfrom typing import Optional\n\nfrom mongodb_odm import ASCENDING, Document, IndexModel, connect\n\n\nclass Player(Document):\n    name: str\n    country_code: str\n    rating: Optional[int] = None\n\n    class Config(Document.Config):\n        indexes = [\n            IndexModel([("rating", ASCENDING)]),\n        ]\n```\n\n### Set Connection\n\n```Python\nconnect(os.environ.get("MONGO_URL", "mongodb://localhost:27017/testdb"))\n```\n\n### Create Document\n\n```Python\npele = Player(name="Pelé", country_code="BRA").create()\nmaradona = Player(name="Diego Maradona", country_code="ARG", rating=97).create()\nzidane = Player(name="Zinedine Zidane", country_code="FRA", rating=96).create()\n```\n\n### Retrieve Document\n\n#### Find data from collection\n\n```Python\nfor player in Player.find():\n    print(player)\n```\n\n#### Find one object with filter\n\n```Python\nplayer = Player.find_one({"name": "Pelé"})\n```\n\n### Update Data\n\n```Python\nplayer = Player.find_one({"name": "Pelé"})\nif player:\n    player.rating = 98  # potential\n    player.update()\n```\n\n### Delete Data\n\n```Python\nplayer = Player.find_one({"name": "Pelé"})\nif player:\n    player.delete()\n```\n\n### Apply Indexes\n\n```Python\nimport os\nfrom typing import Optional\n\nfrom mongodb_odm import ASCENDING, Document, IndexModel, connect\n\n\nclass Player(Document):\n    name: str\n    country_code: str\n    rating: Optional[int] = None\n\n    class Config(Document.Config):\n        indexes = [\n            IndexModel([("rating", ASCENDING)]),\n        ]\n```\n\n- To create indexes in the database declare [IndexModel](https://pymongo.readthedocs.io/en/stable/tutorial.html#indexing) and assign in indexes array in Config class. **IndexModel** modules that are directly imported from **pymongo**.\n- Import the `apply_indexes` from `mongodb_odm`. Call the `apply_indexes` function from your CLI. You can use <a href="https://typer.tiangolo.com" class="external-link" target="_blank">Typer</a> to implement CLI.\n\n## Example Code\n\nThis is the example of full code of above.\n\n```python\nimport os\nfrom typing import Optional\n\nfrom mongodb_odm import ASCENDING, Document, IndexModel, connect\n\n\nclass Player(Document):\n    name: str\n    country_code: str\n    rating: Optional[int] = None\n\n    class Config(Document.Config):\n        indexes = [\n            IndexModel([("rating", ASCENDING)]),\n        ]\n\n\nconnect(os.environ.get("MONGO_URL", "mongodb://localhost:27017/testdb"))\n\npele = Player(name="Pelé", country_code="BRA").create()\nmaradona = Player(name="Diego Maradona", country_code="ARG", rating=97).create()\nzidane = Player(name="Zinedine Zidane", country_code="FRA", rating=96).create()\n\nfor player in Player.find():\n    print(player)\n\nplayer = Player.find_one({"name": "Pelé"})\nif player:\n    player.rating = 98  # potential\n    player.update()\n\nplayer = Player.find_one({"name": "Pelé"})\nif player:\n    player.delete()\n```\n\n### Supported Framework\n\n**MongoDB-ODM** is not framework dependent. We can use this package in any system. But we take special consideration being compatible with <a href="https://fastapi.tiangolo.com/" class="external-link" target="_blank">FastAPI</a> and <a href="https://flask.palletsprojects.com/en/2.2.x/" class="external-link" target="_blank">Flask</a>.\n\n### Credit\n\nThis package is built on top of <a href="https://pymongo.readthedocs.io/en/stable" class="external-link" target="_blank">PyMongo</a> and <a href="https://docs.pydantic.dev" class="external-link" target="_blank">Pydantic</a>.\n\nDocumentation generated by <a href="https://www.mkdocs.org/" class="external-link" target="_blank">MkDocs</a> and <a href="https://squidfunk.github.io/mkdocs-material/" class="external-link" target="_blank">Material for MkDocs</a>.\n\nDocumentation inspired by <a href="https://sqlmodel.tiangolo.com" class="external-link" target="_blank">SQLModel</a>.\n\nBut we use other packages for development and other purposes. Check **pyproject.toml** to know about all packages we use to build this package.\n\n## License\n\nThis project is licensed under the terms of the [MIT license](https://github.com/nayan32biswas/mongodb-odm/blob/main/LICENSE).\n',
    'author': 'Nayan Biswas',
    'author_email': 'nayan32biswas@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/nayan32biswas/mongodb-odm',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
