# -*- coding: utf-8 -*-

#  Licensed under the Apache License, Version 2.0 (the "License"); you may
#  not use this file except in compliance with the License. You may obtain
#  a copy of the License at
#
#       https://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#  License for the specific language governing permissions and limitations
#  under the License.

from __future__ import unicode_literals, absolute_import

import json
import unittest

import responses

from linebot import (
    LineBotApi
)
from linebot.models import (
    ImagemapSendMessage, BaseSize, URIImagemapAction,
    ImagemapArea, MessageImagemapAction,
    Video, ExternalLink
)


class TestLineBotApi(unittest.TestCase):
    def setUp(self):
        self.tested = LineBotApi('channel_secret')

        self.imagemap_message = ImagemapSendMessage(
            base_url='https://example.com/base',
            alt_text='this is an imagemap',
            base_size=BaseSize(height=1040, width=1040),
            video=Video(
                original_content_url='https://example.com/video.mp4',
                preview_image_url='https://example.com/video_preview.jpg',
                area=ImagemapArea(
                    x=0, y=0, width=1040, height=585
                ),
                external_link=ExternalLink(
                    link_uri='https://example.com/see_more.html',
                    label='See More',
                ),
            ),
            actions=[
                URIImagemapAction(
                    link_uri='https://example.com/',
                    area=ImagemapArea(
                        x=0, y=0, width=520, height=1040
                    )
                ),
                MessageImagemapAction(
                    text='hello',
                    area=ImagemapArea(
                        x=520, y=0, width=520, height=1040
                    )
                )
            ]
        )

        self.message = [{
            "type": "imagemap",
            "baseUrl": "https://example.com/base",
            "altText": "this is an imagemap",
            "baseSize": {
                "height": 1040,
                "width": 1040
            },
            "video": {
                "originalContentUrl": "https://example.com/video.mp4",
                "previewImageUrl": "https://example.com/video_preview.jpg",
                "area": {
                    "x": 0,
                    "y": 0,
                    "width": 1040,
                    "height": 585
                },
                "externalLink": {
                    "linkUri": "https://example.com/see_more.html",
                    "label": "See More"
                }
            },
            "actions": [
                {
                    "type": "uri",
                    "linkUri": "https://example.com/",
                    "area": {
                        "x": 0,
                        "y": 0,
                        "width": 520,
                        "height": 1040
                    }
                },
                {
                    "type": "message",
                    "text": "hello",
                    "area": {
                        "x": 520,
                        "y": 0,
                        "width": 520,
                        "height": 1040
                    }
                }
            ]
        }]

    @responses.activate
    def test_push_imagemap_message(self):
        responses.add(
            responses.POST,
            LineBotApi.DEFAULT_API_ENDPOINT + '/v2/bot/message/push',
            json={}, status=200
        )

        self.tested.push_message('to', self.imagemap_message)

        request = responses.calls[0].request
        self.assertEqual(request.method, 'POST')
        self.assertEqual(
            request.url,
            LineBotApi.DEFAULT_API_ENDPOINT + '/v2/bot/message/push')
        self.assertEqual(
            json.loads(request.body),
            {
                "to": "to",
                'notificationDisabled': False,
                "messages": self.message
            }
        )

    @responses.activate
    def test_reply_imagemap_message(self):
        responses.add(
            responses.POST,
            LineBotApi.DEFAULT_API_ENDPOINT + '/v2/bot/message/reply',
            json={}, status=200
        )

        self.tested.reply_message('replyToken', self.imagemap_message)

        request = responses.calls[0].request
        self.assertEqual(request.method, 'POST')
        self.assertEqual(
            request.url,
            LineBotApi.DEFAULT_API_ENDPOINT + '/v2/bot/message/reply')
        self.assertEqual(
            json.loads(request.body),
            {
                "replyToken": "replyToken",
                'notificationDisabled': False,
                "messages": self.message
            }
        )

    @responses.activate
    def test_multicast_imagemap_message(self):
        responses.add(
            responses.POST,
            LineBotApi.DEFAULT_API_ENDPOINT + '/v2/bot/message/multicast',
            json={}, status=200
        )

        self.tested.multicast(['to1', 'to2'], self.imagemap_message)

        request = responses.calls[0].request
        self.assertEqual(request.method, 'POST')
        self.assertEqual(
            request.url,
            LineBotApi.DEFAULT_API_ENDPOINT + '/v2/bot/message/multicast')
        self.assertEqual(
            json.loads(request.body),
            {
                "to": ['to1', 'to2'],
                'notificationDisabled': False,
                "messages": self.message
            }
        )


if __name__ == '__main__':
    unittest.main()
