"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(JSON.stringify(event, null, 2));
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        if (language !== 'typescript' && assembly.targets[language] == null) {
            console.error(`Package ${assembly.name}@${assembly.version} does not support ${language}, skipping!`);
            console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
            for (const submodule of [undefined, ...submodules]) {
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(language), submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                const response = await uploadFile(event.bucket, key, event.assembly.versionId);
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            }
            return created;
        }
        const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
            metrics.setDimensions();
            metrics.setNamespace(constants_1.METRICS_NAMESPACE);
            const uploads = new Map();
            const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
            function renderAndDispatch(submodule) {
                console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                const page = Buffer.from(docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render());
                metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                const { buffer: body, contentEncoding } = compressContent(page);
                metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                console.log(`Uploading ${key}`);
                const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                uploads.set(key, upload);
            }
            renderAndDispatch();
            for (const submodule of submodules) {
                renderAndDispatch(submodule);
            }
            for (const [key, upload] of uploads.entries()) {
                const response = await upload;
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
            }
        });
        await generateDocs(language);
        return created;
    });
}
exports.handler = handler;
function compressContent(buffer) {
    if (buffer.length < 1024) {
        return { buffer };
    }
    const gz = zlib_1.gzipSync(buffer, { level: 9 });
    // If it did not compress well, we'll keep the un-compressed original...
    if (gz.length >= buffer.length) {
        return { buffer };
    }
    return { buffer, contentEncoding: 'gzip' };
}
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public',
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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