"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT, language_1.DocumentationLanguage.JAVA];
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" throttled', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['Lambda.TooManyRequestsException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            // We have to prepare the input to be a JSON string, within an array, because the ECS task integration expects
            // an array of strings (the model if that of a CLI invocation).
            // Unfortunately, we have to split this in two Pass states, because I don't know how to make it work otherwise.
            return new aws_stepfunctions_1.Pass(this, `Prepare ${language}`, {
                parameters: { command: { 'bucket.$': '$.bucket', 'assembly.$': '$.assembly', '$TaskExecution.$': '$.$TaskExecution' } },
                resultPath: '$',
            })
                .next(new aws_stepfunctions_1.Pass(this, `Stringify ${language} input`, {
                parameters: { 'commands.$': 'States.Array(States.JsonToString($.command))' },
                resultPath: '$',
            })
                .next(this.transliterator.createEcsRunTask(this, `Generate ${language} docs`, {
                cluster: this.ecsCluster,
                inputPath: '$.commands',
                language,
                resultSelector: { result: { 'language': language, 'success.$': '$' } },
                vpcSubnets: props.vpcSubnets,
            })
                // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
                .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
                .addRetry({
                errors: [
                    'ECS.AmazonECSException',
                    'jsii-docgen.NpmError.E429',
                    'jsii-codgen.NpmError.EPROTO',
                ],
                ...THROTTLE_RETRY_POLICY,
            })
                .addRetry({ maxAttempts: 3 })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" timed out`, { parameters: { error: 'Timed out!', language } }), { errors: ['States.Timeout'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" service error`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['ECS.AmazonECSException'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', language } }), { errors: ['States.TaskFailed'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" fault`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['States.ALL'] })));
        }))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it.
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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