"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, _context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const match = submoduleKeyRegexp(language).exec(key);
                if (match != null) {
                    const [, submodule, isUnsupported] = match;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language) + constants.NOT_SUPPORTED_SUFFIX)) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
    })();
    for (const entry of Array.from(perLanguage.entries())) {
        await aws_embedded_metrics_1.metricScope((metrics) => (language, data) => {
            console.log('');
            console.log('##################################################');
            console.log(`### Start of data for ${language}`);
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of ["Supported" /* SUPPORTED */, "Unsupported" /* UNSUPPORTED */, "Missing" /* MISSING */]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.length} entries`);
                    if (forStatus === "Missing" /* MISSING */) {
                        // List out selected packages for posterity (and troubleshooting)
                        for (const [name] of filtered) {
                            // Rendering them one-by-one to avoid saturating the CloudWatch logs buffer...
                            console.log(`  -> ${name}`);
                        }
                    }
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
            console.log(`### End of data for ${language}`);
            console.log('##################################################');
            console.log('');
        })(...entry);
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder);
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
                // If we already have a status, only override with "UNSUPPORTED" if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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