"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    console.log('Loading existing catalog (if present)...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (data.Body) {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
    }
    // If defined, the function will invoke itself again to resume the work from that key (this
    // happens only in "from scratch" or "rebuild" cases).
    let nextStartAfter;
    if (event.package) {
        if (!event.package.key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            throw new Error(`The provided package key is invalid: ${event.package.key} does not end in ${constants.PACKAGE_KEY_SUFFIX}`);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // If we don't have a package in event, then we're refreshing the catalog. This is also true if we
    // don't have a catalog body (from scratch) or if "startAfter" is set (continuation of from
    // scratch).
    if (!(event === null || event === void 0 ? void 0 : event.package) || !data.Body || event.startAfter) {
        console.log('Recreating or refreshing catalog...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME, event.startAfter)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
            // If we're getting short on time (1 minute out of 15 left), we'll be continuing in a new
            // invocation after writing what we've done so far to S3...
            if (context.getRemainingTimeInMillis() <= 60000) {
                nextStartAfter = pkgKey;
                break;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Clean up existing entries if necessary. In particular, remove the license texts as they make
    // the catalog unnecessarily large, and may hinder some search queries' result quality.
    for (const entry of catalog.packages) {
        if (entry.metadata) {
            delete entry.metadata.licenseText;
        }
    }
    // Upload the result to S3 and exit.
    const result = await aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
    if (nextStartAfter != null) {
        console.log(`Will continue from ${nextStartAfter} in new invocation...`);
        const nextEvent = { ...event, startAfter: nextStartAfter };
        // We start it asynchronously, as this function has a provisionned
        // concurrency of 1 (so a synchronous attempt would always be throttled).
        await aws.lambda().invokeAsync({
            FunctionName: context.functionName,
            InvokeArgs: JSON.stringify(nextEvent, null, 2),
        }).promise();
    }
    return result;
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 *
 * @param bucket the bucket in which to list objects
 * @param startAfter the key to start reading from, if provided.
 */
async function* relevantObjects(bucket, startAfter) {
    var _a, _b;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
        StartAfter: startAfter,
    };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST docs in one language.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            const javaDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_JAVA}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) &&
                !(await aws.s3ObjectExists(bucket, pyDocs)) &&
                !(await aws.s3ObjectExists(bucket, javaDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    // If the version is === to the current latest, we'll be replacing that (so re-generated metadata are taken into account)
    if (found != null && version.compare(found.version) < 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2F0YWxvZy1idWlsZGVyLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2NhdGFsb2ctYnVpbGRlci9jYXRhbG9nLWJ1aWxkZXIubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtCQUE4QjtBQUU5QiwrREFBeUQ7QUFHekQsbUNBQWdDO0FBQ2hDLDJDQUFxQztBQUVyQyw0RUFBbUU7QUFFbkUsbURBQW1EO0FBQ25ELGlEQUFpRDtBQUNqRCxtRUFBeUQ7QUFFekQsTUFBTSxpQkFBaUIsR0FBRyw2QkFBNkIsQ0FBQztBQUV4RDs7Ozs7Ozs7O0dBU0c7QUFDSSxLQUFLLFVBQVUsT0FBTyxDQUFDLEtBQTBCLEVBQUUsT0FBZ0I7SUFDeEUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUU1QyxNQUFNLFdBQVcsR0FBRyw4QkFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRTlDLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxFQUFvQyxDQUFDO0lBQzdELE1BQU0sUUFBUSxHQUFHLE1BQU0scUNBQWMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUVsRCxPQUFPLENBQUMsR0FBRyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7SUFFeEQsTUFBTSxJQUFJLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsU0FBUyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsT0FBTyxFQUFFO1NBQ2pHLEtBQUssQ0FBQyxDQUFDLEdBQWEsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyxXQUFXO1FBQ2hELENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQztRQUNyQixDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLGFBQWEsQ0FBd0IsQ0FBQyxDQUFDLENBQUM7SUFFaEUsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO1FBQ2IsT0FBTyxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3pDLE1BQU0sT0FBTyxHQUFpQixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDdEUsS0FBSyxNQUFNLElBQUksSUFBSSxPQUFPLENBQUMsUUFBUSxFQUFFO1lBQ25DLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtnQkFDNUIsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksR0FBRyxFQUFFLENBQUMsQ0FBQzthQUNwQztZQUNELFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQ2hEO0tBQ0Y7SUFFRCwyRkFBMkY7SUFDM0Ysc0RBQXNEO0lBQ3RELElBQUksY0FBa0MsQ0FBQztJQUV2QyxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUU7UUFDakIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsa0JBQWtCLENBQUMsRUFBRTtZQUM3RCxNQUFNLElBQUksS0FBSyxDQUFDLHdDQUF3QyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsb0JBQW9CLFNBQVMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUM7U0FDOUg7UUFFRCxPQUFPLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDM0Msa0VBQWtFO1FBQ2xFLDZDQUE2QztRQUM3QyxNQUFNLGFBQWEsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsV0FBVyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0tBQ3pFO0lBRUQsa0dBQWtHO0lBQ2xHLDJGQUEyRjtJQUMzRixZQUFZO0lBQ1osSUFBSSxFQUFDLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxPQUFPLENBQUEsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtRQUVyRCxPQUFPLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDbkQsTUFBTSxRQUFRLEdBQVEsRUFBRSxDQUFDO1FBQ3pCLElBQUksS0FBSyxFQUFFLE1BQU0sRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLElBQUksZUFBZSxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDbEYsSUFBSTtnQkFDRixNQUFNLGFBQWEsQ0FBQyxRQUFRLEVBQUUsTUFBTyxFQUFFLFdBQVcsRUFBRSxRQUFRLENBQUMsQ0FBQzthQUMvRDtZQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUNWLFFBQVEsQ0FBQyxNQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7YUFDdkI7WUFDRCx5RkFBeUY7WUFDekYsMkRBQTJEO1lBQzNELElBQUksT0FBTyxDQUFDLHdCQUF3QixFQUFFLElBQUksS0FBTSxFQUFFO2dCQUNoRCxjQUFjLEdBQUcsTUFBTSxDQUFDO2dCQUN4QixNQUFNO2FBQ1A7U0FDRjtRQUNELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ25ELE9BQU8sQ0FBQyxHQUFHLENBQUMscUJBQXFCLEdBQUcsS0FBSyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1NBQ25EO1FBRUQsTUFBTSxrQ0FBVyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxLQUFLLElBQUksRUFBRTtZQUN4QyxPQUFPLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLENBQUM7WUFDeEMsTUFBTSxXQUFXLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxNQUFNLENBQUM7WUFDakQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxXQUFXLFdBQVcsa0JBQWtCLENBQUMsQ0FBQztZQUN0RCxPQUFPLENBQUMsU0FBUyxDQUFDLDRCQUE0QixFQUFFLFdBQVcsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNFLENBQUMsQ0FBQyxFQUFFLENBQUM7S0FDTjtJQUVELGtDQUFrQztJQUNsQyxPQUFPLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDeEMsTUFBTSxPQUFPLEdBQWlCLEVBQUUsUUFBUSxFQUFFLElBQUksS0FBSyxFQUFlLEVBQUUsT0FBTyxFQUFFLElBQUksSUFBSSxFQUFFLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztJQUN4RyxLQUFLLE1BQU0sTUFBTSxJQUFJLFFBQVEsQ0FBQyxNQUFNLEVBQUUsRUFBRTtRQUN0QyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNqQyxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUM1QjtLQUNGO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLG9DQUFvQyxDQUFDLENBQUM7SUFDMUYsTUFBTSxrQ0FBVyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxLQUFLLElBQUksRUFBRTtRQUN4QyxPQUFPLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDeEMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxnQ0FBZ0MsRUFBRSxPQUFPLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNGLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFHTCwrRkFBK0Y7SUFDL0YsdUZBQXVGO0lBQ3ZGLEtBQUssTUFBTSxLQUFLLElBQUksT0FBTyxDQUFDLFFBQVEsRUFBRTtRQUNwQyxJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDbEIsT0FBUSxLQUFLLENBQUMsUUFBZ0IsQ0FBQyxXQUFXLENBQUM7U0FDNUM7S0FDRjtJQUVELG9DQUFvQztJQUNwQyxNQUFNLE1BQU0sR0FBRyxNQUFNLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUM7UUFDdEMsTUFBTSxFQUFFLFdBQVc7UUFDbkIsR0FBRyxFQUFFLFNBQVMsQ0FBQyxXQUFXO1FBQzFCLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3RDLFdBQVcsRUFBRSxrQkFBa0I7UUFDL0IsWUFBWSxFQUFFLHFCQUFxQjtRQUNuQyxRQUFRLEVBQUU7WUFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtZQUN4QyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsYUFBYTtZQUMxQyxlQUFlLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDckMsZUFBZSxFQUFFLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUU7U0FDOUM7S0FDRixDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7SUFFYixJQUFJLGNBQWMsSUFBSSxJQUFJLEVBQUU7UUFDMUIsT0FBTyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsY0FBYyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ3pFLE1BQU0sU0FBUyxHQUF3QixFQUFFLEdBQUcsS0FBSyxFQUFFLFVBQVUsRUFBRSxjQUFjLEVBQUUsQ0FBQztRQUNoRixrRUFBa0U7UUFDbEUseUVBQXlFO1FBQ3pFLE1BQU0sR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDLFdBQVcsQ0FBQztZQUM3QixZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDbEMsVUFBVSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7U0FDL0MsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0tBQ2Q7SUFFRCxPQUFPLE1BQU0sQ0FBQztBQUNoQixDQUFDO0FBNUhELDBCQTRIQztBQUVEOzs7Ozs7O0dBT0c7QUFDSCxLQUFLLFNBQVMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxNQUFjLEVBQUUsVUFBbUI7O0lBQ2pFLE1BQU0sT0FBTyxHQUE0QjtRQUN2QyxNQUFNLEVBQUUsTUFBTTtRQUNkLE1BQU0sRUFBRSxTQUFTLENBQUMsa0JBQWtCO1FBQ3BDLFVBQVUsRUFBRSxVQUFVO0tBQ3ZCLENBQUM7SUFDRixHQUFHO1FBQ0QsTUFBTSxNQUFNLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQy9ELEtBQUssTUFBTSxNQUFNLFVBQUksTUFBTSxDQUFDLFFBQVEsbUNBQUksRUFBRSxFQUFFO1lBQzFDLElBQUksUUFBQyxNQUFNLENBQUMsR0FBRywwQ0FBRSxRQUFRLENBQUMsU0FBUyxDQUFDLGtCQUFrQixFQUFDLEVBQUU7Z0JBQ3ZELFNBQVM7YUFDVjtZQUNELHdFQUF3RTtZQUN4RSxNQUFNLE1BQU0sR0FBRyxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxNQUFNLENBQUMsR0FBRyxDQUFDLE1BQU0sR0FBRyxTQUFTLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDLEdBQUcsU0FBUyxDQUFDLDBCQUEwQixFQUFFLENBQUM7WUFDNUksTUFBTSxNQUFNLEdBQUcsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxNQUFNLEdBQUcsU0FBUyxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxHQUFHLFNBQVMsQ0FBQyxzQkFBc0IsRUFBRSxDQUFDO1lBQ3hJLE1BQU0sUUFBUSxHQUFHLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxHQUFHLENBQUMsTUFBTSxHQUFHLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsR0FBRyxTQUFTLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztZQUN4SSxJQUFJLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUMzQyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQztnQkFDM0MsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUMsRUFBRTtnQkFDakQsU0FBUzthQUNWO1lBQ0QsTUFBTSxNQUFNLENBQUM7U0FDZDtRQUNELE9BQU8sQ0FBQyxpQkFBaUIsR0FBRyxNQUFNLENBQUMscUJBQXFCLENBQUM7S0FDMUQsUUFBUSxPQUFPLENBQUMsaUJBQWlCLElBQUksSUFBSSxFQUFFO0FBQzlDLENBQUM7QUFFRCxLQUFLLFVBQVUsYUFBYSxDQUFDLFFBQWEsRUFBRSxNQUFjLEVBQUUsVUFBa0IsRUFBRSxRQUF3Qjs7SUFDdEcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsTUFBTSxFQUFFLENBQUMsQ0FBQztJQUN6QyxNQUFNLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLEdBQUcsU0FBUyxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUUsQ0FBQztJQUNyRixNQUFNLE9BQU8sR0FBRyxJQUFJLGVBQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN2QyxNQUFNLEtBQUssU0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQywwQ0FBRSxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzVELHlIQUF5SDtJQUN6SCxJQUFJLEtBQUssSUFBSSxJQUFJLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3ZELE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWSxXQUFXLElBQUksT0FBTyw4Q0FBOEMsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDN0csT0FBTztLQUNSO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxlQUFlLFdBQVcsSUFBSSxPQUFPLENBQUMsT0FBTywyQkFBMkIsQ0FBQyxDQUFDO0lBQ3RGLE1BQU0sT0FBTyxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUM5RCxJQUFJLE9BQU8sRUFBRTtRQUNYLE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWSxXQUFXLElBQUksT0FBTyxDQUFDLE9BQU8saURBQWlELElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ2xJLE9BQU87S0FDUjtJQUVELE9BQU8sQ0FBQyxHQUFHLENBQUMsZUFBZSxXQUFXLElBQUksT0FBTyxFQUFFLENBQUMsQ0FBQztJQUVyRCxtRUFBbUU7SUFDbkUsTUFBTSxHQUFHLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNwRixNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsRUFBRSxTQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUNoRyxNQUFNLGdCQUFnQixHQUFHLE1BQU0sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUUsR0FBRyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7SUFDdEcsTUFBTSxRQUFRLEdBQUcsTUFBTSxJQUFJLE9BQU8sQ0FBUyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRTtRQUNwRCxhQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEVBQUU7WUFDMUMsSUFBSSxHQUFHLEVBQUU7Z0JBQ1AsT0FBTyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7YUFDaEI7WUFDRCxvQkFBTyxFQUFFO2lCQUNOLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxNQUFNLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxFQUFFO2dCQUNwQyxJQUFJLE1BQU0sQ0FBQyxJQUFJLEtBQUssc0JBQXNCLEVBQUU7b0JBQzFDLG9FQUFvRTtvQkFDcEUsT0FBTyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQzNCO2dCQUNELE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7Z0JBQ25DLE9BQU8sTUFBTTtxQkFDVixFQUFFLENBQUMsTUFBTSxFQUFFLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztxQkFDdEQsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUU7b0JBQ2hCLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7b0JBQzFCLElBQUksRUFBRSxDQUFDO2dCQUNULENBQUMsQ0FBQztxQkFDRCxNQUFNLEVBQUUsQ0FBQztZQUNkLENBQUMsQ0FBQztpQkFDRCxJQUFJLENBQUMsUUFBUSxFQUFFLEdBQUcsRUFBRTtnQkFDbkIsRUFBRSxDQUFDLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLENBQUMsQ0FBQztZQUNuRSxDQUFDLENBQUM7aUJBQ0QsS0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLFFBQVEsRUFBRSxFQUFFO2dCQUN2QixJQUFJLFFBQVEsRUFBRTtvQkFDWixFQUFFLENBQUMsUUFBUSxDQUFDLENBQUM7aUJBQ2Q7WUFDSCxDQUFDLENBQUMsQ0FBQztRQUNQLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDRCwyQ0FBMkM7SUFDN0MsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDM0QsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssYUFBQyxnQkFBZ0IsYUFBaEIsZ0JBQWdCLHVCQUFoQixnQkFBZ0IsQ0FBRSxJQUFJLDBDQUFFLFFBQVEsQ0FBQyxPQUFPLG9DQUFLLElBQUksQ0FBQyxDQUFDO0lBQ2xGLE1BQU0sS0FBSyxHQUFHLElBQUksZUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLENBQUM7SUFDcEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ25DLFFBQVEsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxJQUFJLEdBQUcsRUFBRSxDQUFDLENBQUM7S0FDM0M7SUFDRCxRQUFRLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUUsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFO1FBQ3pDLE1BQU0sRUFBRSxXQUFXLENBQUMsTUFBTTtRQUMxQixXQUFXLEVBQUUsV0FBVyxDQUFDLFdBQVc7UUFDcEMsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFRO1FBQzlCLFNBQVMsRUFBRSxXQUFXLENBQUMsSUFBSSxDQUFDLE9BQU87UUFDbkMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxPQUFPO1FBQzVCLEtBQUs7UUFDTCxRQUFRLEVBQUUsV0FBVztRQUNyQixJQUFJLEVBQUUsV0FBVyxDQUFDLElBQUk7UUFDdEIsT0FBTyxFQUFFLFdBQVcsQ0FBQyxPQUFPO0tBQzdCLENBQUMsQ0FBQztBQUVMLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBndW56aXAgfSBmcm9tICd6bGliJztcblxuaW1wb3J0IHsgbWV0cmljU2NvcGUsIFVuaXQgfSBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcyc7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQgfSBmcm9tICdhd3MtbGFtYmRhJztcbmltcG9ydCB7IEFXU0Vycm9yLCBTMyB9IGZyb20gJ2F3cy1zZGsnO1xuaW1wb3J0IHsgU2VtVmVyIH0gZnJvbSAnc2VtdmVyJztcbmltcG9ydCB7IGV4dHJhY3QgfSBmcm9tICd0YXItc3RyZWFtJztcbmltcG9ydCB7IENhdGFsb2dNb2RlbCwgUGFja2FnZUluZm8gfSBmcm9tICcuJztcbmltcG9ydCB7IERlbnlMaXN0Q2xpZW50IH0gZnJvbSAnLi4vZGVueS1saXN0L2NsaWVudC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB0eXBlIHsgQ2F0YWxvZ0J1aWxkZXJJbnB1dCB9IGZyb20gJy4uL3BheWxvYWQtc2NoZW1hJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICcuLi9zaGFyZWQvYXdzLmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0ICogYXMgY29uc3RhbnRzIGZyb20gJy4uL3NoYXJlZC9jb25zdGFudHMnO1xuaW1wb3J0IHsgcmVxdWlyZUVudiB9IGZyb20gJy4uL3NoYXJlZC9lbnYubGFtYmRhLXNoYXJlZCc7XG5cbmNvbnN0IE1FVFJJQ1NfTkFNRVNQQUNFID0gJ0NvbnN0cnVjdEh1Yi9DYXRhbG9nQnVpbGRlcic7XG5cbi8qKlxuICogUmVnZW5lcmF0ZXMgdGhlIGBjYXRhbG9nLmpzb25gIG9iamVjdCBpbiB0aGUgY29uZmlndXJlZCBTMyBidWNrZXQuXG4gKlxuICogQHBhcmFtIGV2ZW50IGNvbmZpZ3VyYXRpb24gZm9yIHRoZSByZWJ1aWxkIGpvYi4gSW4gcGFydGljdWxhciwgdGhlIGByZWJ1aWxkYFxuICogICAgICAgICAgICAgIHByb3BlcnR5IGNhbiBiZSBzZXQgdG8gYHRydWVgIGluIG9yZGVyIHRvIHRyaWdnZXIgYSBmdWxsIChpLmU6XG4gKiAgICAgICAgICAgICAgbm9uLWluY3JlbWVudGFsKSByZWJ1aWxkIG9mIHRoZSBvYmplY3QuXG4gKiBAcGFyYW0gY29udGV4dCB0aGUgbGFtYmRhIGNvbnRleHQgaW4gd2hpY2ggdGhpcyBleGVjdXRpb24gcnVucy5cbiAqXG4gKiBAcmV0dXJucyB0aGUgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHVwZGF0ZWQgUzMgb2JqZWN0LlxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gaGFuZGxlcihldmVudDogQ2F0YWxvZ0J1aWxkZXJJbnB1dCwgY29udGV4dDogQ29udGV4dCkge1xuICBjb25zb2xlLmxvZyhKU09OLnN0cmluZ2lmeShldmVudCwgbnVsbCwgMikpO1xuXG4gIGNvbnN0IEJVQ0tFVF9OQU1FID0gcmVxdWlyZUVudignQlVDS0VUX05BTUUnKTtcblxuICBjb25zdCBwYWNrYWdlcyA9IG5ldyBNYXA8c3RyaW5nLCBNYXA8bnVtYmVyLCBQYWNrYWdlSW5mbz4+KCk7XG4gIGNvbnN0IGRlbnlMaXN0ID0gYXdhaXQgRGVueUxpc3RDbGllbnQubmV3Q2xpZW50KCk7XG5cbiAgY29uc29sZS5sb2coJ0xvYWRpbmcgZXhpc3RpbmcgY2F0YWxvZyAoaWYgcHJlc2VudCkuLi4nKTtcblxuICBjb25zdCBkYXRhID0gYXdhaXQgYXdzLnMzKCkuZ2V0T2JqZWN0KHsgQnVja2V0OiBCVUNLRVRfTkFNRSwgS2V5OiBjb25zdGFudHMuQ0FUQUxPR19LRVkgfSkucHJvbWlzZSgpXG4gICAgLmNhdGNoKChlcnI6IEFXU0Vycm9yKSA9PiBlcnIuY29kZSAhPT0gJ05vU3VjaEtleSdcbiAgICAgID8gUHJvbWlzZS5yZWplY3QoZXJyKVxuICAgICAgOiBQcm9taXNlLnJlc29sdmUoeyAvKiBubyBkYXRhICovIH0gYXMgUzMuR2V0T2JqZWN0T3V0cHV0KSk7XG5cbiAgaWYgKGRhdGEuQm9keSkge1xuICAgIGNvbnNvbGUubG9nKCdDYXRhbG9nIGZvdW5kLiBMb2FkaW5nLi4uJyk7XG4gICAgY29uc3QgY2F0YWxvZzogQ2F0YWxvZ01vZGVsID0gSlNPTi5wYXJzZShkYXRhLkJvZHkudG9TdHJpbmcoJ3V0Zi04JykpO1xuICAgIGZvciAoY29uc3QgaW5mbyBvZiBjYXRhbG9nLnBhY2thZ2VzKSB7XG4gICAgICBpZiAoIXBhY2thZ2VzLmhhcyhpbmZvLm5hbWUpKSB7XG4gICAgICAgIHBhY2thZ2VzLnNldChpbmZvLm5hbWUsIG5ldyBNYXAoKSk7XG4gICAgICB9XG4gICAgICBwYWNrYWdlcy5nZXQoaW5mby5uYW1lKSEuc2V0KGluZm8ubWFqb3IsIGluZm8pO1xuICAgIH1cbiAgfVxuXG4gIC8vIElmIGRlZmluZWQsIHRoZSBmdW5jdGlvbiB3aWxsIGludm9rZSBpdHNlbGYgYWdhaW4gdG8gcmVzdW1lIHRoZSB3b3JrIGZyb20gdGhhdCBrZXkgKHRoaXNcbiAgLy8gaGFwcGVucyBvbmx5IGluIFwiZnJvbSBzY3JhdGNoXCIgb3IgXCJyZWJ1aWxkXCIgY2FzZXMpLlxuICBsZXQgbmV4dFN0YXJ0QWZ0ZXI6IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICBpZiAoZXZlbnQucGFja2FnZSkge1xuICAgIGlmICghZXZlbnQucGFja2FnZS5rZXkuZW5kc1dpdGgoY29uc3RhbnRzLlBBQ0tBR0VfS0VZX1NVRkZJWCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVGhlIHByb3ZpZGVkIHBhY2thZ2Uga2V5IGlzIGludmFsaWQ6ICR7ZXZlbnQucGFja2FnZS5rZXl9IGRvZXMgbm90IGVuZCBpbiAke2NvbnN0YW50cy5QQUNLQUdFX0tFWV9TVUZGSVh9YCk7XG4gICAgfVxuXG4gICAgY29uc29sZS5sb2coJ1JlZ2lzdGVyaW5nIG5ldyBwYWNrYWdlcy4uLicpO1xuICAgIC8vIG5vdGUgdGhhdCB3ZSBpbnRlbnRpb25hbGx5IGRvbid0IGNhdGNoIGVycm9ycyBoZXJlIHRvIGxldCB0aGVzZVxuICAgIC8vIGV2ZW50IGdvIHRvIHRoZSBETFEgZm9yIG1hbnVhbCBpbnNwZWN0aW9uLlxuICAgIGF3YWl0IGFwcGVuZFBhY2thZ2UocGFja2FnZXMsIGV2ZW50LnBhY2thZ2Uua2V5LCBCVUNLRVRfTkFNRSwgZGVueUxpc3QpO1xuICB9XG5cbiAgLy8gSWYgd2UgZG9uJ3QgaGF2ZSBhIHBhY2thZ2UgaW4gZXZlbnQsIHRoZW4gd2UncmUgcmVmcmVzaGluZyB0aGUgY2F0YWxvZy4gVGhpcyBpcyBhbHNvIHRydWUgaWYgd2VcbiAgLy8gZG9uJ3QgaGF2ZSBhIGNhdGFsb2cgYm9keSAoZnJvbSBzY3JhdGNoKSBvciBpZiBcInN0YXJ0QWZ0ZXJcIiBpcyBzZXQgKGNvbnRpbnVhdGlvbiBvZiBmcm9tXG4gIC8vIHNjcmF0Y2gpLlxuICBpZiAoIWV2ZW50Py5wYWNrYWdlIHx8ICFkYXRhLkJvZHkgfHwgZXZlbnQuc3RhcnRBZnRlcikge1xuXG4gICAgY29uc29sZS5sb2coJ1JlY3JlYXRpbmcgb3IgcmVmcmVzaGluZyBjYXRhbG9nLi4uJyk7XG4gICAgY29uc3QgZmFpbHVyZXM6IGFueSA9IHt9O1xuICAgIGZvciBhd2FpdCAoY29uc3QgeyBLZXk6IHBrZ0tleSB9IG9mIHJlbGV2YW50T2JqZWN0cyhCVUNLRVRfTkFNRSwgZXZlbnQuc3RhcnRBZnRlcikpIHtcbiAgICAgIHRyeSB7XG4gICAgICAgIGF3YWl0IGFwcGVuZFBhY2thZ2UocGFja2FnZXMsIHBrZ0tleSEsIEJVQ0tFVF9OQU1FLCBkZW55TGlzdCk7XG4gICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgIGZhaWx1cmVzW3BrZ0tleSFdID0gZTtcbiAgICAgIH1cbiAgICAgIC8vIElmIHdlJ3JlIGdldHRpbmcgc2hvcnQgb24gdGltZSAoMSBtaW51dGUgb3V0IG9mIDE1IGxlZnQpLCB3ZSdsbCBiZSBjb250aW51aW5nIGluIGEgbmV3XG4gICAgICAvLyBpbnZvY2F0aW9uIGFmdGVyIHdyaXRpbmcgd2hhdCB3ZSd2ZSBkb25lIHNvIGZhciB0byBTMy4uLlxuICAgICAgaWYgKGNvbnRleHQuZ2V0UmVtYWluaW5nVGltZUluTWlsbGlzKCkgPD0gNjBfMDAwKSB7XG4gICAgICAgIG5leHRTdGFydEFmdGVyID0gcGtnS2V5O1xuICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG4gICAgZm9yIChjb25zdCBba2V5LCBlcnJvcl0gb2YgT2JqZWN0LmVudHJpZXMoZmFpbHVyZXMpKSB7XG4gICAgICBjb25zb2xlLmxvZyhgRmFpbGVkIHByb2Nlc3NpbmcgJHtrZXl9OiAke2Vycm9yfWApO1xuICAgIH1cblxuICAgIGF3YWl0IG1ldHJpY1Njb3BlKChtZXRyaWNzKSA9PiBhc3luYyAoKSA9PiB7XG4gICAgICBtZXRyaWNzLnNldE5hbWVzcGFjZShNRVRSSUNTX05BTUVTUEFDRSk7XG4gICAgICBjb25zdCBmYWlsZWRDb3VudCA9IE9iamVjdC5rZXlzKGZhaWx1cmVzKS5sZW5ndGg7XG4gICAgICBjb25zb2xlLmxvZyhgTWFya2luZyAke2ZhaWxlZENvdW50fSBmYWlsZWQgcGFja2FnZXNgKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKCdGYWlsZWRQYWNrYWdlc09uUmVjcmVhdGlvbicsIGZhaWxlZENvdW50LCBVbml0LkNvdW50KTtcbiAgICB9KSgpO1xuICB9XG5cbiAgLy8gQnVpbGQgdGhlIGZpbmFsIGRhdGEgcGFja2FnZS4uLlxuICBjb25zb2xlLmxvZygnQ29uc29saWRhdGluZyBjYXRhbG9nLi4uJyk7XG4gIGNvbnN0IGNhdGFsb2c6IENhdGFsb2dNb2RlbCA9IHsgcGFja2FnZXM6IG5ldyBBcnJheTxQYWNrYWdlSW5mbz4oKSwgdXBkYXRlZDogbmV3IERhdGUoKS50b0lTT1N0cmluZygpIH07XG4gIGZvciAoY29uc3QgbWFqb3JzIG9mIHBhY2thZ2VzLnZhbHVlcygpKSB7XG4gICAgZm9yIChjb25zdCBwa2cgb2YgbWFqb3JzLnZhbHVlcygpKSB7XG4gICAgICBjYXRhbG9nLnBhY2thZ2VzLnB1c2gocGtnKTtcbiAgICB9XG4gIH1cblxuICBjb25zb2xlLmxvZyhgVGhlcmUgYXJlIG5vdyAke2NhdGFsb2cucGFja2FnZXMubGVuZ3RofSByZWdpc3RlcmVkIHBhY2thZ2UgbWFqb3IgdmVyc2lvbnNgKTtcbiAgYXdhaXQgbWV0cmljU2NvcGUoKG1ldHJpY3MpID0+IGFzeW5jICgpID0+IHtcbiAgICBtZXRyaWNzLnNldE5hbWVzcGFjZShNRVRSSUNTX05BTUVTUEFDRSk7XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoJ1JlZ2lzdGVyZWRQYWNrYWdlc01ham9yVmVyc2lvbicsIGNhdGFsb2cucGFja2FnZXMubGVuZ3RoLCBVbml0LkNvdW50KTtcbiAgfSkoKTtcblxuXG4gIC8vIENsZWFuIHVwIGV4aXN0aW5nIGVudHJpZXMgaWYgbmVjZXNzYXJ5LiBJbiBwYXJ0aWN1bGFyLCByZW1vdmUgdGhlIGxpY2Vuc2UgdGV4dHMgYXMgdGhleSBtYWtlXG4gIC8vIHRoZSBjYXRhbG9nIHVubmVjZXNzYXJpbHkgbGFyZ2UsIGFuZCBtYXkgaGluZGVyIHNvbWUgc2VhcmNoIHF1ZXJpZXMnIHJlc3VsdCBxdWFsaXR5LlxuICBmb3IgKGNvbnN0IGVudHJ5IG9mIGNhdGFsb2cucGFja2FnZXMpIHtcbiAgICBpZiAoZW50cnkubWV0YWRhdGEpIHtcbiAgICAgIGRlbGV0ZSAoZW50cnkubWV0YWRhdGEgYXMgYW55KS5saWNlbnNlVGV4dDtcbiAgICB9XG4gIH1cblxuICAvLyBVcGxvYWQgdGhlIHJlc3VsdCB0byBTMyBhbmQgZXhpdC5cbiAgY29uc3QgcmVzdWx0ID0gYXdhaXQgYXdzLnMzKCkucHV0T2JqZWN0KHtcbiAgICBCdWNrZXQ6IEJVQ0tFVF9OQU1FLFxuICAgIEtleTogY29uc3RhbnRzLkNBVEFMT0dfS0VZLFxuICAgIEJvZHk6IEpTT04uc3RyaW5naWZ5KGNhdGFsb2csIG51bGwsIDIpLFxuICAgIENvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vanNvbicsXG4gICAgQ2FjaGVDb250cm9sOiAncHVibGljLCBtYXgtYWdlPTMwMCcsIC8vIEV4cGlyZSBmcm9tIGNhY2hlIGFmdGVyIDUgbWludXRlc1xuICAgIE1ldGFkYXRhOiB7XG4gICAgICAnTGFtYmRhLUxvZy1Hcm91cCc6IGNvbnRleHQubG9nR3JvdXBOYW1lLFxuICAgICAgJ0xhbWJkYS1Mb2ctU3RyZWFtJzogY29udGV4dC5sb2dTdHJlYW1OYW1lLFxuICAgICAgJ0xhbWJkYS1SdW4tSWQnOiBjb250ZXh0LmF3c1JlcXVlc3RJZCxcbiAgICAgICdQYWNrYWdlLUNvdW50JzogYCR7Y2F0YWxvZy5wYWNrYWdlcy5sZW5ndGh9YCxcbiAgICB9LFxuICB9KS5wcm9taXNlKCk7XG5cbiAgaWYgKG5leHRTdGFydEFmdGVyICE9IG51bGwpIHtcbiAgICBjb25zb2xlLmxvZyhgV2lsbCBjb250aW51ZSBmcm9tICR7bmV4dFN0YXJ0QWZ0ZXJ9IGluIG5ldyBpbnZvY2F0aW9uLi4uYCk7XG4gICAgY29uc3QgbmV4dEV2ZW50OiBDYXRhbG9nQnVpbGRlcklucHV0ID0geyAuLi5ldmVudCwgc3RhcnRBZnRlcjogbmV4dFN0YXJ0QWZ0ZXIgfTtcbiAgICAvLyBXZSBzdGFydCBpdCBhc3luY2hyb25vdXNseSwgYXMgdGhpcyBmdW5jdGlvbiBoYXMgYSBwcm92aXNpb25uZWRcbiAgICAvLyBjb25jdXJyZW5jeSBvZiAxIChzbyBhIHN5bmNocm9ub3VzIGF0dGVtcHQgd291bGQgYWx3YXlzIGJlIHRocm90dGxlZCkuXG4gICAgYXdhaXQgYXdzLmxhbWJkYSgpLmludm9rZUFzeW5jKHtcbiAgICAgIEZ1bmN0aW9uTmFtZTogY29udGV4dC5mdW5jdGlvbk5hbWUsXG4gICAgICBJbnZva2VBcmdzOiBKU09OLnN0cmluZ2lmeShuZXh0RXZlbnQsIG51bGwsIDIpLFxuICAgIH0pLnByb21pc2UoKTtcbiAgfVxuXG4gIHJldHVybiByZXN1bHQ7XG59XG5cbi8qKlxuICogQSBnZW5lcmF0b3IgdGhhdCBhc3luY2hyb25vdXNseSB0cmF2ZXJzZXMgdGhlIHNldCBvZiBcImludGVyZXN0aW5nXCIgb2JqZWN0c1xuICogZm91bmQgYnkgbGlzdGluZyB0aGUgY29uZmlndXJlZCBTMyBidWNrZXQuIFRob3NlIG9iamVjdHMgY29ycmVzcG9uZCB0byBhbGxcbiAqIG5wbSBwYWNrYWdlIHRhcmJhbGxzIHByZXNlbnQgdW5kZXIgdGhlIGBwYWNrYWdlcy9gIHByZWZpeCBpbiB0aGUgYnVja2V0LlxuICpcbiAqIEBwYXJhbSBidWNrZXQgdGhlIGJ1Y2tldCBpbiB3aGljaCB0byBsaXN0IG9iamVjdHNcbiAqIEBwYXJhbSBzdGFydEFmdGVyIHRoZSBrZXkgdG8gc3RhcnQgcmVhZGluZyBmcm9tLCBpZiBwcm92aWRlZC5cbiAqL1xuYXN5bmMgZnVuY3Rpb24qIHJlbGV2YW50T2JqZWN0cyhidWNrZXQ6IHN0cmluZywgc3RhcnRBZnRlcj86IHN0cmluZykge1xuICBjb25zdCByZXF1ZXN0OiBTMy5MaXN0T2JqZWN0c1YyUmVxdWVzdCA9IHtcbiAgICBCdWNrZXQ6IGJ1Y2tldCxcbiAgICBQcmVmaXg6IGNvbnN0YW50cy5TVE9SQUdFX0tFWV9QUkVGSVgsXG4gICAgU3RhcnRBZnRlcjogc3RhcnRBZnRlcixcbiAgfTtcbiAgZG8ge1xuICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGF3cy5zMygpLmxpc3RPYmplY3RzVjIocmVxdWVzdCkucHJvbWlzZSgpO1xuICAgIGZvciAoY29uc3Qgb2JqZWN0IG9mIHJlc3VsdC5Db250ZW50cyA/PyBbXSkge1xuICAgICAgaWYgKCFvYmplY3QuS2V5Py5lbmRzV2l0aChjb25zdGFudHMuUEFDS0FHRV9LRVlfU1VGRklYKSkge1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIC8vIFdlIG9ubHkgcmVnaXN0ZXIgcGFja2FnZXMgaWYgdGhleSBoYXZlIEFUIExFQVNUIGRvY3MgaW4gb25lIGxhbmd1YWdlLlxuICAgICAgY29uc3QgdHNEb2NzID0gYCR7b2JqZWN0LktleS5zdWJzdHJpbmcoMCwgb2JqZWN0LktleS5sZW5ndGggLSBjb25zdGFudHMuUEFDS0FHRV9LRVlfU1VGRklYLmxlbmd0aCl9JHtjb25zdGFudHMuRE9DU19LRVlfU1VGRklYX1RZUEVTQ1JJUFR9YDtcbiAgICAgIGNvbnN0IHB5RG9jcyA9IGAke29iamVjdC5LZXkuc3Vic3RyaW5nKDAsIG9iamVjdC5LZXkubGVuZ3RoIC0gY29uc3RhbnRzLlBBQ0tBR0VfS0VZX1NVRkZJWC5sZW5ndGgpfSR7Y29uc3RhbnRzLkRPQ1NfS0VZX1NVRkZJWF9QWVRIT059YDtcbiAgICAgIGNvbnN0IGphdmFEb2NzID0gYCR7b2JqZWN0LktleS5zdWJzdHJpbmcoMCwgb2JqZWN0LktleS5sZW5ndGggLSBjb25zdGFudHMuUEFDS0FHRV9LRVlfU1VGRklYLmxlbmd0aCl9JHtjb25zdGFudHMuRE9DU19LRVlfU1VGRklYX0pBVkF9YDtcbiAgICAgIGlmICghKGF3YWl0IGF3cy5zM09iamVjdEV4aXN0cyhidWNrZXQsIHRzRG9jcykpICYmXG4gICAgICAgICAgIShhd2FpdCBhd3MuczNPYmplY3RFeGlzdHMoYnVja2V0LCBweURvY3MpKSAmJlxuICAgICAgICAgICEoYXdhaXQgYXdzLnMzT2JqZWN0RXhpc3RzKGJ1Y2tldCwgamF2YURvY3MpKSkge1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIHlpZWxkIG9iamVjdDtcbiAgICB9XG4gICAgcmVxdWVzdC5Db250aW51YXRpb25Ub2tlbiA9IHJlc3VsdC5OZXh0Q29udGludWF0aW9uVG9rZW47XG4gIH0gd2hpbGUgKHJlcXVlc3QuQ29udGludWF0aW9uVG9rZW4gIT0gbnVsbCk7XG59XG5cbmFzeW5jIGZ1bmN0aW9uIGFwcGVuZFBhY2thZ2UocGFja2FnZXM6IGFueSwgcGtnS2V5OiBzdHJpbmcsIGJ1Y2tldE5hbWU6IHN0cmluZywgZGVueUxpc3Q6IERlbnlMaXN0Q2xpZW50KSB7XG4gIGNvbnNvbGUubG9nKGBQcm9jZXNzaW5nIGtleTogJHtwa2dLZXl9YCk7XG4gIGNvbnN0IFssIHBhY2thZ2VOYW1lLCB2ZXJzaW9uU3RyXSA9IGNvbnN0YW50cy5TVE9SQUdFX0tFWV9GT1JNQVRfUkVHRVguZXhlYyhwa2dLZXkpITtcbiAgY29uc3QgdmVyc2lvbiA9IG5ldyBTZW1WZXIodmVyc2lvblN0cik7XG4gIGNvbnN0IGZvdW5kID0gcGFja2FnZXMuZ2V0KHBhY2thZ2VOYW1lKT8uZ2V0KHZlcnNpb24ubWFqb3IpO1xuICAvLyBJZiB0aGUgdmVyc2lvbiBpcyA9PT0gdG8gdGhlIGN1cnJlbnQgbGF0ZXN0LCB3ZSdsbCBiZSByZXBsYWNpbmcgdGhhdCAoc28gcmUtZ2VuZXJhdGVkIG1ldGFkYXRhIGFyZSB0YWtlbiBpbnRvIGFjY291bnQpXG4gIGlmIChmb3VuZCAhPSBudWxsICYmIHZlcnNpb24uY29tcGFyZShmb3VuZC52ZXJzaW9uKSA8IDApIHtcbiAgICBjb25zb2xlLmxvZyhgU2tpcHBpbmcgJHtwYWNrYWdlTmFtZX1AJHt2ZXJzaW9ufSBiZWNhdXNlIGl0IGlzIG5vdCBuZXdlciB0aGFuIHRoZSBleGlzdGluZyAke2ZvdW5kLnZlcnNpb259YCk7XG4gICAgcmV0dXJuO1xuICB9XG5cbiAgY29uc29sZS5sb2coYENoZWNraW5nIGlmICR7cGFja2FnZU5hbWV9QCR7dmVyc2lvbi52ZXJzaW9ufSBtYXRjaGVzIGEgZGVueSBsaXN0IHJ1bGVgKTtcbiAgY29uc3QgYmxvY2tlZCA9IGRlbnlMaXN0Lmxvb2t1cChwYWNrYWdlTmFtZSwgdmVyc2lvbi52ZXJzaW9uKTtcbiAgaWYgKGJsb2NrZWQpIHtcbiAgICBjb25zb2xlLmxvZyhgU2tpcHBpbmcgJHtwYWNrYWdlTmFtZX1AJHt2ZXJzaW9uLnZlcnNpb259IGJlY2F1c2UgaXQgaXMgYmxvY2tlZCBieSB0aGUgZGVueSBsaXN0IHJ1bGU6ICR7SlNPTi5zdHJpbmdpZnkoYmxvY2tlZCl9YCk7XG4gICAgcmV0dXJuO1xuICB9XG5cbiAgY29uc29sZS5sb2coYFJlZ2lzdGVyaW5nICR7cGFja2FnZU5hbWV9QCR7dmVyc2lvbn1gKTtcblxuICAvLyBEb253bG9hZCB0aGUgdGFyYmFsbCB0byBpbnNwZWN0IHRoZSBgcGFja2FnZS5qc29uYCBkYXRhIHRoZXJlaW4uXG4gIGNvbnN0IHBrZyA9IGF3YWl0IGF3cy5zMygpLmdldE9iamVjdCh7IEJ1Y2tldDogYnVja2V0TmFtZSwgS2V5OiBwa2dLZXkgfSkucHJvbWlzZSgpO1xuICBjb25zdCBtZXRhZGF0YUtleSA9IHBrZ0tleS5yZXBsYWNlKGNvbnN0YW50cy5QQUNLQUdFX0tFWV9TVUZGSVgsIGNvbnN0YW50cy5NRVRBREFUQV9LRVlfU1VGRklYKTtcbiAgY29uc3QgbWV0YWRhdGFSZXNwb25zZSA9IGF3YWl0IGF3cy5zMygpLmdldE9iamVjdCh7IEJ1Y2tldDogYnVja2V0TmFtZSwgS2V5OiBtZXRhZGF0YUtleSB9KS5wcm9taXNlKCk7XG4gIGNvbnN0IG1hbmlmZXN0ID0gYXdhaXQgbmV3IFByb21pc2U8QnVmZmVyPigob2ssIGtvKSA9PiB7XG4gICAgZ3VuemlwKEJ1ZmZlci5mcm9tKHBrZy5Cb2R5ISksIChlcnIsIHRhcikgPT4ge1xuICAgICAgaWYgKGVycikge1xuICAgICAgICByZXR1cm4ga28oZXJyKTtcbiAgICAgIH1cbiAgICAgIGV4dHJhY3QoKVxuICAgICAgICAub24oJ2VudHJ5JywgKGhlYWRlciwgc3RyZWFtLCBuZXh0KSA9PiB7XG4gICAgICAgICAgaWYgKGhlYWRlci5uYW1lICE9PSAncGFja2FnZS9wYWNrYWdlLmpzb24nKSB7XG4gICAgICAgICAgICAvLyBOb3QgdGhlIGZpbGUgd2UgYXJlIGxvb2tpbmcgZm9yLCBza2lwIGFoZWFkIChuZXh0IHJ1bi1sb29wIHRpY2spLlxuICAgICAgICAgICAgcmV0dXJuIHNldEltbWVkaWF0ZShuZXh0KTtcbiAgICAgICAgICB9XG4gICAgICAgICAgY29uc3QgY2h1bmtzID0gbmV3IEFycmF5PEJ1ZmZlcj4oKTtcbiAgICAgICAgICByZXR1cm4gc3RyZWFtXG4gICAgICAgICAgICAub24oJ2RhdGEnLCAoY2h1bmspID0+IGNodW5rcy5wdXNoKEJ1ZmZlci5mcm9tKGNodW5rKSkpXG4gICAgICAgICAgICAub25jZSgnZW5kJywgKCkgPT4ge1xuICAgICAgICAgICAgICBvayhCdWZmZXIuY29uY2F0KGNodW5rcykpO1xuICAgICAgICAgICAgICBuZXh0KCk7XG4gICAgICAgICAgICB9KVxuICAgICAgICAgICAgLnJlc3VtZSgpO1xuICAgICAgICB9KVxuICAgICAgICAub25jZSgnZmluaXNoJywgKCkgPT4ge1xuICAgICAgICAgIGtvKG5ldyBFcnJvcignQ291bGQgbm90IGZpbmQgcGFja2FnZS9wYWNrYWdlLmpzb24gaW4gdGFyYmFsbCEnKSk7XG4gICAgICAgIH0pXG4gICAgICAgIC53cml0ZSh0YXIsICh3cml0ZUVycikgPT4ge1xuICAgICAgICAgIGlmICh3cml0ZUVycikge1xuICAgICAgICAgICAga28od3JpdGVFcnIpO1xuICAgICAgICAgIH1cbiAgICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuICAgIC8vIEFkZCB0aGUgUGFja2FnZUluZm8gaW50byB0aGUgd29ya2luZyBzZXRcbiAgY29uc3QgcGtnTWV0YWRhdGEgPSBKU09OLnBhcnNlKG1hbmlmZXN0LnRvU3RyaW5nKCd1dGYtOCcpKTtcbiAgY29uc3QgbnBtTWV0YWRhdGEgPSBKU09OLnBhcnNlKG1ldGFkYXRhUmVzcG9uc2U/LkJvZHk/LnRvU3RyaW5nKCd1dGYtOCcpID8/ICd7fScpO1xuICBjb25zdCBtYWpvciA9IG5ldyBTZW1WZXIocGtnTWV0YWRhdGEudmVyc2lvbikubWFqb3I7XG4gIGlmICghcGFja2FnZXMuaGFzKHBrZ01ldGFkYXRhLm5hbWUpKSB7XG4gICAgcGFja2FnZXMuc2V0KHBrZ01ldGFkYXRhLm5hbWUsIG5ldyBNYXAoKSk7XG4gIH1cbiAgcGFja2FnZXMuZ2V0KHBrZ01ldGFkYXRhLm5hbWUpIS5zZXQobWFqb3IsIHtcbiAgICBhdXRob3I6IHBrZ01ldGFkYXRhLmF1dGhvcixcbiAgICBkZXNjcmlwdGlvbjogcGtnTWV0YWRhdGEuZGVzY3JpcHRpb24sXG4gICAga2V5d29yZHM6IHBrZ01ldGFkYXRhLmtleXdvcmRzLFxuICAgIGxhbmd1YWdlczogcGtnTWV0YWRhdGEuanNpaS50YXJnZXRzLFxuICAgIGxpY2Vuc2U6IHBrZ01ldGFkYXRhLmxpY2Vuc2UsXG4gICAgbWFqb3IsXG4gICAgbWV0YWRhdGE6IG5wbU1ldGFkYXRhLFxuICAgIG5hbWU6IHBrZ01ldGFkYXRhLm5hbWUsXG4gICAgdmVyc2lvbjogcGtnTWV0YWRhdGEudmVyc2lvbixcbiAgfSk7XG5cbn1cbiJdfQ==