"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: constants_lambda_shared_1.MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(constants_lambda_shared_1.MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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