"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifact = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const code_artifact_forwarder_1 = require("./codeartifact/code-artifact-forwarder");
const constants_lambda_shared_1 = require("./codeartifact/constants.lambda-shared");
/**
 * (experimental) A package source that obtains package data from an npm CodeArtifact repository.
 *
 * @experimental
 */
class CodeArtifact {
    /**
     * @experimental
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * (experimental) Binds the package source to a scope and target queue.
     *
     * @experimental
     */
    bind(scope, { denyList, ingestion, licenseList, monitoring, queue }) {
        const idPrefix = this.props.repository.node.path;
        const repositoryId = `${this.props.repository.attrDomainOwner}:${this.props.repository.attrDomainName}/${this.props.repository.attrName}`;
        const bucket = this.props.bucket || new aws_s3_1.Bucket(scope, `${idPrefix}/StagingBucket`, {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const dlq = new aws_sqs_1.Queue(scope, `${idPrefix}/DLQ`, {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const forwarder = new code_artifact_forwarder_1.CodeArtifactForwarder(scope, `${idPrefix}/Forwarder`, {
            deadLetterQueue: dlq,
            description: `[${scope.node.path}/CodeArtifact/${repositoryId}] Handle CodeArtifact EventBridge events`,
            environment: {
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.seconds(60),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(forwarder);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(forwarder);
        licenseList.grantRead(forwarder);
        queue.grantSendMessages(forwarder);
        forwarder.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['codeartifact:GetPackageVersionAsset'],
            resources: [
                core_1.Stack.of(scope).formatArn({
                    service: 'codeartifact',
                    resource: 'package',
                    arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                    resourceName: [
                        this.props.repository.attrDomainName,
                        this.props.repository.attrName,
                        'npm',
                        '*',
                    ].join('/'),
                }),
            ],
        }));
        const rule = new aws_events_1.Rule(scope, `${idPrefix}/EventBridge`, {
            description: `${scope.node.path}/CodeArtifact/${repositoryId}/EventBridge`,
            eventPattern: {
                source: ['aws.codeartifact'],
                detailType: ['CodeArtifact Package Version State Change'],
                detail: {
                    domainOwner: this.props.repository.attrDomainOwner,
                    domainName: this.props.repository.attrDomainName,
                    repositoryName: this.props.repository.attrName,
                    packageFormat: 'npm',
                },
            },
            targets: [new aws_events_targets_1.LambdaFunction(forwarder)],
        });
        const failureAlarm = forwarder.metricErrors().createAlarm(scope, `${idPrefix}/Forwarder/Failure`, {
            alarmName: `${scope.node.path}/CodeArtifact/${repositoryId}/Forwarder`,
            alarmDescription: [
                `The CodeArtifact fowarder for ${repositoryId} is failing`,
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(forwarder)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm(`CodeArtifact:${this.props.repository.attrName} Failures`, failureAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: dlq.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: dlq.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${idPrefix}/Forwarder/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/CodeArtifact/${repositoryId}/DLQNotEmpty`,
            alarmDescription: [
                `The CodeArtifact fowarder for ${repositoryId} is failing`,
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(forwarder)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(dlq)}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addHighSeverityAlarm(`CodeArtifact/${repositoryId} DLQ Not Empty`, dlqNotEmptyAlarm);
        rule.node.addDependency(failureAlarm, dlqNotEmptyAlarm);
        return {
            name: `CodeArtifact: ${repositoryId}`,
            links: [{
                    name: 'CodeArtifact',
                    url: deep_link_1.codeArtifactRepositoryUrl(this.props.repository),
                    primary: true,
                }, {
                    name: 'Forwarder Function',
                    url: deep_link_1.lambdaFunctionUrl(forwarder),
                }, {
                    name: 'Forwarder Dead-Letter Queue',
                    url: deep_link_1.sqsQueueUrl(dlq),
                }],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(forwarder.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(forwarder.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            forwarder.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            dlq.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            dlq.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            dlq.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Quality Metrics',
                        left: [
                            metric_utils_1.fillMetric(this.metricNotJsiiEnabledCount({ label: 'Not a jsii package' }), 0),
                            metric_utils_1.fillMetric(this.metricIneligibleLicense({ label: 'Ineligible License' }), 0),
                            metric_utils_1.fillMetric(this.metricDenyListedCount({ label: 'Deny Listed' }), 0),
                            metric_utils_1.fillMetric(this.metricDeletedCount({ label: 'Deletion Events' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
            ],
        };
    }
    /**
     * (experimental) The count of package versions that were ignored due to being in the deny list.
     *
     * @experimental
     */
    metricDenyListedCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: this.props.repository.attrDomainOwner,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "DenyListedCount" /* DENY_LISTED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The number of package versions that were ignored due to using an ineloigible license.
     *
     * @experimental
     */
    metricIneligibleLicense(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "IneligibleLicenseCount" /* INELIGIBLE_LICENSE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The number of package versions that were deleted from CodeArtifact (those events are not handled currently).
     *
     * @experimental
     */
    metricDeletedCount(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "DeletedCount" /* DELETED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The number of package versions that do not have a jsii assembly in the package.
     *
     * @experimental
     */
    metricNotJsiiEnabledCount(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.CodeArtifact = CodeArtifact;
_a = JSII_RTTI_SYMBOL_1;
CodeArtifact[_a] = { fqn: "construct-hub.sources.CodeArtifact", version: "0.1.177" };
//# sourceMappingURL=data:application/json;base64,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