import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { Alarm, Metric, MetricOptions } from '@aws-cdk/aws-cloudwatch';
import { Construct } from '@aws-cdk/core';
export interface MonitoredCertificateProps {
    /**
     * The ACM certificate to be monitored.
     */
    readonly certificate: ICertificate;
    /**
     * The DNS name of an endpoint using the monitored certificate.
     */
    readonly domainName: string;
    /**
     * The namespace of the CloudWatch metric emitted for the amount of days
     * remaining before expiry of the certificate used to serve HTTPS traffic on
     * the configured `domainName`.
     *
     * @default Stack.of(this).stackName
     */
    readonly metricNamespace?: string;
    /**
      * The name of the CloudWatch metric emitted for the amount of days remaining
      * before expiry of the certificate used to serve HTTPS traffic on the
      * configured `domainName`.
      *
      * @default 'DaysToExpiry'
      */
    readonly metricName?: string;
}
/**
 * Monitors an ACM certificate's expiration date. Tracks the days to expiry
 * metric published by ACM (until the certificate has expired), and also
 * inspects the certificate used by the provided `domainName` endpoint for
 * serving HTTPS webpages.
 *
 * These ensure ACM certificates are rotated by the time automated renewal would
 * have happened (60 to 45 days prior to expiration), and that the endpoint is
 * updated to a renewed certificate in due time.
 */
export declare class MonitoredCertificate extends Construct {
    private readonly props;
    /**
     * Alarms when the ACM certificate expiry is less than 45 days from now.
     */
    readonly alarmAcmCertificateExpiresSoon: Alarm;
    /**
     * Alarms when the Endpoint certificate expiry is less than 45 days from now.
     */
    readonly alarmEndpointCertificateExpiresSoon: Alarm;
    private readonly endpointMetricName;
    private readonly endpointMetricNamespace;
    constructor(scope: Construct, id: string, props: MonitoredCertificateProps);
    /**
     * The remaining days before the monitored certificate expires, as far as ACM
     * is concerned. This metric is no longer emitted after the certificate has
     * expired (alarms should treat missing data as `<= 0`).
     */
    metricAcmCertificateDaysToExpiry(opts?: MetricOptions): Metric;
    /**
     * The remaining days before the certificate served by the configured
     * `domainName` expires. This metric is published as 0 if the certificate has
     * already expired.
     */
    metricEndpointCertificateDaysToExpiry(opts?: MetricOptions): Metric;
}
