"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        const metadata = { date: payload.time, licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8') };
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const { license, name, version } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense } = JSON.parse(packageJson.toString('utf-8'));
        if (packageJsonName !== packageName || packageJsonVersion !== packageVersion || packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws.s3().putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        }).promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws.stepFunctions().startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        }).promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return possibleExtensions.has(ext.toLowerCase())
        && path_1.basename(fileName, ext).toUpperCase() === 'LICENSE';
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/ig, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5nZXN0aW9uLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2luZ2VzdGlvbi9pbmdlc3Rpb24ubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1DQUFvQztBQUNwQywrQkFBeUM7QUFDekMsNkJBQTBCO0FBRTFCLHFDQUE4QztBQUM5QywrREFBd0U7QUFDeEUsb0ZBQTZFO0FBRzdFLG1EQUFtRDtBQUNuRCxpREFBaUQ7QUFDakQsbUVBQXlEO0FBRXpELCtFQUE4RDtBQUM5RCwyRUFBaUU7QUFDakUsMkNBQTREO0FBRTVELG9DQUFhLENBQUMsbUJBQW1CLEdBQUcsc0JBQVksQ0FBQyxNQUFNLENBQUM7QUFDeEQsb0NBQWEsQ0FBQyxTQUFTLEdBQUcsNkJBQWlCLENBQUM7QUFFL0IsUUFBQSxPQUFPLEdBQUcsa0NBQVcsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQWUsRUFBRSxPQUFnQixFQUFFLEVBQUU7O0lBQzFGLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRXhELHdEQUF3RDtJQUN4RCxPQUFPLENBQUMsYUFBYSxFQUFFLENBQUM7SUFFeEIsTUFBTSxXQUFXLEdBQUcsOEJBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM5QyxNQUFNLGlCQUFpQixHQUFHLDhCQUFVLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUUxRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO0lBRW5DLEtBQUssTUFBTSxNQUFNLFVBQUksS0FBSyxDQUFDLE9BQU8sbUNBQUksRUFBRSxFQUFFO1FBQ3hDLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBbUIsQ0FBQztRQUUxRCxNQUFNLFVBQVUsR0FBRyxJQUFJLFNBQUcsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDL0MsSUFBSSxVQUFVLENBQUMsUUFBUSxLQUFLLEtBQUssRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLGdDQUFnQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQy9EO1FBQ0QsTUFBTSxPQUFPLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLGlHQUFpRztZQUNqRyxNQUFNLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JDLG9FQUFvRTtZQUNwRSxHQUFHLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztZQUMzQyxTQUFTLFFBQUUsVUFBVSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLG1DQUFJLFNBQVM7U0FDakUsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRWIsTUFBTSxjQUFjLEdBQUcsbUNBQVMsQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSyxDQUFDLENBQUMsQ0FBQztRQUN0RSxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssY0FBYyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLE9BQU8sQ0FBQyxTQUFTLFFBQVEsY0FBYyxFQUFFLENBQUMsQ0FBQztTQUN2RjtRQUVELElBQUksT0FBZSxDQUFDO1FBQ3BCLElBQUksV0FBbUIsQ0FBQztRQUN4QixJQUFJLFdBQStCLENBQUM7UUFDcEMsSUFBSTtZQUNGLENBQUMsRUFBRSxPQUFPLEVBQUUsV0FBVyxFQUFFLFdBQVcsRUFBRSxHQUFHLE1BQU0sc0NBQWMsQ0FDM0QsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSyxDQUFDLEVBQzFCO2dCQUNFLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRTtnQkFDbEQsV0FBVyxFQUFFLEVBQUUsSUFBSSxFQUFFLHNCQUFzQixFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUU7Z0JBQzdELFdBQVcsRUFBRSxFQUFFLE1BQU0sRUFBRSxhQUFhLEVBQUU7YUFDdkMsQ0FDRixDQUFDLENBQUM7U0FDSjtRQUFDLE9BQU8sR0FBRyxFQUFFO1lBQ1osT0FBTyxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUNqRCxPQUFPLENBQUMsU0FBUyx5Q0FBNkIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDN0QsT0FBTztTQUNSO1FBQ0QsTUFBTSxRQUFRLEdBQUcsRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUUsV0FBVyxhQUFYLFdBQVcsdUJBQVgsV0FBVyxDQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBRXJGLElBQUksY0FBc0IsQ0FBQztRQUMzQixJQUFJLFdBQW1CLENBQUM7UUFDeEIsSUFBSSxjQUFzQixDQUFDO1FBQzNCLElBQUk7WUFDRixNQUFNLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsR0FBRyx1QkFBZ0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzNGLGNBQWMsR0FBRyxPQUFPLENBQUM7WUFDekIsV0FBVyxHQUFHLElBQUksQ0FBQztZQUNuQixjQUFjLEdBQUcsT0FBTyxDQUFDO1lBQ3pCLE9BQU8sQ0FBQyxTQUFTLDJDQUE4QixDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMvRDtRQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ1gsT0FBTyxDQUFDLEtBQUssQ0FBQywwREFBMEQsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUM5RSxPQUFPLENBQUMsU0FBUywyQ0FBOEIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDOUQsT0FBTztTQUNSO1FBRUQsb0ZBQW9GO1FBQ3BGLE1BQU0sRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLE9BQU8sRUFBRSxrQkFBa0IsRUFBRSxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN0SSxJQUFJLGVBQWUsS0FBSyxXQUFXLElBQUksa0JBQWtCLEtBQUssY0FBYyxJQUFJLGtCQUFrQixLQUFLLGNBQWMsRUFBRTtZQUNySCxPQUFPLENBQUMsR0FBRyxDQUFDLG1FQUFtRSxlQUFlLElBQUksa0JBQWtCLE9BQU8sa0JBQWtCLFFBQVEsV0FBVyxJQUFJLGNBQWMsT0FBTyxjQUFjLEdBQUcsQ0FBQyxDQUFDO1lBQzVNLE9BQU8sQ0FBQyxTQUFTLHNFQUE0QyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUM1RSxTQUFTO1NBQ1Y7UUFDRCxPQUFPLENBQUMsU0FBUyxzRUFBNEMsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFNUUseUNBQXlDO1FBQ3pDLE9BQU8sQ0FBQyxTQUFTLDhDQUFnQyxXQUFXLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFGLE1BQU0sRUFBRSxXQUFXLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxHQUFHLFNBQVMsQ0FBQyxhQUFhLENBQUMsV0FBVyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3RHLE9BQU8sQ0FBQyxHQUFHLENBQUMsdUJBQXVCLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFDbEQsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUNqRCxPQUFPLENBQUMsR0FBRyxDQUFDLHdCQUF3QixXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBRW5ELDJFQUEyRTtRQUMzRSxtREFBbUQ7UUFDbkQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLFdBQVcsSUFBSSxjQUFjLHlDQUF5QyxDQUFDLENBQUM7UUFDdkYsTUFBTSxDQUFDLEdBQUcsRUFBRSxjQUFjLENBQUMsR0FBRyxNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDOUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztnQkFDakIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEdBQUcsRUFBRSxVQUFVO2dCQUNmLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsWUFBWSxFQUFFLFFBQVE7Z0JBQ3RCLFdBQVcsRUFBRSwwQkFBMEI7Z0JBQ3ZDLFFBQVEsRUFBRTtvQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtvQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7b0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTtpQkFDdEM7YUFDRixDQUFDLENBQUMsT0FBTyxFQUFFO1lBQ1osR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztnQkFDakIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEdBQUcsRUFBRSxXQUFXO2dCQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUM7Z0JBQzlCLFlBQVksRUFBRSxRQUFRO2dCQUN0QixXQUFXLEVBQUUsa0JBQWtCO2dCQUMvQixRQUFRLEVBQUU7b0JBQ1Isa0JBQWtCLEVBQUUsT0FBTyxDQUFDLFlBQVk7b0JBQ3hDLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxhQUFhO29CQUMxQyxlQUFlLEVBQUUsT0FBTyxDQUFDLFlBQVk7aUJBQ3RDO2FBQ0YsQ0FBQyxDQUFDLE9BQU8sRUFBRTtTQUNiLENBQUMsQ0FBQztRQUVILGtDQUFrQztRQUNsQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxJQUFJLGNBQWMsNEJBQTRCLENBQUMsQ0FBQztRQUMxRSxNQUFNLFFBQVEsR0FBRyxNQUFNLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUM7WUFDeEMsTUFBTSxFQUFFLFdBQVc7WUFDbkIsR0FBRyxFQUFFLFdBQVc7WUFDaEIsSUFBSSxFQUFFLE9BQU87WUFDYixZQUFZLEVBQUUsUUFBUTtZQUN0QixXQUFXLEVBQUUsa0JBQWtCO1lBQy9CLFFBQVEsRUFBRTtnQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7Z0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTthQUN0QztTQUNGLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUViLE1BQU0sT0FBTyxHQUFzQjtZQUNqQyxNQUFNLEVBQUUsV0FBVztZQUNuQixRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUzthQUM5QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxHQUFHLEVBQUUsVUFBVTtnQkFDZixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7YUFDekI7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxjQUFjLENBQUMsU0FBUzthQUNwQztTQUNGLENBQUM7UUFDRixPQUFPLENBQUMsR0FBRyxDQUFDLG9CQUFvQixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE1BQU0sR0FBRyxHQUFHLE1BQU0sR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDLGNBQWMsQ0FBQztZQUNuRCxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUM7WUFDOUIsSUFBSSxFQUFFLHlCQUF5QixDQUFDLFdBQVcsRUFBRSxJQUFJLGNBQWMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUM7WUFDeEYsZUFBZSxFQUFFLGlCQUFpQjtTQUNuQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDYixPQUFPLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxHQUFHLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUNuRSxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsQ0FBQztLQUMvQjtJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUMsQ0FBQyxDQUFDO0FBRUg7Ozs7Ozs7R0FPRztBQUNILFNBQVMsYUFBYSxDQUFDLFFBQWdCO0lBQ3JDLE1BQU0sR0FBRyxHQUFHLGNBQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUM5QixNQUFNLGtCQUFrQixHQUFHLElBQUksR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBQ3hELE9BQU8sa0JBQWtCLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztXQUMzQyxlQUFRLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDLFdBQVcsRUFBRSxLQUFLLFNBQVMsQ0FBQztBQUMzRCxDQUFDO0FBR0Q7Ozs7OztHQU1HO0FBQ0gsU0FBUyx5QkFBeUIsQ0FBQyxLQUFhLEVBQUUsR0FBRyxJQUF1QjtJQUMxRSxNQUFNLEtBQUssR0FBRyxDQUFDLEtBQUssRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQy9CLE1BQU0sSUFBSSxHQUFHLEtBQUs7U0FDZixHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDbEQsSUFBSSxDQUFDLEdBQUcsQ0FBQztTQUNULE9BQU8sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDO1NBQ2xCLE9BQU8sQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDMUIsSUFBSSxJQUFJLENBQUMsTUFBTSxJQUFJLEVBQUUsRUFBRTtRQUNyQixPQUFPLElBQUksQ0FBQztLQUNiO0lBQ0QsTUFBTSxNQUFNLEdBQUcsbUJBQVUsQ0FBQyxRQUFRLENBQUM7UUFDbkMscUVBQXFFO1NBQ2xFLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZCLE1BQU0sQ0FBQyxLQUFLLENBQUM7U0FDYixTQUFTLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25CLE9BQU8sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxFQUFFLEdBQUcsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsSUFBSSxNQUFNLEVBQUUsQ0FBQztBQUNsRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgY3JlYXRlSGFzaCB9IGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgeyBiYXNlbmFtZSwgZXh0bmFtZSB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgVVJMIH0gZnJvbSAndXJsJztcblxuaW1wb3J0IHsgdmFsaWRhdGVBc3NlbWJseSB9IGZyb20gJ0Bqc2lpL3NwZWMnO1xuaW1wb3J0IHsgbWV0cmljU2NvcGUsIENvbmZpZ3VyYXRpb24sIFVuaXQgfSBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcyc7XG5pbXBvcnQgRW52aXJvbm1lbnRzIGZyb20gJ2F3cy1lbWJlZGRlZC1tZXRyaWNzL2xpYi9lbnZpcm9ubWVudC9FbnZpcm9ubWVudHMnO1xuaW1wb3J0IHR5cGUgeyBDb250ZXh0LCBTUVNFdmVudCB9IGZyb20gJ2F3cy1sYW1iZGEnO1xuaW1wb3J0IHR5cGUgeyBTdGF0ZU1hY2hpbmVJbnB1dCB9IGZyb20gJy4uL3BheWxvYWQtc2NoZW1hJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICcuLi9zaGFyZWQvYXdzLmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0ICogYXMgY29uc3RhbnRzIGZyb20gJy4uL3NoYXJlZC9jb25zdGFudHMnO1xuaW1wb3J0IHsgcmVxdWlyZUVudiB9IGZyb20gJy4uL3NoYXJlZC9lbnYubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBJbmdlc3Rpb25JbnB1dCB9IGZyb20gJy4uL3NoYXJlZC9pbmdlc3Rpb24taW5wdXQubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBpbnRlZ3JpdHkgfSBmcm9tICcuLi9zaGFyZWQvaW50ZWdyaXR5LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgZXh0cmFjdE9iamVjdHMgfSBmcm9tICcuLi9zaGFyZWQvdGFyYmFsbC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IE1ldHJpY05hbWUsIE1FVFJJQ1NfTkFNRVNQQUNFIH0gZnJvbSAnLi9jb25zdGFudHMnO1xuXG5Db25maWd1cmF0aW9uLmVudmlyb25tZW50T3ZlcnJpZGUgPSBFbnZpcm9ubWVudHMuTGFtYmRhO1xuQ29uZmlndXJhdGlvbi5uYW1lc3BhY2UgPSBNRVRSSUNTX05BTUVTUEFDRTtcblxuZXhwb3J0IGNvbnN0IGhhbmRsZXIgPSBtZXRyaWNTY29wZSgobWV0cmljcykgPT4gYXN5bmMgKGV2ZW50OiBTUVNFdmVudCwgY29udGV4dDogQ29udGV4dCkgPT4ge1xuICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQsIG51bGwsIDIpfWApO1xuXG4gIC8vIENsZWFyIG91dCB0aGUgZGVmYXVsdCBkaW1lbnNpb25zLCB3ZSB3b24ndCBuZWVkIHRoZW0uXG4gIG1ldHJpY3Muc2V0RGltZW5zaW9ucygpO1xuXG4gIGNvbnN0IEJVQ0tFVF9OQU1FID0gcmVxdWlyZUVudignQlVDS0VUX05BTUUnKTtcbiAgY29uc3QgU1RBVEVfTUFDSElORV9BUk4gPSByZXF1aXJlRW52KCdTVEFURV9NQUNISU5FX0FSTicpO1xuXG4gIGNvbnN0IHJlc3VsdCA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG5cbiAgZm9yIChjb25zdCByZWNvcmQgb2YgZXZlbnQuUmVjb3JkcyA/PyBbXSkge1xuICAgIGNvbnN0IHBheWxvYWQgPSBKU09OLnBhcnNlKHJlY29yZC5ib2R5KSBhcyBJbmdlc3Rpb25JbnB1dDtcblxuICAgIGNvbnN0IHRhcmJhbGxVcmkgPSBuZXcgVVJMKHBheWxvYWQudGFyYmFsbFVyaSk7XG4gICAgaWYgKHRhcmJhbGxVcmkucHJvdG9jb2wgIT09ICdzMzonKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIHByb3RvY29sIGluIFVSSTogJHt0YXJiYWxsVXJpfWApO1xuICAgIH1cbiAgICBjb25zdCB0YXJiYWxsID0gYXdhaXQgYXdzLnMzKCkuZ2V0T2JqZWN0KHtcbiAgICAgIC8vIE5vdGU6IHdlIGRyb3AgYW55dGhpbmcgYWZ0ZXIgdGhlIGZpcnN0IGAuYCBpbiB0aGUgaG9zdCwgYXMgd2Ugb25seSBjYXJlIGFib3V0IHRoZSBidWNrZXQgbmFtZS5cbiAgICAgIEJ1Y2tldDogdGFyYmFsbFVyaS5ob3N0LnNwbGl0KCcuJylbMF0sXG4gICAgICAvLyBOb3RlOiB0aGUgcGF0aG5hbWUgcGFydCBpcyBhYnNvbHV0ZSwgc28gd2Ugc3RyaXAgdGhlIGxlYWRpbmcgYC9gLlxuICAgICAgS2V5OiB0YXJiYWxsVXJpLnBhdGhuYW1lLnJlcGxhY2UoL15cXC8vLCAnJyksXG4gICAgICBWZXJzaW9uSWQ6IHRhcmJhbGxVcmkuc2VhcmNoUGFyYW1zLmdldCgndmVyc2lvbklkJykgPz8gdW5kZWZpbmVkLFxuICAgIH0pLnByb21pc2UoKTtcblxuICAgIGNvbnN0IGludGVncml0eUNoZWNrID0gaW50ZWdyaXR5KHBheWxvYWQsIEJ1ZmZlci5mcm9tKHRhcmJhbGwuQm9keSEpKTtcbiAgICBpZiAocGF5bG9hZC5pbnRlZ3JpdHkgIT09IGludGVncml0eUNoZWNrKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEludGVncml0eSBjaGVjayBmYWlsZWQ6ICR7cGF5bG9hZC5pbnRlZ3JpdHl9ICE9PSAke2ludGVncml0eUNoZWNrfWApO1xuICAgIH1cblxuICAgIGxldCBkb3RKc2lpOiBCdWZmZXI7XG4gICAgbGV0IHBhY2thZ2VKc29uOiBCdWZmZXI7XG4gICAgbGV0IGxpY2Vuc2VUZXh0OiBCdWZmZXIgfCB1bmRlZmluZWQ7XG4gICAgdHJ5IHtcbiAgICAgICh7IGRvdEpzaWksIHBhY2thZ2VKc29uLCBsaWNlbnNlVGV4dCB9ID0gYXdhaXQgZXh0cmFjdE9iamVjdHMoXG4gICAgICAgIEJ1ZmZlci5mcm9tKHRhcmJhbGwuQm9keSEpLFxuICAgICAgICB7XG4gICAgICAgICAgZG90SnNpaTogeyBwYXRoOiAncGFja2FnZS8uanNpaScsIHJlcXVpcmVkOiB0cnVlIH0sXG4gICAgICAgICAgcGFja2FnZUpzb246IHsgcGF0aDogJ3BhY2thZ2UvcGFja2FnZS5qc29uJywgcmVxdWlyZWQ6IHRydWUgfSxcbiAgICAgICAgICBsaWNlbnNlVGV4dDogeyBmaWx0ZXI6IGlzTGljZW5zZUZpbGUgfSxcbiAgICAgICAgfSxcbiAgICAgICkpO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcihgSW52YWxpZCB0YXJiYWxsIGNvbnRlbnQ6ICR7ZXJyfWApO1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX1RBUkJBTEwsIDEsIFVuaXQuQ291bnQpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBtZXRhZGF0YSA9IHsgZGF0ZTogcGF5bG9hZC50aW1lLCBsaWNlbnNlVGV4dDogbGljZW5zZVRleHQ/LnRvU3RyaW5nKCd1dGYtOCcpIH07XG5cbiAgICBsZXQgcGFja2FnZUxpY2Vuc2U6IHN0cmluZztcbiAgICBsZXQgcGFja2FnZU5hbWU6IHN0cmluZztcbiAgICBsZXQgcGFja2FnZVZlcnNpb246IHN0cmluZztcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBsaWNlbnNlLCBuYW1lLCB2ZXJzaW9uIH0gPSB2YWxpZGF0ZUFzc2VtYmx5KEpTT04ucGFyc2UoZG90SnNpaS50b1N0cmluZygndXRmLTgnKSkpO1xuICAgICAgcGFja2FnZUxpY2Vuc2UgPSBsaWNlbnNlO1xuICAgICAgcGFja2FnZU5hbWUgPSBuYW1lO1xuICAgICAgcGFja2FnZVZlcnNpb24gPSB2ZXJzaW9uO1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX0FTU0VNQkxZLCAwLCBVbml0LkNvdW50KTtcbiAgICB9IGNhdGNoIChleCkge1xuICAgICAgY29uc29sZS5lcnJvcihgUGFja2FnZSBkb2VzIG5vdCBjb250YWluIGEgdmFsaWQgYXNzZW1ibHkgLS0gaWdub3Jpbmc6ICR7ZXh9YCk7XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLklOVkFMSURfQVNTRU1CTFksIDEsIFVuaXQuQ291bnQpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIC8vIEVuc3VyZSB0aGUgYC5qc2lpYCBuYW1lLCB2ZXJzaW9uICYgbGljZW5zZSBjb3JyZXNwb25kcyB0byB0aG9zZSBpbiBgcGFja2FnZS5qc29uYFxuICAgIGNvbnN0IHsgbmFtZTogcGFja2FnZUpzb25OYW1lLCB2ZXJzaW9uOiBwYWNrYWdlSnNvblZlcnNpb24sIGxpY2Vuc2U6IHBhY2thZ2VKc29uTGljZW5zZSB9ID0gSlNPTi5wYXJzZShwYWNrYWdlSnNvbi50b1N0cmluZygndXRmLTgnKSk7XG4gICAgaWYgKHBhY2thZ2VKc29uTmFtZSAhPT0gcGFja2FnZU5hbWUgfHwgcGFja2FnZUpzb25WZXJzaW9uICE9PSBwYWNrYWdlVmVyc2lvbiB8fCBwYWNrYWdlSnNvbkxpY2Vuc2UgIT09IHBhY2thZ2VMaWNlbnNlKSB7XG4gICAgICBjb25zb2xlLmxvZyhgSWdub3JpbmcgcGFja2FnZSB3aXRoIG1pc21hdGNoZWQgbmFtZSwgdmVyc2lvbiwgYW5kL29yIGxpY2Vuc2UgKCR7cGFja2FnZUpzb25OYW1lfUAke3BhY2thZ2VKc29uVmVyc2lvbn0gaXMgJHtwYWNrYWdlSnNvbkxpY2Vuc2V9ICE9PSAke3BhY2thZ2VOYW1lfUAke3BhY2thZ2VWZXJzaW9ufSBpcyAke3BhY2thZ2VMaWNlbnNlfSlgKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuTUlTTUFUQ0hFRF9JREVOVElUWV9SRUpFQ1RJT05TLCAxLCBVbml0LkNvdW50KTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLk1JU01BVENIRURfSURFTlRJVFlfUkVKRUNUSU9OUywgMCwgVW5pdC5Db3VudCk7XG5cbiAgICAvLyBEaWQgd2UgaWRlbnRpZnkgYSBsaWNlbnNlIGZpbGUgb3Igbm90P1xuICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuRk9VTkRfTElDRU5TRV9GSUxFLCBsaWNlbnNlVGV4dCAhPSBudWxsID8gMSA6IDAsIFVuaXQuQ291bnQpO1xuXG4gICAgY29uc3QgeyBhc3NlbWJseUtleSwgbWV0YWRhdGFLZXksIHBhY2thZ2VLZXkgfSA9IGNvbnN0YW50cy5nZXRPYmplY3RLZXlzKHBhY2thZ2VOYW1lLCBwYWNrYWdlVmVyc2lvbik7XG4gICAgY29uc29sZS5sb2coYFdyaXRpbmcgYXNzZW1ibHkgYXQgJHthc3NlbWJseUtleX1gKTtcbiAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBwYWNrYWdlIGF0ICAke3BhY2thZ2VLZXl9YCk7XG4gICAgY29uc29sZS5sb2coYFdyaXRpbmcgbWV0YWRhdGEgYXQgICR7bWV0YWRhdGFLZXl9YCk7XG5cbiAgICAvLyB3ZSB1cGxvYWQgdGhlIG1ldGFkYXRhIGZpbGUgZmlyc3QgYmVjYXVzZSB0aGUgY2F0YWxvZyBidWlsZGVyIGRlcGVuZHMgb25cbiAgICAvLyBpdCBhbmQgaXMgdHJpZ2dlcmVkIGJ5IHRoZSBhc3NlbWJseSBmaWxlIHVwbG9hZC5cbiAgICBjb25zb2xlLmxvZyhgJHtwYWNrYWdlTmFtZX1AJHtwYWNrYWdlVmVyc2lvbn0gfCBVcGxvYWRpbmcgcGFja2FnZSBhbmQgbWV0YWRhdGEgZmlsZXNgKTtcbiAgICBjb25zdCBbcGtnLCBzdG9yZWRNZXRhZGF0YV0gPSBhd2FpdCBQcm9taXNlLmFsbChbXG4gICAgICBhd3MuczMoKS5wdXRPYmplY3Qoe1xuICAgICAgICBCdWNrZXQ6IEJVQ0tFVF9OQU1FLFxuICAgICAgICBLZXk6IHBhY2thZ2VLZXksXG4gICAgICAgIEJvZHk6IHRhcmJhbGwuQm9keSxcbiAgICAgICAgQ2FjaGVDb250cm9sOiAncHVibGljJyxcbiAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9vY3RldC1zdHJlYW0nLFxuICAgICAgICBNZXRhZGF0YToge1xuICAgICAgICAgICdMYW1iZGEtTG9nLUdyb3VwJzogY29udGV4dC5sb2dHcm91cE5hbWUsXG4gICAgICAgICAgJ0xhbWJkYS1Mb2ctU3RyZWFtJzogY29udGV4dC5sb2dTdHJlYW1OYW1lLFxuICAgICAgICAgICdMYW1iZGEtUnVuLUlkJzogY29udGV4dC5hd3NSZXF1ZXN0SWQsXG4gICAgICAgIH0sXG4gICAgICB9KS5wcm9taXNlKCksXG4gICAgICBhd3MuczMoKS5wdXRPYmplY3Qoe1xuICAgICAgICBCdWNrZXQ6IEJVQ0tFVF9OQU1FLFxuICAgICAgICBLZXk6IG1ldGFkYXRhS2V5LFxuICAgICAgICBCb2R5OiBKU09OLnN0cmluZ2lmeShtZXRhZGF0YSksXG4gICAgICAgIENhY2hlQ29udHJvbDogJ3B1YmxpYycsXG4gICAgICAgIENvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vanNvbicsXG4gICAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgICAnTGFtYmRhLUxvZy1TdHJlYW0nOiBjb250ZXh0LmxvZ1N0cmVhbU5hbWUsXG4gICAgICAgICAgJ0xhbWJkYS1SdW4tSWQnOiBjb250ZXh0LmF3c1JlcXVlc3RJZCxcbiAgICAgICAgfSxcbiAgICAgIH0pLnByb21pc2UoKSxcbiAgICBdKTtcblxuICAgIC8vIG5vdyB3ZSBjYW4gdXBsb2FkIHRoZSBhc3NlbWJseS5cbiAgICBjb25zb2xlLmxvZyhgJHtwYWNrYWdlTmFtZX1AJHtwYWNrYWdlVmVyc2lvbn0gfCBVcGxvYWRpbmcgYXNzZW1ibHkgZmlsZWApO1xuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXdhaXQgYXdzLnMzKCkucHV0T2JqZWN0KHtcbiAgICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICBLZXk6IGFzc2VtYmx5S2V5LFxuICAgICAgQm9keTogZG90SnNpaSxcbiAgICAgIENhY2hlQ29udHJvbDogJ3B1YmxpYycsXG4gICAgICBDb250ZW50VHlwZTogJ2FwcGxpY2F0aW9uL2pzb24nLFxuICAgICAgTWV0YWRhdGE6IHtcbiAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgJ0xhbWJkYS1Mb2ctU3RyZWFtJzogY29udGV4dC5sb2dTdHJlYW1OYW1lLFxuICAgICAgICAnTGFtYmRhLVJ1bi1JZCc6IGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgfSxcbiAgICB9KS5wcm9taXNlKCk7XG5cbiAgICBjb25zdCBjcmVhdGVkOiBTdGF0ZU1hY2hpbmVJbnB1dCA9IHtcbiAgICAgIGJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICBhc3NlbWJseToge1xuICAgICAgICBrZXk6IGFzc2VtYmx5S2V5LFxuICAgICAgICB2ZXJzaW9uSWQ6IGFzc2VtYmx5LlZlcnNpb25JZCxcbiAgICAgIH0sXG4gICAgICBwYWNrYWdlOiB7XG4gICAgICAgIGtleTogcGFja2FnZUtleSxcbiAgICAgICAgdmVyc2lvbklkOiBwa2cuVmVyc2lvbklkLFxuICAgICAgfSxcbiAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgIGtleTogbWV0YWRhdGFLZXksXG4gICAgICAgIHZlcnNpb25JZDogc3RvcmVkTWV0YWRhdGEuVmVyc2lvbklkLFxuICAgICAgfSxcbiAgICB9O1xuICAgIGNvbnNvbGUubG9nKGBDcmVhdGVkIG9iamVjdHM6ICR7SlNPTi5zdHJpbmdpZnkoY3JlYXRlZCwgbnVsbCwgMil9YCk7XG5cbiAgICBjb25zdCBzZm4gPSBhd2FpdCBhd3Muc3RlcEZ1bmN0aW9ucygpLnN0YXJ0RXhlY3V0aW9uKHtcbiAgICAgIGlucHV0OiBKU09OLnN0cmluZ2lmeShjcmVhdGVkKSxcbiAgICAgIG5hbWU6IHNmbkV4ZWN1dGlvbk5hbWVGcm9tUGFydHMocGFja2FnZU5hbWUsIGB2JHtwYWNrYWdlVmVyc2lvbn1gLCBjb250ZXh0LmF3c1JlcXVlc3RJZCksXG4gICAgICBzdGF0ZU1hY2hpbmVBcm46IFNUQVRFX01BQ0hJTkVfQVJOLFxuICAgIH0pLnByb21pc2UoKTtcbiAgICBjb25zb2xlLmxvZyhgU3RhcnRlZCBTdGF0ZU1hY2hpbmUgZXhlY3V0aW9uOiAke3Nmbi5leGVjdXRpb25Bcm59YCk7XG4gICAgcmVzdWx0LnB1c2goc2ZuLmV4ZWN1dGlvbkFybik7XG4gIH1cblxuICByZXR1cm4gcmVzdWx0O1xufSk7XG5cbi8qKlxuICogQ2hlY2tzIHdoZXRoZXIgdGhlIHByb3ZpZGVkIGZpbGUgbmFtZSBjb3JyZXNwb25kcyB0byBhIGxpY2Vuc2UgZmlsZSBvciBub3QuXG4gKlxuICogQHBhcmFtIGZpbGVOYW1lIHRoZSBmaWxlIG5hbWUgdG8gYmUgY2hlY2tlZC5cbiAqXG4gKiBAcmV0dXJucyBgdHJ1ZWAgSUlGIHRoZSBmaWxlIGlzIG5hbWVkIExJQ0VOU0UgYW5kIGhhcyB0aGUgLk1EIG9yIC5UWFRcbiAqICAgICAgICAgIGV4dGVuc2lvbiwgb3Igbm8gZXh0ZW5zaW9uIGF0IGFsbC4gVGhlIHRlc3QgaXMgY2FzZS1pbnNlbnNpdGl2ZS5cbiAqL1xuZnVuY3Rpb24gaXNMaWNlbnNlRmlsZShmaWxlTmFtZTogc3RyaW5nKTogYm9vbGVhbiB7XG4gIGNvbnN0IGV4dCA9IGV4dG5hbWUoZmlsZU5hbWUpO1xuICBjb25zdCBwb3NzaWJsZUV4dGVuc2lvbnMgPSBuZXcgU2V0KFsnJywgJy5tZCcsICcudHh0J10pO1xuICByZXR1cm4gcG9zc2libGVFeHRlbnNpb25zLmhhcyhleHQudG9Mb3dlckNhc2UoKSlcbiAgICAmJiBiYXNlbmFtZShmaWxlTmFtZSwgZXh0KS50b1VwcGVyQ2FzZSgpID09PSAnTElDRU5TRSc7XG59XG5cblxuLyoqXG4gKiBDcmVhdGVzIGEgU3RlcEZ1bmN0aW9uIGV4ZWN1dGlvbiByZXF1ZXN0IG5hbWUgYmFzZWQgb24gdGhlIHByb3ZpZGVkIHBhcnRzLlxuICogVGhlIHJlc3VsdCBpcyBndWFyYW50ZWVkIHRvIGJlIDgwIGNoYXJhY3RlcnMgb3IgbGVzcyBhbmQgdG8gY29udGFpbiBvbmx5XG4gKiBjaGFyYWN0ZXJzIHRoYXQgYXJlIHZhbGlkIGZvciBhIFN0ZXBGdW5jdGlvbiBleGVjdXRpb24gcmVxdWVzdCBuYW1lIGZvciB3aGljaFxuICogQ2xvdWRXYXRjaCBMb2dnaW5nIGNhbiBiZSBlbmFibGVkLiBUaGUgcmVzdWx0aW5nIG5hbWUgaXMgdmVyeSBsaWtlbHkgdG9cbiAqIGJlIHVuaXF1ZSBmb3IgYSBnaXZlbiBpbnB1dC5cbiAqL1xuZnVuY3Rpb24gc2ZuRXhlY3V0aW9uTmFtZUZyb21QYXJ0cyhmaXJzdDogc3RyaW5nLCAuLi5yZXN0OiByZWFkb25seSBzdHJpbmdbXSk6IHN0cmluZyB7XG4gIGNvbnN0IHBhcnRzID0gW2ZpcnN0LCAuLi5yZXN0XTtcbiAgY29uc3QgbmFtZSA9IHBhcnRzXG4gICAgLm1hcCgocGFydCkgPT4gcGFydC5yZXBsYWNlKC9bXmEtejAtOV8tXSsvaWcsICdfJykpXG4gICAgLmpvaW4oJ18nKVxuICAgIC5yZXBsYWNlKC9eXy9nLCAnJylcbiAgICAucmVwbGFjZSgvX3syLH0vZywgJ18nKTtcbiAgaWYgKG5hbWUubGVuZ3RoIDw9IDgwKSB7XG4gICAgcmV0dXJuIG5hbWU7XG4gIH1cbiAgY29uc3Qgc3VmZml4ID0gY3JlYXRlSGFzaCgnc2hhMjU2JylcbiAgLy8gVGhlIGhhc2ggaXMgY29tcHV0ZWQgYmFzZWQgb24gaW5wdXQgYXJndW1lbnRzLCB0byBtYXhpbWl6ZSB1bmljaXR5XG4gICAgLnVwZGF0ZShwYXJ0cy5qb2luKCdfJykpXG4gICAgLmRpZ2VzdCgnaGV4JylcbiAgICAuc3Vic3RyaW5nKDAsIDYpO1xuICByZXR1cm4gYCR7bmFtZS5zdWJzdHJpbmcoMCwgODAgLSBzdWZmaXgubGVuZ3RoIC0gMSl9XyR7c3VmZml4fWA7XG59XG4iXX0=