"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const denyListClient = require("../../../backend/deny-list/client.lambda-shared");
const licenseListClient = require("../../../backend/license-list/client.lambda-shared");
const aws_lambda_shared_1 = require("../../../backend/shared/aws.lambda-shared");
const env = require("../../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../../backend/shared/integrity.lambda-shared");
const tarball = require("../../../backend/shared/tarball.lambda-shared");
const code_artifact_forwarder_lambda_1 = require("../../../package-sources/codeartifact/code-artifact-forwarder.lambda");
const safe_mock_1 = require("../../safe-mock");
const mockBucketName = 'mock-bucket-name';
const mockQueueUrl = 'https://mock-queue-url/dummy';
jest.mock('../../../backend/shared/env.lambda-shared');
env.requireEnv
    .mockName('mockEnv.requireEnv')
    .mockImplementation((name) => {
    switch (name) {
        case 'BUCKET_NAME':
            return mockBucketName;
        case 'QUEUE_URL':
            return mockQueueUrl;
        default:
            fail(`Attempted to use unexpected environment variable: ${name}`);
    }
});
const mockDenyListClientLookup = jest.fn()
    .mockName('mockDenyListClient.lookup');
jest.mock('../../../backend/deny-list/client.lambda-shared');
denyListClient.DenyListClient.newClient
    .mockName('DenyListClient.newClient')
    .mockImplementation(() => Promise.resolve(safe_mock_1.safeMock('mockDenyListClient', {
    lookup: mockDenyListClientLookup,
})));
const mockLicenseListLookup = jest.fn()
    .mockName('mockLicenseListClient.lookup');
jest.mock('../../../backend/license-list/client.lambda-shared');
licenseListClient.LicenseListClient.newClient
    .mockName('LicenseListClient.newClient')
    .mockImplementation(() => Promise.resolve(safe_mock_1.safeMock('mockLicenseListClient', {
    lookup: mockLicenseListLookup,
})));
jest.mock('../../../backend/shared/tarball.lambda-shared');
const mockExtractObjects = tarball.extractObjects
    .mockName('tarball.extractObjects');
beforeEach(() => {
    AWSMock.setSDKInstance(AWS);
});
afterEach(() => {
    AWSMock.restore();
    aws_lambda_shared_1.reset();
});
const detail = safe_mock_1.safeMock('request.detail', {
    domainOwner: '123456789012',
    domainName: 'mock-domain-name',
    repositoryName: 'mock-repository-name',
    operationType: 'Created',
    packageName: 'pkg-name',
    packageNamespace: 'pkg-namespace',
    packageVersion: '1.2.3-dev.1337',
    packageFormat: 'npm',
    packageVersionRevision: crypto_1.pseudoRandomBytes(10).toString('base64'),
    eventDeduplicationId: crypto_1.pseudoRandomBytes(10).toString('base64'),
});
test('happy path', async () => {
    // GIVEN
    const mockContext = safe_mock_1.safeMock('mockContext', {
        awsRequestId: crypto_1.pseudoRandomBytes(6).toString('base64'),
        logGroupName: `mock-log-group-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
        logStreamName: `mock-log-stream-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
    });
    const mockGetPackageVersionAssetResult = {
        asset: 'mock-asset-content',
        assetName: 'package.tgz',
        packageVersion: '1.2.3-dev.1337',
        packageVersionRevision: crypto_1.pseudoRandomBytes(10).toString('base64'),
    };
    AWSMock.mock('CodeArtifact', 'getPackageVersionAsset', (request, cb) => {
        try {
            expect(request).toEqual({
                asset: 'package.tgz',
                format: 'npm',
                domainOwner: detail.domainOwner,
                domain: detail.domainName,
                repository: detail.repositoryName,
                namespace: detail.packageNamespace,
                package: detail.packageName,
                packageVersion: detail.packageVersion,
            });
            cb(null, mockGetPackageVersionAssetResult);
        }
        catch (e) {
            cb(e);
        }
    });
    const mockAssembly = Buffer.from('mock-assembly-content');
    const mockPackageJson = safe_mock_1.safeMock('package.json', { license: 'Apache-2.0' });
    mockExtractObjects.mockImplementationOnce(async (tgz, selector) => {
        expect(tgz).toEqual(Buffer.from(mockGetPackageVersionAssetResult.asset));
        expect(selector).toHaveProperty('assemblyJson', { path: 'package/.jsii' });
        expect(selector).toHaveProperty('packageJson', { path: 'package/package.json', required: true });
        return {
            assemblyJson: mockAssembly,
            packageJson: Buffer.from(JSON.stringify(mockPackageJson)),
        };
    });
    mockLicenseListLookup.mockReturnValueOnce('Apache-2.0');
    const stagingKey = `@${detail.packageNamespace}/${detail.packageName}/${detail.packageVersion}/${mockGetPackageVersionAssetResult.packageVersionRevision}/package.tgz`;
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        try {
            expect(req).toEqual({
                Bucket: mockBucketName,
                Key: stagingKey,
                Body: mockGetPackageVersionAssetResult.asset,
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': mockContext.logGroupName,
                    'Lambda-Log-Stream': mockContext.logStreamName,
                    'Lambda-Run-Id': mockContext.awsRequestId,
                },
            });
            cb(null, safe_mock_1.safeMock('mockS3PutObjectOutput', {}));
        }
        catch (e) {
            cb(e);
        }
    });
    const mockSendMessageResult = {
        MessageId: crypto_1.pseudoRandomBytes(10).toString('base64'),
    };
    const time = new Date().toISOString();
    const resources = ['arn:obviously:made:up'];
    AWSMock.mock('SQS', 'sendMessage', (req, cb) => {
        try {
            expect(req).toEqual({
                MessageAttributes: {
                    AWS_REQUEST_ID: { DataType: 'String', StringValue: mockContext.awsRequestId },
                    LOG_GROUP_NAME: { DataType: 'String', StringValue: mockContext.logGroupName },
                    LOG_STREAM_NAME: { DataType: 'String', StringValue: mockContext.logStreamName },
                },
                MessageBody: JSON.stringify({
                    tarballUri: `s3://${mockBucketName}/${stagingKey}`,
                    metadata: { resources: resources.join(', ') },
                    time,
                    integrity: integrity_lambda_shared_1.integrity({
                        tarballUri: `s3://${mockBucketName}/${stagingKey}`,
                        metadata: { resources: resources.join(', ') },
                        time,
                    }, Buffer.from(mockGetPackageVersionAssetResult.asset)),
                }),
                MessageDeduplicationId: detail.eventDeduplicationId,
                QueueUrl: mockQueueUrl,
            });
            cb(null, mockSendMessageResult);
        }
        catch (e) {
            cb(e);
        }
    });
    // WHEN
    const request = safe_mock_1.safeMock('request', { resources, time, detail });
    // THEN
    await expect(code_artifact_forwarder_lambda_1.handler(request, mockContext)).resolves.toEqual(mockSendMessageResult);
    expect(mockDenyListClientLookup).toHaveBeenCalledWith('@pkg-namespace/pkg-name', '1.2.3-dev.1337');
    return expect(mockLicenseListLookup).toHaveBeenCalledWith('Apache-2.0');
});
test('no license (i.e: UNLICENSED)', async () => {
    // GIVEN
    const mockContext = safe_mock_1.safeMock('mockContext', {
        awsRequestId: crypto_1.pseudoRandomBytes(6).toString('base64'),
        logGroupName: `mock-log-group-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
        logStreamName: `mock-log-stream-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
    });
    const mockGetPackageVersionAssetResult = {
        asset: 'mock-asset-content',
        assetName: 'package.tgz',
        packageVersion: '1.2.3-dev.1337',
        packageVersionRevision: crypto_1.pseudoRandomBytes(10).toString('base64'),
    };
    AWSMock.mock('CodeArtifact', 'getPackageVersionAsset', (request, cb) => {
        try {
            expect(request).toEqual({
                asset: 'package.tgz',
                format: 'npm',
                domainOwner: detail.domainOwner,
                domain: detail.domainName,
                repository: detail.repositoryName,
                namespace: detail.packageNamespace,
                package: detail.packageName,
                packageVersion: detail.packageVersion,
            });
            cb(null, mockGetPackageVersionAssetResult);
        }
        catch (e) {
            cb(e);
        }
    });
    const mockAssembly = Buffer.from('mock-assembly-content');
    const mockPackageJson = safe_mock_1.safeMock('package.json', { license: undefined });
    mockExtractObjects.mockImplementationOnce(async (tgz, selector) => {
        expect(tgz).toEqual(Buffer.from(mockGetPackageVersionAssetResult.asset));
        expect(selector).toHaveProperty('assemblyJson', { path: 'package/.jsii' });
        expect(selector).toHaveProperty('packageJson', { path: 'package/package.json', required: true });
        return {
            assemblyJson: mockAssembly,
            packageJson: Buffer.from(JSON.stringify(mockPackageJson)),
        };
    });
    // WHEN
    const request = safe_mock_1.safeMock('request', { time: new Date().toISOString(), detail });
    // THEN
    await expect(code_artifact_forwarder_lambda_1.handler(request, mockContext)).resolves.toBeUndefined();
    return expect(mockLicenseListLookup).toHaveBeenCalledWith('UNLICENSED');
});
test('ineligible license', async () => {
    // GIVEN
    const mockContext = safe_mock_1.safeMock('mockContext', {
        awsRequestId: crypto_1.pseudoRandomBytes(6).toString('base64'),
        logGroupName: `mock-log-group-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
        logStreamName: `mock-log-stream-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
    });
    const mockGetPackageVersionAssetResult = {
        asset: 'mock-asset-content',
        assetName: 'package.tgz',
        packageVersion: '1.2.3-dev.1337',
        packageVersionRevision: crypto_1.pseudoRandomBytes(10).toString('base64'),
    };
    AWSMock.mock('CodeArtifact', 'getPackageVersionAsset', (request, cb) => {
        try {
            expect(request).toEqual({
                asset: 'package.tgz',
                format: 'npm',
                domainOwner: detail.domainOwner,
                domain: detail.domainName,
                repository: detail.repositoryName,
                namespace: detail.packageNamespace,
                package: detail.packageName,
                packageVersion: detail.packageVersion,
            });
            cb(null, mockGetPackageVersionAssetResult);
        }
        catch (e) {
            cb(e);
        }
    });
    const mockAssembly = Buffer.from('mock-assembly-content');
    const mockPackageJson = safe_mock_1.safeMock('package.json', { license: 'Phony-MOCK' });
    mockExtractObjects.mockImplementationOnce(async (tgz, selector) => {
        expect(tgz).toEqual(Buffer.from(mockGetPackageVersionAssetResult.asset));
        expect(selector).toHaveProperty('assemblyJson', { path: 'package/.jsii' });
        expect(selector).toHaveProperty('packageJson', { path: 'package/package.json', required: true });
        return {
            assemblyJson: mockAssembly,
            packageJson: Buffer.from(JSON.stringify(mockPackageJson)),
        };
    });
    // WHEN
    const request = safe_mock_1.safeMock('request', { time: new Date().toISOString(), detail });
    // THEN
    await expect(code_artifact_forwarder_lambda_1.handler(request, mockContext)).resolves.toBeUndefined();
    return expect(mockLicenseListLookup).toHaveBeenCalledWith('Phony-MOCK');
});
test('not a jsii package', async () => {
    // GIVEN
    const mockContext = safe_mock_1.safeMock('mockContext', {
        awsRequestId: crypto_1.pseudoRandomBytes(6).toString('base64'),
        logGroupName: `mock-log-group-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
        logStreamName: `mock-log-stream-name-${crypto_1.pseudoRandomBytes(6).toString('base64')}`,
    });
    const mockGetPackageVersionAssetResult = {
        asset: 'mock-asset-content',
        assetName: 'package.tgz',
        packageVersion: '1.2.3-dev.1337',
        packageVersionRevision: crypto_1.pseudoRandomBytes(10).toString('base64'),
    };
    AWSMock.mock('CodeArtifact', 'getPackageVersionAsset', (request, cb) => {
        try {
            expect(request).toEqual({
                asset: 'package.tgz',
                format: 'npm',
                domainOwner: detail.domainOwner,
                domain: detail.domainName,
                repository: detail.repositoryName,
                namespace: detail.packageNamespace,
                package: detail.packageName,
                packageVersion: detail.packageVersion,
            });
            cb(null, mockGetPackageVersionAssetResult);
        }
        catch (e) {
            cb(e);
        }
    });
    const mockPackageJson = safe_mock_1.safeMock('package.json', { license: 'Apache-2.0' });
    mockExtractObjects.mockImplementationOnce(async (tgz, selector) => {
        expect(tgz).toEqual(Buffer.from(mockGetPackageVersionAssetResult.asset));
        expect(selector).toHaveProperty('assemblyJson', { path: 'package/.jsii' });
        expect(selector).toHaveProperty('packageJson', { path: 'package/package.json', required: true });
        return {
            assemblyJson: undefined,
            packageJson: Buffer.from(JSON.stringify(mockPackageJson)),
        };
    });
    // WHEN
    const request = safe_mock_1.safeMock('request', { time: new Date().toISOString(), detail });
    // THEN
    return expect(code_artifact_forwarder_lambda_1.handler(request, mockContext)).resolves.toBeUndefined();
});
test('deny-listed package', async () => {
    // GIVEN
    const mockContext = safe_mock_1.safeMock('mockContext', {});
    mockDenyListClientLookup.mockImplementationOnce((name, version) => {
        expect(name).toBe('@pkg-namespace/pkg-name');
        expect(version).toBe('1.2.3-dev.1337');
        return {
            packageName: '@pkg-namespace/pkg-name',
            reason: 'I decided so',
        };
    });
    // WHEN
    const request = safe_mock_1.safeMock('request', { time: new Date().toISOString(), detail });
    // THEN
    return expect(code_artifact_forwarder_lambda_1.handler(request, mockContext)).resolves.toBeUndefined();
});
test('deleted package', async () => {
    // GIVEN
    const mockContext = safe_mock_1.safeMock('mockContext', {});
    // WHEN
    const request = safe_mock_1.safeMock('request', { time: new Date().toISOString(), detail: { ...detail, operationType: 'Deleted' } });
    // THEN
    return expect(code_artifact_forwarder_lambda_1.handler(request, mockContext)).resolves.toBeUndefined();
});
//# sourceMappingURL=data:application/json;base64,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