"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const code_artifact_1 = require("../../package-sources/code-artifact");
const safe_mock_1 = require("../safe-mock");
test('default configuration', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Test');
    const mockRepository = safe_mock_1.safeMock('mockRepository', {
        attrDomainOwner: '123456789012',
        attrDomainName: 'mock-domain-name',
        attrName: 'mock-repository-name',
        node: safe_mock_1.safeMock('mockRepository.node', { path: 'fake/path/to/repository' }),
    });
    const mockDenyListGrantRead = jest.fn().mockName('mockDenyList.grantRead');
    const mockLicenseListGrantRead = jest.fn().mockName('mockLicenseList.grantRead');
    const mockQueueGrantSendMessages = jest.fn().mockName('mockQueue.grantSendMessages');
    const mockMonitoringAddHighSeverityAlarm = jest.fn().mockName('mockMonitoring.addHighSeverityAlarm');
    const mockDenyList = safe_mock_1.safeMock('mockDenyList', {
        grantRead: mockDenyListGrantRead,
    });
    const mockLicenseList = safe_mock_1.safeMock('mockLicenseList', {
        grantRead: mockLicenseListGrantRead,
    });
    const mockIngestion = safe_mock_1.safeMock('mockIngestion', {
        grantPrincipal: new aws_iam_1.User(stack, 'MockIngestionRole'),
    });
    const mockMonitoring = safe_mock_1.safeMock('mockMonitoring', {
        addHighSeverityAlarm: mockMonitoringAddHighSeverityAlarm,
    });
    const mockQueue = safe_mock_1.safeMock('mockQueue', {
        grantSendMessages: mockQueueGrantSendMessages,
        queueUrl: 'https://fake-queue-url/phony',
    });
    // WHEN
    const source = new code_artifact_1.CodeArtifact({ repository: mockRepository });
    const bound = source.bind(stack, {
        denyList: mockDenyList,
        licenseList: mockLicenseList,
        ingestion: mockIngestion,
        monitoring: mockMonitoring,
        queue: mockQueue,
    });
    // THEN
    // Not gonna check all the contents therein... This is very UI-ey.
    expect(bound).toBeDefined();
    expect(mockDenyListGrantRead).toHaveBeenCalledTimes(1);
    expect(mockLicenseListGrantRead).toHaveBeenCalledTimes(1);
    expect(mockQueueGrantSendMessages).toHaveBeenCalledTimes(1);
    expect(mockMonitoringAddHighSeverityAlarm).toHaveBeenCalledTimes(2);
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot();
});
test('user-provided staging bucket', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Test');
    const mockRepository = safe_mock_1.safeMock('mockRepository', {
        attrDomainOwner: '123456789012',
        attrDomainName: 'mock-domain-name',
        attrName: 'mock-repository-name',
        node: safe_mock_1.safeMock('mockRepository.node', { path: 'fake/path/to/repository' }),
    });
    const mockBucketGrantRead = jest.fn().mockName('mockBucket.grantRead');
    const mockBucketGrantReadWrite = jest.fn().mockName('mockBucket.grantReadWrite');
    const mockDenyListGrantRead = jest.fn().mockName('mockDenyList.grantRead');
    const mockLicenseListGrantRead = jest.fn().mockName('mockLicenseList.grantRead');
    const mockQueueGrantSendMessages = jest.fn().mockName('mockQueue.grantSendMessages');
    const mockMonitoringAddHighSeverityAlarm = jest.fn().mockName('mockMonitoring.addHighSeverityAlarm');
    const mockBucket = safe_mock_1.safeMock('mockBucket', {
        bucketName: 'mock-bucket',
        grantRead: mockBucketGrantRead,
        grantReadWrite: mockBucketGrantReadWrite,
    });
    const mockDenyList = safe_mock_1.safeMock('mockDenyList', {
        grantRead: mockDenyListGrantRead,
    });
    const mockLicenseList = safe_mock_1.safeMock('mockLicenseList', {
        grantRead: mockLicenseListGrantRead,
    });
    const mockIngestion = safe_mock_1.safeMock('mockIngestion', {});
    const mockMonitoring = safe_mock_1.safeMock('mockMonitoring', {
        addHighSeverityAlarm: mockMonitoringAddHighSeverityAlarm,
    });
    const mockQueue = safe_mock_1.safeMock('mockQueue', {
        grantSendMessages: mockQueueGrantSendMessages,
        queueUrl: 'https://fake-queue-url/phony',
    });
    // WHEN
    const source = new code_artifact_1.CodeArtifact({ bucket: mockBucket, repository: mockRepository });
    const bound = source.bind(stack, {
        denyList: mockDenyList,
        licenseList: mockLicenseList,
        ingestion: mockIngestion,
        monitoring: mockMonitoring,
        queue: mockQueue,
    });
    // THEN
    // Not gonna check all the contents therein... This is very UI-ey.
    expect(bound).toBeDefined();
    expect(mockBucketGrantRead).toHaveBeenCalledWith(mockIngestion);
    expect(mockBucketGrantReadWrite).toHaveBeenCalledWith(expect.any(aws_lambda_1.Function));
    expect(mockDenyListGrantRead).toHaveBeenCalledWith(expect.any(aws_lambda_1.Function));
    expect(mockLicenseListGrantRead).toHaveBeenCalledWith(expect.any(aws_lambda_1.Function));
    expect(mockQueueGrantSendMessages).toHaveBeenCalledWith(expect.any(aws_lambda_1.Function));
    expect(mockMonitoringAddHighSeverityAlarm).toHaveBeenCalledTimes(2);
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZS1hcnRpZmFjdC50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL19fdGVzdHNfXy9wYWNrYWdlLXNvdXJjZXMvY29kZS1hcnRpZmFjdC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQ0EsOENBQW9EO0FBQ3BELG9EQUErQztBQUcvQyx3Q0FBMEQ7QUFJMUQsdUVBQW1FO0FBQ25FLDRDQUF3QztBQUV4QyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO0lBQ2pDLFFBQVE7SUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0lBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNyQyxNQUFNLGNBQWMsR0FBRyxvQkFBUSxDQUFnQixnQkFBZ0IsRUFBRTtRQUMvRCxlQUFlLEVBQUUsY0FBYztRQUMvQixjQUFjLEVBQUUsa0JBQWtCO1FBQ2xDLFFBQVEsRUFBRSxzQkFBc0I7UUFDaEMsSUFBSSxFQUFFLG9CQUFRLENBQWdCLHFCQUFxQixFQUFFLEVBQUUsSUFBSSxFQUFFLHlCQUF5QixFQUFFLENBQUM7S0FDMUYsQ0FBQyxDQUFDO0lBRUgsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0UsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDakYsTUFBTSwwQkFBMEIsR0FBRyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDckYsTUFBTSxrQ0FBa0MsR0FBRyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7SUFFckcsTUFBTSxZQUFZLEdBQUcsb0JBQVEsQ0FBVyxjQUFjLEVBQUU7UUFDdEQsU0FBUyxFQUFFLHFCQUFxQjtLQUNqQyxDQUFDLENBQUM7SUFDSCxNQUFNLGVBQWUsR0FBRyxvQkFBUSxDQUFlLGlCQUFpQixFQUFFO1FBQ2hFLFNBQVMsRUFBRSx3QkFBd0I7S0FDcEMsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxhQUFhLEdBQUcsb0JBQVEsQ0FBYSxlQUFlLEVBQUU7UUFDMUQsY0FBYyxFQUFFLElBQUksY0FBSSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsQ0FBQztLQUNyRCxDQUFDLENBQUM7SUFDSCxNQUFNLGNBQWMsR0FBRyxvQkFBUSxDQUFhLGdCQUFnQixFQUFFO1FBQzVELG9CQUFvQixFQUFFLGtDQUFrQztLQUN6RCxDQUFDLENBQUM7SUFDSCxNQUFNLFNBQVMsR0FBRyxvQkFBUSxDQUFTLFdBQVcsRUFBRTtRQUM5QyxpQkFBaUIsRUFBRSwwQkFBMEI7UUFDN0MsUUFBUSxFQUFFLDhCQUE4QjtLQUN6QyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSw0QkFBWSxDQUFDLEVBQUUsVUFBVSxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7SUFDaEUsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUU7UUFDL0IsUUFBUSxFQUFFLFlBQVk7UUFDdEIsV0FBVyxFQUFFLGVBQWU7UUFDNUIsU0FBUyxFQUFFLGFBQWE7UUFDeEIsVUFBVSxFQUFFLGNBQWM7UUFDMUIsS0FBSyxFQUFFLFNBQVM7S0FDakIsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGtFQUFrRTtJQUNsRSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDNUIsTUFBTSxDQUFDLHFCQUFxQixDQUFDLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdkQsTUFBTSxDQUFDLHdCQUF3QixDQUFDLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDMUQsTUFBTSxDQUFDLDBCQUEwQixDQUFDLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUQsTUFBTSxDQUFDLGtDQUFrQyxDQUFDLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDcEUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO0FBQ2pGLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtJQUN4QyxRQUFRO0lBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDckMsTUFBTSxjQUFjLEdBQUcsb0JBQVEsQ0FBZ0IsZ0JBQWdCLEVBQUU7UUFDL0QsZUFBZSxFQUFFLGNBQWM7UUFDL0IsY0FBYyxFQUFFLGtCQUFrQjtRQUNsQyxRQUFRLEVBQUUsc0JBQXNCO1FBQ2hDLElBQUksRUFBRSxvQkFBUSxDQUFnQixxQkFBcUIsRUFBRSxFQUFFLElBQUksRUFBRSx5QkFBeUIsRUFBRSxDQUFDO0tBQzFGLENBQUMsQ0FBQztJQUVILE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3ZFLE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQ2pGLE1BQU0scUJBQXFCLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNFLE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQ2pGLE1BQU0sMEJBQTBCLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ3JGLE1BQU0sa0NBQWtDLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBRXJHLE1BQU0sVUFBVSxHQUFHLG9CQUFRLENBQVUsWUFBWSxFQUFFO1FBQ2pELFVBQVUsRUFBRSxhQUFhO1FBQ3pCLFNBQVMsRUFBRSxtQkFBbUI7UUFDOUIsY0FBYyxFQUFFLHdCQUF3QjtLQUN6QyxDQUFDLENBQUM7SUFDSCxNQUFNLFlBQVksR0FBRyxvQkFBUSxDQUFZLGNBQWMsRUFBRTtRQUN2RCxTQUFTLEVBQUUscUJBQXFCO0tBQ2pDLENBQUMsQ0FBQztJQUNILE1BQU0sZUFBZSxHQUFHLG9CQUFRLENBQWUsaUJBQWlCLEVBQUU7UUFDaEUsU0FBUyxFQUFFLHdCQUF3QjtLQUNwQyxDQUFDLENBQUM7SUFDSCxNQUFNLGFBQWEsR0FBRyxvQkFBUSxDQUFhLGVBQWUsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNoRSxNQUFNLGNBQWMsR0FBRyxvQkFBUSxDQUFhLGdCQUFnQixFQUFFO1FBQzVELG9CQUFvQixFQUFFLGtDQUFrQztLQUN6RCxDQUFDLENBQUM7SUFDSCxNQUFNLFNBQVMsR0FBRyxvQkFBUSxDQUFTLFdBQVcsRUFBRTtRQUM5QyxpQkFBaUIsRUFBRSwwQkFBMEI7UUFDN0MsUUFBUSxFQUFFLDhCQUE4QjtLQUN6QyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSw0QkFBWSxDQUFDLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxVQUFVLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztJQUNwRixNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRTtRQUMvQixRQUFRLEVBQUUsWUFBWTtRQUN0QixXQUFXLEVBQUUsZUFBZTtRQUM1QixTQUFTLEVBQUUsYUFBYTtRQUN4QixVQUFVLEVBQUUsY0FBYztRQUMxQixLQUFLLEVBQUUsU0FBUztLQUNqQixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1Asa0VBQWtFO0lBQ2xFLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUM1QixNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoRSxNQUFNLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLHFCQUFRLENBQUMsQ0FBQyxDQUFDO0lBQzVFLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMscUJBQVEsQ0FBQyxDQUFDLENBQUM7SUFDekUsTUFBTSxDQUFDLHdCQUF3QixDQUFDLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxxQkFBUSxDQUFDLENBQUMsQ0FBQztJQUM1RSxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLHFCQUFRLENBQUMsQ0FBQyxDQUFDO0lBQzlFLE1BQU0sQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDLHFCQUFxQixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3BFLE1BQU0sQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUNqRixDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENmblJlcG9zaXRvcnkgfSBmcm9tICdAYXdzLWNkay9hd3MtY29kZWFydGlmYWN0JztcbmltcG9ydCB7IElHcmFudGFibGUsIFVzZXIgfSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEZ1bmN0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBJQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IElRdWV1ZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zcXMnO1xuaW1wb3J0IHsgQXBwLCBDb25zdHJ1Y3ROb2RlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgRGVueUxpc3QsIElEZW55TGlzdCB9IGZyb20gJy4uLy4uL2JhY2tlbmQnO1xuaW1wb3J0IHsgSUxpY2Vuc2VMaXN0IH0gZnJvbSAnLi4vLi4vYmFja2VuZC9saWNlbnNlLWxpc3QvYXBpJztcbmltcG9ydCB7IE1vbml0b3JpbmcgfSBmcm9tICcuLi8uLi9tb25pdG9yaW5nJztcbmltcG9ydCB7IENvZGVBcnRpZmFjdCB9IGZyb20gJy4uLy4uL3BhY2thZ2Utc291cmNlcy9jb2RlLWFydGlmYWN0JztcbmltcG9ydCB7IHNhZmVNb2NrIH0gZnJvbSAnLi4vc2FmZS1tb2NrJztcblxudGVzdCgnZGVmYXVsdCBjb25maWd1cmF0aW9uJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1Rlc3QnKTtcbiAgY29uc3QgbW9ja1JlcG9zaXRvcnkgPSBzYWZlTW9jazxDZm5SZXBvc2l0b3J5PignbW9ja1JlcG9zaXRvcnknLCB7XG4gICAgYXR0ckRvbWFpbk93bmVyOiAnMTIzNDU2Nzg5MDEyJyxcbiAgICBhdHRyRG9tYWluTmFtZTogJ21vY2stZG9tYWluLW5hbWUnLFxuICAgIGF0dHJOYW1lOiAnbW9jay1yZXBvc2l0b3J5LW5hbWUnLFxuICAgIG5vZGU6IHNhZmVNb2NrPENvbnN0cnVjdE5vZGU+KCdtb2NrUmVwb3NpdG9yeS5ub2RlJywgeyBwYXRoOiAnZmFrZS9wYXRoL3RvL3JlcG9zaXRvcnknIH0pLFxuICB9KTtcblxuICBjb25zdCBtb2NrRGVueUxpc3RHcmFudFJlYWQgPSBqZXN0LmZuKCkubW9ja05hbWUoJ21vY2tEZW55TGlzdC5ncmFudFJlYWQnKTtcbiAgY29uc3QgbW9ja0xpY2Vuc2VMaXN0R3JhbnRSZWFkID0gamVzdC5mbigpLm1vY2tOYW1lKCdtb2NrTGljZW5zZUxpc3QuZ3JhbnRSZWFkJyk7XG4gIGNvbnN0IG1vY2tRdWV1ZUdyYW50U2VuZE1lc3NhZ2VzID0gamVzdC5mbigpLm1vY2tOYW1lKCdtb2NrUXVldWUuZ3JhbnRTZW5kTWVzc2FnZXMnKTtcbiAgY29uc3QgbW9ja01vbml0b3JpbmdBZGRIaWdoU2V2ZXJpdHlBbGFybSA9IGplc3QuZm4oKS5tb2NrTmFtZSgnbW9ja01vbml0b3JpbmcuYWRkSGlnaFNldmVyaXR5QWxhcm0nKTtcblxuICBjb25zdCBtb2NrRGVueUxpc3QgPSBzYWZlTW9jazxEZW55TGlzdD4oJ21vY2tEZW55TGlzdCcsIHtcbiAgICBncmFudFJlYWQ6IG1vY2tEZW55TGlzdEdyYW50UmVhZCxcbiAgfSk7XG4gIGNvbnN0IG1vY2tMaWNlbnNlTGlzdCA9IHNhZmVNb2NrPElMaWNlbnNlTGlzdD4oJ21vY2tMaWNlbnNlTGlzdCcsIHtcbiAgICBncmFudFJlYWQ6IG1vY2tMaWNlbnNlTGlzdEdyYW50UmVhZCxcbiAgfSk7XG4gIGNvbnN0IG1vY2tJbmdlc3Rpb24gPSBzYWZlTW9jazxJR3JhbnRhYmxlPignbW9ja0luZ2VzdGlvbicsIHtcbiAgICBncmFudFByaW5jaXBhbDogbmV3IFVzZXIoc3RhY2ssICdNb2NrSW5nZXN0aW9uUm9sZScpLFxuICB9KTtcbiAgY29uc3QgbW9ja01vbml0b3JpbmcgPSBzYWZlTW9jazxNb25pdG9yaW5nPignbW9ja01vbml0b3JpbmcnLCB7XG4gICAgYWRkSGlnaFNldmVyaXR5QWxhcm06IG1vY2tNb25pdG9yaW5nQWRkSGlnaFNldmVyaXR5QWxhcm0sXG4gIH0pO1xuICBjb25zdCBtb2NrUXVldWUgPSBzYWZlTW9jazxJUXVldWU+KCdtb2NrUXVldWUnLCB7XG4gICAgZ3JhbnRTZW5kTWVzc2FnZXM6IG1vY2tRdWV1ZUdyYW50U2VuZE1lc3NhZ2VzLFxuICAgIHF1ZXVlVXJsOiAnaHR0cHM6Ly9mYWtlLXF1ZXVlLXVybC9waG9ueScsXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3Qgc291cmNlID0gbmV3IENvZGVBcnRpZmFjdCh7IHJlcG9zaXRvcnk6IG1vY2tSZXBvc2l0b3J5IH0pO1xuICBjb25zdCBib3VuZCA9IHNvdXJjZS5iaW5kKHN0YWNrLCB7XG4gICAgZGVueUxpc3Q6IG1vY2tEZW55TGlzdCxcbiAgICBsaWNlbnNlTGlzdDogbW9ja0xpY2Vuc2VMaXN0LFxuICAgIGluZ2VzdGlvbjogbW9ja0luZ2VzdGlvbixcbiAgICBtb25pdG9yaW5nOiBtb2NrTW9uaXRvcmluZyxcbiAgICBxdWV1ZTogbW9ja1F1ZXVlLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIC8vIE5vdCBnb25uYSBjaGVjayBhbGwgdGhlIGNvbnRlbnRzIHRoZXJlaW4uLi4gVGhpcyBpcyB2ZXJ5IFVJLWV5LlxuICBleHBlY3QoYm91bmQpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChtb2NrRGVueUxpc3RHcmFudFJlYWQpLnRvSGF2ZUJlZW5DYWxsZWRUaW1lcygxKTtcbiAgZXhwZWN0KG1vY2tMaWNlbnNlTGlzdEdyYW50UmVhZCkudG9IYXZlQmVlbkNhbGxlZFRpbWVzKDEpO1xuICBleHBlY3QobW9ja1F1ZXVlR3JhbnRTZW5kTWVzc2FnZXMpLnRvSGF2ZUJlZW5DYWxsZWRUaW1lcygxKTtcbiAgZXhwZWN0KG1vY2tNb25pdG9yaW5nQWRkSGlnaFNldmVyaXR5QWxhcm0pLnRvSGF2ZUJlZW5DYWxsZWRUaW1lcygyKTtcbiAgZXhwZWN0KGFwcC5zeW50aCgpLmdldFN0YWNrQnlOYW1lKHN0YWNrLnN0YWNrTmFtZSkudGVtcGxhdGUpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbnRlc3QoJ3VzZXItcHJvdmlkZWQgc3RhZ2luZyBidWNrZXQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnVGVzdCcpO1xuICBjb25zdCBtb2NrUmVwb3NpdG9yeSA9IHNhZmVNb2NrPENmblJlcG9zaXRvcnk+KCdtb2NrUmVwb3NpdG9yeScsIHtcbiAgICBhdHRyRG9tYWluT3duZXI6ICcxMjM0NTY3ODkwMTInLFxuICAgIGF0dHJEb21haW5OYW1lOiAnbW9jay1kb21haW4tbmFtZScsXG4gICAgYXR0ck5hbWU6ICdtb2NrLXJlcG9zaXRvcnktbmFtZScsXG4gICAgbm9kZTogc2FmZU1vY2s8Q29uc3RydWN0Tm9kZT4oJ21vY2tSZXBvc2l0b3J5Lm5vZGUnLCB7IHBhdGg6ICdmYWtlL3BhdGgvdG8vcmVwb3NpdG9yeScgfSksXG4gIH0pO1xuXG4gIGNvbnN0IG1vY2tCdWNrZXRHcmFudFJlYWQgPSBqZXN0LmZuKCkubW9ja05hbWUoJ21vY2tCdWNrZXQuZ3JhbnRSZWFkJyk7XG4gIGNvbnN0IG1vY2tCdWNrZXRHcmFudFJlYWRXcml0ZSA9IGplc3QuZm4oKS5tb2NrTmFtZSgnbW9ja0J1Y2tldC5ncmFudFJlYWRXcml0ZScpO1xuICBjb25zdCBtb2NrRGVueUxpc3RHcmFudFJlYWQgPSBqZXN0LmZuKCkubW9ja05hbWUoJ21vY2tEZW55TGlzdC5ncmFudFJlYWQnKTtcbiAgY29uc3QgbW9ja0xpY2Vuc2VMaXN0R3JhbnRSZWFkID0gamVzdC5mbigpLm1vY2tOYW1lKCdtb2NrTGljZW5zZUxpc3QuZ3JhbnRSZWFkJyk7XG4gIGNvbnN0IG1vY2tRdWV1ZUdyYW50U2VuZE1lc3NhZ2VzID0gamVzdC5mbigpLm1vY2tOYW1lKCdtb2NrUXVldWUuZ3JhbnRTZW5kTWVzc2FnZXMnKTtcbiAgY29uc3QgbW9ja01vbml0b3JpbmdBZGRIaWdoU2V2ZXJpdHlBbGFybSA9IGplc3QuZm4oKS5tb2NrTmFtZSgnbW9ja01vbml0b3JpbmcuYWRkSGlnaFNldmVyaXR5QWxhcm0nKTtcblxuICBjb25zdCBtb2NrQnVja2V0ID0gc2FmZU1vY2s8SUJ1Y2tldD4oJ21vY2tCdWNrZXQnLCB7XG4gICAgYnVja2V0TmFtZTogJ21vY2stYnVja2V0JyxcbiAgICBncmFudFJlYWQ6IG1vY2tCdWNrZXRHcmFudFJlYWQsXG4gICAgZ3JhbnRSZWFkV3JpdGU6IG1vY2tCdWNrZXRHcmFudFJlYWRXcml0ZSxcbiAgfSk7XG4gIGNvbnN0IG1vY2tEZW55TGlzdCA9IHNhZmVNb2NrPElEZW55TGlzdD4oJ21vY2tEZW55TGlzdCcsIHtcbiAgICBncmFudFJlYWQ6IG1vY2tEZW55TGlzdEdyYW50UmVhZCxcbiAgfSk7XG4gIGNvbnN0IG1vY2tMaWNlbnNlTGlzdCA9IHNhZmVNb2NrPElMaWNlbnNlTGlzdD4oJ21vY2tMaWNlbnNlTGlzdCcsIHtcbiAgICBncmFudFJlYWQ6IG1vY2tMaWNlbnNlTGlzdEdyYW50UmVhZCxcbiAgfSk7XG4gIGNvbnN0IG1vY2tJbmdlc3Rpb24gPSBzYWZlTW9jazxJR3JhbnRhYmxlPignbW9ja0luZ2VzdGlvbicsIHt9KTtcbiAgY29uc3QgbW9ja01vbml0b3JpbmcgPSBzYWZlTW9jazxNb25pdG9yaW5nPignbW9ja01vbml0b3JpbmcnLCB7XG4gICAgYWRkSGlnaFNldmVyaXR5QWxhcm06IG1vY2tNb25pdG9yaW5nQWRkSGlnaFNldmVyaXR5QWxhcm0sXG4gIH0pO1xuICBjb25zdCBtb2NrUXVldWUgPSBzYWZlTW9jazxJUXVldWU+KCdtb2NrUXVldWUnLCB7XG4gICAgZ3JhbnRTZW5kTWVzc2FnZXM6IG1vY2tRdWV1ZUdyYW50U2VuZE1lc3NhZ2VzLFxuICAgIHF1ZXVlVXJsOiAnaHR0cHM6Ly9mYWtlLXF1ZXVlLXVybC9waG9ueScsXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3Qgc291cmNlID0gbmV3IENvZGVBcnRpZmFjdCh7IGJ1Y2tldDogbW9ja0J1Y2tldCwgcmVwb3NpdG9yeTogbW9ja1JlcG9zaXRvcnkgfSk7XG4gIGNvbnN0IGJvdW5kID0gc291cmNlLmJpbmQoc3RhY2ssIHtcbiAgICBkZW55TGlzdDogbW9ja0RlbnlMaXN0LFxuICAgIGxpY2Vuc2VMaXN0OiBtb2NrTGljZW5zZUxpc3QsXG4gICAgaW5nZXN0aW9uOiBtb2NrSW5nZXN0aW9uLFxuICAgIG1vbml0b3Jpbmc6IG1vY2tNb25pdG9yaW5nLFxuICAgIHF1ZXVlOiBtb2NrUXVldWUsXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgLy8gTm90IGdvbm5hIGNoZWNrIGFsbCB0aGUgY29udGVudHMgdGhlcmVpbi4uLiBUaGlzIGlzIHZlcnkgVUktZXkuXG4gIGV4cGVjdChib3VuZCkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KG1vY2tCdWNrZXRHcmFudFJlYWQpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKG1vY2tJbmdlc3Rpb24pO1xuICBleHBlY3QobW9ja0J1Y2tldEdyYW50UmVhZFdyaXRlKS50b0hhdmVCZWVuQ2FsbGVkV2l0aChleHBlY3QuYW55KEZ1bmN0aW9uKSk7XG4gIGV4cGVjdChtb2NrRGVueUxpc3RHcmFudFJlYWQpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKGV4cGVjdC5hbnkoRnVuY3Rpb24pKTtcbiAgZXhwZWN0KG1vY2tMaWNlbnNlTGlzdEdyYW50UmVhZCkudG9IYXZlQmVlbkNhbGxlZFdpdGgoZXhwZWN0LmFueShGdW5jdGlvbikpO1xuICBleHBlY3QobW9ja1F1ZXVlR3JhbnRTZW5kTWVzc2FnZXMpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKGV4cGVjdC5hbnkoRnVuY3Rpb24pKTtcbiAgZXhwZWN0KG1vY2tNb25pdG9yaW5nQWRkSGlnaFNldmVyaXR5QWxhcm0pLnRvSGF2ZUJlZW5DYWxsZWRUaW1lcygyKTtcbiAgZXhwZWN0KGFwcC5zeW50aCgpLmdldFN0YWNrQnlOYW1lKHN0YWNrLnN0YWNrTmFtZSkudGVtcGxhdGUpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG4iXX0=