"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const repository_1 = require("../../codeartifact/repository");
test('basic usage', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new repository_1.Repository(stack, 'Repo');
    // THEN
    expect(stack).toMatchTemplate({
        Resources: {
            RepoDomainC79FB030: {
                Type: 'AWS::CodeArtifact::Domain',
                Properties: {
                    DomainName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                },
            },
            Repo: {
                Type: 'AWS::CodeArtifact::Repository',
                Properties: {
                    DomainName: {
                        'Fn::GetAtt': [
                            'RepoDomainC79FB030',
                            'Name',
                        ],
                    },
                    RepositoryName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                },
            },
        },
    }, assert_1.MatchStyle.SUPERSET);
});
test('using upstreams', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new repository_1.Repository(stack, 'Repo', { domainExists: true, domainName: 'test-domain', upstreams: ['upstream-1', 'upstream-2'] });
    // THEN
    expect(stack).toMatchTemplate({
        Resources: {
            Repo: {
                Type: 'AWS::CodeArtifact::Repository',
                Properties: {
                    DomainName: 'test-domain',
                    RepositoryName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                    Upstreams: ['upstream-1', 'upstream-2'],
                },
            },
        },
    }, assert_1.MatchStyle.SUPERSET);
});
test('external connection', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new repository_1.Repository(stack, 'Repo').addExternalConnection('public:npmjs');
    // THEN
    expect(stack).toMatchTemplate({
        Resources: {
            RepoDomainC79FB030: {
                Type: 'AWS::CodeArtifact::Domain',
                Properties: {
                    DomainName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                },
            },
            Repo: {
                Type: 'AWS::CodeArtifact::Repository',
                Properties: {
                    DomainName: {
                        'Fn::GetAtt': [
                            'RepoDomainC79FB030',
                            'Name',
                        ],
                    },
                    RepositoryName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                    ExternalConnections: ['public:npmjs'],
                },
            },
        },
    }, assert_1.MatchStyle.SUPERSET);
});
test('custom domain name', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new repository_1.Repository(stack, 'Repo', { domainName: 'custom-domain' });
    // THEN
    expect(stack).toMatchTemplate({
        Resources: {
            RepoDomainC79FB030: {
                Type: 'AWS::CodeArtifact::Domain',
                Properties: {
                    DomainName: 'custom-domain',
                },
            },
            Repo: {
                Type: 'AWS::CodeArtifact::Repository',
                Properties: {
                    DomainName: {
                        'Fn::GetAtt': [
                            'RepoDomainC79FB030',
                            'Name',
                        ],
                    },
                    RepositoryName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                },
            },
        },
    }, assert_1.MatchStyle.SUPERSET);
});
test('custom repository name', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new repository_1.Repository(stack, 'Repo', { repositoryName: 'custom-repo' });
    // THEN
    expect(stack).toMatchTemplate({
        Resources: {
            RepoDomainC79FB030: {
                Type: 'AWS::CodeArtifact::Domain',
                Properties: {
                    DomainName: 'c8d064061d1c8680a574cd5a9f9c9c69b475d41907',
                },
            },
            Repo: {
                Type: 'AWS::CodeArtifact::Repository',
                Properties: {
                    DomainName: {
                        'Fn::GetAtt': [
                            'RepoDomainC79FB030',
                            'Name',
                        ],
                    },
                    RepositoryName: 'custom-repo',
                },
            },
        },
    }, assert_1.MatchStyle.SUPERSET);
});
test('custom domain & repository name', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new repository_1.Repository(stack, 'Repo', { domainName: 'custom-domain', repositoryName: 'custom-repo' });
    // THEN
    expect(stack).toMatchTemplate({
        Resources: {
            RepoDomainC79FB030: {
                Type: 'AWS::CodeArtifact::Domain',
                Properties: {
                    DomainName: 'custom-domain',
                },
            },
            Repo: {
                Type: 'AWS::CodeArtifact::Repository',
                Properties: {
                    DomainName: {
                        'Fn::GetAtt': [
                            'RepoDomainC79FB030',
                            'Name',
                        ],
                    },
                    RepositoryName: 'custom-repo',
                },
            },
        },
    }, assert_1.MatchStyle.SUPERSET);
});
test('npm repository endpoint', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    // WHEN
    const repo = new repository_1.Repository(stack, 'Repo');
    new core_1.CfnOutput(stack, 'NpmRepositoryEndpoint', { value: repo.repositoryNpmEndpoint });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Parameters: expect.anything(),
        Resources: {
            AWS679f53fac002430cb0da5b7982bd22872D164C4C: {
                Properties: expect.anything(),
                Type: 'AWS::Lambda::Function',
            },
            AWS679f53fac002430cb0da5b7982bd2287ServiceRoleC1EA0FF2: {
                Properties: expect.anything(),
                Type: 'AWS::IAM::Role',
            },
            Repo: expect.anything(),
            RepoDomainC79FB030: expect.anything(),
        },
    });
});
test('S3 bucket ARN', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    // WHEN
    const repo = new repository_1.Repository(stack, 'Repo');
    new core_1.CfnOutput(stack, 'S3BucketArn', { value: repo.s3BucketArn });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Parameters: expect.anything(),
        Resources: {
            AWS679f53fac002430cb0da5b7982bd22872D164C4C: {
                Properties: expect.anything(),
                Type: 'AWS::Lambda::Function',
            },
            AWS679f53fac002430cb0da5b7982bd2287ServiceRoleC1EA0FF2: {
                Properties: expect.anything(),
                Type: 'AWS::IAM::Role',
            },
            Repo: expect.anything(),
            RepoDomainC79FB030: expect.anything(),
        },
    });
});
test('grantReadFromRepository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.ServicePrincipal('test.service-principal') });
    const repo = new repository_1.Repository(stack, 'Repo');
    // WHEN
    repo.grantReadFromRepository(role);
    // THEN
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: 'sts:GetServiceBearerToken',
                    Condition: {
                        StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' },
                    },
                    Effect: 'Allow',
                    Resource: '*',
                }, {
                    Action: ['codeartifact:GetAuthorizationToken', 'codeartifact:GetRepositoryEndpoint', 'codeartifact:ReadFromRepository'],
                    Effect: 'Allow',
                    Resource: [
                        stack.resolve(repo.repositoryDomainArn),
                        stack.resolve(repo.repositoryArn),
                    ],
                }],
            Version: '2012-10-17',
        },
        PolicyName: 'RoleDefaultPolicy5FFB7DAB',
        Roles: [stack.resolve(role.roleName)],
    });
});
test('throughVpcEndpoint', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    const api = vpc.addInterfaceEndpoint('API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const repositories = vpc.addInterfaceEndpoint('Repositories', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.ServicePrincipal('test.service-principal') });
    const repo = new repository_1.Repository(stack, 'Repo');
    const vpcRepo = repo.throughVpcEndpoint(api, repositories);
    // WHEN
    vpcRepo.grantReadFromRepository(role);
    // THEN
    expect(vpcRepo.repositoryDomainOwner).toBe(repo.repositoryDomainOwner); // Example pass-through...
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: 'sts:GetServiceBearerToken',
                    Condition: {
                        StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' },
                    },
                    Effect: 'Allow',
                    Resource: '*',
                }, {
                    Action: ['codeartifact:GetAuthorizationToken', 'codeartifact:GetRepositoryEndpoint', 'codeartifact:ReadFromRepository'],
                    Effect: 'Allow',
                    Resource: [
                        stack.resolve(repo.repositoryDomainArn),
                        stack.resolve(repo.repositoryArn),
                    ],
                }],
            Version: '2012-10-17',
        },
        PolicyName: 'RoleDefaultPolicy5FFB7DAB',
        Roles: [stack.resolve(role.roleName)],
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        PolicyDocument: {
            Statement: [{
                    Action: 'sts:GetServiceBearerToken',
                    Condition: {
                        StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' },
                    },
                    Effect: 'Allow',
                    Principal: { AWS: stack.resolve(role.roleArn) },
                    Resource: '*',
                }, {
                    Action: ['codeartifact:GetAuthorizationToken', 'codeartifact:GetRepositoryEndpoint'],
                    Effect: 'Allow',
                    Principal: { AWS: stack.resolve(role.roleArn) },
                    Resource: [
                        stack.resolve(repo.repositoryDomainArn),
                        stack.resolve(repo.repositoryArn),
                    ],
                }],
            Version: '2012-10-17',
        },
        ServiceName: stack.resolve(`com.amazonaws.${stack.region}.codeartifact.api`),
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        PolicyDocument: {
            Statement: [{
                    Action: 'codeartifact:ReadFromRepository',
                    Effect: 'Allow',
                    Principal: { AWS: stack.resolve(role.roleArn) },
                    Resource: stack.resolve(repo.repositoryArn),
                }],
            Version: '2012-10-17',
        },
        ServiceName: stack.resolve(`com.amazonaws.${stack.region}.codeartifact.repositories`),
    });
});
//# sourceMappingURL=data:application/json;base64,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