"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const backend_1 = require("../../../../backend");
const constants_1 = require("../../../../backend/shared/constants");
const monitoring_1 = require("../../../../monitoring");
const catalog_builder_mock_1 = require("./catalog-builder-mock");
const trigger_client_test_1 = require("./trigger.client-test");
const trigger_prune_test_1 = require("./trigger.prune-test");
// we need to pull mock package data from `src/` because we execute in `lib/`
const mockPackageDataDir = path_1.join(__dirname, '..', '..', '..', '..', '..', 'src', '__tests__', 'backend', 'deny-list', 'integ', 'package-data');
const app = new core_1.App();
const stack = new core_1.Stack(app, 'TestDenyList');
const packageData = new s3.Bucket(stack, 'MockDataBucket', {
    autoDeleteObjects: true,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
new aws_s3_deployment_1.BucketDeployment(stack, 'MockData', {
    destinationBucket: packageData,
    sources: [aws_s3_deployment_1.Source.asset(mockPackageDataDir)],
});
const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring');
const denylist = new backend_1.DenyList(stack, 'DenyList', {
    monitoring: monitoring,
    packageDataBucket: packageData,
    packageDataKeyPrefix: constants_1.STORAGE_KEY_PREFIX,
    rules: [
        { packageName: 'mypackage', reason: '"mypackage" is deprecated' },
        { packageName: 'your', version: '1.2.3', reason: 'v1.2.3 of "your" has a security issue' },
    ],
});
const catalogBuilderMock = new catalog_builder_mock_1.CatalogBuilderMock(stack, 'CatalogBuilderMock');
denylist.prune.onChangeInvoke(catalogBuilderMock);
const test1 = new trigger_client_test_1.TriggerClientTest(stack, 'ClientTest', {
    invokeAfter: [denylist],
    environment: {
        BUCKET_NAME: denylist.bucket.bucketName,
        FILE_NAME: denylist.objectKey,
    },
});
denylist.grantRead(test1);
const test2 = new trigger_prune_test_1.TriggerPruneTest(stack, 'PruneTest', {
    invokeAfter: [denylist],
    timeout: core_1.Duration.minutes(5),
    environment: {
        BUCKET_NAME: packageData.bucketName,
        TIMEOUT_SEC: core_1.Duration.minutes(2).toSeconds().toString(),
        EXPECTED_KEYS: JSON.stringify([
            'data/your/v1.2.4/world.txt',
            'data/your/v1.2.4/hello.txt',
        ]),
    },
});
packageData.grantRead(test2);
app.synth();
//# sourceMappingURL=data:application/json;base64,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