"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const client_lambda_shared_1 = require("../../../backend/deny-list/client.lambda-shared");
const constants_1 = require("../../../backend/deny-list/constants");
const aws = require("../../../backend/shared/aws.lambda-shared");
const sample = {
    'foo/v1.2.3': {
        packageName: 'foo',
        version: '1.2.3',
        reason: 'bar',
    },
    'bar': {
        packageName: 'bar',
        reason: 'hello bar',
    },
};
beforeEach(() => {
    process.env[constants_1.ENV_DENY_LIST_BUCKET_NAME] = 'deny-list-bucket-name';
    process.env[constants_1.ENV_DENY_LIST_OBJECT_KEY] = 'deny-list.json';
    AWSMock.setSDKInstance(AWS);
});
afterEach(() => {
    delete process.env[constants_1.ENV_DENY_LIST_BUCKET_NAME];
    delete process.env[constants_1.ENV_DENY_LIST_OBJECT_KEY];
    AWSMock.restore();
    aws.reset();
});
test('s3 object not found error', async () => {
    AWSMock.mock('S3', 'getObject', (_, callback) => {
        const err = new Error('NoSuchKey');
        err.code = 'NoSuchKey';
        callback(err, null);
    });
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    expect(client.lookup('foo', '1.2.3')).toBeUndefined();
});
test('s3 bucket not found error', async () => {
    AWSMock.mock('S3', 'getObject', (_, callback) => {
        const err = new Error('NoSuchBucket');
        err.code = 'NoSuchBucket';
        callback(err, null);
    });
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    expect(client.lookup('foo', '1.2.3')).toBeUndefined();
});
test('empty file', async () => {
    AWSMock.mock('S3', 'getObject', (params, callback) => {
        expect(params.Bucket).toBe('deny-list-bucket-name');
        expect(params.Key).toBe('deny-list.json');
        callback(null, { Body: '' });
    });
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    expect(client.lookup('foo', '1.2.3')).toBeUndefined();
});
test('json parsing error', async () => {
    AWSMock.mock('S3', 'getObject', (params, callback) => {
        expect(params.Bucket).toBe('deny-list-bucket-name');
        expect(params.Key).toBe('deny-list.json');
        callback(null, { Body: '09x{}' });
    });
    const expected = new Error('Unable to parse deny list file deny-list-bucket-name/deny-list.json: SyntaxError: Unexpected number in JSON at position 1');
    await expect(client_lambda_shared_1.DenyListClient.newClient()).rejects.toEqual(expected);
});
describe('lookup', () => {
    let client;
    beforeEach(async () => {
        AWSMock.mock('S3', 'getObject', (params, callback) => {
            expect(params.Bucket).toBe('deny-list-bucket-name');
            expect(params.Key).toBe('deny-list.json');
            callback(null, { Body: JSON.stringify(sample) });
        });
        client = await client_lambda_shared_1.DenyListClient.newClient();
    });
    test('match specific package + version', () => {
        expect(client.lookup('foo', '1.2.3')).toStrictEqual({
            packageName: 'foo',
            version: '1.2.3',
            reason: 'bar',
        });
    });
    test('match any version', () => {
        const expected = {
            packageName: 'bar',
            reason: 'hello bar',
        };
        expect(client.lookup('bar', '1.2.3')).toStrictEqual(expected);
        expect(client.lookup('bar', '4.4.4')).toStrictEqual(expected);
    });
    test('version does not match', () => {
        expect(client.lookup('foo', '4.4.4')).toBeUndefined();
    });
});
//# sourceMappingURL=data:application/json;base64,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