# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyrdf2vec',
 'pyrdf2vec.embedders',
 'pyrdf2vec.graphs',
 'pyrdf2vec.samplers',
 'pyrdf2vec.utils',
 'pyrdf2vec.walkers']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=20.3.0,<21.0.0',
 'cachetools>=4.2.1,<5.0.0',
 'matplotlib>=3.3.1,<4.0.0',
 'networkx>=2.5,<3.0',
 'python-louvain>=0.14,<0.15',
 'tomlkit>=0.7.0,<0.8.0',
 'torch>=1.7.1,<2.0.0',
 'tqdm>=4.48.2,<5.0.0',
 'transformers>=4.3.2,<5.0.0']

extras_require = \
{':extra == "docs"': ['scikit-learn>=0.23.2,<0.24.0'],
 ':extra == "docs" or extra == "tests"': ['gensim>=3.8.3,<4.0.0',
                                          'rdflib>=5.0.0,<6.0.0'],
 ':extra == "tests"': ['aiohttp>=3.7.3,<4.0.0',
                       'nest-asyncio>=1.5.1,<2.0.0',
                       'numpy>=1.20.0,<2.0.0',
                       'pandas>=1.1.1,<2.0.0']}

setup_kwargs = {
    'name': 'pyrdf2vec',
    'version': '0.2.1',
    'description': 'Python implementation and extension of RDF2Vec',
    'long_description': '\n.. rdf2vec-begin\n\nWhat is RDF2Vec?\n----------------\n\nRDF2Vec is an unsupervised technique that builds further on\n`Word2Vec <https://en.wikipedia.org/wiki/Word2vec>`__, where an\nembedding is learned per word, in two ways:\n\n1. **the word based on its context**: Continuous Bag-of-Words (CBOW);\n2. **the context based on a word**: Skip-Gram (SG).\n\nTo create this embedding, RDF2Vec first creates "sentences" which can be\nfed to Word2Vec by extracting walks of a certain depth from a Knowledge\nGraph.\n\nThis repository contains an implementation of the algorithm in "RDF2Vec:\nRDF Graph Embeddings and Their Applications" by Petar Ristoski, Jessica\nRosati, Tommaso Di Noia, Renato De Leone, Heiko Paulheim\n(`[paper] <http://semantic-web-journal.net/content/rdf2vec-rdf-graph-embeddings-and-their-applications-0>`__\n`[original\ncode] <http://data.dws.informatik.uni-mannheim.de/rdf2vec/>`__).\n\n.. rdf2vec-end\n.. getting-started-begin\n\nGetting Started\n---------------\n\nFor most uses-cases, here is how ``pyRDF2Vec`` should be used to generate\nembeddings and get literals from a given Knowledge Graph (KG) and entities:\n\n.. code:: python\n\n   import pandas as pd\n\n   from pyrdf2vec import RDF2VecTransformer\n   from pyrdf2vec.embedders import Word2Vec\n   from pyrdf2vec.graphs import KG\n   from pyrdf2vec.walkers import RandomWalker\n\n   data = pd.read_csv("samples/countries-cities/entities.tsv", sep="\\t")\n   entities = [entity for entity in data["location"]]\n   print(entities)\n   # [\n   #    "http://dbpedia.org/resource/Belgium",\n   #    "http://dbpedia.org/resource/France",\n   #    "http://dbpedia.org/resource/Germany",\n   # ]\n\n   transformer = RDF2VecTransformer(\n       Word2Vec(iter=10),\n       walkers=[RandomWalker(4, 10, with_reverse=False, n_jobs=2)],\n       # verbose=1\n   )\n   embeddings, literals = transformer.fit_transform(\n       KG(\n           "https://dbpedia.org/sparql",\n           skip_predicates={"www.w3.org/1999/02/22-rdf-syntax-ns#type"},\n           literals=[\n               [\n                   "http://dbpedia.org/ontology/wikiPageWikiLink",\n                   "http://www.w3.org/2004/02/skos/core#prefLabel",\n               ],\n               ["http://dbpedia.org/ontology/humanDevelopmentIndex"],\n           ],\n       ),\n       entities\n   )\n   print(embeddings)\n   # [\n   #     array([ 1.5737595e-04,  1.1333118e-03, -2.9838676e-04,  ..., -5.3064007e-04,\n   #             4.3192197e-04,  1.4529384e-03], dtype=float32),\n   #     array([-5.9027621e-04,  6.1689125e-04, -1.1987977e-03,  ...,  1.1066757e-03,\n   #            -1.0603866e-05,  6.6087965e-04], dtype=float32),\n   #     array([ 7.9996325e-04,  7.2907173e-04, -1.9482171e-04,  ...,  5.6251377e-04,\n   #             4.1435464e-04,  1.4478950e-04], dtype=float32)\n   # ]\n\n   print(literals)\n   # [\n   #     [(\'1830 establishments in Belgium\', \'States and territories established in 1830\',\n   #       \'Western European countries\', ..., \'Member states of the Organisation\n   #       internationale de la Francophonie\', \'Member states of the Union for the\n   #       Mediterranean\', \'Member states of the United Nations\'), 0.919],\n   #     [(\'Group of Eight nations\', \'Southwestern European countries\', \'1792\n   #       establishments in Europe\', ..., \'Member states of the Union for the\n   #       Mediterranean\', \'Member states of the United Nations\', \'Transcontinental\n   #       countries\'), 0.891]\n   #     [(\'Germany\', \'Group of Eight nations\', \'Articles containing video clips\', ...,\n   #       \'Member states of the European Union\', \'Member states of the Union for the\n   #       Mediterranean\', \'Member states of the United Nations\'), 0.939]\n   #  ]\n\nIf you are using a dataset other than MUTAG (where the interested entities have\nno parents in the KG), it is **highly recommended** to specify\n``with_reverse=True`` (defaults to ``False``) in the walking strategy (e.g.,\n``RandomWalker``). Such a parameter **allows Word2Vec** to have a better\nlearning window for an entity based on its parents and children and thus\n**predict test data with better accuracy**.\n\nIn a more concrete way, we provide a blog post with a tutorial on how to use\n``pyRDF2Vec`` `here\n<https://towardsdatascience.com/how-to-create-representations-of-entities-in-a-knowledge-graph-using-pyrdf2vec-82e44dad1a0>`__.\n\n**NOTE:** this blog uses an older version of ``pyRDF2Vec``, some commands need\nbe to adapted.\n\nIf you run the above snippet, you will not necessarily have the same\nembeddings, because there is no conservation of the random determinism, however\nit remains possible to do it (**SEE:** `FAQ <#faq>`__).\n\nInstallation\n~~~~~~~~~~~~\n\n``pyRDF2Vec`` can be installed in two ways:\n\n1. from `PyPI <https://pypi.org/project/pyrdf2vec>`__ using ``pip``:\n\n.. code:: bash\n\n   pip install pyRDF2vec\n\n2. from any compatible Python dependency manager (e.g., ``poetry``):\n\n.. code:: bash\n\n   poetry add pyRDF2vec\n\nIntroduction\n~~~~~~~~~~~~\n\nTo create embeddings for a list of entities, there are two steps to do\nbeforehand:\n\n1. **use a KG**;\n2. **define a walking strategy**.\n\nFor more elaborate examples, check the `examples\n<https://github.com/IBCNServices/pyRDF2Vec/blob/master/examples>`__ folder.\n\nIf no sampling strategy is defined, ``UniformSampler`` is used. Similarly for\nthe embedding techniques, ``Word2Vec`` is used by default.\n\nUse a Knowledge Graph\n~~~~~~~~~~~~~~~~~~~~~\n\nTo use a KG, you can initialize it in three ways:\n\n1. **From a endpoint server using SPARQL**:\n\n.. code:: python\n\n   from pyrdf2vec.graphs import KG\n\n   # Defined the DBpedia endpoint server, as well as a set of predicates to\n   # exclude from this KG and a list of predicate chains to fetch the literals.\n   KG(\n       "https://dbpedia.org/sparql",\n       skip_predicates={"www.w3.org/1999/02/22-rdf-syntax-ns#type"},\n       literals=[\n           [\n               "http://dbpedia.org/ontology/wikiPageWikiLink",\n               "http://www.w3.org/2004/02/skos/core#prefLabel",\n           ],\n           ["http://dbpedia.org/ontology/humanDevelopmentIndex"],\n        ],\n    ),\n\n2. **From a file using RDFLib**:\n\n.. code:: python\n\n   from pyrdf2vec.graphs import KG\n\n   # Defined the MUTAG KG, as well as a set of predicates to exclude from\n   # this KG and a list of predicate chains to get the literals.\n   KG(\n       "samples/mutag/mutag.owl",\n       skip_predicates={"http://dl-learner.org/carcinogenesis#isMutagenic"},\n       literals=[\n           [\n               "http://dl-learner.org/carcinogenesis#hasBond",\n               "http://dl-learner.org/carcinogenesis#inBond",\n           ],\n           [\n               "http://dl-learner.org/carcinogenesis#hasAtom",\n               "http://dl-learner.org/carcinogenesis#charge",\n           ],\n       ],\n   ),\n\n3. **From scratch**:\n\n.. code:: python\n\n   from pyrdf2vec.graphs import KG, Vertex\n\n    GRAPH = [\n        ["Alice", "knows", "Bob"],\n        ["Alice", "knows", "Dean"],\n        ["Dean", "loves", "Alice"],\n    ]\n    URL = "http://pyRDF2Vec"\n    CUSTOM_KG = KG()\n\n    for row in GRAPH:\n        subj = Vertex(f"{URL}#{row[0]}")\n        obj = Vertex((f"{URL}#{row[2]}"))\n        pred = Vertex((f"{URL}#{row[1]}"), predicate=True, vprev=subj, vnext=obj)\n        CUSTOM_KG.add_walk(subj, pred, obj)\n\nDefine Walking Strategies With Their Sampling Strategy\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nAll supported walking strategies can be found on the\n`Wiki\npage <https://github.com/IBCNServices/pyRDF2Vec/wiki/Walking-Strategies>`__.\n\nAs the number of walks grows exponentially in function of the depth,\nexhaustively extracting all walks quickly becomes infeasible for larger\nKnowledge Graphs. In order to avoid this issue, `sampling strategies\n<http://www.heikopaulheim.com/docs/wims2017.pdf>`__ can be applied. These will\nextract a fixed maximum number of walks per entity and sampling the walks\naccording to a certain metric.\n\nFor example, if one wants to extract a maximum of 10 walks of a maximum depth\nof 4 for each entity using the random walking strategy and Page Rank sampling\nstrategy, the following code snippet can be used:\n\n.. code:: python\n\n   from pyrdf2vec.samplers import PageRankSampler\n   from pyrdf2vec.walkers import RandomWalker\n\n   walkers = [RandomWalker(4, 10, PageRankSampler())]\n\n.. getting-started-end\n\nSpeed up the Extraction of Walks\n--------------------------------\n\nThe extraction of walks can take hours, days if not more in some cases. That\'s\nwhy it is important to use certain attributes and optimize ``pyRDF2Vec``\nparameters as much as possible according to your use cases.\n\nThis section aims to help you to set up these parameters with some advice.\n\nConfigure the ``n_jobs`` attribute to use multiple processors\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nBy default multiprocessing is disabled (``n_jobs=1``). If your machine allows\nit, it is recommended to use multiprocessing by incrementing the number of\nprocessors used for the extraction of walks:\n\n.. code:: python\n\n   from pyrdf2vec.walkers import RandomWalker\n\n   RDF2VecTransformer(walkers=[RandomWalker(4, 10, n_jobs=4)])\n\nIn the above snippet, the random walking strategy will use 4 processors to\nextract the walks, whether for a local or remote KG.\n\n**WARNING: using a large number of processors may violate the policy of some\nSPARQL endpoint servers**. This being that using multiprocessing means that\neach processor will send a SPARQL request to one server to fetch the hops of\nthe entity it is processing. Therefore, since these requests may take place in\na short time, this server could consider them as a Denial-Of-Service attack\n(DOS). Of course, these risks are multiplied in the absence of cache and when\nthe entities to be treated are of a consequent number.\n\nBundle SPARQL requests\n~~~~~~~~~~~~~~~~~~~~~~\n\nBy default the SPARQL requests bundling is disabled\n(``mul_req=False``). However, if you are using a remote KG and have a large\nnumber of entities, this option can greatly speed up the extraction of walks:\n\n.. code:: python\n\n   import pandas as pd\n\n   from pyrdf2vec import RDF2VecTransformer\n   from pyrdf2vec.graphs import KG\n   from pyrdf2vec.walkers import RandomWalker\n\n   data = pd.read_csv("samples/countries-cities/entities.tsv", sep="\\t")\n\n\n   RDF2VecTransformer(walkers=[RandomWalker(4, 10)]).fit_transform(\n       KG("https://dbpedia.org/sparql", mul_req=True),\n       [entity for entity in data["location"]],\n   )\n\nIn the above snippet, the KG specifies to the internal connector that it uses,\nto fetch the hops of the specified entities in an asynchronous way. These hops\nwill then be stored in cache and be accessed by the walking strategy to\naccelerate the extraction of walks for these entities.\n\n**WARNING: bundling SPARQL requests for a number of entities considered too\nlarge can may violate the policy of some SPARQL endpoint servers**. As for the\nuse of multiprocessing (which can be combined with ``mul_req``), sending a\nlarge number of SPARQL requests simultaneously could be seen by a server as a\nDOS. Be aware that the number of entities you have in your file corresponds to\nthe number of simultaneous requests that will be made and stored in cache.\n\nModify the Cache Settings\n~~~~~~~~~~~~~~~~~~~~~~~~~\n\nBy default, ``pyRDF2Vec`` uses a cache that provides a `Least Recently Used\n(LRU) <https://www.interviewcake.com/concept/java/lru-cache>`__ policy, with a\nsize that can hold 1024 entries, and a Time To Live (TTL) of 1200 seconds. For\nsome use cases, you would probably want to modify the `cache policy\n<https://cachetools.readthedocs.io/en/stable/>`__, increase (or decrease) the\ncache size and/or change the TTL:\n\n.. code:: python\n\n   import pandas as pd\n   from cachetools import MRUCache\n\n   from pyrdf2vec import RDF2VecTransformer\n   from pyrdf2vec.graphs import KG\n   from pyrdf2vec.walkers import RandomWalker\n\n   data = pd.read_csv("samples/countries-cities/entities.tsv", sep="\\t")\n\n   RDF2VecTransformer(walkers=[RandomWalker(4, 10)]).fit_transform(\n       KG("https://dbpedia.org/sparql", cache=MRUCache(maxsize=2048),\n       [entity for entity in data["location"]],\n   )\n\nModify the Walking Strategy Settings\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nBy default, ``pyRDF2Vec`` uses ``[RandomWalker(2, None, UniformSampler())]`` as\nwalking strategy. Using a greater maximum depth indicates a longer extraction\ntime for walks. Add to this that using ``max_walks=None``, extracts more walks\nand is faster in most cases than when giving a number (**SEE:** `FAQ <#faq>`__).\n\nIn some cases, using another sampling strategy can speed up the extraction of\nwalks by assigning a higher weight to some paths than others:\n\n.. code:: python\n\n   import pandas as pd\n\n   from pyrdf2vec import RDF2VecTransformer\n   from pyrdf2vec.graphs import KG\n   from pyrdf2vec.samplers import PageRankSampler\n   from pyrdf2vec.walkers import RandomWalker\n\n   data = pd.read_csv("samples/countries-cities/entities.tsv", sep="\\t")\n\n   RDF2VecTransformer(\n       walkers=[RandomWalker(2, None, PageRankSampler())]\n   ).fit_transform(\n       KG("https://dbpedia.org/sparql"),\n       [entity for entity in data["location"]],\n   )\n\nSet Up a Local Server\n~~~~~~~~~~~~~~~~~~~~~\n\nLoading large RDF files into memory will cause memory issues. Remote KGs serve\nas a solution for larger KGs, but **using a public endpoint will be slower**\ndue to overhead caused by HTTP requests. For that reason, it is better to set\nup your own local server and use that for your "Remote" KG.\n\nTo set up such a server, a tutorial has been made `on our wiki\n<https://github.com/IBCNServices/pyRDF2Vec/wiki/Fast-generation-of-RDF2Vec-embeddings-with-a-SPARQL-endpoint>`__.\n\nDocumentation\n-------------\n\nFor more information on how to use ``pyRDF2Vec``, `visit our online documentation\n<https://pyrdf2vec.readthedocs.io/en/latest/>`__ which is automatically updated\nwith the latest version of the ``master`` branch.\n\nFrom then on, you will be able to learn more about the use of the\nmodules as well as their functions available to you.\n\nContributions\n-------------\n\nYour help in the development of ``pyRDF2Vec`` is more than welcome. In order to\nbetter understand how you can help either through pull requests and/or issues,\nplease take a look at the `CONTRIBUTING\n<https://github.com/IBCNServices/pyRDF2Vec/blob/master/CONTRIBUTING.rst>`__\nfile.\n\nFAQ\n---\nHow to Ensure the Generation of Similar Embeddings?\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n``pyRDF2Vec``\'s walking strategies, sampling strategies and Word2Vec work with\nrandomness. To get reproducible embeddings, you firstly need to **use a seed** to\nensure determinism:\n\n.. code:: bash\n\n   PYTHONHASHSEED=42 python foo.py\n\nAdded to this, you must **also specify a random state** to the walking strategy\nwhich will implicitly use it for the sampling strategy:\n\n.. code:: python\n\n   from pyrdf2vec.walkers import RandomWalker\n\n   RandomWalker(2, None, random_state=42)\n\n**NOTE:** the ``PYTHONHASHSEED`` (e.g., 42) is to ensure determinism.\n\nFinally, to ensure random determinism for Word2Vec, you must **specify a single\nworker**:\n\n.. code:: python\n\n   from pyrdf2vec.embedders import Word2Vec\n\n   Word2Vec(workers=1)\n\n**NOTE:** using the ``n_jobs`` and ``mul_req`` parameters does not affect the\nrandom determinism.\n\nWhy the extraction time of walks is faster if ``max_walks=None``?\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nCurrently, **the BFS function** (using the Breadth-first search algorithm) is used\nwhen ``max_walks=None`` which is significantly **faster** than the DFS function\n(using the Depth-first search algorithm) **and extract more walks**.\n\nWe hope that this algorithmic complexity issue will be solved for the next\nrelease of ``pyRDf2Vec``\n\nReferencing\n-----------\n\nIf you use ``pyRDF2Vec`` in a scholarly article, we would appreciate a\ncitation:\n\n.. code:: bibtex\n\n   @inproceedings{pyrdf2vec,\n     author       = {Gilles Vandewiele and Bram Steenwinckel and Terencio Agozzino\n                     and Michael Weyns and Pieter Bonte and Femke Ongenae\n                     and Filip De Turck},\n     title        = {{pyRDF2Vec: Python Implementation and Extension of RDF2Vec}},\n     organization = {IDLab},\n     year         = {2020},\n     url          = {https://github.com/IBCNServices/pyRDF2Vec}\n   }\n',
    'author': 'Gilles Vandewiele',
    'author_email': 'gilles.vandewiele@ugent.be',
    'maintainer': 'Gilles Vandewiele',
    'maintainer_email': 'gilles.vandewiele@ugent.be',
    'url': 'https://github.com/IBCNServices/pyRDF2Vec',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
