import os, re
import openapiart.goserver.string_util as util
import openapiart.goserver.generator_context as ctx
from openapiart.goserver.writer import Writer

class GoServerInterfaceGenerator(object):

    def __init__(self, context):
        # type: (ctx.GeneratorContext) -> None
        self._indent = '\t'
        self._models_prefix = context.models_prefix
        self._root_package = context.module_path
        self._package_name = "interfaces"
        self._ctx = context
        self._output_path = os.path.join(context.output_path, 'interfaces')
    
    def generate(self):
        self._write_interfaces()

    def _write_interfaces(self):
        if not os.path.exists(self._output_path):
            os.makedirs(self._output_path)
        for ctrl in self._ctx.controllers:
            self._write_interface(ctrl)

    def _write_interface(self, ctrl):
        # type: (ctx.Controller) -> None
        filename = ctrl.yamlname.lower() + "_interface.go"
        fullname = os.path.join(self._output_path, filename)
        w = Writer(self._indent)
        self._write_header(w)
        self._write_import(w)
        self._write_path_param_const(w, ctrl)
        self._write_controller_interface(w, ctrl)
        self._write_servicehandler_interface(w, ctrl)
        with open(fullname, 'w') as file:
            print("Interface: {}".format(fullname))
            for line in w.strings:
                file.write(line + '\n')
            pass
        pass

    def _write_header(self, w):
        # type: (Writer) -> None
        w.write_line(
            "// This file is autogenerated. Do not modify",
            "package {name}".format(name=self._package_name),
            ""
        )

    def _write_import(self, w):
        # type: (Writer) -> None
        w.write_line(
            "import ("
        ).push_indent(
        ).write_line(
            '"net/http"',
            '"{pkg}/httpapi"'.format(pkg=self._root_package),
            """{prefix} \"{path}\"""".format(
                prefix=re.sub("[.]", "", self._ctx.models_prefix),
                path=self._ctx.models_path
            ),
        ).pop_indent(
        ).write_line(
            ")",
            ""
        )

    def _write_path_param_const(self, w, ctrl):
        # type: (Writer, ctx.Controller) -> None
        params = []  # type: [str]
        for r in ctrl.routes:
            for param in r.route_parameters:
                if param not in params:
                    params.append(param)
        if len(params) > 0:
            w.write_line("const (")
            w.push_indent()
            for param in params:
                w.write_line("""{ps_yamlname}{ps_param} = \"{param}\"""".format(
                    ps_yamlname=util.pascal_case(ctrl.yamlname),
                    ps_param=util.pascal_case(param),
                    param=param
                ))
            w.pop_indent()
            w.write_line(")", "")
        pass

    def _write_route_description(self, w, r):
        # type: (Writer, ctx.ControllerRoute) -> None
        w.write_line("/*")
        w.write_line("""{operation_name}: {method} {url}""".format(
            operation_name=r.operation_name,
            method=r.method,
            url=r.url
        ))
        w.write_line("Description: " + r.description)
        w.write_line("*/")

    def _write_controller_interface(self, w, ctrl):
        # type: (Writer, ctx.Controller) -> None
        w.write_line(
            "type {name} interface {{".format(name=ctrl.controller_name),
        )
        w.push_indent()
        w.write_line(
            "Routes() []httpapi.Route",
        )
        for r in ctrl.routes:
            # self._write_route_description(w, r)
            w.write_line(
                "{operation_name}(w http.ResponseWriter, r *http.Request)".format(
                    operation_name=r.operation_name
                ),
            )
        w.pop_indent()
        w.write_line(
            "}",
            ""
        )

    def _write_servicehandler_interface(self, w, ctrl):
        # type: (Writer, ctx.Controller) -> None
        w.write_line(
            "type {name} interface {{".format(
                name=ctrl.service_handler_name
            )
        )
        w.push_indent()
        w.write_line(
            "GetController() {name}".format(
                name=ctrl.controller_name
            )
        )
        for r in ctrl.routes:
            self._write_route_description(w, r)
            full_responsename = r.full_responsename
            request_body = r.requestBody()   # type: ctx.Component
            if request_body != None:
                full_requestname = request_body.full_model_name
                w.write_line(
                    """{operation_name}(rbody {full_requestname}, r *http.Request) {full_responsename}""".format(
                        operation_name=r.operation_name,
                        full_requestname=full_requestname,
                        full_responsename=full_responsename
                    )
                )
            else:
                w.write_line(
                    """{operation_name}(r *http.Request) {full_responsename}""".format(
                        operation_name=r.operation_name,
                        full_responsename=full_responsename
                    )
                )
        w.pop_indent()
        w.write_line(
            "}",
            ""
        )


