"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateJobExecutor = exports.FargateRunner = exports.JobExecutorImage = exports.FargateCapacityProviderType = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * Amazon ECS Capacity Providers for AWS Fargate.
 *
 * @stability stable
 */
var FargateCapacityProviderType;
(function (FargateCapacityProviderType) {
    FargateCapacityProviderType["FARGATE"] = "FARGATE";
    FargateCapacityProviderType["FARGATE_SPOT"] = "FARGATE_SPOT";
})(FargateCapacityProviderType = exports.FargateCapacityProviderType || (exports.FargateCapacityProviderType = {}));
/**
 * The docker image for the job executor.
 *
 * @stability stable
 */
class JobExecutorImage {
    /**
     *
     * @param image job executor image URI
     */
    constructor(uri) {
        this.uri = uri;
    }
    /**
     * Custom image.
     *
     * @param image custom image registry URI.
     * @stability stable
     */
    static of(image) { return new JobExecutorImage(image); }
}
exports.JobExecutorImage = JobExecutorImage;
/**
 * Debian.
 *
 * @see https://gitlab.com/tmaczukin-test-projects/fargate-driver-debian
 * @stability stable
 */
JobExecutorImage.DEBIAN = JobExecutorImage.of('registry.gitlab.com/tmaczukin-test-projects/fargate-driver-debian:latest');
/**
 * Node.
 *
 * @see https://gitlab.com/aws-fargate-driver-demo/docker-nodejs-gitlab-ci-fargate
 * @stability stable
 */
JobExecutorImage.NODE = JobExecutorImage.of('registry.gitlab.com/aws-fargate-driver-demo/docker-nodejs-gitlab-ci-fargate:latest');
/**
 * JSII for AWS CDK.
 *
 * @see https://gitlab.com/pahud/docker-jsii-cdk-gitlab-ci-fargate
 * @stability stable
 */
JobExecutorImage.JSII = JobExecutorImage.of('registry.gitlab.com/pahud/docker-jsii-cdk-gitlab-ci-fargate:latest');
const CLUSTER_DEFAULT_CAPACITY_PROVIDER = [
    FargateCapacityProviderType.FARGATE,
    FargateCapacityProviderType.FARGATE_SPOT,
];
const DEFAULT_CLUSTER_CAPACITY_PROVIDER_STRATEGY = [{
        base: 0,
        weight: 1,
        capacityProvider: FargateCapacityProviderType.FARGATE_SPOT,
    }];
const DEFAULT_SERVICE_CAPACITY_PROVIDER_STRATEGY = [
    { base: 0, weight: 0, capacityProvider: FargateCapacityProviderType.FARGATE },
    { weight: 1, capacityProvider: FargateCapacityProviderType.FARGATE_SPOT },
];
/**
 * The FargateRunner.
 *
 * @stability stable
 */
class FargateRunner extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t;
        super(scope, id);
        this.vpc = props.vpc;
        const stack = cdk.Stack.of(this);
        const fargateSubnet = this.vpc.selectSubnets(props.fargateJobSubnet);
        const cluster = new ecs.Cluster(this, 'Cluster', { vpc: this.vpc });
        const cfnCluster = cluster.node.tryFindChild('Resource');
        cfnCluster.addPropertyOverride('CapacityProviders', CLUSTER_DEFAULT_CAPACITY_PROVIDER);
        cfnCluster.addPropertyOverride('DefaultCapacityProviderStrategy', (_a = props.serviceDefaultCapacityProviderStrategy) !== null && _a !== void 0 ? _a : DEFAULT_CLUSTER_CAPACITY_PROVIDER_STRATEGY);
        const runnerTask = new ecs.FargateTaskDefinition(this, 'RunnerTask', {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        const fargateSecurityGroup = (_b = props.securityGroup) !== null && _b !== void 0 ? _b : this.createSecurityGroup();
        const registrationToken = (_c = props.registrationToken) !== null && _c !== void 0 ? _c : (this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN);
        if (!registrationToken) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        runnerTask.addContainer('runner', {
            image: ecs.ContainerImage.fromRegistry('registry.gitlab.com/pahud/docker-gitlab-runner-fargate-driver'),
            logging: new ecs.AwsLogDriver({ streamPrefix: 'GitlabRunnerManager', logRetention: 7 }),
            environment: {
                GITLAB_REGISTRATION_TOKEN: registrationToken,
                GITLAB_URL: (_d = props.gitlabURL) !== null && _d !== void 0 ? _d : 'https://gitlab.com',
                FARGATE_REGION: (_f = (_e = props.executor) === null || _e === void 0 ? void 0 : _e.region) !== null && _f !== void 0 ? _f : stack.region,
                FARGATE_CLUSTER: (_j = (_h = (_g = props.executor) === null || _g === void 0 ? void 0 : _g.cluster) === null || _h === void 0 ? void 0 : _h.clusterName) !== null && _j !== void 0 ? _j : cluster.clusterName,
                FARGATE_SUBNET: (_m = (_l = (_k = props.executor) === null || _k === void 0 ? void 0 : _k.subnet) === null || _l === void 0 ? void 0 : _l.subnetId) !== null && _m !== void 0 ? _m : fargateSubnet.subnetIds[0],
                FARGATE_SECURITY_GROUP: (_q = (_p = (_o = props.executor) === null || _o === void 0 ? void 0 : _o.securityGroup) === null || _p === void 0 ? void 0 : _p.securityGroupId) !== null && _q !== void 0 ? _q : fargateSecurityGroup.securityGroupId,
                FARGATE_TASK_DEFINITION: ((_r = props.executor) === null || _r === void 0 ? void 0 : _r.taskDefinitionArn) || new FargateJobExecutor(this, 'JobExecutor').taskDefinitionArn,
                RUNNER_TAG_LIST: this.synthesizeTags((_s = props.tags) !== null && _s !== void 0 ? _s : ['fargate', 'gitlab', 'aws', 'docker']),
            },
        });
        runnerTask.taskRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonECS_FullAccess'));
        const svc = new ecs.FargateService(this, 'RunnerManagerService', {
            cluster,
            taskDefinition: runnerTask,
            securityGroups: [fargateSecurityGroup],
        });
        const cfnService = svc.node.tryFindChild('Service');
        cfnService.addPropertyDeletionOverride('LaunchType');
        cfnService.addPropertyOverride('CapacityProviderStrategy', (_t = props.serviceDefaultCapacityProviderStrategy) !== null && _t !== void 0 ? _t : DEFAULT_SERVICE_CAPACITY_PROVIDER_STRATEGY);
    }
    synthesizeTags(tags) {
        return tags.join(',');
    }
    createSecurityGroup() {
        const sg = new ec2.SecurityGroup(this, 'FargateSecurityGroup', { vpc: this.vpc });
        sg.connections.allowInternally(ec2.Port.allTraffic());
        return sg;
    }
}
exports.FargateRunner = FargateRunner;
/**
 * The FargateJobExecutor.
 *
 * @stability stable
 */
class FargateJobExecutor extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.region = (_a = props.region) !== null && _a !== void 0 ? _a : cdk.Stack.of(this).region;
        this.cluster = props.cluster;
        this.subnet = props.subnet;
        this.securityGroup = props.securityGroup;
        const task = new ecs.FargateTaskDefinition(this, 'JobsTask', {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        this.taskDefinitionArn = task.taskDefinitionArn;
        const image = (_b = props.image) !== null && _b !== void 0 ? _b : JobExecutorImage.DEBIAN;
        task.addContainer('ci-coordinator', {
            image: ecs.ContainerImage.fromRegistry(image.uri.toString()),
            logging: new ecs.AwsLogDriver({ streamPrefix: 'GitlabRunnerJob', logRetention: 7 }),
        }).addPortMappings({ containerPort: 22 });
    }
}
exports.FargateJobExecutor = FargateJobExecutor;
//# sourceMappingURL=data:application/json;base64,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