"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const eks = require("@aws-cdk/aws-eks");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const runner = require("./runner");
const gitLabClusterRoleBinding = {
    apiVersion: 'rbac.authorization.k8s.io/v1beta1',
    kind: 'ClusterRoleBinding',
    metadata: { name: 'gitlab-admin' },
    roleRef: {
        apiGroup: 'rbac.authorization.k8s.io',
        kind: 'ClusterRole',
        name: 'cluster-admin',
    },
    subjects: [
        {
            kind: 'ServiceAccount',
            name: 'gitlab',
            namespace: 'kube-system',
        },
    ],
};
/**
 * The Provider to create GitLab Integrations with AWS.
 *
 * @stability stable
 */
class Provider extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { natGateways: 1 });
        new cdk.CfnOutput(this, 'Region', { value: stack.region });
    }
    /**
     * @stability stable
     */
    createFargateRunner(executor) {
        const token = this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN;
        if (!token) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        new runner.FargateRunner(this, 'FargateRunner', {
            vpc: this.vpc,
            registrationToken: this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN'),
            executor,
        });
    }
    /**
     * @stability stable
     */
    createEksCluster(scope, id, props) {
        const cluster = new eks.Cluster(scope, id, props.clusterOptions);
        if (props.rbac != false) {
            cluster.addServiceAccount('gitlab');
            cluster.addManifest('ClusterRoleBinding', gitLabClusterRoleBinding);
        }
        return cluster;
    }
    /**
     * @stability stable
     */
    createFargateEksCluster(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const cluster = new eks.FargateCluster(scope, id, props.clusterOptions);
        if (props.rbac != false) {
            cluster.addServiceAccount('gitlab');
            cluster.addManifest('ClusterRoleBinding', gitLabClusterRoleBinding);
        }
        const registrationToken = (_b = (_a = props.helmRunnerOptions) === null || _a === void 0 ? void 0 : _a.registrationToken) !== null && _b !== void 0 ? _b : (this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN);
        if (!registrationToken) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        cluster.addHelmChart('helmrunner', {
            chart: 'gitlab-runner',
            repository: 'https://charts.gitlab.io',
            namespace: 'default',
            values: {
                // use fargate run job ,always need to pull.
                imagePullPolicy: 'Always',
                terminationGracePeriodSeconds: 3600,
                concurrent: (_d = (_c = props.helmRunnerOptions) === null || _c === void 0 ? void 0 : _c.concurrent) !== null && _d !== void 0 ? _d : 10,
                checkInterval: 30,
                gitlabUrl: (_f = (_e = props.helmRunnerOptions) === null || _e === void 0 ? void 0 : _e.gitlabURL) !== null && _f !== void 0 ? _f : 'https://gitlab.com/',
                runnerRegistrationToken: registrationToken,
                unregisterRunners: true,
                rbac: {
                    create: true,
                    clusterWideAccess: false,
                },
                metrics: {
                    enabled: true,
                },
                securityContext: {
                    fsGroup: 65533,
                    runAsUser: 100,
                },
                // runners default image when job start not set "image" in gitlab-ci.yaml.
                runners: {
                    image: (_h = (_g = props.helmRunnerOptions) === null || _g === void 0 ? void 0 : _g.jobDefaultImage) !== null && _h !== void 0 ? _h : 'alpine:3.11',
                    tags: this.synthesizeTags((_k = (_j = props.helmRunnerOptions) === null || _j === void 0 ? void 0 : _j.tags) !== null && _k !== void 0 ? _k : ['eks', 'fargate', 'runner']),
                    privileged: false,
                },
                envVars: [
                    {
                        name: 'RUNNER_EXECUTOR',
                        value: 'kubernetes',
                    },
                ],
            },
        });
        return cluster;
    }
    /**
     * @stability stable
     */
    createSecurityGroup() {
        const sg = new ec2.SecurityGroup(this, 'GitlabEksSecurityGroup', {
            vpc: this.vpc,
        });
        new cdk.CfnOutput(this, 'SecurityGroup', { value: sg.securityGroupName });
        return sg;
    }
    /**
     * @stability stable
     */
    createEksServiceRole() {
        return new iam.Role(this, 'GitlabEksServiceRole', {
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy')],
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
        });
    }
    // this.gitlabEksRole = new iam.Role(this, 'GitlabManagedEksRole', {
    //   assumedBy: new iam.AccountPrincipal(props.accountId),
    //   externalIds: [props.externalId],
    //   inlinePolicies: {
    //     default: policy,
    //   },
    // });
    //   new cdk.CfnOutput(this, 'RoleArn', { value: this.gitlabEksRole.roleArn });
    // }
    /**
     * @stability stable
     */
    createGitlabManagedEksRole(props) {
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'autoscaling:CreateAutoScalingGroup',
                        'autoscaling:DescribeAutoScalingGroups',
                        'autoscaling:DescribeScalingActivities',
                        'autoscaling:UpdateAutoScalingGroup',
                        'autoscaling:CreateLaunchConfiguration',
                        'autoscaling:DescribeLaunchConfigurations',
                        'cloudformation:CreateStack',
                        'cloudformation:DescribeStacks',
                        'ec2:AuthorizeSecurityGroupEgress',
                        'ec2:AuthorizeSecurityGroupIngress',
                        'ec2:RevokeSecurityGroupEgress',
                        'ec2:RevokeSecurityGroupIngress',
                        'ec2:CreateSecurityGroup',
                        'ec2:createTags',
                        'ec2:DescribeImages',
                        'ec2:DescribeKeyPairs',
                        'ec2:DescribeRegions',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeVpcs',
                        'eks:CreateCluster',
                        'eks:DescribeCluster',
                        'iam:AddRoleToInstanceProfile',
                        'iam:AttachRolePolicy',
                        'iam:CreateRole',
                        'iam:CreateInstanceProfile',
                        'iam:CreateServiceLinkedRole',
                        'iam:GetRole',
                        'iam:ListRoles',
                        'iam:PassRole',
                        'ssm:GetParameters',
                    ],
                    resources: ['*'],
                }),
            ],
        });
        this.gitlabEksRole = new iam.Role(this, 'GitlabManagedEksRole', {
            assumedBy: new iam.AccountPrincipal(props.accountId),
            externalIds: [props.externalId],
            inlinePolicies: {
                default: policy,
            },
        });
        const eksAdminRole = this.createEksServiceRole();
        this.createSecurityGroup();
        const subnetIds = this.vpc.selectSubnets({
            subnetType: ec2.SubnetType.PRIVATE,
        }).subnetIds;
        new cdk.CfnOutput(this, 'EksClusterSubnetIds', {
            value: subnetIds.join(','),
        });
        new cdk.CfnOutput(this, 'RoleArn', { value: this.gitlabEksRole.roleArn });
        new cdk.CfnOutput(this, 'EksAdminRole', { value: eksAdminRole.roleName });
    }
    synthesizeTags(tags) {
        return tags.join(',');
    }
}
exports.Provider = Provider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFpbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9tYWluLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHFDQUFxQztBQUNyQyxtQ0FBbUM7QUF5Rm5DLE1BQU0sd0JBQXdCLEdBQUc7SUFDL0IsVUFBVSxFQUFFLG1DQUFtQztJQUMvQyxJQUFJLEVBQUUsb0JBQW9CO0lBQzFCLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7SUFDbEMsT0FBTyxFQUFFO1FBQ1AsUUFBUSxFQUFFLDJCQUEyQjtRQUNyQyxJQUFJLEVBQUUsYUFBYTtRQUNuQixJQUFJLEVBQUUsZUFBZTtLQUN0QjtJQUNELFFBQVEsRUFBRTtRQUNSO1lBQ0UsSUFBSSxFQUFFLGdCQUFnQjtZQUN0QixJQUFJLEVBQUUsUUFBUTtZQUNkLFNBQVMsRUFBRSxhQUFhO1NBQ3pCO0tBQ0Y7Q0FDRixDQUFDOzs7Ozs7QUFLRixNQUFhLFFBQVMsU0FBUSxHQUFHLENBQUMsU0FBUzs7OztJQUd6QyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLFFBQXVCLEVBQUU7O1FBQ3JFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakMsSUFBSSxDQUFDLEdBQUcsU0FBRyxLQUFLLENBQUMsR0FBRyxtQ0FBSSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXJFLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBRTdELENBQUM7Ozs7SUFDTSxtQkFBbUIsQ0FBQyxRQUFvQztRQUM3RCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUM7UUFDNUcsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNWLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELENBQUMsQ0FBQztTQUM5RTtRQUNELElBQUksTUFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQzlDLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLGlCQUFpQixFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDJCQUEyQixDQUFDO1lBQ3ZFLFFBQVE7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7O0lBQ00sZ0JBQWdCLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDaEYsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ2pFLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxLQUFLLEVBQUU7WUFDdkIsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3BDLE9BQU8sQ0FBQyxXQUFXLENBQUMsb0JBQW9CLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztTQUNyRTtRQUNELE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7Ozs7SUFDTSx1QkFBdUIsQ0FBQyxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUErQjs7UUFDOUYsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3hFLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxLQUFLLEVBQUU7WUFDdkIsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3BDLE9BQU8sQ0FBQyxXQUFXLENBQUMsb0JBQW9CLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztTQUNyRTtRQUNELE1BQU0saUJBQWlCLGVBQUcsS0FBSyxDQUFDLGlCQUFpQiwwQ0FBRSxpQkFBaUIsbUNBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUV4SyxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsT0FBTyxDQUFDLFlBQVksQ0FBQyxZQUFZLEVBQUU7WUFDakMsS0FBSyxFQUFFLGVBQWU7WUFDdEIsVUFBVSxFQUFFLDBCQUEwQjtZQUN0QyxTQUFTLEVBQUUsU0FBUztZQUNwQixNQUFNLEVBQUU7Z0JBQ04sNENBQTRDO2dCQUM1QyxlQUFlLEVBQUUsUUFBUTtnQkFDekIsNkJBQTZCLEVBQUUsSUFBSTtnQkFDbkMsVUFBVSxjQUFFLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsVUFBVSxtQ0FBSSxFQUFFO2dCQUNyRCxhQUFhLEVBQUUsRUFBRTtnQkFDakIsU0FBUyxjQUFFLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsU0FBUyxtQ0FBSSxxQkFBcUI7Z0JBQ3RFLHVCQUF1QixFQUFFLGlCQUFpQjtnQkFDMUMsaUJBQWlCLEVBQUUsSUFBSTtnQkFDdkIsSUFBSSxFQUFFO29CQUNKLE1BQU0sRUFBRSxJQUFJO29CQUNaLGlCQUFpQixFQUFFLEtBQUs7aUJBQ3pCO2dCQUNELE9BQU8sRUFBRTtvQkFDUCxPQUFPLEVBQUUsSUFBSTtpQkFDZDtnQkFDRCxlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFLEtBQUs7b0JBQ2QsU0FBUyxFQUFFLEdBQUc7aUJBQ2Y7Z0JBQ0QsMEVBQTBFO2dCQUMxRSxPQUFPLEVBQUU7b0JBQ1AsS0FBSyxjQUFFLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsZUFBZSxtQ0FBSSxhQUFhO29CQUNoRSxJQUFJLEVBQUUsSUFBSSxDQUFDLGNBQWMsYUFBQyxLQUFLLENBQUMsaUJBQWlCLDBDQUFFLElBQUksbUNBQUksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO29CQUN4RixVQUFVLEVBQUUsS0FBSztpQkFDbEI7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQO3dCQUNFLElBQUksRUFBRSxpQkFBaUI7d0JBQ3ZCLEtBQUssRUFBRSxZQUFZO3FCQUNwQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQzs7OztJQUNNLG1CQUFtQjtRQUN4QixNQUFNLEVBQUUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO1lBQy9ELEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztTQUNkLENBQUMsQ0FBQztRQUNILElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLEVBQUUsS0FBSyxFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDMUUsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDOzs7O0lBQ00sb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRTtZQUNoRCxlQUFlLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLHdCQUF3QixDQUFDLENBQUM7WUFDdkYsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO1NBQ3pELENBQUMsQ0FBQztJQUNMLENBQUM7SUFDRCxvRUFBb0U7SUFDcEUsMERBQTBEO0lBQzFELHFDQUFxQztJQUNyQyxzQkFBc0I7SUFDdEIsdUJBQXVCO0lBQ3ZCLE9BQU87SUFDUCxNQUFNO0lBQ04sK0VBQStFO0lBQy9FLElBQUk7Ozs7SUFDRywwQkFBMEIsQ0FBQyxLQUFnQjtRQUNoRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7WUFDcEMsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFO3dCQUNQLG9DQUFvQzt3QkFDcEMsdUNBQXVDO3dCQUN2Qyx1Q0FBdUM7d0JBQ3ZDLG9DQUFvQzt3QkFDcEMsdUNBQXVDO3dCQUN2QywwQ0FBMEM7d0JBQzFDLDRCQUE0Qjt3QkFDNUIsK0JBQStCO3dCQUMvQixrQ0FBa0M7d0JBQ2xDLG1DQUFtQzt3QkFDbkMsK0JBQStCO3dCQUMvQixnQ0FBZ0M7d0JBQ2hDLHlCQUF5Qjt3QkFDekIsZ0JBQWdCO3dCQUNoQixvQkFBb0I7d0JBQ3BCLHNCQUFzQjt3QkFDdEIscUJBQXFCO3dCQUNyQiw0QkFBNEI7d0JBQzVCLHFCQUFxQjt3QkFDckIsa0JBQWtCO3dCQUNsQixtQkFBbUI7d0JBQ25CLHFCQUFxQjt3QkFDckIsOEJBQThCO3dCQUM5QixzQkFBc0I7d0JBQ3RCLGdCQUFnQjt3QkFDaEIsMkJBQTJCO3dCQUMzQiw2QkFBNkI7d0JBQzdCLGFBQWE7d0JBQ2IsZUFBZTt3QkFDZixjQUFjO3dCQUNkLG1CQUFtQjtxQkFDcEI7b0JBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNqQixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7WUFDOUQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDcEQsV0FBVyxFQUFFLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztZQUMvQixjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFLE1BQU07YUFDaEI7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztRQUNqRCxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUMzQixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQztZQUN2QyxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxPQUFPO1NBQ25DLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDYixJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQzdDLEtBQUssRUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztTQUMzQixDQUFDLENBQUM7UUFDSCxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDMUUsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUUsRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUNPLGNBQWMsQ0FBQyxJQUFjO1FBQ25DLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN4QixDQUFDO0NBQ0Y7QUF6S0QsNEJBeUtDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWtzIGZyb20gJ0Bhd3MtY2RrL2F3cy1la3MnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgcnVubmVyIGZyb20gJy4vcnVubmVyJztcblxuZXhwb3J0IGludGVyZmFjZSBQcm92aWRlclByb3BzIHtcbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUm9sZVByb3BzIHtcbiAgcmVhZG9ubHkgYWNjb3VudElkOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGV4dGVybmFsSWQ6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBFa3NDbHVzdGVyT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmJhYz86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsdXN0ZXJPcHRpb25zOiBla3MuQ2x1c3RlclByb3BzO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEhlbG1SdW5uZXJPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhZ3M/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVnaXN0cmF0aW9uVG9rZW4/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZ2l0bGFiVVJMPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbmN1cnJlbnQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGpvYkRlZmF1bHRJbWFnZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbmFtZXNwYWNlPzogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEZhcmdhdGVFa3NDbHVzdGVyT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmJhYz86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsdXN0ZXJPcHRpb25zOiBla3MuRmFyZ2F0ZUNsdXN0ZXJQcm9wcztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaGVsbVJ1bm5lck9wdGlvbnM/OiBIZWxtUnVubmVyT3B0aW9ucztcbn1cblxuY29uc3QgZ2l0TGFiQ2x1c3RlclJvbGVCaW5kaW5nID0ge1xuICBhcGlWZXJzaW9uOiAncmJhYy5hdXRob3JpemF0aW9uLms4cy5pby92MWJldGExJyxcbiAga2luZDogJ0NsdXN0ZXJSb2xlQmluZGluZycsXG4gIG1ldGFkYXRhOiB7IG5hbWU6ICdnaXRsYWItYWRtaW4nIH0sXG4gIHJvbGVSZWY6IHtcbiAgICBhcGlHcm91cDogJ3JiYWMuYXV0aG9yaXphdGlvbi5rOHMuaW8nLFxuICAgIGtpbmQ6ICdDbHVzdGVyUm9sZScsXG4gICAgbmFtZTogJ2NsdXN0ZXItYWRtaW4nLFxuICB9LFxuICBzdWJqZWN0czogW1xuICAgIHtcbiAgICAgIGtpbmQ6ICdTZXJ2aWNlQWNjb3VudCcsXG4gICAgICBuYW1lOiAnZ2l0bGFiJyxcbiAgICAgIG5hbWVzcGFjZTogJ2t1YmUtc3lzdGVtJyxcbiAgICB9LFxuICBdLFxufTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBQcm92aWRlciBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICBwdWJsaWMgZ2l0bGFiRWtzUm9sZT86IGlhbS5JUm9sZTtcbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBQcm92aWRlclByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2YodGhpcyk7XG5cbiAgICB0aGlzLnZwYyA9IHByb3BzLnZwYyA/PyBuZXcgZWMyLlZwYyh0aGlzLCAnVnBjJywgeyBuYXRHYXRld2F5czogMSB9KTtcblxuICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdSZWdpb24nLCB7IHZhbHVlOiBzdGFjay5yZWdpb24gfSk7XG5cbiAgfVxuICBwdWJsaWMgY3JlYXRlRmFyZ2F0ZVJ1bm5lcihleGVjdXRvcj86IHJ1bm5lci5GYXJnYXRlSm9iRXhlY3V0b3IpIHtcbiAgICBjb25zdCB0b2tlbiA9IHRoaXMubm9kZS50cnlHZXRDb250ZXh0KCdHSVRMQUJfUkVHSVNUUkFUSU9OX1RPS0VOJykgfHwgcHJvY2Vzcy5lbnYuR0lUTEFCX1JFR0lTVFJBVElPTl9UT0tFTjtcbiAgICBpZiAoIXRva2VuKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ21pc3NpbmcgR0lUTEFCX1JFR0lTVFJBVElPTl9UT0tFTiBpbiB0aGUgY29udGV4dCB2YXJpYWJsZScpO1xuICAgIH1cbiAgICBuZXcgcnVubmVyLkZhcmdhdGVSdW5uZXIodGhpcywgJ0ZhcmdhdGVSdW5uZXInLCB7XG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgcmVnaXN0cmF0aW9uVG9rZW46IHRoaXMubm9kZS50cnlHZXRDb250ZXh0KCdHSVRMQUJfUkVHSVNUUkFUSU9OX1RPS0VOJyksXG4gICAgICBleGVjdXRvcixcbiAgICB9KTtcbiAgfVxuICBwdWJsaWMgY3JlYXRlRWtzQ2x1c3RlcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEVrc0NsdXN0ZXJPcHRpb25zICk6IGVrcy5DbHVzdGVyIHtcbiAgICBjb25zdCBjbHVzdGVyID0gbmV3IGVrcy5DbHVzdGVyKHNjb3BlLCBpZCwgcHJvcHMuY2x1c3Rlck9wdGlvbnMpO1xuICAgIGlmIChwcm9wcy5yYmFjICE9IGZhbHNlKSB7XG4gICAgICBjbHVzdGVyLmFkZFNlcnZpY2VBY2NvdW50KCdnaXRsYWInKTtcbiAgICAgIGNsdXN0ZXIuYWRkTWFuaWZlc3QoJ0NsdXN0ZXJSb2xlQmluZGluZycsIGdpdExhYkNsdXN0ZXJSb2xlQmluZGluZyk7XG4gICAgfVxuICAgIHJldHVybiBjbHVzdGVyO1xuICB9XG4gIHB1YmxpYyBjcmVhdGVGYXJnYXRlRWtzQ2x1c3RlcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEZhcmdhdGVFa3NDbHVzdGVyT3B0aW9ucyk6IGVrcy5DbHVzdGVyIHtcbiAgICBjb25zdCBjbHVzdGVyID0gbmV3IGVrcy5GYXJnYXRlQ2x1c3RlcihzY29wZSwgaWQsIHByb3BzLmNsdXN0ZXJPcHRpb25zKTtcbiAgICBpZiAocHJvcHMucmJhYyAhPSBmYWxzZSkge1xuICAgICAgY2x1c3Rlci5hZGRTZXJ2aWNlQWNjb3VudCgnZ2l0bGFiJyk7XG4gICAgICBjbHVzdGVyLmFkZE1hbmlmZXN0KCdDbHVzdGVyUm9sZUJpbmRpbmcnLCBnaXRMYWJDbHVzdGVyUm9sZUJpbmRpbmcpO1xuICAgIH1cbiAgICBjb25zdCByZWdpc3RyYXRpb25Ub2tlbiA9IHByb3BzLmhlbG1SdW5uZXJPcHRpb25zPy5yZWdpc3RyYXRpb25Ub2tlbiA/PyAodGhpcy5ub2RlLnRyeUdldENvbnRleHQoJ0dJVExBQl9SRUdJU1RSQVRJT05fVE9LRU4nKSB8fCBwcm9jZXNzLmVudi5HSVRMQUJfUkVHSVNUUkFUSU9OX1RPS0VOKTtcblxuICAgIGlmICghcmVnaXN0cmF0aW9uVG9rZW4pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignbWlzc2luZyBHSVRMQUJfUkVHSVNUUkFUSU9OX1RPS0VOIGluIHRoZSBjb250ZXh0IHZhcmlhYmxlJyk7XG4gICAgfVxuXG4gICAgY2x1c3Rlci5hZGRIZWxtQ2hhcnQoJ2hlbG1ydW5uZXInLCB7XG4gICAgICBjaGFydDogJ2dpdGxhYi1ydW5uZXInLFxuICAgICAgcmVwb3NpdG9yeTogJ2h0dHBzOi8vY2hhcnRzLmdpdGxhYi5pbycsXG4gICAgICBuYW1lc3BhY2U6ICdkZWZhdWx0JyxcbiAgICAgIHZhbHVlczoge1xuICAgICAgICAvLyB1c2UgZmFyZ2F0ZSBydW4gam9iICxhbHdheXMgbmVlZCB0byBwdWxsLlxuICAgICAgICBpbWFnZVB1bGxQb2xpY3k6ICdBbHdheXMnLFxuICAgICAgICB0ZXJtaW5hdGlvbkdyYWNlUGVyaW9kU2Vjb25kczogMzYwMCxcbiAgICAgICAgY29uY3VycmVudDogcHJvcHMuaGVsbVJ1bm5lck9wdGlvbnM/LmNvbmN1cnJlbnQgPz8gMTAsIC8vIG51bWJlciBvZiBydW4gam9iIGluIHRoZSBzYW1lIHRpbWUuXG4gICAgICAgIGNoZWNrSW50ZXJ2YWw6IDMwLFxuICAgICAgICBnaXRsYWJVcmw6IHByb3BzLmhlbG1SdW5uZXJPcHRpb25zPy5naXRsYWJVUkwgPz8gJ2h0dHBzOi8vZ2l0bGFiLmNvbS8nLFxuICAgICAgICBydW5uZXJSZWdpc3RyYXRpb25Ub2tlbjogcmVnaXN0cmF0aW9uVG9rZW4sXG4gICAgICAgIHVucmVnaXN0ZXJSdW5uZXJzOiB0cnVlLFxuICAgICAgICByYmFjOiB7XG4gICAgICAgICAgY3JlYXRlOiB0cnVlLFxuICAgICAgICAgIGNsdXN0ZXJXaWRlQWNjZXNzOiBmYWxzZSxcbiAgICAgICAgfSxcbiAgICAgICAgbWV0cmljczoge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICAgIHNlY3VyaXR5Q29udGV4dDoge1xuICAgICAgICAgIGZzR3JvdXA6IDY1NTMzLFxuICAgICAgICAgIHJ1bkFzVXNlcjogMTAwLFxuICAgICAgICB9LFxuICAgICAgICAvLyBydW5uZXJzIGRlZmF1bHQgaW1hZ2Ugd2hlbiBqb2Igc3RhcnQgbm90IHNldCBcImltYWdlXCIgaW4gZ2l0bGFiLWNpLnlhbWwuXG4gICAgICAgIHJ1bm5lcnM6IHtcbiAgICAgICAgICBpbWFnZTogcHJvcHMuaGVsbVJ1bm5lck9wdGlvbnM/LmpvYkRlZmF1bHRJbWFnZSA/PyAnYWxwaW5lOjMuMTEnLFxuICAgICAgICAgIHRhZ3M6IHRoaXMuc3ludGhlc2l6ZVRhZ3MocHJvcHMuaGVsbVJ1bm5lck9wdGlvbnM/LnRhZ3MgPz8gWydla3MnLCAnZmFyZ2F0ZScsICdydW5uZXInXSksXG4gICAgICAgICAgcHJpdmlsZWdlZDogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICAgIGVudlZhcnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAnUlVOTkVSX0VYRUNVVE9SJyxcbiAgICAgICAgICAgIHZhbHVlOiAna3ViZXJuZXRlcycsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgcmV0dXJuIGNsdXN0ZXI7XG4gIH1cbiAgcHVibGljIGNyZWF0ZVNlY3VyaXR5R3JvdXAoKTogZWMyLlNlY3VyaXR5R3JvdXAge1xuICAgIGNvbnN0IHNnID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdHaXRsYWJFa3NTZWN1cml0eUdyb3VwJywge1xuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICB9KTtcbiAgICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnU2VjdXJpdHlHcm91cCcsIHsgdmFsdWU6IHNnLnNlY3VyaXR5R3JvdXBOYW1lIH0pO1xuICAgIHJldHVybiBzZztcbiAgfVxuICBwdWJsaWMgY3JlYXRlRWtzU2VydmljZVJvbGUoKTogaWFtLlJvbGUge1xuICAgIHJldHVybiBuZXcgaWFtLlJvbGUodGhpcywgJ0dpdGxhYkVrc1NlcnZpY2VSb2xlJywge1xuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25FS1NDbHVzdGVyUG9saWN5JyldLFxuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2Vrcy5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG4gIH1cbiAgLy8gdGhpcy5naXRsYWJFa3NSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdHaXRsYWJNYW5hZ2VkRWtzUm9sZScsIHtcbiAgLy8gICBhc3N1bWVkQnk6IG5ldyBpYW0uQWNjb3VudFByaW5jaXBhbChwcm9wcy5hY2NvdW50SWQpLFxuICAvLyAgIGV4dGVybmFsSWRzOiBbcHJvcHMuZXh0ZXJuYWxJZF0sXG4gIC8vICAgaW5saW5lUG9saWNpZXM6IHtcbiAgLy8gICAgIGRlZmF1bHQ6IHBvbGljeSxcbiAgLy8gICB9LFxuICAvLyB9KTtcbiAgLy8gICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnUm9sZUFybicsIHsgdmFsdWU6IHRoaXMuZ2l0bGFiRWtzUm9sZS5yb2xlQXJuIH0pO1xuICAvLyB9XG4gIHB1YmxpYyBjcmVhdGVHaXRsYWJNYW5hZ2VkRWtzUm9sZShwcm9wczogUm9sZVByb3BzKSB7XG4gICAgY29uc3QgcG9saWN5ID0gbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAnYXV0b3NjYWxpbmc6Q3JlYXRlQXV0b1NjYWxpbmdHcm91cCcsXG4gICAgICAgICAgICAnYXV0b3NjYWxpbmc6RGVzY3JpYmVBdXRvU2NhbGluZ0dyb3VwcycsXG4gICAgICAgICAgICAnYXV0b3NjYWxpbmc6RGVzY3JpYmVTY2FsaW5nQWN0aXZpdGllcycsXG4gICAgICAgICAgICAnYXV0b3NjYWxpbmc6VXBkYXRlQXV0b1NjYWxpbmdHcm91cCcsXG4gICAgICAgICAgICAnYXV0b3NjYWxpbmc6Q3JlYXRlTGF1bmNoQ29uZmlndXJhdGlvbicsXG4gICAgICAgICAgICAnYXV0b3NjYWxpbmc6RGVzY3JpYmVMYXVuY2hDb25maWd1cmF0aW9ucycsXG4gICAgICAgICAgICAnY2xvdWRmb3JtYXRpb246Q3JlYXRlU3RhY2snLFxuICAgICAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tzJyxcbiAgICAgICAgICAgICdlYzI6QXV0aG9yaXplU2VjdXJpdHlHcm91cEVncmVzcycsXG4gICAgICAgICAgICAnZWMyOkF1dGhvcml6ZVNlY3VyaXR5R3JvdXBJbmdyZXNzJyxcbiAgICAgICAgICAgICdlYzI6UmV2b2tlU2VjdXJpdHlHcm91cEVncmVzcycsXG4gICAgICAgICAgICAnZWMyOlJldm9rZVNlY3VyaXR5R3JvdXBJbmdyZXNzJyxcbiAgICAgICAgICAgICdlYzI6Q3JlYXRlU2VjdXJpdHlHcm91cCcsXG4gICAgICAgICAgICAnZWMyOmNyZWF0ZVRhZ3MnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZUltYWdlcycsXG4gICAgICAgICAgICAnZWMyOkRlc2NyaWJlS2V5UGFpcnMnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZVJlZ2lvbnMnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZVNlY3VyaXR5R3JvdXBzJyxcbiAgICAgICAgICAgICdlYzI6RGVzY3JpYmVTdWJuZXRzJyxcbiAgICAgICAgICAgICdlYzI6RGVzY3JpYmVWcGNzJyxcbiAgICAgICAgICAgICdla3M6Q3JlYXRlQ2x1c3RlcicsXG4gICAgICAgICAgICAnZWtzOkRlc2NyaWJlQ2x1c3RlcicsXG4gICAgICAgICAgICAnaWFtOkFkZFJvbGVUb0luc3RhbmNlUHJvZmlsZScsXG4gICAgICAgICAgICAnaWFtOkF0dGFjaFJvbGVQb2xpY3knLFxuICAgICAgICAgICAgJ2lhbTpDcmVhdGVSb2xlJyxcbiAgICAgICAgICAgICdpYW06Q3JlYXRlSW5zdGFuY2VQcm9maWxlJyxcbiAgICAgICAgICAgICdpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGUnLFxuICAgICAgICAgICAgJ2lhbTpHZXRSb2xlJyxcbiAgICAgICAgICAgICdpYW06TGlzdFJvbGVzJyxcbiAgICAgICAgICAgICdpYW06UGFzc1JvbGUnLFxuICAgICAgICAgICAgJ3NzbTpHZXRQYXJhbWV0ZXJzJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRoaXMuZ2l0bGFiRWtzUm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnR2l0bGFiTWFuYWdlZEVrc1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uQWNjb3VudFByaW5jaXBhbChwcm9wcy5hY2NvdW50SWQpLFxuICAgICAgZXh0ZXJuYWxJZHM6IFtwcm9wcy5leHRlcm5hbElkXSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIGRlZmF1bHQ6IHBvbGljeSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgZWtzQWRtaW5Sb2xlID0gdGhpcy5jcmVhdGVFa3NTZXJ2aWNlUm9sZSgpO1xuICAgIHRoaXMuY3JlYXRlU2VjdXJpdHlHcm91cCgpO1xuICAgIGNvbnN0IHN1Ym5ldElkcyA9IHRoaXMudnBjLnNlbGVjdFN1Ym5ldHMoe1xuICAgICAgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICB9KS5zdWJuZXRJZHM7XG4gICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0Vrc0NsdXN0ZXJTdWJuZXRJZHMnLCB7XG4gICAgICB2YWx1ZTogc3VibmV0SWRzLmpvaW4oJywnKSxcbiAgICB9KTtcbiAgICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnUm9sZUFybicsIHsgdmFsdWU6IHRoaXMuZ2l0bGFiRWtzUm9sZS5yb2xlQXJuIH0pO1xuICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdFa3NBZG1pblJvbGUnLCB7IHZhbHVlOiBla3NBZG1pblJvbGUucm9sZU5hbWUgfSk7XG4gIH1cbiAgcHJpdmF0ZSBzeW50aGVzaXplVGFncyh0YWdzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRhZ3Muam9pbignLCcpO1xuICB9XG59Il19