"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnumType = exports.UnionType = exports.InputType = exports.ObjectType = exports.InterfaceType = void 0;
const private_1 = require("./private");
const schema_field_1 = require("./schema-field");
/**
 * (experimental) Interface Types are abstract types that includes a certain set of fields that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    /**
     * @experimental
     */
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
        this.directives = props.directives;
    }
    /**
     * (experimental) Create a GraphQL Type representing this Intermediate Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this object type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'interface',
            name: this.name,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Interface Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * (experimental) Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    /**
     * @experimental
     */
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
            directives: props.directives,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.resolvers = [];
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        // If the resolvers have been generated, skip the bind
        if (this.resolvers && this.resolvers.length > 0) {
            return this;
        }
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            if (field instanceof schema_field_1.ResolvableField) {
                if (!this.resolvers)
                    this.resolvers = [];
                this.resolvers.push(this.generateResolver(api, fieldName, field.fieldOptions));
            }
        });
        return this;
    }
    /**
     * (experimental) Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Object Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * (experimental) Generate the string of this object type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'type',
            name: this.name,
            interfaceTypes: this.interfaceTypes,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Generate the resolvers linked to this Object Type.
     *
     * @experimental
     */
    generateResolver(api, fieldName, options) {
        return api.createResolver({
            typeName: this.name,
            fieldName: fieldName,
            dataSource: options === null || options === void 0 ? void 0 : options.dataSource,
            pipelineConfig: options === null || options === void 0 ? void 0 : options.pipelineConfig,
            requestMappingTemplate: options === null || options === void 0 ? void 0 : options.requestMappingTemplate,
            responseMappingTemplate: options === null || options === void 0 ? void 0 : options.responseMappingTemplate,
        });
    }
}
exports.ObjectType = ObjectType;
/**
 * (experimental) Input Types are abstract types that define complex objects.
 *
 * They are used in arguments to represent
 *
 * @experimental
 */
class InputType {
    /**
     * @experimental
     */
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * (experimental) Create a GraphQL Type representing this Input Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this input type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'input',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Input Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InputType = InputType;
/**
 * (experimental) Union Types are abstract types that are similar to Interface Types, but they cannot to specify any common fields between types.
 *
 * Note that fields of a union type need to be object types. In other words,
 * you can't create a union type out of interfaces, other unions, or inputs.
 *
 * @experimental
 */
class UnionType {
    /**
     * @experimental
     */
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((def) => this.addField({ field: def.attribute() }));
    }
    /**
     * (experimental) Create a GraphQL Type representing this Union Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this Union type.
     *
     * @experimental
     */
    toString() {
        // Return a string that appends all Object Types for this Union Type
        // i.e. 'union Example = example1 | example2'
        return Object.values(this.definition).reduce((acc, field) => `${acc} ${field.toString()} |`, `union ${this.name} =`).slice(0, -2);
    }
    /**
     * (experimental) Add a field to this Union Type.
     *
     * Input Types must have field options and the IField must be an Object Type.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        var _a;
        if (options.fieldName) {
            throw new Error('Union Types cannot be configured with the fieldName option. Use the field option instead.');
        }
        if (!options.field) {
            throw new Error('Union Types must be configured with the field option.');
        }
        if (options.field && !(options.field.intermediateType instanceof ObjectType)) {
            throw new Error('Fields for Union Types must be Object Types.');
        }
        this.definition[((_a = options.field) === null || _a === void 0 ? void 0 : _a.toString()) + 'id'] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.UnionType = UnionType;
/**
 * (experimental) Enum Types are abstract types that includes a set of fields that represent the strings this type can create.
 *
 * @experimental
 */
class EnumType {
    /**
     * @experimental
     */
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((fieldName) => this.addField({ fieldName }));
    }
    /**
     * (experimental) Create an GraphQL Type representing this Enum Type.
     *
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this enum type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'enum',
            name: this.name,
            fields: Object.keys(this.definition),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Enum Type.
     *
     * To add a field to this Enum Type, you must only configure
     * addField with the fieldName options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (options.field) {
            throw new Error('Enum Type fields consist of strings. Use the fieldName option instead of the field option.');
        }
        if (!options.fieldName) {
            throw new Error('When adding a field to an Enum Type, you must configure the fieldName option.');
        }
        if (options.fieldName.indexOf(' ') > -1) {
            throw new Error(`Enum Type values cannot have whitespace. Received: ${options.fieldName}`);
        }
        this.definition[options.fieldName] = schema_field_1.GraphqlType.string();
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.EnumType = EnumType;
//# sourceMappingURL=data:application/json;base64,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