"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Type = exports.Directive = void 0;
const graphqlapi_1 = require("./graphqlapi");
/**
 * (experimental) Directives for types.
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 * @experimental
 */
class Directive {
    constructor(statement, options) {
        this.statement = statement;
        this.mode = options === null || options === void 0 ? void 0 : options.mode;
        this.mutationFields = options === null || options === void 0 ? void 0 : options.mutationFields;
    }
    /**
     * (experimental) Add the @aws_iam directive.
     *
     * @experimental
     */
    static iam() {
        return new Directive('@aws_iam', { mode: graphqlapi_1.AuthorizationType.IAM });
    }
    /**
     * (experimental) Add the @aws_oidc directive.
     *
     * @experimental
     */
    static oidc() {
        return new Directive('@aws_oidc', { mode: graphqlapi_1.AuthorizationType.OIDC });
    }
    /**
     * (experimental) Add the @aws_api_key directive.
     *
     * @experimental
     */
    static apiKey() {
        return new Directive('@aws_api_key', { mode: graphqlapi_1.AuthorizationType.API_KEY });
    }
    /**
     * (experimental) Add the @aws_auth or @aws_cognito_user_pools directive.
     *
     * @param groups the groups to allow access to.
     * @experimental
     */
    static cognito(...groups) {
        if (groups.length === 0) {
            throw new Error(`Cognito authorization requires at least one Cognito group to be supplied. Received: ${groups.length}`);
        }
        // this function creates the cognito groups as a string (i.e. ["group1", "group2", "group3"])
        const stringify = (array) => {
            return array.reduce((acc, element) => `${acc}"${element}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_auth(cognito_groups: [${stringify(groups)}])`, { mode: graphqlapi_1.AuthorizationType.USER_POOL });
    }
    /**
     * (experimental) Add the @aws_subscribe directive.
     *
     * Only use for top level Subscription type.
     *
     * @param mutations the mutation fields to link to.
     * @experimental
     */
    static subscribe(...mutations) {
        if (mutations.length === 0) {
            throw new Error(`Subscribe directive requires at least one mutation field to be supplied. Received: ${mutations.length}`);
        }
        // this function creates the subscribe directive as a string (i.e. ["mutation_field_1", "mutation_field_2"])
        const stringify = (array) => {
            return array.reduce((acc, mutation) => `${acc}"${mutation}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_subscribe(mutations: [${stringify(mutations)}])`, { mutationFields: mutations });
    }
    /**
     * (experimental) Add a custom directive.
     *
     * @param statement - the directive statement to append.
     * @experimental
     */
    static custom(statement) {
        return new Directive(statement);
    }
    /**
     * Method called when the stringifying Directive for schema generation
     *
     * @param modes the authorization modes
     *
     * @internal
     */
    _bindToAuthModes(modes) {
        this.modes = modes;
        return this;
    }
    /**
     * (experimental) Generate the directive statement.
     *
     * @experimental
     */
    toString() {
        if (this.modes && this.mode && !this.modes.some((mode) => mode === this.mode)) {
            throw new Error(`No Authorization Type ${this.mode} declared in GraphQL Api.`);
        }
        if (this.mode === graphqlapi_1.AuthorizationType.USER_POOL && this.modes && this.modes.length > 1) {
            this.statement = this.statement.replace('@aws_auth', '@aws_cognito_user_pools');
        }
        return this.statement;
    }
}
exports.Directive = Directive;
/**
 * (experimental) Enum containing the Types that can be used to define ObjectTypes.
 *
 * @experimental
 */
var Type;
(function (Type) {
    Type["ID"] = "ID";
    Type["STRING"] = "String";
    Type["INT"] = "Int";
    Type["FLOAT"] = "Float";
    Type["BOOLEAN"] = "Boolean";
    Type["AWS_DATE"] = "AWSDate";
    Type["AWS_TIME"] = "AWSTime";
    Type["AWS_DATE_TIME"] = "AWSDateTime";
    Type["AWS_TIMESTAMP"] = "AWSTimestamp";
    Type["AWS_EMAIL"] = "AWSEmail";
    Type["AWS_JSON"] = "AWSJSON";
    Type["AWS_URL"] = "AWSURL";
    Type["AWS_PHONE"] = "AWSPhone";
    Type["AWS_IP_ADDRESS"] = "AWSIPAddress";
    Type["INTERMEDIATE"] = "INTERMEDIATE";
})(Type = exports.Type || (exports.Type = {}));
//# sourceMappingURL=data:application/json;base64,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