"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * (experimental) enum with all possible values for AppSync authorization type.
 *
 * @experimental
 */
var AuthorizationType;
(function (AuthorizationType) {
    AuthorizationType["API_KEY"] = "API_KEY";
    AuthorizationType["IAM"] = "AWS_IAM";
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * (experimental) enum with all possible values for Cognito user-pool default actions.
 *
 * @experimental
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * (experimental) log-level for fields in AppSync.
 *
 * @experimental
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    FieldLogLevel["NONE"] = "NONE";
    FieldLogLevel["ERROR"] = "ERROR";
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * (experimental) A class used to generate resource arns for AppSync.
 *
 * @experimental
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * (experimental) Generate the resource names given custom arns.
     *
     * @param arns The custom arns that need to be permissioned.
     * @experimental
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * (experimental) Generate the resource names given a type and fields.
     *
     * @param type The type that needs to be allowed.
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields.
     * @experimental
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * (experimental) Generate the resource names that accepts all types: `*`.
     *
     * @experimental
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * (experimental) Return the Resource ARN.
     *
     * @param api The GraphQL API to give permissions.
     * @experimental
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * (experimental) An AppSync GraphQL API.
 *
 * @experimental
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultMode = (_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) !== null && _b !== void 0 ? _b : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.additionalAuthorizationModes) !== null && _d !== void 0 ? _d : [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_e = props.schema) !== null && _e !== void 0 ? _e : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_f = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _f === void 0 ? void 0 : _f.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * (experimental) Import a GraphQL API through this function.
     *
     * @param scope scope.
     * @param id id.
     * @param attrs GraphQL API Attributes of an API.
     * @experimental
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _a;
        const arn = (_a = attrs.graphqlApiArn) !== null && _a !== void 0 ? _a : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Adds an IAM policy statement associated with this GraphQLApi to an IAM principal's policy.
     *
     * @param grantee The principal.
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...).
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql ).
     * @experimental
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * (experimental) Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM principal's policy.
     *
     * @param grantee The principal.
     * @param fields The fields to grant access to that are Mutations (leave blank for all).
     * @experimental
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * (experimental) Adds an IAM policy statement for Query access to this GraphQLApi to an IAM principal's policy.
     *
     * @param grantee The principal.
     * @param fields The fields to grant access to that are Queries (leave blank for all).
     * @experimental
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * (experimental) Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM principal's policy.
     *
     * @param grantee The principal.
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all).
     * @experimental
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * (experimental) Add schema dependency to a given construct.
     *
     * @param construct the dependee.
     * @experimental
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        var _a, _b;
        if (!config)
            return undefined;
        const logsRoleArn = (_b = (_a = config.role) === null || _a === void 0 ? void 0 : _a.roleArn) !== null && _b !== void 0 ? _b : new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        }).roleArn;
        return {
            cloudWatchLogsRoleArn: logsRoleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        var _a, _b;
        if (((_a = config === null || config === void 0 ? void 0 : config.expires) === null || _a === void 0 ? void 0 : _a.isBefore(core_1.Duration.days(1))) || ((_b = config === null || config === void 0 ? void 0 : config.expires) === null || _b === void 0 ? void 0 : _b.isAfter(core_1.Duration.days(365)))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = (config === null || config === void 0 ? void 0 : config.expires) ? config === null || config === void 0 ? void 0 : config.expires.toEpoch() : undefined;
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * (experimental) Escape hatch to append to Schema as desired.
     *
     * Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema.
     * @param delimiter the delimiter between schema and addition.
     * @default - ''
     * @experimental
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * (experimental) Add type to the schema.
     *
     * @param type the intermediate type to add to the schema.
     * @experimental
     */
    addType(type) {
        return this.schema.addType(type);
    }
    /**
     * (experimental) Add a query field to the schema's Query. CDK will create an Object Type called 'Query'. For example,.
     *
     * type Query {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query.
     * @param field the resolvable field to for this query.
     * @experimental
     */
    addQuery(fieldName, field) {
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * (experimental) Add a mutation field to the schema's Mutation. CDK will create an Object Type called 'Mutation'. For example,.
     *
     * type Mutation {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation.
     * @param field the resolvable field to for this Mutation.
     * @experimental
     */
    addMutation(fieldName, field) {
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * (experimental) Add a subscription field to the schema's Subscription. CDK will create an Object Type called 'Subscription'. For example,.
     *
     * type Subscription {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription.
     * @param field the resolvable field to for this Subscription.
     * @experimental
     */
    addSubscription(fieldName, field) {
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhcGhxbGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdyYXBocWxhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsOENBQW1HO0FBQ25HLHdDQUFzRjtBQUV0RiwyREFBaUY7QUFDakYsdURBQWdFO0FBQ2hFLHFDQUFrQzs7Ozs7O0FBUWxDLElBQVksaUJBaUJYO0FBakJELFdBQVksaUJBQWlCO0lBSTNCLHdDQUFtQixDQUFBO0lBSW5CLG9DQUFlLENBQUE7SUFJZiw0REFBdUMsQ0FBQTtJQUl2Qyw0Q0FBdUIsQ0FBQTtBQUN6QixDQUFDLEVBakJXLGlCQUFpQixHQUFqQix5QkFBaUIsS0FBakIseUJBQWlCLFFBaUI1Qjs7Ozs7O0FBa0NELElBQVkscUJBU1g7QUFURCxXQUFZLHFCQUFxQjtJQUkvQix3Q0FBZSxDQUFBO0lBSWYsc0NBQWEsQ0FBQTtBQUNmLENBQUMsRUFUVyxxQkFBcUIsR0FBckIsNkJBQXFCLEtBQXJCLDZCQUFxQixRQVNoQzs7Ozs7O0FBb0dELElBQVksYUFhWDtBQWJELFdBQVksYUFBYTtJQUl2Qiw4QkFBYSxDQUFBO0lBSWIsZ0NBQWUsQ0FBQTtJQUlmLDRCQUFXLENBQUE7QUFDYixDQUFDLEVBYlcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFheEI7Ozs7OztBQXVFRCxNQUFhLFdBQVc7SUFxQ3RCLFlBQW9CLElBQWM7UUFDaEMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7SUFDbkIsQ0FBQzs7Ozs7OztJQS9CTSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsSUFBYztRQUNwQyxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUM1RDtRQUNELE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7Ozs7Ozs7SUFVTSxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVksRUFBRSxHQUFHLE1BQWdCO1FBQ3BELE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLFNBQVMsSUFBSSxXQUFXLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQyxDQUFDO1FBQzFHLE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLEdBQUc7UUFDZixPQUFPLElBQUksV0FBVyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUNoQyxDQUFDOzs7Ozs7O0lBYU0sWUFBWSxDQUFDLEdBQWU7UUFDakMsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsWUFBSyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDcEQsT0FBTyxFQUFFLFNBQVM7WUFDbEIsUUFBUSxFQUFFLFFBQVEsR0FBRyxDQUFDLEtBQUssRUFBRTtZQUM3QixHQUFHLEVBQUUsR0FBRztZQUNSLFlBQVksRUFBRSxHQUFHLEdBQUcsRUFBRTtTQUN2QixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUM7Q0FDRjtBQXRERCxrQ0FzREM7Ozs7Ozs7QUF3QkQsTUFBYSxVQUFXLFNBQVEsZ0NBQWM7Ozs7SUFtRTVDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7O1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxXQUFXLGVBQUcsS0FBSyxDQUFDLG1CQUFtQiwwQ0FBRSxvQkFBb0IsbUNBQ2pFLEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDbkQsTUFBTSxlQUFlLGVBQUcsS0FBSyxDQUFDLG1CQUFtQiwwQ0FBRSw0QkFBNEIsbUNBQUksRUFBRSxDQUFDO1FBQ3RGLE1BQU0sS0FBSyxHQUFHLENBQUMsV0FBVyxFQUFFLEdBQUcsZUFBZSxDQUFDLENBQUM7UUFFaEQsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUUsQ0FBQztRQUUxRCxJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFdkMsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLGlDQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM3QyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsa0JBQWtCLEVBQUUsV0FBVyxDQUFDLGlCQUFpQjtZQUNqRCxTQUFTLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQy9DLG1CQUFtQixFQUFFLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsbUJBQW1CLENBQUM7WUFDbkYsY0FBYyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDO1lBQ3BFLGlDQUFpQyxFQUFFLElBQUksQ0FBQyxpQ0FBaUMsQ0FBQyxlQUFlLENBQUM7WUFDMUYsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQy9CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUM7UUFDaEMsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQztRQUM1QixJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDO1FBQzFDLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7UUFDMUIsSUFBSSxDQUFDLE1BQU0sU0FBRyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxJQUFJLGVBQU0sRUFBRSxDQUFDO1FBQzNDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0MsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEtBQUssaUJBQWlCLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDOUUsTUFBTSxNQUFNLFNBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQXVCLEVBQUUsRUFBRTtnQkFDcEQsT0FBTyxJQUFJLENBQUMsaUJBQWlCLEtBQUssaUJBQWlCLENBQUMsT0FBTyxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUM7WUFDbkYsQ0FBQyxDQUFDLDBDQUFFLFlBQVksQ0FBQztZQUNqQixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDaEQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQ3RELElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUM7U0FDOUM7SUFDSCxDQUFDOzs7Ozs7Ozs7SUFoR00sTUFBTSxDQUFDLHdCQUF3QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTJCOztRQUM5RixNQUFNLEdBQUcsU0FBRyxLQUFLLENBQUMsYUFBYSxtQ0FBSSxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUMzRCxPQUFPLEVBQUUsU0FBUztZQUNsQixRQUFRLEVBQUUsUUFBUSxLQUFLLENBQUMsWUFBWSxFQUFFO1NBQ3ZDLENBQUMsQ0FBQztRQUNILE1BQU0sTUFBTyxTQUFRLGdDQUFjO1lBR2pDLFlBQWEsQ0FBWSxFQUFFLENBQVM7Z0JBQ2xDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBSEUsVUFBSyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7Z0JBQzNCLFFBQUcsR0FBRyxHQUFHLENBQUM7WUFHMUIsQ0FBQztTQUNGO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7Ozs7Ozs7O0lBNkZNLEtBQUssQ0FBQyxPQUFtQixFQUFFLFNBQXNCLEVBQUUsR0FBRyxPQUFpQjtRQUM1RSxPQUFPLGVBQUssQ0FBQyxjQUFjLENBQUM7WUFDMUIsT0FBTztZQUNQLE9BQU87WUFDUCxZQUFZLEVBQUUsU0FBUyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7WUFDMUMsS0FBSyxFQUFFLElBQUk7U0FDWixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7OztJQVNNLGFBQWEsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDM0QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDM0YsQ0FBQzs7Ozs7Ozs7SUFTTSxVQUFVLENBQUMsT0FBbUIsRUFBRSxHQUFHLE1BQWdCO1FBQ3hELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsR0FBRyxNQUFNLENBQUMsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3hGLENBQUM7Ozs7Ozs7O0lBU00saUJBQWlCLENBQUMsT0FBbUIsRUFBRSxHQUFHLE1BQWdCO1FBQy9ELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUUsR0FBRyxNQUFNLENBQUMsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBQy9GLENBQUM7SUFFTywwQkFBMEIsQ0FBQyxLQUEwQjtRQUMzRCxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7WUFDakIsSUFBSSxJQUFJLENBQUMsaUJBQWlCLEtBQUssaUJBQWlCLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixFQUFFO2dCQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7YUFDdkQ7WUFDRCxJQUFJLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxTQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO2dCQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7YUFDdkQ7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDM0YsTUFBTSxJQUFJLEtBQUssQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO1NBQ3RJO1FBQ0QsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEtBQUssaUJBQWlCLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUN2RixNQUFNLElBQUksS0FBSyxDQUFDLCtHQUErRyxDQUFDLENBQUM7U0FDbEk7SUFDSCxDQUFDOzs7Ozs7O0lBT00sbUJBQW1CLENBQUMsU0FBc0I7UUFDL0MsU0FBUyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8sY0FBYyxDQUFDLE1BQWtCOztRQUN2QyxJQUFJLENBQUMsTUFBTTtZQUFFLE9BQU8sU0FBUyxDQUFDO1FBQzlCLE1BQU0sV0FBVyxlQUFXLE1BQU0sQ0FBQyxJQUFJLDBDQUFFLE9BQU8sbUNBQUksSUFBSSxjQUFJLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUNoRixTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyx1QkFBdUIsQ0FBQztZQUN4RCxlQUFlLEVBQUU7Z0JBQ2YsdUJBQWEsQ0FBQyx3QkFBd0IsQ0FBQyw2Q0FBNkMsQ0FBQzthQUN0RjtTQUNGLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDWCxPQUFPO1lBQ0wscUJBQXFCLEVBQUUsV0FBVztZQUNsQyxxQkFBcUIsRUFBRSxNQUFNLENBQUMscUJBQXFCO1lBQ25ELGFBQWEsRUFBRSxNQUFNLENBQUMsYUFBYTtTQUNwQyxDQUFDO0lBQ0osQ0FBQztJQUVPLHdCQUF3QixDQUFDLE1BQTRCO1FBQzNELElBQUksQ0FBQyxNQUFNO1lBQUUsT0FBTyxTQUFTLENBQUM7UUFDOUIsT0FBTztZQUNMLE9BQU8sRUFBRSxNQUFNLENBQUMsbUJBQW1CO1lBQ25DLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUTtZQUN6QixNQUFNLEVBQUUsTUFBTSxDQUFDLG9CQUFvQjtZQUNuQyxNQUFNLEVBQUUsTUFBTSxDQUFDLFlBQVk7U0FDNUIsQ0FBQztJQUNKLENBQUM7SUFFTyxtQkFBbUIsQ0FBQyxNQUF1QjtRQUNqRCxJQUFJLENBQUMsTUFBTTtZQUFFLE9BQU8sU0FBUyxDQUFDO1FBQzlCLE9BQU87WUFDTCxVQUFVLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVO1lBQ3RDLFNBQVMsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxNQUFNO1lBQ3ZDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxnQkFBZ0I7WUFDekMsYUFBYSxFQUFFLE1BQU0sQ0FBQyxhQUFhLElBQUkscUJBQXFCLENBQUMsS0FBSztTQUNuRSxDQUFDO0lBQ0osQ0FBQztJQUVPLGlDQUFpQyxDQUFDLEtBQTJCO1FBQ25FLElBQUksQ0FBQyxLQUFLLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDO1lBQUUsT0FBTyxTQUFTLENBQUM7UUFDbkQsT0FBTyxLQUFLLENBQUMsTUFBTSxDQUEyRCxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsRUFBRSxDQUFDO1lBQzNGLEdBQUcsR0FBRyxFQUFFO2dCQUNOLGtCQUFrQixFQUFFLElBQUksQ0FBQyxpQkFBaUI7Z0JBQzFDLGNBQWMsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQztnQkFDN0QsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQzthQUM3RTtTQUNGLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDVCxDQUFDO0lBRU8sWUFBWSxDQUFDLE1BQXFCOztRQUN4QyxJQUFJLE9BQUEsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLE9BQU8sMENBQUUsUUFBUSxDQUFDLGVBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLGFBQUssTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLE9BQU8sMENBQUUsT0FBTyxDQUFDLGVBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUMsRUFBRTtZQUMvRixNQUFNLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1NBQ25FO1FBQ0QsTUFBTSxPQUFPLEdBQUcsQ0FBQSxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsT0FBTyxFQUFDLENBQUMsQ0FBQyxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsT0FBTyxDQUFDLE9BQU8sR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ3hFLE9BQU8sSUFBSSw2QkFBUyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUEsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLElBQUksS0FBSSxTQUFTLFFBQVEsRUFBRTtZQUMvRCxPQUFPO1lBQ1AsV0FBVyxFQUFFLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxXQUFXO1lBQ2hDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztTQUNsQixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7Ozs7Ozs7SUFZTSxXQUFXLENBQUMsUUFBZ0IsRUFBRSxTQUFrQjtRQUNyRCxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDL0MsQ0FBQzs7Ozs7OztJQVNNLE9BQU8sQ0FBQyxJQUF1QjtRQUNwQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25DLENBQUM7Ozs7Ozs7Ozs7OztJQWFNLFFBQVEsQ0FBQyxTQUFpQixFQUFFLEtBQXNCO1FBQ3ZELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2hELENBQUM7Ozs7Ozs7Ozs7OztJQWFNLFdBQVcsQ0FBQyxTQUFpQixFQUFFLEtBQXNCO1FBQzFELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ25ELENBQUM7Ozs7Ozs7Ozs7OztJQWFNLGVBQWUsQ0FBQyxTQUFpQixFQUFFLEtBQXNCO1FBQzlELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3ZELENBQUM7Q0FDRjtBQXRURCxnQ0FzVEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJVXNlclBvb2wgfSBmcm9tICdAYXdzLWNkay9hd3MtY29nbml0byc7XG5pbXBvcnQgeyBNYW5hZ2VkUG9saWN5LCBSb2xlLCBJUm9sZSwgU2VydmljZVByaW5jaXBhbCwgR3JhbnQsIElHcmFudGFibGUgfSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IENmblJlc291cmNlLCBEdXJhdGlvbiwgRXhwaXJhdGlvbiwgSVJlc29sdmFibGUsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmbkFwaUtleSwgQ2ZuR3JhcGhRTEFwaSwgQ2ZuR3JhcGhRTFNjaGVtYSB9IGZyb20gJy4vYXBwc3luYy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUdyYXBocWxBcGksIEdyYXBocWxBcGlCYXNlIH0gZnJvbSAnLi9ncmFwaHFsYXBpLWJhc2UnO1xuaW1wb3J0IHsgU2NoZW1hIH0gZnJvbSAnLi9zY2hlbWEnO1xuaW1wb3J0IHsgSUludGVybWVkaWF0ZVR5cGUgfSBmcm9tICcuL3NjaGVtYS1iYXNlJztcbmltcG9ydCB7IFJlc29sdmFibGVGaWVsZCB9IGZyb20gJy4vc2NoZW1hLWZpZWxkJztcbmltcG9ydCB7IE9iamVjdFR5cGUgfSBmcm9tICcuL3NjaGVtYS1pbnRlcm1lZGlhdGUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gQXV0aG9yaXphdGlvblR5cGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBUElfS0VZID0gJ0FQSV9LRVknLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIElBTSA9ICdBV1NfSUFNJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFVTRVJfUE9PTCA9ICdBTUFaT05fQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIE9JREMgPSAnT1BFTklEX0NPTk5FQ1QnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhvcml6YXRpb25Nb2RlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyUG9vbENvbmZpZz86IFVzZXJQb29sQ29uZmlnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwaUtleUNvbmZpZz86IEFwaUtleUNvbmZpZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3BlbklkQ29ubmVjdENvbmZpZz86IE9wZW5JZENvbm5lY3RDb25maWc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gVXNlclBvb2xEZWZhdWx0QWN0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBTExPVyA9ICdBTExPVycsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBERU5ZID0gJ0RFTlknLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sQ29uZmlnIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlclBvb2w6IElVc2VyUG9vbDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcHBJZENsaWVudFJlZ2V4Pzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZWZhdWx0QWN0aW9uPzogVXNlclBvb2xEZWZhdWx0QWN0aW9uO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFwaUtleUNvbmZpZyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGV4cGlyZXM/OiBFeHBpcmF0aW9uO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBPcGVuSWRDb25uZWN0Q29uZmlnIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRva2VuRXhwaXJ5RnJvbUF1dGg/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0b2tlbkV4cGlyeUZyb21Jc3N1ZT86IG51bWJlcjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsaWVudElkPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvaWRjUHJvdmlkZXI6IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQXV0aG9yaXphdGlvbkNvbmZpZyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVmYXVsdEF1dGhvcml6YXRpb24/OiBBdXRob3JpemF0aW9uTW9kZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2Rlcz86IEF1dGhvcml6YXRpb25Nb2RlW107XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gRmllbGRMb2dMZXZlbCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgTk9ORSA9ICdOT05FJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBFUlJPUiA9ICdFUlJPUicsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIEFMTCA9ICdBTEwnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBMb2dDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXhjbHVkZVZlcmJvc2VDb250ZW50PzogYm9vbGVhbiB8IElSZXNvbHZhYmxlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGZpZWxkTG9nTGV2ZWw/OiBGaWVsZExvZ0xldmVsO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogSVJvbGU7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBHcmFwaHFsQXBpUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXV0aG9yaXphdGlvbkNvbmZpZz86IEF1dGhvcml6YXRpb25Db25maWc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ0NvbmZpZz86IExvZ0NvbmZpZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzY2hlbWE/OiBTY2hlbWE7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB4cmF5RW5hYmxlZD86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBJYW1SZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjdXN0b20oLi4uYXJuczogc3RyaW5nW10pOiBJYW1SZXNvdXJjZSB7XG4gICAgaWYgKGFybnMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IDEgY3VzdG9tIEFSTiBtdXN0IGJlIHByb3ZpZGVkLicpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IElhbVJlc291cmNlKGFybnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgb2ZUeXBlKHR5cGU6IHN0cmluZywgLi4uZmllbGRzOiBzdHJpbmdbXSk6IElhbVJlc291cmNlIHtcbiAgICBjb25zdCBhcm5zID0gZmllbGRzLmxlbmd0aCA/IGZpZWxkcy5tYXAoKGZpZWxkKSA9PiBgdHlwZXMvJHt0eXBlfS9maWVsZHMvJHtmaWVsZH1gKSA6IFtgdHlwZXMvJHt0eXBlfS8qYF07XG4gICAgcmV0dXJuIG5ldyBJYW1SZXNvdXJjZShhcm5zKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGwoKTogSWFtUmVzb3VyY2Uge1xuICAgIHJldHVybiBuZXcgSWFtUmVzb3VyY2UoWycqJ10pO1xuICB9XG5cbiAgcHJpdmF0ZSBhcm5zOiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGFybnM6IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5hcm5zID0gYXJucztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZXNvdXJjZUFybnMoYXBpOiBHcmFwaHFsQXBpKTogc3RyaW5nW10ge1xuICAgIHJldHVybiB0aGlzLmFybnMubWFwKChhcm4pID0+IFN0YWNrLm9mKGFwaSkuZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdhcHBzeW5jJyxcbiAgICAgIHJlc291cmNlOiBgYXBpcy8ke2FwaS5hcGlJZH1gLFxuICAgICAgc2VwOiAnLycsXG4gICAgICByZXNvdXJjZU5hbWU6IGAke2Fybn1gLFxuICAgIH0pKTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBHcmFwaHFsQXBpQXR0cmlidXRlcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGdyYXBocWxBcGlJZDogc3RyaW5nLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZ3JhcGhxbEFwaUFybj86IHN0cmluZyxcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBHcmFwaHFsQXBpIGV4dGVuZHMgR3JhcGhxbEFwaUJhc2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21HcmFwaHFsQXBpQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogR3JhcGhxbEFwaUF0dHJpYnV0ZXMpOiBJR3JhcGhxbEFwaSB7XG4gICAgY29uc3QgYXJuID0gYXR0cnMuZ3JhcGhxbEFwaUFybiA/PyBTdGFjay5vZihzY29wZSkuZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdhcHBzeW5jJyxcbiAgICAgIHJlc291cmNlOiBgYXBpcy8ke2F0dHJzLmdyYXBocWxBcGlJZH1gLFxuICAgIH0pO1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIEdyYXBocWxBcGlCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhcGlJZCA9IGF0dHJzLmdyYXBocWxBcGlJZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhcm4gPSBhcm47XG4gICAgICBjb25zdHJ1Y3RvciAoczogQ29uc3RydWN0LCBpOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIocywgaSk7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgYXBpSWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFybjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBncmFwaHFsVXJsOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHNjaGVtYTogU2NoZW1hO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBtb2RlczogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFwaUtleT86IHN0cmluZztcblxuICBwcml2YXRlIHNjaGVtYVJlc291cmNlOiBDZm5HcmFwaFFMU2NoZW1hO1xuICBwcml2YXRlIGFwaTogQ2ZuR3JhcGhRTEFwaTtcbiAgcHJpdmF0ZSBhcGlLZXlSZXNvdXJjZT86IENmbkFwaUtleTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogR3JhcGhxbEFwaVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IGRlZmF1bHRNb2RlID0gcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZz8uZGVmYXVsdEF1dGhvcml6YXRpb24gPz9cbiAgICAgIHsgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkgfTtcbiAgICBjb25zdCBhZGRpdGlvbmFsTW9kZXMgPSBwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnPy5hZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzID8/IFtdO1xuICAgIGNvbnN0IG1vZGVzID0gW2RlZmF1bHRNb2RlLCAuLi5hZGRpdGlvbmFsTW9kZXNdO1xuXG4gICAgdGhpcy5tb2RlcyA9IG1vZGVzLm1hcCgobW9kZSkgPT4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSApO1xuXG4gICAgdGhpcy52YWxpZGF0ZUF1dGhvcml6YXRpb25Qcm9wcyhtb2Rlcyk7XG5cbiAgICB0aGlzLmFwaSA9IG5ldyBDZm5HcmFwaFFMQXBpKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG5hbWU6IHByb3BzLm5hbWUsXG4gICAgICBhdXRoZW50aWNhdGlvblR5cGU6IGRlZmF1bHRNb2RlLmF1dGhvcml6YXRpb25UeXBlLFxuICAgICAgbG9nQ29uZmlnOiB0aGlzLnNldHVwTG9nQ29uZmlnKHByb3BzLmxvZ0NvbmZpZyksXG4gICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB0aGlzLnNldHVwT3BlbklkQ29ubmVjdENvbmZpZyhkZWZhdWx0TW9kZS5vcGVuSWRDb25uZWN0Q29uZmlnKSxcbiAgICAgIHVzZXJQb29sQ29uZmlnOiB0aGlzLnNldHVwVXNlclBvb2xDb25maWcoZGVmYXVsdE1vZGUudXNlclBvb2xDb25maWcpLFxuICAgICAgYWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiB0aGlzLnNldHVwQWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcyhhZGRpdGlvbmFsTW9kZXMpLFxuICAgICAgeHJheUVuYWJsZWQ6IHByb3BzLnhyYXlFbmFibGVkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcGlJZCA9IHRoaXMuYXBpLmF0dHJBcGlJZDtcbiAgICB0aGlzLmFybiA9IHRoaXMuYXBpLmF0dHJBcm47XG4gICAgdGhpcy5ncmFwaHFsVXJsID0gdGhpcy5hcGkuYXR0ckdyYXBoUWxVcmw7XG4gICAgdGhpcy5uYW1lID0gdGhpcy5hcGkubmFtZTtcbiAgICB0aGlzLnNjaGVtYSA9IHByb3BzLnNjaGVtYSA/PyBuZXcgU2NoZW1hKCk7XG4gICAgdGhpcy5zY2hlbWFSZXNvdXJjZSA9IHRoaXMuc2NoZW1hLmJpbmQodGhpcyk7XG5cbiAgICBpZiAobW9kZXMuc29tZSgobW9kZSkgPT4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSkpIHtcbiAgICAgIGNvbnN0IGNvbmZpZyA9IG1vZGVzLmZpbmQoKG1vZGU6IEF1dGhvcml6YXRpb25Nb2RlKSA9PiB7XG4gICAgICAgIHJldHVybiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZICYmIG1vZGUuYXBpS2V5Q29uZmlnO1xuICAgICAgfSk/LmFwaUtleUNvbmZpZztcbiAgICAgIHRoaXMuYXBpS2V5UmVzb3VyY2UgPSB0aGlzLmNyZWF0ZUFQSUtleShjb25maWcpO1xuICAgICAgdGhpcy5hcGlLZXlSZXNvdXJjZS5hZGREZXBlbmRzT24odGhpcy5zY2hlbWFSZXNvdXJjZSk7XG4gICAgICB0aGlzLmFwaUtleSA9IHRoaXMuYXBpS2V5UmVzb3VyY2UuYXR0ckFwaUtleTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdyYW50KGdyYW50ZWU6IElHcmFudGFibGUsIHJlc291cmNlczogSWFtUmVzb3VyY2UsIC4uLmFjdGlvbnM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgIHJldHVybiBHcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9ucyxcbiAgICAgIHJlc291cmNlQXJuczogcmVzb3VyY2VzLnJlc291cmNlQXJucyh0aGlzKSxcbiAgICAgIHNjb3BlOiB0aGlzLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBncmFudE11dGF0aW9uKGdyYW50ZWU6IElHcmFudGFibGUsIC4uLmZpZWxkczogc3RyaW5nW10pOiBHcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgSWFtUmVzb3VyY2Uub2ZUeXBlKCdNdXRhdGlvbicsIC4uLmZpZWxkcyksICdhcHBzeW5jOkdyYXBoUUwnKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdyYW50UXVlcnkoZ3JhbnRlZTogSUdyYW50YWJsZSwgLi4uZmllbGRzOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCBJYW1SZXNvdXJjZS5vZlR5cGUoJ1F1ZXJ5JywgLi4uZmllbGRzKSwgJ2FwcHN5bmM6R3JhcGhRTCcpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdyYW50U3Vic2NyaXB0aW9uKGdyYW50ZWU6IElHcmFudGFibGUsIC4uLmZpZWxkczogc3RyaW5nW10pOiBHcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgSWFtUmVzb3VyY2Uub2ZUeXBlKCdTdWJzY3JpcHRpb24nLCAuLi5maWVsZHMpLCAnYXBwc3luYzpHcmFwaFFMJyk7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlQXV0aG9yaXphdGlvblByb3BzKG1vZGVzOiBBdXRob3JpemF0aW9uTW9kZVtdKSB7XG4gICAgbW9kZXMubWFwKChtb2RlKSA9PiB7XG4gICAgICBpZiAobW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuT0lEQyAmJiAhbW9kZS5vcGVuSWRDb25uZWN0Q29uZmlnKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignTWlzc2luZyBkZWZhdWx0IE9JREMgQ29uZmlndXJhdGlvbicpO1xuICAgICAgfVxuICAgICAgaWYgKG1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCAmJiAhbW9kZS51c2VyUG9vbENvbmZpZykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pc3NpbmcgZGVmYXVsdCBPSURDIENvbmZpZ3VyYXRpb24nKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgICBpZiAobW9kZXMuZmlsdGVyKChtb2RlKSA9PiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZKS5sZW5ndGggPiAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBBUElfS0VZIGNvbmZpZ3VyYXRpb24uIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbCcpO1xuICAgIH1cbiAgICBpZiAobW9kZXMuZmlsdGVyKChtb2RlKSA9PiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5JQU0pLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IGNhblxcJ3QgZHVwbGljYXRlIElBTSBjb25maWd1cmF0aW9uLiBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL3NlY3VyaXR5Lmh0bWwnKTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRTY2hlbWFEZXBlbmRlbmN5KGNvbnN0cnVjdDogQ2ZuUmVzb3VyY2UpOiBib29sZWFuIHtcbiAgICBjb25zdHJ1Y3QuYWRkRGVwZW5kc09uKHRoaXMuc2NoZW1hUmVzb3VyY2UpO1xuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cExvZ0NvbmZpZyhjb25maWc/OiBMb2dDb25maWcpIHtcbiAgICBpZiAoIWNvbmZpZykgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICBjb25zdCBsb2dzUm9sZUFybjogc3RyaW5nID0gY29uZmlnLnJvbGU/LnJvbGVBcm4gPz8gbmV3IFJvbGUodGhpcywgJ0FwaUxvZ3NSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnYXBwc3luYy5hbWF6b25hd3MuY29tJyksXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NBcHBTeW5jUHVzaFRvQ2xvdWRXYXRjaExvZ3MnKSxcbiAgICAgIF0sXG4gICAgfSkucm9sZUFybjtcbiAgICByZXR1cm4ge1xuICAgICAgY2xvdWRXYXRjaExvZ3NSb2xlQXJuOiBsb2dzUm9sZUFybixcbiAgICAgIGV4Y2x1ZGVWZXJib3NlQ29udGVudDogY29uZmlnLmV4Y2x1ZGVWZXJib3NlQ29udGVudCxcbiAgICAgIGZpZWxkTG9nTGV2ZWw6IGNvbmZpZy5maWVsZExvZ0xldmVsLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHNldHVwT3BlbklkQ29ubmVjdENvbmZpZyhjb25maWc/OiBPcGVuSWRDb25uZWN0Q29uZmlnKSB7XG4gICAgaWYgKCFjb25maWcpIHJldHVybiB1bmRlZmluZWQ7XG4gICAgcmV0dXJuIHtcbiAgICAgIGF1dGhUdGw6IGNvbmZpZy50b2tlbkV4cGlyeUZyb21BdXRoLFxuICAgICAgY2xpZW50SWQ6IGNvbmZpZy5jbGllbnRJZCxcbiAgICAgIGlhdFR0bDogY29uZmlnLnRva2VuRXhwaXJ5RnJvbUlzc3VlLFxuICAgICAgaXNzdWVyOiBjb25maWcub2lkY1Byb3ZpZGVyLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHNldHVwVXNlclBvb2xDb25maWcoY29uZmlnPzogVXNlclBvb2xDb25maWcpIHtcbiAgICBpZiAoIWNvbmZpZykgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICByZXR1cm4ge1xuICAgICAgdXNlclBvb2xJZDogY29uZmlnLnVzZXJQb29sLnVzZXJQb29sSWQsXG4gICAgICBhd3NSZWdpb246IGNvbmZpZy51c2VyUG9vbC5zdGFjay5yZWdpb24sXG4gICAgICBhcHBJZENsaWVudFJlZ2V4OiBjb25maWcuYXBwSWRDbGllbnRSZWdleCxcbiAgICAgIGRlZmF1bHRBY3Rpb246IGNvbmZpZy5kZWZhdWx0QWN0aW9uIHx8IFVzZXJQb29sRGVmYXVsdEFjdGlvbi5BTExPVyxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cEFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXMobW9kZXM/OiBBdXRob3JpemF0aW9uTW9kZVtdKSB7XG4gICAgaWYgKCFtb2RlcyB8fCBtb2Rlcy5sZW5ndGggPT09IDApIHJldHVybiB1bmRlZmluZWQ7XG4gICAgcmV0dXJuIG1vZGVzLnJlZHVjZTxDZm5HcmFwaFFMQXBpLkFkZGl0aW9uYWxBdXRoZW50aWNhdGlvblByb3ZpZGVyUHJvcGVydHlbXT4oKGFjYywgbW9kZSkgPT4gW1xuICAgICAgLi4uYWNjLCB7XG4gICAgICAgIGF1dGhlbnRpY2F0aW9uVHlwZTogbW9kZS5hdXRob3JpemF0aW9uVHlwZSxcbiAgICAgICAgdXNlclBvb2xDb25maWc6IHRoaXMuc2V0dXBVc2VyUG9vbENvbmZpZyhtb2RlLnVzZXJQb29sQ29uZmlnKSxcbiAgICAgICAgb3BlbklkQ29ubmVjdENvbmZpZzogdGhpcy5zZXR1cE9wZW5JZENvbm5lY3RDb25maWcobW9kZS5vcGVuSWRDb25uZWN0Q29uZmlnKSxcbiAgICAgIH0sXG4gICAgXSwgW10pO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVBUElLZXkoY29uZmlnPzogQXBpS2V5Q29uZmlnKSB7XG4gICAgaWYgKGNvbmZpZz8uZXhwaXJlcz8uaXNCZWZvcmUoRHVyYXRpb24uZGF5cygxKSkgfHwgY29uZmlnPy5leHBpcmVzPy5pc0FmdGVyKER1cmF0aW9uLmRheXMoMzY1KSkpIHtcbiAgICAgIHRocm93IEVycm9yKCdBUEkga2V5IGV4cGlyYXRpb24gbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDM2NSBkYXlzLicpO1xuICAgIH1cbiAgICBjb25zdCBleHBpcmVzID0gY29uZmlnPy5leHBpcmVzID8gY29uZmlnPy5leHBpcmVzLnRvRXBvY2goKSA6IHVuZGVmaW5lZDtcbiAgICByZXR1cm4gbmV3IENmbkFwaUtleSh0aGlzLCBgJHtjb25maWc/Lm5hbWUgfHwgJ0RlZmF1bHQnfUFwaUtleWAsIHtcbiAgICAgIGV4cGlyZXMsXG4gICAgICBkZXNjcmlwdGlvbjogY29uZmlnPy5kZXNjcmlwdGlvbixcbiAgICAgIGFwaUlkOiB0aGlzLmFwaUlkLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFRvU2NoZW1hKGFkZGl0aW9uOiBzdHJpbmcsIGRlbGltaXRlcj86IHN0cmluZyk6IHZvaWQge1xuICAgIHRoaXMuc2NoZW1hLmFkZFRvU2NoZW1hKGFkZGl0aW9uLCBkZWxpbWl0ZXIpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRUeXBlKHR5cGU6IElJbnRlcm1lZGlhdGVUeXBlKTogSUludGVybWVkaWF0ZVR5cGUge1xuICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRUeXBlKHR5cGUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRRdWVyeShmaWVsZE5hbWU6IHN0cmluZywgZmllbGQ6IFJlc29sdmFibGVGaWVsZCk6IE9iamVjdFR5cGUge1xuICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRRdWVyeShmaWVsZE5hbWUsIGZpZWxkKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkTXV0YXRpb24oZmllbGROYW1lOiBzdHJpbmcsIGZpZWxkOiBSZXNvbHZhYmxlRmllbGQpOiBPYmplY3RUeXBlIHtcbiAgICByZXR1cm4gdGhpcy5zY2hlbWEuYWRkTXV0YXRpb24oZmllbGROYW1lLCBmaWVsZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFN1YnNjcmlwdGlvbihmaWVsZE5hbWU6IHN0cmluZywgZmllbGQ6IFJlc29sdmFibGVGaWVsZCk6IE9iamVjdFR5cGUge1xuICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRTdWJzY3JpcHRpb24oZmllbGROYW1lLCBmaWVsZCk7XG4gIH1cbn1cbiJdfQ==