#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
    Pantone LLC is a limited liability company headquartered in Carlstadt, New Jersey.
    The company is best known for its Pantone Matching System (PMS), a proprietary color space used in a variety of industries,
    notably graphic design, fashion design, product design, printing and manufacturing
    and supporting the management of color from design to production, in physical and digital formats,
    among coated and uncoated materials, cotton, polyester, nylon and plastics.
"""
import typing
from decimal import Decimal

from borb.pdf.canvas.color.color import Color, HexColor, RGBColor


class Pantone(HexColor):
    """
    Pantone LLC is a limited liability company headquartered in Carlstadt, New Jersey.
    The company is best known for its Pantone Matching System (PMS), a proprietary color space used in a variety of industries,
    notably graphic design, fashion design, product design, printing and manufacturing
    and supporting the management of color from design to production, in physical and digital formats,
    among coated and uncoated materials, cotton, polyester, nylon and plastics.
    """

    COLOR_DEFINITION = {
        "abbey-stone": "#aba798",
        "absinthe-green": "#76b583",
        "abyss": "#8f9e9d",
        "acacia": "#dacd65",
        "acai": "#46295a",
        "acid-lime": "#badf30",
        "acorn": "#7e5e52",
        "adobe": "#a3623b",
        "adobe-rose": "#ba9f99",
        "adriatic-blue": "#5c899b",
        "aegean-blue": "#4e6e81",
        "african-violet": "#b085b7",
        "after-dark": "#3c3535",
        "afterglow": "#f3e6c9",
        "agate-gray": "#b1b09f",
        "agate-green": "#599f99",
        "agave-green": "#6b7169",
        "air-blue": "#77acc7",
        "airy-blue": "#92b6d5",
        "alabaster-gleam": "#f0debd",
        "alaskan-blue": "#6da9d2",
        "alesan": "#f1ceb3",
        "aleutian": "#9a9eb3",
        "alfalfa": "#b7b59f",
        "algiers-blue": "#00859c",
        "alhambra": "#008778",
        "alloy": "#98979a",
        "allure": "#7291b4",
        "almond": "#a7754d",
        "almond-blossom": "#f5bec7",
        "almond-buff": "#ccb390",
        "almond-cream": "#f4c29f",
        "almond-milk": "#d6cebe",
        "almond-oil": "#f4efc1",
        "almondine": "#a78c8b",
        "almost-apricot": "#e5b39b",
        "almost-aqua": "#cad3c1",
        "almost-mauve": "#e7dcd9",
        "aloe": "#817a60",
        "aloe-wash": "#d0d3b7",
        "alpine-green": "#005f56",
        "aluminum": "#9f9586",
        "amaranth": "#6f3c56",
        "amaranth-purple": "#6a397b",
        "amazon": "#1f7349",
        "amber": "#efad55",
        "amber-brown": "#a66646",
        "amber-gold": "#c19552",
        "amber-green": "#9a803a",
        "amber-yellow": "#fab75a",
        "amberglow": "#dc793e",
        "amberlight": "#e2bea2",
        "ambrosia": "#d2e7ca",
        "american-beauty": "#a73340",
        "amethyst": "#864d75",
        "amethyst-orchid": "#926aa6",
        "amparo-blue": "#4960a8",
        "amphora": "#9f8672",
        "andorra": "#603535",
        "anemone": "#842c48",
        "angel-blue": "#83c5cd",
        "angel-falls": "#a3bdd3",
        "angel-wing": "#f3dfd7",
        "angora": "#dfd1bb",
        "anise-flower": "#f4e3b5",
        "antarctica": "#c6c5c6",
        "antelope": "#b19664",
        "anthracite": "#28282d",
        "antiqua-sand": "#83c2cd",
        "antique-bronze": "#907954",
        "antique-gold": "#b59e5f",
        "antique-green": "#29675c",
        "antique-moss": "#b9a023",
        "antique-white": "#ede3d2",
        "antler": "#957a76",
        "apple-butter": "#844b4d",
        "apple-cinnamon": "#b0885a",
        "apple-green": "#b5b644",
        "appleblossom": "#ddbca0",
        "apricot": "#f19035",
        "apricot-blush": "#feaea5",
        "apricot-brandy": "#c26a5a",
        "apricot-buff": "#cd7e4d",
        "apricot-cream": "#f1bd89",
        "apricot-gelato": "#f5d7af",
        "apricot-ice": "#fbbe99",
        "apricot-illusion": "#e2c4a6",
        "apricot-nectar": "#ecaa79",
        "apricot-orange": "#c86b3c",
        "apricot-sherbet": "#facd9e",
        "apricot-tan": "#dd9760",
        "apricot-wash": "#fbac82",
        "aqua": "#64a1ad",
        "aqua-foam": "#adc3b4",
        "aqua-glass": "#d2e8e0",
        "aqua-gray": "#a5b2aa",
        "aqua-green": "#00b89f",
        "aqua-haze": "#87b9bc",
        "aqua-sea": "#6baaae",
        "aqua-sky": "#7bc4c4",
        "aqua-splash": "#85ced1",
        "aquamarine": "#9dc3d4",
        "aquarelle": "#61aab1",
        "aquarius": "#3cadd4",
        "aquatic": "#99c1cc",
        "aquifer": "#89acac",
        "arabesque": "#d16f52",
        "arabian-spice": "#884332",
        "aragon": "#b06455",
        "arcadia": "#00a28a",
        "arcadian-green": "#a3c893",
        "arctic": "#648589",
        "arctic-dusk": "#735b6a",
        "arctic-ice": "#bfc7d6",
        "argan-oil": "#8b593e",
        "argyle-purple": "#895c79",
        "arona": "#879ba3",
        "arrowwood": "#bc8d1f",
        "artichoke-green": "#4b6d41",
        "artisans-gold": "#f2ab46",
        "aruba-blue": "#81d7d3",
        "ash": "#a09998",
        "ash-rose": "#b5817d",
        "ashes-of-roses": "#b5acab",
        "ashley-blue": "#8699ab",
        "asparagus-green": "#d2cdb4",
        "aspen-gold": "#ffd662",
        "aspen-green": "#7e9b76",
        "asphalt": "#434447",
        "aster-purple": "#7d74a8",
        "astral-aura": "#363151",
        "atlantic-deep": "#274e55",
        "atlantis": "#00af9f",
        "atmosphere": "#a89c94",
        "auburn": "#a15843",
        "aura-orange": "#b4262a",
        "aurora": "#eddd59",
        "aurora-pink": "#e881a6",
        "aurora-red": "#b93a32",
        "autumn-blaze": "#d9922e",
        "autumn-blonde": "#eed0ae",
        "autumn-glaze": "#b3573f",
        "autumn-glory": "#ff8812",
        "autumn-leaf": "#b56a4c",
        "autumn-maple": "#c46215",
        "autumn-sunset": "#f38554",
        "autumnal": "#a15325",
        "avocado": "#676232",
        "azalea": "#d42e5b",
        "azalea-pink": "#e96a97",
        "aztec": "#7a5747",
        "azure-blue": "#4d91c6",
        "baby-blue": "#b5c7d3",
        "bachelor-button": "#4abbd5",
        "baja-blue": "#5f6db0",
        "baked-apple": "#b34646",
        "baked-clay": "#9c5642",
        "baleine-blue": "#155187",
        "ballad-blue": "#c0ceda",
        "ballerina": "#f2cfdc",
        "ballet-slipper": "#ebced5",
        "balsam": "#33565e",
        "balsam-green": "#576664",
        "baltic": "#279d9f",
        "baltic-sea": "#79b5db",
        "bamboo": "#d2b04c",
        "banana": "#fcb953",
        "banana-cream": "#ffcf73",
        "banana-crepe": "#e7d3ad",
        "barbados-cherry": "#aa0a27",
        "barberry": "#bf1945",
        "barely-blue": "#dde0df",
        "barely-pink": "#f8d7dd",
        "bark": "#a99592",
        "barn-red": "#8f423b",
        "baroque-rose": "#b35a66",
        "barrier-reef": "#0084a1",
        "basil": "#879f84",
        "baton-rouge": "#973c6c",
        "bay": "#bae5d6",
        "bayberry": "#255958",
        "bayou": "#20706f",
        "beach-glass": "#96dfce",
        "beach-sand": "#fbb995",
        "beaujolais": "#80304c",
        "beaver-fur": "#997867",
        "beech": "#5b4f3b",
        "beechnut": "#c2c18d",
        "beeswax": "#eba851",
        "beet-red": "#7a1f3d",
        "beetle": "#55584c",
        "beetroot-purple": "#cf2d71",
        "begonia-pink": "#ec9abe",
        "beige": "#d5ba98",
        "bel-air-blue": "#819ac1",
        "belgian-block": "#a3a9a6",
        "bellflower": "#9469a2",
        "bellini": "#f4c9b1",
        "beluga": "#4a4843",
        "bering-sea": "#4b5b6e",
        "bermuda": "#60c9b3",
        "berry-conserve": "#765269",
        "beryl-green": "#619187",
        "beveled-glass": "#7accb8",
        "bijou-blue": "#4e5e7f",
        "biking-red": "#77212e",
        "billiard": "#00aa92",
        "billowing-sail": "#d8e7e7",
        "birch": "#ddd5c7",
        "bird-of-paradise": "#ff8c55",
        "birds-egg-green": "#aaccb9",
        "biscay-bay": "#097988",
        "biscay-green": "#55c6a9",
        "biscotti": "#dac7ab",
        "biscuit": "#b4835b",
        "bison": "#6e4f3a",
        "bisque": "#edcab5",
        "bistre": "#98754a",
        "bistro-green": "#395551",
        "bit-of-blue": "#e2eaeb",
        "bitter-chocolate": "#503130",
        "bittersweet": "#d93744",
        "black-bean": "#2e272a",
        "black-beauty": "#26262a",
        "black-coffee": "#3b302f",
        "black-forest": "#414f3c",
        "black-ink": "#44413c",
        "black-iris": "#2b3042",
        "black-olive": "#48413b",
        "black-onyx": "#2b272b",
        "black-plum": "#6c5765",
        "blackberry-cordial": "#3f2a47",
        "blackberry-wine": "#4d3246",
        "blackened-pearl": "#4d4b50",
        "blanc-de-blanc": "#e7e9e7",
        "blarney": "#00a776",
        "blazing-orange": "#ffa64f",
        "blazing-yellow": "#fee715",
        "bleached-apricot": "#fccaac",
        "bleached-aqua": "#bce3df",
        "bleached-denim": "#646f9b",
        "bleached-sand": "#daccb4",
        "blithe": "#0084bd",
        "blooming-dahlia": "#eb9687",
        "blossom": "#f2b2ae",
        "blue-ashes": "#3b5f78",
        "blue-aster": "#0077b3",
        "blue-atoll": "#00b1d2",
        "blue-bell": "#93b4d7",
        "blue-blush": "#d6dbd9",
        "blue-bonnet": "#6384b8",
        "blue-coral": "#1b5366",
        "blue-curacao": "#32becc",
        "blue-danube": "#0087b6",
        "blue-depths": "#263056",
        "blue-flower": "#d0d9d4",
        "blue-fog": "#9babbb",
        "blue-fox": "#b9bcb6",
        "blue-glass": "#c6e3e1",
        "blue-glow": "#b2d4dd",
        "blue-granite": "#717388",
        "blue-graphite": "#323137",
        "blue-grass": "#007c7a",
        "blue-grotto": "#5cacce",
        "blue-haze": "#a5bcbb",
        "blue-heaven": "#5b7e98",
        "blue-heron": "#96a3c7",
        "blue-horizon": "#4e6482",
        "blue-ice": "#70789b",
        "blue-indigo": "#49516d",
        "blue-iris": "#5a5b9f",
        "blue-jewel": "#007baa",
        "blue-light": "#acdfdd",
        "blue-mirage": "#5c6d7c",
        "blue-mist": "#5bacc3",
        "blue-moon": "#3686a0",
        "blue-nights": "#363b48",
        "blue-opal": "#0f3b57",
        "blue-radiance": "#58c9d4",
        "blue-ribbon": "#3a395f",
        "blue-sapphire": "#09577b",
        "blue-shadow": "#66829a",
        "blue-spruce": "#486b67",
        "blue-surf": "#90a8a4",
        "blue-tint": "#9fd9d7",
        "blue-topaz": "#78bdd4",
        "blue-turquoise": "#53b0ae",
        "blue-wing-teal": "#2c4053",
        "blue-yonder": "#5a77a8",
        "blueberry": "#2c333e",
        "bluebird": "#009dae",
        "bluejay": "#157ea0",
        "blueprint": "#2d3359",
        "bluesteel": "#35637c",
        "bluestone": "#577284",
        "bluewash": "#e2e6e0",
        "blush": "#d1969a",
        "blushing-bride": "#fbd3d9",
        "boa": "#8e855f",
        "bodacious": "#b76ba3",
        "bog": "#bab696",
        "bok-choy": "#bccab3",
        "bombay-brown": "#9f5130",
        "bone-brown": "#9d7446",
        "bone-white": "#d7d0c0",
        "bonnie-blue": "#539ccc",
        "bordeaux": "#96637b",
        "bosphorus": "#007558",
        "bossa-nova": "#973a36",
        "botanical-garden": "#12403c",
        "bottle-green": "#427d6d",
        "bougainvillea": "#9884b9",
        "boulder": "#d1be9b",
        "boysenberry": "#85325c",
        "bracken": "#4f3f3b",
        "bran": "#a66e4a",
        "brandied-apricot": "#ca848a",
        "brandied-melon": "#ce7b5b",
        "brandy-brown": "#73362a",
        "brazilian-sand": "#dacab7",
        "breen": "#795d34",
        "brick-dust": "#b07069",
        "brick-red": "#8c373e",
        "bridal-blush": "#eee2dd",
        "bridal-rose": "#d69fa2",
        "bright-aqua": "#30a299",
        "bright-chartreuse": "#b5bf50",
        "bright-cobalt": "#385d8d",
        "bright-gold": "#cf9f52",
        "bright-green": "#009b5c",
        "bright-lime-green": "#97bc62",
        "bright-marigold": "#ff8d00",
        "bright-rose": "#c51959",
        "bright-violet": "#784384",
        "bright-white": "#f4f5f0",
        "brilliant-blue": "#0075b3",
        "brilliant-white": "#edf1fe",
        "brindle": "#82776b",
        "bristol-blue": "#558f91",
        "brittany-blue": "#4c7e86",
        "bronze-brown": "#825e2f",
        "bronze-green": "#525f48",
        "bronze-mist": "#9c7e41",
        "brook-green": "#afddcc",
        "brown-patina": "#834f3d",
        "brown-rice": "#c7bba4",
        "brown-stone": "#593c39",
        "brown-sugar": "#a17249",
        "brownie": "#8f7265",
        "brunette": "#664238",
        "brunnera-blue": "#9ba9ca",
        "bruschetta": "#a75949",
        "brush": "#b99984",
        "brushed-nickel": "#73706f",
        "bubblegum": "#ea738d",
        "buckskin": "#d18e54",
        "buckthorn-brown": "#a76f1f",
        "bud-green": "#79b465",
        "buff": "#ebc396",
        "buff-orange": "#ffbb7c",
        "buff-yellow": "#f1bf70",
        "bungee-cord": "#696156",
        "burgundy": "#64313e",
        "burlwood": "#9b716b",
        "burnished-gold": "#aa9855",
        "burnished-lilac": "#c5aeb1",
        "burnt-brick": "#a14d3a",
        "burnt-coral": "#e9897e",
        "burnt-henna": "#7e392f",
        "burnt-ochre": "#bb4f35",
        "burnt-olive": "#646049",
        "burnt-orange": "#c86733",
        "burnt-russet": "#7e3940",
        "burnt-sienna": "#c65d52",
        "burro": "#947764",
        "buttercream": "#efe0cd",
        "buttercup": "#fae03c",
        "butterfly": "#cadea5",
        "butternut": "#7a643f",
        "butterscotch": "#e19640",
        "butterum": "#c68f65",
        "byzantium": "#853b7b",
        "cabaret": "#cb3373",
        "cabbage": "#87d7be",
        "cabernet": "#64242e",
        "cactus": "#53713d",
        "cactus-flower": "#a83e6c",
        "cadet": "#6a6378",
        "cadmium-green": "#00675b",
        "cadmium-orange": "#f99471",
        "cadmium-yellow": "#ee9626",
        "cafe-au-lait": "#ae8774",
        "cafe-creme": "#c79685",
        "calla-green": "#6a6f34",
        "calliste-green": "#757a4e",
        "calypso-coral": "#ee5c6c",
        "camel": "#b0846a",
        "camellia": "#f6745f",
        "camellia-rose": "#eb6081",
        "cameo-blue": "#769da6",
        "cameo-brown": "#c08a80",
        "cameo-green": "#aac0ad",
        "cameo-pink": "#dba9b8",
        "cameo-rose": "#d7b8ab",
        "campanula": "#3272af",
        "canal-blue": "#9cc2c5",
        "canary-green": "#d6dec9",
        "canary-yellow": "#dfd87e",
        "candied-ginger": "#bfa387",
        "candlelight-peach": "#f8a39d",
        "candy-pink": "#f5b0bd",
        "caneel-bay": "#00849f",
        "cannoli-cream": "#f0efe2",
        "cantaloupe": "#ffa177",
        "canteen": "#5e5347",
        "canton": "#6da29e",
        "canyon-clay": "#ce8477",
        "canyon-rose": "#af6c67",
        "canyon-sunset": "#e1927a",
        "capers": "#695e4b",
        "cappuccino": "#633f33",
        "capri": "#44bbca",
        "capri-breeze": "#008799",
        "captains-blue": "#557088",
        "capulet-olive": "#656344",
        "carafe": "#5d473a",
        "caramel": "#c37c54",
        "caramel-cafe": "#864c24",
        "caramel-cream": "#f4ba94",
        "carbon": "#272f38",
        "cardinal": "#ad3e48",
        "caribbean-sea": "#00819d",
        "caribou": "#816d5e",
        "carmine": "#bc4869",
        "carmine-rose": "#e35b8f",
        "carnelian": "#ce785d",
        "carob-brown": "#855c4c",
        "carrot": "#fd6f3b",
        "carrot-curl": "#fe8c18",
        "cascade": "#76c1b2",
        "cashew": "#a47149",
        "cashmere-blue": "#a5b8d0",
        "cashmere-rose": "#ce879f",
        "castle-wall": "#c8c1ab",
        "castlerock": "#5f5e62",
        "castor-gray": "#646762",
        "catawba-grape": "#5d3c43",
        "cathay-spice": "#99642c",
        "caviar": "#292a2d",
        "cayenne": "#e04951",
        "cedar": "#928e64",
        "cedar-green": "#5e6737",
        "cedar-wood": "#a1655b",
        "celadon": "#b8ccba",
        "celadon-green": "#b5c1a5",
        "celadon-tint": "#cbcebe",
        "celandine": "#ebdf67",
        "celery": "#cec153",
        "celery-green": "#c5cc7b",
        "celestial": "#006380",
        "celestial-blue": "#a3b4c4",
        "celosia-orange": "#e8703a",
        "cement": "#c4b6a6",
        "cendre-blue": "#3e7fa5",
        "ceramic": "#00aaa9",
        "cerise": "#a41247",
        "cerulean": "#9bb7d4",
        "ceylon-yellow": "#d4ae40",
        "chai-tea": "#b1832f",
        "chalk-blue": "#ccdad7",
        "chalk-pink": "#e6c5ca",
        "chalk-violet": "#8f7da5",
        "chambray-blue": "#9eb4d3",
        "chamois": "#f7b26a",
        "chamomile": "#e8d0a7",
        "chanterelle": "#a28776",
        "charcoal-gray": "#6c6868",
        "chardonnay": "#e7df99",
        "charisma": "#632a60",
        "charlock": "#e5e790",
        "chateau-gray": "#bbb1a8",
        "chateau-rose": "#d2738f",
        "cherries-jubilee": "#a22452",
        "cherry-blossom": "#f7cee0",
        "cherry-mahogany": "#66352b",
        "cherry-tomato": "#eb3c27",
        "chestnut": "#584039",
        "chicory-coffee": "#4a342e",
        "chili": "#be5141",
        "chili-oil": "#8e3c36",
        "chili-pepper": "#9b1b30",
        "china-blue": "#546477",
        "chinchilla": "#9c8e7b",
        "chinese-red": "#be132d",
        "chinese-violet": "#835e81",
        "chino-green": "#d9caa5",
        "chinois-green": "#7c8c87",
        "chintz-rose": "#eec4be",
        "chipmunk": "#976f4c",
        "chive": "#4a5335",
        "chive-blossom": "#7d5d99",
        "chocolate-brown": "#4e403b",
        "chocolate-chip": "#685a4e",
        "chocolate-fondant": "#56352d",
        "chocolate-lab": "#5c3e35",
        "chocolate-plum": "#3c2d2e",
        "chocolate-torte": "#382e2d",
        "chocolate-truffle": "#612e35",
        "chrysanthemum": "#be454f",
        "chutney": "#98594b",
        "cilantro": "#43544b",
        "cinder": "#8a7e78",
        "cinnabar": "#9c453b",
        "cinnamon": "#6b4139",
        "cinnamon-stick": "#9b4722",
        "citadel": "#748995",
        "citron": "#dfde9b",
        "citronelle": "#b8af23",
        "citrus": "#f9ac2f",
        "claret-red": "#c84c61",
        "classic-blue": "#0f4c81",
        "classic-green": "#39a845",
        "clay": "#d2a172",
        "clearly-aqua": "#cee1d4",
        "clearwater": "#aad5db",
        "clematis-blue": "#363b7c",
        "climbing-ivy": "#444940",
        "cloisonne": "#0075af",
        "cloud-blue": "#a2b6b9",
        "cloud-cream": "#e6ddc5",
        "cloud-dancer": "#f0eee9",
        "cloud-gray": "#b7a9ac",
        "cloud-pink": "#f5d1c8",
        "cloudburst": "#837f7f",
        "clove": "#876155",
        "clover": "#8a3371",
        "coastal-fjord": "#505d7e",
        "cobblestone": "#a89a8e",
        "cockatoo": "#58c8b6",
        "cocoa-brown": "#6c5043",
        "coconut-milk": "#f0ede5",
        "coconut-shell": "#874e3c",
        "cocoon": "#c9b27c",
        "coffee-bean": "#40312f",
        "coffee-liqueur": "#6a513b",
        "cognac": "#8b645a",
        "colonial-blue": "#2d6471",
        "colony-blue": "#65769a",
        "columbia": "#009288",
        "comfrey": "#5b7961",
        "conch-shell": "#fc8f9b",
        "concord-grape": "#7c5379",
        "confetti": "#e6798e",
        "cool-blue": "#a5c5d9",
        "copen-blue": "#516b84",
        "copper": "#c47e5a",
        "copper-brown": "#9a6051",
        "copper-coin": "#ba6b57",
        "copper-tan": "#de8e65",
        "coral": "#ed7464",
        "coral-almond": "#e29d94",
        "coral-blush": "#e6b2b8",
        "coral-cloud": "#e2a9a1",
        "coral-gold": "#d27d56",
        "coral-haze": "#e38e84",
        "coral-pink": "#e8a798",
        "coral-quartz": "#f77464",
        "coral-reef": "#faa181",
        "coral-rose": "#f3774d",
        "coral-sands": "#edaa86",
        "cordovan": "#702f3b",
        "coriander": "#938772",
        "cork": "#ba8671",
        "cornflower-blue": "#7391c8",
        "cornhusk": "#f2d6ae",
        "cornsilk": "#edc373",
        "cornstalk": "#a9947a",
        "coronet-blue": "#59728e",
        "corsair": "#18576c",
        "corsican-blue": "#646093",
        "corydalis-blue": "#a9cada",
        "cosmic-sky": "#aaaac4",
        "country-blue": "#717f9b",
        "covert-green": "#80765f",
        "cowhide": "#884344",
        "crabapple": "#d77e70",
        "cradle-pink": "#edd0dd",
        "cranberry": "#bb4a4d",
        "cream-blush": "#f8c19a",
        "cream-gold": "#dec05f",
        "cream-pink": "#f6e4d9",
        "cream-tan": "#e4c7b8",
        "creampuff": "#ffcda8",
        "creme-brulee": "#dbccb5",
        "creme-de-menthe": "#70a38d",
        "creme-de-peche": "#f5d6c6",
        "creole-pink": "#f7d5cc",
        "cress-green": "#bca949",
        "crimson": "#ae0e36",
        "crockery": "#a49887",
        "crocodile": "#5d5348",
        "crocus": "#c67fae",
        "crocus-petal": "#b99bc5",
        "croissant": "#c4ab86",
        "crown-blue": "#464b65",
        "crown-jewel": "#482d54",
        "crushed-berry": "#804f5a",
        "crushed-grape": "#7a547f",
        "crushed-violets": "#643a4c",
        "crystal-blue": "#a1c8db",
        "crystal-gray": "#d7cbc4",
        "crystal-pink": "#edd0ce",
        "crystal-rose": "#fdc3c6",
        "crystal-seas": "#5dafce",
        "crystal-teal": "#00637c",
        "cub": "#6e5c4b",
        "cuban-sand": "#c1a68d",
        "cumin": "#927240",
        "curds-and-whey": "#bca483",
        "curry": "#be9e6f",
        "custard": "#e5d68e",
        "cyan-blue": "#14a3c7",
        "cyber-yellow": "#ffd400",
        "cyclamen": "#d687ba",
        "cypress": "#545a3e",
        "dachshund": "#704f37",
        "daffodil": "#fdc04e",
        "dahlia": "#843e83",
        "dahlia-mauve": "#a64f82",
        "dahlia-purple": "#7e6eac",
        "daiquiri-green": "#c9d77e",
        "damson": "#854c65",
        "dandelion": "#ffd02e",
        "daphne": "#0f5f9a",
        "dapple-gray": "#9c9ba7",
        "dark-blue": "#305679",
        "dark-cheddar": "#e08119",
        "dark-citron": "#a0ac4f",
        "dark-denim": "#35465e",
        "dark-earth": "#5c4939",
        "dark-forest": "#556962",
        "dark-green": "#314f40",
        "dark-gull-gray": "#625d5d",
        "dark-ivy": "#5b7763",
        "dark-navy": "#232f36",
        "dark-olive": "#574d35",
        "dark-purple": "#582147",
        "dark-sapphire": "#262b37",
        "dark-shadow": "#4a4b4d",
        "dark-slate": "#46515a",
        "darkest-spruce": "#303d3c",
        "dawn": "#ebd2b7",
        "dawn-blue": "#cacccb",
        "dawn-pink": "#bfa3af",
        "daybreak": "#8981a0",
        "dazzling-blue": "#3850a0",
        "deauville-mauve": "#af9294",
        "decadent-chocolate": "#513235",
        "december-sky": "#767275",
        "deco-rose": "#985f68",
        "deep-blue": "#44377d",
        "deep-claret": "#973443",
        "deep-cobalt": "#404466",
        "deep-depths": "#46483c",
        "deep-dive": "#29495c",
        "deep-forest": "#37413a",
        "deep-grass-green": "#558367",
        "deep-green": "#009276",
        "deep-jungle": "#36716f",
        "deep-lagoon": "#005265",
        "deep-lake": "#00656b",
        "deep-lavender": "#775496",
        "deep-lichen-green": "#6e6e5c",
        "deep-mahogany": "#553b39",
        "deep-mint": "#009e6d",
        "deep-orchid": "#903f75",
        "deep-peacock-blue": "#008381",
        "deep-periwinkle": "#7c83bc",
        "deep-purple": "#50314c",
        "deep-sea": "#4f7c74",
        "deep-sea-coral": "#d9615b",
        "deep-taupe": "#7b6660",
        "deep-teal": "#18454b",
        "deep-ultramarine": "#384883",
        "deep-water": "#266691",
        "deep-well": "#2c2a33",
        "deep-wisteria": "#443f6f",
        "delft": "#3d5e8c",
        "delicacy": "#f5e3e2",
        "delicioso": "#3f352f",
        "della-robbia-blue": "#7a9dcb",
        "delphinium-blue": "#6198ae",
        "demitasse": "#40342b",
        "desert-dust": "#e3bc8e",
        "desert-flower": "#ff9687",
        "desert-mist": "#e0b589",
        "desert-palm": "#5a4632",
        "desert-rose": "#cf6977",
        "desert-sage": "#a7ae9e",
        "desert-sand": "#bd7b74",
        "desert-sun": "#c87629",
        "desert-taupe": "#8d7e71",
        "dew": "#eeded1",
        "dewberry": "#8b5987",
        "dewkist": "#c4d1c2",
        "diffused-orchid": "#9879a2",
        "dijon": "#97754c",
        "dill": "#6f7755",
        "directoire-blue": "#0061a3",
        "diva-blue": "#007bb2",
        "doe": "#b98e68",
        "doeskin": "#bdab9b",
        "double-cream": "#f3e0ac",
        "dove": "#b3ada7",
        "dragon-fire": "#fc642d",
        "dragonfly": "#2a5c6a",
        "dream-blue": "#a0bcd0",
        "dresden-blue": "#0086bb",
        "dress-blues": "#2a3244",
        "dried-herb": "#847a59",
        "dried-moss": "#ccb97e",
        "dried-tobacco": "#997b38",
        "driftwood": "#847a75",
        "drizzle": "#a09f9c",
        "dry-rose": "#8c4759",
        "dubarry": "#f25f66",
        "duck-green": "#53665c",
        "duffel-bag": "#394034",
        "dull-gold": "#8a6f48",
        "dune": "#998978",
        "dusk": "#897f98",
        "dusk-blue": "#7ba0c0",
        "dusky-citron": "#e3cc81",
        "dusky-green": "#746c57",
        "dusky-orchid": "#9a7182",
        "dusted-clay": "#cc7357",
        "dusted-peri": "#696ba0",
        "dusty-aqua": "#c0dccd",
        "dusty-blue": "#8c9dad",
        "dusty-cedar": "#ad5d5d",
        "dusty-coral": "#d29b83",
        "dusty-jade-green": "#7bb5a3",
        "dusty-lavender": "#a1759c",
        "dusty-olive": "#646356",
        "dusty-orange": "#e27a53",
        "dusty-pink": "#deaa9b",
        "dusty-rose": "#ba797d",
        "dusty-turquoise": "#649b9e",
        "dusty-yellow": "#d4cc9a",
        "dutch-blue": "#4a638d",
        "dynasty-green": "#008e80",
        "earth-red": "#95424e",
        "easter-egg": "#919bc9",
        "ebony": "#41424a",
        "eclipse": "#343148",
        "ecru": "#f3dfca",
        "ecru-olive": "#927b3c",
        "eden": "#264e36",
        "eggnog": "#ece1d3",
        "eggplant": "#613f4c",
        "eggshell-blue": "#a3ccc9",
        "egret": "#f3ece0",
        "eiffel-tower": "#5c5658",
        "elderberry": "#9d848e",
        "electric-green": "#4bc3a8",
        "elephant-skin": "#8f8982",
        "elfin-yellow": "#eeea97",
        "elm": "#a39f86",
        "elm-green": "#547053",
        "elmwood": "#8c7c61",
        "emberglow": "#ea6759",
        "emerald": "#009473",
        "emperador": "#684832",
        "empire-yellow": "#f7d000",
        "enamel-blue": "#007a8e",
        "endive": "#d2cc81",
        "english-ivy": "#61845b",
        "english-lavendar": "#9d7bb0",
        "english-manor": "#7181a4",
        "english-rose": "#f4c6c3",
        "ensign-blue": "#384c67",
        "ephemera": "#6f5965",
        "epsom": "#849161",
        "ermine": "#836b4f",
        "espresso": "#363031",
        "estate-blue": "#233658",
        "ether": "#9eb6b8",
        "etherea": "#a5958f",
        "ethereal-blue": "#5ca6ce",
        "ethereal-green": "#f1ecca",
        "etruscan-red": "#a2574b",
        "eucalyptus": "#b1a992",
        "evening-blue": "#2a293e",
        "evening-haze": "#bdb8c7",
        "evening-primrose": "#ccdb1e",
        "evening-sand": "#ddb6ab",
        "eventide": "#959eb7",
        "everglade": "#005b5d",
        "evergreen": "#11574a",
        "excalibur": "#676168",
        "exotic-orange": "#f96531",
        "exuberance": "#e86800",
        "faded-denim": "#798ea4",
        "faded-rose": "#bf6464",
        "faience": "#2a6a8b",
        "fair-aqua": "#b8e2dc",
        "fair-green": "#92af88",
        "fair-orchid": "#c0aac0",
        "fairest-jade": "#d8e3d7",
        "fairway": "#477050",
        "fairy-tale": "#f2c1d1",
        "fairy-wren": "#9479af",
        "falcon": "#6d625b",
        "fall-leaf": "#c9a86a",
        "fallen-rock": "#807669",
        "fandango-pink": "#e04f80",
        "fanfare": "#006d70",
        "fawn": "#ae9490",
        "feather-gray": "#b8ad9e",
        "federal-blue": "#43628b",
        "feldspar": "#729b8b",
        "fennel-seed": "#998456",
        "fenugreek": "#c0916c",
        "fern": "#9aa067",
        "fern-green": "#008c45",
        "festival-fuchsia": "#9e2c6a",
        "fiery-red": "#d01c1f",
        "fiesta": "#dd4132",
        "fig": "#532d3b",
        "fir": "#3a725f",
        "fir-green": "#67592a",
        "firecracker": "#f36944",
        "fired-brick": "#6a2e2a",
        "fjord-blue": "#007290",
        "flame": "#f2552c",
        "flame-orange": "#fb8b23",
        "flame-scarlet": "#cd212a",
        "flamingo": "#df7253",
        "flamingo-pink": "#f7969e",
        "flan": "#f6e3b4",
        "flax": "#ffc87d",
        "flint": "#705861",
        "flint-gray": "#a09c98",
        "flint-stone": "#677283",
        "florida-keys": "#56beab",
        "fluorite-green": "#699158",
        "foam-green": "#b4c79c",
        "fog": "#d0c5b1",
        "fog-green": "#c2cbb4",
        "foggy-dew": "#d1d5d0",
        "foliage": "#75a14f",
        "foliage-green": "#3e6f58",
        "folkstone-gray": "#626879",
        "fondue-fudge": "#5d4236",
        "forest-biome": "#184a45",
        "forest-green": "#6b8d53",
        "forest-night": "#434237",
        "forest-shade": "#91ac80",
        "forever-blue": "#899bb8",
        "forged-iron": "#48464a",
        "forget-me-not": "#8fadbd",
        "fossil": "#806f63",
        "four-leaf-clover": "#616652",
        "foxglove": "#b98391",
        "fragrant-lilac": "#ceadbe",
        "frappe": "#d1b7a0",
        "freesia": "#f3c12c",
        "french-blue": "#0072b5",
        "french-roast": "#58423f",
        "french-vanilla": "#efe1a7",
        "fresh-salmon": "#ff7f6a",
        "friar-brown": "#6e493a",
        "frost": "#dde2d6",
        "frost-gray": "#848283",
        "frosted-almond": "#d2c2ac",
        "frosty-green": "#a3b5a6",
        "frosty-spruce": "#578270",
        "frozen-dew": "#d8cfb2",
        "fruit-dove": "#ce5b78",
        "fuchsia-pink": "#df88b7",
        "fuchsia-purple": "#d33479",
        "fuchsia-red": "#ab3475",
        "fuchsia-rose": "#c74375",
        "fudge": "#493338",
        "fudgesickle": "#63403a",
        "fungi": "#8f8177",
        "fusion-coral": "#ff8576",
        "galapagos-green": "#29685f",
        "galaxy-blue": "#2a4b7c",
        "ganache": "#34292a",
        "garden-glade": "#dcd8a8",
        "garden-green": "#495e35",
        "garden-topiary": "#3e524b",
        "gardenia": "#f1e8df",
        "gargoyle": "#686767",
        "garnet": "#953640",
        "garnet-rose": "#ac4b55",
        "gentian-violet": "#544275",
        "georgia-peach": "#f97272",
        "geranium": "#da3d58",
        "geranium-pink": "#f6909d",
        "ghost-gray": "#9c9b98",
        "gibraltar-sea": "#123850",
        "gilded-beige": "#b39f8d",
        "ginger": "#c96551",
        "ginger-bread": "#8c4a2f",
        "ginger-root": "#bfa58a",
        "ginger-snap": "#977d70",
        "ginger-spice": "#b65d48",
        "glacier": "#c3dbd4",
        "glacier-gray": "#c5c6c7",
        "glass-green": "#ecead0",
        "glazed-ginger": "#91552b",
        "gleam": "#bfd1ad",
        "gloxinia": "#622e5a",
        "goat": "#a89a91",
        "goblin-blue": "#5f7278",
        "goji-berry": "#b91228",
        "gold-earth": "#dd9c6b",
        "gold-flame": "#b45422",
        "gold-fusion": "#ffb000",
        "golden-apricot": "#dda758",
        "golden-brown": "#91672f",
        "golden-cream": "#f7b768",
        "golden-fleece": "#f2d1a0",
        "golden-glow": "#d99938",
        "golden-green": "#bdb369",
        "golden-haze": "#fbd897",
        "golden-kiwi": "#f3dd3e",
        "golden-lime": "#9a9738",
        "golden-mist": "#d5cd94",
        "golden-nugget": "#db9b59",
        "golden-oak": "#be752d",
        "golden-ochre": "#c77943",
        "golden-olive": "#af9841",
        "golden-orange": "#d7942d",
        "golden-palm": "#aa8805",
        "golden-poppy": "#f56733",
        "golden-rod": "#e2a829",
        "golden-spice": "#c6973f",
        "golden-straw": "#e6bd8f",
        "golden-yellow": "#cb8e16",
        "goldfinch": "#f8dc6c",
        "golf-green": "#008763",
        "gossamer-green": "#b2cfbe",
        "gossamer-pink": "#fac8c3",
        "gothic-grape": "#473951",
        "gothic-olive": "#7c6e4f",
        "granada-sky": "#5d81bb",
        "granita": "#a52350",
        "granite-gray": "#615e5f",
        "granite-green": "#86a293",
        "grape": "#433455",
        "grape-compote": "#6b5876",
        "grape-jam": "#725671",
        "grape-juice": "#682961",
        "grape-kiss": "#7b4368",
        "grape-leaf": "#545144",
        "grape-nectar": "#8d5c74",
        "grape-royale": "#4f2d54",
        "grape-shake": "#886971",
        "grape-wine": "#5a2f43",
        "grapeade": "#85677b",
        "grapemist": "#8398ca",
        "graphite": "#3b3b48",
        "grass-green": "#7bb369",
        "grasshopper": "#77824a",
        "gravel": "#cbbfa2",
        "gray-blue": "#4d587a",
        "gray-dawn": "#bbc1cc",
        "gray-flannel": "#848182",
        "gray-green": "#a49a79",
        "gray-lilac": "#d4cacd",
        "gray-mist": "#99aeae",
        "gray-morn": "#cabeb5",
        "gray-pinstripe": "#49494d",
        "gray-ridge": "#847986",
        "gray-sand": "#e5ccaf",
        "gray-violet": "#bbbcbc",
        "grayed-jade": "#9bbea9",
        "graystone": "#4d495b",
        "green-ash": "#a0daa9",
        "green-banana": "#babc72",
        "green-bay": "#7e9285",
        "green-blue-slate": "#358082",
        "green-essence": "#e9eac8",
        "green-eyes": "#7d956d",
        "green-flash": "#79c753",
        "green-gables": "#324241",
        "green-glow": "#b0c965",
        "green-haze": "#cac4a4",
        "green-lily": "#c1cec1",
        "green-milieu": "#8a9992",
        "green-moss": "#857946",
        "green-oasis": "#b0b454",
        "green-olive": "#8d8b55",
        "green-sheen": "#d9ce52",
        "green-spruce": "#589f7e",
        "green-sulphur": "#ae8e2c",
        "green-tea": "#86a96f",
        "green-tint": "#c5ccc0",
        "greenbriar": "#4b9b69",
        "greener-pastures": "#37503d",
        "greenery": "#88b04b",
        "greengage": "#8bc28c",
        "greenlake": "#007d69",
        "greige": "#928475",
        "grenadine": "#df3f32",
        "griffin": "#8d8f8f",
        "grisaille": "#585e6f",
        "guacamole": "#797b3a",
        "gulf-stream": "#88c3d0",
        "gull": "#918c8f",
        "gull-gray": "#a49ca0",
        "gumdrop-green": "#2ea785",
        "gunmetal": "#5c5d5b",
        "habanero-gold": "#fed450",
        "halogen-blue": "#bdc6dc",
        "harbor-blue": "#00656e",
        "harbor-gray": "#a8c0bb",
        "harbor-mist": "#afb1b4",
        "harvest-gold": "#b68a3a",
        "harvest-pumpkin": "#d56231",
        "haute-red": "#a11729",
        "hawaiian-ocean": "#008db9",
        "hawaiian-sunset": "#bb5c14",
        "hawaiian-surf": "#0078a7",
        "hawthorn-rose": "#884c5e",
        "hay": "#d3cca3",
        "hazel": "#ae7250",
        "hazelnut": "#cfb095",
        "heather": "#b7c0d6",
        "heather-rose": "#ad6d7f",
        "heavenly-pink": "#f4dede",
        "hedge-green": "#768a75",
        "heirloom-lilac": "#9d96b2",
        "heliotrope": "#4f3872",
        "hemlock": "#97c1a1",
        "hemp": "#c0ad7c",
        "henna": "#7c423c",
        "herbal-garden": "#9cad60",
        "heritage-blue": "#5d96bc",
        "heron": "#62617e",
        "hibiscus": "#dd3848",
        "high-rise": "#aeb2b5",
        "high-risk-red": "#c71f2d",
        "hint-of-mint": "#d8e8e6",
        "holiday": "#81c3b4",
        "holly-berry": "#b44e5d",
        "holly-green": "#0f9d76",
        "hollyhock": "#823270",
        "honey": "#ba9238",
        "honey-ginger": "#a86217",
        "honey-gold": "#d1a054",
        "honey-mustard": "#b68f52",
        "honey-peach": "#dcbd9e",
        "honey-yellow": "#ca9456",
        "honeydew": "#bae1d3",
        "honeysuckle": "#d94f70",
        "horizon-blue": "#289dbe",
        "hortensia": "#553b50",
        "hot-chocolate": "#683b39",
        "hot-coral": "#f35b53",
        "hot-pink": "#e55982",
        "hot-sauce": "#ab4f41",
        "huckleberry": "#5b4349",
        "humus": "#b7a793",
        "hunter-green": "#335749",
        "hushed-green": "#d8e9e5",
        "hushed-violet": "#d1c0bf",
        "hyacinth": "#936ca7",
        "hyacinth-violet": "#8d4687",
        "hydrangea": "#849bcc",
        "hydro": "#426972",
        "ibis-rose": "#ca628f",
        "ibiza-blue": "#007cb7",
        "ice": "#e0e4d9",
        "ice-flow": "#c6d2d2",
        "ice-green": "#87d8c3",
        "iceberg-green": "#8c9c92",
        "iced-aqua": "#abd3db",
        "iced-coffee": "#b18f6a",
        "iceland-poppy": "#f4963a",
        "icelandic-blue": "#a9adc2",
        "icicle": "#dadcd0",
        "icy-morn": "#b0d3d1",
        "iguana": "#818455",
        "illusion-blue": "#c9d3dc",
        "impala": "#f8ce97",
        "impatiens-pink": "#ffc4bc",
        "imperial-blue": "#005a92",
        "imperial-palace": "#604e7a",
        "imperial-purple": "#542c5d",
        "inca-gold": "#bb7a2c",
        "incense": "#af9a7e",
        "india-ink": "#3c3f4a",
        "indian-tan": "#ad8567",
        "indian-teal": "#3c586b",
        "indigo": "#4c3957",
        "indigo-bunting": "#006ca9",
        "infinity": "#6e7e99",
        "ink-blue": "#0b5369",
        "insignia-blue": "#2f3e55",
        "iolite": "#707bb4",
        "iris": "#baafbc",
        "iris-bloom": "#5b609e",
        "iris-orchid": "#a767a2",
        "irish-cream": "#c0ac92",
        "irish-green": "#45be76",
        "iron": "#736460",
        "iron-gate": "#4e5055",
        "island-green": "#2bae66",
        "island-paradise": "#95dee3",
        "italian-plum": "#533146",
        "italian-straw": "#e7d1a1",
        "ivory-cream": "#dac0a7",
        "ivy": "#226c63",
        "ivy-green": "#585442",
        "jacaranda": "#848dc5",
        "jade-cream": "#60b892",
        "jade-green": "#759465",
        "jade-lime": "#a1ca7b",
        "jadeite": "#95a69f",
        "jadesheen": "#77a276",
        "jaffa-orange": "#d86d39",
        "jalapeno-red": "#b2103c",
        "jasmine-green": "#7ec845",
        "jasper": "#335959",
        "java": "#433331",
        "jazzy": "#b61c50",
        "jelly-bean": "#008658",
        "jester-red": "#9e1030",
        "jet-black": "#2d2c2f",
        "jet-set": "#262c2a",
        "jet-stream": "#ede6de",
        "jojoba": "#dabe81",
        "jolly-green": "#007844",
        "june-bug": "#264a48",
        "jungle-green": "#3c4e47",
        "juniper": "#3d7245",
        "jurassic-gold": "#e7aa56",
        "kalamata": "#5f5b4c",
        "kale": "#5a7247",
        "kangaroo": "#725e43",
        "kashmir": "#6f8d6a",
        "katydid": "#66bc91",
        "keepsake-lilac": "#c0a5ae",
        "kelly-green": "#339c5e",
        "kelp": "#988467",
        "kentucky-blue": "#a5b3cc",
        "ketchup": "#9a382d",
        "khaki": "#a39264",
        "kiwi": "#7aab55",
        "koi": "#d15837",
        "kombu-green": "#3a4032",
        "kumquat": "#fbaa4c",
        "lagoon": "#4d9e9a",
        "lake-blue": "#008c96",
        "lambs-wool": "#e5d0b1",
        "langoustino": "#ca6c56",
        "languid-lavender": "#a2a1ba",
        "lantana": "#da7e7a",
        "lapis": "#008684",
        "lapis-blue": "#004b8d",
        "lark": "#b89b72",
        "larkspur": "#3c7d90",
        "latigo-bay": "#379190",
        "latte": "#c5a582",
        "laurel-green": "#adbba1",
        "laurel-oak": "#918c7e",
        "laurel-wreath": "#616f65",
        "lavender": "#afa4ce",
        "lavender-aura": "#9f99aa",
        "lavender-blue": "#c5c0d0",
        "lavender-crystal": "#936a98",
        "lavender-fog": "#d2c4d6",
        "lavender-frost": "#bdabbe",
        "lavender-gray": "#9890a2",
        "lavender-herb": "#b18eaa",
        "lavender-lustre": "#8c9cc1",
        "lavender-mist": "#ae90a7",
        "lavender-violet": "#767ba5",
        "lavendula": "#bca4cb",
        "lead": "#7a898f",
        "lead-gray": "#8a7963",
        "leaf-green": "#9faf6c",
        "leather-brown": "#97572b",
        "leek-green": "#b7b17a",
        "legion-blue": "#1f495b",
        "lemon": "#f3bf08",
        "lemon-chrome": "#ffc300",
        "lemon-curry": "#cda323",
        "lemon-drop": "#fdd878",
        "lemon-grass": "#dcd494",
        "lemon-icing": "#f6ebc8",
        "lemon-meringue": "#f6e199",
        "lemon-verbena": "#f3e779",
        "lemon-zest": "#f9d857",
        "lemonade": "#f0e79d",
        "lentil-sprout": "#aba44d",
        "leprechaun": "#378661",
        "lettuce-green": "#bed38e",
        "liberty": "#4d448a",
        "lichen": "#9bc2b1",
        "lichen-blue": "#5d89b3",
        "licorice": "#3a3536",
        "light-grass-green": "#7cb68e",
        "light-gray": "#dad8c9",
        "light-lilac": "#dec6d3",
        "light-mahogany": "#ad6d68",
        "light-taupe": "#b19d8d",
        "lightest-sky": "#e4eadf",
        "lilac-ash": "#d7cdcd",
        "lilac-breeze": "#b3a0c9",
        "lilac-chiffon": "#de9bc4",
        "lilac-gray": "#9896a4",
        "lilac-hint": "#d0d0da",
        "lilac-marble": "#c3babf",
        "lilac-rose": "#bd4275",
        "lilac-sachet": "#e9adca",
        "lilac-snow": "#e0c7d7",
        "lilas": "#b88995",
        "lily-green": "#c5cf98",
        "lily-pad": "#818f84",
        "lily-white": "#e2e2da",
        "lima-bean": "#e1d590",
        "lime-cream": "#d7e8bc",
        "lime-green": "#9fc131",
        "lime-popsicle": "#c0db3a",
        "lime-punch": "#c0d725",
        "lime-sherbet": "#cdd78a",
        "limeade": "#d3d95f",
        "limelight": "#f0e87d",
        "limestone": "#989a98",
        "limoges": "#243f6c",
        "limpet-shell": "#98ddde",
        "linden-green": "#c4bf71",
        "linen": "#edd2c0",
        "lint": "#b6ba99",
        "lion": "#a0714f",
        "lipstick-red": "#b31a38",
        "little-boy-blue": "#6ea2d5",
        "living-coral": "#ff6f61",
        "lizard": "#71643e",
        "lobster-bisque": "#dd9289",
        "loden-frost": "#788f74",
        "loden-green": "#6e7153",
        "loganberry": "#5a4769",
        "lollipop": "#cc1c3b",
        "london-fog": "#a29e92",
        "lotus": "#e2c1c0",
        "love-potion": "#c01352",
        "luminary-green": "#e3eaa5",
        "lunar-rock": "#c5c5c5",
        "lupine": "#be9cc1",
        "lush-meadow": "#006e51",
        "lychee": "#ba0b32",
        "lyons-blue": "#005871",
        "macadamia": "#e4cfb6",
        "macaroon": "#b38b71",
        "macaw-green": "#9bb53e",
        "madder-brown": "#6a3331",
        "magenta": "#d23c77",
        "magenta-haze": "#9d446e",
        "magenta-purple": "#6b264b",
        "magnet": "#4d4b4f",
        "mahogany": "#824d46",
        "mahogany-rose": "#c5a193",
        "maize": "#eec843",
        "majesty": "#593761",
        "majolica-blue": "#274357",
        "major-brown": "#5b5149",
        "malachite-green": "#709a89",
        "malaga": "#9f5069",
        "malibu-blue": "#008cc1",
        "mallard-blue": "#3a5c6e",
        "mallard-green": "#405e5c",
        "mandarin-orange": "#ec6a37",
        "mandarin-red": "#e74a33",
        "mango": "#b75e41",
        "mango-mojito": "#d69c2f",
        "maple-sugar": "#c9a38d",
        "margarita": "#b5c38e",
        "marigold": "#fadc53",
        "marina": "#4f84c4",
        "marine-blue": "#76afb6",
        "marine-green": "#40a48e",
        "maritime-blue": "#27293d",
        "marlin": "#515b87",
        "marmalade": "#c16512",
        "maroon": "#834655",
        "marron": "#6e4c4b",
        "mars-red": "#bc2731",
        "marsala": "#964f4c",
        "marshmallow": "#f0eee4",
        "martini-olive": "#716a4d",
        "marys-rose": "#f7d1d4",
        "marzipan": "#d8c09d",
        "maui-blue": "#52a2b4",
        "mauve-chalk": "#e5d0cf",
        "mauve-mist": "#c49bd4",
        "mauve-morn": "#ecd6d6",
        "mauve-orchid": "#b58299",
        "mauve-shadows": "#b598a3",
        "mauve-wine": "#5b3644",
        "mauveglow": "#d18489",
        "mauvewood": "#a75d67",
        "mayfly": "#65663f",
        "mazarine-blue": "#273c76",
        "meadow": "#8bba94",
        "meadow-green": "#739957",
        "meadow-mauve": "#a9568c",
        "meadow-mist": "#d3dec4",
        "meadow-violet": "#764f82",
        "meadowbrook": "#60a0a3",
        "meadowlark": "#ead94e",
        "mecca-orange": "#bd5745",
        "medal-bronze": "#977547",
        "medieval-blue": "#29304e",
        "mediterranea": "#32575d",
        "mediterranean-blue": "#1478a7",
        "medium-green": "#3c824e",
        "meerkat": "#a46f44",
        "mellow-buff": "#d8b998",
        "mellow-green": "#d5d593",
        "mellow-mauve": "#996378",
        "mellow-rose": "#d9a6a1",
        "mellow-yellow": "#f0dd9d",
        "melon": "#fe8863",
        "mercury": "#bac2ba",
        "merlot": "#72262c",
        "mermaid": "#817a65",
        "mesa-rose": "#a66e7a",
        "metal": "#babfbc",
        "meteorite": "#2b2929",
        "methyl-blue": "#0074a8",
        "micro-chip": "#babcc0",
        "midnight": "#325b74",
        "midnight-navy": "#34414e",
        "military-olive": "#63563b",
        "milky-blue": "#72a8ba",
        "milky-green": "#cfdbd1",
        "mimosa": "#f0c05a",
        "mineral-blue": "#6d9192",
        "mineral-gray": "#b2b6ac",
        "mineral-green": "#7a9b78",
        "mineral-red": "#b35457",
        "mineral-yellow": "#d39c43",
        "ming": "#7cb08a",
        "ming-green": "#3aa278",
        "minimal-gray": "#948d99",
        "minion-yellow": "#fed55d",
        "mink": "#734b42",
        "mint": "#00a170",
        "mint-green": "#487d49",
        "mint-leaf": "#00b694",
        "mirage-gray": "#abafae",
        "mist-green": "#aacebc",
        "misted-yellow": "#dab965",
        "mistletoe": "#8aa282",
        "misty-blue": "#bfcdcc",
        "misty-jade": "#bcd9c8",
        "misty-lilac": "#bcb4c4",
        "misty-rose": "#caa39a",
        "mocha-bisque": "#8c543a",
        "mocha-mousse": "#a47864",
        "mock-orange": "#ffa368",
        "mojave-desert": "#c7b595",
        "mole": "#392d2b",
        "molten-lava": "#b5332e",
        "monaco-blue": "#274374",
        "monks-robe": "#704822",
        "montana-grape": "#6c5971",
        "monument": "#84898c",
        "mood-indigo": "#353a4c",
        "moon-mist": "#80817d",
        "moon-rock": "#958b84",
        "moonbeam": "#cdc6bd",
        "moonless-night": "#2f2d30",
        "moonlight": "#c5b1a0",
        "moonlight-blue": "#506886",
        "moonlight-jade": "#c7e5df",
        "moonlit-ocean": "#293b4d",
        "moonlite-mauve": "#d28fb0",
        "moonscape": "#725f69",
        "moonstruck": "#c2beb6",
        "morel": "#685c53",
        "morganite": "#dfcdc6",
        "morning-glory": "#ee819f",
        "morning-mist": "#cfdfdb",
        "moroccan-blue": "#0f4e67",
        "mosaic-blue": "#00758f",
        "moss": "#a09d59",
        "moss-gray": "#afab97",
        "mosstone": "#858961",
        "moth": "#d2cbaf",
        "mother-of-pearl": "#e9d4c3",
        "mountain-view": "#2e3d30",
        "mourning-dove": "#94908b",
        "mulberry": "#a76c97",
        "mulberry-purple": "#493c62",
        "mulch": "#433937",
        "mulled-grape": "#675a74",
        "murmur": "#d2d8d2",
        "mushroom": "#bdaca3",
        "muskmelon": "#ec935e",
        "mustang": "#684b40",
        "mustard-gold": "#b08e51",
        "muted-clay": "#d29380",
        "muted-lime": "#d1c87c",
        "mykonos-blue": "#005780",
        "myrtle": "#4f6b58",
        "mysterioso": "#46394b",
        "mystic-blue": "#e1e3de",
        "mystical": "#5f4e72",
        "narcissus": "#c39449",
        "nasturtium": "#fe6347",
        "natural": "#aa907d",
        "nautical-blue": "#1a5091",
        "navagio-bay": "#3183a0",
        "navajo": "#efdcc3",
        "navigate": "#008583",
        "navy-blazer": "#282d3c",
        "navy-blue": "#403f6f",
        "navy-cosmos": "#503b53",
        "navy-peony": "#223a5e",
        "nebulas-blue": "#2d62a3",
        "nectarine": "#ff8656",
        "neptune-green": "#7fbb9e",
        "neutral-gray": "#8e918f",
        "new-wheat": "#d7b57f",
        "niagara": "#5487a4",
        "night-sky": "#2a2a35",
        "nightshade": "#433748",
        "nightshadow-blue": "#4e5368",
        "nile": "#b4bb85",
        "nile-blue": "#76a7ab",
        "nile-green": "#a7c796",
        "nimbus-cloud": "#d5d5d8",
        "nine-iron": "#46434a",
        "nirvana": "#a2919b",
        "nocturne": "#7a4b56",
        "nomad": "#b49f89",
        "norse-blue": "#4ca5c7",
        "north-atlantic": "#536d70",
        "north-sea": "#316c6b",
        "nostalgia-rose": "#a4777e",
        "nougat": "#b69885",
        "novelle-peach": "#e7cfbd",
        "nude": "#f2d3bc",
        "nugget": "#cf8848",
        "nugget-gold": "#c89720",
        "nutmeg": "#7e5c54",
        "nutria": "#75663e",
        "oak-buff": "#cf9c63",
        "oasis": "#a3a04e",
        "oatmeal": "#cbc3b4",
        "obsidian": "#3a363b",
        "ocean-depths": "#006175",
        "ocean-wave": "#8ec5b6",
        "ochre": "#d6af66",
        "odyssey-gray": "#434452",
        "oil-blue": "#658c88",
        "oil-green": "#80856d",
        "oil-yellow": "#c4a647",
        "old-gold": "#eca825",
        "old-rose": "#b47b77",
        "olive-branch": "#646a45",
        "olive-drab": "#756d47",
        "olive-gray": "#a6997a",
        "olive-night": "#535040",
        "olive-oil": "#a98b2d",
        "olivenite": "#c1a65c",
        "olivine": "#666b54",
        "olympian-blue": "#1a4c8b",
        "ombre-blue": "#434854",
        "omphalodes": "#b5cedf",
        "online-lime": "#44883c",
        "opal": "#77cfb7",
        "opal-blue": "#c3ddd6",
        "opal-gray": "#a49e9e",
        "opaline-green": "#a3c57d",
        "opera-mauve": "#ca80b1",
        "orange-chiffon": "#f9aa7d",
        "orange-com": "#da321c",
        "orange-ochre": "#dc793a",
        "orange-peel": "#fa7a35",
        "orange-pepper": "#df7500",
        "orange-popsicle": "#ff7913",
        "orange-rust": "#c25a3c",
        "orange-tiger": "#f96714",
        "orangeade": "#e2552c",
        "orchid": "#d198c5",
        "orchid-bloom": "#c5aecf",
        "orchid-bouquet": "#d1acce",
        "orchid-haze": "#b0879b",
        "orchid-hush": "#cec3d2",
        "orchid-ice": "#e0d0db",
        "orchid-mist": "#917798",
        "orchid-petal": "#bfb4cb",
        "orchid-pink": "#f3bbca",
        "orchid-smoke": "#d294aa",
        "orchid-tint": "#dbd2db",
        "orient-blue": "#47457a",
        "orion-blue": "#3e4f5c",
        "otter": "#7f674f",
        "outer-space": "#2f3441",
        "overcast": "#c3bdab",
        "oxblood-red": "#70393f",
        "oxford-tan": "#b8a99a",
        "oyster-gray": "#cbc1ae",
        "oyster-mushroom": "#c3c6c8",
        "oyster-white": "#d2caaf",
        "pacific": "#1f595c",
        "pacific-coast": "#5480ac",
        "pagoda-blue": "#1a7f8e",
        "paisley-purple": "#8b79b1",
        "palace-blue": "#346cb0",
        "pale-aqua": "#c1ccc2",
        "pale-banana": "#fae199",
        "pale-blue": "#c4d6d3",
        "pale-blush": "#e4bfb3",
        "pale-dogwood": "#edcdc2",
        "pale-gold": "#bd9865",
        "pale-green": "#cbce91",
        "pale-iris": "#8895c5",
        "pale-khaki": "#bfaf92",
        "pale-lilac": "#e1c6cc",
        "pale-lime-yellow": "#dfe69f",
        "pale-marigold": "#ffc66e",
        "pale-mauve": "#c6a4a4",
        "pale-olive-green": "#b5ad88",
        "pale-peach": "#fed1bd",
        "palm": "#afaf5e",
        "paloma": "#9f9c99",
        "pampas": "#cfbb7b",
        "pansy": "#653d7c",
        "papaya": "#fea166",
        "papaya-punch": "#fca289",
        "paprika": "#ce4d42",
        "papyrus": "#f5edd6",
        "parachute-purple": "#392852",
        "paradise-green": "#b2e79f",
        "paradise-pink": "#e4445e",
        "parakeet": "#008c69",
        "parasailing": "#00736c",
        "parchment": "#dfd1be",
        "parfait-pink": "#e9c3cf",
        "parisian-blue": "#4f7ca4",
        "parisian-night": "#323441",
        "parrot-green": "#8db051",
        "parsnip": "#d6c69a",
        "partridge": "#725440",
        "passion-flower": "#6d5698",
        "pastel-blue": "#bcd3d5",
        "pastel-green": "#b4d3b2",
        "pastel-lavender": "#d8a1c4",
        "pastel-lilac": "#bdb0d0",
        "pastel-parchment": "#e5d9d3",
        "pastel-rose-tan": "#e9d1bf",
        "pastel-turquoise": "#99c5c4",
        "pastel-yellow": "#f2e6b1",
        "pastry-shell": "#bd8c66",
        "patina-green": "#b9eab3",
        "patrician-purple": "#6c4e79",
        "patriot-blue": "#363756",
        "pavement": "#524d50",
        "peach": "#f2a987",
        "peach-amber": "#fb9f93",
        "peach-beige": "#d3a297",
        "peach-bloom": "#d99b7c",
        "peach-blossom": "#de8286",
        "peach-blush": "#e4ccc6",
        "peach-bud": "#fdb2ab",
        "peach-caramel": "#c5733d",
        "peach-cobbler": "#ffb181",
        "peach-dust": "#f0d8cc",
        "peach-echo": "#f7786b",
        "peach-fuzz": "#ffbe98",
        "peach-melba": "#fbbdaf",
        "peach-nectar": "#ffb59b",
        "peach-nougat": "#e6af91",
        "peach-parfait": "#f8bfa8",
        "peach-pearl": "#ffb2a5",
        "peach-pink": "#fa9a85",
        "peach-puree": "#efcfba",
        "peach-quartz": "#f5b895",
        "peach-whip": "#dbbeb7",
        "peaches-n-cream": "#f4a6a3",
        "peachskin": "#dfb8b6",
        "peachy-keen": "#e2bdb3",
        "peacoat": "#2b2e43",
        "peacock-blue": "#00a0b0",
        "peacock-green": "#00a78b",
        "peapod": "#82b185",
        "pear-sorbet": "#f3eac3",
        "pearl": "#f9dbd8",
        "pearl-blue": "#b0b7be",
        "pearl-blush": "#f4cec5",
        "pearled-ivory": "#f0dfcc",
        "peat": "#3b3a36",
        "pebble": "#cab698",
        "pecan-brown": "#a36e51",
        "pelican": "#c1bcac",
        "peony": "#ed9ca8",
        "pepper-green": "#007d60",
        "peppercorn": "#6c5656",
        "peppermint": "#699e6d",
        "peridot": "#819548",
        "periscope": "#46444c",
        "persian-jewel": "#6e81be",
        "persian-red": "#a21441",
        "persian-violet": "#8c8eb2",
        "persimmon": "#f67866",
        "persimmon-orange": "#f47327",
        "pesto": "#595f34",
        "petal-pink": "#f2e2e0",
        "petit-four": "#87c2d4",
        "petrified-oak": "#8d7960",
        "petunia": "#4f3466",
        "pewter": "#666564",
        "peyote": "#c5bbae",
        "phantom": "#39373b",
        "phantom-green": "#dce4d7",
        "pheasant": "#c68463",
        "phlox": "#692d5d",
        "phlox-pink": "#ce5e9a",
        "picante": "#8d3f2d",
        "picasso-lily": "#634878",
        "pickled-beet": "#4d233d",
        "pigeon": "#a9afaa",
        "pine-bark": "#827064",
        "pine-green": "#3a795e",
        "pine-grove": "#213631",
        "pineapple-slice": "#e7d391",
        "pinecone": "#61473b",
        "pineneedle": "#334d41",
        "pink-carnation": "#ed7a9e",
        "pink-dogwood": "#f7d1d1",
        "pink-flambe": "#d3507a",
        "pink-icing": "#eea0a6",
        "pink-lady": "#efc1d6",
        "pink-lavender": "#d9afca",
        "pink-lemonade": "#ee6d8a",
        "pink-mist": "#e6bccd",
        "pink-nectar": "#d8aab7",
        "pink-peacock": "#c62168",
        "pink-salt": "#f7cdc7",
        "pink-sand": "#dfb19b",
        "pink-tint": "#dbcbbd",
        "pink-yarrow": "#ce3175",
        "piquant-green": "#769358",
        "pirate-black": "#363838",
        "pistachio-green": "#a9d39e",
        "pistachio-shell": "#d7cfbb",
        "placid-blue": "#8cadd3",
        "plantation": "#7a6332",
        "plaza-taupe": "#aea393",
        "plein-air": "#bfcad6",
        "plum": "#5a315d",
        "plum-caspia": "#61224a",
        "plum-jam": "#624076",
        "plum-kitten": "#625b5c",
        "plum-perfect": "#473442",
        "plum-purple": "#51304e",
        "plum-truffle": "#675657",
        "plum-wine": "#674550",
        "plume": "#a5cfd5",
        "poinciana": "#ca3422",
        "poinsettia": "#cb3441",
        "poison-green": "#4db560",
        "polignac": "#c28799",
        "pomegranate": "#6c2831",
        "pompeian-red": "#a4292e",
        "ponderosa-pine": "#203b3d",
        "pool-blue": "#67bcb3",
        "pool-green": "#00af9d",
        "popcorn": "#f8de8d",
        "poppy-red": "#dc343b",
        "porcelain": "#5d9ca4",
        "porcelain-blue": "#95c0cb",
        "porcelain-green": "#108780",
        "porcelain-rose": "#ea6b6a",
        "porcini": "#cca580",
        "porpoise": "#a7a19e",
        "port": "#663336",
        "port-royale": "#502b33",
        "portabella": "#937b6a",
        "poseidon": "#123955",
        "posy-green": "#325b51",
        "potent-purple": "#462639",
        "potpourri": "#e7c9ca",
        "potters-clay": "#9e4624",
        "potting-soil": "#54392d",
        "powder-blue": "#96b3d2",
        "powder-pink": "#ecb2b3",
        "powder-puff": "#f3e0d6",
        "prairie-sand": "#b59a6a",
        "prairie-sunset": "#ffbb9e",
        "praline": "#ad8b75",
        "primrose-pink": "#eed4d9",
        "primrose-yellow": "#f6d155",
        "princess-blue": "#00539c",
        "prism-pink": "#f0a1bf",
        "prism-violet": "#53357d",
        "pristine": "#f2e8da",
        "provence": "#658dc6",
        "provincial-blue": "#5c798e",
        "prune": "#603749",
        "prune-purple": "#5c3a4d",
        "puce": "#503938",
        "puffins-bill": "#e95c20",
        "pumice-stone": "#cac2b9",
        "pumpkin": "#f5a26f",
        "pumpkin-spice": "#a05c17",
        "pure-cashmere": "#ada396",
        "pureed-pumpkin": "#c34121",
        "puritan-gray": "#a8b0ae",
        "purple-ash": "#8f8395",
        "purple-corallite": "#5a4e8f",
        "purple-dove": "#98878c",
        "purple-gumdrop": "#7a596f",
        "purple-haze": "#807396",
        "purple-heart": "#745587",
        "purple-heather": "#bab8d3",
        "purple-impression": "#858fb1",
        "purple-magic": "#663271",
        "purple-opulence": "#60569a",
        "purple-orchid": "#ad4d8c",
        "purple-passion": "#683d62",
        "purple-pennant": "#432c47",
        "purple-plumeria": "#473854",
        "purple-potion": "#692746",
        "purple-reign": "#56456b",
        "purple-rose": "#b09fca",
        "purple-sage": "#75697e",
        "purple-sapphire": "#6f4685",
        "purple-velvet": "#41354d",
        "purple-wine": "#8c3573",
        "pussywillow-gray": "#aeaca1",
        "putty": "#d4cab0",
        "quail": "#98868c",
        "quarry": "#98a0a5",
        "quartz-pink": "#efa6aa",
        "quetzal-green": "#006865",
        "quicksilver": "#7e7d88",
        "quiet-gray": "#b9babd",
        "quiet-green": "#9ebc97",
        "quiet-harbor": "#5a789a",
        "quiet-shade": "#66676d",
        "rabbit": "#5f575c",
        "racing-red": "#bd162c",
        "radiant-orchid": "#ad5e99",
        "radiant-yellow": "#fc9e21",
        "raffia": "#dac483",
        "rain-drum": "#5f4c40",
        "rain-forest": "#15463e",
        "raindrops": "#b1aab3",
        "rainy-day": "#cfc8bd",
        "raisin": "#524144",
        "rapture-rose": "#d16277",
        "raspberry": "#d32e5e",
        "raspberry-radiance": "#802a50",
        "raspberry-rose": "#cc4385",
        "raspberry-sorbet": "#d2386c",
        "raspberry-wine": "#b63753",
        "rattan": "#d1b272",
        "raven": "#413e3d",
        "raw-sienna": "#b9714f",
        "raw-umber": "#92705f",
        "rawhide": "#865e49",
        "real-teal": "#405d73",
        "red-bud": "#962d49",
        "red-clay": "#c2452d",
        "red-dahlia": "#7d2027",
        "red-mahogany": "#60373d",
        "red-ochre": "#913832",
        "red-orange": "#f05627",
        "red-pear": "#7b3539",
        "red-plum": "#7c2946",
        "red-violet": "#a35776",
        "redwood": "#a6594c",
        "reed": "#c3d3a8",
        "reed-yellow": "#dcc99e",
        "reef-waters": "#6f9fa9",
        "reflecting-pond": "#203e4a",
        "regal-orchid": "#a98baf",
        "regatta": "#487ab7",
        "renaissance-rose": "#865560",
        "reseda": "#a1ad92",
        "rhapsody": "#9f86aa",
        "rhododendron": "#722b3f",
        "rhubarb": "#77202f",
        "ribbon-red": "#b92636",
        "rich-gold": "#c8b273",
        "rifle-green": "#414832",
        "rio-red": "#8a2232",
        "river-blue": "#38afcd",
        "riverside": "#4c6a92",
        "riviera": "#5879a2",
        "roan-rouge": "#885157",
        "roasted-cashew": "#918579",
        "roasted-pecan": "#93592b",
        "rock-ridge": "#918c86",
        "rocky-road": "#5a3e36",
        "rococco-red": "#bb363f",
        "roebuck": "#b09080",
        "rooibos-tea": "#a23c26",
        "root-beer": "#714a41",
        "rose-brown": "#80565b",
        "rose-cloud": "#dbb0a2",
        "rose-dawn": "#c2877b",
        "rose-dust": "#cdb2a5",
        "rose-of-sharon": "#dc5b62",
        "rose-quartz": "#f7cac9",
        "rose-red": "#c92351",
        "rose-shadow": "#f9c2cd",
        "rose-smoke": "#d3b4ad",
        "rose-tan": "#d19c97",
        "rose-taupe": "#806062",
        "rose-violet": "#c0428a",
        "rose-water": "#f8e0e7",
        "rose-wine": "#a4596d",
        "rosebloom": "#e290b2",
        "rosebud": "#b65f9a",
        "rosette": "#ce8e8b",
        "rosewater": "#f6dbd8",
        "rosewood": "#813639",
        "rosin": "#36362d",
        "rouge-red": "#e24666",
        "royal-blue": "#3d428b",
        "royal-lilac": "#774d8e",
        "royal-purple": "#603f83",
        "rubber": "#815b37",
        "ruby-wine": "#77333b",
        "rugby-tan": "#c2a594",
        "rum-raisin": "#583432",
        "rumba-red": "#7c2439",
        "russet": "#8f5f50",
        "russet-brown": "#743332",
        "russet-orange": "#e47127",
        "rust": "#b55a30",
        "rustic-brown": "#855141",
        "rutabaga": "#ecddbe",
        "sable": "#6e403c",
        "sachet-pink": "#f18aad",
        "safari": "#baaa91",
        "saffron": "#ffa500",
        "sage": "#91946e",
        "sage-green": "#b2ac88",
        "sagebrush-green": "#567572",
        "sahara-sun": "#dfc08a",
        "sailor-blue": "#0e3a53",
        "salmon": "#faaa94",
        "salmon-buff": "#feaa7b",
        "salmon-rose": "#ff8d94",
        "salsa": "#aa182b",
        "salute": "#282b34",
        "samba": "#a2242f",
        "samoan-sun": "#fbc85f",
        "sand": "#cca67f",
        "sand-dollar": "#decdbe",
        "sand-verbena": "#9f90c1",
        "sandshell": "#d8ccbb",
        "sandstone": "#c48a69",
        "sandstorm": "#bd8b69",
        "sangria": "#982551",
        "sap-green": "#afcb80",
        "sargasso-sea": "#35435a",
        "sassafras": "#54353b",
        "satellite": "#9f8d89",
        "sauterne": "#c5a253",
        "saxony-blue": "#1f6680",
        "scallop-shell": "#fbd8c9",
        "scarab": "#23312d",
        "scarlet": "#bc2b3d",
        "scarlet-ibis": "#f45520",
        "scarlet-sage": "#9d202f",
        "scooter": "#941e32",
        "scuba-blue": "#00abc0",
        "sea-angel": "#98bfca",
        "sea-blue": "#549f98",
        "sea-foam": "#b7c2b2",
        "sea-fog": "#a5929d",
        "sea-green": "#149c88",
        "sea-mist": "#d8c9a3",
        "sea-moss": "#254445",
        "sea-pine": "#4c6969",
        "sea-pink": "#de98ab",
        "sea-salt": "#f1e6de",
        "sea-spray": "#717e6f",
        "sea-turtle": "#5e5749",
        "seacrest": "#bfd1b3",
        "seafoam-green": "#cbd5b1",
        "seagrass": "#959889",
        "seal-brown": "#493b39",
        "seaport": "#005e7d",
        "seashell-pink": "#f7c8c2",
        "sedona-sage": "#686d6c",
        "seedling": "#c0cba1",
        "seedpearl": "#e6dac4",
        "semolina": "#ceb899",
        "seneca-rock": "#9a927f",
        "sepia": "#6b543e",
        "sepia-rose": "#d4bab6",
        "sepia-tint": "#897560",
        "sequoia": "#804839",
        "serenity": "#91a8d0",
        "sesame": "#baa38b",
        "shaded-spruce": "#00585e",
        "shadow": "#888d82",
        "shadow-gray": "#bba5a0",
        "shadow-green": "#cfc486",
        "shadow-lime": "#cfe09d",
        "shadow-purple": "#4e334e",
        "shady-glade": "#006e5b",
        "shale": "#4a3f41",
        "shale-green": "#739072",
        "shamrock": "#6fa26b",
        "shark": "#6d636b",
        "sharkskin": "#838487",
        "sharp-green": "#c6ec7a",
        "shaved-chocolate": "#543b35",
        "sheepskin": "#dab58f",
        "sheer-lilac": "#b793c0",
        "sheer-pink": "#f6e5db",
        "shell": "#e1cfc6",
        "shell-coral": "#ea9575",
        "shell-pink": "#f88180",
        "shifting-sand": "#d8c0ad",
        "shitake": "#736253",
        "shocking-pink": "#de5b8c",
        "shopping-bag": "#5a4743",
        "shrimp": "#e29a86",
        "shrinking-violet": "#f4e1e6",
        "sierra": "#985c41",
        "silt-green": "#a9bdb1",
        "silver": "#a2a2a1",
        "silver-birch": "#d2cfc4",
        "silver-blue": "#8a9a9a",
        "silver-bullet": "#81839a",
        "silver-cloud": "#beb7b0",
        "silver-fern": "#bbaa7e",
        "silver-filigree": "#7f7c81",
        "silver-gray": "#c1b7b0",
        "silver-green": "#d7d7c7",
        "silver-lake-blue": "#618bb9",
        "silver-lining": "#bdb6ab",
        "silver-mink": "#9f8d7c",
        "silver-peony": "#e7cfc7",
        "silver-pine": "#4e6866",
        "silver-pink": "#dcb1af",
        "silver-sage": "#938b78",
        "silver-sconce": "#a19fa5",
        "simply-green": "#009b75",
        "simply-taupe": "#ad9f93",
        "sirocco": "#c39d88",
        "ski-patrol": "#bb1237",
        "skipper-blue": "#484a72",
        "sky-blue": "#8abad3",
        "sky-captain": "#262934",
        "sky-gray": "#bcc8c6",
        "skydiver": "#00589b",
        "skylight": "#c8e0e0",
        "skyway": "#adbed3",
        "slate": "#8c9fa1",
        "slate-black": "#4b3d33",
        "slate-gray": "#8a9691",
        "slate-green": "#a0987c",
        "slate-rose": "#b45865",
        "sleet": "#92949b",
        "smoke": "#bfc8c3",
        "smoke-blue": "#6d8994",
        "smoke-gray": "#cebaa8",
        "smoke-green": "#a8bba2",
        "smoke-pine": "#3e6257",
        "smoked-paprika": "#6e362c",
        "smoked-pearl": "#656466",
        "smoky-grape": "#b88aac",
        "snapdragon": "#fed777",
        "snorkel-blue": "#034f84",
        "snow-white": "#f2f0eb",
        "sodalite-blue": "#253668",
        "soft-pink": "#f2d8cd",
        "solar-power": "#f4bf3a",
        "soothing-sea": "#c3e9e4",
        "southern-moss": "#bca66a",
        "soybean": "#d2c29d",
        "spa-blue": "#d3dedf",
        "spanish-villa": "#dfbaa9",
        "sparkling-grape": "#773376",
        "sparrow": "#69595c",
        "spearmint": "#64bfa4",
        "spectra-green": "#009b8c",
        "spectra-yellow": "#f7b718",
        "spectrum-blue": "#3d3c7c",
        "sphinx": "#ab9895",
        "spice-route": "#b95b3f",
        "spiced-apple": "#783937",
        "spiced-coral": "#d75c5d",
        "spiced-plum": "#6d4773",
        "spicy-mustard": "#d8ae47",
        "spicy-orange": "#d73c26",
        "spinach-green": "#909b4c",
        "split-pea": "#9c9a40",
        "sponge": "#a49775",
        "spray": "#bed3bb",
        "spray-green": "#aea692",
        "spring-bouquet": "#6dce87",
        "spring-bud": "#6bcd9c",
        "spring-crocus": "#ba69a1",
        "sprout-green": "#cbd7d2",
        "spruce-yellow": "#be8a4a",
        "sprucestone": "#9fc09c",
        "star-sapphire": "#386192",
        "star-white": "#efefe8",
        "starfish": "#b09a77",
        "stargazer": "#39505c",
        "starlight-blue": "#b5ced4",
        "steel-gray": "#726f70",
        "steeple-gray": "#827e7c",
        "stellar": "#46647e",
        "sterling-blue": "#a2b9c2",
        "stillwater": "#70a4b0",
        "stone-blue": "#829ca5",
        "stone-gray": "#685e4f",
        "stone-green": "#658e67",
        "stonewash": "#74809a",
        "storm": "#035453",
        "storm-blue": "#47788a",
        "storm-front": "#787376",
        "storm-gray": "#b5bab6",
        "stormy-sea": "#6e8082",
        "stormy-weather": "#58646d",
        "stratosphere": "#9ec1cc",
        "straw": "#e0c992",
        "strawberry-cream": "#f4c3c4",
        "strawberry-ice": "#e78b90",
        "strawberry-pink": "#f57f8e",
        "stretch-limo": "#2b2c30",
        "striking-purple": "#944e87",
        "string": "#aa9f96",
        "strong-blue": "#1f5da0",
        "stucco": "#a58d7f",
        "subtle-green": "#b5cbbb",
        "sudan-brown": "#ac6b29",
        "sugar-almond": "#935529",
        "sugar-coral": "#f56c73",
        "sugar-swizzle": "#f3eee7",
        "sulphur": "#ddb614",
        "sulphur-spring": "#d5d717",
        "summer-fig": "#be4b3b",
        "summer-green": "#7ed37f",
        "summer-melon": "#ead3ae",
        "summer-shower": "#e5ebe3",
        "sun-baked": "#d27f63",
        "sun-dried-tomato": "#752329",
        "sun-kiss": "#ebd1bb",
        "sun-orange": "#f48037",
        "sunburn": "#b37256",
        "sunburst": "#f6c289",
        "sundress": "#ebcf89",
        "sunflower": "#d39237",
        "sunkist-coral": "#ea6676",
        "sunlight": "#edd59e",
        "sunlit-allium": "#9787bb",
        "sunny-lime": "#dfef87",
        "sunset-gold": "#f7c46c",
        "sunset-purple": "#6f456e",
        "sunshine": "#fade85",
        "super-lemon": "#e4bf45",
        "super-pink": "#ce6ba4",
        "surf-spray": "#b4c8c2",
        "surf-the-web": "#203c7f",
        "swamp": "#a8b197",
        "swedish-blue": "#007eb1",
        "sweet-cream": "#f0ead6",
        "sweet-grape": "#4b3b4f",
        "sweet-lavender": "#9a9bc1",
        "sweet-lilac": "#e8b5ce",
        "sweet-pea": "#a3a969",
        "sycamore": "#35463d",
        "sylvan-green": "#e7eacb",
        "syrah": "#6a282c",
        "taffy": "#c39b6a",
        "tahitian-tide": "#006b7e",
        "tan": "#b69574",
        "tanager-turquoise": "#91dce8",
        "tandori-spice": "#9f4440",
        "tangelo": "#fe7e03",
        "tangerine": "#f88f58",
        "tangerine-tango": "#dd4124",
        "tango-red": "#ac0e2e",
        "tannin": "#a68a6d",
        "taos-taupe": "#bfa77f",
        "tap-shoe": "#2a2b2d",
        "tapenade": "#805d24",
        "tapestry": "#436573",
        "tapioca": "#dccdbc",
        "tarmac": "#5a5348",
        "tarragon": "#a4ae77",
        "taupe-gray": "#8e7c71",
        "tawny-birch": "#ae856c",
        "tawny-brown": "#ab856f",
        "tawny-olive": "#c4962c",
        "tawny-orange": "#d37f6f",
        "tawny-port": "#5c2c35",
        "tea": "#999b85",
        "tea-rose": "#dc7178",
        "teaberry": "#dc3855",
        "teak": "#655341",
        "teal": "#478589",
        "teal-blue": "#007f7c",
        "teal-green": "#006361",
        "tempest": "#79839b",
        "tender-greens": "#c5cfb6",
        "tender-peach": "#f8d5b8",
        "tender-shoots": "#b5cc39",
        "tender-yellow": "#ededb7",
        "tendril": "#89a06b",
        "terra-cotta": "#d38377",
        "thai-curry": "#ab6819",
        "thistle": "#b9b3c5",
        "thistle-down": "#9499bb",
        "thrush": "#936b4f",
        "thyme": "#50574c",
        "tibetan-red": "#782a39",
        "tibetan-stone": "#82c2c7",
        "tidal-foam": "#bfb9a3",
        "tidepool": "#0a6f69",
        "tigerlily": "#e2583e",
        "tigers-eye": "#977c61",
        "tile-blue": "#008491",
        "tillandsia-purple": "#563474",
        "timber-wolf": "#8d8070",
        "tinsel": "#c3964d",
        "tiramisu": "#634235",
        "titanium": "#807d7f",
        "toadstool": "#988088",
        "toast": "#ca9978",
        "toasted-almond": "#d2b49c",
        "toasted-coconut": "#8b6a4f",
        "toasted-nut": "#c08768",
        "tobacco-brown": "#9a7352",
        "toffee": "#755139",
        "tofu": "#e8e3d9",
        "tomato": "#ce2939",
        "tomato-cream": "#c57644",
        "tomato-puree": "#c53346",
        "topaz": "#d08344",
        "toreador": "#b61032",
        "tornado": "#5e5b60",
        "tortoise-shell": "#754734",
        "total-eclipse": "#2c313d",
        "tourmaline": "#86a1a9",
        "tradewinds": "#7f8793",
        "transparent-yellow": "#f4ecc2",
        "travertine": "#ae997d",
        "tree-house": "#988c75",
        "treetop": "#476a30",
        "trekking-green": "#355048",
        "trellis": "#6a8988",
        "trooper": "#697a7e",
        "tropical-green": "#008786",
        "tropical-peach": "#ffc4b2",
        "true-blue": "#1e4477",
        "true-navy": "#3f5277",
        "true-red": "#bf1932",
        "tuffet": "#a59788",
        "tulipwood": "#805466",
        "turbulence": "#4e545b",
        "turf-green": "#6f8c69",
        "turkish-coffee": "#483f39",
        "turkish-sea": "#195190",
        "turkish-tile": "#00698b",
        "turmeric": "#fe840e",
        "turquoise": "#45b5aa",
        "turtle-green": "#81894e",
        "turtledove": "#ded7c8",
        "tuscany": "#be9785",
        "twilight-blue": "#313d64",
        "twilight-mauve": "#8b6f70",
        "twilight-purple": "#66648b",
        "twill": "#a79b82",
        "twist-of-lime": "#4e632c",
        "ultra-violet": "#5f4b8b",
        "ultramarine": "#5b7ebd",
        "ultramarine-green": "#006b54",
        "umber": "#944a1f",
        "urban-chic": "#464e4d",
        "valerian": "#9f7a93",
        "valiant-poppy": "#bc322c",
        "vallarta-blue": "#30658e",
        "vanilla": "#f4e1c1",
        "vanilla-cream": "#f4d8c6",
        "vanilla-custard": "#f3e0be",
        "vanilla-ice": "#f0eada",
        "vapor-blue": "#bebdbd",
        "vaporous-gray": "#dfddd7",
        "veiled-rose": "#f8cdc9",
        "velvet-morning": "#60688d",
        "verdant-green": "#12674a",
        "vermillion-orange": "#f9633b",
        "veronica": "#6d6695",
        "very-berry": "#b73275",
        "very-grape": "#927288",
        "vetiver": "#807d6f",
        "vibrant-green": "#55a860",
        "vibrant-orange": "#ff7420",
        "vibrant-yellow": "#ffda29",
        "victoria-blue": "#08589d",
        "vineyard-green": "#5f7355",
        "vineyard-wine": "#58363d",
        "vintage-indigo": "#4a556b",
        "vintage-khaki": "#9a9186",
        "vintage-violet": "#634f62",
        "viola": "#a692ba",
        "violet": "#c17fb5",
        "violet-ice": "#c2acb1",
        "violet-indigo": "#3e285c",
        "violet-quartz": "#8b4963",
        "violet-storm": "#5c619d",
        "violet-tulip": "#9e91c3",
        "violet-tulle": "#c193c0",
        "viridian-green": "#009499",
        "viridis": "#00846b",
        "virtual-pink": "#c6174e",
        "vista-blue": "#81a0d4",
        "vivacious": "#a32857",
        "vivid-blue": "#0088b0",
        "vivid-green": "#009e82",
        "vivid-viola": "#993c7c",
        "volcanic-glass": "#615c60",
        "vulcan": "#2d3036",
        "walnut": "#776a5f",
        "wan-blue": "#cbdcdf",
        "warm-apricot": "#ffb865",
        "warm-olive": "#c7b63c",
        "warm-sand": "#c5ae91",
        "warm-taupe": "#af9483",
        "wasabi": "#73a89e",
        "water-lily": "#dde3d5",
        "watercress": "#748c69",
        "waterfall": "#3ab0a2",
        "wax-yellow": "#ede9ad",
        "wedgewood": "#6479b3",
        "weeping-willow": "#b3b17b",
        "wet-weather": "#929090",
        "wheat": "#dec5a5",
        "whisper-green": "#e0e6d7",
        "whisper-pink": "#dacbbe",
        "whisper-white": "#ede6db",
        "whispering-blue": "#c9dcdc",
        "white-alyssum": "#efebe7",
        "white-asparagus": "#e1dbc8",
        "white-jade": "#d4dbb2",
        "white-pepper": "#b6a893",
        "white-sand": "#dbd5d1",
        "white-smoke": "#eddcc9",
        "white-swan": "#e4d7c5",
        "whitecap-gray": "#e0d5c6",
        "wild-aster": "#92316f",
        "wild-dove": "#8b8c89",
        "wild-ginger": "#7c4c53",
        "wild-lime": "#c3d363",
        "wild-orchid": "#d979a2",
        "wild-rose": "#ce8498",
        "willow": "#9a8b4f",
        "willow-bough": "#59754d",
        "willowherb": "#8e4483",
        "wind-chime": "#cac5c2",
        "windsor-wine": "#582b36",
        "wineberry": "#5a395b",
        "winetasting": "#492a34",
        "winsome-orchid": "#d4b9cb",
        "winter-bloom": "#47243b",
        "winter-green": "#4f9e81",
        "winter-moss": "#5b5a41",
        "winter-pear": "#b0b487",
        "winter-sky": "#a9c0cb",
        "winter-twig": "#948a7a",
        "winter-wheat": "#dfc09f",
        "winter-white": "#f5ecd2",
        "wisteria": "#a198af",
        "wistful-mauve": "#946c74",
        "withered-rose": "#a26666",
        "wood-ash": "#d7cab0",
        "wood-thrush": "#a47d43",
        "wood-violet": "#75406a",
        "woodbine": "#7b7f32",
        "woodrose": "#ae8c8e",
        "woodsmoke": "#947764",
        "wren": "#4a4139",
        "wrought-iron": "#999e98",
        "xenon-blue": "#b7c0d7",
        "yam": "#d0893f",
        "yarrow": "#face6d",
        "yellow-cream": "#efdc75",
        "yellow-iris": "#eee78e",
        "yellow-pear": "#ece99b",
        "yolk-yellow": "#e2b051",
        "young-wheat": "#e1e3a9",
        "yucca": "#a1d7c9",
        "zen-blue": "#9fa9be",
        "zephyr": "#c89fa5",
        "zephyr-blue": "#d3d9d1",
        "zephyr-green": "#7cb083",
        "zinc": "#92898a",
        "zinfandel": "#5c2935",
        "zinnia": "#ffa010",
    }

    def __init__(self, color_name: str):
        assert color_name in Pantone.COLOR_DEFINITION
        self.color_name: str = color_name
        super(Pantone, self).__init__(Pantone.COLOR_DEFINITION[color_name])

    def get_name(self) -> str:
        """
        This function returns the name of this pantone color
        """
        return self.color_name

    @staticmethod
    def find_nearest_pantone_color(color: Color) -> "Pantone":
        """
        This function find the nearest Pantone equivalent for a given Color
        """
        rgb_color_001: RGBColor = color.to_rgb()
        d_min: typing.Optional[Decimal] = None
        c_min: typing.Optional[str] = None
        for n, c in Pantone.COLOR_DEFINITION.items():
            rgb_color_002: RGBColor = HexColor(c)
            d: Decimal = (
                (rgb_color_001.red - rgb_color_002.red) ** 2
                + (rgb_color_001.green - rgb_color_002.green) ** 2
                + (rgb_color_001.blue - rgb_color_002.blue) ** 2
            )
            if d_min is None or d < d_min:
                d_min = d
                c_min = n
        assert d_min is not None
        assert c_min is not None
        return Pantone(c_min)

    def __deepcopy__(self, memodict={}):
        return Pantone(self.color_name)
